# ************************************************************************* 
# Copyright (c) 2014, SUSE LLC
# 
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
# 
# 3. Neither the name of SUSE LLC nor the names of its contributors may be
# used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# ************************************************************************* 
#
# Util module - reusable components
#
package App::Dochazka::CLI::Util;

use 5.012;
use strict;
use warnings;

use App::CELL qw( $CELL $log $meta $site );
use App::Dochazka::CLI qw( $current_emp $current_priv $debug_mode $prompt_date );
use App::Dochazka::CLI::TokenMap qw( $regex_map ); 
use App::Dochazka::Model::Employee;
use Data::Dumper;
use Date::Calc qw( check_date Add_Delta_Days );
use Exporter 'import';
use File::ShareDir;
use Log::Any::Adapter;
use Scalar::Util qw( looks_like_number );
use Time::Piece;
use Time::Seconds;
use Try::Tiny;
use Web::MREST::CLI::UserAgent qw( send_req );
use Web::MREST::Util qw( normalize_filespec );



=head1 NAME

App::Dochazka::CLI::Util - Various reusable components




=head1 VERSION

Version 0.184

=cut

our $VERSION = '0.184';




=head1 PACKAGE VARIABLES AND EXPORTS

=cut

our ( $t, $today, $yesterday, $tomorrow, $current_year, $current_century, $current_month );

our @EXPORT_OK = qw( 
    authenticate_to_server 
    determine_employee
    lookup_employee 
    init_cli_client
    init_timepiece
    normalize_date
    normalize_time
    parse_test
    refresh_current_emp 
    rest_error 
    truncate_to
    tsrange_from_timestamps
    $t
    $today
    $yesterday
    $tomorrow
    $current_year
    $current_century
    $current_month
);



=head1 FUNCTIONS


=head2 authenticate_to_server

Takes PROPLIST with three properties:

=over

=item C<< user >>

The username to authenticate as

=item C<< password >>

The password to use

=item C<< quiet >>

(Optional) boolean value whether to "be quiet" - defaults to 0

=back

=cut

sub authenticate_to_server {
    my %PROPLIST = ( 
        quiet => 0,
        @_,
    );
    $PROPLIST{'user'} = $PROPLIST{'user'} || 'demo';
    $PROPLIST{'password'} = $PROPLIST{'password'} || $PROPLIST{'user'};
    print "Authenticating to server at " . $site->MREST_CLI_URI_BASE . 
        " as user " . $PROPLIST{'user'} . "\n"
        unless $PROPLIST{'quiet'};

    # prompt for nick if none provided in site configuration
    #if ( ! $site->DOCHAZKA_REST_LOGIN_NICK ) {
    #    print "Server auth nick:     ";
    #    $meta->set( 'CURRENT_EMPLOYEE_NICK', <> );
    #}

    $meta->set( 'CURRENT_EMPLOYEE_NICK', $PROPLIST{'user'} );

    # prompt for password if necessary
    if ( ! $PROPLIST{'quiet'} and $PROPLIST{'user'} ne 'demo' and $PROPLIST{'password'} eq 'demo' ) {
        print "Server auth password: ";
        my $input = <STDIN>;
        chomp( $input ); 
        $meta->set( 'CURRENT_EMPLOYEE_PASSWORD', $input );
        #print "Password set to " . $meta->CURRENT_EMPLOYEE_PASSWORD . "\n";
    } else {
        $meta->set( 'CURRENT_EMPLOYEE_PASSWORD', $PROPLIST{'password'} );
    }

    # get info about us
    my $status;
    try {
        $status = send_req( 'GET', '/employee/self/priv' );
    } catch {
        $status = $_;
    };
    if ( !ref( $status ) ) {
        print "$status\n";
        exit;
    }
    return $status unless $status->ok;
    #print Dumper( $status );
    $current_emp = App::Dochazka::Model::Employee->spawn( %{ $status->payload->{'current_emp'} } );
    $current_priv = $status->payload->{'priv'};
    return $CELL->status_ok;
}


=head2 determine_employee

Given what might possibly be an employee specification, return
a status object that will either be an error (not OK) or contain
the employee object in the payload.

=cut

sub determine_employee {
    my $s_key = shift;

    my $status = ( $s_key )
        ? lookup_employee( $s_key )
        : refresh_current_emp();
    return ( $status->ok )
        ? $CELL->status_ok( 'EMPLOYEE_LOOKUP', 
            payload => App::Dochazka::Model::Employee->spawn( %{ $status->payload } ) )
        : rest_error( $status, "Employee lookup" );
}


=head2 lookup_employee

EMPLOYEE_SPEC may be "nick=...", "sec_id=...", "eid=...", or simply
"employee=...", in which case we use a clever algorithm to look up employees
(i.e. try looking up search key as nick, sec_id, and EID - in that order).

=cut

sub lookup_employee {
    my ( $s_key ) = @_;   # search key
    print "Entering " . __PACKAGE__ . "::lookup_employee with search key " . Dumper( $s_key )
        if $debug_mode;

    my ( $key_spec, $key ) = $s_key =~ m/^(.*)\=(.*)$/;

    my $status;
    if ( $key_spec =~ m/^emp/i ) {
        $status = send_req( 'GET', "employee/nick/$key" );
        if ( $status->not_ok and ( $status->{'http_code'} == 404 or $status->{'http_code'} == 403 ) ) {
            $status = send_req( 'GET', "employee/sec_id/$key" );
            if ( $status->not_ok and $status->{'http_code'} != 500 and looks_like_number( $key ) ) {
                $status = send_req( 'GET', "employee/eid/$key" );
            }
        }
    } elsif ( $key_spec =~ m/^nic/ ) {
        $status = send_req( 'GET', "employee/nick/$key" );
    } elsif ( $key_spec =~ m/^sec/ ) {
        $status = send_req( 'GET', "employee/sec_id/$key" );
    } elsif ( $key_spec =~ m/^eid/ ) {
        $status = send_req( 'GET', "employee/eid/$key" );
    } else {
        die "AAAHAAAHHH!!! Invalid employee lookup key " . ( defined( $key_spec ) ? $key_spec : "undefined" )
    }

    return $status;
}


=head2 init_cli_client

CLI client initialization routine: might die

=cut

sub init_cli_client {
    my ( $sitedir ) = @_;

    # always load the App::Dochazka::CLI distro sharedir
    my $target = File::ShareDir::dist_dir('App-Dochazka-CLI');
    print "Loading configuration files from $target\n";
    my $status = $CELL->load( sitedir => $target );
    die $status->text unless $status->ok;

    # load core config params and, if sitedir specified, site config params
    # as well
    my %CELL_ARGS = ( debug_mode => 1 );
    $CELL_ARGS{sitedir} = $sitedir if $sitedir;
    $status = $CELL->load( %CELL_ARGS );
    die $status->text unless $status->ok;

    init_logger(); 

    init_timepiece();

    # initialize the LWP::UserAgent object
    Web::MREST::CLI::UserAgent::init_ua();

    return $CELL->status_ok;
}


=head2 init_logger

Logger initialization routine

=cut

sub init_logger {
    my $log_file = normalize_filespec( $site->DOCHAZKA_CLI_LOG_FILE );
    unlink $log_file if $site->DOCHAZKA_CLI_LOG_FILE_RESET;
    Log::Any::Adapter->set('File', $log_file );
    $log->init( ident => 'dochazka-cli', debug_mode => 1 );
    $log->debug( 'Logger initialized' );
}


=head2 init_timepiece

(Re-)initialize the date/time-related package variables

=cut

sub init_timepiece {
    #print "Entering " . __PACKAGE__ . "::init_timepiece\n";
    $t = localtime;
    $today = $t->ymd;
    $prompt_date = $today unless $prompt_date;
    $yesterday = ($t - ONE_DAY)->ymd;
    $tomorrow = ($t + ONE_DAY)->ymd;
    $current_year = $t->year;
    ( $current_century ) = $current_year =~ m/^(\d{2,2})/;
    $current_month = $t->mon;
}


=head2 normalize_date

Normalize a date entered by the user. A date can take the following forms
(case is insignificant):

    YYYY-MM-DD
    YY-MM-DD
    MM-DD
    TODAY
    TOMORROW
    YESTERDAY
    +n
    -n

and any of the two-digit forms can be fulfilled by a single digit,
for example 2014-3-4 is March 4th, 2014.

All the forms except the first are converted into the YYYY-MM-DD form.
The last two forms listed, C<+n> and C<-n>, are calculated as offsets
from the "prompt date" (the date shown in the prompt), where C<n> is
interpreted as a number of days.

If an undefined or empty string is given, the prompt date is returned.

If the string does not match any of the forms, undef is returned.

Caveats:

=over

=item * two-digit years

If only YY is given, it is converted into YYYY by appending two digits
corresponding to the current century (e.g. 22 becomes 2022 during 2000-2099).

=item * special date forms

The special date forms "TODAY", "TOMORROW", and "YESTERDAY" are recognized,
and only the first three letters are significant, so "todMUMBOJUMBO" converts
to today's date.

=item * offsets

The C<n> in the offset can be any number in the range 0-999.

=item * no year

If no year is given, the current year is used.

=item * no date

If no date is given, the prompt date is used.

=item * single-digit forms

If a single-digit form is given for C<MM> or C<DD>, a leading zero is appended.

=back

=cut

sub normalize_date {
    my $rd = shift;  # rd == raw date
    my $nd;          # nd == normalized date

    # initialize timepiece so we can do things like $today, $tomorrow, etc.
    init_timepiece();

    # return prompt date if no raw date provided
    unless ( defined( $rd ) and length( $rd ) > 0 ) {
        #print "normalize_date(): no date provided, returning prompt date\n";
        #print "Prompt date is " . ( $prompt_date || 'undefined' ) . "\n";
        return $prompt_date;
    }

    if ( $rd =~ m/\A\d{4,4}-\d{1,2}-\d{1,2}\z/ ) {
        $nd = $rd;
    } elsif ( $rd =~ m/\A\d{2,2}-\d{1,2}-\d{1,2}\z/ ) {
        # year has only two digits: add the current century
        $nd = $current_century . $rd;
    } elsif ( $rd =~ m/\A\d{1,2}-\d{1,2}\z/ ) {
        # year omitted: add the current year
        $nd = $current_year . '-' . $rd;
    } elsif ( $rd =~ m/\Atod/i ) {
        $nd = $today;
    } elsif ( $rd =~ m/\Atom/i ) {
        $nd = $tomorrow;
    } elsif ( $rd =~ m/\Ayes/i ) {
        $nd = $yesterday;
    } elsif ( $rd =~ m/\A[\+\-]\d{1,3}\z/ ) {
        # offset from prompt date
        $prompt_date =~ m/\A(?<yyyy>\d{4,4})-(?<mm>\d{1,2})-(?<dd>\d{1,2})\z/;
        if ( check_date( $+{'yyyy'}, $+{'mm'}, $+{'dd'} ) ) {
            # prompt date is OK, apply delta
            my ( $year, $month, $day ) = Add_Delta_Days(
                $+{'yyyy'}, $+{'mm'}, $+{'dd'},
                $rd,
            );
            $nd = "$year-$month-$day";
        } else {
            return undef;
        }
    } else {
        # anything else - invalid timestamp
        return undef;
    }

    # add leading zeroes to month and day, if necessary
    $nd =~ m/\A(?<yyyy>\d{4,4})-(?<mm>\d{1,2})-(?<dd>\d{1,2})\z/;
    return undef unless $+{yyyy} and $+{mm} and $+{dd};
    $nd = sprintf( "%d-%02d-%02d", $+{yyyy}, $+{mm}, $+{dd} );

    return "$nd";
}


=head2 normalize_time

Normalize a time entered by the user. A time can take the following forms

    HH:MM:SS
    HH:MM

and any of the two-digit forms can be fulfilled by a single digit,
for example 6:4:9 is 6:04 a.m. and nine seconds

=over

=item * single-digit forms

If a single-digit form is given, a leading zero is appended.

=item * seconds

If seconds are given, they are ignored.

=item * no validation

No attempt is made to validate the time -- this is done later, by
PostgreSQL.

=back

=cut

sub normalize_time {
    my $rt = shift;  # rt == raw time

    return '00:00' unless $rt;

    # normalize time part
    $rt =~ m/\A(?<hh>\d{1,2}):(?<mm>\d{1,2})(:\d{1,2})?\z/;
    my ( $hours, $minutes ) = ( $+{hh}, $+{mm} );
    return undef unless defined( $hours ) and defined( $minutes );
    # handle single zeroes
    $hours = '00' if $hours eq '0';
    $minutes = '00' if $minutes eq '0';
    return undef unless $hours and $minutes;
    my $nt = sprintf( "%02d:%02d", $+{hh}, $+{mm} );
    
    return "$nt";
}


=head2 parse_test

Given a reference to the PARAMHASH a command handler was called with, check
if there is a PARSE_TEST property there, and if it is true return the
full subroutine name of the caller. 

=cut

sub parse_test {
    #print ( 'parse_test arg list: ' . join( ' ', @_ ) . "\n" );
    my ( %PARAMHASH ) = @_;
    if ( $PARAMHASH{'PARSE_TEST'} ) {
        return $CELL->status_ok( 'DOCHAZKA_CLI_PARSE_TEST', 
            payload => (caller(1))[3] );
    } 
    return $CELL->status_not_ok( 'DOCHAZKA_CLI_PARSE_TEST' );
}


=head2 refresh_current_emp

REST calls are cheap, so look up C<< $current_emp >> again just to make sure.

=cut

sub refresh_current_emp {
    my $status = send_req( 'GET', 'employee/eid/' . $current_emp->eid );
    die "Problem with data integrity (current employee)" unless $status->ok;
    $current_emp = App::Dochazka::Model::Employee->spawn( %{ $status->payload } );
    return $status;
}


=head2 rest_error

Given a non-OK status object and a string briefly identifying (for the user) in
which operation the error occurred, check for and report common errors like 404.

Returns a true value if a common error is successfully identified and reported,
false otherwise.

=cut

sub rest_error {
    my ( $status, $id_str ) = @_;
    my $rv = "\n";
    $rv .= "Entering " . __PACKAGE__ . "::rest_error ($id_str)"
        if $debug_mode;

    $rv .= "Error encountered on attempted operation \"$id_str\"\n";

    # special handling if payload is a string
    if ( ref( $status->payload ) eq '' ) {

        $rv .= $status->payload;
        $rv .= "\n";

    } elsif ( ref( $status->payload ) eq 'HASH' ) {

        my $http_status = $status->{'http_status'} || $status->payload->{'http_code'} || "Cannot be determined";
        my $method = $status->payload->{'http_method'} || "Cannot be determined";
        my $uri_path = $status->payload->{'uri_path'} || '';
        $rv .= "REST operation: $method $uri_path\n";
        $rv .= "HTTP status: $http_status\n";
        $rv .= "Explanation: ";
        $rv .= $status->code;
        $rv .= ( $status->code eq $status->text ) 
            ? "\n"
            : ': ' . $status->text . "\n";
        $rv .= "Permanent? ";
        $rv .= ( $status->payload->{'permanent'} )
            ? "YES\n"
            : "NO\n";

    } else {
        die "AH! in rest_error, payload is neither a hashref nor an ordinary scalar";
    }

    my $status_clone = App::CELL::Status->new( 
        level => $status->level,
        code => 'REST_ERROR',
        payload => $rv,
        rest_payload => $status->payload,
        rest_uri_path => $status->{'uri_path'},
        http_status => $status->{'http_status'},
    );
    return $status_clone;
}


=head2 truncate_to

Given a string and a maximum length (defaults to 32), truncates to that length.
Returns a copy of the string. If any characters were actually removed in the
truncate operation, '...' is appended.

=cut

sub truncate_to {
    my ( $str, $mlen ) = @_;
    $mlen = 32 unless defined( $mlen );
    my $len = length $str || 0;
    return $str unless $len > $mlen;
    my $str_copy = substr( $str, 0, $mlen );
    $str_copy .= '...' if $len > $mlen;
    return $str_copy;
}


=head2 tsrange_from_timestamps

Given two timestamps, normalize them and make a tsrange from them.
If either of the timestamps does not normalize properly, return a NOT_OK status.
If all goes well, return OK status with tsrange in the payload.

=cut

sub tsrange_from_timestamps {
    my ( $ts1, $ts2 ) = @_;

    $ts1 = normalize_timestamp( $ts1 );
    $ts2 = normalize_timestamp( $ts2 );
    return $CELL->status_not_ok( 'TSRANGE_DOES_NOT_COMPUTE' ) unless $ts1 and $ts2;

    return $CELL->status_ok( 'DOCHAZKA_CLI_NORMAL_COMPLETION',
        payload => "[ $ts1, $ts2 )" );
}


1;
