# ************************************************************************* 
# Copyright (c) 2014, SUSE LLC
# 
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
# 
# 3. Neither the name of SUSE LLC nor the names of its contributors may be
# used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# ************************************************************************* 
#
# employee command targets
package App::Dochazka::CLI::Commands::Employee;

use 5.012;
use strict;
use warnings;

use App::CELL qw( $CELL );
use App::Dochazka::CLI qw( $current_emp $debug_mode );
use App::Dochazka::CLI::Util qw( lookup_employee refresh_current_emp rest_error );
use App::Dochazka::REST::Model::Employee;
use Data::Dumper;
use Exporter 'import';
use Web::MREST::CLI::UserAgent qw( send_req );




=head1 NAME

App::Dochazka::CLI::Commands::Employee - Employee commands




=head1 VERSION

Version 0.151

=cut

our $VERSION = '0.151';



=head1 PACKAGE VARIABLES AND EXPORTS

=cut

our @EXPORT_OK = qw( 
    employee_profile
    set_employee_self_sec_id 
    set_employee_self_fullname 
    set_employee_other_sec_id
    set_employee_other_fullname
);


=head1 FUNCTIONS

=head2 employee_profile

    EMPLOYEE
    EMPLOYEE_SPEC
    EMPLOYEE PROFILE
    EMPLOYEE_SPEC PROFILE
    EMPLOYEE SHOW
    EMPLOYEE_SPEC SHOW

=cut

sub employee_profile {
    print "Entering " . __PACKAGE__ . "::employee_profile\n" if $debug_mode;
    my ( $ts, $th ) = @_;
    print "ts " . Dumper( $ts ) if $debug_mode;
    print "th " . Dumper( $th ) if $debug_mode;
    my $s_key = $th->{'EMPLOYEE_SPEC'};

    print "Search key: " . Dumper( $s_key ) . "\n" if $debug_mode;

    die "AAAAGAGAAGGGGHHGHHHH!" if defined( $s_key ) and ref( $s_key ) ne '';

    # GET employee/eid/:eid
    my $status = ( $s_key )
        ? lookup_employee( $s_key )
        : refresh_current_emp();
    return rest_error( $status, "Employee lookup" ) unless $status->ok;
    my $emp = App::Dochazka::REST::Model::Employee->spawn( %{ $status->payload } );

    # GET priv/eid/:eid
    $status = send_req( 'GET', 'priv/eid/' . $emp->eid );
    return rest_error( $status, "Determine employee privlevel" ) unless $status->ok;
    my $priv = $status->payload->{'priv'};

    my $message;
    $message = "\nEmployee profile\n================\n\n";
    $message .= "Full name:    " . ( $emp->fullname ? $emp->fullname : "(not set)" ) . "\n";
    $message .= "Nick:         " . $emp->nick . "\n";
    $message .= "Secondary ID: " . ( $emp->sec_id ? $emp->sec_id : "(not set)" ) . "\n";
    $message .= "Dochazka EID: " . $emp->eid . "\n";
    $message .= "Password*:    (" . ( $emp->passhash ? "set" : "not set" ) . ")\n";
    $message .= "Privlevel:    $priv\n";
    $message .= "\n";
    $message .= "* Internal password stored in the Dochazka database - \n";
    $message .= "  irrelevant if external authentication in use.\n";

    return $CELL->status_ok( 'DOCHAZKA_CLI_NORMAL_COMPLETION', payload => $message );
}

=head2 set_employee_self_sec_id

SET EMPLOYEE SEC_ID _TERM

=cut

sub set_employee_self_sec_id {
    my ( $ts, $th ) = @_;
    return set_employee( 
        emp_obj => $current_emp,
        prop => 'sec_id', 
        val => $th->{'_TERM'},
    );
}


=head2 set_employee_self_fullname

SET EMPLOYEE FULLNAME

=cut

sub set_employee_self_fullname {
    my ( $ts, $th ) = @_;
    return set_employee( 
        emp_obj => $current_emp,
        prop => 'fullname', 
        val => $th->{'_REST'},
    );
}


=head2 set_employee_other_sec_id

EMPLOYEE_SPEC SET SEC_ID _TERM

=cut

sub set_employee_other_sec_id {
    my ( $ts, $th ) = @_;
    return set_employee( 
        emp_spec => $th->{'EMPLOYEE_SPEC'},
        prop => 'sec_id', 
        val => $th->{'_TERM'},
    );
}


=head2 set_employee_other_fullname

EMPLOYEE_SPEC SET FULLNAME

=cut

sub set_employee_other_fullname {
    my ( $ts, $th ) = @_;
    return set_employee( 
        emp_spec => $th->{'EMPLOYEE_SPEC'},
        prop => 'fullname', 
        val => $th->{'_REST'},
    );
}


=head2 set_employee

=cut

sub set_employee {
    my %PROPLIST = @_;
    my $status;
    my $emp_obj;
    if ( my $e_spec = $PROPLIST{'emp_spec'} ) {
        $status = lookup_employee( $e_spec );
        return rest_error( $status, "Employee lookup" ) unless $status->ok;
        $emp_obj = App::Dochazka::REST::Model::Employee->spawn( %{ $status->payload } );
    } elsif ( $emp_obj = $PROPLIST{'emp_obj'} ) {
    } else {
        die "AAAAAAAAAAAAAHHHHH!";
    }
    my $eid = $emp_obj->eid;
    my $prop = $PROPLIST{'prop'};
    my $val = $PROPLIST{'val'};
    $val =~ s/['"]//g;
    $status = send_req( 'POST', "employee/eid", <<"EOS" );
{ "eid" : $eid, "$prop" : "$val" }
EOS
    return rest_error( $status, "Modify employee profile" ) unless $status->ok;

    my $message = "Profile of employee " . $emp_obj->nick . 
        " has been modified ($prop -> $val)\n";

    return $CELL->status_ok( 'DOCHAZKA_CLI_NORMAL_COMPLETION', payload => $message );
}

1;
