# ************************************************************************* 
# Copyright (c) 2014, SUSE LLC
# 
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
# 
# 3. Neither the name of SUSE LLC nor the names of its contributors may be
# used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# ************************************************************************* 
#
# Show.pm - SHOW command implementations
#
package App::Dochazka::CLI::Show;

use 5.012;
use strict;
use warnings;

use App::CELL qw( $CELL $log );
use App::Dochazka::CLI qw( $current_emp $current_priv $message );
use App::Dochazka::Model::Employee;
use Data::Dumper;
use Exporter qw( import );
use Scalar::Util qw( looks_like_number );
use Web::MREST::CLI::UserAgent qw( send_req );



=head1 NAME

App::Dochazka::CLI::Show - SHOW command implementations




=head1 VERSION

Version 0.096

=cut

our $VERSION = '0.096';




=head1 PACKAGE VARIABLES AND EXPORTS

=cut

our @EXPORT_OK = qw( show_employee );




=head1 FUNCTIONS


=head2 show_employee

The SHOW EMPLOYEE command. Possibly takes a single argument containing the nick
of the employee to be shown. If an argument is given and it does not match a
nick, try matching it against sec_id and, finally, against eid (but only if the
argument is a natural number).

=cut

sub show_employee {
    my $s_key = shift;
    return $CELL->status_not_ok if defined( $s_key ) and ref( $s_key ) ne '';
    $log->debug( "Entering " . __PACKAGE__ . ":::show_employee with search key " . 
        ( ( defined $s_key ) ? $s_key : "undefined" ) );

    # GET employee/eid/:eid
    my $status;
    if ( $s_key ) {
        $status = send_req( 'GET', "employee/nick/$s_key" );
        if ( $status->not_ok and $status->{'http_code'} != 500 ) {
            $status = send_req( 'GET', "employee/sec_id/$s_key" );
            if ( $status->not_ok and $status->{'http_code'} != 500 and looks_like_number( $s_key ) ) {
                $status = send_req( 'GET', "employee/eid/$s_key" );
            }
        }
    } else {
        $status = send_req( 'GET', 'employee/eid/' . $current_emp->eid );
    }
    return $status unless $status->ok;
    my $emp = App::Dochazka::Model::Employee->spawn( %{ $status->payload } );

    # GET priv/eid/:eid
    $status = send_req( 'GET', 'priv/eid/' . $emp->eid );
    return $status unless $status->ok;
    my $priv = $status->payload->{'priv'};

    $message = "\nEmployee profile\n================\n\n";
    $message .= "Full name:    " . ( $emp->fullname ? $emp->fullname : "(not set)" ) . "\n";
    $message .= "Nick:         " . $emp->nick . "\n";
    $message .= "Secondary ID: " . ( $emp->sec_id ? $emp->sec_id : "(not set)" ) . "\n";
    $message .= "Dochazka ID:  " . $emp->eid . "\n";
    $message .= "Password*:    (" . ( $emp->passhash ? "set" : "not set" ) . ")\n";
    $message .= "Privlevel:    $priv\n";
    $message .= "\n";
    $message .= "* Internal password stored in the Dochazka database - \n";
    $message .= "  irrelevant if external authentication in use.\n";
    return;
}

1;
