use strict;
use warnings;

use Test::More qw(no_plan);

my $standards; # for the standards map
my $rules;     # rule specifics to a given file extention
my @extentions = ('pl', 'pm', 't', 'pl2', 'pm2', 't2');

my %lines = (
    "# comment rule should cause this to fail" => "fail",
    "}"                                        => "fail",
    "sdflklsdf"                                => "pass",
    ""                                         => "fail",
    "__END__"                                  => "stop",
    "__DATA__"                                 => "stop",
);

my @java_lines = (
    "package foo;",
    "/*****************************",
    " * this shouldn't count",
    "*****************************/",
    "",
    "/*",
    " * dsafthis shouldn't count",
    "*/",
    "// main does something interesting",
    "class man{",
    "public static void main(String[] args) {",
    "System.out.printlin('This should count');",
    "}",
    "}",
);

my %java_status_for = (
    "package foo;"                                        =>"pass",
    "/*****************************"                      =>"fail",
    " * this shouldn't count"                             =>"fail",                
    "*****************************/"                      =>"fail",
    ""                                                    =>"fail",
    "/*"                                                  =>"fail",
    " * dsafthis shouldn't count"                         =>"fail",                
    "*/"                                                  =>"fail",
    ""                                                    =>"fail",
    "// main does something interesting"                  =>"fail",
    "class man{"                                          =>"pass",
    "public static void main(String[] args) {"            =>"pass",
    "System.out.printlin('This should count');"           =>"pass",
    "}"                                                   =>"fail",
    "}"                                                   =>"fail",
);

BEGIN { use_ok( 'CodeCounter::Standards' ); }

ok($standards = load_standards('t/standards.xml'),     'loading standards file');
ok(ref($standards) eq ref({}),                         'checking return type is hash ref');
ok(exists $standards->{'pm'} && exists $standards->{'pl'} && exists $standards->{'t'},
                                                        'ensuring extentions exist');
# single line test using Perl
for my $extention (@extentions){
    $rules = $standards->{$extention};
    ok(ref($rules) eq ref({}),                         'checking return type is hash ref');
    
    
    for my $line (keys %lines){
        my $result;
        
        for my $rule (keys %{$rules}){ 
            $result = $rules->{$rule}->($line);
            last if $result ne "pass";
        }
        
        ok($result eq $lines{$line}, "$line for $result");
    }
}

# multi-line test using Java

$rules = $standards->{'java'};
ok(ref($rules) eq ref({}),                         'checking return type is hash ref');

for my $line (@java_lines){
    my $result;
        
    for my $rule (keys %{$rules}){ 
       $result = $rules->{$rule}->($line);
       last if $result ne "pass";
    }
        
    ok($result eq $java_status_for{$line}, "$line for $result");    
}