package FFI::Platypus;

use strict;
use warnings;
use 5.008001;
use Carp qw( croak );

# ABSTRACT: Write Perl bindings to non-Perl libraries without C or XS
our $VERSION = '0.16'; # VERSION

# Platypus Man,
# Platypus Man,
# Does Everything The Platypus Can
# ...
# Watch Out!
# Here Comes The Platypus Man


our @CARP_NOT = qw( FFI::Platypus::Declare );

require XSLoader;
XSLoader::load(
  'FFI::Platypus', eval q{ $VERSION } || do {
    # this is for testing without dzil
    # it expects MYMETA.json for FFI::Platypus
    # to be in the current working directory.
    require JSON::PP;
    my $fh;
    open($fh, '<', 'MYMETA.json') || die "unable to read MYMETA.json";
    my $config = JSON::PP::decode_json(do { local $/; <$fh> });
    close $fh;
    $config->{version};
  }
);


sub new
{
  my($class, %args) = @_;
  my @lib;
  if(defined $args{lib})
  {
    if(!ref($args{lib}))
    {
      push @lib, $args{lib};
    }
    elsif(ref($args{lib}) eq 'ARRAY')
    {
      push @lib, @{$args{lib}};
    }
    else
    {
      croak "lib argument must be a scalar or array reference";
    }
  }
  bless {
    lib              => \@lib,
    handles          => {},
    types            => {},
    ignore_not_found => defined $args{ignore_not_found} ? $args{ignore_not_found} : 0,
  }, $class;
}


sub lib
{
  my($self, @new) = @_;

  if(@new)
  {
    push @{ $self->{lib} }, @new;
  }
  
  @{ $self->{lib} };
}


sub ignore_not_found
{
  my($self, $value) = @_;
  
  if(defined $value)
  {
    $self->{ignore_not_found} = $value;
  }
  
  $self->{ignore_not_found};
}


sub type
{
  my($self, $name, $alias) = @_;
  croak "usage: \$ffi->type(name => alias) (alias is optional)" unless defined $self && defined $name;
  croak "spaces not allowed in alias" if defined $alias && $alias =~ /\s/;
  croak "allowed characters for alias: [A-Za-z0-9_]+" if defined $alias && $alias =~ /[^A-Za-z0-9_]/;

  require FFI::Platypus::ConfigData;
  my $type_map = FFI::Platypus::ConfigData->config("type_map");

  croak "alias conflicts with existing type" if defined $alias && (defined $type_map->{$alias} || defined $self->{types}->{$alias});

  if($name =~ /-\>/)
  {
    # for closure types we do not try to convet into the basic type
    # so you can have many many many copies of a given closure type
    # if you do not spell it exactly the same each time.  Recommended
    # thsat you use an alias for a closure type anyway.
    $self->{types}->{$name} ||= FFI::Platypus::Type->new($name, $self);
  }
  else
  {
    my $basic = $name;
    my $extra = '';
    if($basic =~ s/\s*((\*|\[|\<).*)$//)
    {
      $extra = " $1";
    }
  
    croak "unknown type: $basic" unless defined $type_map->{$basic};
    $self->{types}->{$name} = $self->{types}->{$type_map->{$basic}.$extra} ||= FFI::Platypus::Type->new($type_map->{$basic}.$extra, $self);
  }
  
  if(defined $alias)
  {
    $self->{types}->{$alias} = $self->{types}->{$name};
  }
  $self;
}


sub custom_type
{
  my($self, $name, $cb) = @_;
  
  my $type = $cb->{native_type};
  $type ||= 'opaque';
  
  my $argument_count = $cb->{argument_count} || 1;
  
  croak "argument_count must be >= 1"
    unless $argument_count >= 1;
  
  croak "Usage: \$ffi->custom_type(\$name, { ... })"
    unless defined $name && ref($cb) eq 'HASH';
  
  croak "must define at least one of native_to_perl, perl_to_native, or perl_to_native_post"
    unless defined $cb->{native_to_perl} || defined $cb->{perl_to_native} || defined $cb->{perl_to_native_post};
  
  require FFI::Platypus::ConfigData;
  my $type_map = FFI::Platypus::ConfigData->config("type_map");  
  croak "$type is not a native type" unless defined $type_map->{$type} || $type eq 'string';
  croak "name conflicts with existing type" if defined $type_map->{$name} || defined $self->{types}->{$name};
  
  $self->{types}->{$name} = FFI::Platypus::Type->_new_custom_perl(
    $type_map->{$type},
    $cb->{perl_to_native},
    $cb->{native_to_perl},
    $cb->{perl_to_native_post},
    $argument_count,
  );
  
  $self;
}


sub load_custom_type
{
  my($self, $name, $alias, @type_args) = @_;

  croak "usage: \$ffi->load_custom_type(\$name, \$alias, ...)"
    unless defined $name && defined $alias;

  $name = "FFI::Platypus::Type$name" if $name =~ /^::/;
  $name = "FFI::Platypus::Type::$name" unless $name =~ /::/;
  
  unless($name->can("ffi_custom_type_api_1"))
  {
    eval qq{ use $name () };
    warn $@ if $@;
  }
  
  unless($name->can("ffi_custom_type_api_1"))
  {
    croak "$name does not appear to conform to the custom type API";
  }
  
  my $cb = $name->ffi_custom_type_api_1($self, @type_args);
  $self->custom_type($alias => $cb);
  
  $self;
}

sub _type_lookup
{
  my($self, $name) = @_;
  $self->type($name) unless defined $self->{types}->{$name};
  $self->{types}->{$name};
}


sub types
{
  my($self) = @_;
  $self = $self->new unless ref $self && eval { $self->isa('FFI::Platypus') };
  require FFI::Platypus::ConfigData;
  my %types = map { $_ => 1 } keys %{ FFI::Platypus::ConfigData->config("type_map") };
  $types{$_} ||= 1 foreach keys %{ $self->{types} };
  sort keys %types;
}


sub type_meta
{
  my($self, $name) = @_;
  $self = $self->new unless ref $self && eval { $self->isa('FFI::Platypus') };
  my $type = $self->_type_lookup($name);
  $type->meta;
}


sub function
{
  my($self, $name, $args, $ret) = @_;
  croak "usage \$ffi->function( name, [ arguments ], return_type)" unless @_ == 4;
  my @args = map { $self->_type_lookup($_) || croak "unknown type: $_" } @$args;
  $ret = $self->_type_lookup($ret) || croak "unknown type: $ret";
  my $address = $name =~ /^-?[0-9]+$/ ? $name : $self->find_symbol($name);
  croak "unable to find $name" unless defined $address || $self->ignore_not_found;
  return unless defined $address;
  FFI::Platypus::Function->new($self, $address, $ret, @args);
}


sub attach
{
  my($self, $name, $args, $ret, $proto) = @_;
  my($c_name, $perl_name) = ref($name) ? @$name : ($name, $name);

  croak "you tried to provide a perl name that looks like an address"
    if $perl_name =~ /^-?[0-9]+$/;
  
  my $function = $self->function($c_name, $args, $ret);
  
  if(defined $function)
  {
    my($caller, $filename, $line) = caller;
    $perl_name = join '::', $caller, $perl_name
      unless $perl_name =~ /::/;
    
    $function->attach($perl_name, "$filename:$line", $proto);
  }
  
  $self;
}


sub closure
{
  my($self, $coderef) = @_;
  FFI::Platypus::Closure->new($coderef);
}


sub cast
{
  $_[0]->function(0 => [$_[1]] => $_[2])->call($_[3]);
}


sub attach_cast
{
  my($self, $name, $type1, $type2) = @_;
  my $caller = caller;
  $name = join '::', $caller, $name unless $name =~ /::/;
  $self->attach([0 => $name] => [$type1] => $type2 => '$');
  $self;
}


sub sizeof
{
  my($self,$type) = @_;
  $self->type($type);
  $self->type_meta($type)->{size};
}


sub find_symbol
{
  my($self, $name) = @_;

  foreach my $path (@{ $self->{lib} })
  {
    my $handle = do { no warnings; $self->{handles}->{$path||0} } || FFI::Platypus::dl::dlopen($path);
    next unless $handle;
    my $address = FFI::Platypus::dl::dlsym($handle, $name);
    if($address)
    {
      $self->{handles}->{$path||0} = $handle;
      return $address;
    }
    else
    {
      FFI::Platypus::dl::dlclose($handle) unless $self->{handles}->{$path||0};
    }
  }
  return;
}


sub package
{
  my($self, $module, $modlibname) = @_;
  
  require FFI::Platypus::ConfigData;
  my $dlext = FFI::Platypus::ConfigData->config("config_dlext");

  ($module, $modlibname) = caller() unless defined $modlibname;  
  my @modparts = split /::/, $module;
  my $modfname = $modparts[-1];
  my $modpname = join('/',@modparts);
  my $c = @modparts;
  $modlibname =~ s,[\\/][^\\/]+$,, while $c--;    # Q&D basename
  my $file = "$modlibname/auto/$modpname/$modfname.$dlext";
  unless(-e $file)
  {
    $modlibname =~ s,[\\/][^\\/]+$,,;
    $file = "$modlibname/arch/auto/$modpname/$modfname.$dlext";
  }
  
  $self->lib($file) if -e $file;
  
  $self;
}


sub DESTROY
{
  my($self) = @_;
  foreach my $handle (values %{ $self->{handles} })
  {
    next unless $handle;
    FFI::Platypus::dl::dlclose($handle);
  }
  delete $self->{handles};
}

package FFI::Platypus::Function;

our $VERSION = '0.16'; # VERSION

use overload '&{}' => sub {
  my $ffi = shift;
  sub { $ffi->call(@_) };
};

package FFI::Platypus::Closure;

use Scalar::Util qw( refaddr);
use Carp qw( croak );

our $VERSION = '0.16'; # VERSION

our %cbdata;

sub new
{
  my($class, $coderef) = @_;
  croak "not a coderef" unless ref($coderef) eq 'CODE';
  my $self = bless $coderef, $class;
  $cbdata{refaddr $self} = [];
  $self;
}

sub add_data
{
  my($self, $payload) = @_;
  push @{ $cbdata{refaddr $self} }, bless \$payload, 'FFI::Platypus::ClosureData';
}

sub DESTROY
{
  my($self) = @_;
  delete $cbdata{refaddr $self};
}

package FFI::Platypus::ClosureData;

our $VERSION = '0.16'; # VERSION

package FFI::Platypus::Type;

use Carp qw( croak );

our $VERSION = '0.16'; # VERSION

sub new
{
  my($class, $type, $platypus) = @_;

  # the platypus object is only needed for closures, so
  # that it can lookup existing types.

  if($type =~ m/^\((.*)\)-\>\s*(.*)\s*$/)
  {
    croak "passing closure into a closure not supported" if $1 =~ /(\(|\)|-\>)/;
    my @argument_types = map { $platypus->_type_lookup($_) } map { s/^\s+//; s/\s+$//; $_ } split /,/, $1;
    my $return_type = $platypus->_type_lookup($2);
    return $class->_new_closure($return_type, @argument_types);
  }
  
  my $ffi_type;
  my $platypus_type;
  my $array_size = 0;
  
  if($type eq 'string')
  {
    $ffi_type = 'pointer';
    $platypus_type = 'string';
  }
  elsif($type =~ s/\s+\*$//)
  {
    $ffi_type = $type;
    $platypus_type = 'pointer';
  }
  elsif($type =~ s/\s+\[([0-9]+)\]$//)
  {
    $ffi_type = $type;
    $platypus_type = 'array';
    $array_size = $1;
  }
  else
  {
    $ffi_type = $type;
    $platypus_type = 'ffi';
  }
  
  $class->_new($ffi_type, $platypus_type, $array_size);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

FFI::Platypus - Write Perl bindings to non-Perl libraries without C or XS

=head1 VERSION

version 0.16

=head1 SYNOPSIS

 use FFI::Platypus;
 
 my $ffi = FFI::Platypus->new;
 $ffi->lib(undef); # search libc
 
 # call dynamically
 $ffi->function( puts => ['string'] => 'int' )->call("hello world");
 
 # attach as a xsub and call (much faster)
 $ffi->attach( puts => ['string'] => 'int' );
 puts("hello world");

=head1 DESCRIPTION

Platypus provides a method for creating interfaces to machine code 
libraries.  This implementation uses C<libffi>, a library that provides 
Foreign Function Interfaces for a number of other languages, including 
Ruby and Python.  Platypus can be used in stand alone scripts, or to 
create Perl extensions for CPAN.  There are a number of reasons why you 
might want to write extensions with FFI instead of XS:

=over 4

=item FFI / Platypus does not require messing with the guts of Perl

XS is less of an API and more of the guts of perl splayed out to do 
whatever you want.  That may at times be very powerful, but it may also 
sometimes be very dangerous to your mental health.

=item FFI / Platypus is portable

Lots of languages have FFI interfaces, and it is subjectively easier to 
port an extension written in FFI in Perl or another language to FFI in 
another language or Perl.  One goal of the Platypus Project is to reduce 
common interface specifications to a common format like JSON that could 
be shared between different languages.

=item FFI / Platypus could be a bridge to Perl 6

One of those "other" languages could be Perl 6 and Perl 6 already has an 
FFI interface I am told.

=item FFI / Platypus is pure perl (sorta)

One Platypus script or module works on any platform where the libraries 
it uses are available.  That means you can deploy your Platypus script 
in a shared filesystem where they may be run on different platforms.  It 
also means that Platypus modules do not need to be installed in the 
platform specific Perl library path.

=item FFI / Platypus is not C or C++ centric

XS is implemented primarily as a bunch of C macros, which requires at 
least some understanding of C, (sadly) the C pre-processor, and some C++ 
caveats (since on some platforms Perl is compiled and linked with a C++ 
compiler). Platypus on the other hand could be used to call other 
compiled languages, like Rust, Go or even assembly, allowing you to 
focus on your strengths.

=item FFI / Platypus does not require a parser

L<Inline> isolates the extension developer from XS to some extent, but 
it also requires a parser.  The various L<Inline> language bindings are 
a great technical achievement, but I think writing a parser for every 
language that you want to interface with is a bit of an anti-pattern.

=back

This document consists of an API reference, a set of examples, some 
support and development (for contributors) information.  If you are new 
to Platypus or FFI, you may want to skip down to the 
L<EXAMPLES|FFI::Platypus#EXAMPLES> to get a taste of what you can do 
with Platypus.

Platypus also provides an declarative interface you may want to use 
instead of the object oriented interface called 
L<FFI::Platypus::Declare>.

Platypus has extensive documentation of types at L<FFI::Platypus::Type> 
and its custom types API at L<FFI::Platypus::API>.

=for stopwords ØMQ

=head1 CONSTRUCTORS

=head2 new

 my $ffi = FFI::Platypus->new(%options);

Create a new instance of L<FFI::Platypus>.

Any types defined with this instance will be valid for this instance 
only, so you do not need to worry about stepping on the toes of other 
CPAN FFI / Platypus Authors.

Any functions found will be out of the list of libraries specified with 
the L<lib|FFI::Platypus#lib> attribute.

=head3 options

=over 4

=item lib

Either a pathname (string) or a list of pathnames (array ref of strings) 
to pre-populate the L<lib|FFI::Platypus#lib> attribute.

=item ignore_not_found

[version 0.15]

Set the L<ignore_not_found|FFI::Platypus#ignore_not_found> attribute.

=back

=head1 ATTRIBUTES

=head2 lib

 $ffi->lib($path1, $path2, ...);
 my @paths = $ffi->lib;

The list of libraries to search for symbols in.

The most portable and reliable way to find dynamic libraries is by using 
L<FFI::CheckLib>, like this:

 use FFI::CheckLib 0.06;
 $ffi->lib(find_lib_or_die lib => 'archive'); 
   # finds libarchive.so on Linux
   #       libarchive.bundle on OS X
   #       libarchive.dll (or archive.dll) on Windows
   #       cygarchive-13.dll on Cygwin
   #       ...
   # and will die if it isn't found

L<FFI::CheckLib> has a number of options, such as checking for specific 
symbols, etc.  You should consult the documentation for that module.

As a special case, if you add C<undef> as a "library" to be searched, 
Platypus will also search the current process for symbols. This is 
mostly useful for finding functions in the standard C library, without 
having to know the name of the standard c library for your platform (as 
it turns out it is different just about everywhere!).

=head2 ignore_not_found

[version 0.15]

 $ffi->ignore_not_found(1);
 my $ignore_not_found = $ffi->ignore_not_found;

Normally the L<attach|FFI::Platypus#attach> and 
L<function|FFI::Platypus#function> methods will throw an exception if it 
cannot find the name of the function you provide it.  This will change 
the behavior such that L<function|FFI::Platypus#function> will return 
C<undef> when the function is not found and 
L<attach|FFI::Platypus#attach> will ignore functions that are not found. 
This is useful when you are writing bindings to a library and have many 
optional functions and you do not wish to wrap every call to 
L<function|FFI::Platypus#function> or L<attach|FFI::Platypus#attach> in 
an C<eval>.

=head1 METHODS

=head2 type

 $ffi->type($typename);
 $ffi->type($typename => $alias);

Define a type.  The first argument is the native or C name of the type.  
The second argument (optional) is an alias name that you can use to 
refer to this new type.  See L<FFI::Platypus::Type> for legal type 
definitions.

Examples:

 $ffi->type('sint32'); # oly checks to see that sint32 is a valid type
 $ffi->type('sint32' => 'myint'); # creates an alias myint for sint32
 $ffi->type('bogus'); # dies with appropriate diagnostic

=head2 custom_type

 $ffi->custom_type($alias => {
   native_type         => $native_type,
   native_to_perl      => $coderef,
   perl_to_native      => $coderef,
   perl_to_native_post => $coderef,
 });

Define a custom type.  See L<FFI::Platypus::Type#Custom-Types> for details.

=head2 load_custom_type

 $ffi->load_custom_type($name => $alias, @type_args);

Load the custom type defined in the module I<$name>, and make an alias 
I<$alias>. If the custom type requires any arguments, they may be passed 
in as I<@type_args>. See L<FFI::Platypus::Type#Custom-Types> for 
details.

If I<$name> contains C<::> then it will be assumed to be a fully 
qualified package name. If not, then C<FFI::Platypus::Type::> will be 
prepended to it.

=head2 types

 my @types = $ffi->types;
 my @types = FFI::Platypus->types;

Returns the list of types that FFI knows about.  This will include the 
native C<libffi> types (example: C<sint32>, C<opaque> and C<double>) and 
the normal C types (example: C<unsigned int>, C<uint32_t>), any types 
that you have defined using the L<type|FFI::Platypus#type> method, and
custom types.

The list of types that Platypus knows about varies somewhat from 
platform to platform, L<FFI::Platypus::Type> includes a list of the core 
types that you can always count on having access to.

It can also be called as a class method, in which case, no user defined 
or custom types will be included in the list.

=head2 type_meta

 my $meta = $ffi->type_meta($type_name);
 my $meta = FFI::Platypus->type_meta($type_name);

Returns a hash reference with the meta information for the given type.

It can also be called as a class method, in which case, you won't be 
able to get meta data on user defined types.

Examples:

 my $meta = $ffi->type_meta('int');        # standard int type
 my $meta = $ffi->type_meta('int[64]');    # array of 64 ints
 $ffi->type('int[128]' => 'myintarray');
 my $meta = $ffi->type_meta('myintarray'); # array of 128 ints

=head2 function

 my $function = $ffi->function($name => \@argument_types => $return_type);
 my $function = $ffi->function($address => \@argument_types => $return_type);

Returns an object that is similar to a code reference in that it can be 
called like one.

Caveat: many situations require a real code reference, so at the price 
of a performance penalty you can get one like this:

 my $function = $ffi->function(...);
 my $coderef = sub { $function->(@_) };

It may be better, and faster to create a real Perl function using the 
L<attach|FFI::Platypus#attach> method.

In addition to looking up a function by name you can provide the address 
of the symbol yourself:

 my $address = $ffi->find_symbol('my_functon');
 my $function = $ffi->function($address => ...);

Under the covers, L<function|FFI::Platypus#function> uses 
L<find_symbol|FFI::Platypus#find_symbol> when you provide it with a 
name, but it is useful to keep this in mind as there are alternative 
ways of obtaining a functions address.  Example: a C function could 
return the address of another C function that you might want to call, or 
modules such as L<FFI::TinyCC> produce machine code at runtime that you 
can call from Platypus.

Examples:

 my $function = $ffi->function('my_function_name', ['int', 'string'] => 'string');
 my $return_string = $function->(1, "hi there");

=head2 attach

 $ffi->attach($name => \@argument_types => $return_type);
 $ffi->attach([$c_name => $perl_name] => \@argument_types => $return_type);
 $ffi->attach([$address => $perl_name] => \@argument_types => $return_type);

Find and attach a C function as a real live Perl xsub.  The advantage of 
attaching a function over using the L<function|FFI::Platypus#function> 
method is that it is much much much faster since no object resolution 
needs to be done.  The disadvantage is that it locks the function and 
the L<FFI::Platypus> instance into memory permanently, since there is no 
way to deallocate an xsub.

If just one I<$name> is given, then the function will be attached in 
Perl with the same name as it has in C.  The second form allows you to 
give the Perl function a different name.  You can also provide an 
address (the third form), just like with the 
L<function|FFI::Platypus#function> method.

Examples:

 $ffi->attach('my_functon_name', ['int', 'string'] => 'string');
 $ffi->attach(['my_c_functon_name' => 'my_perl_function_name'], ['int', 'string'] => 'string');
 my $string1 = my_function_name($int);
 my $string2 = my_perl_function_name($int);

=head2 closure

 my $closure = $ffi->closure($coderef);

Prepares a code reference so that it can be used as a FFI closure (a 
Perl subroutine that can be called from C code).  For details on 
closures, see L<FFI::Platypus::Type#Closures>.

=head2 cast

 my $converted_value = $ffi->cast($original_type, $converted_type, $original_value);

The C<cast> function converts an existing I<$original_value> of type 
I<$original_type> into one of type I<$converted_type>.  Not all types 
are supported, so care must be taken.  For example, to get the address 
of a string, you can do this:

 my $address = $ffi->cast('string' => 'opaque', $string_value);

Something that won't work is trying to cast an array to anything:

 my $address = $ffi->cast('int[10]' => 'opaque', \@list);  # WRONG

=head2 attach_cast

 $ffi->attach_cast("cast_name", $original_type, $converted_type);
 my $converted_value = cast_name($original_value);

This function attaches a cast as a permanent xsub.  This will make it 
faster and may be useful if you are calling a particular cast a lot.

=head2 sizeof

 my $size = $ffi->sizeof($type);

Returns the total size of the given type in bytes.  For example to get 
the size of an integer:

 my $intsize = $ffi->sizeof('int'); # usually 4 or 8 depending on platform

You can also get the size of arrays

 my $intarraysize = $ffi->sizeof('int[64]'); # usually 4*64 or 8*64

Keep in mind that "pointer" types will always be the pointer / word size 
for the platform that you are using.  This includes strings, opaque and 
pointers to other types.

This function is not very fast, so you might want to save this value as 
a constant, particularly if you need the size in a loop with many 
iterations.

=head2 find_symbol

 my $address = $ffi->find_symbol($name);

Return the address of the given symbol (usually function).

=head2 package

 $ffi->package($package, $file); # usually __PACKAGE__ and __FILE__ can be used
 $ffi->package;                  # autodetect

If you have used L<Module::Build::FFI> to bundle C code with your
distribution, you can use this method to tell the L<FFI::Platypus> instance
to look for symbols that came with the dynamic library that was built
when your distribution was installed.

=head1 EXAMPLES

Here are some examples.  Some of them use the L<FFI::Platypus::Declare> 
interface, but the principles apply to the OO interface.  These examples 
are provided in full with the Platypus distribution in the "examples" 
directory.  There are also some more examples in L<FFI::Platypus::Type> 
that are related to types.

=head2 Integer conversions

 use FFI::Platypus::Declare qw( int string );
 
 lib undef;
 attach puts => [string] => int;
 attach atoi => [string] => int;
 
 puts(atoi('56'));

B<Discussion>: C<puts> and C<atoi> should be part of the standard C 
library on all platforms.  C<puts> prints a string to standard output, 
and C<atoi> converts a string to integer.  Specifying C<undef> as a 
library tells Platypus to search the current process for symbols, which 
includes the standard c library.

=head2 libnotify

 use FFI::CheckLib;
 use FFI::Platypus::Declare qw( void string opaque );
 
 # NOTE: I ported this from the like named eg/notify.pl that came with FFI::Raw
 # and it seems to work most of the time, but also seems to SIGSEGV sometimes.
 # I saw the same behavior in the FFI::Raw version, and am not really familiar
 # with the libnotify API to say what is the cause.  Patches welcome to fix it.
 
 lib find_lib_or_exit lib => 'notify';
 
 attach notify_init   => [string] => void;
 attach notify_uninit => []       => void;
 attach [notify_notification_new    => 'notify_new']    => [string,string,string]           => opaque;
 attach [notify_notification_update => 'notify_update'] => [opaque, string, string, string] => void;
 attach [notify_notification_show   => 'notify_show']   => [opaque, opaque]                 => void;
 
 notify_init('FFI::Platypus');
 my $n = notify_new('','','');
 notify_update($n, 'FFI::Platypus', 'It works!!!', 'media-playback-start');
 notify_show($n, undef);
 notify_uninit();

B<Discussion>: libnotify is a desktop GUI notification library for the 
GNOME Desktop environment. This script sends a notification event that 
should show up as a balloon, for me it did so in the upper right hand 
corner of my screen.

The most portable way to find the correct name and location of a dynamic 
library is via the L<FFI::CheckLib#find_lib> family of functions.  If 
you are putting together a CPAN distribution, you should also consider 
using L<FFI::CheckLib#check_lib_or_exit> function in your C<Build.PL> or 
C<Makefile.PL> file (If you are using L<Dist::Zilla>, check out the 
L<Dist::Zilla::Plugin::FFI::CheckLib> plugin). This will provide a user 
friendly diagnostic letting the user know that the required library is 
missing, and reduce the number of bogus CPAN testers results that you 
will get.

=head2 Allocating and freeing memory

 use FFI::Platypus::Declare;
 use FFI::Platypus::Memory qw( malloc free memcpy );
 
 my $buffer = malloc 12;
 
 memcpy $buffer, cast('string' => 'opaque', "hello there"), length "hello there\0";
 
 print cast('opaque' => 'string', $buffer), "\n";
 
 free $buffer;

B<Discussion>: C<malloc> and C<free> are standard memory allocation 
functions available from the standard c library and.  Interfaces to 
these and other memory related functions are provided by the 
L<FFI::Platypus::Memory> module.

=head2 libuuid

 use FFI::CheckLib;
 use FFI::Platypus::Declare qw( void opaque string );
 use FFI::Platypus::Memory qw( malloc free );
 
 lib find_lib_or_exit lib => 'uuid';
 
 attach uuid_generate => [opaque] => void;
 attach uuid_unparse  => [opaque,opaque] => void;
 
 my $uuid = malloc sizeof 'char[16]';  # uuid_t
 uuid_generate($uuid);
 
 my $string_opaque = malloc 37;       # 36 bytes to store a UUID string
 uuid_unparse($uuid, $string_opaque);
 
 print cast( opaque => string, $string_opaque), "\n";

B<Discussion>: libuuid is a library used to generate unique identifiers 
(UUID) for objects that may be accessible beyond the local system.  The 
library is or was part of the Linux e2fsprogs package.

Knowing the size of objects is sometimes important.  In this example, we 
use the L<sizeof|FFI::Platypus#sizeof> function to get the size of 16 
characters (in this case it is simply 16 bytes).  We also know that the 
strings "deparsed" by C<uuid_unparse> are exactly 37 bytes.

=head2 puts and getpid

 use FFI::Platypus::Declare qw( string int );
 
 lib undef;
 attach puts => [string] => int;
 attach getpid => [] => int;
 
 puts(getpid());

B<Discussion>: C<puts> is part of standard C library on all platforms.  
C<getpid> is available on Unix type platforms.

=head2 Math library

 use FFI::Platypus::Declare qw( string int double );
 use FFI::CheckLib;
 
 lib undef;
 attach puts => [string] => int;
 attach fdim => [double,double] => double;
 
 puts(fdim(7.0, 2.0));
 
 attach cos => [double] => double;
 
 puts(cos(2.0));
 
 attach fmax => [double, double] => double;
 
 puts(fmax(2.0,3.0));

B<Discussion>: On UNIX the standard c library math functions are 
frequently provided in a separate library C<libm>, so you could search 
for those symbols in "libm.so", but that won't work on non-UNIX 
platforms like Microsoft Windows.  Fortunately Perl uses the math 
library so these symbols are already in the current process so you can 
use C<undef> as the library to find them.

=head2 Strings

 use FFI::Platypus::Declare qw( int string );
 
 lib undef;
 attach puts => [string] => int;
 attach strlen => [string] => int;
 
 puts(strlen('somestring'));
 
 attach strstr => [string,string] => string;
 
 puts(strstr('somestring', 'string'));
 
 #attach puts => [string] => int;
 
 puts(puts("lol"));
 
 attach strerror => [int] => string;
 
 puts(strerror(2));

B<Discussion>: Strings are not a native type to C<libffi> but the are 
handled seamlessly by Platypus.

=head2 Attach function from pointer

 use FFI::TinyCC;
 use FFI::Platypus;
 
 my $ffi = FFI::Platypus->new;
 my $tcc = FFI::TinyCC->new;
 
 $tcc->compile_string(q{
   int
   add(int a, int b)
   {
     return a+b;
   }
 });
 
 my $address = $tcc->get_symbol('add');
 
 $ffi->attach( [ $address => 'add' ] => ['int','int'] => 'int' );
 
 print add(1,2), "\n";

B<Discussion>: Sometimes you will have a pointer to a function from a 
source other than Platypus that you want to call.  You can use that 
address instead of a function name for either of the 
L<function|FFI::Platypus#function> or L<attach|FFI::Platypus#attach> 
methods.  In this example we use L<FFI::TinyCC> to compile a short piece 
of C code and to give us the address of one of its functions, which we 
then use to create a perl xsub to call it.

L<FFI::TinyCC> embeds the Tiny C Compiler (tcc) to provide a 
just-in-time (JIT) compilation service for FFI.

=head2 libzmq

 use constant ZMQ_IO_THREADS  => 1;
 use constant ZMQ_MAX_SOCKETS => 2;
 use constant ZMQ_REQ => 3;
 use constant ZMQ_REP => 4;
 use FFI::CheckLib qw( find_lib_or_exit );
 use FFI::Platypus;
 use FFI::Platypus::Memory qw( malloc );
 use FFI::Platypus::Buffer qw( scalar_to_buffer buffer_to_scalar );
 
 my $endpoint = "ipc://zmq-ffi-$$";
 #my $endpoint = "tcp://localhost:3000";
 my $ffi = FFI::Platypus->new;
 
 $ffi->lib(undef); # for puts
 $ffi->attach(puts => ['string'] => 'int');
 
 $ffi->lib(find_lib_or_exit lib => 'zmq');
 $ffi->attach(zmq_version => ['int*', 'int*', 'int*'] => 'void');
 
 my($major,$minor,$patch);
 zmq_version(\$major, \$minor, \$patch);
 puts("libzmq version $major.$minor.$patch");
 die "this script only works with libzmq 3 or better" unless $major >= 3;
 
 $ffi->type('opaque'       => 'zmq_context');
 $ffi->type('opaque'       => 'zmq_socket');
 $ffi->type('opaque'       => 'zmq_msg_t');
 $ffi->attach(zmq_ctx_new  => [] => 'zmq_context');
 $ffi->attach(zmq_ctx_set  => ['zmq_context', 'int', 'int'] => 'int');
 $ffi->attach(zmq_socket   => ['zmq_context', 'int'] => 'zmq_socket');
 $ffi->attach(zmq_connect  => ['opaque', 'string'] => 'int');
 $ffi->attach(zmq_bind     => ['zmq_socket', 'string'] => 'int');
 $ffi->attach(zmq_send     => ['zmq_socket', 'opaque', 'size_t', 'int'] => 'int');
 $ffi->attach(zmq_msg_init => ['zmq_msg_t'] => 'int');
 $ffi->attach(zmq_msg_recv => ['zmq_msg_t', 'zmq_socket', 'int'] => 'int');
 $ffi->attach(zmq_msg_data => ['zmq_msg_t'] => 'opaque');
 $ffi->attach(zmq_errno    => [] => 'int');
 $ffi->attach(zmq_strerror => ['int'] => 'string');
 
 my $context = zmq_ctx_new();
 zmq_ctx_set($context, ZMQ_IO_THREADS, 1);
 
 my $socket1 = zmq_socket($context, ZMQ_REQ);
 zmq_connect($socket1, $endpoint);
 
 my $socket2 = zmq_socket($context, ZMQ_REP);
 zmq_bind($socket2, $endpoint);
 
 do { # send
   our $sent_message = "hello there";
   my($pointer, $size) = scalar_to_buffer $sent_message;
   my $r = zmq_send($socket1, $pointer, $size, 0);
   die zmq_strerror(zmq_errno()) if $r == -1;
 };
 
 do { # recv
   my $msg_ptr  = malloc 100;
   zmq_msg_init($msg_ptr);
   my $size     = zmq_msg_recv($msg_ptr, $socket2, 0);
   die zmq_strerror(zmq_errno()) if $size == -1;
   my $data_ptr = zmq_msg_data($msg_ptr);
   my $recv_message = buffer_to_scalar $data_ptr, $size;
   print "recv_message = $recv_message\n";
 };

B<Discussion>: ØMQ is a high-performance asynchronous messaging library.  
There are a few things to note here.

Firstly, sometimes there may be multiple versions of a library in the 
wild and you may need to verify that the library on a system meets your 
needs (alternatively you could support multiple versions and configure 
your bindings dynamically).  Here we use C<zmq_version> to ask libzmq 
which version it is.

C<zmq_version> returns the version number via three integer pointer 
arguments, so we use the pointer to integer type: C<int *>.  In order to 
pass pointer types, we pass a reference. In this case it is a reference 
to an undefined value, because zmq_version will write into the pointers 
the output values, but you can also pass in references to integers, 
floating point values and opaque pointer types.  When the function 
returns the C<$major> variable (and the others) has been updated and we 
can use it to verify that it supports the API that we require.

Notice that we define three aliases for the C<opaque> type: 
C<zmq_context>, C<zmq_socket> and C<zmq_msg_t>.  While this isn't 
strictly necessary, since Platypus and C treat all three of these types 
the same, it is useful form of documentation that helps describe the 
functionality of the interface.

Finally we attach the necessary functions, send and receive a message. 
If you are interested, there is a fully fleshed out ØMQ Perl interface 
implemented using FFI called L<ZMQ::FFI>.

=head2 libarchive

 use FFI::Platypus      ();
 use FFI::Platypus::API ();
 use FFI::CheckLib      ();
 
 # This example uses FreeBSD's libarchive to list the contents of any
 # archive format that it suppors.  We've also filled out a part of
 # the ArchiveWrite class that could be used for writing archive formats
 # supported by libarchive
 
 my $ffi = FFI::Platypus->new;
 $ffi->lib(FFI::CheckLib::find_lib_or_exit lib => 'archive');
 
 $ffi->custom_type(archive => {
   native_type    => 'opaque',
   perl_to_native => sub { ${$_[0]} },
   native_to_perl => sub {
     # this works because archive_read_new ignores any arguments
     # and we pass in the class name which we can get here.
     my $class = FFI::Platypus::API::arguments_get_string(0);
     bless \$_[0], $class;
   },
 });
 
 $ffi->custom_type(archive_entry => {
   native_type => 'opaque',
   perl_to_native => sub { ${$_[0]} },
   native_to_perl => sub {
     # works likewise for archive_entry objects
     my $class = FFI::Platypus::API::arguments_get_string(0);
     bless \$_[0], $class,
   },
 });
 
 package Archive;
 
 # base class is "abstract" having no constructor or destructor
 
 $ffi->attach( [ archive_error_string => 'error_string' ] => ['archive'] => 'string' );
 
 package ArchiveRead;
 
 our @ISA = qw( Archive );
 
 $ffi->attach( [ archive_read_new => 'new' ] => ['string'] => 'archive' );
 $ffi->attach( [ archive_read_free => 'DESTROY' ] => ['archive'] => 'void' );
 $ffi->attach( [ archive_read_support_filter_all => 'support_filter_all' ] => ['archive'] => 'int' );
 $ffi->attach( [ archive_read_support_format_all => 'support_format_all' ] => ['archive'] => 'int' );
 $ffi->attach( [ archive_read_open_filename => 'open_filename' ] => ['archive','string','size_t'] => 'int' );
 $ffi->attach( [ archive_read_next_header2 => 'next_header2' ] => ['archive', 'archive_entry' ] => 'int' );
 $ffi->attach( [ archive_read_data_skip => 'data_skip' ] => ['archive'] => 'int' );
 # ... define additional read methods
 
 package ArchiveWrite;
 
 our @ISA = qw( Archive );
 
 $ffi->attach( [ archive_write_new => 'new' ] => ['string'] => 'archive' );
 $ffi->attach( [ archive_write_free => 'DESTROY' ] => ['archive'] => 'void' );
 # ... define additional write methods
 
 package ArchiveEntry;
 
 $ffi->attach( [ archive_entry_new => 'new' ] => ['string'] => 'archive_entry' );
 $ffi->attach( [ archive_entry_free => 'DESTROY' ] => ['archive_entry'] => 'void' );
 $ffi->attach( [ archive_entry_pathname => 'pathname' ] => ['archive_entry'] => 'string' );
 # ... define additional entry methods
 
 package main;
 
 use constant ARCHIVE_OK => 0;
 
 # this is a Perl version of the C code here:
 # https://github.com/libarchive/libarchive/wiki/Examples#List_contents_of_Archive_stored_in_File
 
 my $archive_filename = shift @ARGV;
 die "usage: $0 archive.tar" unless defined $archive_filename;
 
 my $archive = ArchiveRead->new;
 $archive->support_filter_all;
 $archive->support_format_all;
 
 my $r = $archive->open_filename($archive_filename, 1024);
 die "error opening $archive_filename: ", $archive->error_string
   unless $r == ARCHIVE_OK;
 
 my $entry = ArchiveEntry->new;
 
 while($archive->next_header2($entry) == ARCHIVE_OK)
 {
   print $entry->pathname, "\n";
   $archive->data_skip;
 }

B<Discussion>: libarchive is the implementation of C<tar> for FreeBSD 
provided as a library and available on a number of platforms.

One interesting thing about libarchive is that it provides a kind of 
object oriented interface via opaque pointers.  This example creates an 
abstract class C<Archive>, and concrete classes C<ArchiveWrite>, 
C<ArchiveRead> and C<ArchiveEntry>.  The concrete classes can even be 
inherited from and extended just like any Perl classes because of the 
way the custom types are implemented.  For more details on custom types 
see L<FFI::Platypus::Type> and L<FFI::Platypus::API>.

=head1 SUPPORT

If something does not work the way you think it should, or if you have a 
feature request, please open an issue on this project's GitHub Issue 
tracker:

L<https://github.com/plicease/FFI-Platypus/issues>

=head1 CONTRIBUTING

If you have implemented a new feature or fixed a bug then you may make a 
pull request on this project's GitHub repository:

L<https://github.com/plicease/FFI-Platypus/pulls>

This project is developed using L<Dist::Zilla>.  The project's git 
repository also comes with C<Build.PL> and C<cpanfile> files necessary 
for building, testing (and even installing if necessary) without 
L<Dist::Zilla>.  Please keep in mind though that these files are 
generated so if changes need to be made to those files they should be 
done through the project's C<dist.ini> file.  If you do use 
L<Dist::Zilla> and already have the necessary plugins installed, then I 
encourage you to run C<dzil test> before making any pull requests.  This 
is not a requirement, however, I am happy to integrate especially 
smaller patches that need tweaking to fit the project standards.  I may 
push back and ask you to write a test case or alter the formatting of a 
patch depending on the amount of time I have and the amount of code that 
your patch touches.

This project's GitHub issue tracker listed above is not Write-Only.  If 
you want to contribute then feel free to browse through the existing 
issues and see if there is something you feel you might be good at and 
take a whack at the problem.  I frequently open issues myself that I 
hope will be accomplished by someone in the future but do not have time 
to immediately implement myself.

Another good area to help out in is documentation.  I try to make sure 
that there is good document coverage, that is there should be 
documentation describing all the public features and warnings about 
common pitfalls, but an outsider's or alternate view point on such 
things would be welcome; if you see something confusing or lacks 
sufficient detail I encourage documentation only pull requests to 
improve things.

The Platypus distribution comes with a test library named C<libtest> 
that is normally automatically built by C<./Build test>.  If you prefer 
to use C<prove> or run tests directly, you can use the C<./Build 
libtest> command to build it.  Example:

 % perl Build.PL
 % ./Build
 % ./Build libtest
 % prove -bv t
 # or an individual test
 % perl -Mblib t/ffi_platypus_memory.t

The build process also respects these environment variables:

=over 4

=item FFI_PLATYPUS_DEBUG

Build the XS code portion of Platypus with -g3 instead of what ever 
optimizing flags that your Perl normally uses.  This is useful if you 
need to debug the C or XS code that comes with Platypus, but do not have 
a debugging Perl.

 % env FFI_PLATYPUS_DEBUG=1 perl Build.PL
 
 
 DEBUG:
   + $Config{lddlflags} = -shared -O2 -L/usr/local/lib -fstack-protector
   - $Config{lddlflags} = -shared -g3 -L/usr/local/lib -fstack-protector
   + $Config{optimize} = -O2
   - $Config{optimize} = -g3
 
 
 Created MYMETA.yml and MYMETA.json
 Creating new 'Build' script for 'FFI-Platypus' version '0.10'

=item FFI_PLATYPUS_DEBUG_FAKE32

When building Platypus on 32 bit Perls, it will use the L<Math::Int64> C 
API and make L<Math::Int64> a prerequisite.  Setting this environment 
variable will force Platypus to build with both of those options on a 64 
bit Perl as well.

 % env FFI_PLATYPUS_DEBUG_FAKE32=1 perl Build.PL
 
 
 DEBUG_FAKE32:
   + making Math::Int64 a prerequsite (not normally done on 64 bit Perls)
   + using Math::Int64's C API to manipulate 64 bit values (not normally done on 64 bit Perls)
 
 Created MYMETA.yml and MYMETA.json
 Creating new 'Build' script for 'FFI-Platypus' version '0.10'

=item FFI_PLATYPUS_NO_ALLOCA

Platypus uses the non-standard and somewhat controversial C function 
C<alloca> by default on platforms that support it.  I believe that 
Platypus uses it responsibly to allocate small amounts of memory for 
argument type parameters, and does not use it to allocate large 
structures like arrays or buffers.  If you prefer not to use C<alloca> 
despite these precautions, then you can turn its use off by setting this 
environment variable when you run C<Build.PL>:

 % env FFI_PLATYPUS_NO_ALLOCA=1 perl Build.PL 
 
 
 NO_ALLOCA:
   + alloca() will not be used, even if your platform supports it.
 
 
  Created MYMETA.yml and MYMETA.json
  Creating new 'Build' script for 'FFI-Platypus' version '0.10'

=back

=head1 SEE ALSO

=over 4

=item L<FFI::Platypus::Declare>

Declarative interface to Platypus.

=item L<FFI::Platypus::Type>

Type definitions for Platypus.

=item L<FFI::Platypus::API>

The custom types API for Platypus.

=item L<FFI::Platypus::Memory>

Memory functions for FFI.

=item L<FFI::CheckLib>

Find dynamic libraries in a portable way.

=item L<Module::Build::FFI>

Bundle C code with your FFI extension.

=item L<FFI::TinyCC>

JIT compiler for FFI.

=item L<Convert::Binary::C>

An interface for interacting with C C<struct> types.  Unfortunately it 
appears to be unmaintained, and has a failing pod test, so I cannot 
recommend it for use by CPAN modules.

=item L<pack|perlfunc#pack> and L<unpack|perlfunc#unpack>

Native to Perl functions that can be used to decode C C<struct> types.

=item L<FFI::Raw>

Alternate interface to libffi with fewer features.  It notably lacks the 
ability to create real xsubs, which may make L<FFI::Platypus> much 
faster.  Also lacking are pointers to native types, arrays and custom 
types.  In its favor, it has been around for longer that Platypus, and 
has been battle tested to some success.

=item L<Win32::API>

Microsoft Windows specific FFI style interface.

=item L<Ctypes|https://gitorious.org/perl-ctypes>

Ctypes was intended as a FFI style interface for Perl, but was never 
part of CPAN, and at least the last time I tried it did not work with 
recent versions of Perl.

=item L<FFI>

Foreign function interface based on (nomenclature is everything) FSF's 
C<ffcall>. It hasn't worked for quite some time, and C<ffcall> is no 
longer supported or distributed.

=item L<C::DynaLib>

Another FFI for Perl that doesn't appear to have worked for a long time.

=back

=cut

=head1 AUTHOR

Graham Ollis <plicease@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Graham Ollis.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
