package Bencher::Scenario::StringFunctions::Indent;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2021-07-31'; # DATE
our $DIST = 'Bencher-Scenarios-StringFunctions'; # DIST
our $VERSION = '0.005'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => "Benchmark string indenting (adding whitespace to lines of text)",
    participants => [
        {fcall_template=>'String::Nudge::nudge(<num_spaces>, <str>)'},
        {fcall_template=>'String::Indent::indent(<indent>, <str>)'},
        {fcall_template=>'String::Indent::Join::indent(<indent>, <str>)'},
        # TODO: Indent::String
        # TODO: Indent::Utils
        # TODO: Text::Indent
    ],
    datasets => [
        {name=>'empty'        , args=>{num_spaces=>4, indent=>'    ', str=>''}},
        {name=>'1line'        , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..1)}},
        {name=>'10line'       , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..10)}},
        {name=>'100line'      , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..100)}},
        {name=>'1000line'     , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..1000)}},
    ],
};

1;
# ABSTRACT: Benchmark string indenting (adding whitespace to lines of text)

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::StringFunctions::Indent - Benchmark string indenting (adding whitespace to lines of text)

=head1 VERSION

This document describes version 0.005 of Bencher::Scenario::StringFunctions::Indent (from Perl distribution Bencher-Scenarios-StringFunctions), released on 2021-07-31.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m StringFunctions::Indent

To run module startup overhead benchmark:

 % bencher --module-startup -m StringFunctions::Indent

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<String::Nudge> 1.0002

L<String::Indent> 0.03

L<String::Indent::Join>

=head1 BENCHMARK PARTICIPANTS

=over

=item * String::Nudge::nudge (perl_code)

Function call template:

 String::Nudge::nudge(<num_spaces>, <str>)



=item * String::Indent::indent (perl_code)

Function call template:

 String::Indent::indent(<indent>, <str>)



=item * String::Indent::Join::indent (perl_code)

Function call template:

 String::Indent::Join::indent(<indent>, <str>)



=back

=head1 BENCHMARK DATASETS

=over

=item * empty

=item * 1line

=item * 10line

=item * 100line

=item * 1000line

=back

=head1 BENCHMARK SAMPLE RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 5.3.0-68-generic >>.

Benchmark command (default options):

 % bencher -m StringFunctions::Indent

Result formatted as table (split, part 1 of 5):

 #table1#
 {dataset=>"1000line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |  3000     |  333      |                 0.00% |                61.96% | 4.9e-08 |      24 |
 | String::Nudge::nudge         |  3990     |  251      |                33.01% |                21.76% | 2.1e-07 |      20 |
 | String::Indent::Join::indent |  4859.002 |  205.8036 |                61.96% |                 0.00% | 5.6e-12 |      21 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

               Rate  SI:i  SN:n  SIJ:i 
  SI:i       3000/s    --  -24%   -38% 
  SN:n       3990/s   32%    --   -18% 
  SIJ:i  4859.002/s   61%   21%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 2 of 5):

 #table2#
 {dataset=>"100line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |   28400   |   35.3    |                 0.00% |                87.83% | 1.2e-08 |      26 |
 | String::Nudge::nudge         |   37437.5 |   26.7112 |                32.05% |                42.24% | 5.8e-12 |      20 |
 | String::Indent::Join::indent |   53300   |   18.8    |                87.83% |                 0.00% | 6.7e-09 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  SI:i  SN:n  SIJ:i 
  SI:i     28400/s    --  -24%   -46% 
  SN:n   37437.5/s   32%    --   -29% 
  SIJ:i    53300/s   87%   42%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 3 of 5):

 #table3#
 {dataset=>"10line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |    218140 |    4.5843 |                 0.00% |               110.87% | 1.7e-11 |      20 |
 | String::Nudge::nudge         |    279260 |    3.5808 |                28.02% |                64.71% | 5.8e-12 |      20 |
 | String::Indent::Join::indent |    459980 |    2.174  |               110.87% |                 0.00% | 5.8e-12 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

             Rate  SI:i  SN:n  SIJ:i 
  SI:i   218140/s    --  -21%   -52% 
  SN:n   279260/s   28%    --   -39% 
  SIJ:i  459980/s  110%   64%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 4 of 5):

 #table4#
 {dataset=>"1line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |    822660 |    1215.6 |                 0.00% |               179.86% | 5.8e-12 |      20 |
 | String::Nudge::nudge         |   1040000 |     962   |                26.42% |               121.37% | 3.8e-10 |      24 |
 | String::Indent::Join::indent |   2302000 |     434.3 |               179.86% |                 0.00% | 5.6e-12 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  SI:i  SN:n  SIJ:i 
  SI:i    822660/s    --  -20%   -64% 
  SN:n   1040000/s   26%    --   -54% 
  SIJ:i  2302000/s  179%  121%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 5 of 5):

 #table5#
 {dataset=>"empty"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |    867600 |    1153   |                 0.00% |               591.98% | 1.7e-11 |      20 |
 | String::Nudge::nudge         |   1200000 |     850   |                35.15% |               412.01% | 1.2e-09 |      21 |
 | String::Indent::Join::indent |   6004000 |     166.6 |               591.98% |                 0.00% | 5.7e-12 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  SI:i  SN:n  SIJ:i 
  SI:i    867600/s    --  -26%   -85% 
  SN:n   1200000/s   35%    --   -80% 
  SIJ:i  6004000/s  592%  410%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Benchmark module startup overhead (C<< bencher -m StringFunctions::Indent --module-startup >>):

Result formatted as table:

 #table6#
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | String::Nudge        |         8 |                 2 |                 0.00% |                42.10% | 0.00013 |      20 |
 | String::Indent::Join |         8 |                 2 |                 4.35% |                36.19% | 0.00017 |      20 |
 | String::Indent       |         8 |                 2 |                 7.06% |                32.74% | 0.00013 |      21 |
 | perl -e1 (baseline)  |         6 |                 0 |                42.10% |                 0.00% | 0.00012 |      20 |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                Rate  S:N  SI:J  S:I  :perl -e1 ( 
  S:N          0.1/s   --    0%   0%         -25% 
  SI:J         0.1/s   0%    --   0%         -25% 
  S:I          0.1/s   0%    0%   --         -25% 
  :perl -e1 (  0.2/s  33%   33%  33%           -- 
 
 Legends:
   :perl -e1 (: mod_overhead_time=0 participant=perl -e1 (baseline)
   S:I: mod_overhead_time=2 participant=String::Indent
   S:N: mod_overhead_time=2 participant=String::Nudge
   SI:J: mod_overhead_time=2 participant=String::Indent::Join

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 BENCHMARK NOTES

Joining is faster than regex substitution for the datasets tested (0-1000
lines of short text).

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-StringFunctions>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-StringFunctions>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-StringFunctions>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2018 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
