package Bencher::Scenario::StringFunctions::CommonPrefix;

our $DATE = '2021-07-31'; # DATE
our $VERSION = '0.005'; # VERSION

use strict;
use warnings;

our $scenario = {
    summary => "Benchmark calculating common prefix",
    participants => [
        {fcall_template=>'String::CommonPrefix::common_prefix(@{<strings>})'},
    ],
    datasets => [
        {name=>'elems0'          , args=>{strings=>[]}},
        {name=>'elems1'          , args=>{strings=>['x']}},
        {name=>'elems10prefix0'  , args=>{strings=>[map{sprintf "%02d", $_} 1..10]}},
        {name=>'elems10prefix1'  , args=>{strings=>[map{sprintf "%02d", $_} 0..9]}},
        {name=>'elems100prefix0' , args=>{strings=>[map{sprintf "%03d", $_} 1..100]}},
        {name=>'elems100prefix1' , args=>{strings=>[map{sprintf "%03d", $_} 0..99]}},
        {name=>'elems1000prefix0', args=>{strings=>[map{sprintf "%04d", $_} 1..1000]}},
        {name=>'elems1000prefix1', args=>{strings=>[map{sprintf "%04d", $_} 0..999]}},
    ],
};

1;
# ABSTRACT: Benchmark calculating common prefix

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::StringFunctions::CommonPrefix - Benchmark calculating common prefix

=head1 VERSION

This document describes version 0.005 of Bencher::Scenario::StringFunctions::CommonPrefix (from Perl distribution Bencher-Scenarios-StringFunctions), released on 2021-07-31.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m StringFunctions::CommonPrefix

To run module startup overhead benchmark:

 % bencher --module-startup -m StringFunctions::CommonPrefix

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<String::CommonPrefix> 0.01

=head1 BENCHMARK PARTICIPANTS

=over

=item * String::CommonPrefix::common_prefix (perl_code)

Function call template:

 String::CommonPrefix::common_prefix(@{<strings>})



=back

=head1 BENCHMARK DATASETS

=over

=item * elems0

=item * elems1

=item * elems10prefix0

=item * elems10prefix1

=item * elems100prefix0

=item * elems100prefix1

=item * elems1000prefix0

=item * elems1000prefix1

=back

=head1 BENCHMARK SAMPLE RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 5.3.0-68-generic >>.

Benchmark command (default options):

 % bencher -m StringFunctions::CommonPrefix

Result formatted as table (split, part 1 of 8):

 #table1#
 {dataset=>"elems0"}
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset | ds_tags | p_tags | perl | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems0  |         |        | perl |   8690000 |       115 |                 0.00% |                 0.00% | 5.4e-11 |      21 |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

          Rate     
     8690000/s  -- 
 
 Legends:
   : dataset=elems0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 2 of 8):

 #table2#
 {dataset=>"elems1"}
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset | ds_tags | p_tags | perl | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems1  |         |        | perl |   2470000 |       405 |                 0.00% |                 0.00% | 1.9e-10 |      24 |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

          Rate     
     2470000/s  -- 
 
 Legends:
   : dataset=elems1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAAI1QTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlADVlQDWmADalADUjQDKhgDAAAAAPQBYJAA0AAAAEQAYFgAfBgAIAAAAAAAAlADURQBj////hSAPfgAAACt0Uk5TABFEMyJm3bvumcx3iKpVcM7Vx9I/+vbs+fR1vtpOdSDf6/bh/eDx9PbvW5DuQQIAAAABYktHRACIBR1IAAAACXBIWXMAAABIAAAASABGyWs+AAAAB3RJTUUH5QcfBzgrjmvodgAAE/NJREFUeNrt3Q1z23Z2RnEQJME3SO1u2t2m2W5fdvuK9vt/vQKgpNgWhAsbf5LnKuc3syvH0SQ3njMckpb8VJUkSZIkSZIkSZIkSZIkSZKkB9jULz+oN1/87Hb36Luk77B/C7buXn7Q1W9/d9t0XbN99I3SYoe3eqeCPp6qzal59I3SUtvzZVft27Yegt6NH8eg67bdDj/VP/vYdT5EK4t9c6oPTdse9329x9Opa8egL8f21O2vT6v70B99pbRU/5Tj0AfbPvXhXqrq0tfb1eOD8v44fsLufHr0jdJi43Po7fP58PIcun947up9U/eGqjft8JgtZdEH3XaH0+E16OMQdHs8DLb9U+yDT6CVyaF+Pg5POQ7XV4Cb8RH6eXhjY3gC3fh0Q7kcnvd9vZvxKUdfb9sMzzo2/WvE4YfP3fDUo17/b5Hu5Kn5m3NzaE7HfX0+N81xOz6N3h/P5/6HbTd69I3SYpt6V9X1pqqH9+bqt9/13tRf/Qa4JEmSJEmSVNRzcz74BpQ+i02zqZ4uj75CKmTvFyPoM2mfmu7kUw7hfP3dx9evadxOl/ryxTTj322Hpxx+Ra9gvvnu4/ZQDS/3uu6pj3b8CprDr5+7G7+YZnfuxq8e62OuD9/7r5Nu6+vvPq6HfjfHS7U5970+neq6fot993wegz48bXZNW20P16glkK+/+3hzfOozvQx9Px/7dPfjpwz/f6mr/WEMevzky7mqTuens8+hxfL1dx8/tcNTjvZcXf8siW78LvzN8bnPe/Pyc1/83/aLr0//29+Nfi/9mJ/GgH76uwJRv3338f48PofeDo/Bp+E7hprxW+6fm91xbHfMeH8N+pvH5u7v/zD4oz72D48+AO3nMaDuH9c/Rr999/Gu2V1fFJ6Oh/Oh2+7avtpL/9Tj6Xj9jDHoS1dN/IkSBQ759HwFHVvf0Rfffdye+2ccTdunum2fX//4quF7O/fd+GT626ccZQ/5/Aw6tr6jL777uG6vQY/Pq/unH+Orwf414ObYjk+hrxnvhgfn/bd/KptBxww6trqjt+8+3l8fhIenHJvuudo0bTX+eSinc3Voq6en4W9eH5f7vxz/V/aQ3wCDjq3u6O27jw/XX+7xOfS+OxxP4988NM320myq3fG5eg16ezw3796vM+iYQcdu1NHu5bdTdpN/csRm4mcNOrZ/9AEJYDrCHKLUMB1hDlFqmI4whyg1TEeYQ5QapiPMIUoN0xHmEKWG6QhziFLDdIQ5RKlhOsIcotQwHWEOUWqYjjCHKDVMR5hDlBqmI8whSg3TEeYQpYbpCHOIUsN0hDlEqWE6whyi1DAdYQ5RapiOMIcoNUxHmEOUGqYjzCFKDdMR5hClhukIc4hSw3SEOUSpYTrCHKLUMB1hDlFqmI4whyg1TEeYQ5QapiPMIUrtkR19NV5v0CrhgR19PV5v0CrhgR19PV5v0CrhgR19PV5v0CqhZEcLRuw/Hq83aJVQrqO5EftXM+P1Bq0SynU0N2J/NTteb9AqoVhHMyP2i8brDVolFOtoZsR+0Xh990s7ePSvh9LajwEVfWD8aMR+0Xj96WXAU/oh2zGggkF/PGLveL3upeC7HDMj9o7X607KdTQzYu94ve6lWEdzI/aO1+teinU0M2LveL3u5sYdvY7Yf83xet0KpiPMIUoN0xHmEKWG6QhziFLDdIQ5RKlhOsIcotQwHWEOUWqYjjCHKDVMR5hDlBqmI8whSg3TEeYQpYbpCHOIUsN0hDlEqWE6whyi1DAdYQ5RapiOMIcoNUxHmEOUGqYjzCFKDdMR5hClhukIc4hSw3SEOUSpYTrCHKLUMB1hDlFqmI4whyg1TEeYQ5QapiPMIUoN0xHmEKWG6QhziFLDdIQ5RKlhOsIcotQcr9en4ni9PhXH6/WpOF6vT6VER5Oj9bvJT3W8Xre1vqOp0frh54bXe+PwyuHXz3W8Xje2vqOJ0fqqaavNsCr7NMzDvsXueL1ubnVHU6P1VVdfH6oP48Sb4/W6m9UdTY3WV8fL8Ojc/9P3bVtXy8br/3QYPPrXQ2m1Y0BFHhi/Ha2v6mNz7F/yVV3Tnrp9tWi83kdoFVCgo/ej9ZvzU/3cR75r+2ovx8rxet1LgXc53o/Wj0Pe25dkN/0TasfrdSfrO5oYrW/7V3xDyOOrwf41oOP1upfVHU2N1m+H9zHaY9/vdngvw/F63c3qjj4YrT83xyHq7tA0W8frdTc36mj3Mk+/m5ipd7xet4PpCHOIUsN0hDlEqWE6whyi1DAdYQ5RapiOMIcoNUxHmEOUGqYjzCFKDdMR5hClhukIc4hSw3SEOUSpYTrCHKLUMB1hDlFqmI4whyg1TEeYQ5QapiPMIUoN0xHmEKWG6QhziFLDdIQ5RKlhOsIcotQwHWEOUWqYjjCHKDVMR5hDlBqmI8whSg3TEeYQpYbpCHOIUsN0hDlEqWE6whyi1DAdYQ5RapiOMIcoNcfr9ak4Xq9PxfF6fSqO1+tTKdnRghF7x+t1W+U6mhuxf+V4vW6sXEdzI/ZXjtfr5op1NDNi73i97qZYRzMj9o7X6w4Kjte/+mjE3vF63UvBjj4esXe8XvdS8F2OmRF7x+t1J+U6mhmxd7xe91Kso7kRe8frdS/FOpoZsXe8Xndz444cr9d9YTrCHKLUMB1hDlFqmI4whyg1TEeYQ5QapiPMIUoN0xHmEKWG6QhziFLDdIQ5RKlhOsIcotQwHWEOUWqYjjCHKDVMR5hDlBqmI8whSg3TEeYQpYbpCHOIUsN0hDlEqWE6whyi1DAdYQ5RapiOMIcoNUxHmEOUGqYjzCFKDdMR5hClhukIc4hSw3SEOUSpYTrCHKLUMB1hDlFqmI4whyg1TEeYQ5SaW9/6VNz61qfi1rc+Fbe+9amU6Ghq4/sDbn3rttZ3NLHxXV/nKerrTsUXY5pufevG1nc0sfG9GeaDLsf+AfjU/+Atdre+dXOrO5rc+B6cn/t0x0Ust751N6s7mtz47l2GHbdu37Z1tWzr26BVQJGO3m18Dw/VQ+Jd0566/bKtb8frtUqx8fr3G9/DP31IfNf21V6Obn3rbgq8y/F+43t4gH57NrHpare+dS/rO5rY+H6d8h5fDfavAd361r2s7mhq47t/jjFWXg/vZ5zObn3rblZ3NLnxXR1f6u4OTePWt+7HrW99KpiOMIcoNUxHmEOUGqYjzCFKDdMR5hClhukIc4hSw3SEOUSpYTrCHKLUMB1hDlFqmI4whyg1TEeYQ5QapiPMIUoN0xHmEKWG6QhziFLDdIQ5RKlhOsIcotQwHWEOUWqYjjCHKDVMR5hDlBqmI8whSg3TEeYQpYbpCHOIUsN0hDlEqWE6whyi1DAdYQ5RapiOMIcoNUxHmEOU2sKOtttFn3b7Q6RZizraH7tD3dy2aYNWCUs62nb7+rB5+SOeH3mIFFnSUXsa9wTPdfyp38XxehW3KOj2JkE7Xq/ylnRUH7d90PvSTzkcr1d5izq6dM2xeflD+ctxvF7lLetot2+f48fnBSP2jtfrtpZ0tLtmuN/NftbciP3bP8nxet1W3NGuvgwT9PVzM/+icG7E/uWf5Hi9bi3uaH84N+Po7NPsk46ZEXvH63U3i35j5fpycP4px8yI/bLx+tPL3qE+9E9/vrV/fvR/4grbMaBlv/X9NDxCH8PaPhqxXzRe/0s7ePSvCtq//O+t/fnR/4kr7MeAlr0P3Z4P7fkUfNrHI/aO15dh0LGFv1P4fBp/Y2/O3Ii94/VFGHRsYdDDOxKH+accMyP2jteXYdCxJR3t+2cQfYTzb9vNjdg7Xl+GQccWdXQ4VO2xOc9+zsyIveP1hRh0bHFHz/tyX5vkeP2PMejYonc5Sn9Z0o8e8ltn0LElHV3O8efc5ZDfOoOOLero1N7+d/EMOmbQsUVPObrXF3wPPuS3zqBjmI4wh4AZdAzTEeYQMIOOYTrCHAJm0DFMR5hDwAw6hukIcwiYQccwHWEOATPoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXQM0xHmEDCDjmE6whwCZtAxTEeYQ8AMOobpCHMImEHHMB1hDgEz6BimI8whYAYdw3SEOQTMoGOYjjCHgBl0DNMR5hAwg45hOsIcAmbQMUxHmEPADDqG6QhzCJhBxzAdYQ4BM+gYpiPMIWAGHcN0hDkEzKBjmI4wh4AZdOyRHTle/50MOvbAjhyv/14GHXtgR47Xfy+Djj2wI8frv5dBx0p0NDVav5ncrne8fhWDjq3vaGK0vto8dd159zK8cvj1cx2vX8WgY+s7mhitr07nzWbYvXoa9o7fYne8fiWDjq3uaGq0fjP85W4YbxvXWRyvL8SgY6s7mhqtr7tqWw+pdvu2rReO17v1HTLoOcu3vkPfjtY/d4emOfaPw13Tnrr9svH60+13XLIz6DnbMaACQb8frR9/oj32zzr6ai9Hx+sLMehYgXc53o/Wj91uuuvD7fDR8foiDDq2vqOJ0frtNejt+Gqwfw3oeH0ZBh1b3dHkaH1zqapT0/e7Hd7LcLy+EIOOre5ocrR+mKcfXhS2w6vDreP1hRh07EYdvc7T7ybft3C8/scYdAzTEeYQMIOOYTrCHAJm0DFMR5hDwAw6hukIcwiYQccwHWEOATPoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXQM0xHmEDCDjmE6whwCZtAxTEeYQ8AMOobpCHMImEHHMB1hDgEz6BimI8whYAYdw3SEOQTMoGOYjjCHgBl0DNMR5hAwg45hOsIcAmbQMUxHmEPADDqG6QhzCJhBxzAdYQ4BM+gYpiPMIWAGHcN0hDkEzKBjmI4wh4AZdAzTEeYQMIOOYTrCHAJm0DFMR5hDwAw6hukIcwiYQccwHWEOATPomOP1iRh0zPH6RAw65nh9IgYdc7w+EYOOlexowYi94/VrGHSsXEdzI/avHK9fxaBj5TqaG7G/crx+JYOOFetoZsTe8fpCDDpWrKOZEftl4/V/Ogwe/euBZtBz2jGgog+MH43YLxqv9xE6ZNCxgh19PGLveH0ZBh0r+C7HzIi94/VFGHSsXEczI/aO15dh0LFiHc2N2DteX4ZBx4p1NDNi73h9IQYdu3FHG8frCzLoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXQM0xHmEDCDjmE6whwCZtAxTEeYQ8AMOobpCHMImEHHMB1hDgEz6BimI8whYAYdw3SEOQTMoGOYjjCHgBl0DNMR5hAwg45hOsIcAmbQMUxHmEPADDqG6QhzCJhBxzAdYQ4BM+gYpiPMIWAGHcN0hDkEzKBjmI4wh4AZdAzTEeYQMIOOYTrCHAJm0DFMR5hDwAw6hukIcwiYQccwHWEOATPoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXQM0xHmEDCDjrn1nYhBx9z6TsSgY259J2LQMbe+EzHoWImOpja+P+DW9xoGHVvf0dTG97hPcfj14yu3vlcx6Nj6jqY2vp9OdV1vf/145db3SgYdW93R1MZ3dbguYb18dOu7EIOOre5oauO76vZtW799XLb1bdAhg44V6ejbje+qa9rTMB778nHR1rfj9SGDnlNsvP79xveu7Wu9HN8+uvVdhkHHCrzL8X7je/yLYeP79aNb30UYdGx9RxMb3+OrwP613+tHt77LMOjY6o6mNr7r4X2M0/nto1vfZRh0bHVHkxvfbXdoht9suX5067sQg47dqKPdy5r3zq3vggw6hukIcwiYQccwHWEOATPoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXQM0xHmEDCDjmE6whwCZtAxTEeYQ8AMOobpCHMImEHHMB1hDgEz6BimI8whYAYdw3SEOQTMoGOYjjCHgBl0DNMR5hAwg45hOsIcAmbQMUxHmEPADDqG6QhzCJhBxzAdYQ4BM+gYpiPMIWAGHcN0hDkEzKBjmI4wh4AZdAzTEeYQMIOOYTrCHAJm0DFMR5hDwAw6hukIcwiYQccwHWEOATPoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXTM8fpEDDrmeH0iBh1zvD4Rg445Xp+IQcdKdrRgxN7x+jUMOlauo7kR+1eO169i0LFyHc2N2F85Xr+SQceKdTQzYu94fSEGHSvW0cyI/bLx+tPL3qE+ZNBztmNARR8YPxqxXzRe/0s7ePSvCppBz9mPARUM+uMRe8fryzDoWMF3OWZG7B2vL8KgY+U6mhmxd7y+DIOOFetobsTe8foyDDpWrKOZEXvH6wsx6NiNO3K8viSDjmE6whwCZtAxTEeYQ8AMOobpCHMImEHHMB1hDgEz6BimI8whYAYdw3SEOQTMoGOYjjCHgBl0DNMR5hAwg45hOsIcAmbQMUxHmEPADDqG6QhzCJhBxzAdYQ4BM+gYpiPMIWAGHcN0hDkEzKBjmI4wh4AZdAzTEeYQMIOOYTrCHAJm0DFMR5hDwAw6hukIcwiYQccwHWEOATPoGKYjzCFgBh3DdIQ5BMygY5iOMIeAGXQM0xHmEDCDjmE6whwCZtAxTEeYQ8AMOobpCHMImEHHMB1hDgEz6BimI8whYAYdc7w+EYOOOV6fiEHHHK9P5F8NOuR4fSL/ZtChEh1Nj9Y7Xl+cQcfWdzQ1Wl85Xn8LBh1b39HUaL3j9Tdh0LHVHU2O1jtefxMGHVvd0eRo/Y+M1//ld4Pf62N//b9b+/dH/yeu8NMYUJEHxnej9T8wXv8ffxj9UR/7z/+6tf9+9H/iCj+PAf38P6tznhit/4HxeolhYrT+R8brJYap0fofGK+XEKZG618/fs94vYQwOVpfff94vZTA8vF6SZIkSZIkSdJvzP8Dq8BI9N1EYXMAAAAldEVYdGRhdGU6Y3JlYXRlADIwMjEtMDctMzFUMDc6NTY6NDMrMDc6MDCFN7XvAAAAJXRFWHRkYXRlOm1vZGlmeQAyMDIxLTA3LTMxVDA3OjU2OjQzKzA3OjAw9GoNUwAAACF0RVh0cHM6SGlSZXNCb3VuZGluZ0JveAA1MDR4NzIwKzUwKzUw1uIiwwAAABN0RVh0cHM6TGV2ZWwAQWRvYmUtMi4wCjmTdA0AAAAASUVORK5CYII=" />

=end html


Result formatted as table (split, part 3 of 8):

 #table3#
 {dataset=>"elems1000prefix0"}
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset          | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems1000prefix0 |         |        | perl |      5580 |       179 |                 0.00% |                 0.00% | 5.3e-08 |      20 |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

       Rate     
     5580/s  -- 
 
 Legends:
   : dataset=elems1000prefix0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 4 of 8):

 #table4#
 {dataset=>"elems1000prefix1"}
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset          | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems1000prefix1 |         |        | perl |      4800 |       210 |                 0.00% |                 0.00% | 2.1e-07 |      20 |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

       Rate     
     4800/s  -- 
 
 Legends:
   : dataset=elems1000prefix1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 5 of 8):

 #table5#
 {dataset=>"elems100prefix0"}
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset         | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems100prefix0 |         |        | perl |     55000 |        18 |                 0.00% |                 0.00% | 2.6e-08 |      21 |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

        Rate     
     55000/s  -- 
 
 Legends:
   : dataset=elems100prefix0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 6 of 8):

 #table6#
 {dataset=>"elems100prefix1"}
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset         | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems100prefix1 |         |        | perl |     47200 |      21.2 |                 0.00% |                 0.00% | 5.7e-09 |      27 |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

        Rate     
     47200/s  -- 
 
 Legends:
   : dataset=elems100prefix1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 7 of 8):

 #table7#
 {dataset=>"elems10prefix0"}
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset        | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems10prefix0 |         |        | perl |    460000 |       2.2 |                 0.00% |                 0.00% | 2.5e-09 |      20 |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

         Rate     
     460000/s  -- 
 
 Legends:
   : dataset=elems10prefix0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAAJxQTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlQDVlQDWlADVlQDVlADUlQDVkADOjQDKAAAAQgBePgBZAAAAFgAfEQAYAAAAAAAAlADURQBj////gsBvMwAAADB0Uk5TABFEZiK7d+4zzIhV3ZmqqdXKx9I/7/z27Pn0dVxO5O3waXVO9d/s9Pnx/vjO9vRbldFvCAAAAAFiS0dEAIgFHUgAAAAJcEhZcwAAAEgAAABIAEbJaz4AAAAHdElNRQflBx8HOCwQD33VAAAPcUlEQVR42u3di3Lb1hWFYVyJC0HGSeomTdPYvd9btO//cMUBSdeiTWNR2iC0d/5vpoaoKJU8/Yc9pCitLAMAAAAAAAAAAAAAAAAAAADwQHlxfqPIt/5SgGcqqw9vFuP5jfFD2eOsuPe/FNhK/f9aPw06Lya7hjtseNF2u+kuuuz7IgVdzdc56KLv29OHdPutv0hAVQ6HIquHvm/KKejmcBj7Oehd0x/GMn3E7rj11wjo0pGjnu6k++MU9G7qd6ymoKtxunsum+mf50314s8BPMzpDN3uu/p8hp7unseiHNLpOVXdH7b+CoE7pKD7sT7Ul6CbFHTf1Ek73UHzFAc8mYLep1NFn4LOpyPGfA+9H7LT89LlsPUXCNyj3s/R5vORYzpe9EM6deTTY8T5zSMnDrhyHMq8G+rh0JRF1w1D087H6LLpuvRmU279BQL3yIvpvFEUeZau8xuX9/MNcAAAAAAAAAAAALw2849S5O3VO/PPX4FXrq+nnI/j2KUXiaUf4pxuV92YXlrzyRV47YoU8KHL8+MxvRqsKIrpvro+5tXQf3oFXrm8OdZZnn6Yokrhnl4NNv/I0K775Aq8dsd+fi161s6vBhsvP7CczX9cX4FXruzSGXo/1ueX7g7zTyeXY5YCzq+v53/pqzezr78BbHw7F/XtL17aczVUKeg+/bR930ynjqnZXZPtTgFX19fzv/X2l98lb7/Hsl9t/QW48MNc1Pjrlwbdd9OJY+h/TMHm59/wM10Xjhzff7Pu/22EUm/9BTjy8qCLfg76N2OWQm6L9JhwegRYpTvjcsiur2cEfQeC1r086CQ9Dz3ssuwwZPOvjjh00/8K/ef/c0LQdyBonV3QbdPNDwr79OiwPd3u8k+vJwR9B4LW2QQ9y4vTAbo6X/Mb1xlB34GgdYZB34Wg78AvLtARNEIhaIRC0AiFoBEKQSMUgkYoBI1QCBqhEDRCIWiEQtAIhaARCkEjFIJGKASNUAgaoRA0QiFohELQCIWgEQpBIxSCRigEjVAIGqEQNEIhaIRC0AiFoBEKQSMUgkYoBI1QCBqhGAV9NV5/euNL4/UEjVXYbaxcxuvPNxfG6wkaqzAJ+sl4/fnmwng9QWMVFkE/Ha8/3VwarydorMIi6Kfj9aebS+P1BI1VGAR9NV5/urk0Xv/2pz6pnvk5gWvlXNTLg74erz/dXBqvf/emSPJnflLgWjsXZT5ef75ZceTAFuzH6083K8brsQXz8frLTcbrsQX78frzTcbrsYUVxuuvbzNej8fhxUkIhaARCkEjFIJGKASNUAgaoRA0QiFohELQCIWgEQpBIxSCRigEjVAIGqEQNEIhaIRC0AiFoBEKQSMUgkYoBI1QCBqhEDRCIWiEQtAIhaARCkEjFIJGKASNUAgaoRA0QiFohLLG1ndxnrpi6xsPZ7/1XTbjWOdsfWMT5lvfeVNmeXdg6xubMN/6nqcIp3tstr6xhRW2vtN7jmx9YxP2W9/T4WIY8qWtb4LGKsy3vidFOZ2VF7a+37+vk3brvz/C6OeizLe+5/ftR44c2IT51ve8IjuFy9Y3tmC+9V2kZzPS5jdb39iA/db3YXp0mK5sfWMDK2x9Vzc2vtn6xvp4cRJCIWiEQtAIhaARCkEjFIJGKASNUAgaoRA0QiFohELQCIWgEQpBIxSCRigEjVAIGqEQNEIhaIRC0AiFoBEKQSMUgkYoBI1QCBqhEDRCIWiEQtAIhaARCkEjFIJGKASNUAgaoRA0QlljvL5lvB5bsR+vb4dxHFrG67EJ8/H6rDlk+byCxXg9Hm+F8frpYFGNLeP12IL5eH2epq6KsWJJFltYY7x+Oi4fGK/HJlYYr8/nNxmvx2OtNV7fdnOmHDmwBfPx+mw4PTfHeD22YD5evx+LhPF6bMJ8vL4fZ4zXYxMrjNdf32a8Ho/Di5MQCkEjFIJGKASNUAgaoRA0QiFohELQCIWgEQpBIxSCRigEjVAIGqEQNEIhaIRC0AiFoBEKQSMUgkYoBI1QCBqhEDRCIWiEQtAIhaARCkEjFIJGKASNUAgaoRA0QiFohELQCIWgEcoa4/XZ+Tf1M16Ph7Mfr08DWPOfjNfj8ezH66t9N6Z3Ml6PDZiP12dlPQfNeD22YD5en2UfjcayJIsHW2O8fs6W8XpsYYXx+lPQjNfjsdYar+fIgQ3Zj9efs2W8HlswH6/PLvfDjNdjA+bj9dklaMbrsQHG6xEKL05CKASNUAgaoRA0QiFohELQCIWgEQpBIxSCRigEjVAIGqEQNEIhaIRC0AiFoBEKQSMUgkYoBI1QCBqhEDRCIWiEQtAIhaARCkEjFIJGKASNUAgaoYhBt9a/mZygsQop6LIZ62IwbZqgsQol6HYsizrvm3z5Q2UEjVUoQfeHrKizrCuWP1RG0FiFFHRP0HBCCbpo2inokiMHXj/pQeFuHJqhKb/wEU/G62+N1jNej9VpT9tVZb//0v3zx+P1t0brGa/HAyhBV6fDc1nd+oAn4/W3RusZr8cDLAddFbtDMdkPtx4UPhmvvzVaz3g9HmE56LLuhnl09njr0PFkvP7WgixLsngE6Rsrp4eDt44cT8frb43WX43Xv3uT7vULyydO4vrt79b3+63/ki/WzkVp3/o+pnvo5vNHjqvx+luj9Vfj9W9/6pNq+ZMj+8N/1vfHrf+SL1bORWnPQ/dd3XeHz//Tq/F6jhz2/kTQMvE7hftDlg+fPx9cjdffGq1nvP75CFonBt3WWVbf/tb3x+P1t0brGa9/NoLWKUGX0yl5un8dvhz0Zbz+1mg94/XPRtA66UFhPT3ma4Zu4aNujdUzXv9SBK2TfwRrX5o+xUbQdyBonfQsR7n8Mfci6DsQtE4Jerd02HgGgr4DQeukI8ehn78JY/l5CfoOBK2TjhzjieXnJeg7ELSO38vhAEHrCNoBgtYRtAMErSNoBwhaR9AOELSOoB0gaB1BO0DQOoJ2gKB1BO0AQesI2gGC1hG0AwStI2gHCFpH0A4QtI6gHSBoHUE7QNA6gnaAoHUE7QBB6wjaAYLWEbQDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHrCNoBgtYRtAMErTMKun1yg61vWwStswk6TVL06fc5Ttf9MI5po5OtbzMErTMJet76PqYB5TbLm12Wd2x9WyJonUXQ89Z3Vp9+z/8uLbftG7a+DRG0ziLoees7G8u+L9IOZ8bwpjGC1hkEfdr6zsahP4xl1qZ74sNYLGx9E/QdCFr38qDPW99VP7W6a6aYm7qrx5atbzsErdC3vr/svPU9p5mP06Gj7ffLR453b+bZFtOpuLAIWtHORb086PPW94/pMeH0yC9P20Jlx9a3IYLWmT0PXcxn5266k95n+cDWtyWC1ll+Y6Uehinqcqyb9A0Utr7NELTO8LUc1XnJsCpO3whn69sKQet4cZIDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHrCNoBgtYRtAMErSNoBwhaR9AOELSOoB0gaB1BO0DQOoJ2gKB1BO0AQesI2gGC1hG0AwStI2gHCFpH0A4QtI6gHSBoHUE7QNA6gnaAoHUE7QBB6wjaAYLWEbQDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHrCNoBgtYRtAMErVtlvP481sZ4vRGC1tmP17fDdGG83hJB68zH67Ohz/LuwHi9IYLWmY/XZ2l7My3LMl5vhqB15uP1WbNL99aM1xsiaJ35eH1WNEMz5Bnj9XYIWmc+Xp93x2I/naEXxuvfv6+T9pmf9OeFoBX9XJT5eP08592OFUcOOwStMx+v79PjvilsxuvtELTOfLy+Tde+YbzeEEHr1hiv74amZbzeEEHrVhmv//xoPeP1z0XQOl6c5ABB6wjaAYLWEbQDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHrCNoBgtYRtAMErSNoBwhaR9AOELSOoB0gaB1BO0DQOoJ2gKB1BO0AQesI2gGC1hG0AwStI2gHCFpH0A4QtI6gHSBoHUE7QNA6gnaAoHUE7QBB6wjaAYLWEbQDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHr1hiv//BOxuttELTOfLy+GGcF4/V2CFpnPl6fT38WuyZnvN4OQevsx+uTbs94vSGC1tmP1092x4zxekMErbMfr0/32Gk0dswYrzdC0Drz8fpJnx77MV5vh6AVK43XpzvodPLgyGGHoHXm4/WXhXrG6+0QtM58vD49fze/j/F6MwStsx+vz5rT83eM15shaN0K4/UXjNdbIWgdL05ygKB1BO0AQesI2gGC1hG0AwStI2gHCFpH0A4QtI6gHSBoHUE7QNA6gnaAoHUE7QBB6wjaAYLWEbQDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHrCNoBgtYRtAMErSNoBwhaR9AOELSOoB0gaB1BO0DQOoJ2gKB1BO0AQesI2gGC1hG0AwStI2gHCFpH0A4QtI6gHSBoHUE7QNA6gnaAoHVrjNfn51uM1xshaJ35eH2WH8exqzLG6+0QtM58vD47dHl+PGaM19shaJ35eH2e5t2qnvF6QwStMx+vL8asLXKWZC0RtM58vH6f5gqbdmm8/t2bIsmf+Tl/Xgha0c5FmY/X9+N0Tu6bpfH6tz/1SfXMT/rzQtCKci7KfLx+PlVcrhw5TBC0zny8vh2z+ZEh4/V2CFpnP14/7KbrwHi9IYLW2Y/Xp5H6pmW83hBB61YYr88XrjOCvgNB63hxkgMErSNoBwhaR9AOELSOoB0gaB1BO0DQOoJ2gKB1BO0AQesI2gGC1hG0AwStI2gHCFpH0A4QtI6gHSBoHUE7QNA6gnaAoHUE7QBB6wjaAYLWEbQDBK0jaAcIWkfQDhC0jqAdIGgdQTtA0DqCdoCgdQTtAEHrCNoBgtYRtAMErSNoBwhaR9AOELSOoB0gaB1BO0DQOoJ2gKB1BO0AQesI2gGC1hG0A38maNka4/Uf3sl4vY2/ELTMfrz+cmW83gxB6+zH6y9XxuvNELTOfLz+cmW83g5B68zH6z8asWdJ1ghB68zH6y/XpfH6v75Jvv4Gy/723/X9feu/5It9OxdlPl5/uS6M1//ju9nb77Hsnw/wr63/ki/2w1zUD/9+adBX4/XZ+bpw5ABeqavx+uJ8XRivB16xj8brP4zYf3m8HnjFPh6v/3jE/vZ4PfDaXcbrl0bsAQAAAAAAAAAAovkfzbxGttlRMgQAAAAldEVYdGRhdGU6Y3JlYXRlADIwMjEtMDctMzFUMDc6NTY6NDQrMDc6MDBAkIthAAAAJXRFWHRkYXRlOm1vZGlmeQAyMDIxLTA3LTMxVDA3OjU2OjQ0KzA3OjAwMc0z3QAAACF0RVh0cHM6SGlSZXNCb3VuZGluZ0JveAA1MDR4NzIwKzUwKzUw1uIiwwAAABN0RVh0cHM6TGV2ZWwAQWRvYmUtMi4wCjmTdA0AAAAASUVORK5CYII=" />

=end html


Result formatted as table (split, part 8 of 8):

 #table8#
 {dataset=>"elems10prefix1"}
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset        | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems10prefix1 |         |        | perl |    396000 |      2.53 |                 0.00% |                 0.00% | 7.3e-10 |      26 |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

         Rate     
     396000/s  -- 
 
 Legends:
   : dataset=elems10prefix1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Benchmark module startup overhead (C<< bencher -m StringFunctions::CommonPrefix --module-startup >>):

Result formatted as table:

 #table9#
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix |         8 |                 4 |                 0.00% |                92.35% | 0.00022 |      20 |
 | perl -e1 (baseline)  |         4 |                 0 |                92.35% |                 0.00% | 0.00013 |      20 |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                Rate   S:C  :perl -e1 ( 
  S:C          0.1/s    --         -50% 
  :perl -e1 (  0.2/s  100%           -- 
 
 Legends:
   :perl -e1 (: mod_overhead_time=0 participant=perl -e1 (baseline)
   S:C: mod_overhead_time=4 participant=String::CommonPrefix

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-StringFunctions>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-StringFunctions>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-StringFunctions>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2018 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
