package Mojo::PgX::Cursor::Results;

require Mojo::IOLoop;
require Mojo::PgX::Cursor::Cursor;

use Time::HiRes qw(time);

use Mojo::Base -base;

has [qw(seconds_blocked fetch reload_at)];

sub array {
  my $self = shift->_fetch;
  $self->{remaining}--;
  return $self->{results}->array;
}

sub columns { shift->{results}->columns }

sub cursor {
  my $self = shift;
  if (@_) {
    $self->{cursor} = shift;
    $self->{remaining} = 0;
    delete $self->{results};
    return $self;
  }
  return $self->{cursor};
}

sub hash {
  my $self = shift->_fetch;
  $self->{remaining}--;
  return $self->{results}->hash;
}

sub expand {
  my $self = shift;
  $self->{expand}++;
  $self->{results}->expand;
  return $self;
}

sub new {
  my $self = shift->SUPER::new(
    fetch => 10,
    reload_at => 10,
    remaining => 0,
    @_
  );
  return $self->_fetch
}

sub reload {
  my ($self, $fetch) = (shift, shift);
  unless (defined $fetch) {
      $fetch = $self->{fetch};
  }
  $self->{delay} = Mojo::IOLoop->delay(
    sub {
      my $delay = shift;
      $self->cursor->fetch($fetch, $delay->begin);
    },
    sub {
      my ($delay, $err, $results) = @_;
      $self->{results} = $results;
      $self->{results}->expand if ($self->{expand});
      $self->{remaining} = $self->{results}->rows;
    },
  );
  return $self;
}

sub rows { shift->{results}->rows }

sub _fetch {
  my $self = shift;
  if (not $self->{delay} and $self->{remaining} <= $self->{reload_at}) {
    $self->reload;
  }
  unless ($self->{remaining}) {
    my $start = time;
    $self->{delay}->wait;
    $self->{seconds_blocked} += time - $start;
    delete $self->{delay};
  }
  return $self;
}

1;
__END__

=encoding utf-8

=head1 NAME

Mojo::PgX::Cursor::Results

=head1 SYNOPSIS

=head1 DESCRIPTION

L<Mojo::PgX::Cursor::Results> is a container for L<Mojo::PgX::Cursor> cursor
like L<Mojo::Pg::Results> is for statement handles.

=head1 ATTRIBUTES

=head2 cursor

    my $cursor = $results->cursor;
    $results = $results->cursor($cursor);

L<Mojo::PgX::Cursor::Cursor> results are fetched from.

=head2 fetch

    $results->fetch(100);

The quantity of rows to fetch in each batch of rows.  Smaller uses less memory.

=head2 reload_at

    $results->reload_at(50);

The threshold of remaining rows which will trigger a non-blocking reload to
start.  Smaller uses less memory.

=head2 seconds_blocked

    my $time_wasted = $results->seconds_blocked;

The cumulative time the cursor has spent blocking upon running out of rows.

=head1 METHODS

=head2 array

    my $row = $results->array;

Return next row from L</"cursor"> as an array reference.  If necessary, the
next row will be fetched first.

=head2 columns

    my $columns = $results->columns;

Return column names as an array reference.

=head2 expand

    $results = $results->expand;

Decode C<json> and C<jsonb> fields automatically for all rows.

=head2 hash

    my $row = $results->hash;

Return next row from L</"cursor"> as a hash reference.  If necessary, the next
row will be fetched first.

=head2 new

    my $results = Mojo::PgX::Cursor::Results->new(cursor => $cursor);

Construct a new L<Mojo::PgX::Cursor::Results> object.

=head2 reload

    $results->reload;
    $results->reload(1000);

Manually trigger a new batch of rows to be fetched.  The
L<Mojo::PgX::Cursor::Results> object uses the L<"/fetch"> and L<"/reload_at">
attributes to automatically reload.  This method is just available in case you
want to implement more advanced logic in your app.  If passed an argument that
value overrides L<"/fetch">.

=head2 rows

    my $num = $results->rows;

Number of rows in current batch; not total.

=head1 LICENSE

Copyright (C) Nathaniel Nutter.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHOR

Nathaniel Nutter C<nnutter@cpan.org>

=head1 SEE ALSO

L<Mojo::PgX::Cursor>, L<Mojo::PgX::Cursor::Cursor>

=cut

