package Test::Builder2::Streamer::Print;
use Test::Builder2::Mouse;
with 'Test::Builder2::Streamer';


=head1 NAME

Test::Builder2::Streamer::Print - A simple streamer that prints

=head1 DESCRIPTION

This is a L<Test::Builder2::Streamer> which prints to a filehandle.

You are encouraged to subclass this Streamer if you're writing one
which prints.

=head2 Destinations

It ignores your destination.  Everything goes to the L<output_fh>.

=head2 Attributes

=head3 output_fh

The filehandle to which it should write.

=cut

has output_fh =>
  is            => 'rw',
  # "FileHandle" does not appear to include glob filehandles.
  #  isa           => 'FileHandle',
  default       => *STDOUT,
;


=head2 Methods

=head3 safe_print

    $streamer->safe_print($fh, @hunks);

Works like C<print> but is not effected by the global variables which
change print's behavior such as C<$\> and C<$,>.  This allows a test
to play with these variables without affecting test output.

Subclasses are encouraged to take advantage of this method rather than
calling print themselves.

=cut

sub safe_print {
    my $self = shift;
    my $fh   = shift;

    local( $\, $, ) = ( undef, '' );
    print $fh @_;
}

sub write {
    my $self = shift;
    my $dest = shift;

    $self->safe_print($self->output_fh, @_);
}

no Test::Builder2::Mouse;
1;
