package App::bif::push::project;
use strict;
use warnings;
use AnyEvent;
use Bif::Sync::Client;
use Bif::Mo;
use Coro;

our $VERSION = '0.1.5_6';
extends 'App::bif';

sub run {
    my $self  = shift;
    my $opts  = $self->opts;
    my $dbw   = $self->dbw;
    my $pinfo = $self->get_project( $opts->{path} );
    my $hinfo = $self->get_hub( $opts->{hub} );

    my $re_push = $dbw->xval(
        select => '1',
        from   => 'project_hubs ph',
        where  => {
            'ph.project_id' => $pinfo->{id},
            'ph.hub_id'     => $hinfo->{id},
        },
    );
    warn "warning: project already pushed: $pinfo->{path} -> $hinfo->{name}\n"
      if $re_push;

    $|++;    # no buffering
    my $error;
    my $cv = AE::cv;

    my $client = Bif::Sync::Client->new(
        name          => $hinfo->{name},
        db            => $dbw,
        location      => $hinfo->{location},
        debug         => $opts->{debug},
        debug_bifsync => $opts->{debug_bifsync},
        on_update     => sub {
            my $client = shift;
            $self->lprint( $client->name . ': ' . $_[0] );
        },
        on_error => sub {
            $error = shift;
            $cv->send;
        },
    );

    my $fh   = select;
    my $coro = async {
        select $fh;

        eval {
            $dbw->txn(
                sub {
                    my $msg = "[ push: $opts->{path} ($opts->{hub}) ]";
                    if ( $opts->{message} ) {
                        $msg .= "\n\n$opts->{message}\n";
                    }

                    $opts->{change_id} =
                      $self->new_change( parent_id => $pinfo->{first_change_id},
                      );

                    $dbw->xdo(
                        insert_into => 'func_update_project',
                        values      => {
                            change_id => $opts->{change_id},
                            id        => $pinfo->{id},
                            hub_id    => $hinfo->{id},
                        },
                    ) unless $re_push;

                    $self->end_change(
                        id            => $opts->{change_id},
                        action_format => "push project $pinfo->{path} "
                          . "(%s) $hinfo->{name}",
                        action_node_id_1 => $pinfo->{id},
                        message          => $msg,
                    );

                    $dbw->xdo(
                        update => 'projects',
                        set    => { default_hub_id => $hinfo->{id} },
                        where  => { id => $pinfo->{id} },
                    );

                    my $status = $client->sync_hub( $hinfo->{id} );

                    unless ( $status eq 'HubSync'
                        or $status eq 'HubMatch' )
                    {
                        $dbw->rollback;
                        $error =
                          "unexpected HubSync/Match status received: $status";
                        return;
                    }

                    if ( $status eq 'HubSync' ) {
                        $status = $client->transfer_hub_changes;
                        if ( $status ne 'TransferHubChanges' ) {
                            $dbw->rollback;
                            $error =
"unexpected TransferHubChanges status received: $status";
                            return;
                        }
                    }

                    $status = $client->sync_project( $pinfo->{id} );

                    unless ( $status eq 'ProjectSync'
                        or $status eq 'ProjectMatch' )
                    {
                        $dbw->rollback;
                        $error =
"unexpected ProjectSync/Match status received: $status";
                        return;
                    }

                    if ( $status eq 'ProjectSync' ) {
                        $status = $client->transfer_project_related_changes;

                        if ( $status ne 'TransferProjectRelatedChanges' ) {
                            $dbw->rollback;
                            $error =
"unexpected TransferProjectRelatedChanges status received: $status";
                            return;
                        }
                    }
                    print "\n";
                    print "Project(s) exported: $opts->{path}\n";

                    return;
                }
            );
        };

        if ($@) {
            $error = $@;
            print "\n";
        }

        return $cv->send( !$error );
    };

    $cv->recv;
    $client->disconnect;
    return $self->err( 'Unknown', $error ) if $error;
    return $self->ok('PushProject');
}

1;
__END__

=head1 NAME

=for bif-doc #sync

bif-push-project -  export a project to a remote hub

=head1 VERSION

0.1.5_6 (2015-10-20)

=head1 SYNOPSIS

    bif push project PATH HUB [OPTIONS...]

=head1 DESCRIPTION

The B<bif-push-project> command exports a project to a hub.

=head1 ARGUMENTS & OPTIONS

=over

=item PATH

The path of the local project to export.  An error will be raised if a
project with the same path exists at the remote HUB.

=item HUB

The name of a previously registered hub.

=item --message, -m MESSAGE

Add the optional MESSAGE to the change created for this action.

=back

=head1 SEE ALSO

L<bif>(1)

=head1 AUTHOR

Mark Lawrence E<lt>nomad@null.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2013-2015 Mark Lawrence <nomad@null.net>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

