CREATE TABLE hub_changes (
    hub_id INTEGER NOT NULL,
    -- Collate below for LIKE/index support
    prefix VARCHAR NOT NULL COLLATE NOCASE CHECK (LENGTH(prefix) > 0),
    change_id INTEGER, -- Leaf nodes only
    -- real_hub_id? Consider for per-hub changes?
    hash VARCHAR NOT NULL DEFAULT '*not-yet-defined*',
    num_changes INTEGER NOT NULL DEFAULT 1,
    PRIMARY KEY(hub_id,prefix),
    FOREIGN KEY(hub_id) REFERENCES hubs(id) ON DELETE CASCADE,
    FOREIGN KEY(change_id) REFERENCES changes(id) ON DELETE CASCADE
);


-- If current prefix exists then insert one level down
CREATE TRIGGER
    hub_changes_bi_1
BEFORE INSERT ON
    hub_changes
FOR EACH ROW WHEN
    EXISTS (SELECT
        1
    FROM
        hub_changes hc
    WHERE
        hc.hub_id = NEW.hub_id AND
        hc.prefix = NEW.prefix AND
        hc.change_id IS NULL
    ) AND NEW.prefix NOT LIKE 'z%'
BEGIN
    SELECT debug(
        NEW.hub_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- Insert next level down
    INSERT INTO
        hub_changes(
            hub_id,
            prefix,
            change_id,
            num_changes
        )
    SELECT
        NEW.hub_id,
        SUBSTR(c.uuid, 1, LENGTH(NEW.prefix) + 1),
        NEW.change_id,
        1
    FROM
        changes c
    WHERE
        c.id = NEW.change_id 
    ;

    SELECT RAISE(IGNORE);
END;

-- conflict, reparent current node
CREATE TRIGGER
    hub_changes_bi_2
BEFORE INSERT ON
    hub_changes
FOR EACH ROW WHEN
    EXISTS (SELECT
        1
    FROM
        hub_changes hc
    WHERE
        hc.hub_id = NEW.hub_id AND
        hc.prefix = NEW.prefix AND
        hc.change_id != NEW.change_id AND
        hc.num_changes = 1
    ) AND NEW.prefix NOT LIKE 'z%'
BEGIN
    SELECT debug(
        NEW.hub_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- Push existing node one level down
    INSERT INTO
        hub_changes(
            hub_id,
            prefix,
            change_id,
            hash,
            num_changes
        )
    SELECT
        hc.hub_id,
        SUBSTR(c.uuid, 1, LENGTH(NEW.prefix) + 1),
        c.id,
        c.uuid,
        1
    FROM
        hub_changes hc
    INNER JOIN
        changes c
    ON
        c.id = hc.change_id
    WHERE
        hc.hub_id = NEW.hub_id AND
        hc.prefix = NEW.prefix
    ;

    -- Insert this node next level down
    INSERT INTO
        hub_changes(
            hub_id,
            prefix,
            change_id,
            hash,
            num_changes
        )
    SELECT
        NEW.hub_id,
        SUBSTR(c.uuid, 1, LENGTH(NEW.prefix) + 1),
        NEW.change_id,
        c.uuid,
        1
    FROM
        changes c
    WHERE
        c.id = NEW.change_id 
    ;

    SELECT RAISE(IGNORE);
END;

-- Inserted, now work back up the tree
CREATE TRIGGER
    hub_changes_ai_1
AFTER INSERT ON
    hub_changes
FOR EACH ROW WHEN
    NEW.change_id IS NOT NULL AND NEW.prefix NOT LIKE 'z%'
BEGIN
    SELECT debug(
        NEW.hub_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- Special case for prefix length 1
    UPDATE
        hub_changes
    SET
        hash = (
            SELECT
                c.uuid
            FROM
                changes c
            WHERE
                c.id = NEW.change_id
        )
    WHERE
        hub_id = NEW.hub_id AND
        prefix = NEW.prefix
    ;

    UPDATE
        hub_changes
    SET
        change_id = NULL,
        hash = (
            SELECT
                agg_sha1_hex(hc.hash, hc.hash)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND
                hc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        ),
        num_changes = (
            SELECT
                SUM(hc.num_changes)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND
                hc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        )
    WHERE
        hub_id = NEW.hub_id AND
        prefix = SUBSTR(NEW.prefix,1,LENGTH(NEW.prefix) - 1)
    ;
END;

-- intermediate node update, so update the parent
CREATE TRIGGER
    hub_changes_au_1
AFTER UPDATE OF
    num_changes
ON
    hub_changes
FOR EACH ROW WHEN
    NEW.change_id IS NULL
BEGIN
    SELECT debug(
        NEW.hub_id,
        NEW.prefix,
        NEW.change_id,
        NEW.hash,
        NEW.num_changes
    );

    -- If no children remove ourself
    DELETE FROM
        hub_changes
    WHERE
        NEW.num_changes = 0 AND
        hub_id = NEW.hub_id AND
        prefix = NEW.prefix
    ;

    -- If one child replace ourself with it
    UPDATE
        hub_changes
    SET
        change_id = (
            SELECT
                hc.change_id
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND
                hc.prefix LIKE NEW.prefix || '_'
        ),
        hash = (
            SELECT
                hc.hash
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND
                hc.prefix LIKE NEW.prefix || '_'
        )
    WHERE
        NEW.num_changes = 1 AND OLD.num_changes > 1 AND
        hub_id = NEW.hub_id AND
        prefix = NEW.prefix
    ;

    DELETE FROM
        hub_changes
    WHERE
        NEW.num_changes = 1 AND OLD.num_changes > 1 AND
        hub_id = NEW.hub_id AND
        prefix LIKE NEW.prefix || '_'
    ;

    -- Update our parent
    UPDATE
        hub_changes
    SET
        change_id = NULL,
        hash = (
            SELECT
                agg_sha1_hex(hc.hash, hc.hash)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND
                hc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        ),
        num_changes = (
            SELECT
                SUM(hc.num_changes)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND
                hc.prefix LIKE
                    SUBSTR(NEW.prefix, 1, LENGTH(NEW.prefix) - 1) || '_'
        )
    WHERE
        hub_id = NEW.hub_id AND
        prefix = SUBSTR(NEW.prefix,1,LENGTH(NEW.prefix) - 1)
    ;

END;


-- Housekeeping - calculate topics.hash
CREATE TRIGGER
    hub_changes_au_2
AFTER UPDATE OF
    hash
ON
    hub_changes
FOR EACH ROW WHEN
    1 = LENGTH(NEW.prefix)
BEGIN
    UPDATE
        hubs
    SET
        hash = (
            SELECT
                agg_sha1_hex(hc.hash, hc.hash)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND hc.prefix LIKE '_'
        ),
        num_changes = (
            SELECT
                SUM(hc.num_changes)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = NEW.hub_id AND hc.prefix LIKE '_'
        )
    WHERE
        id = NEW.hub_id
    ;
END;


CREATE TRIGGER
    hub_changes_ad_1
AFTER DELETE ON
    hub_changes
FOR EACH ROW WHEN
    OLD.change_id IS NOT NULL
BEGIN
    SELECT debug(
        OLD.hub_id,
        OLD.prefix,
        OLD.change_id,
        OLD.hash,
        OLD.num_changes
    );

    UPDATE
        hub_changes
    SET
        change_id = NULL,
        hash = (
            SELECT
                agg_sha1_hex(hc.hash, hc.hash)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = OLD.hub_id AND
                hc.prefix LIKE
                    SUBSTR(OLD.prefix, 1, LENGTH(OLD.prefix) - 1) || '_'
        ),
        num_changes = (
            SELECT
                SUM(hc.num_changes)
            FROM
                hub_changes hc
            WHERE
                hc.hub_id = OLD.hub_id AND
                hc.prefix LIKE
                    SUBSTR(OLD.prefix, 1, LENGTH(OLD.prefix) - 1) || '_'
        )
    WHERE
        change_id IS NULL AND -- Don't update when this is a move
        hub_id = OLD.hub_id AND
        prefix = SUBSTR(OLD.prefix,1,LENGTH(OLD.prefix) - 1)
    ;

END;

