package Bio::Mitomaster::RefHuman;
use MooseX::Singleton;
use Carp;

=head1 NAME

Bio::Mitomaster::RefHuman - Singleton for a human mitochondrial reference object.

=head1 VERSION

Version 0.1.0, released July 15 09

=head1 SYNOPSIS

  use Bio::Mitomaster::RefHuman;
  $ref = Bio::Mitomaster::RefHuman->new();

  $amino_acid = $ref->codon_code('TTC');
  # $amino_acid is now F
  $amino_acid = $ref->codon_code('UUC');  
  # Same thing but with Uracil

  print "Position 100 of the rCRS is: ", $ref->ref_seq(100);
  $locus_id = $LOCUS_LOOKUP{'ND1'};  # Gets the database locus ID for ND1 locus
  $transcript = $ref->transcript_seq($locus_id);  # Get the ND1 transcript
  $translation = $ref->translation_seq($locus_id); # Get the ND1 translation

=cut


our $VERSION = '0.1';


=head1 Attributes


=head2 Seq String Attributes

All attributes that return a sequence of nucleotides have a modified accessor for retrieving a partial sequence.

 $r = Bio::Mitomaster::RefHuman->new();
 $s = $r->ref_seq(11);  # skips the first ten bases of the seq
 $s = $r->ref_seq(-11);  # only the last 11 bases
 $s = $r->ref_seq(11,20);  # 10 bases from positions 11 to 20
 $s = $r->ref_seq(11,-20);  # skip the first 10 bases and truncate the last 20
 $s = $r->ref_seq(-20,-10);  # ten bases starting 20 bases from the end

=cut



=head2 ref_seq 

    A string for the the MITOMAP reference sequence (essentially the revised Cambridge sequence).  Note that the revised Cambridge sequence (rCRS) has a placeholder at position 3107.  Because this causes numerous problems it is omitted in this sequence, making it one base shorter than the actual reference used by MITOMAP.  If you want to use the actual MITOMAP sequence with the placeholder than set the 'revised' attribute to true during instantiation.  Also note that the placeholder base is represented by an 'X'.  The accessor for this attribute is documented in the 'Seq String Attributes' section.
=cut

has 'ref_seq' => (
    is => 'ro',
    isa => 'Str',
    lazy_build => 1,
);

around 'ref_seq' => \&_ref_seq_lookup;

sub _build_ref_seq {#FOLDBEG
    my $self = shift;
    my $r = <<REFSEQ;
GATCACAGGTCTATCACCCTATTAACCACTCACGGGAGCTCTCCATGCATTTGGTATTTT
CGTCTGGGGGGTATGCACGCGATAGCATTGCGAGACGCTGGAGCCGGAGCACCCTATGTC
GCAGTATCTGTCTTTGATTCCTGCCTCATCCTATTATTTATCGCACCTACGTTCAATATT
ACAGGCGAACATACTTACTAAAGTGTGTTAATTAATTAATGCTTGTAGGACATAATAATA
ACAATTGAATGTCTGCACAGCCACTTTCCACACAGACATCATAACAAAAAATTTCCACCA
AACCCCCCCTCCCCCGCTTCTGGCCACAGCACTTAAACACATCTCTGCCAAACCCCAAAA
ACAAAGAACCCTAACACCAGCCTAACCAGATTTCAAATTTTATCTTTTGGCGGTATGCAC
TTTTAACAGTCACCCCCCAACTAACACATTATTTTCCCCTCCCACTCCCATACTACTAAT
CTCATCAATACAACCCCCGCCCATCCTACCCAGCACACACACACCGCTGCTAACCCCATA
CCCCGAACCAACCAAACCCCAAAGACACCCCCCACAGTTTATGTAGCTTACCTCCTCAAA
GCAATACACTGAAAATGTTTAGACGGGCTCACATCACCCCATAAACAAATAGGTTTGGTC
CTAGCCTTTCTATTAGCTCTTAGTAAGATTACACATGCAAGCATCCCCGTTCCAGTGAGT
TCACCCTCTAAATCACCACGATCAAAAGGAACAAGCATCAAGCACGCAGCAATGCAGCTC
AAAACGCTTAGCCTAGCCACACCCCCACGGGAAACAGCAGTGATTAACCTTTAGCAATAA
ACGAAAGTTTAACTAAGCTATACTAACCCCAGGGTTGGTCAATTTCGTGCCAGCCACCGC
GGTCACACGATTAACCCAAGTCAATAGAAGCCGGCGTAAAGAGTGTTTTAGATCACCCCC
TCCCCAATAAAGCTAAAACTCACCTGAGTTGTAAAAAACTCCAGTTGACACAAAATAGAC
TACGAAAGTGGCTTTAACATATCTGAACACACAATAGCTAAGACCCAAACTGGGATTAGA
TACCCCACTATGCTTAGCCCTAAACCTCAACAGTTAAATCAACAAAACTGCTCGCCAGAA
CACTACGAGCCACAGCTTAAAACTCAAAGGACCTGGCGGTGCTTCATATCCCTCTAGAGG
AGCCTGTTCTGTAATCGATAAACCCCGATCAACCTCACCACCTCTTGCTCAGCCTATATA
CCGCCATCTTCAGCAAACCCTGATGAAGGCTACAAAGTAAGCGCAAGTACCCACGTAAAG
ACGTTAGGTCAAGGTGTAGCCCATGAGGTGGCAAGAAATGGGCTACATTTTCTACCCCAG
AAAACTACGATAGCCCTTATGAAACTTAAGGGTCGAAGGTGGATTTAGCAGTAAACTAAG
AGTAGAGTGCTTAGTTGAACAGGGCCCTGAAGCGCGTACACACCGCCCGTCACCCTCCTC
AAGTATACTTCAAAGGACATTTAACTAAAACCCCTACGCATTTATATAGAGGAGACAAGT
CGTAACATGGTAAGTGTACTGGAAAGTGCACTTGGACGAACCAGAGTGTAGCTTAACACA
AAGCACCCAACTTACACTTAGGAGATTTCAACTTAACTTGACCGCTCTGAGCTAAACCTA
GCCCCAAACCCACTCCACCTTACTACCAGACAACCTTAGCCAAACCATTTACCCAAATAA
AGTATAGGCGATAGAAATTGAAACCTGGCGCAATAGATATAGTACCGCAAGGGAAAGATG
AAAAATTATAACCAAGCATAATATAGCAAGGACTAACCCCTATACCTTCTGCATAATGAA
TTAACTAGAAATAACTTTGCAAGGAGAGCCAAAGCTAAGACCCCCGAAACCAGACGAGCT
ACCTAAGAACAGCTAAAAGAGCACACCCGTCTATGTAGCAAAATAGTGGGAAGATTTATA
GGTAGAGGCGACAAACCTACCGAGCCTGGTGATAGCTGGTTGTCCAAGATAGAATCTTAG
TTCAACTTTAAATTTGCCCACAGAACCCTCTAAATCCCCTTGTAAATTTAACTGTTAGTC
CAAAGAGGAACAGCTCTTTGGACACTAGGAAAAAACCTTGTAGAGAGAGTAAAAAATTTA
ACACCCATAGTAGGCCTAAAAGCAGCCACCAATTAAGAAAGCGTTCAAGCTCAACACCCA
CTACCTAAAAAATCCCAAACATATAACTGAACTCCTCACACCCAATTGGACCAATCTATC
ACCCTATAGAAGAACTAATGTTAGTATAAGTAACATGAAAACATTCTCCTCCGCATAAGC
CTGCGTCAGATTAAAACACTGAACTGACAATTAACAGCCCAATATCTACAATCAACCAAC
AAGTCATTATTACCCTCACTGTCAACCCAACACAGGCATGCTCATAAGGAAAGGTTAAAA
AAAGTAAAAGGAACTCGGCAAATCTTACCCCGCCTGTTTACCAAAAACATCACCTCTAGC
ATCACCAGTATTAGAGGCACCGCCTGCCCAGTGACACATGTTTAACGGCCGCGGTACCCT
AACCGTGCAAAGGTAGCATAATCACTTGTTCCTTAAATAGGGACCTGTATGAATGGCTCC
ACGAGGGTTCAGCTGTCTCTTACTTTTAACCAGTGAAATTGACCTGCCCGTGAAGAGGCG
GGCATAACACAGCAAGACGAGAAGACCCTATGGAGCTTTAATTTATTAATGCAAACAGTA
CCTAACAAACCCACAGGTCCTAAACTACCAAACCTGCATTAAAAATTTCGGTTGGGGCGA
CCTCGGAGCAGAACCCAACCTCCGAGCAGTACATGCTAAGACTTCACCAGTCAAAGCGAA
CTACTATACTCAATTGATCCAATAACTTGACCAACGGAACAAGTTACCCTAGGGATAACA
GCGCAATCCTATTCTAGAGTCCATATCAACAATAGGGTTTACGACCTCGATGTTGGATCA
GGACATCCCGATGGTGCAGCCGCTATTAAAGGTTCGTTTGTTCAACGATTAAAGTCCTAC
GTGATCTGAGTTCAGACCGGAGTAATCCAGGTCGGTTTCTATCTACXTTCAAATTCCTCC
CTGTACGAAAGGACAAGAGAAATAAGGCCTACTTCACAAAGCGCCTTCCCCCGTAAATGA
TATCATCTCAACTTAGTATTATACCCACACCCACCCAAGAACAGGGTTTGTTAAGATGGC
AGAGCCCGGTAATCGCATAAAACTTAAAACTTTACAGTCAGAGGTTCAATTCCTCTTCTT
AACAACATACCCATGGCCAACCTCCTACTCCTCATTGTACCCATTCTAATCGCAATGGCA
TTCCTAATGCTTACCGAACGAAAAATTCTAGGCTATATACAACTACGCAAAGGCCCCAAC
GTTGTAGGCCCCTACGGGCTACTACAACCCTTCGCTGACGCCATAAAACTCTTCACCAAA
GAGCCCCTAAAACCCGCCACATCTACCATCACCCTCTACATCACCGCCCCGACCTTAGCT
CTCACCATCGCTCTTCTACTATGAACCCCCCTCCCCATACCCAACCCCCTGGTCAACCTC
AACCTAGGCCTCCTATTTATTCTAGCCACCTCTAGCCTAGCCGTTTACTCAATCCTCTGA
TCAGGGTGAGCATCAAACTCAAACTACGCCCTGATCGGCGCACTGCGAGCAGTAGCCCAA
ACAATCTCATATGAAGTCACCCTAGCCATCATTCTACTATCAACATTACTAATAAGTGGC
TCCTTTAACCTCTCCACCCTTATCACAACACAAGAACACCTCTGATTACTCCTGCCATCA
TGACCCTTGGCCATAATATGATTTATCTCCACACTAGCAGAGACCAACCGAACCCCCTTC
GACCTTGCCGAAGGGGAGTCCGAACTAGTCTCAGGCTTCAACATCGAATACGCCGCAGGC
CCCTTCGCCCTATTCTTCATAGCCGAATACACAAACATTATTATAATAAACACCCTCACC
ACTACAATCTTCCTAGGAACAACATATGACGCACTCTCCCCTGAACTCTACACAACATAT
TTTGTCACCAAGACCCTACTTCTAACCTCCCTGTTCTTATGAATTCGAACAGCATACCCC
CGATTCCGCTACGACCAACTCATACACCTCCTATGAAAAAACTTCCTACCACTCACCCTA
GCATTACTTATATGATATGTCTCCATACCCATTACAATCTCCAGCATTCCCCCTCAAACC
TAAGAAATATGTCTGATAAAAGAGTTACTTTGATAGAGTAAATAATAGGAGCTTAAACCC
CCTTATTTCTAGGACTATGAGAATCGAACCCATCCCTGAGAATCCAAAATTCTCCGTGCC
ACCTATCACACCCCATCCTAAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCG
AAAATGTTGGTTATACCCTTCCCGTACTAATTAATCCCCTGGCCCAACCCGTCATCTACT
CTACCATCTTTGCAGGCACACTCATCACAGCGCTAAGCTCGCACTGATTTTTTACCTGAG
TAGGCCTAGAAATAAACATGCTAGCTTTTATTCCAGTTCTAACCAAAAAAATAAACCCTC
GTTCCACAGAAGCTGCCATCAAGTATTTCCTCACGCAAGCAACCGCATCCATAATCCTTC
TAATAGCTATCCTCTTCAACAATATACTCTCCGGACAATGAACCATAACCAATACTACCA
ATCAATACTCATCATTAATAATCATAATAGCTATAGCAATAAAACTAGGAATAGCCCCCT
TTCACTTCTGAGTCCCAGAGGTTACCCAAGGCACCCCTCTGACATCCGGCCTGCTTCTTC
TCACATGACAAAAACTAGCCCCCATCTCAATCATATACCAAATCTCTCCCTCACTAAACG
TAAGCCTTCTCCTCACTCTCTCAATCTTATCCATCATAGCAGGCAGTTGAGGTGGATTAA
ACCAAACCCAGCTACGCAAAATCTTAGCATACTCCTCAATTACCCACATAGGATGAATAA
TAGCAGTTCTACCGTACAACCCTAACATAACCATTCTTAATTTAACTATTTATATTATCC
TAACTACTACCGCATTCCTACTACTCAACTTAAACTCCAGCACCACGACCCTACTACTAT
CTCGCACCTGAAACAAGCTAACATGACTAACACCCTTAATTCCATCCACCCTCCTCTCCC
TAGGAGGCCTGCCCCCGCTAACCGGCTTTTTGCCCAAATGGGCCATTATCGAAGAATTCA
CAAAAAACAATAGCCTCATCATCCCCACCATCATAGCCACCATCACCCTCCTTAACCTCT
ACTTCTACCTACGCCTAATCTACTCCACCTCAATCACACTACTCCCCATATCTAACAACG
TAAAAATAAAATGACAGTTTGAACATACAAAACCCACCCCATTCCTCCCCACACTCATCG
CCCTTACCACGCTACTCCTACCTATCTCCCCTTTTATACTAATAATCTTATAGAAATTTA
GGTTAAATACAGACCAAGAGCCTTCAAAGCCCTCAGTAAGTTGCAATACTTAATTTCTGT
AACAGCTAAGGACTGCAAAACCCCACTCTGCATCAACTGAACGCAAATCAGCCACTTTAA
TTAAGCTAAGCCCTTACTAGACCAATGGGACTTAAACCCACAAACACTTAGTTAACAGCT
AAGCACCCTAATCAACTGGCTTCAATCTACTTCTCCCGCCGCCGGGAAAAAAGGCGGGAG
AAGCCCCGGCAGGTTTGAAGCTGCTTCTTCGAATTTGCAATTCAATATGAAAATCACCTC
GGAGCTGGTAAAAAGAGGCCTAACCCCTGTCTTTAGATTTACAGTCCAATGCTTCACTCA
GCCATTTTACCTCACCCCCACTGATGTTCGCCGACCGTTGACTATTCTCTACAAACCACA
AAGACATTGGAACACTATACCTATTATTCGGCGCATGAGCTGGAGTCCTAGGCACAGCTC
TAAGCCTCCTTATTCGAGCCGAGCTGGGCCAGCCAGGCAACCTTCTAGGTAACGACCACA
TCTACAACGTTATCGTCACAGCCCATGCATTTGTAATAATCTTCTTCATAGTAATACCCA
TCATAATCGGAGGCTTTGGCAACTGACTAGTTCCCCTAATAATCGGTGCCCCCGATATGG
CGTTTCCCCGCATAAACAACATAAGCTTCTGACTCTTACCTCCCTCTCTCCTACTCCTGC
TCGCATCTGCTATAGTGGAGGCCGGAGCAGGAACAGGTTGAACAGTCTACCCTCCCTTAG
CAGGGAACTACTCCCACCCTGGAGCCTCCGTAGACCTAACCATCTTCTCCTTACACCTAG
CAGGTGTCTCCTCTATCTTAGGGGCCATCAATTTCATCACAACAATTATCAATATAAAAC
CCCCTGCCATAACCCAATACCAAACGCCCCTCTTCGTCTGATCCGTCCTAATCACAGCAG
TCCTACTTCTCCTATCTCTCCCAGTCCTAGCTGCTGGCATCACTATACTACTAACAGACC
GCAACCTCAACACCACCTTCTTCGACCCCGCCGGAGGAGGAGACCCCATTCTATACCAAC
ACCTATTCTGATTTTTCGGTCACCCTGAAGTTTATATTCTTATCCTACCAGGCTTCGGAA
TAATCTCCCATATTGTAACTTACTACTCCGGAAAAAAAGAACCATTTGGATACATAGGTA
TGGTCTGAGCTATGATATCAATTGGCTTCCTAGGGTTTATCGTGTGAGCACACCATATAT
TTACAGTAGGAATAGACGTAGACACACGAGCATATTTCACCTCCGCTACCATAATCATCG
CTATCCCCACCGGCGTCAAAGTATTTAGCTGACTCGCCACACTCCACGGAAGCAATATGA
AATGATCTGCTGCAGTGCTCTGAGCCCTAGGATTCATCTTTCTTTTCACCGTAGGTGGCC
TGACTGGCATTGTATTAGCAAACTCATCACTAGACATCGTACTACACGACACGTACTACG
TTGTAGCCCACTTCCACTATGTCCTATCAATAGGAGCTGTATTTGCCATCATAGGAGGCT
TCATTCACTGATTTCCCCTATTCTCAGGCTACACCCTAGACCAAACCTACGCCAAAATCC
ATTTCACTATCATATTCATCGGCGTAAATCTAACTTTCTTCCCACAACACTTTCTCGGCC
TATCCGGAATGCCCCGACGTTACTCGGACTACCCCGATGCATACACCACATGAAACATCC
TATCATCTGTAGGCTCATTCATTTCTCTAACAGCAGTAATATTAATAATTTTCATGATTT
GAGAAGCCTTCGCTTCGAAGCGAAAAGTCCTAATAGTAGAAGAACCCTCCATAAACCTGG
AGTGACTATATGGATGCCCCCCACCCTACCACACATTCGAAGAACCCGTATACATAAAAT
CTAGACAAAAAAGGAAGGAATCGAACCCCCCAAAGCTGGTTTCAAGCCAACCCCATGGCC
TCCATGACTTTTTCAAAAAGGTATTAGAAAAACCATTTCATAACTTTGTCAAAGTTAAAT
TATAGGCTAAATCCTATATATCTTAATGGCACATGCAGCGCAAGTAGGTCTACAAGACGC
TACTTCCCCTATCATAGAAGAGCTTATCACCTTTCATGATCACGCCCTCATAATCATTTT
CCTTATCTGCTTCCTAGTCCTGTATGCCCTTTTCCTAACACTCACAACAAAACTAACTAA
TACTAACATCTCAGACGCTCAGGAAATAGAAACCGTCTGAACTATCCTGCCCGCCATCAT
CCTAGTCCTCATCGCCCTCCCATCCCTACGCATCCTTTACATAACAGACGAGGTCAACGA
TCCCTCCCTTACCATCAAATCAATTGGCCACCAATGGTACTGAACCTACGAGTACACCGA
CTACGGCGGACTAATCTTCAACTCCTACATACTTCCCCCATTATTCCTAGAACCAGGCGA
CCTGCGACTCCTTGACGTTGACAATCGAGTAGTACTCCCGATTGAAGCCCCCATTCGTAT
AATAATTACATCACAAGACGTCTTGCACTCATGAGCTGTCCCCACATTAGGCTTAAAAAC
AGATGCAATTCCCGGACGTCTAAACCAAACCACTTTCACCGCTACACGACCGGGGGTATA
CTACGGTCAATGCTCTGAAATCTGTGGAGCAAACCACAGTTTCATGCCCATCGTCCTAGA
ATTAATTCCCCTAAAAATCTTTGAAATAGGGCCCGTATTTACCCTATAGCACCCCCTCTA
CCCCCTCTAGAGCCCACTGTAAAGCTAACTTAGCATTAACCTTTTAAGTTAAAGATTAAG
AGAACCAACACCTCTTTACAGTGAAATGCCCCAACTAAATACTACCGTATGGCCCACCAT
AATTACCCCCATACTCCTTACACTATTCCTCATCACCCAACTAAAAATATTAAACACAAA
CTACCACCTACCTCCCTCACCAAAGCCCATAAAAATAAAAAATTATAACAAACCCTGAGA
ACCAAAATGAACGAAAATCTGTTCGCTTCATTCATTGCCCCCACAATCCTAGGCCTACCC
GCCGCAGTACTGATCATTCTATTTCCCCCTCTATTGATCCCCACCTCCAAATATCTCATC
AACAACCGACTAATCACCACCCAACAATGACTAATCAAACTAACCTCAAAACAAATGATA
ACCATACACAACACTAAAGGACGAACCTGATCTCTTATACTAGTATCCTTAATCATTTTT
ATTGCCACAACTAACCTCCTCGGACTCCTGCCTCACTCATTTACACCAACCACCCAACTA
TCTATAAACCTAGCCATGGCCATCCCCTTATGAGCGGGCACAGTGATTATAGGCTTTCGC
TCTAAGATTAAAAATGCCCTAGCCCACTTCTTACCACAAGGCACACCTACACCCCTTATC
CCCATACTAGTTATTATCGAAACCATCAGCCTACTCATTCAACCAATAGCCCTGGCCGTA
CGCCTAACCGCTAACATTACTGCAGGCCACCTACTCATGCACCTAATTGGAAGCGCCACC
CTAGCAATATCAACCATTAACCTTCCCTCTACACTTATCATCTTCACAATTCTAATTCTA
CTGACTATCCTAGAAATCGCTGTCGCCTTAATCCAAGCCTACGTTTTCACACTTCTAGTA
AGCCTCTACCTGCACGACAACACATAATGACCCACCAATCACATGCCTATCATATAGTAA
AACCCAGCCCATGACCCCTAACAGGGGCCCTCTCAGCCCTCCTAATGACCTCCGGCCTAG
CCATGTGATTTCACTTCCACTCCATAACGCTCCTCATACTAGGCCTACTAACCAACACAC
TAACCATATACCAATGATGGCGCGATGTAACACGAGAAAGCACATACCAAGGCCACCACA
CACCACCTGTCCAAAAAGGCCTTCGATACGGGATAATCCTATTTATTACCTCAGAAGTTT
TTTTCTTCGCAGGATTTTTCTGAGCCTTTTACCACTCCAGCCTAGCCCCTACCCCCCAAT
TAGGAGGGCACTGGCCCCCAACAGGCATCACCCCGCTAAATCCCCTAGAAGTCCCACTCC
TAAACACATCCGTATTACTCGCATCAGGAGTATCAATCACCTGAGCTCACCATAGTCTAA
TAGAAAACAACCGAAACCAAATAATTCAAGCACTGCTTATTACAATTTTACTGGGTCTCT
ATTTTACCCTCCTACAAGCCTCAGAGTACTTCGAGTCTCCCTTCACCATTTCCGACGGCA
TCTACGGCTCAACATTTTTTGTAGCCACAGGCTTCCACGGACTTCACGTCATTATTGGCT
CAACTTTCCTCACTATCTGCTTCATCCGCCAACTAATATTTCACTTTACATCCAAACATC
ACTTTGGCTTCGAAGCCGCCGCCTGATACTGGCATTTTGTAGATGTGGTTTGACTATTTC
TGTATGTCTCCATCTATTGATGAGGGTCTTACTCTTTTAGTATAAATAGTACCGTTAACT
TCCAATTAACTAGTTTTGACAACATTCAAAAAAGAGTAATAAACTTCGCCTTAATTTTAA
TAATCAACACCCTCCTAGCCTTACTACTAATAATTATTACATTTTGACTACCACAACTCA
ACGGCTACATAGAAAAATCCACCCCTTACGAGTGCGGCTTCGACCCTATATCCCCCGCCC
GCGTCCCTTTCTCCATAAAATTCTTCTTAGTAGCTATTACCTTCTTATTATTTGATCTAG
AAATTGCCCTCCTTTTACCCCTACCATGAGCCCTACAAACAACTAACCTGCCACTAATAG
TTATGTCATCCCTCTTATTAATCATCATCCTAGCCCTAAGTCTGGCCTATGAGTGACTAC
AAAAAGGATTAGACTGAACCGAATTGGTATATAGTTTAAACAAAACGAATGATTTCGACT
CATTAAATTATGATAATCATATTTACCAAATGCCCCTCATTTACATAAATATTATACTAG
CATTTACCATCTCACTTCTAGGAATACTAGTATATCGCTCACACCTCATATCCTCCCTAC
TATGCCTAGAAGGAATAATACTATCGCTGTTCATTATAGCTACTCTCATAACCCTCAACA
CCCACTCCCTCTTAGCCAATATTGTGCCTATTGCCATACTAGTCTTTGCCGCCTGCGAAG
CAGCGGTGGGCCTAGCCCTACTAGTCTCAATCTCCAACACATATGGCCTAGACTACGTAC
ATAACCTAAACCTACTCCAATGCTAAAACTAATCGTCCCAACAATTATATTACTACCACT
GACATGACTTTCCAAAAAACACATAATTTGAATCAACACAACCACCCACAGCCTAATTAT
TAGCATCATCCCTCTACTATTTTTTAACCAAATCAACAACAACCTATTTAGCTGTTCCCC
AACCTTTTCCTCCGACCCCCTAACAACCCCCCTCCTAATACTAACTACCTGACTCCTACC
CCTCACAATCATGGCAAGCCAACGCCACTTATCCAGTGAACCACTATCACGAAAAAAACT
CTACCTCTCTATACTAATCTCCCTACAAATCTCCTTAATTATAACATTCACAGCCACAGA
ACTAATCATATTTTATATCTTCTTCGAAACCACACTTATCCCCACCTTGGCTATCATCAC
CCGATGAGGCAACCAGCCAGAACGCCTGAACGCAGGCACATACTTCCTATTCTACACCCT
AGTAGGCTCCCTTCCCCTACTCATCGCACTAATTTACACTCACAACACCCTAGGCTCACT
AAACATTCTACTACTCACTCTCACTGCCCAAGAACTATCAAACTCCTGAGCCAACAACTT
AATATGACTAGCTTACACAATAGCTTTTATAGTAAAGATACCTCTTTACGGACTCCACTT
ATGACTCCCTAAAGCCCATGTCGAAGCCCCCATCGCTGGGTCAATAGTACTTGCCGCAGT
ACTCTTAAAACTAGGCGGCTATGGTATAATACGCCTCACACTCATTCTCAACCCCCTGAC
AAAACACATAGCCTACCCCTTCCTTGTACTATCCCTATGAGGCATAATTATAACAAGCTC
CATCTGCCTACGACAAACAGACCTAAAATCGCTCATTGCATACTCTTCAATCAGCCACAT
AGCCCTCGTAGTAACAGCCATTCTCATCCAAACCCCCTGAAGCTTCACCGGCGCAGTCAT
TCTCATAATCGCCCACGGGCTTACATCCTCATTACTATTCTGCCTAGCAAACTCAAACTA
CGAACGCACTCACAGTCGCATCATAATCCTCTCTCAAGGACTTCAAACTCTACTCCCACT
AATAGCTTTTTGATGACTTCTAGCAAGCCTCGCTAACCTCGCCTTACCCCCCACTATTAA
CCTACTGGGAGAACTCTCTGTGCTAGTAACCACGTTCTCCTGATCAAATATCACTCTCCT
ACTTACAGGACTCAACATACTAGTCACAGCCCTATACTCCCTCTACATATTTACCACAAC
ACAATGGGGCTCACTCACCCACCACATTAACAACATAAAACCCTCATTCACACGAGAAAA
CACCCTCATGTTCATACACCTATCCCCCATTCTCCTCCTATCCCTCAACCCCGACATCAT
TACCGGGTTTTCCTCTTGTAAATATAGTTTAACCAAAACATCAGATTGTGAATCTGACAA
CAGAGGCTTACGACCCCTTATTTACCGAGAAAGCTCACAAGAACTGCTAACTCATGCCCC
CATGTCTAACAACATGGCTTTCTCAACTTTTAAAGGATAACAGCTATCCATTGGTCTTAG
GCCCCAAAAATTTTGGTGCAACTCCAAATAAAAGTAATAACCATGCACACTACTATAACC
ACCCTAACCCTGACTTCCCTAATTCCCCCCATCCTTACCACCCTCGTTAACCCTAACAAA
AAAAACTCATACCCCCATTATGTAAAATCCATTGTCGCATCCACCTTTATTATCAGTCTC
TTCCCCACAACAATATTCATGTGCCTAGACCAAGAAGTTATTATCTCGAACTGACACTGA
GCCACAACCCAAACAACCCAGCTCTCCCTAAGCTTCAAACTAGACTACTTCTCCATAATA
TTCATCCCTGTAGCATTGTTCGTTACATGGTCCATCATAGAATTCTCACTGTGATATATA
AACTCAGACCCAAACATTAATCAGTTCTTCAAATATCTACTCATCTTCCTAATTACCATA
CTAATCTTAGTTACCGCTAACAACCTATTCCAACTGTTCATCGGCTGAGAGGGCGTAGGA
ATTATATCCTTCTTGCTCATCAGTTGATGATACGCCCGAGCAGATGCCAACACAGCAGCC
ATTCAAGCAATCCTATACAACCGTATCGGCGATATCGGTTTCATCCTCGCCTTAGCATGA
TTTATCCTACACTCCAACTCATGAGACCCACAACAAATAGCCCTTCTAAACGCTAATCCA
AGCCTCACCCCACTACTAGGCCTCCTCCTAGCAGCAGCAGGCAAATCAGCCCAATTAGGT
CTCCACCCCTGACTCCCCTCAGCCATAGAAGGCCCCACCCCAGTCTCAGCCCTACTCCAC
TCAAGCACTATAGTTGTAGCAGGAATCTTCTTACTCATCCGCTTCCACCCCCTAGCAGAA
AATAGCCCACTAATCCAAACTCTAACACTATGCTTAGGCGCTATCACCACTCTGTTCGCA
GCAGTCTGCGCCCTTACACAAAATGACATCAAAAAAATCGTAGCCTTCTCCACTTCAAGT
CAACTAGGACTCATAATAGTTACAATCGGCATCAACCAACCACACCTAGCATTCCTGCAC
ATCTGTACCCACGCCTTCTTCAAAGCCATACTATTTATGTGCTCCGGGTCCATCATCCAC
AACCTTAACAATGAACAAGATATTCGAAAAATAGGAGGACTACTCAAAACCATACCTCTC
ACTTCAACCTCCCTCACCATTGGCAGCCTAGCATTAGCAGGAATACCTTTCCTCACAGGT
TTCTACTCCAAAGACCACATCATCGAAACCGCAAACATATCATACACAAACGCCTGAGCC
CTATCTATTACTCTCATCGCTACCTCCCTGACAAGCGCCTATAGCACTCGAATAATTCTT
CTCACCCTAACAGGTCAACCTCGCTTCCCCACCCTTACTAACATTAACGAAAATAACCCC
ACCCTACTAAACCCCATTAAACGCCTGGCAGCCGGAAGCCTATTCGCAGGATTTCTCATT
ACTAACAACATTTCCCCCGCATCCCCCTTCCAAACAACAATCCCCCTCTACCTAAAACTC
ACAGCCCTCGCTGTCACTTTCCTAGGACTTCTAACAGCCCTAGACCTCAACTACCTAACC
AACAAACTTAAAATAAAATCCCCACTATGCACATTTTATTTCTCCAACATACTCGGATTC
TACCCTAGCATCACACACCGCACAATCCCCTATCTAGGCCTTCTTACGAGCCAAAACCTG
CCCCTACTCCTCCTAGACCTAACCTGACTAGAAAAGCTATTACCTAAAACAATTTCACAG
CACCAAATCTCCACCTCCATCATCACCTCAACCCAAAAAGGCATAATTAAACTTTACTTC
CTCTCTTTCTTCTTCCCACTCATCCTAACCCTACTCCTAATCACATAACCTATTCCCCCG
AGCAATCTCAATTACAATATATACACCAACAAACAATGTTCAACCAGTAACTACTACTAA
TCAACGCCCATAATCATACAAAGCCCCCGCACCAATAGGATCCTCCCGAATCAACCCTGA
CCCCTCTCCTTCATAAATTATTCAGCTTCCTACACTATTAAAGTTTACCACAACCACCAC
CCCATCATACTCTTTCACCCACAGCACCAATCCTACCTCCATCGCTAACCCCACTAAAAC
ACTCACCAAGACCTCAACCCCTGACCCCCATGCCTCAGGATACTCCTCAATAGCCATCGC
TGTAGTATATCCAAAGACAACCATCATTCCCCCTAAATAAATTAAAAAAACTATTAAACC
CATATAACCTCCCCCAAAATTCAGAATAATAACACACCCGACCACACCGCTAACAATCAA
TACTAAACCCCCATAAATAGGAGAAGGCTTAGAAGAAAACCCCACAAACCCCATTACTAA
ACCCACACTCAACAGAAACAAAGCATACATCATTATTCTCGCACGGACTACAACCACGAC
CAATGATATGAAAAACCATCGTTGTATTTCAACTACAAGAACACCAATGACCCCAATACG
CAAAACTAACCCCCTAATAAAATTAATTAACCACTCATTCATCGACCTCCCCACCCCATC
CAACATCTCCGCATGATGAAACTTCGGCTCACTCCTTGGCGCCTGCCTGATCCTCCAAAT
CACCACAGGACTATTCCTAGCCATGCACTACTCACCAGACGCCTCAACCGCCTTTTCATC
AATCGCCCACATCACTCGAGACGTAAATTATGGCTGAATCATCCGCTACCTTCACGCCAA
TGGCGCCTCAATATTCTTTATCTGCCTCTTCCTACACATCGGGCGAGGCCTATATTACGG
ATCATTTCTCTACTCAGAAACCTGAAACATCGGCATTATCCTCCTGCTTGCAACTATAGC
AACAGCCTTCATAGGCTATGTCCTCCCGTGAGGCCAAATATCATTCTGAGGGGCCACAGT
AATTACAAACTTACTATCCGCCATCCCATACATTGGGACAGACCTAGTTCAATGAATCTG
AGGAGGCTACTCAGTAGACAGTCCCACCCTCACACGATTCTTTACCTTTCACTTCATCTT
GCCCTTCATTATTGCAGCCCTAGCAACACTCCACCTCCTATTCTTGCACGAAACGGGATC
AAACAACCCCCTAGGAATCACCTCCCATTCCGATAAAATCACCTTCCACCCTTACTACAC
AATCAAAGACGCCCTCGGCTTACTTCTCTTCCTTCTCTCCTTAATGACATTAACACTATT
CTCACCAGACCTCCTAGGCGACCCAGACAATTATACCCTAGCCAACCCCTTAAACACCCC
TCCCCACATCAAGCCCGAATGATATTTCCTATTCGCCTACACAATTCTCCGATCCGTCCC
TAACAAACTAGGAGGCGTCCTTGCCCTATTACTATCCATCCTCATCCTAGCAATAATCCC
CATCCTCCATATATCCAAACAACAAAGCATAATATTTCGCCCACTAAGCCAATCACTTTA
TTGACTCCTAGCCGCAGACCTCCTCATTCTAACCTGAATCGGAGGACAACCAGTAAGCTA
CCCTTTTACCATCATTGGACAAGTAGCATCCGTACTATACTTCACAACAATCCTAATCCT
AATACCAACTATCTCCCTAATTGAAAACAAAATACTCAAATGGGCCTGTCCTTGTAGTAT
AAACTAATACACCAGTCTTGTAAACCGGAGATGAAAACCTTTTTCCAAGGACAAATCAGA
GAAAAAGTCTTTAACTCCACCATTAGCACCCAAAGCTAAGATTCTAATTTAAACTATTCT
CTGTTCTTTCATGGGGAAGCAGATTTGGGTACCACCCAAGTATTGACTCACCCATCAACA
ACCGCTATGTATTTCGTACATTACTGCCAGCCACCATGAATATTGTACGGTACCATAAAT
ACTTGACCACCTGTAGTACATAAAAACCCAATCCACATCAAAACCCCCTCCCCATGCTTA
CAAGCAAGTACAGCAATCAACCCTCAACTATCACACATCAACTGCAACTCCAAAGCCACC
CCTCACCCACTAGGATACCAACAAACCTACCCACCCTTAACAGTACATAGTACATAAAGC
CATTTACCGTACATAGCACATTACAGTCAAATCCCTTCTCGTCCCCATGGATGACCCCCC
TCAGATAGGGGTCCCTTGACCACCATCCTCCGTGAAATCAATATCCCGCACAAGAGTGCT
ACTCTCCTCGCTCCGGGCCCATAACACTTGGGGGTAGCTAAAGTGAACTGTATCCGACAT
CTGGTTCCTACTTCAGGGTCATAAAGCCTAAATAGCCCACACGTTCCCCTTAAATAAGAC
ATCACGATG
REFSEQ

    $r =~ s/\n//g;
    unless ($self->revised) {
        # remove the 3107 placeholder position unless we want the revised seq
        $r = substr($r, 0, 3106) . substr($r, 3107);
    }
    return $r;
}#FOLDEND

sub _ref_seq_lookup {#FOLDBEG
    my ($orig, $self, $b, $e) = @_;

    if ( !(defined $b) and !(defined $e ) ){ return $self->$orig(); }

    return $self->_seq_string($self->$orig, $b, $e);
}#FOLDEND



=head2 revised

A flag that can be set so that the ref_seq attribute will include the 3107 placeholder and use the numbering system of the rCRS.

=cut

has 'revised' => (
    is => 'ro',
    default => 0,
);



=head2 codon_code

The human mitochondrial translation code that maps codons to their amino acid.  Note that the mitochondrial code has some exceptions to the standard code.  There are key values for both thymine and uracil codes, so you can use either one.

 $r = Bio::Mitomaster::RefHuman->new();
 $c = $r->codon_code('TTT');  # $c is now 'F'
 $c = $r->codon_code('UUU');  # same thing
 $c = $r->codon_code();  # $c is a hash reference to a codon lookup table

=cut

has 'codon_code' => (
    is => 'ro',
    lazy_build => 1,
);

around 'codon_code' => \&_codon_lookup;


sub _build_codon_code {#FOLDBEG
    my $self = shift;
    my %code = (
	'TTT'=>'F','TTC'=>'F','TTA'=>'L','TTG'=>'L','CTT'=>'L','CTC'=>'L','CTA'=>'L',
	'CTG'=>'L','ATT'=>'I','ATC'=>'I','ATA'=>'M','ATG'=>'M','GTT'=>'V','GTC'=>'V',
	'GTA'=>'V','GTG'=>'V','TCT'=>'S','TCC'=>'S','TCA'=>'S','TCG'=>'S','CCT'=>'P',
	'CCC'=>'P','CCA'=>'P','CCG'=>'P','ACT'=>'T','ACC'=>'T','ACA'=>'T','ACG'=>'T',
	'GCT'=>'A','GCC'=>'A','GCA'=>'A','GCG'=>'A','TAT'=>'Y','TAC'=>'Y','TAA'=>'TERM',
	'TAG'=>'TERM','CAT'=>'H','CAC'=>'H','CAA'=>'Q','CAG'=>'Q','AAT'=>'N','AAC'=>'N',
	'AAA'=>'K','AAG'=>'K','GAT'=>'D','GAC'=>'D','GAA'=>'E','GAG'=>'E','TGT'=>'C',
	'TGC'=>'C','TGA'=>'W','TGG'=>'W','CGT'=>'R','CGC'=>'R','CGA'=>'R','CGG'=>'R',
	'AGT'=>'S','AGC'=>'S','AGA'=>'TERM','AGG'=>'TERM','GGT'=>'G','GGC'=>'G',
	'GGA'=>'G','GGG'=>'G',
	'UUU'=>'F','UUC'=>'F','UUA'=>'L','UUG'=>'L','CUU'=>'L','CUC'=>'L','CUA'=>'L',
	'CUG'=>'L','AUU'=>'I','AUC'=>'I','AUA'=>'M','AUG'=>'M','GUU'=>'V','GUC'=>'V',
	'GUA'=>'V','GUG'=>'V','UCU'=>'S','UCC'=>'S','UCA'=>'S','UCG'=>'S','CCU'=>'P',
	'CCC'=>'P','CCA'=>'P','CCG'=>'P','ACU'=>'T','ACC'=>'T','ACA'=>'T','ACG'=>'T',
	'GCU'=>'A','GCC'=>'A','GCA'=>'A','GCG'=>'A','UAU'=>'Y','UAC'=>'Y','UAA'=>'TERM',
	'UAG'=>'TERM','CAU'=>'H','CAC'=>'H','CAA'=>'Q','CAG'=>'Q','AAU'=>'N','AAC'=>'N',
	'AAA'=>'K','AAG'=>'K','GAU'=>'D','GAC'=>'D','GAA'=>'E','GAG'=>'E','UGU'=>'C',
	'UGC'=>'C','UGA'=>'W','UGG'=>'W','CGU'=>'R','CGC'=>'R','CGA'=>'R','CGG'=>'R',
	'AGU'=>'S','AGC'=>'S','AGA'=>'TERM','AGG'=>'TERM','GGU'=>'G','GGC'=>'G',
	'GGA'=>'G','GGG'=>'G'
    );

    return \%code;
}#FOLDEND


sub _codon_lookup {#FOLDBEG
    my ($orig, $self, $code) = @_;

    my $c = $self->$orig;
    return $c unless defined $code;

    croak "$code is not a valid codon" unless $c->{$code};

    return $c->{$code};
}#FOLDEND



=head2 locus

The human mitochondrial locus regions.  Contains information for these regions.  Locus regions are indexed by ID# and name.
 $r = Bio::Mitomaster::RefHuman->new();
 $l = $r->locus(16);  # $l is a hashref for information about the Cytochrome oxidase subunit I locus
 $l = $r->locus('COI');  # same thing
 $s = $l->{starting};  # $s is 5903, the starting for position for MTCO1
 $l = $r->locus();  # $l is a hashref to a hash that contains all the locus hashrefs indexed by ID#
Each locus hashref contains the keys: name, common_name, starting, ending, strand, type, product.  Note that position values for the loci will be adjusted depending upong the value of the revised attribute.

=cut

has 'locus' => (
    is => 'ro',
    lazy_build => 1,
);

around 'locus' => \&_locus_lookup;


sub _build_locus {#FOLDBEG

    my $self = shift;

    my %locus_table = (
	38=>{name=>'MTHV2',common_name=>'HVS2',starting=>'57',ending=>'372',strand=>'N',type=>'n',product=>'Hypervariable segment 2'},
	39=>{name=>'MTOHR',common_name=>'OH',starting=>'110',ending=>'441',strand=>'H',type=>'n',product=>'H-strand origin'},
	40=>{name=>'MTCSB1',common_name=>'CSB1',starting=>'213',ending=>'235',strand=>'N',type=>'n',product=>'Conserved sequence block 1'},
	41=>{name=>'MTTFX',common_name=>'TFX',starting=>'233',ending=>'260',strand=>'N',type=>'n',product=>'mtTF1 binding site'},
	42=>{name=>'MTTFY',common_name=>'TFY',starting=>'276',ending=>'303',strand=>'N',type=>'n',product=>'mtTF1 binding site'},
	43=>{name=>'MTCSB2',common_name=>'CSB2',starting=>'299',ending=>'315',strand=>'N',type=>'n',product=>'Conserved sequence block 2'},
	44=>{name=>'MTHPR',common_name=>'HPR',starting=>'317',ending=>'321',strand=>'N',type=>'n',product=>'replication primer'},
	45=>{name=>'MTCSB3',common_name=>'CSB3',starting=>'346',ending=>'363',strand=>'N',type=>'n',product=>'Conserved sequence block 3'},
	46=>{name=>'MTMT4H',common_name=>'mt4H',starting=>'371',ending=>'379',strand=>'H',type=>'n',product=>'mt4 H-strand control element'},
	47=>{name=>'MTMT3H',common_name=>'mt3H',starting=>'384',ending=>'391',strand=>'H',type=>'n',product=>'mt3 H-strand control element'},
	48=>{name=>'MTLSP',common_name=>'PL',starting=>'392',ending=>'445',strand=>'L',type=>'n',product=>'L-strand promoter'},
	49=>{name=>'MTTFL',common_name=>'TFL',starting=>'418',ending=>'445',strand=>'N',type=>'n',product=>'mtTF1 binding site'},
	50=>{name=>'MTTFH',common_name=>'TFH',starting=>'523',ending=>'550',strand=>'N',type=>'n',product=>'mtTF1 binding site'},
	51=>{name=>'MTHSP1',common_name=>'PH1',starting=>'545',ending=>'567',strand=>'H',type=>'n',product=>'Major H-strand promoter'},
	1=>{name=>'MTTF',common_name=>'F',starting=>'577',ending=>'647',strand=>'H',type=>'t',product=>'tRNA phenylanine'},
	52=>{name=>'MTHSP2',common_name=>'PH2',starting=>'645',ending=>'645',strand=>'H',type=>'n',product=>'Minor H-strand promoter'},
	2=>{name=>'MTRNR1',common_name=>'12S',starting=>'648',ending=>'1601',strand=>'H',type=>'r',product=>'12S ribosomal RNA'},
	3=>{name=>'MTTV',common_name=>'V',starting=>'1602',ending=>'1670',strand=>'H',type=>'t',product=>'tRNA valine'},
	4=>{name=>'MTRNR2',common_name=>'16S',starting=>'1671',ending=>'3229',strand=>'H',type=>'r',product=>'16S ribosomal RNA'},
	53=>{name=>'MTRNR3',common_name=>'RNR3',starting=>'3205',ending=>'3229',strand=>'N',type=>'n',product=>'5S-like sequence'},
	54=>{name=>'MTTER',common_name=>'TER',starting=>'3228',ending=>'3255',strand=>'N',type=>'n',product=>'Transcription terminator'},
	5=>{name=>'MTTL1',common_name=>'L(UUA/G)',starting=>'3229',ending=>'3303',strand=>'H',type=>'t',product=>'tRNA leucine 1'},
	55=>{name=>'MTNC1',common_name=>'NC1',starting=>'3304',ending=>'3305',strand=>'N',type=>'n',product=>'non-coding nucleotides between MTTL1 and MTND1'},
	6=>{name=>'MTND1',common_name=>'ND1',starting=>'3306',ending=>'4261',strand=>'H',type=>'m',product=>'NADH Dehydrogenase subunit 1'},
	7=>{name=>'MTTI',common_name=>'I',starting=>'4262',ending=>'4330',strand=>'H',type=>'t',product=>'tRNA isoleucine'},
	8=>{name=>'MTTQ',common_name=>'Q',starting=>'4328',ending=>'4399',strand=>'L',type=>'t',product=>'tRNA glutamine'},
	56=>{name=>'MTNC2',common_name=>'NC2',starting=>'4400',ending=>'4400',strand=>'N',type=>'n',product=>'non-coding nucleotide between MTTQ and MTTM'},
	9=>{name=>'MTTM',common_name=>'M',starting=>'4401',ending=>'4468',strand=>'H',type=>'t',product=>'tRNA methionine'},
	10=>{name=>'MTND2',common_name=>'ND2',starting=>'4469',ending=>'5510',strand=>'H',type=>'m',product=>'NADH dehydrogenase subunit 2'},
	11=>{name=>'MTTW',common_name=>'W',starting=>'5511',ending=>'5578',strand=>'H',type=>'t',product=>'tRNA tryptophan'},
	57=>{name=>'MTNC3',common_name=>'NC3',starting=>'5576',ending=>'5585',strand=>'N',type=>'n',product=>'non-coding nucleotides between MTTW and MTTA'},
	12=>{name=>'MTTA',common_name=>'A',starting=>'5586',ending=>'5654',strand=>'L',type=>'t',product=>'tRNA alanine'},
	58=>{name=>'MTNC4',common_name=>'NC4',starting=>'5655',ending=>'5655',strand=>'N',type=>'n',product=>'non-coding nucleotide between MTTA and MTTN'},
	13=>{name=>'MTTN',common_name=>'N',starting=>'5656',ending=>'5728',strand=>'L',type=>'t',product=>'tRNA asparagine'},
	59=>{name=>'MTOLR',common_name=>'OL',starting=>'5720',ending=>'5797',strand=>'L',type=>'n',product=>'L-strand origin'},
	14=>{name=>'MTTC',common_name=>'C',starting=>'5760',ending=>'5825',strand=>'L',type=>'t',product=>'tRNA cysteine'},
	15=>{name=>'MTTY',common_name=>'Y',starting=>'5825',ending=>'5890',strand=>'L',type=>'t',product=>'tRNA tyrosine'},
	60=>{name=>'MTNC5',common_name=>'NC5',starting=>'5891',ending=>'5902',strand=>'N',type=>'n',product=>'non-coding nucleotides between MTTY and MTCO1'},
	16=>{name=>'MTCO1',common_name=>'COI',starting=>'5903',ending=>'7444',strand=>'H',type=>'m',product=>'Cytochrome c oxidase subunit I'},
	17=>{name=>'MTTS1',common_name=>'S(UCN)',starting=>'7445',ending=>'7515',strand=>'L',type=>'t',product=>'tRNA serine 1'},
	61=>{name=>'MTNC6',common_name=>'NC6',starting=>'7516',ending=>'7516',strand=>'N',type=>'n',product=>'non-coding nucleotide between MTTS1 and MTTD'},
	18=>{name=>'MTTD',common_name=>'D',starting=>'7517',ending=>'7584',strand=>'H',type=>'t',product=>'tRNA aspartic acid'},
	19=>{name=>'MTCO2',common_name=>'COII',starting=>'7585',ending=>'8268',strand=>'H',type=>'m',product=>'Cytochrome c oxidase subunit II'},
	62=>{name=>'MTNC7',common_name=>'NC7',starting=>'8269',ending=>'8293',strand=>'N',type=>'n',product=>'non-coding nucleotides between MTCO2 and MTTK'},
	20=>{name=>'MTTK',common_name=>'K',starting=>'8294',ending=>'8363',strand=>'H',type=>'t',product=>'tRNA lysine'},
	63=>{name=>'MTNC8',common_name=>'NC8',starting=>'8364',ending=>'8364',strand=>'N',type=>'n',product=>'non-coding nucleotide between MTTK and MTATP8'},
	21=>{name=>'MTATP8',common_name=>'ATPase8',starting=>'8365',ending=>'8571',strand=>'H',type=>'m',product=>'ATP synthase F0 subunit 8'},
	22=>{name=>'MTATP6',common_name=>'ATPase6',starting=>'8526',ending=>'9206',strand=>'H',type=>'m',product=>'ATP synthase F0 subunit 6'},
	23=>{name=>'MTCO3',common_name=>'COIII',starting=>'9206',ending=>'9989',strand=>'H',type=>'m',product=>'Cytochrome c oxidase subunit III'},
	24=>{name=>'MTTG',common_name=>'G',starting=>'9990',ending=>'10057',strand=>'H',type=>'t',product=>'tRNA glycine'},
	25=>{name=>'MTND3',common_name=>'ND3',starting=>'10058',ending=>'10403',strand=>'H',type=>'m',product=>'NADH dehydrogenase subunit 3'},
	26=>{name=>'MTTR',common_name=>'R',starting=>'10404',ending=>'10468',strand=>'H',type=>'t',product=>'tRNA arginine'},
	27=>{name=>'MTND4L',common_name=>'ND4L',starting=>'10469',ending=>'10765',strand=>'H',type=>'m',product=>'NADH dehydrogenase subunit 4L'},
	28=>{name=>'MTND4',common_name=>'ND4',starting=>'10759',ending=>'12136',strand=>'H',type=>'m',product=>'NADH dehydrogenase subunit 4'},
	29=>{name=>'MTTH',common_name=>'H',starting=>'12137',ending=>'12205',strand=>'H',type=>'t',product=>'tRNA histidine'},
	30=>{name=>'MTTS2',common_name=>'S(AGY)',starting=>'12206',ending=>'12264',strand=>'H',typ=>'t',type=>'tRNA',product=>'serine2'},
	31=>{name=>'MTTL2',common_name=>'L(CUN)',starting=>'12265',ending=>'12335',strand=>'H',type=>'t',product=>'tRNA leucine2'},
	32=>{name=>'MTND5',common_name=>'ND5',starting=>'12336',ending=>'14147',strand=>'H',type=>'m',product=>'NADH dehydrogenase subunit 5'},
	33=>{name=>'MTND6',common_name=>'ND6',starting=>'14148',ending=>'14672',strand=>'L',type=>'m',product=>'NADH dehydrogenase subunit 6'},
	34=>{name=>'MTTE',common_name=>'E',starting=>'14673',ending=>'14741',strand=>'L',type=>'t',product=>'tRNA glutamic acid'},
	64=>{name=>'MTNC9',common_name=>'NC9',starting=>'14742',ending=>'14745',strand=>'N',type=>'n',product=>'non-coding nucleotides between MTTE and MTCYB'},
	35=>{name=>'MTCYB',common_name=>'Cytb',starting=>'14746',ending=>'15886',strand=>'H',type=>'m',product=>'Cytochrome b'},
	36=>{name=>'MTTT',common_name=>'T',starting=>'15887',ending=>'15952',strand=>'H',type=>'t',product=>'tRNA threonine'},
	65=>{name=>'MTATT',common_name=>'ATT',starting=>'15924',ending=>'499',strand=>'N',type=>'n',product=>'membrane attachment site'},
	66=>{name=>'MTNC10',common_name=>'NC10',starting=>'15953',ending=>'15953',strand=>'N',type=>'n',product=>'non-coding nucleotide between MTTT and MTTP'},
	37=>{name=>'MTTP',common_name=>'P',starting=>'15954',ending=>'16022',strand=>'L',type=>'t',product=>'tRNA proline'},
	67=>{name=>'MTDLOOP',common_name=>'D-Loop',starting=>'16023',ending=>'576',strand=>'N',type=>'n',product=>'Displacement loop'},
	68=>{name=>'MTHV1',common_name=>'HVS1',starting=>'16023',ending=>'16382',strand=>'N',type=>'n',product=>'Hypervariable segment 1'},
	69=>{name=>'MT7SDNA',common_name=>'7S DNA',starting=>'16105',ending=>'191',strand=>'N',type=>'n',product=>'7S DNA'},
	70=>{name=>'MTTAS',common_name=>'TAS',starting=>'16156',ending=>'16171',strand=>'N',type=>'n',product=>'termination-associated sequence'},
	71=>{name=>'MTMT5',common_name=>'mt5',starting=>'16193',ending=>'16207',strand=>'N',type=>'n',product=>'control element'},
	72=>{name=>'MTMT3L',common_name=>'mt3L',starting=>'16498',ending=>'16505',strand=>'L',type=>'n',product=>'L-strand control element'},
    );

    if ($self->revised) {
        # adjust position numbers for the rCRS numbering
        for (1..72) {
            $locus_table{$_}->{'starting'}++ if $locus_table{$_}->{'starting'} > 3106;
            $locus_table{$_}->{'ending'}++ if $locus_table{$_}->{'ending'} > 3106;
        }
    }

    return \%locus_table;
}#FOLDEND

sub _locus_lookup {#FOLDBEG
    my ($orig, $self, $index) = @_;

    my $l = $self->$orig;
    return $l unless defined $index;


    my $locus_id = $self->_get_locus_id($index);
    $index = $locus_id if $locus_id;

    croak "$index not a valid locus index" unless $index =~ /^\d+$/;
    croak "$index not within the range 1-72 for locus indices" unless $index > 0 and $index <= 72;

    return $l->{$index};

}#FOLDEND



=head2 protein

The human mitochondrially encoded proteins.  Contains information for these.  Proteins are indexed by locus ID# and name.
 $r = Bio::Mitomaster::RefHuman->new();
 $p = $r->protein(16);  # $p is a hashref for information about the Cytochrome oxidase subunit I
 $p = $r->protein('COI');  # same thing
 $w = $p->{predicted_weight};  # $w is 57000, the molecular weight predicted for Cytochrome oxidase subunit I
 $p = $r->protein();  # $l is a hashref to a hash that contains all the protein hashrefs indexed by locus ID#
Each protein hashref contains the keys:  id, predicted_weight, glycine_number, glycine_weight, urea_number, urea_weight, name, npid, and giid.

=cut


has 'protein' => (
    is => 'ro',
    lazy_build => 1,
);

around 'protein' => \&_protein_lookup;


sub _build_protein {#FOLDBEG

    my $self = shift;

    my %protein_table = (
	    32=>{id=>1, predicted_weight=>66600, glycine_number=>1, glycine_weight=>43.5, 
	        urea_number=>1, urea_weight=>51, name=>'NADH dehydrogenase subunit 5', 
	        npid=>'NP_536853.1', giid=>'GI:17981863'}, 
	    16=>{id=>2, predicted_weight=>57000, glycine_number=>2, glycine_weight=>39.5, 
	        urea_number=>23, urea_weight=>45, name=>'cytochrome c oxidase subunit I', 
	        npid=>'NP_536845.1', giid=>'GI:17981855'}, 
	    28=>{id=>3, predicted_weight=>51400, glycine_number=>3, glycine_weight=>36.5, 
	        urea_number=>45, urea_weight=>'39, 36', name=>'NADH dehydrogenase subunit 4', 
	        npid=>'NP_536852.1', giid=>'GI:17981862'}, 
	    35=>{id=>4, predicted_weight=>42700, glycine_number=>6, glycine_weight=>27.5, 
	        urea_number=>'7-10', urea_weight=>29, name=>'cytochrome b', npid=>'NP_536855.1', 
	        giid=>'GI:17981865'}, 
	    10=>{id=>5, predicted_weight=>38900, glycine_number=>4, glycine_weight=>31.5, 
	        urea_number=>11, urea_weight=>25, name=>'NADH dehydrogenase subunit 2', 
	        npid=>'NP_536844.1', giid=>'GI:17981854'}, 
	    6=>{id=>6, predicted_weight=>35600, glycine_number=>5, glycine_weight=>29.5, 
	        urea_number=>12, urea_weight=>24, 
	        name=>'NADH dehydrogenase subunit 1', npid=>'NP_536843.1', giid=>'GI:17981853'}, 
	    23=>{id=>7, predicted_weight=>30000, glycine_number=>8, glycine_weight=>22.5, 
	        urea_number=>'15, 16', urea_weight=>18, name=>'cytochrome c oxidase subunit 3', 
	        npid=>'NP_536849.1', giid=>'GI:17981859'}, 
	    19=>{id=>8, predicted_weight=>25500, glycine_number=>7, glycine_weight=>23.6, 
	        urea_number=>'13, 14', urea_weight=>20, name=>'cytochrome c oxidase subunit 2', 
	        npid=>'NP_536846.1', giid=>'GI:17981856'}, 
	    22=>{id=>9, predicted_weight=>24800, glycine_number=>9, glycine_weight=>21.6, 
	        urea_number=>'17, 18', urea_weight=>16, name=>'ATP synthase F0 subunit 6', 
	        npid=>'NP_536848.1', giid=>'GI:17981858'}, 
	    33=>{id=>10, predicted_weight=>18600, glycine_number=>10, glycine_weight=>16.7, 
	        urea_number=>, urea_weight=>, name=>'NADH dehydrogenase subunit 6', 
	        npid=>'NP_536854.1', giid=>'GI:17981864'}, 
	    25=>{id=>11, predicted_weight=>13200, glycine_number=>12, glycine_weight=>13.5, 
	        urea_number=>'23, 24', urea_weight=>6, name=>'NADH dehydrogenase subunit 3', 
	        npid=>'NP_536850.1', giid=>'GI:17981860'}, 
	    27=>{id=>12, predicted_weight=>10700, glycine_number=>11, glycine_weight=>14.8, 
	        urea_number=>26, urea_weight=>3.5, name=>'NADH dehydrogenase subunit 4L', 
	        npid=>'NP_536851.1', giid=>'GI:17981861'}, 
	    21=>{id=>13, predicted_weight=>7900, glycine_number=>'13+13a', glycine_weight=>9.8, 
	        urea_number=>25, urea_weight=>4.5, name=>'ATP synthase F0 subunit 8', 
	        npid=>'NP_536847.1', giid=>'GI:17981857'}
    );

    return \%protein_table;
}#FOLDEND

sub _protein_lookup {#FOLDBEG
    my ($orig, $self, $index) = @_;

    my $p = $self->$orig;
    return $p unless defined $index;

    my $locus_id = $self->_get_locus_id($index);
    $index = $locus_id if $locus_id;

    croak "$index not a valid locus index" unless $index =~ /^\d+$/;
    croak "$index not within the range 1-72 for locus indices" unless $index > 0 and $index <= 72;

    return $p->{$index};

}#FOLDEND



=head2 transcript

The human mitochondrial transcript sequences.  Values are the RNA strings that would be seen following transcription: compensation has been made for the strand on which the locus is encoded, uracil replaces thymine, and any poly-adenlyation.

    $r = Bio::Mitomaster::RefHuman->new();
    $t = $r->transcript(16);  # $t contains the transcript string for the Cytochrome oxidase subunit I
    $t = $r->ref_seq(16,1,10);  # same thing but only the first 10 nucleotides
    $t = $r->transcript();  # $t is a hashref to a hash that contains all the mitochondrially encoded transcripts indexed by locus ID#

The options for retrieving sub string values of a transcript is documented in the section Seq String Attributes.

=cut


has 'transcript' => (
    is => 'ro',
    lazy_build => 1,
);

around 'transcript' => \&_transcript_lookup;


sub _build_transcript {#FOLDBEG

    my $self = shift;

    my %transcript_table = (
	22=>'AUGAACGAAAAUCUGUUCGCUUCAUUCAUUGCCCCCACAAUCCUAGGCCUACCCGCCGCAGUACUGAUCAUUCUAUUUCCCCCUCUAUUGAUCCCCACCUCCAAAUAUCUCAUCAACAACCGACUAAUCACCACCCAACAAUGACUAAUCAAACUAACCUCAAAACAAAUGAUAACCAUACACAACACUAAAGGACGAACCUGAUCUCUUAUACUAGUAUCCUUAAUCAUUUUUAUUGCCACAACUAACCUCCUCGGACUCCUGCCUCACUCAUUUACACCAACCACCCAACUAUCUAUAAACCUAGCCAUGGCCAUCCCCUUAUGAGCGGGCACAGUGAUUAUAGGCUUUCGCUCUAAGAUUAAAAAUGCCCUAGCCCACUUCUUACCACAAGGCACACCUACACCCCUUAUCCCCAUACUAGUUAUUAUCGAAACCAUCAGCCUACUCAUUCAACCAAUAGCCCUGGCCGUACGCCUAACCGCUAACAUUACUGCAGGCCACCUACUCAUGCACCUAAUUGGAAGCGCCACCCUAGCAAUAUCAACCAUUAACCUUCCCUCUACACUUAUCAUCUUCACAAUUCUAAUUCUACUGACUAUCCUAGAAAUCGCUGUCGCCUUAAUCCAAGCCUACGUUUUCACACUUCUAGUAAGCCUCUACCUGCACGACAACACAUAA',
	32=>'AUAACCAUGCACACUACUAUAACCACCCUAACCCUGACUUCCCUAAUUCCCCCCAUCCUUACCACCCUCGUUAACCCUAACAAAAAAAACUCAUACCCCCAUUAUGUAAAAUCCAUUGUCGCAUCCACCUUUAUUAUCAGUCUCUUCCCCACAACAAUAUUCAUGUGCCUAGACCAAGAAGUUAUUAUCUCGAACUGACACUGAGCCACAACCCAAACAACCCAGCUCUCCCUAAGCUUCAAACUAGACUACUUCUCCAUAAUAUUCAUCCCUGUAGCAUUGUUCGUUACAUGGUCCAUCAUAGAAUUCUCACUGUGAUAUAUAAACUCAGACCCAAACAUUAAUCAGUUCUUCAAAUAUCUACUCAUCUUCCUAAUUACCAUACUAAUCUUAGUUACCGCUAACAACCUAUUCCAACUGUUCAUCGGCUGAGAGGGCGUAGGAAUUAUAUCCUUCUUGCUCAUCAGUUGAUGAUACGCCCGAGCAGAUGCCAACACAGCAGCCAUUCAAGCAAUCCUAUACAACCGUAUCGGCGAUAUCGGUUUCAUCCUCGCCUUAGCAUGAUUUAUCCUACACUCCAACUCAUGAGACCCACAACAAAUAGCCCUUCUAAACGCUAAUCCAAGCCUCACCCCACUACUAGGCCUCCUCCUAGCAGCAGCAGGCAAAUCAGCCCAAUUAGGUCUCCACCCCUGACUCCCCUCAGCCAUAGAAGGCCCCACCCCAGUCUCAGCCCUACUCCACUCAAGCACUAUAGUUGUAGCAGGAAUCUUCUUACUCAUCCGCUUCCACCCCCUAGCAGAAAAUAGCCCACUAAUCCAAACUCUAACACUAUGCUUAGGCGCUAUCACCACUCUGUUCGCAGCAGUCUGCGCCCUUACACAAAAUGACAUCAAAAAAAUCGUAGCCUUCUCCACUUCAAGUCAACUAGGACUCAUAAUAGUUACAAUCGGCAUCAACCAACCACACCUAGCAUUCCUGCACAUCUGUACCCACGCCUUCUUCAAAGCCAUACUAUUUAUGUGCUCCGGGUCCAUCAUCCACAACCUUAACAAUGAACAAGAUAUUCGAAAAAUAGGAGGACUACUCAAAACCAUACCUCUCACUUCAACCUCCCUCACCAUUGGCAGCCUAGCAUUAGCAGGAAUACCUUUCCUCACAGGUUUCUACUCCAAAGACCACAUCAUCGAAACCGCAAACAUAUCAUACACAAACGCCUGAGCCCUAUCUAUUACUCUCAUCGCUACCUCCCUGACAAGCGCCUAUAGCACUCGAAUAAUUCUUCUCACCCUAACAGGUCAACCUCGCUUCCCCACCCUUACUAACAUUAACGAAAAUAACCCCACCCUACUAAACCCCAUUAAACGCCUGGCAGCCGGAAGCCUAUUCGCAGGAUUUCUCAUUACUAACAACAUUUCCCCCGCAUCCCCCUUCCAAACAACAAUCCCCCUCUACCUAAAACUCACAGCCCUCGCUGUCACUUUCCUAGGACUUCUAACAGCCCUAGACCUCAACUACCUAACCAACAAACUUAAAAUAAAAUCCCCACUAUGCACAUUUUAUUUCUCCAACAUACUCGGAUUCUACCCUAGCAUCACACACCGCACAAUCCCCUAUCUAGGCCUUCUUACGAGCCAAAACCUGCCCCUACUCCUCCUAGACCUAACCUGACUAGAAAAGCUAUUACCUAAAACAAUUUCACAGCACCAAAUCUCCACCUCCAUCAUCACCUCAACCCAAAAAGGCAUAAUUAAACUUUACUUCCUCUCUUUCUUCUUCCCACUCAUCCUAACCCUACUCCUAAUCACAUAA',
	28=>'AUGCUAAAACUAAUCGUCCCAACAAUUAUAUUACUACCACUGACAUGACUUUCCAAAAAACACAUAAUUUGAAUCAACACAACCACCCACAGCCUAAUUAUUAGCAUCAUCCCUCUACUAUUUUUUAACCAAAUCAACAACAACCUAUUUAGCUGUUCCCCAACCUUUUCCUCCGACCCCCUAACAACCCCCCUCCUAAUACUAACUACCUGACUCCUACCCCUCACAAUCAUGGCAAGCCAACGCCACUUAUCCAGUGAACCACUAUCACGAAAAAAACUCUACCUCUCUAUACUAAUCUCCCUACAAAUCUCCUUAAUUAUAACAUUCACAGCCACAGAACUAAUCAUAUUUUAUAUCUUCUUCGAAACCACACUUAUCCCCACCUUGGCUAUCAUCACCCGAUGAGGCAACCAGCCAGAACGCCUGAACGCAGGCACAUACUUCCUAUUCUACACCCUAGUAGGCUCCCUUCCCCUACUCAUCGCACUAAUUUACACUCACAACACCCUAGGCUCACUAAACAUUCUACUACUCACUCUCACUGCCCAAGAACUAUCAAACUCCUGAGCCAACAACUUAAUAUGACUAGCUUACACAAUAGCUUUUAUAGUAAAGAUACCUCUUUACGGACUCCACUUAUGACUCCCUAAAGCCCAUGUCGAAGCCCCCAUCGCUGGGUCAAUAGUACUUGCCGCAGUACUCUUAAAACUAGGCGGCUAUGGUAUAAUACGCCUCACACUCAUUCUCAACCCCCUGACAAAACACAUAGCCUACCCCUUCCUUGUACUAUCCCUAUGAGGCAUAAUUAUAACAAGCUCCAUCUGCCUACGACAAACAGACCUAAAAUCGCUCAUUGCAUACUCUUCAAUCAGCCACAUAGCCCUCGUAGUAACAGCCAUUCUCAUCCAAACCCCCUGAAGCUUCACCGGCGCAGUCAUUCUCAUAAUCGCCCACGGGCUUACAUCCUCAUUACUAUUCUGCCUAGCAAACUCAAACUACGAACGCACUCACAGUCGCAUCAUAAUCCUCUCUCAAGGACUUCAAACUCUACUCCCACUAAUAGCUUUUUGAUGACUUCUAGCAAGCCUCGCUAACCUCGCCUUACCCCCCACUAUUAACCUACUGGGAGAACUCUCUGUGCUAGUAACCACGUUCUCCUGAUCAAAUAUCACUCUCCUACUUACAGGACUCAACAUACUAGUCACAGCCCUAUACUCCCUCUACAUAUUUACCACAACACAAUGGGGCUCACUCACCCACCACAUUAACAACAUAAAACCCUCAUUCACACGAGAAAACACCCUCAUGUUCAUACACCUAUCCCCCAUUCUCCUCCUAUCCCUCAACCCCGACAUCAUUACCGGGUUUUCCUCUUAA',
	25=>'AUAAACUUCGCCUUAAUUUUAAUAAUCAACACCCUCCUAGCCUUACUACUAAUAAUUAUUACAUUUUGACUACCACAACUCAACGGCUACAUAGAAAAAUCCACCCCUUACGAGUGCGGCUUCGACCCUAUAUCCCCCGCCCGCGUCCCUUUCUCCAUAAAAUUCUUCUUAGUAGCUAUUACCUUCUUAUUAUUUGAUCUAGAAAUUGCCCUCCUUUUACCCCUACCAUGAGCCCUACAAACAACUAACCUGCCACUAAUAGUUAUGUCAUCCCUCUUAUUAAUCAUCAUCCUAGCCCUAAGUCUGGCCUAUGAGUGACUACAAAAAGGAUUAGACUGAACCGAAUAA',
	10=>'AUUAAUCCCCUGGCCCAACCCGUCAUCUACUCUACCAUCUUUGCAGGCACACUCAUCACAGCGCUAAGCUCGCACUGAUUUUUUACCUGAGUAGGCCUAGAAAUAAACAUGCUAGCUUUUAUUCCAGUUCUAACCAAAAAAAUAAACCCUCGUUCCACAGAAGCUGCCAUCAAGUAUUUCCUCACGCAAGCAACCGCAUCCAUAAUCCUUCUAAUAGCUAUCCUCUUCAACAAUAUACUCUCCGGACAAUGAACCAUAACCAAUACUACCAAUCAAUACUCAUCAUUAAUAAUCAUAAUAGCUAUAGCAAUAAAACUAGGAAUAGCCCCCUUUCACUUCUGAGUCCCAGAGGUUACCCAAGGCACCCCUCUGACAUCCGGCCUGCUUCUUCUCACAUGACAAAAACUAGCCCCCAUCUCAAUCAUAUACCAAAUCUCUCCCUCACUAAACGUAAGCCUUCUCCUCACUCUCUCAAUCUUAUCCAUCAUAGCAGGCAGUUGAGGUGGAUUAAACCAAACCCAGCUACGCAAAAUCUUAGCAUACUCCUCAAUUACCCACAUAGGAUGAAUAAUAGCAGUUCUACCGUACAACCCUAACAUAACCAUUCUUAAUUUAACUAUUUAUAUUAUCCUAACUACUACCGCAUUCCUACUACUCAACUUAAACUCCAGCACCACGACCCUACUACUAUCUCGCACCUGAAACAAGCUAACAUGACUAACACCCUUAAUUCCAUCCACCCUCCUCUCCCUAGGAGGCCUGCCCCCGCUAACCGGCUUUUUGCCCAAAUGGGCCAUUAUCGAAGAAUUCACAAAAAACAAUAGCCUCAUCAUCCCCACCAUCAUAGCCACCAUCACCCUCCUUAACCUCUACUUCUACCUACGCCUAAUCUACUCCACCUCAAUCACACUACUCCCCAUAUCUAACAACGUAAAAAUAAAAUGACAGUUUGAACAUACAAAACCCACCCCAUUCCUCCCCACACUCAUCGCCCUUACCACGCUACUCCUACCUAUCUCCCCUUUUAUACUAAUAAUCUUAUAA',
	6=>'AUACCCAUGGCCAACCUCCUACUCCUCAUUGUACCCAUUCUAAUCGCAAUGGCAUUCCUAAUGCUUACCGAACGAAAAAUUCUAGGCUAUAUACAACUACGCAAAGGCCCCAACGUUGUAGGCCCCUACGGGCUACUACAACCCUUCGCUGACGCCAUAAAACUCUUCACCAAAGAGCCCCUAAAACCCGCCACAUCUACCAUCACCCUCUACAUCACCGCCCCGACCUUAGCUCUCACCAUCGCUCUUCUACUAUGAACCCCCCUCCCCAUACCCAACCCCCUGGUCAACCUCAACCUAGGCCUCCUAUUUAUUCUAGCCACCUCUAGCCUAGCCGUUUACUCAAUCCUCUGAUCAGGGUGAGCAUCAAACUCAAACUACGCCCUGAUCGGCGCACUGCGAGCAGUAGCCCAAACAAUCUCAUAUGAAGUCACCCUAGCCAUCAUUCUACUAUCAACAUUACUAAUAAGUGGCUCCUUUAACCUCUCCACCCUUAUCACAACACAAGAACACCUCUGAUUACUCCUGCCAUCAUGACCCUUGGCCAUAAUAUGAUUUAUCUCCACACUAGCAGAGACCAACCGAACCCCCUUCGACCUUGCCGAAGGGGAGUCCGAACUAGUCUCAGGCUUCAACAUCGAAUACGCCGCAGGCCCCUUCGCCCUAUUCUUCAUAGCCGAAUACACAAACAUUAUUAUAAUAAACACCCUCACCACUACAAUCUUCCUAGGAACAACAUAUGACGCACUCUCCCCUGAACUCUACACAACAUAUUUUGUCACCAAGACCCUACUUCUAACCUCCCUGUUCUUAUGAAUUCGAACAGCAUACCCCCGAUUCCGCUACGACCAACUCAUACACCUCCUAUGAAAAAACUUCCUACCACUCACCCUAGCAUUACUUAUAUGAUAUGUCUCCAUACCCAUUACAAUCUCCAGCAUUCCCCCUCAAACCUAA',
	16=>'AUGUUCGCCGACCGUUGACUAUUCUCUACAAACCACAAAGACAUUGGAACACUAUACCUAUUAUUCGGCGCAUGAGCUGGAGUCCUAGGCACAGCUCUAAGCCUCCUUAUUCGAGCCGAGCUGGGCCAGCCAGGCAACCUUCUAGGUAACGACCACAUCUACAACGUUAUCGUCACAGCCCAUGCAUUUGUAAUAAUCUUCUUCAUAGUAAUACCCAUCAUAAUCGGAGGCUUUGGCAACUGACUAGUUCCCCUAAUAAUCGGUGCCCCCGAUAUGGCGUUUCCCCGCAUAAACAACAUAAGCUUCUGACUCUUACCUCCCUCUCUCCUACUCCUGCUCGCAUCUGCUAUAGUGGAGGCCGGAGCAGGAACAGGUUGAACAGUCUACCCUCCCUUAGCAGGGAACUACUCCCACCCUGGAGCCUCCGUAGACCUAACCAUCUUCUCCUUACACCUAGCAGGUGUCUCCUCUAUCUUAGGGGCCAUCAAUUUCAUCACAACAAUUAUCAAUAUAAAACCCCCUGCCAUAACCCAAUACCAAACGCCCCUCUUCGUCUGAUCCGUCCUAAUCACAGCAGUCCUACUUCUCCUAUCUCUCCCAGUCCUAGCUGCUGGCAUCACUAUACUACUAACAGACCGCAACCUCAACACCACCUUCUUCGACCCCGCCGGAGGAGGAGACCCCAUUCUAUACCAACACCUAUUCUGAUUUUUCGGUCACCCUGAAGUUUAUAUUCUUAUCCUACCAGGCUUCGGAAUAAUCUCCCAUAUUGUAACUUACUACUCCGGAAAAAAAGAACCAUUUGGAUACAUAGGUAUGGUCUGAGCUAUGAUAUCAAUUGGCUUCCUAGGGUUUAUCGUGUGAGCACACCAUAUAUUUACAGUAGGAAUAGACGUAGACACACGAGCAUAUUUCACCUCCGCUACCAUAAUCAUCGCUAUCCCCACCGGCGUCAAAGUAUUUAGCUGACUCGCCACACUCCACGGAAGCAAUAUGAAAUGAUCUGCUGCAGUGCUCUGAGCCCUAGGAUUCAUCUUUCUUUUCACCGUAGGUGGCCUGACUGGCAUUGUAUUAGCAAACUCAUCACUAGACAUCGUACUACACGACACGUACUACGUUGUAGCCCACUUCCACUAUGUCCUAUCAAUAGGAGCUGUAUUUGCCAUCAUAGGAGGCUUCAUUCACUGAUUUCCCCUAUUCUCAGGCUACACCCUAGACCAAACCUACGCCAAAAUCCAUUUCACUAUCAUAUUCAUCGGCGUAAAUCUAACUUUCUUCCCACAACACUUUCUCGGCCUAUCCGGAAUGCCCCGACGUUACUCGGACUACCCCGAUGCAUACACCACAUGAAACAUCCUAUCAUCUGUAGGCUCAUUCAUUUCUCUAACAGCAGUAAUAUUAAUAAUUUUCAUGAUUUGAGAAGCCUUCGCUUCGAAGCGAAAAGUCCUAAUAGUAGAAGAACCCUCCAUAAACCUGGAGUGACUAUAUGGAUGCCCCCCACCCUACCACACAUUCGAAGAACCCGUAUACAUAAAAUCUAGA',
	19=>'AUGGCACAUGCAGCGCAAGUAGGUCUACAAGACGCUACUUCCCCUAUCAUAGAAGAGCUUAUCACCUUUCAUGAUCACGCCCUCAUAAUCAUUUUCCUUAUCUGCUUCCUAGUCCUGUAUGCCCUUUUCCUAACACUCACAACAAAACUAACUAAUACUAACAUCUCAGACGCUCAGGAAAUAGAAACCGUCUGAACUAUCCUGCCCGCCAUCAUCCUAGUCCUCAUCGCCCUCCCAUCCCUACGCAUCCUUUACAUAACAGACGAGGUCAACGAUCCCUCCCUUACCAUCAAAUCAAUUGGCCACCAAUGGUACUGAACCUACGAGUACACCGACUACGGCGGACUAAUCUUCAACUCCUACAUACUUCCCCCAUUAUUCCUAGAACCAGGCGACCUGCGACUCCUUGACGUUGACAAUCGAGUAGUACUCCCGAUUGAAGCCCCCAUUCGUAUAAUAAUUACAUCACAAGACGUCUUGCACUCAUGAGCUGUCCCCACAUUAGGCUUAAAAACAGAUGCAAUUCCCGGACGUCUAAACCAAACCACUUUCACCGCUACACGACCGGGGGUAUACUACGGUCAAUGCUCUGAAAUCUGUGGAGCAAACCACAGUUUCAUGCCCAUCGUCCUAGAAUUAAUUCCCCUAAAAAUCUUUGAAAUAGGGCCCGUAUUUACCCUAUAG',
	35=>'AUGACCCCAAUACGCAAAACUAACCCCCUAAUAAAAUUAAUUAACCACUCAUUCAUCGACCUCCCCACCCCAUCCAACAUCUCCGCAUGAUGAAACUUCGGCUCACUCCUUGGCGCCUGCCUGAUCCUCCAAAUCACCACAGGACUAUUCCUAGCCAUGCACUACUCACCAGACGCCUCAACCGCCUUUUCAUCAAUCGCCCACAUCACUCGAGACGUAAAUUAUGGCUGAAUCAUCCGCUACCUUCACGCCAAUGGCGCCUCAAUAUUCUUUAUCUGCCUCUUCCUACACAUCGGGCGAGGCCUAUAUUACGGAUCAUUUCUCUACUCAGAAACCUGAAACAUCGGCAUUAUCCUCCUGCUUGCAACUAUAGCAACAGCCUUCAUAGGCUAUGUCCUCCCGUGAGGCCAAAUAUCAUUCUGAGGGGCCACAGUAAUUACAAACUUACUAUCCGCCAUCCCAUACAUUGGGACAGACCUAGUUCAAUGAAUCUGAGGAGGCUACUCAGUAGACAGUCCCACCCUCACACGAUUCUUUACCUUUCACUUCAUCUUGCCCUUCAUUAUUGCAGCCCUAGCAACACUCCACCUCCUAUUCUUGCACGAAACGGGAUCAAACAACCCCCUAGGAAUCACCUCCCAUUCCGAUAAAAUCACCUUCCACCCUUACUACACAAUCAAAGACGCCCUCGGCUUACUUCUCUUCCUUCUCUCCUUAAUGACAUUAACACUAUUCUCACCAGACCUCCUAGGCGACCCAGACAAUUAUACCCUAGCCAACCCCUUAAACACCCCUCCCCACAUCAAGCCCGAAUGAUAUUUCCUAUUCGCCUACACAAUUCUCCGAUCCGUCCCUAACAAACUAGGAGGCGUCCUUGCCCUAUUACUAUCCAUCCUCAUCCUAGCAAUAAUCCCCAUCCUCCAUAUAUCCAAACAACAAAGCAUAAUAUUUCGCCCACUAAGCCAAUCACUUUAUUGACUCCUAGCCGCAGACCUCCUCAUUCUAACCUGAAUCGGAGGACAACCAGUAAGCUACCCUUUUACCAUCAUUGGACAAGUAGCAUCCGUACUAUACUUCACAACAAUCCUAAUCCUAAUACCAACUAUCUCCCUAAUUGAAAACAAAAUACUCAAAUGGGCCUAA',
	23=>'AUGACCCACCAAUCACAUGCCUAUCAUAUAGUAAAACCCAGCCCAUGACCCCUAACAGGGGCCCUCUCAGCCCUCCUAAUGACCUCCGGCCUAGCCAUGUGAUUUCACUUCCACUCCAUAACGCUCCUCAUACUAGGCCUACUAACCAACACACUAACCAUAUACCAAUGAUGGCGCGAUGUAACACGAGAAAGCACAUACCAAGGCCACCACACACCACCUGUCCAAAAAGGCCUUCGAUACGGGAUAAUCCUAUUUAUUACCUCAGAAGUUUUUUUCUUCGCAGGAUUUUUCUGAGCCUUUUACCACUCCAGCCUAGCCCCUACCCCCCAAUUAGGAGGGCACUGGCCCCCAACAGGCAUCACCCCGCUAAAUCCCCUAGAAGUCCCACUCCUAAACACAUCCGUAUUACUCGCAUCAGGAGUAUCAAUCACCUGAGCUCACCAUAGUCUAAUAGAAAACAACCGAAACCAAAUAAUUCAAGCACUGCUUAUUACAAUUUUACUGGGUCUCUAUUUUACCCUCCUACAAGCCUCAGAGUACUUCGAGUCUCCCUUCACCAUUUCCGACGGCAUCUACGGCUCAACAUUUUUUGUAGCCACAGGCUUCCACGGACUUCACGUCAUUAUUGGCUCAACUUUCCUCACUAUCUGCUUCAUCCGCCAACUAAUAUUUCACUUUACAUCCAAACAUCACUUUGGCUUCGAAGCCGCCGCCUGAUACUGGCAUUUUGUAGAUGUGGUUUGACUAUUUCUGUAUGUCUCCAUCUAUUGAUGAGGGUCUUAA',
	21=>'AUGCCCCAACUAAAUACUACCGUAUGGCCCACCAUAAUUACCCCCAUACUCCUUACACUAUUCCUCAUCACCCAACUAAAAAUAUUAAACACAAACUACCACCUACCUCCCUCACCAAAGCCCAUAAAAAUAAAAAAUUAUAACAAACCCUGAGAACCAAAAUGAACGAAAAUCUGUUCGCUUCAUUCAUUGCCCCCACAAUCCUAG',
	33=>'AUGAUGUAUGCUUUGUUUCUGUUGAGUGUGGGUUUAGUAAUGGGGUUUGUGGGGUUUUCUUCUAAGCCUUCUCCUAUUUAUGGGGGUUUAGUAUUGAUUGUUAGCGGUGUGGUCGGGUGUGUUAUUAUUCUGAAUUUUGGGGGAGGUUAUAUGGGUUUAAUAGUUUUUUUAAUUUAUUUAGGGGGAAUGAUGGUUGUCUUUGGAUAUACUACAGCGAUGGCUAUUGAGGAGUAUCCUGAGGCAUGGGGGUCAGGGGUUGAGGUCUUGGUGAGUGUUUUAGUGGGGUUAGCGAUGGAGGUAGGAUUGGUGCUGUGGGUGAAAGAGUAUGAUGGGGUGGUGGUUGUGGUAAACUUUAAUAGUGUAGGAAGCUGAAUAAUUUAUGAAGGAGAGGGGUCAGGGUUGAUUCGGGAGGAUCCUAUUGGUGCGGGGGCUUUGUAUGAUUAUGGGCGUUGAUUAGUAGUAGUUACUGGUUGAACAUUGUUUGUUGGUGUAUAUAUUGUAAUUGAGAUUGCUCGGGGGAAUAGG',
	27=>'AUGCCCCUCAUUUACAUAAAUAUUAUACUAGCAUUUACCAUCUCACUUCUAGGAAUACUAGUAUAUCGCUCACACCUCAUAUCCUCCCUACUAUGCCUAGAAGGAAUAAUACUAUCGCUGUUCAUUAUAGCUACUCUCAUAACCCUCAACACCCACUCCCUCUUAGCCAAUAUUGUGCCUAUUGCCAUACUAGUCUUUGCCGCCUGCGAAGCAGCGGUGGGCCUAGCCCUACUAGUCUCAAUCUCCAACACAUAUGGCCUAGACUACGUACAUAACCUAAACCUACUCCAAUGCUAA'

    );

    return \%transcript_table;
}#FOLDEND

sub _transcript_lookup {#FOLDBEG
    my ($orig, $self, $index, $beg, $end) = @_;

    my $t = $self->$orig;
    return $t unless defined $index;

    my $locus_id = $self->_get_locus_id($index);
    $index = $locus_id if $locus_id;

    croak "$index not a valid transcript index" unless $index =~ /^\d+$/;
    croak "$index not a valid transcript index" unless $t->{$index};

    return $t->{$index} unless $beg or $end;
    return $self->_seq_string($t->{$index}, $beg, $end);
}#FOLDEND



=head2 translation

The human mitochondrial translation sequences.  Values are the amino acid strings (IUPAC single-letter symbols) that would be seen following translation using the transcript strings returned by the transcript method.

    $r = Bio::Mitomaster::RefHuman->new();
    $t = $r->translation(16);  # $t contains the translation string for the Cytochrome oxidase subunit I
    $t = $r->ref_seq(16,1,10);  # same thing but only the first 10 amino acids
    $t = $r->translation();  # $t is a hashref to a hash that contains all the mitochondrially encoded translations indexed by locus ID#

The options for retrieving sub string values of a translation is documented in the section Seq String Attributes.

=cut


has 'translation' => (
    is => 'ro',
    lazy_build => 1,
);

around 'translation' => \&_translation_lookup;


sub _build_translation {#FOLDBEG

    my $self = shift;

    my %translation_table = (
	32=>'MTMHTTMTTLTLTSLIPPILTTLVNPNKKNSYPHYVKSIVASTFIISLFPTTMFMCLDQEVIISNWHWATTQTTQLSLSFKLDYFSMMFIPVALFVTWSIMEFSLWYMNSDPNINQFFKYLLIFLITMLILVTANNLFQLFIGWEGVGIMSFLLISWWYARADANTAAIQAILYNRIGDIGFILALAWFILHSNSWDPQQMALLNANPSLTPLLGLLLAAAGKSAQLGLHPWLPSAMEGPTPVSALLHSSTMVVAGIFLLIRFHPLAENSPLIQTLTLCLGAITTLFAAVCALTQNDIKKIVAFSTSSQLGLMMVTIGINQPHLAFLHICTHAFFKAMLFMCSGSIIHNLNNEQDIRKMGGLLKTMPLTSTSLTIGSLALAGMPFLTGFYSKDHIIETANMSYTNAWALSITLIATSLTSAYSTRMILLTLTGQPRFPTLTNINENNPTLLNPIKRLAAGSLFAGFLITNNISPASPFQTTIPLYLKLTALAVTFLGLLTALDLNYLTNKLKMKSPLCTFYFSNMLGFYPSITHRTIPYLGLLTSQNLPLLLLDLTWLEKLLPKTISQHQISTSIITSTQKGMIKLYFLSFFFPLILTLLLIT',
	16=>'MFADRWLFSTNHKDIGTLYLLFGAWAGVLGTALSLLIRAELGQPGNLLGNDHIYNVIVTAHAFVMIFFMVMPIMIGGFGNWLVPLMIGAPDMAFPRMNNMSFWLLPPSLLLLLASAMVEAGAGTGWTVYPPLAGNYSHPGASVDLTIFSLHLAGVSSILGAINFITTIINMKPPAMTQYQTPLFVWSVLITAVLLLLSLPVLAAGITMLLTDRNLNTTFFDPAGGGDPILYQHLFWFFGHPEVYILILPGFGMISHIVTYYSGKKEPFGYMGMVWAMMSIGFLGFIVWAHHMFTVGMDVDTRAYFTSATMIIAIPTGVKVFSWLATLHGSNMKWSAAVLWALGFIFLFTVGGLTGIVLANSSLDIVLHDTYYVVAHFHYVLSMGAVFAIMGGFIHWFPLFSGYTLDQTYAKIHFTIMFIGVNLTFFPQHFLGLSGMPRRYSDYPDAYTTWNILSSVGSFISLTAVMLMIFMIWEAFASKRKVLMVEEPSMNLEWLYGCPPPYHTFEEPVYMKS',
	28=>'MLKLIVPTIMLLPLTWLSKKHMIWINTTTHSLIISIIPLLFFNQINNNLFSCSPTFSSDPLTTPLLMLTTWLLPLTIMASQRHLSSEPLSRKKLYLSMLISLQISLIMTFTATELIMFYIFFETTLIPTLAIITRWGNQPERLNAGTYFLFYTLVGSLPLLIALIYTHNTLGSLNILLLTLTAQELSNSWANNLMWLAYTMAFMVKMPLYGLHLWLPKAHVEAPIAGSMVLAAVLLKLGGYGMMRLTLILNPLTKHMAYPFLVLSLWGMIMTSSICLRQTDLKSLIAYSSISHMALVVTAILIQTPWSFTGAVILMIAHGLTSSLLFCLANSNYERTHSRIMILSQGLQTLLPLMAFWWLLASLANLALPPTINLLGELSVLVTTFSWSNITLLLTGLNMLVTALYSLYMFTTTQWGSLTHHINNMKPSFTRENTLMFMHLSPILLLSLNPDIITGFSS',
	35=>'MTPMRKTNPLMKLINHSFIDLPTPSNISAWWNFGSLLGACLILQITTGLFLAMHYSPDASTAFSSIAHITRDVNYGWIIRYLHANGASMFFICLFLHIGRGLYYGSFLYSETWNIGIILLLATMATAFMGYVLPWGQMSFWGATVITNLLSAIPYIGTDLVQWIWGGYSVDSPTLTRFFTFHFILPFIIAALATLHLLFLHETGSNNPLGITSHSDKITFHPYYTIKDALGLLLFLLSLMTLTLFSPDLLGDPDNYTLANPLNTPPHIKPEWYFLFAYTILRSVPNKLGGVLALLLSILILAMIPILHMSKQQSMMFRPLSQSLYWLLAADLLILTWIGGQPVSYPFTIIGQVASVLYFTTILILMPTISLIENKMLKWA',
	10=>'INPLAQPVIYSTIFAGTLITALSSHWFFTWVGLEMNMLAFIPVLTKKMNPRSTEAAIKYFLTQATASMILLMAILFNNMLSGQWTMTNTTNQYSSLMIMMAMAMKLGMAPFHFWVPEVTQGTPLTSGLLLLTWQKLAPISIMYQISPSLNVSLLLTLSILSIMAGSWGGLNQTQLRKILAYSSITHMGWMMAVLPYNPNMTILNLTIYIILTTTAFLLLNLNSSTTTLLLSRTWNKLTWLTPLIPSTLLSLGGLPPLTGFLPKWAIIEEFTKNNSLIIPTIMATITLLNLYFYLRLIYSTSITLLPMSNNVKMKWQFEHTKPTPFLPTLIALTTLLLPISPFMLMIL',
	6=>'MPMANLLLLIVPILIAMAFLMLTERKILGYMQLRKGPNVVGPYGLLQPFADAMKLFTKEPLKPATSTITLYITAPTLALTIALLLWTPLPMPNPLVNLNLGLLFILATSSLAVYSILWSGWASNSNYALIGALRAVAQTISYEVTLAIILLSTLLMSGSFNLSTLITTQEHLWLLLPSWPLAMMWFISTLAETNRTPFDLAEGESELVSGFNIEYAAGPFALFFMAEYTNIIMMNTLTTTIFLGTTYDALSPELYTTYFVTKTLLLTSLFLWIRTAYPRFRYDQLMHLLWKNFLPLTLALLMWYVSMPITISSIPPQT',
	23=>'MTHQSHAYHMVKPSPWPLTGALSALLMTSGLAMWFHFHSMTLLMLGLLTNTLTMYQWWRDVTRESTYQGHHTPPVQKGLRYGMILFITSEVFFFAGFFWAFYHSSLAPTPQLGGHWPPTGITPLNPLEVPLLNTSVLLASGVSITWAHHSLMENNRNQMIQALLITILLGLYFTLLQASEYFESPFTISDGIYGSTFFVATGFHGLHVIIGSTFLTICFIRQLMFHFTSKHHFGFEAAAWYWHFVDVVWLFLYVSIYWWGS',
	19=>'MAHAAQVGLQDATSPIMEELITFHDHALMIIFLICFLVLYALFLTLTTKLTNTNISDAQEMETVWTILPAIILVLIALPSLRILYMTDEVNDPSLTIKSIGHQWYWTYEYTDYGGLIFNSYMLPPLFLEPGDLRLLDVDNRVVLPIEAPIRMMITSQDVLHSWAVPTLGLKTDAIPGRLNQTTFTATRPGVYYGQCSEICGANHSFMPIVLELIPLKIFEMGPVFTL',
	22=>'MNENLFASFIAPTILGLPAAVLIILFPPLLIPTSKYLINNRLITTQQWLIKLTSKQMMTMHNTKGRTWSLMLVSLIIFIATTNLLGLLPHSFTPTTQLSMNLAMAIPLWAGTVIMGFRSKIKNALAHFLPQGTPTPLIPMLVIIETISLLIQPMALAVRLTANITAGHLLMHLIGSATLAMSTINLPSTLIIFTILILLTILEIAVALIQAYVFTLLVSLYLHDNT',
	33=>'MMYALFLLSVGLVMGFVGFSSKPSPIYGGLVLIVSGVVGCVIILNFGGGYMGLMVFLIYLGGMMVVFGYTTAMAIEEYPEAWGSGVEVLVSVLVGLAMEVGLVLWVKEYDGVVVVVNFNSVGSWMIYEGEGSGLIREDPIGAGALYDYGRWLVVVTGWTLFVGVYIVIEIARGN',
	    25=>'MNFALILMINTLLALLLMIITFWLPQLNGYMEKSTPYECGFDPMSPARVPFSMKFFLVAITFLLFDLEIALLLPLPWALQTTNLPLMVMSSLLLIIILALSLAYEWLQKGLDWTE',
	    27=>'MPLIYMNIMLAFTISLLGMLVYRSHLMSSLLCLEGMMLSLFIMATLMTLNTHSLLANIVPIAMLVFAACEAAVGLALLVSISNTYGLDYVHNLNLLQC',
	    21=>'MPQLNTTVWPTMITPMLLTLFLITQLKMLNTNYHLPPSPKPMKMKNYNKPWEPKWTKICSLHSLPPQS'
	);   

    return \%translation_table;
}#FOLDEND

sub _translation_lookup {#FOLDBEG
    my ($orig, $self, $index, $beg, $end) = @_;

    my $t = $self->$orig;
    return $t unless defined $index;

    my $locus_id = $self->_get_locus_id($index);
    $index = $locus_id if $locus_id;

    croak "$index not a valid translation index" unless $index =~ /^\d+$/;
    croak "$index not a valid translation index" unless $t->{$index};

    return $t->{$index} unless $beg or $end;
    return $self->_seq_string($t->{$index}, $beg, $end);
}#FOLDEND



sub _seq_string {#FOLDBEG
    my ($self, $s, $b, $e) = @_;

    croak "Beginning value must be an integer." unless defined $b and $b =~ /-?\d+/;
    croak "Ending value must be an integer." if defined $e and !($e =~ /-?\d+/);
    croak "Beginning value cannot be zero." if $b == 0;
    croak "Ending value cannot be zero." if defined $e and $e == 0;
    
    if (defined $b and defined $e) {
        if ( $b > 0 ) {
            if ( $e > 0 ) {
                $s = substr($s, $b - 1, ( $e - $b + 1));
            }
            else {
                $s = substr($s, $b - 1, $e);
            }
        }
        else {
            if ( $e > 0 ) {
                $s = substr($s, $b - 1, ( $e - $b + 1));
            }
            else {
                $s = substr($s, $b, $e);
            }
        }
    }
    else {
        if ( $b > 0 ) {
            $s = substr($s, $b - 1);
        }
        else {
            $s = substr($s, $b);
        }
    }

    return $s;
}#FOLDEND


sub _get_locus_id {#FOLDBEG
    my ($self, $key_val) = @_;

    my %locus_lookup = (
	    'F'           =>  1,
	    '12S'         =>  2,
	    'V'           =>  3,
	    '16S'         =>  4,
	    'L(UUA/G)'    =>  5,
	    'L-UUA'       =>  5,
	    'L-UUG'       =>  5,
	    'NADH1'       =>  6,
	    'ND1'         =>  6,
	    'I'           =>  7,
	    'Q'           =>  8,
	    'M'           =>  9,
	    'NADH2'       => 10,
	    'ND2'         => 10,
	    'W'           => 11,
	    'A'           => 12,
	    'N'           => 13,
	    'C'           => 14,
	    'Y'           => 15,
	    'COI'         => 16,
	    'S(UCN)'      => 17,
	    'S-UCN'       => 17,
	    'D'           => 18,
	    'COII'        => 19,
	    'K'           => 20,
	    'ATP8'        => 21,
	    'ATPase8'     => 21,
	    'ATPASE8'     => 21,
	    'ATP6'        => 22,
	    'ATPase6'     => 22,
	    'ATPASE6'     => 22,
	    'COIII'       => 23,
	    'G'           => 24,
	    'NADH3'       => 25,
	    'ND3'         => 25,
	    'R'           => 26,
	    'NADH4L'      => 27,
	    'ND4L'        => 27,
	    'NADH4'       => 28,
	    'ND4'         => 28,
	    'H'           => 29,
	    'S(AGY)'      => 30,
	    'S-AGY'       => 30,
	    'L(CUN)'      => 31,
	    'L-CUN'       => 31,
	    'NADH5'       => 32,
	    'ND5'         => 32,
	    'NADH6'       => 33,
	    'ND6'         => 33,
	    'E'           => 34,
	    'Cytb'        => 35,
	    'CYTB'        => 35,
	    'T'           => 36,
	    'P'           => 37,
	    'HVS2'        => 38,
	    'OH'          => 39,
	    'CSB1'        => 40,
	    'TFX'         => 41,
	    'TFY'         => 42,
	    'CSB2'        => 43,
	    'HPR'         => 44,
	    'CSB3'        => 45,
	    'mt4H'        => 46,
	    'MT4H'        => 46,
	    'mt3H'        => 47,
	    'MT3H'        => 47,
	    'PL'          => 48,
	    'TFL'         => 49,
	    'TFH'         => 50,
	    'PH1'         => 51,
	    'PH2'         => 52,
	    'RNR3'        => 53,
	    'TER'         => 54,
	    'NC1'         => 55,
	    'NC2'         => 56,
	    'NC3'         => 57,
	    'NC4'         => 58,
	    'OL'          => 59,
	    'NC5'         => 60,
	    'NC6'         => 61,
	    'NC7'         => 62,
	    'NC8'         => 63,
	    'NC9'         => 64,
	    'ATT'         => 65,
	    'NC10'        => 66,
	    'DLOOP'       => 67,
	    'HVS1'        => 68,
	    '7S'          => 69,
	    'TAS'         => 70,
	    'mt5'         => 71,
	    'MT5'         => 71,
	    'mt3L'        => 72,
	    'MT3L'        => 72,
	);

    return $locus_lookup{$key_val} ? $locus_lookup{$key_val} : undef;
}#FOLDEND



=head1 AUTHOR

Marty Brandon, C<< <mbrandon at uci.edu> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-bio-mm-refmitohuman at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Bio-Bio::Mitomaster-RefMitoHuman>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Bio::Mitomaster::RefHuman

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Bio-Bio::Mitomaster-RefMitoHuman>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Bio-Bio::Mitomaster-RefMitoHuman>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Bio-Bio::Mitomaster-RefMitoHuman>

=item * Search CPAN

L<http://search.cpan.org/dist/Bio-Bio::Mitomaster-RefMitoHuman/>

=back


=head1 COPYRIGHT & LICENSE

Copyright 2008 Marty Brandon, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.


=cut

no Moose;
__PACKAGE__->meta->make_immutable;
1; # End of Bio::Mitomaster::RefHuman
