########################################################################
# housekeeping
########################################################################

package Devel::SharedLibs;
use v5.22;
use autodie qw( open close );

use IO::File;

use List::MoreUtils qw( uniq );

########################################################################
# package variables
########################################################################

our $VERSION = '0.1.0';
$VERSION    = eval "$VERSION";

sub import
{
    my ( undef, $path ) = @_;

    END
    {
        $ENV{ DEVEL_SHAREDLIBS_PRINT }
        or return;

        local $,    = "\n";
        local $\    = "\n";

        my $fh
        = $path
        ? IO::File->new( $path, 'w' )
        : *STDOUT{ IO }
        ;

        print $fh
        "# ldd '$^X', '$0'" =>
        uniq
        sort
        map
        {
            # the literal '=>' dodges linux-vdso & friends,
            # which lack a path and also the 
            # "not a dynamic executable" messages from 
            # most of the contents.

            my ( $lib ) = m{ => \s+ (\S+) }x;

            $lib || ()
        }
        map
        {
            split "\n" => qx(ldd $_ 2>/dev/null ) 
        }
        ( $^X,  values %INC );

        close $fh;
    }
}

# keep require happy
1
__END__

=head1 NAME

Devel::SharedLibs -- output of ldd for anythingin %INC

=head1 SYNOPSIS

    # if $ENV{ DEVEL_SHAREDLIBS_PRINT } is true at the end 
    # of execution then print results of runing ldd on 
    # everything in %INC to STDOUT.

    use Devel::SharedLibs;

    # this can be mangled any time during execution it is not
    # checked until END time.
    #
    # true dumps list to stdout or path; false does nothing.

    $ENV{ DEVEL_SHAREDLIBS_PRINT } = 1;

    __END__


    # use ./libfiles.out insted of stdout for the list.

    use Devel::SharedLibs qw( ./libfiles.out );

=head1 DESCRIPTION

=head2 Output

All this does it get a unique output from running "ldd" on all
of the paths in values %INC. The scan is done at END time to 
ensure that anything pulled in via dynamic loading is listed.

Ouptut includes the executable path ("$^X") and the executable
path ($0, which may have been munged during execution).

For example, to see which shared object lib's (.so files) perl
itself is linked with use:

    #!/usr/bin/env  perl

    use Devel::SharedLibs;
    __END__

which outputs something like:

    $ DEVEL_SHAREDLIBS_PRINT=1 perl t/bin/exec-stdout
    # ldd '/opt/perl/5.22/bin/perl', 't/bin/exec-stdout'
    /lib64/libc.so.6
    /lib64/libcrypt.so.1
    /lib64/libdl.so.2
    /lib64/libm.so.6
    /lib64/libnsl.so.1
    /lib64/libpthread.so.0
    /lib64/libutil.so.1

=head2 Why bother?

A truly idiotic thing about most linux container doc's (Docker
is a great example, but not alone in this) is putting a full
copy of the OS as a 'base layer' for the container. The claimed
reason is having access to necessary shared object lib's in the
distro.

Catch: Aside from security and bloat issues, it simply does not work.
Problem is that there is no gurarantee that the executables being
run on the container system were complied on the same distro with 
compatable lib's. 

A really simple fix is to build the container using whatever 
.so files are actually *used* by the executable. 

For something running #!perl this can be done by examining the
values of %^X (i.e., perl executable) and %INC (i.e., included paths)
with ldd.

At the very least it'll give you a good place to start. For a an 
reasonable set of modules (i.e., the usuall collection of files that 
go along with a crawler, PSGI or CGI web back end, or log watcher)
this runs in the 10MB - 20MB range (vs. 1+GB for a full linux distro).


=head1 CAVEAT UTILITOR

=over 4

=item 

This is probably only useful to use once as there is no bookkeeping
of the output path: the path is examined once at END time based on
the arguments to the last call to import. Then again, it is probably
only useful to put this in #! code anway, which solves that problem.

=item 

You will require a usable ldd to run this. Nearly general-purpose
*NIX system from the last 30 years should have one; if you are
running an embedded system (e.g., openwrt, andriod) this may not
work.

=item 

This will help deal with so's, it will not help you if libc or the
kernel headers are far enough out of sync. Most of the time using
something based on lxc the libc and kernel versions are close enough
that it won't kill you.

=back

=head1 SEE ALSO

=over 4

=item ldd(1)

Speicfics of running ldd on your system.

=item How to use this with Docker

Brief explanation of building a '-lib' image for docker using 
ldd output:

<http://www.slideshare.net/lembark/shared-object-images-in-docker-when-what-you-need-is-what-you-want>

=back


