package Net::OAuth2::AuthorizationServer;

=head1 NAME

Net::OAuth2::AuthorizationServer - Easier implementation of an OAuth2
Authorization Server

=for html
<a href='https://travis-ci.org/G3S/net-oauth2-authorizationserver?branch=master'><img src='https://travis-ci.org/G3S/net-oauth2-authorizationserver.svg?branch=master' alt='Build Status' /></a>
<a href='https://coveralls.io/github/G3S/net-oauth2-authorizationserver?branch=master'><img src='https://coveralls.io/repos/github/G3S/net-oauth2-authorizationserver/badge.svg?branch=master' alt='Coverage Status' /></a>

=head1 VERSION

0.06

=head1 SYNOPSIS

    my $Server = Net::OAuth2::AuthorizationServer->new;

    my $Grant  = $Server->auth_code_grant(
        ...
    );

=head1 DESCRIPTION

This module is the gateway to the various OAuth2 grant flows, as documented
at L<https://tools.ietf.org/html/rfc6749>. Each module implements a specific
grant flow and is designed to "just work" with minimal detail and effort.

Please see L<Net::OAuth2::AuthorizationServer::Manual> for more information
on how to use this module and the various grant types. You should use the manual
in conjunction with the grant type module you are using to understand how to
override the defaults if the "just work" mode isn't good enough for you.

=cut

use strict;
use warnings;

use Moo;
use Types::Standard qw/ :all /;

use Net::OAuth2::AuthorizationServer::AuthorizationCodeGrant;
use Net::OAuth2::AuthorizationServer::PasswordGrant;

our $VERSION = '0.06';

=head1 GRANT TYPES

=head2 auth_code_grant

OAuth Authorisation Code Grant as document at L<http://tools.ietf.org/html/rfc6749#section-4.1>.

See L<Net::OAuth2::AuthorizationServer::AuthorizationCodeGrant>.

=cut

sub auth_code_grant {
    my ( $self, @args ) = @_;
    return Net::OAuth2::AuthorizationServer::AuthorizationCodeGrant->new( @args, );
}

=head2 implicit_grant

Not yet implemented.

=head2 password_grant

OAuth Resource Owner Password Grant as document at L<http://tools.ietf.org/html/rfc6749#section-4.3>.

See L<Net::OAuth2::AuthorizationServer::PasswordGrant>.

=cut

sub password_grant {
    my ( $self, @args ) = @_;
    return Net::OAuth2::AuthorizationServer::PasswordGrant->new( @args, );
}

=head2 client_grant

Not yet implemented.

=head2 extension_grant

Not yet implemented.

=head1 SEE ALSO

L<Mojolicious::Plugin::OAuth2::Server> - A Mojolicious plugin using this module

L<Mojo::JWT> - encode/decode JWTs

=head1 AUTHOR

Lee Johnson - C<leejo@cpan.org>

=head1 LICENSE

This library is free software; you can redistribute it and/or modify it under
the same terms as Perl itself. If you would like to contribute documentation
or file a bug report then please raise an issue / pull request:

    https://github.com/G3S/net-oauth2-authorizationserver

=cut

__PACKAGE__->meta->make_immutable;
