use strict;
use CGI::Wiki::TestConfig::Utilities;
use CGI::Wiki;
use URI::Escape;

use Test::More tests =>
  (3 + 15 * $CGI::Wiki::TestConfig::Utilities::num_stores);

use_ok( "CGI::Wiki::Plugin::RSS::ModWiki" );

eval { my $rss = CGI::Wiki::Plugin::RSS::ModWiki->new; };
ok( $@, "new croaks if no wiki object supplied" );

eval {
        my $rss = CGI::Wiki::Plugin::RSS::ModWiki->new( wiki => "foo" );
     };
ok( $@, "new croaks if something that isn't a wiki object supplied" );

my %stores = CGI::Wiki::TestConfig::Utilities->stores;

my ($store_name, $store);
while ( ($store_name, $store) = each %stores ) {
  SKIP: {
      skip "$store_name storage backend not configured for testing", 15
          unless $store;

      print "#\n##### TEST CONFIG: Store: $store_name\n#\n";

      my $wiki = CGI::Wiki->new( store => $store );
      my %default_config = (
              wiki => $wiki,
              site_name => "CGI::Wiki Test Site",
              make_node_url => sub {
                  "http://example.com/?id=" . uri_escape($_[0])
                    . ";version=" . uri_escape($_[1]) },
              recent_changes_link => "http://example.com/recentchanges"
      );
      my $rss = eval {
          CGI::Wiki::Plugin::RSS::ModWiki->new( %default_config );
      };
      is( $@, "",
         "'new' doesn't croak if wiki object and mandatory parameters supplied"
      );
      isa_ok( $rss, "CGI::Wiki::Plugin::RSS::ModWiki" );

      my $feed = eval { $rss->recent_changes; };
      is( $@, "", "->recent_changes doesn't croak" );

      # Check the things that are generated by the mandatory arguments.
      like( $feed, qr|<item rdf:about="http://example.com/\?id=Test%20Node%201;version=1">|,
	    "make_node_url is used" );

      like( $feed, qr|<wiki:version>1</wiki:version>|,
	    "version numbers included in feed" );

      like( $feed, qr|<wiki:status>new</wiki:status>|,
	    "page status included in feed" );

      like( $feed, qr|<wiki:importance>major</wiki:importance>|,
	    "change importance included and defaults to 'major'" );

      # Check stuff that comes from the metadata.
      like( $feed, qr|<dc:contributor>Kake</dc:contributor>|,
	    "username picked up as contributor" );

      like( $feed, qr|<description>.*\[nou]</description>|,
            "username included in description" );

      # Check that interwiki things are passed through right.
      $rss = CGI::Wiki::Plugin::RSS::ModWiki->new(
          %default_config,
	  interwiki_identifier => "KakesWiki"
      );
      $feed = $rss->recent_changes;
      like( $feed, qr|<wiki:interwiki>KakesWiki</wiki:interwiki>|,
            "interwiki identifier passed through OK" );

      # Check that diff url comes through.
      $rss = CGI::Wiki::Plugin::RSS::ModWiki->new(
          %default_config,
          make_diff_url        => sub {
              my $node_name = shift;
              return "http://example.com/?action=show_diff;id="
                   . uri_escape($node_name)
                                      }
      );
      $feed = $rss->recent_changes;
      like( $feed, qr|<wiki:diff>http://example.com/\?action=show_diff;id=Calthorpe%20Arms</wiki:diff>|,
	    "make_diff_url used" );

      # Check that history url comes through.
      $rss = CGI::Wiki::Plugin::RSS::ModWiki->new(
          %default_config,
          make_history_url        => sub {
              my $node_name = shift;
              return "http://example.com/?action=history;id="
                   . uri_escape($node_name)
                                      }
      );
      $feed = $rss->recent_changes;
      like( $feed, qr|<wiki:history>http://example.com/\?action=history;id=Calthorpe%20Arms</wiki:history>|,
	    "make_history_url used" );

      # Test the 'items' parameter.
      $feed = $rss->recent_changes( items => 2 );
      unlike( $feed, qr|<title>Test Node 1</title>|, "items param works" );

      # Test the 'days' parameter.
      $feed = $rss->recent_changes( days => 2 );
      like( $feed, qr|<title>Old Node</title>|, "days param works" );

      # Test ignoring minor changes.
      $feed = $rss->recent_changes( ignore_minor_changes => 1 );
      unlike( $feed, qr|This is a minor change.|,
              "ignore_minor_changes works" );
  }
}
