package CGI::Wiki::Formatter::Multiple;
use strict;

use vars qw( $VERSION );
$VERSION = '0.01';

=head1 NAME

CGI::Wiki::Formatter::Multiple - Allows a CGI::Wiki wiki to use more than one formatter.

=head1 DESCRIPTION

A "dummy" formatter for L<CGI::Wiki>.  Passes methods through to other
CGI::Wiki formatters, depending on supplied metadata.

=head1 SYNOPSIS

  use CGI::Wiki::Formatter::Multiple;
  use CGI::Wiki::Formatter::Pod;
  use CGI::Wiki::Formatter::UseMod;

  my $pod_fmtr = CGI::Wiki::Formatter::Pod->new(
      node_prefix => "wiki.cgi?node=",
  );

  my $usemod_fmtr = CGI::Wiki::Formatter::UseMod->new(
      node_prefix    => "wiki.cgi?node=",
      extended_links => 1,
      allowed_tags   => [ qw( p b i div br ) ],
  );

  my $formatter = CGI::Wiki::Formatter::Multiple->new(
      documentation => $pod_fmtr,
      discussion    => $usemod_fmtr,
      _DEFAULT      => $usemod_fmtr,
  );

  my $wiki = CGI::Wiki->new( store     => ...,
                             formatter => $formatter );
  my $output = $wiki->format( "This is some discussion.",
                              { formatter => "discussion" } );
  
=head1 METHODS

=over

=item B<new>

  my $formatter = CGI::Wiki::Formatter::Multiple->new(
      label_1  => Formatter1->new( ... ),
      label_2  => Formatter2->new( ... ),
      _DEFAULT => CGI::Wiki::Formatter::Default->new,
  );

You may supply as many formatter objects as you wish.  They don't have
to be of different classes; you may just wish to, for example, permit
different HTML tags to be used on different types of pages.

The "labels" supplied as the keys of the parameter hash should be
unique.  When you write a node, you should store a key-value pair in
its metadata where the key is C<formatter> and the value is the label
of the formatter that should be used to render that node.

The C<_DEFAULT> label is special - it defines the formatter that will
be used for any node that does not have a C<formatter> stored in its
metadata.  The C<_DEFAULT> formatter, if not supplied to C<< ->new >>,
will default to the very basic L<CGI::Wiki::Formatter::Default>.

=cut

sub new {
    my ($class, %args) = @_;
    my $self = bless {}, $class;
    unless ( $args{_DEFAULT} ) {
        require CGI::Wiki::Formatter::Default;
        $args{_DEFAULT} = CGI::Wiki::Formatter::Default->new;
    }
    $self->{formatters} = \%args;
    return $self;
}

=item B<format( $raw, \%metadata )>

    my $output = $formatter->format( "Here is some text.", undef,
                                     { formatter => "discussion" } );

Uses the value of C<formatter> given in the metadata to decide which
of the formatter objects passed on instantiation to use, then uses it
to format the provided rawwikitext.

The C<undef> second element of the parameter array in the example is
there because when this is called from a L<CGI::Wiki> object, the wiki
object passes itself in as the second parameter.

=cut

sub format {
    my ($self, $raw, $wiki, $metadata) = @_;
    my $formatter_label = $metadata->{formatter}[0] || "_DEFAULT";
    return $self->{formatters}{$formatter_label}->format( $raw, $wiki );
}

=back

=head1 SEE ALSO

L<CGI::Wiki>

=head1 AUTHOR

Kake Pugh <kake@earth.li>

=head1 SUPPORT

Bug reports, questions and feature requests should go to cgi-wiki-dev@earth.li

=head1 COPYRIGHT

     Copyright (C) 2003 Kake Pugh.  All Rights Reserved.

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
