package Parse::EventLog;

require Exporter;
@ISA     = qw(Exporter);
@EXPORT  = qw(EVENT_INFORMATION EVENT_WARNING EVENT_ERROR EVENT_AUDIT_SUCCESS EVENT_AUDIT_FAILURE GetEventType);
$VERSION = '0.5';

use warnings;
use strict;
use constant { EVENT_ERROR => 1, EVENT_WARNING => 2, EVENT_INFORMATION => 4,
               EVENT_AUDIT_SUCCESS => 8, EVENT_AUDIT_FAILURE => 16};

=head1 NAME

  Parse::EventLog - Parse information from Windows Event Logs

=head1 SYNOPSIS

 use Parse::EventLog;
 my $elog =
  Parse::EventLog->new("c:\\winnt\\system32\\config\\SysEvent.Evt");

 my %c = $elog->getNewestEvent();
 # Do something interesting here.

 while (%c = $elog->getPreviousEvent()) {
  print localtime($c{TimeGenerated}) .
   ": $c{EventID} generated by $c{Source} on $c{Computer}.\n";
  # Do a lot of interesting somethings here.
 }

=head1 DESCRIPTION

Parse::EventLog is a module designed to allow parsing of Windows 2000 Event Logs
on any platform. For better Event Log parsing on Windows platforms, see
Win32::EventLog.

When instansiated with the location of an Event Log file, you will be able to
walk through the existing events in chronological order from oldest to newest or
vice versa, as well as being able to extract the whole log to a hash table and
seek through however you wish.

=head1 METHODS

=head2 new

 my $eventlog =
  Parse::EventLog->new("c:\\winnt\\system32\\config\\SysEvent.Evt");

Parse::EventLog constructor, called with the location of the event log file to
open. Returns a Parse::EventLog object or dies on error. Stores the internal
record pointer at the newest event, so the next call to getPreviousEvent will
succeed but getNextEvent will fail (there are no newer events).

=cut

sub new {
	my $class = shift;
  my $filename = shift;
  my %events;
  my @events_by_time;
  my @data;
  my $RecordNumber;
  my $TimeGenerated;
  my $TimeWritten;
  my $EventID;
  my $EventType;
  my $NumStrings;
  my $EventCategory;
  my $unused;
  my $Source;
  my $Computer;
  my @bmsgs;
  my $msgct;

 {
  local $/; #$INPUT_RECORD_SEPARATOR;
  my $filedata;
  my $file;
  open($file, $filename) or die "Unable to open file $filename";
  binmode($file);
  $filedata = <$file>;
  close $file;
  # Split each record out by its header.
  @data = split(/LfLe/, $filedata); #chr(0x4c).chr(0x66).chr(0x4c).chr(0x65)
 }

 foreach my $rec (@data) {
  if (length($rec) < 92) {  # Magic number!
   next;
  }

  ($RecordNumber, $TimeGenerated, $TimeWritten,
   $EventID, $unused, $unused, $EventType, $unused,
   $NumStrings) = unpack('VVVvccccvc', $rec); # More magic.

   if ($events{$RecordNumber}) {
    die "Error parsing event log, duplicate record number encountered ($RecordNumber)";
   }

   $events{$RecordNumber}{'TimeGenerated'} = $TimeGenerated;
   $events{$RecordNumber}{'TimeWritten'} = $TimeWritten;
   $events{$RecordNumber}{'RecordNumber'} = $RecordNumber;
   $events{$RecordNumber}{'EventID'} = $EventID;
   $events{$RecordNumber}{'EventType'} = $EventType;
   $events{$RecordNumber}{'NumStrings'} = $NumStrings;
   $events{$RecordNumber}{'Source'} = $Source;
   $events{$RecordNumber}{'Computer'} = $Computer;

   # Strings are unicode and in a null terminated array. Split on double-null
   # (a unicode null), and then remove nulls. Note: Replace with a real Unicode
   # to ASCII function.
   @bmsgs = split(/\0\0/, substr($rec, 48));  # Another magic number
   $msgct = -2;

   foreach my $m (@bmsgs) {
    $msgct++;
    if ($msgct == -1) {
     $Source = join('', split(/\0/, $m));
    } elsif ($msgct == 0) {
     $Computer = join('', split(/\0/, $m));
     if ($NumStrings == 0) { last; }
    } else {
     push @{$events{$RecordNumber}{'Strings'}}, join('', split(/\0/, $m));
     if ($msgct >= $NumStrings) { last; }
    }
   }

 }

 # Sort the events and store it in an array with the oldest being at
 # $events_by_time[0].
 @events_by_time = sort {
  $events{$b}{'TimeGenerated'} <=> $events{$a}{'TimeGenerated'}
 } keys %events;

 my $self = {
		'filename' => $filename,
    'events' => \%events,
    'events_by_time' => \@events_by_time,
    'lastindex' => 0,
    'totalevents' => scalar @events_by_time,
 };
 bless $self, $class;

 return( $self );
};

=head2 getOldestEvent

 my %event = $eventlog->getOldestEvent();

Returns a hash table containing the oldest entry in the event logand moves the
current event pointer to the oldest event.

=cut

#Remember, the list (@events_by_time) goes from newest -> oldest, so we want
# the last entry in that array. Extract the array into $ebt
sub getOldestEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 $self->{'lastindex'} = (scalar @ebt) - 1;
 return %{$self->{'events'}->{$ebt[(scalar @ebt) - 1]}};
}

=head2 getNewestEvent

 my %event = $eventlog->getNewestEvent();

Returns a hash table containing the newest entry in the event log and moves the
current event pointer to the newest event.

=cut

# Get the newest entry in the list. Remember, the list goes from
# newest -> oldest, so we want the first entry. Code looks a lot like
# getOldestEvent.
sub getNewestEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 $self->{'lastindex'} = 0;
 return %{$self->{'events'}->{$ebt[0]}};
}

=head2 getNextEvent

 my %event = $eventlog->getNextEvent();

Moves us forward in time, returning a hash table containing the next event in
the event log. Returns an empty array/hashtable if there are no more events to
get.

=cut

# Move forward in time.
sub getNextEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 my $newindex = $self->{'lastindex'} - 1;
 if ($newindex < 0) { return (); }
 $self->{'lastindex'} = $newindex;
 return %{$self->{'events'}->{$ebt[$newindex]}};
}

=head2 getPreviousEvent

 my %event = $eventlog->getPreviousEvent();

Moves us backward in time, returning a hash table containing the previous event
in the event log. Returns an empty array/hashtable if there are no more events
to get.

=cut

# Move backward in time.
sub getPreviousEvent {
 my $self = shift;
 my @ebt = @{$self->{'events_by_time'}};
 my $newindex = $self->{'lastindex'} + 1;
 if ($newindex > (scalar @ebt) - 1) { return (); }
 $self->{'lastindex'} = $newindex;
 return %{$self->{'events'}->{$ebt[$newindex]}};
}

=head2 getAll

 my %event = $eventlog->getAll();

Returns a copy of the internal Event Log hash table for your own nefarious
pleasure. The main hash table is keyed by the internal RecordNumber (which is
unique to each entry in the file). The value of each of those keys is another
hash table containing the actual event (in the same format as is returned by
the other get*Event functions).

=head3 Example

 my %c = $eventlog->getAll();
 foreach my $a (keys %c) {
  print 'Time: ' . localtime($c{$a}{'TimeGenerated'} . "\n";
  print "Computer: $c{$a}{'Computer'}\n";
  print "Source: $c{$a}{'Source'}\n";
  print 'Type: ' . GetEventType($c{$a}{'EventType'}) . "\n";
  print '-' x 10;
 }

=cut

# Return a copy everything, might be slow.
sub getAll {
 my $self = shift;
 return %{$self->{'events'}};
}

=head2 count

 my %event = $eventlog->count();

Returns the number of entries loaded from the event log.

=cut

sub count {
 my $self = shift;
 return $self->{'totalevents'};
}

=head1 FUNCTIONS

=head2 GetEventType

 print 'Event type:' . GetEventType($c{'EventType'}) . "\n";

Returns a string containing the type of event (Error, Warning, Information,
Audit Success, Audit Failure, or Unknown).

GetEventType is exported by default.

=cut

sub GetEventType {
 my $EventType = shift;
 
 if ($EventType == EVENT_ERROR) {
  return 'Error';
 } elsif ($EventType == EVENT_WARNING) {
  return 'Warning';
 }if ($EventType == EVENT_INFORMATION) {
  return 'Information';
 }if ($EventType == EVENT_AUDIT_SUCCESS) {
  return 'Audit Success';
 }if ($EventType == EVENT_AUDIT_FAILURE) {
  return 'Audit Failure';
 } else {
  return 'Unknown';
 }
}

=head1 CONSTANTS

Several constants are exported by default. These constants are Event Type
definitions. They are as follows:

=over 4

=item * EVENT_ERROR (1)

=item * EVENT_WARNING (2)

=item * EVENT_INFORMATION (4)

=item * EVENT_AUDIT_SUCCESS (8)

=item * EVENT_AUDIT_FAILURE (16)

=back

=head1 RETURN VALUES

On success, all of the get*Event functions return a hash of the current Event.
The hash itself has several keys made up of each of the values in the Event
itself. The following keys are defined (note the capitals):

=head2 TimeGenerated

The epoch time the event was generated, suitable for feeding to gmtime and localtime.

=head2 TimeWritten

The epoch time the event was written to the event log,
usually the same as TimeGenerated (give or take a second). Suitable for feeding
to gmtime and localtime.

=head2 RecordNumber

The record number of the event. This is assigned and due
to the binary nature of the file, and can not be expected to be consistant or
unique on subsequent parsings of the same file.

=head2 EventID

The event identification number, vendor and application
specific. Details can often be found in the Microsoft Knowledge Base.

=head2 EventType

The event type, vendor and application specific.

=head2 Source

Source of the event, usually the application or service name.

=head2 Computer

The name of the computer that generated the event.

=head2 NumStrings

Number of strings in the event.

=head2 Strings

An array reference to all (if any) of the strings included
with the event. Events work a lot like printf format strings, that is, there is
a fixed message with a variable part, such as:

 The %s service failed to start due to the following error:
 The system cannot find the file specified.

In this case, the Strings parameter would contain the name of the service
that would be placed in %s above to complete the event message.

=head1 TO DO

=over 4

=item * Support for random access reads.

=item * Parsing without loading the whole file into memory. By default, the files are only 64k, but it's possible to specify a much larger size.

=item * Descriptions for general Microsoft Windows events.

=item * Rename/alias getNextEvent and getPreviousEvent to getNewerEvent and getOlderEvent?

=back

=head1 KNOWN BUGS

=over 4

=item * Occasional spurious Strings are read from the Event Log. It seems the log
itself specifies they should be there, but there is only garbage data to be
read.

=item * Incomplete parsing. Due to the mysterious nature of the Event Logs and
the lack of documentation on their format, there are several unknown fields in
the log. These fields are currently ignored. If anyone knows what these fields
are for, please let me know.

=back

=head1 SEE ALSO

L<Win32::EventLog>

=head1 COPYRIGHT

Copyright 2005, John Eaglesham

This module is free software. It may be used, redistributed and/or modified
under the same terms as Perl itself.

