#!env perl
use strict;
use diagnostics;
use Archive::Tar;
use Config::AutoConf::INI;
use ExtUtils::CBuilder;
use File::chdir;
use File::Basename qw/basename dirname/;
use File::Find qw/find/;
use File::Path qw/make_path remove_tree/;
use File::Copy qw/copy/;
use File::Copy::Recursive qw/dircopy/;
use IPC::Run qw/run/;
use Perl::OSType qw/is_os_type/;
use POSIX qw/EXIT_SUCCESS/;
use IO::Handle;

autoflush STDOUT 1;

# ------------------------------------------------------------
# Generation of objects using perl setup, for use in perl's XS
# ------------------------------------------------------------
my $version = get_version();
print "Generating config for tconv version $version\n";

# ------------------------
# Write configuration file
# ------------------------
my $config_h_in = File::Spec->catfile('include', 'tconv_config.h.in');
my $config_h    = File::Spec->catfile('output', 'include', 'tconv_config.h');
make_path(dirname($config_h));

my $ac = Config::AutoConf::INI->new(logfile => 'config.log');
$ac->define_var('TCONV_NTRACE', 1);
$ac->check;
write_config($version, $config_h_in, $config_h);

# -------------
# Fake export.h
# -------------
my $export_h = File::Spec->catfile('output', 'include', 'tconv', 'export.h');
make_path(dirname($export_h));
open(my $fh, '>', $export_h) || die "Cannot open $export_h, $!";
print $fh "#define tconv_EXPORT\n";
print $fh "#define TCONV_NO_EXPORT\n";
close($fh) || warn "Cannot close $export_h, $!";

my @additional_includes = ();
my @additional_srcdir = ();

# -----------------------------------------------------
# We depend on cchardet that is in the form of a tar.gz
# -----------------------------------------------------
my $tar=Archive::Tar->new();
my $cchardet_tar = File::Spec->catfile('3rdparty', 'tar', 'cchardet-1.0.0.tar.gz');
$tar->read($cchardet_tar);
print "... Extracting $cchardet_tar in 3rdparty/output\n";
make_path(File::Spec->catdir('3rdparty', 'output'));
{
    local $CWD = File::Spec->catdir('3rdparty', 'output');
    $tar->extract();
}
undef $tar;
#
# nspr-emu mess, we fix it by generating ourself what is needed
#
my $libcharsetdetect = File::Spec->catdir('3rdparty', 'output', 'cchardet-1.0.0', 'src', 'ext', 'libcharsetdetect');
my $nspremu = File::Spec->catdir($libcharsetdetect, 'nspr-emu');
print "... Suppress directory $nspremu\n";
remove_tree($nspremu);
print "... Generate directory $nspremu\n";
make_path($nspremu);
#
# nsDebug.h.in have no dependency on cmake discoveries
#
my $nsDebug = File::Spec->catfile($nspremu, 'nsDebug.h');
print "... Generate file $nsDebug\n";
copy(File::Spec->catfile('include', 'nsDebug.h.in'), $nsDebug);
#
# prmem.h.in have no dependency on cmake discoveries
#
my $prmem = File::Spec->catfile($nspremu, 'prmem.h');
print "... Generate file $prmem\n";
copy(File::Spec->catfile('include', 'prmem.h.in'), $prmem);
#
# nscore.h.in depend on some type sizes
#
my $nscore = File::Spec->catfile($libcharsetdetect, 'nscore.h');
print "... Suppress file $nscore\n";
unlink($nscore);
print "... Generate file $nscore\n";
write_nscore(File::Spec->catfile('include', 'nscore.h.in'), $nscore);

push(@additional_srcdir, File::Spec->catdir('3rdparty', 'output', 'cchardet-1.0.0', 'src', 'ext', 'libcharsetdetect', 'mozilla', 'extensions', 'universalchardet', 'src', 'base'));
push(@additional_srcdir, File::Spec->catfile($libcharsetdetect, 'charsetdetect.cpp'));
push(@additional_includes, File::Spec->catdir($libcharsetdetect, 'mozilla', 'extensions', 'universalchardet', 'src', 'base'));
push(@additional_includes, File::Spec->catdir($libcharsetdetect, 'nspr-emu'));
push(@additional_includes, $libcharsetdetect);

my $extra_compiler_flags = '';
# ------------------------------------------------
# On Windows, we will use dlfcn-win32 and winiconv
# ------------------------------------------------
my @additional = ();
if (is_os_type('Windows')) {
    push(@additional, 'dlfcn-win32');
    push(@additional, 'winiconv');
    #
    # This depend on psapi, and perl's cflags always include that by default
    #
    push(@additional_srcdir, File::Spec->catdir('output', '3rdparty', 'dlfcn-win32', 'dlfcn.c'));
    push(@additional_srcdir, File::Spec->catdir('output', '3rdparty', 'winiconv', 'win_iconv.c'));
    #
    # Well, dlfcn.h is at dlfcn-win32 level
    #
    push(@additional_includes, File::Spec->catdir('output', '3rdparty', 'dlfcn-win32'));
    #
    # And we want to make sure we will get the iconv.h from winiconv, not the one from the system
    #
    push(@additional_includes, File::Spec->catdir('output', '3rdparty', 'winiconv'));
    $extra_compiler_flags = "-DICONV_SECOND_ARGUMENT_IS_CONST";
}

$ac->define_var('TCONV_HAVE_ICONV', 1);

# --------------------------------------------------------------------------------
# getopt is only used for the tconv binary - not needed from library point of view
# --------------------------------------------------------------------------------

# -----------------------------------------------------------------
# We depend on these* stuff that can manage themselvess, eventually
# -----------------------------------------------------------------
foreach (qw/genericLogger/, @additional) {
    if (! dircopy(File::Spec->catdir('3rdparty', 'github', $_),
                  File::Spec->catdir('output', '3rdparty', $_))) {
        die "Failed to copy $_";
    }
    print "... Copying $_\n";
    my $CMakeObjects = File::Spec->catfile('output', '3rdparty', $_, 'CMakeObjects.PL');
    if (-e $CMakeObjects) {
        print "... Executing $CMakeObjects\n";
        my $in;
        my $out;
        my $err;
        {
            local $CWD = dirname($CMakeObjects);
            run([$^X, basename($CMakeObjects)], \$in, \$out, \$err) || die "Failed to execute $CMakeObjects" . ($err ? "\n$err" : '');
        }
        if ($out) {
            foreach (grep {defined} split(/\R/, $out)) {
                print "... ... $_\n";
            }
        }
        my $inc = File::Spec->catdir('output', '3rdparty', $_, 'output', 'include');
        push(@additional_includes, $inc) if (-d $inc);
    }
    my $inc = File::Spec->catdir('output', '3rdparty', $_, 'include');
    push(@additional_includes, $inc) if (-d $inc);
}

# ----------------
# Get source files
# ----------------
my @sources;
find(
    {
	no_chdir => 1,
	wanted => sub {
	    my $file = File::Spec->canonpath($_);
	    if (-f $file && ($file =~ /\.c$/ || $file =~ /\.cpp$/)) {
                #
                # We know that ICU.c is an exception
                #
                if (basename($file) ne 'ICU.c') {
                    push(@sources, $file)
                }
	    }
	},
    },
    'src', @additional_srcdir);

# ----------------------------------------------------------------------------------------
# Generate objects
# (individually- not efficient but I do not see how CBuilder mixes C and C++ source files)
# ----------------------------------------------------------------------------------------
my $cbuilder = ExtUtils::CBuilder->new();
my @objects;
my $obj_dir = File::Spec->catfile('output', 'obj4perl');

make_path($obj_dir);
foreach my $source (@sources) {
    my $is_cplusplus = ($source =~ /\.cpp$/i || $source =~ /\.c\+\+$/i);
    my $obj = basename($cbuilder->object_file($source));
    $obj = File::Spec->catfile($obj_dir, basename($cbuilder->object_file($source)));
    push(@objects, $cbuilder->object_file($source));
    $cbuilder->compile(
                       source        => $source,
                       extra_compiler_flags => $extra_compiler_flags,
                       object_file   => $obj,
                       include_dirs  => [ 'include', File::Spec->catdir('output', 'include'), @additional_includes ],
                       'C++'         => $is_cplusplus
                      );
}

# ----
# Done
# ----
exit(EXIT_SUCCESS);

sub get_version {
    open(my $fh, '<', 'CMakeLists.txt') || die "Cannot open CMakeLists.txt, $!";
    my $content = do { local $/; <$fh>; };
    close($fh) || warn "Failed to close CMakeLists.txt, $!";

    my @rc;
    if ($content =~ /^MYPACKAGESTART\s*\(\s*tconv\s+(\d+)\s+(\d+)\s+(\d+)\s*\)/sm) {
        @rc = ($1, $2, $3);
    } else {
        foreach (qw/TCONV_VERSION_MAJOR TCONV_VERSION_MINOR TCONV_VERSION_PATCH/) {
            if ($content =~ /^SET\s*\(\s*$_\s*(\d+)\s*\)/sm) {
                push(@rc, $1);
            } else {
                die "Failed to find $_",
            }
        }
    }

    return join('.', @rc)
}

sub write_config {
    my ($version, $input, $output) = @_;

    make_path(dirname($output));
    open(my $fh, '<', $input) || die "Cannot open $input, $!";
    my $source = do { local $/; <$fh>; };
    close($fh) || warn "Cannot close $input, $!";

    $source =~ s/^[ \t]*#[ \t]*cmakedefine[ \t]+(\w+)+[ \t]+\@([^ \t@]*)\@//smg;

    open($fh, '>', $output) || die "Cannot open $output, $!";
    my $c_va_copy = get_C_WRAPPER('va_copy');
    print $fh <<CONFIG;
#ifndef TCONV_CONFIG_WRAPPER_H
#define TCONV_CONFIG_WRAPPER_H

#include <tconv/config_autoconf.h>

#define TCONV_C_INLINE
#define TCONV_HAVE_ICONV 1
#define TCONV_VERSION "$version"

$c_va_copy

#ifdef C_VA_COPY
#define TCONV_C_VA_COPY C_VA_COPY
#endif

$source

#endif /* TCONV_CONFIG_WRAPPER_H */
CONFIG
    close($fh) || warn "Cannot close $output, $!";
}

sub get_C_WRAPPER {
    my ($what, $required) = @_;

    $required //= 1;

    my $WHAT = uc($what);
    my $error = $required ? "#error \"C_$WHAT is undefined\"" : '';

    my $c_wrapper = <<C_WRAPPER;
#ifdef HAVE_$WHAT
#  define C_$WHAT $what
#else
#  ifdef HAVE__$WHAT
#    define C_$WHAT _$what
#  else
#    ifdef HAVE___$WHAT
#      define C_$WHAT __$what
#    else
$error
#    endif
#  endif
#endif
C_WRAPPER
        
    return $c_wrapper
}

sub write_nscore {
    my ($input, $output) = @_;

    make_path(dirname($output));
    open(my $fh, '<', $input) || die "Cannot open $input, $!";
    my $source = do { local $/; <$fh>; };
    close($fh) || warn "Cannot close $input, $!";

    $source =~ s/^[ \t]*#[ \t]*cmakedefine\b.*?$//smg;
    $source =~ s/^[ \t]*#[ \t]*define\s+SIZEOF.*?$//smg;

    open($fh, '>', $output) || die "Cannot open $output, $!";
    print $fh <<NSCORE;
#ifndef NSCORE_WRAPPER_H
#define NSCORE_WRAPPER_H

#include <tconv/config_autoconf.h>

$source

#endif /* TCONV_CONFIG_WRAPPER_H */
NSCORE
    close($fh) || warn "Cannot close $output, $!";
}
