package Tinkerforge;

# setting the module version
use vars '$VERSION';
our $VERSION = '2.1.0';

=pod

=encoding UTF-8

=head1 NAME

Tinkerforge - Official Perl support for all Tinkerforge Bricks and Bricklets

=head1 VERSION

Version 2.1.0

=head1 ABSTRACT

The Tinkerforge Perl module provides official API support for all Tinkerforge
Bricks and Bricklets. A comprehensive documentation for the Perl bindings can
be found at,
English: L<strict|http://www.tinkerforge.com/en/doc/Software/API_Bindings_Perl.html>
German: L<strict|http://www.tinkerforge.com/de/doc/Software/API_Bindings_Perl.html>

=cut

=head1 INSTALLATION

Just install the package for the device you want to use.
For example, if you want to use the Humidity Bricklet then install
the following package,

Tinkerforge::BrickletHumidity

Installing any package from the Tinkerforge namespace will provide you with everything
that you need to use the perl bindings and you won't have to install anything for a
new device again.

=head1 EXAMPLES

=head2 AUTHENTICATION

    #!/usr/bin/perl

    use Tinkerforge::IPConnection;

    use constant HOST => 'localhost';
    use constant PORT => 4223;
    use constant SECRET => 'My Authentication Secret!';

    # Create IPConnection
    our $ipcon = Tinkerforge::IPConnection->new();

    # Authenticate each time the connection got (re-)established
    sub cb_connected
    {
        my ($connect_reason) = @_;

        if ($connect_reason == $ipcon->CONNECT_REASON_REQUEST)
        {
            print "Connected by request\n";
        }
        elsif ($connect_reason == $ipcon->CONNECT_REASON_AUTO_RECONNECT)
        {
            print "Auto-Reconnect\n";
        }

        # Authenticate first...
        eval
        {
            $ipcon->authenticate(&SECRET);
            print "Authentication succeeded\n";
        };
        if ($!)
        {
            print "Could not authenticate: $!\n";
            return;
        }

        # ...then trigger Enumerate
        $ipcon->enumerate();
    }

    # Print incoming enumeration
    sub cb_enumerate
    {
        my ($uid, $connected_uid, $position, $hardware_version,
        $firmware_version, $device_identifier, $enumeration_type) = @_;

        print "UID: $uid, Enumeration Type: $enumeration_type\n";
    }

    # Register Connected Callback
    $ipcon->register_callback($ipcon->CALLBACK_CONNECTED, 'cb_connected');

    # Register Enumerate Callback
    $ipcon->register_callback($ipcon->CALLBACK_ENUMERATE, 'cb_enumerate');

    # Connect to brickd
    $ipcon->connect(&HOST, &PORT);

    print "Press any key to exit...\n";
    <STDIN>;
    $ipcon->disconnect();

=head2 ENUMERATION

    #!/usr/bin/perl

    use Tinkerforge::IPConnection;

    use constant HOST => 'localhost';
    use constant PORT => 4223;

    # Print incoming enumeration
    sub cb_enumerate()
    {
        my ($uid, $connected_uid, $position, $hardware_version,
            $firmware_version, $device_identifier, $enumeration_type) = @_;

        print "\nUID:               ".$uid;
        print "\nEnumeration Type:  ".$enumeration_type;

        if ($enumeration_type == Tinkerforge::IPConnection->ENUMERATION_TYPE_DISCONNECTED)
        {
            print "\n";
            return 1;
        }

        print "\nConnected UID:     ".$connected_uid;
        print "\nPosition:          ".$position;
        print "\nHardware Version:  ".join('.', @$hardware_version);
        print "\nFirmware Version:  ".join('.', @$firmware_version);
        print "\nDevice Identifier: ".$device_identifier;
        print "\n";
    }

    # Create connection and connect to brickd
    my $ipcon = Tinkerforge::IPConnection->new();

    $ipcon->connect(&HOST, &PORT);

    # Register Enumerate Callback
    $ipcon->register_callback($ipcon->CALLBACK_ENUMERATE, 'cb_enumerate');

    # Trigger Enumerate
    $ipcon->enumerate();

    print "\nPress any key to exit...\n";
    <STDIN>;
    $ipcon->disconnect();

=head2 GETTER CALL

    #!/usr/bin/perl

    use Tinkerforge::IPConnection;
    use Tinkerforge::BrickletHumidity;

    use constant HOST => 'localhost';
    use constant PORT => 4223;
    use constant UID => '7bA'; # Change to your UID

    my $ipcon = Tinkerforge::IPConnection->new(); # Create IP connection
    my $h = Tinkerforge::BrickletHumidity->new(&UID, $ipcon); # Create device object

    $ipcon->connect(&HOST, &PORT); # Connect to brickd
    # Don't use device before ipcon is connected

    # Get current humidity (unit is %RH/10)
    my $rh = $h->get_humidity()/10.0;

    print "\nRelative Humidity: ".$rh." RH%\n";

    print "\nPress any key to exit...\n";
    <STDIN>;
    $ipcon->disconnect();

=head2 SETTER AND CALLBACK

    #!/usr/bin/perl

    use Tinkerforge::IPConnection;
    use Tinkerforge::BrickletHumidity;

    use constant HOST => 'localhost';
    use constant PORT => 4223;
    use constant UID => '7bA'; # Change to your UID

    my $ipcon = Tinkerforge::IPConnection->new(); # Create IP connection
    my $h = Tinkerforge::BrickletHumidity->new(&UID, $ipcon); # Create device object

    # Callback function for humidity callback (parameter has unit %RH/10)
    sub cb_reached
    {
        my ($humidity) = @_;

        if($humidity < 30*10)
        {
            print "\nHumidity too low: ".$humidity/10.0." RH%\n";
        }
        if($humidity > 60*10)
        {
            print "\nHumidity too high: ".$humidity/10.0." RH%\n";
        }

        print "\nRecommended humiditiy for human comfort is 30 to 60 %RH.\n";
    }

    $ipcon->connect(&HOST, &PORT); # Connect to brickd
    # Don't use device before ipcon is connected

    # Get threshold callbacks with a debounce time of 10 seconds (10000ms)
    $h->set_debounce_period(100);

    # Register threshold reached callback to function cb_reached
    $h->register_callback($h->CALLBACK_HUMIDITY_REACHED, 'cb_reached');

    # Configure threshold for "outside of 30 to 60 %RH" (unit is %RH/10)
    $h->set_humidity_callback_threshold('o', 30*10, 60*10);

    print "\nPress any key to exit...\n";
    <STDIN>;
    $ipcon->disconnect();

=head1 AUTHOR

Ishraq Ibne Ashraf C<ishraq@tinkerforge.com>

=head1 LICENCE AND COPYRIGHT

Copyright 2014 Ishraq Ibne Ashraf.

Redistribution and use in source and binary forms of this distribution,
with or without modification, are permitted. See the Creative Commons
Zero (CC0 1.0) License for more details.

=cut
