#!/usr/local/bin/perl -w
use 5.014;
use warnings;

use Test::More tests => 10;

# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl Quote.t'
# Note:
# This test requires internet connection.
# And it will take several minutes to process.

BEGIN {
	use_ok('Finance::YahooJPN::Quote')
};

#######################################################
my $obj1 = Finance::YahooJPN::Quote->new('4689.t');
isa_ok($obj1, 'Finance::YahooJPN::Quote');

my(@noadjust, @adjust, @TOPIX, @Nikkei, @JASDAQ);
my $file = 'noadjust';
while (<DATA>) {
	my $row = $_;
	$row =~ s/\x0D\x0A|\x0D|\x0A/\n/;
	given ($row) {
		when (/^cut1$/) {
			$file = 'adjust';
		}
		when (/^cut2$/) {
			$file = 'TOPIX';
		}
		when (/^cut3$/) {
			$file = 'Nikkei';
		}
		when (/^cut4$/) {
			$file = 'JASDAQ';
		}
		default {
			given ($file) {
				when ('noadjust') {
					push @noadjust, $row;
				}
				when ('adjust') {
					push @adjust, $row;
				}
				when ('TOPIX') {
					push @TOPIX, $row;
				}
				when ('Nikkei') {
					push @Nikkei, $row;
				}
				when ('JASDAQ') {
					push @JASDAQ, $row;
				}
			}
		}
	}
}
my $noadjust = join '', @noadjust;
my $adjust   = join '', @adjust;
my $TOPIX    = join '', @TOPIX;
my $Nikkei   = join '', @Nikkei;
my $JASDAQ   = join '', @JASDAQ;

$obj1->scan('start' => '1997-11-04');
my @tmp = $obj1->output('noadjust' => 1);
my @quote;
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
my $quote = join '', @quote;

is( $quote, $noadjust,
	'run scan() method w/o modification for splits' );

#######################################################
my $obj2 = Finance::YahooJPN::Quote->new('4689.t');
$obj2->scan('start' => '1997-11-04');
@tmp = $obj2->output();
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;

is( $quote, $adjust,
	'run scan() method w/  modification for splits' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('4689.t', 'Start' => '1997-11-04');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $adjust,
	'run historical() method (includes new(), scan() and output())' );

#######################################################
my $obj3 = Finance::YahooJPN::Quote->new('9437.t');
$obj3->scan('start' => '2002-03-26', 'last' => '2002-03-26');
@tmp = $obj3->output();
$quote = $tmp[0];
my $expected = '2002-03-26	341000	351000	329000	333000	208021';

is( $quote, $expected,
	'in a rare case split data is at the top row' );

#######################################################
my $obj4 = Finance::YahooJPN::Quote->new('0000.t');
$obj4->scan('start' => '2003-09-01', 'last' => '2003-09-30');
@tmp = $obj4->output();
$quote = join "\n", @tmp;
$expected = '';

is( $quote, $expected,
	'in a case the specified symbol is absent' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('998405', 'Start' => '2009-01-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $TOPIX,
	'run historical() method for TOPIX' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('998407', 'Start' => '2009-01-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $Nikkei,
	'run historical() method for Nikkei' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('23337', 'Start' => '2009-01-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $JASDAQ,
	'run historical() method for JASDAQ' );

__END__
1997-11-04	2000000	2000000	2000000	2000000	318
1997-11-05	1930000	2170000	1750000	2040000	632
1997-11-06	2070000	2190000	2060000	2060000	220
1997-11-07	2100000	2170000	2030000	2110000	106
1997-11-10	2050000	2060000	1990000	2000000	40
1997-11-11	2010000	2010000	1970000	1970000	22
1997-11-12	1900000	1900000	1700000	1700000	34
1997-11-13	1640000	1900000	1540000	1900000	54
1997-11-14	1850000	1850000	1800000	1800000	4
1997-11-17	1800000	1900000	1800000	1900000	12
1997-11-18	1900000	1900000	1850000	1900000	10
1997-11-19	1950000	2010000	1900000	1900000	24
1997-11-20	1950000	2050000	1950000	2010000	13
1997-11-21	2010000	2100000	2010000	2100000	17
1997-11-25	2070000	2070000	1940000	1940000	15
1997-11-26	1910000	1940000	1910000	1920000	8
1997-11-27	1900000	1910000	1850000	1850000	12
1997-11-28	1840000	1840000	1700000	1840000	18
1997-12-01	1910000	2070000	1910000	2040000	21
1997-12-02	2040000	2140000	2040000	2140000	57
1997-12-03	2140000	2200000	2100000	2150000	60
1997-12-04	2190000	2410000	2190000	2380000	108
1997-12-05	2410000	2780000	2410000	2780000	159
1997-12-08	2980000	3180000	2960000	3000000	106
1997-12-09	2890000	2950000	2850000	2890000	26
1997-12-10	2890000	2900000	2710000	2900000	15
1997-12-11	2940000	3140000	2900000	2900000	20
1997-12-15	2840000	2850000	2750000	2800000	8
1997-12-16	2760000	2800000	2760000	2800000	3
1997-12-17	2870000	2880000	2800000	2850000	4
1997-12-18	2810000	2830000	2810000	2830000	3
1997-12-19	2810000	2810000	2700000	2700000	3
1997-12-22	2690000	2690000	2650000	2650000	2
1997-12-24	2590000	2650000	2590000	2630000	13
1997-12-25	2700000	2940000	2700000	2800000	9
1997-12-26	2800000	2850000	2800000	2850000	2
1997-12-29	2800000	2800000	2720000	2800000	3
1997-12-30	2900000	3000000	2900000	2990000	6
1998-01-05	3000000	3000000	2990000	2990000	3
1998-01-06	2990000	2990000	2900000	2960000	5
1998-01-07	2960000	2960000	2900000	2900000	8
1998-01-08	2900000	2900000	2890000	2890000	4
1998-01-12	2870000	2870000	2800000	2800000	2
1998-01-13	2800000	2800000	2800000	2800000	2
1998-01-14	2810000	2810000	2710000	2710000	4
1998-01-16	2800000	2890000	2790000	2890000	7
1998-01-19	2930000	3050000	2900000	3010000	15
1998-01-20	3000000	3000000	3000000	3000000	4
1998-01-21	2900000	2930000	2900000	2930000	2
1998-01-22	2900000	2950000	2900000	2910000	10
1998-01-23	2910000	2940000	2900000	2900000	4
1998-01-26	2900000	2950000	2900000	2930000	4
1998-01-27	2950000	2950000	2900000	2900000	8
1998-01-28	2890000	2890000	2890000	2890000	1
1998-01-29	2790000	2790000	2700000	2710000	4
1998-01-30	2730000	2750000	2700000	2750000	3
1998-02-02	2750000	2750000	2750000	2750000	1
1998-02-03	2760000	2800000	2750000	2750000	7
1998-02-04	2730000	2730000	2700000	2700000	2
1998-02-05	2650000	2650000	2650000	2650000	1
1998-02-06	2700000	2700000	2600000	2600000	11
1998-02-09	2610000	2610000	2610000	2610000	1
1998-02-10	2610000	2650000	2610000	2650000	2
1998-02-12	2700000	2700000	2700000	2700000	12
1998-02-13	2690000	2700000	2690000	2700000	3
1998-02-16	2700000	2700000	2690000	2700000	18
1998-02-17	2700000	2700000	2610000	2610000	4
1998-02-18	2700000	2780000	2700000	2780000	14
1998-02-19	2800000	2900000	2800000	2840000	13
1998-02-20	2880000	3130000	2880000	3130000	25
1998-02-23	3230000	3300000	3080000	3140000	25
1998-02-24	3140000	3140000	3030000	3030000	10
1998-02-25	3000000	3110000	3000000	3100000	8
1998-02-26	3150000	3160000	3150000	3150000	5
1998-02-27	3200000	3650000	3200000	3650000	41
1998-03-02	3650000	3990000	3630000	3990000	27
1998-03-03	3930000	3930000	3800000	3900000	18
1998-03-04	3890000	3890000	3790000	3870000	6
1998-03-05	3880000	3880000	3690000	3690000	9
1998-03-10	3820000	3950000	3820000	3950000	21
1998-03-11	4050000	4100000	4000000	4100000	22
1998-03-12	4150000	4150000	4000000	4000000	7
1998-03-13	4050000	4350000	4050000	4350000	11
1998-03-16	4280000	4350000	4280000	4350000	4
1998-03-17	4400000	4820000	4400000	4820000	9
1998-03-18	4850000	4850000	4730000	4740000	20
1998-03-19	4700000	4700000	4600000	4630000	8
1998-03-20	4680000	4700000	4650000	4700000	3
1998-03-23	4700000	4700000	4700000	4700000	1
1998-03-24	4690000	4690000	4400000	4500000	4
1998-03-25	4500000	4500000	4450000	4450000	7
1998-03-26	4600000	4650000	4550000	4550000	6
1998-03-27	4550000	4550000	4500000	4500000	6
1998-03-30	4400000	4410000	4400000	4400000	7
1998-03-31	4500000	4500000	4490000	4500000	4
1998-04-01	4500000	4500000	4500000	4500000	2
1998-04-02	4560000	4580000	4560000	4580000	2
1998-04-03	4700000	5050000	4690000	4990000	18
1998-04-06	5300000	5300000	5250000	5300000	8
1998-04-07	5300000	5390000	5290000	5290000	8
1998-04-08	5280000	5280000	5280000	5280000	1
1998-04-09	5270000	5270000	5260000	5270000	3
1998-04-10	5470000	5800000	5470000	5750000	14
1998-04-13	5850000	5990000	5800000	5960000	19
1998-04-14	5960000	6140000	5960000	6100000	15
1998-04-15	6100000	6120000	6050000	6050000	9
1998-04-16	6090000	6090000	6090000	6090000	1
1998-04-17	6030000	6030000	5960000	6000000	11
1998-04-20	6000000	6000000	5960000	5980000	7
1998-04-21	5960000	5960000	5790000	5790000	5
1998-04-23	5090000	5150000	5090000	5150000	8
1998-04-24	5150000	5150000	4450000	4660000	16
1998-04-27	4710000	5160000	4710000	5160000	11
1998-04-28	5000000	5000000	4600000	4600000	10
1998-04-30	4980000	4990000	4900000	4990000	5
1998-05-01	5010000	5490000	5010000	5490000	5
1998-05-06	5190000	5190000	5000000	5000000	6
1998-05-08	4810000	4810000	4800000	4800000	3
1998-05-11	4810000	4810000	4800000	4800000	5
1998-05-12	4800000	4800000	4800000	4800000	1
1998-05-14	4810000	4810000	4800000	4810000	3
1998-05-18	4800000	4900000	4800000	4900000	2
1998-05-19	4900000	5090000	4900000	5090000	7
1998-05-20	5090000	5090000	5090000	5090000	1
1998-05-22	5100000	5100000	4900000	4900000	2
1998-05-26	4800000	4800000	4800000	4800000	1
1998-05-27	4800000	4800000	4800000	4800000	1
1998-05-28	4700000	4700000	4700000	4700000	2
1998-05-29	4650000	4650000	4600000	4600000	2
1998-06-01	4590000	4600000	4590000	4600000	2
1998-06-04	4400000	4400000	4400000	4400000	3
1998-06-05	4300000	4300000	3900000	3900000	6
1998-06-08	3790000	3790000	3400000	3400000	13
1998-06-09	3470000	3470000	3470000	3470000	2
1998-06-10	4050000	4200000	4000000	4200000	11
1998-06-11	4400000	4400000	4360000	4360000	4
1998-06-12	4370000	4370000	4250000	4250000	6
1998-06-15	4250000	4250000	4210000	4210000	7
1998-06-16	4200000	4200000	4200000	4200000	8
1998-06-17	4250000	4300000	4250000	4300000	3
1998-06-18	4410000	4700000	4410000	4700000	7
1998-06-19	4750000	4750000	4700000	4700000	6
1998-06-23	4850000	4900000	4850000	4850000	6
1998-06-24	4960000	4960000	4960000	4960000	2
1998-06-25	4960000	5050000	4900000	5000000	9
1998-06-26	4950000	4950000	4950000	4950000	3
1998-06-29	5000000	5000000	4900000	4900000	2
1998-06-30	4900000	4900000	4850000	4850000	2
1998-07-01	4900000	5000000	4900000	5000000	6
1998-07-02	5050000	5200000	5050000	5200000	7
1998-07-03	5200000	5250000	5200000	5250000	4
1998-07-06	5300000	5350000	5300000	5350000	3
1998-07-07	5510000	6000000	5510000	5770000	13
1998-07-08	6010000	6300000	6000000	6300000	30
1998-07-09	6300000	6300000	6100000	6200000	14
1998-07-10	6200000	6250000	6190000	6200000	24
1998-07-13	6200000	6200000	6200000	6200000	1
1998-07-15	6080000	6080000	6080000	6080000	1
1998-07-17	5980000	5980000	5980000	5980000	1
1998-07-21	5910000	5910000	5910000	5910000	2
1998-07-22	5910000	5910000	5750000	5750000	3
1998-07-23	5600000	5750000	5550000	5750000	8
1998-07-27	5400000	5400000	5400000	5400000	3
1998-08-03	5610000	5610000	5610000	5610000	1
1998-08-04	5610000	5610000	5610000	5610000	1
1998-08-06	5810000	5810000	5500000	5500000	7
1998-08-07	5300000	5300000	5300000	5300000	1
1998-08-10	5300000	5300000	5300000	5300000	2
1998-08-11	5200000	5200000	5200000	5200000	1
1998-08-12	5100000	5100000	5100000	5100000	1
1998-08-13	5100000	5100000	5100000	5100000	1
1998-08-17	5000000	5000000	4800000	4800000	4
1998-08-18	4700000	4700000	4700000	4700000	2
1998-08-19	4630000	4720000	4630000	4710000	9
1998-08-20	4700000	4710000	4690000	4710000	6
1998-08-21	4750000	4800000	4750000	4790000	7
1998-08-24	4790000	4790000	4790000	4790000	1
1998-08-27	4500000	4500000	4500000	4500000	1
1998-08-28	4010000	4110000	4000000	4110000	12
1998-08-31	4060000	4090000	4010000	4010000	18
1998-09-01	3970000	4010000	3900000	4010000	12
1998-09-02	4360000	4370000	4360000	4370000	2
1998-09-03	4460000	4870000	4450000	4870000	5
1998-09-04	4890000	5100000	4890000	5010000	5
1998-09-07	5010000	5020000	5010000	5020000	2
1998-09-08	5020000	5020000	5020000	5020000	1
1998-09-09	5100000	5200000	5100000	5120000	4
1998-09-10	5100000	5100000	5100000	5100000	1
1998-09-11	5020000	5020000	4800000	4800000	3
1998-09-16	4190000	4310000	4190000	4310000	4
1998-09-17	4410000	4410000	4410000	4410000	1
1998-09-18	4450000	4600000	4450000	4600000	3
1998-09-21	4460000	4460000	4460000	4460000	1
1998-09-22	4460000	4460000	4450000	4450000	2
1998-09-24	4540000	4950000	4540000	4950000	3
1998-09-30	4560000	4560000	4560000	4560000	2
1998-10-01	4500000	4500000	4450000	4450000	2
1998-10-02	4300000	4300000	4300000	4300000	2
1998-10-05	4300000	4300000	4300000	4300000	1
1998-10-06	4250000	4250000	4250000	4250000	1
1998-10-07	4250000	4500000	4250000	4500000	5
1998-10-08	4700000	4800000	4700000	4700000	10
1998-10-09	4700000	4700000	4700000	4700000	2
1998-10-12	4700000	4700000	4560000	4560000	11
1998-10-13	4460000	4460000	4250000	4300000	14
1998-10-14	4300000	4300000	4300000	4300000	1
1998-10-15	4300000	4310000	4300000	4300000	9
1998-10-16	4300000	4300000	4300000	4300000	4
1998-10-19	4350000	4350000	4350000	4350000	1
1998-10-21	4300000	4300000	4300000	4300000	1
1998-10-22	4310000	4310000	4310000	4310000	1
1998-10-23	4350000	4400000	4350000	4400000	4
1998-10-26	4400000	4400000	4400000	4400000	2
1998-10-27	4500000	4600000	4500000	4600000	3
1998-10-28	4600000	4600000	4600000	4600000	1
1998-10-30	4650000	4650000	4650000	4650000	2
1998-11-02	4650000	4650000	4650000	4650000	1
1998-11-04	4800000	4850000	4800000	4850000	3
1998-11-05	5000000	5050000	5000000	5000000	7
1998-11-06	5050000	5300000	5050000	5300000	9
1998-11-09	5300000	5300000	5300000	5300000	2
1998-11-10	5300000	5310000	5300000	5300000	3
1998-11-11	5700000	5800000	5690000	5800000	7
1998-11-12	5850000	6040000	5850000	6000000	14
1998-11-13	6010000	6850000	6010000	6850000	50
1998-11-16	6950000	7500000	6950000	7200000	24
1998-11-17	7400000	7400000	6700000	6800000	12
1998-11-18	6800000	6800000	6400000	6500000	9
1998-11-19	6700000	7000000	6700000	7000000	6
1998-11-20	6990000	6990000	6990000	6990000	2
1998-11-24	7030000	7200000	7030000	7190000	8
1998-11-25	7170000	7170000	7050000	7050000	6
1998-11-26	7060000	7100000	7050000	7100000	6
1998-11-27	7150000	7150000	7150000	7150000	1
1998-11-30	7160000	8000000	7160000	7900000	28
1998-12-01	7490000	7490000	7300000	7300000	14
1998-12-02	7500000	7600000	7300000	7300000	4
1998-12-03	7400000	7400000	7400000	7400000	2
1998-12-04	7100000	7100000	6950000	6950000	4
1998-12-07	6950000	7000000	6950000	7000000	2
1998-12-09	6900000	7000000	6900000	7000000	5
1998-12-10	7200000	7200000	7100000	7190000	4
1998-12-11	7200000	7200000	7200000	7200000	1
1998-12-15	7100000	7100000	7100000	7100000	1
1998-12-16	7150000	7150000	7150000	7150000	4
1998-12-17	7150000	7300000	7150000	7170000	10
1998-12-18	7150000	7350000	7100000	7300000	9
1998-12-21	7500000	7800000	7500000	7800000	9
1998-12-22	8120000	8500000	8120000	8400000	18
1998-12-24	8650000	8650000	8400000	8400000	3
1998-12-25	8300000	8300000	8300000	8300000	3
1998-12-28	8250000	8250000	8250000	8250000	3
1998-12-29	8200000	8390000	8200000	8390000	7
1998-12-30	8490000	8490000	8490000	8490000	3
1999-01-04	8400000	8490000	8400000	8450000	3
1999-01-05	8450000	8600000	8450000	8500000	6
1999-01-06	8600000	9500000	8600000	9500000	18
1999-01-07	10500000	10500000	10500000	10500000	9
1999-01-08	11500000	12300000	11500000	12000000	21
1999-01-11	14000000	14000000	14000000	14000000	9
1999-01-12	16000000	16000000	13800000	13800000	16
1999-01-13	13790000	13800000	13000000	13500000	21
1999-01-14	13500000	13500000	13000000	13200000	6
1999-01-18	13000000	15010000	13000000	15010000	15
1999-01-19	15830000	17000000	15830000	17000000	16
1999-01-20	17000000	17000000	16500000	16500000	24
1999-01-21	14500000	14500000	14500000	14500000	6
1999-01-22	12500000	12500000	12500000	12500000	17
1999-01-25	12210000	13800000	12210000	13500000	17
1999-01-26	14310000	14500000	14300000	14500000	5
1999-01-27	14500000	15500000	13500000	14000000	6
1999-01-28	13000000	13000000	13000000	13000000	1
1999-01-29	13800000	15500000	13800000	14800000	26
1999-02-01	14800000	15500000	14400000	14900000	21
1999-02-02	14800000	14800000	14500000	14800000	9
1999-02-03	14800000	14800000	14400000	14400000	5
1999-02-04	14700000	14700000	14200000	14600000	9
1999-02-05	14000000	14400000	14000000	14300000	9
1999-02-08	14200000	14600000	14200000	14600000	5
1999-02-09	14600000	17700000	14600000	16400000	21
1999-02-10	16500000	17500000	15500000	16100000	10
1999-02-12	18500000	19000000	17100000	17200000	7
1999-02-15	17200000	17200000	17200000	17200000	1
1999-02-16	18200000	18200000	17000000	17000000	3
1999-02-17	17000000	18000000	17000000	17500000	7
1999-02-18	17500000	17500000	17500000	17500000	3
1999-02-19	17500000	20000000	17500000	17700000	7
1999-02-22	18500000	20000000	18200000	19000000	11
1999-02-23	19000000	20000000	19000000	19000000	14
1999-02-24	20500000	20500000	19900000	20000000	3
1999-02-25	20500000	21000000	19700000	20000000	7
1999-02-26	20000000	22000000	19600000	22000000	8
1999-03-01	22000000	27000000	22000000	25000000	17
1999-03-02	26000000	27000000	24500000	26000000	16
1999-03-03	26000000	27500000	24500000	26000000	10
1999-03-04	28000000	28000000	25000000	26200000	12
1999-03-05	24500000	28000000	24000000	28000000	8
1999-03-08	30500000	30500000	25500000	27500000	12
1999-03-09	30000000	30000000	25000000	25200000	14
1999-03-10	26000000	27000000	24000000	26000000	15
1999-03-11	24500000	26500000	23500000	25000000	8
1999-03-12	24500000	25000000	22500000	22800000	12
1999-03-15	22800000	24900000	22800000	23000000	7
1999-03-16	25500000	28000000	23500000	25000000	13
1999-03-17	28500000	28500000	25500000	26500000	8
1999-03-18	27000000	28000000	25000000	26250000	11
1999-03-19	26250000	26800000	26250000	26800000	2
1999-03-23	28000000	31000000	26500000	27300000	32
1999-03-24	28000000	30000000	27300000	28000000	38
1999-03-25	31500000	35000000	27600000	32000000	38
1999-03-26	17000000	22500000	16000000	20500000	67
1999-03-29	20500000	26500000	20500000	21700000	71
1999-03-30	22700000	26000000	22700000	24800000	77
1999-03-31	25000000	25500000	24000000	24000000	31
1999-04-01	26500000	26800000	24000000	25900000	72
1999-04-02	27000000	28500000	26000000	26000000	74
1999-04-05	27500000	30000000	27200000	29300000	72
1999-04-06	32500000	36000000	31000000	31800000	128
1999-04-07	32000000	33000000	29800000	32000000	83
1999-04-08	34000000	42000000	34000000	42000000	208
1999-04-09	44000000	60000000	42000000	60000000	231
1999-04-12	59000000	60000000	46000000	50500000	59
1999-04-13	50500000	50500000	36500000	40500000	33
1999-04-14	38000000	50500000	38000000	50000000	73
1999-04-15	48000000	50000000	43000000	47000000	45
1999-04-16	47000000	49000000	41000000	43000000	26
1999-04-19	43000000	44000000	36000000	36000000	26
1999-04-20	32000000	36000000	27000000	33200000	61
1999-04-21	37000000	38000000	33500000	35300000	66
1999-04-22	37000000	41000000	34500000	41000000	37
1999-04-23	41000000	46500000	41000000	43100000	45
1999-04-26	45000000	45000000	38500000	40000000	20
1999-04-27	44000000	46000000	41500000	41500000	20
1999-04-28	41700000	42200000	41500000	42100000	9
1999-04-30	42090000	42090000	41400000	41400000	8
1999-05-06	39400000	39400000	39400000	39400000	7
1999-05-07	37400000	37400000	37400000	37400000	20
1999-05-10	35400000	35400000	35400000	35400000	5
1999-05-11	34000000	34000000	33400000	33400000	18
1999-05-12	35400000	35400000	35400000	35400000	10
1999-05-13	37400000	37400000	37400000	37400000	7
1999-05-14	37400000	38000000	37400000	38000000	10
1999-05-17	36600000	36600000	36000000	36000000	5
1999-05-18	35990000	36000000	34000000	34000000	7
1999-05-19	34010000	34010000	33110000	33500000	9
1999-05-20	33100000	34300000	33100000	34300000	36
1999-05-21	34100000	34100000	32300000	32900000	30
1999-05-24	32890000	32900000	30900000	30900000	9
1999-05-25	28900000	29000000	28900000	29000000	33
1999-05-26	27000000	27000000	27000000	27000000	11
1999-05-27	27400000	27900000	26800000	26800000	44
1999-05-28	24800000	25010000	24800000	24800000	51
1999-05-31	22800000	22800000	22800000	22800000	14
1999-06-01	21400000	24800000	21390000	24800000	39
1999-06-02	26800000	26800000	26800000	26800000	15
1999-06-03	28800000	28800000	28800000	28800000	8
1999-06-04	30800000	30800000	30800000	30800000	13
1999-06-07	32800000	32800000	32800000	32800000	5
1999-06-08	34800000	34800000	34800000	34800000	7
1999-06-09	36800000	36800000	35000000	35600000	53
1999-06-10	37000000	37600000	37000000	37600000	19
1999-06-11	39000000	39200000	37600000	37600000	16
1999-06-14	37600000	37600000	35600000	35600000	3
1999-06-15	33600000	33600000	33600000	33600000	1
1999-06-16	31600000	33000000	31600000	31600000	23
1999-06-17	33600000	33600000	33600000	33600000	7
1999-06-18	35600000	35600000	35600000	35600000	2
1999-06-21	37600000	37600000	37600000	37600000	10
1999-06-22	39600000	39600000	38800000	39300000	19
1999-06-23	39300000	39300000	37300000	37300000	14
1999-06-24	37310000	37900000	36800000	37300000	21
1999-06-25	39300000	39300000	38800000	39300000	9
1999-06-28	40100000	41300000	40000000	41300000	14
1999-06-29	43300000	43300000	43300000	43300000	7
1999-06-30	45300000	45300000	45300000	45300000	7
1999-07-01	47300000	47300000	47300000	47300000	5
1999-07-02	49300000	49300000	49300000	49300000	3
1999-07-05	51300000	51300000	51300000	51300000	19
1999-07-06	53300000	53300000	53300000	53300000	7
1999-07-07	55000000	55000000	51300000	51300000	20
1999-07-08	51700000	53300000	51700000	52900000	28
1999-07-09	50900000	50900000	50900000	50900000	4
1999-07-12	48900000	48900000	48900000	48900000	6
1999-07-13	46900000	47500000	46900000	47500000	20
1999-07-14	47510000	49500000	47510000	49500000	12
1999-07-15	51500000	51500000	51500000	51500000	1
1999-07-19	55500000	55500000	55500000	55500000	1
1999-07-21	57500000	57500000	57500000	57500000	18
1999-07-22	59500000	59500000	59500000	59500000	19
1999-07-23	61500000	61500000	61500000	61500000	14
1999-07-26	59500000	59500000	59500000	59500000	9
1999-07-27	57500000	57500000	57500000	57500000	3
1999-07-28	55500000	55500000	55500000	55500000	8
1999-07-29	54500000	57500000	54500000	57500000	32
1999-07-30	57500000	57500000	57300000	57500000	9
1999-08-02	57510000	57510000	57500000	57500000	5
1999-08-03	57530000	59500000	57530000	59500000	15
1999-08-04	59490000	59490000	57500000	57500000	6
1999-08-05	55500000	55500000	55500000	55500000	3
1999-08-06	53500000	53500000	53500000	53500000	11
1999-08-09	51500000	51500000	51500000	51500000	16
1999-08-10	49500000	49500000	49500000	49500000	7
1999-08-11	47500000	47500000	47500000	47500000	4
1999-08-12	47300000	49500000	47300000	49500000	14
1999-08-13	50000000	51000000	49990000	50000000	22
1999-08-16	52000000	52000000	51500000	51500000	18
1999-08-17	51500000	51500000	51500000	51500000	6
1999-08-18	52000000	53500000	52000000	53500000	11
1999-08-19	55500000	55500000	55500000	55500000	8
1999-08-20	57500000	57500000	57500000	57500000	10
1999-08-23	58700000	58700000	55500000	55500000	5
1999-08-24	55500000	57500000	55500000	57500000	13
1999-08-25	57510000	59500000	57300000	59500000	9
1999-08-26	60900000	61000000	60500000	60500000	12
1999-08-27	60300000	60300000	60000000	60000000	4
1999-08-30	59990000	60210000	59990000	60210000	5
1999-08-31	60200000	62210000	60200000	62210000	8
1999-09-01	64210000	64210000	64210000	64210000	10
1999-09-02	64300000	66210000	64300000	66210000	11
1999-09-03	68000000	68210000	68000000	68000000	8
1999-09-06	69400000	69600000	69400000	69500000	8
1999-09-07	70100000	71500000	70000000	71000000	19
1999-09-08	72800000	73000000	72800000	72800000	11
1999-09-09	73010000	74800000	73010000	74800000	8
1999-09-10	74810000	76000000	74810000	76000000	13
1999-09-13	76000000	78000000	76000000	78000000	14
1999-09-14	80000000	80000000	78000000	78000000	7
1999-09-16	76000000	76000000	76000000	76000000	3
1999-09-17	74000000	74000000	74000000	74000000	3
1999-09-20	72000000	72000000	72000000	72000000	13
1999-09-21	70010000	70010000	70000000	70000000	8
1999-09-22	68000000	68000000	68000000	68000000	1
1999-09-24	66000000	66000000	66000000	66000000	8
1999-09-27	35000000	35000000	33000000	33000000	46
1999-09-28	33200000	35000000	33200000	35000000	35
1999-09-29	37000000	37000000	37000000	37000000	12
1999-09-30	39000000	39000000	39000000	39000000	39
1999-10-01	41000000	41000000	41000000	41000000	12
1999-10-04	43000000	43000000	43000000	43000000	7
1999-10-05	45000000	45000000	45000000	45000000	12
1999-10-06	47000000	47000000	47000000	47000000	17
1999-10-07	49000000	49000000	49000000	49000000	8
1999-10-08	51000000	51000000	51000000	51000000	61
1999-10-12	52400000	52400000	49000000	49000000	60
1999-10-13	47000000	47000000	47000000	47000000	4
1999-10-14	45000000	45000000	45000000	45000000	8
1999-10-15	43000000	43000000	43000000	43000000	8
1999-10-18	41000000	41000000	41000000	41000000	3
1999-10-19	39000000	39000000	39000000	39000000	11
1999-10-20	41000000	41000000	41000000	41000000	41
1999-10-21	43000000	43000000	43000000	43000000	5
1999-10-22	45000000	45000000	45000000	45000000	2
1999-10-26	49000000	49000000	47890000	48800000	119
1999-10-27	51000000	51000000	51000000	51000000	33
1999-10-28	53000000	53000000	52400000	53000000	69
1999-10-29	55000000	55000000	55000000	55000000	40
1999-11-01	57000000	57000000	57000000	57000000	15
1999-11-02	59000000	59000000	59000000	59000000	23
1999-11-04	61000000	61000000	61000000	61000000	27
1999-11-05	63000000	63000000	63000000	63000000	16
1999-11-08	65000000	65000000	65000000	65000000	26
1999-11-09	67000000	67000000	67000000	67000000	16
1999-11-10	69000000	69000000	68000000	69000000	74
1999-11-11	71000000	71000000	71000000	71000000	31
1999-11-12	73000000	73000000	73000000	73000000	8
1999-11-15	75000000	75000000	75000000	75000000	26
1999-11-16	76000000	76000000	73000000	73500000	112
1999-11-17	73700000	73700000	71500000	71500000	29
1999-11-18	69500000	69500000	69500000	69500000	17
1999-11-19	67500000	71500000	67500000	71500000	68
1999-11-22	73500000	73500000	70990000	71500000	59
1999-11-24	71100000	71300000	69500000	69500000	46
1999-11-25	71500000	71500000	71000000	71000000	22
1999-11-26	70400000	70700000	69000000	70700000	23
1999-11-29	72010000	72700000	72000000	72600000	14
1999-11-30	74600000	74600000	71000000	72000000	20
1999-12-01	70600000	70600000	70000000	70000000	21
1999-12-02	70400000	70400000	68000000	68000000	8
1999-12-03	67600000	70000000	67500000	67500000	31
1999-12-06	69500000	69500000	69500000	69500000	2
1999-12-07	71500000	71500000	71500000	71500000	7
1999-12-08	73500000	73500000	73500000	73500000	6
1999-12-09	75500000	75500000	75500000	75500000	15
1999-12-10	77500000	77500000	77500000	77500000	7
1999-12-13	79500000	79500000	79500000	79500000	4
1999-12-14	81500000	81500000	81500000	81500000	18
1999-12-15	83500000	83500000	83500000	83500000	33
1999-12-16	83500000	85500000	81600000	85400000	77
1999-12-17	86800000	87000000	83400000	83400000	39
1999-12-20	83200000	83200000	81400000	81400000	50
1999-12-21	79400000	79400000	79400000	79400000	20
1999-12-22	81400000	81400000	81400000	81400000	9
1999-12-24	83400000	83400000	83400000	83400000	3
1999-12-27	85400000	85400000	85400000	85400000	15
1999-12-28	87400000	87400000	87400000	87400000	2
1999-12-30	91400000	91400000	91400000	91400000	4
2000-01-04	93400000	93400000	93400000	93400000	4
2000-01-05	93390000	93400000	91400000	91400000	46
2000-01-06	91400000	91400000	89400000	89400000	8
2000-01-07	87400000	87400000	87400000	87400000	8
2000-01-11	89400000	89400000	89400000	89400000	3
2000-01-12	91400000	91400000	91400000	91400000	27
2000-01-13	91800000	93400000	91200000	93400000	31
2000-01-14	95400000	95400000	95400000	95400000	27
2000-01-17	97400000	97400000	97400000	97400000	32
2000-01-18	99400000	99400000	99400000	99400000	12
2000-01-19	101400000	101400000	99100000	99100000	54
2000-01-20	95400000	95400000	95400000	95400000	13
2000-01-21	93600000	93700000	93400000	93400000	30
2000-01-24	93390000	95400000	93390000	95400000	26
2000-01-25	95200000	97400000	94200000	97000000	21
2000-01-27	101000000	101000000	101000000	101000000	5
2000-01-28	103000000	103000000	103000000	103000000	3
2000-01-31	105000000	105000000	105000000	105000000	2
2000-02-01	107000000	107000000	107000000	107000000	2
2000-02-02	109000000	109000000	109000000	109000000	3
2000-02-03	111000000	111000000	111000000	111000000	14
2000-02-04	113000000	113000000	113000000	113000000	42
2000-02-07	115000000	115000000	115000000	115000000	7
2000-02-08	117000000	117000000	117000000	117000000	9
2000-02-09	119000000	119000000	119000000	119000000	1
2000-02-10	121000000	121000000	121000000	121000000	15
2000-02-14	128000000	128000000	128000000	128000000	6
2000-02-15	135000000	135000000	135000000	135000000	8
2000-02-16	142000000	142000000	142000000	142000000	9
2000-02-17	149000000	149000000	149000000	149000000	11
2000-02-18	156000000	156000000	156000000	156000000	16
2000-02-21	163000000	163000000	163000000	163000000	23
2000-02-22	167900000	167900000	156000000	156000000	77
2000-02-23	153200000	163000000	153000000	160950000	60
2000-02-24	158850000	159550000	153950000	157000000	11
2000-02-25	150000000	150000000	150000000	150000000	5
2000-02-28	150000000	150000000	143000000	143000000	11
2000-02-29	141700000	150000000	141000000	150000000	36
2000-03-01	154900000	156000000	153600000	156000000	18
2000-03-02	151100000	154000000	149000000	154000000	7
2000-03-03	149950000	150000000	147000000	147000000	5
2000-03-06	140000000	140000000	140000000	140000000	5
2000-03-07	133000000	133000000	133000000	133000000	7
2000-03-08	126000000	134450000	126000000	134450000	22
2000-03-09	138650000	140150000	138000000	140000000	7
2000-03-10	134400000	141400000	133000000	133000000	15
2000-03-13	126000000	126000000	126000000	126000000	8
2000-03-14	119000000	119000000	119000000	119000000	3
2000-03-15	112000000	126000000	112000000	126000000	37
2000-03-16	127400000	131600000	119500000	120000000	26
2000-03-17	124200000	125600000	115000000	115000000	39
2000-03-21	112900000	122000000	112900000	122000000	23
2000-03-22	129000000	129000000	129000000	129000000	14
2000-03-23	135950000	136000000	133000000	134400000	9
2000-03-24	134400000	134400000	127400000	127400000	7
2000-03-27	120400000	120400000	120400000	120400000	15
2000-03-28	53200000	55300000	53200000	55300000	110
2000-03-29	62300000	62300000	62300000	62300000	13
2000-03-30	69300000	69300000	65600000	69300000	138
2000-03-31	64400000	64400000	62300000	62300000	30
2000-04-03	55300000	55300000	55300000	55300000	22
2000-04-04	48300000	48300000	48300000	48300000	14
2000-04-05	43450000	53300000	43450000	50000000	171
2000-04-06	49300000	49800000	43600000	45900000	41
2000-04-07	42900000	50000000	42500000	50000000	69
2000-04-10	55000000	57000000	55000000	56700000	67
2000-04-11	52500000	53200000	52500000	53000000	20
2000-04-12	48500000	50300000	48000000	49950000	42
2000-04-13	49950000	49950000	44950000	44950000	49
2000-04-14	39950000	39950000	39950000	39950000	20
2000-04-17	34950000	34950000	34950000	34950000	21
2000-04-18	29950000	29950000	29950000	29950000	19
2000-04-19	31950000	31950000	31100000	31950000	184
2000-04-20	36950000	36950000	36950000	36950000	78
2000-04-21	41950000	41950000	41950000	41950000	2
2000-04-24	46950000	46950000	46950000	46950000	10
2000-04-25	51950000	51950000	51950000	51950000	37
2000-04-26	58950000	58950000	58950000	58950000	55
2000-04-27	59650000	63000000	56900000	59500000	244
2000-04-28	57400000	57400000	52500000	54000000	60
2000-05-01	50500000	51600000	49500000	51000000	75
2000-05-02	55000000	55000000	52900000	53700000	45
2000-05-08	53700000	53700000	50500000	51000000	14
2000-05-09	48100000	48100000	46200000	46200000	27
2000-05-10	43000000	44500000	42500000	44000000	42
2000-05-11	39000000	40000000	39000000	39000000	47
2000-05-12	44000000	44000000	43000000	44000000	25
2000-05-15	43950000	43950000	41000000	41000000	13
2000-05-16	41500000	42500000	39000000	39100000	51
2000-05-17	43100000	43100000	38550000	38600000	55
2000-05-18	38600000	40000000	34000000	39500000	48
2000-05-19	34500000	37000000	34500000	37000000	80
2000-05-22	32000000	32000000	32000000	32000000	24
2000-05-23	30500000	31900000	27000000	27000000	182
2000-05-24	25000000	27200000	25000000	25500000	216
2000-05-25	27500000	27500000	27350000	27500000	139
2000-05-26	28500000	29500000	28000000	29500000	92
2000-05-29	31500000	31500000	31500000	31500000	9
2000-05-30	36500000	36500000	36500000	36500000	66
2000-05-31	37500000	40000000	32000000	35400000	131
2000-06-01	33000000	38000000	33000000	37600000	40
2000-06-02	39000000	39000000	37100000	38800000	49
2000-06-05	43300000	43800000	42800000	43800000	54
2000-06-06	43300000	47000000	42000000	46100000	81
2000-06-07	45100000	45100000	42200000	43500000	36
2000-06-08	44150000	44150000	41000000	41800000	24
2000-06-09	41500000	41900000	41500000	41500000	19
2000-06-12	41500000	43400000	41500000	43000000	25
2000-06-13	43500000	44000000	43000000	43650000	44
2000-06-14	44100000	46000000	44100000	46000000	39
2000-06-15	43450000	44300000	43000000	43850000	41
2000-06-16	42950000	43000000	42500000	43000000	24
2000-06-19	43500000	44000000	43500000	44000000	12
2000-06-20	45900000	47000000	45500000	47000000	24
2000-06-21	47500000	50000000	47300000	49600000	53
2000-06-22	49550000	49550000	45000000	45000000	47
2000-06-23	41000000	43000000	41000000	43000000	29
2000-06-26	41000000	41500000	39900000	39900000	24
2000-06-27	37900000	39900000	37900000	39000000	19
2000-06-28	38950000	40000000	38950000	40000000	11
2000-06-29	41500000	42000000	40100000	41500000	26
2000-06-30	41550000	43000000	41200000	42050000	23
2000-07-03	42500000	42500000	41000000	41000000	12
2000-07-04	41000000	41400000	39000000	40000000	22
2000-07-05	39950000	39950000	38000000	38000000	10
2000-07-06	35000000	35500000	33550000	33550000	37
2000-07-07	34650000	38000000	34650000	38000000	21
2000-07-10	40000000	40000000	39000000	39000000	17
2000-07-11	35500000	37200000	35100000	37200000	25
2000-07-12	38550000	39000000	33500000	34100000	83
2000-07-13	35950000	38000000	35500000	38000000	46
2000-07-14	37500000	38000000	37000000	37000000	14
2000-07-17	36000000	36400000	35000000	35000000	57
2000-07-18	34000000	34000000	32000000	33500000	52
2000-07-19	33500000	33500000	31500000	33300000	47
2000-07-21	33800000	34500000	32500000	32600000	27
2000-07-24	31800000	31800000	30500000	31000000	41
2000-07-25	31000000	31400000	29600000	31000000	68
2000-07-26	31000000	33000000	31000000	31100000	38
2000-07-27	29000000	29000000	27800000	28200000	46
2000-07-28	26200000	26500000	26200000	26200000	124
2000-07-31	24200000	24200000	24200000	24200000	49
2000-08-01	26200000	26200000	26200000	26200000	53
2000-08-02	28200000	28200000	27800000	28050000	129
2000-08-03	28850000	28850000	26700000	27700000	61
2000-08-04	27650000	27650000	26300000	26500000	49
2000-08-07	25950000	28000000	25950000	27950000	52
2000-08-08	29000000	29950000	29000000	29950000	37
2000-08-09	31950000	31950000	31950000	31950000	34
2000-08-10	36950000	36950000	35000000	36950000	125
2000-08-11	35000000	35500000	33000000	34500000	72
2000-08-14	35000000	36500000	35000000	35850000	9
2000-08-15	35900000	37500000	35000000	35800000	72
2000-08-16	35800000	36000000	35500000	35800000	33
2000-08-17	34500000	34500000	33200000	34450000	8
2000-08-18	32450000	32950000	31000000	31900000	31
2000-08-21	31900000	33100000	31900000	32900000	25
2000-08-22	32800000	32800000	31650000	32500000	13
2000-08-23	32500000	32650000	32450000	32650000	15
2000-08-24	32800000	34000000	32800000	33650000	75
2000-08-25	34650000	35500000	34500000	35300000	98
2000-08-28	35800000	37000000	35000000	35000000	62
2000-08-29	30500000	31000000	30000000	30000000	80
2000-08-30	29500000	32400000	29500000	31500000	52
2000-08-31	31500000	32800000	31500000	32000000	13
2000-09-01	33500000	33900000	32000000	32500000	39
2000-09-04	32000000	32000000	29500000	30200000	28
2000-09-05	31200000	32400000	29100000	29300000	72
2000-09-06	29300000	30500000	28700000	29400000	42
2000-09-07	28400000	29400000	28400000	28600000	43
2000-09-08	28200000	30000000	28200000	28900000	38
2000-09-11	28300000	28700000	27500000	28500000	46
2000-09-12	28300000	28300000	27550000	28000000	26
2000-09-13	28100000	29100000	28100000	28300000	25
2000-09-14	28900000	28900000	28000000	28050000	14
2000-09-18	26050000	26500000	26050000	26050000	56
2000-09-19	24050000	24550000	24050000	24550000	117
2000-09-20	25000000	26500000	25000000	26400000	52
2000-09-21	26550000	27500000	25800000	26500000	69
2000-09-22	25500000	25500000	24600000	25000000	44
2000-09-25	25050000	26000000	25050000	25550000	52
2000-09-26	13000000	14750000	13000000	14600000	91
2000-09-27	14600000	14600000	13700000	14000000	64
2000-09-28	13050000	13900000	13050000	13550000	50
2000-09-29	14000000	14200000	13200000	13400000	90
2000-10-02	12800000	13500000	12800000	13500000	42
2000-10-03	13500000	13500000	12400000	12700000	108
2000-10-04	11900000	12950000	11700000	12950000	102
2000-10-05	12950000	13100000	12550000	13100000	61
2000-10-06	12500000	13000000	12500000	12700000	57
2000-10-10	12550000	12600000	12100000	12450000	88
2000-10-11	12100000	12500000	12000000	12300000	83
2000-10-12	11950000	12000000	11200000	11850000	99
2000-10-13	10850000	12500000	10850000	12350000	174
2000-10-16	12150000	12450000	11800000	12250000	144
2000-10-17	12100000	12150000	11200000	11500000	91
2000-10-18	9500000	10200000	9500000	9900000	309
2000-10-19	10400000	10900000	10300000	10900000	143
2000-10-20	12300000	12500000	11700000	12100000	95
2000-10-23	12200000	12200000	11300000	11300000	43
2000-10-24	10950000	11500000	10950000	11200000	45
2000-10-25	11000000	11200000	10400000	10400000	44
2000-10-26	10000000	10400000	9900000	10300000	53
2000-10-27	10600000	10900000	10500000	10500000	33
2000-10-30	10500000	10500000	10100000	10200000	28
2000-10-31	10050000	10050000	9300000	9400000	139
2000-11-01	10400000	10400000	10400000	10400000	54
2000-11-02	10800000	11400000	10600000	11200000	77
2000-11-06	11500000	12700000	11500000	12700000	142
2000-11-07	12500000	12500000	11800000	12500000	52
2000-11-08	12550000	13300000	12500000	12750000	99
2000-11-09	12450000	12750000	11900000	12000000	64
2000-11-10	11500000	11800000	11300000	11300000	64
2000-11-13	9900000	10450000	9800000	10300000	124
2000-11-14	10500000	10950000	10500000	10900000	60
2000-11-15	11700000	12000000	11550000	11550000	52
2000-11-16	11200000	11200000	10600000	10900000	20
2000-11-17	10900000	10900000	10450000	10450000	66
2000-11-20	10450000	10550000	9600000	9800000	140
2000-11-21	9000000	9400000	8910000	8920000	362
2000-11-22	8320000	8330000	7920000	7920000	300
2000-11-24	6920000	6920000	6920000	6920000	221
2000-11-27	7010000	7300000	6500000	6750000	880
2000-11-28	6500000	7750000	6020000	7750000	464
2000-11-29	7150000	7200000	6850000	7000000	305
2000-11-30	6900000	7150000	6500000	6800000	233
2000-12-01	6540000	7190000	6540000	7150000	304
2000-12-04	7400000	7920000	7340000	7860000	289
2000-12-05	7960000	8050000	7670000	7880000	188
2000-12-06	8880000	8880000	8880000	8880000	232
2000-12-07	8180000	8600000	8000000	8000000	294
2000-12-08	7800000	8460000	7550000	8460000	233
2000-12-11	8730000	8950000	8620000	8840000	222
2000-12-12	8800000	8800000	8050000	8350000	166
2000-12-13	8340000	8490000	7810000	8480000	179
2000-12-14	7990000	8250000	7950000	7950000	93
2000-12-15	7650000	7700000	7260000	7290000	127
2000-12-18	7090000	7090000	6680000	7050000	225
2000-12-19	6750000	6750000	6400000	6470000	206
2000-12-20	5710000	6300000	5710000	5910000	228
2000-12-21	5500000	5600000	5120000	5200000	274
2000-12-22	5610000	5960000	5610000	5900000	116
2000-12-25	6500000	6680000	6400000	6480000	81
2000-12-26	6500000	6500000	6210000	6350000	75
2000-12-27	6500000	6850000	6440000	6780000	181
2000-12-28	6510000	6850000	6500000	6700000	80
2000-12-29	6800000	6840000	6720000	6720000	49
2001-01-04	7120000	7120000	6620000	6760000	129
2001-01-05	6760000	6760000	6360000	6500000	95
2001-01-09	6000000	6010000	5500000	5500000	189
2001-01-10	5660000	5900000	5580000	5750000	124
2001-01-11	5490000	5490000	4760000	5240000	176
2001-01-12	5210000	5340000	4880000	5090000	385
2001-01-15	4990000	5290000	4950000	5100000	582
2001-01-16	5300000	6100000	5280000	6100000	490
2001-01-17	5930000	6200000	5600000	6080000	401
2001-01-18	6580000	7000000	6500000	6850000	585
2001-01-19	7260000	7850000	7210000	7850000	463
2001-01-22	7450000	7750000	6850000	6940000	464
2001-01-23	6790000	7010000	6580000	6890000	148
2001-01-24	7190000	7650000	7090000	7520000	355
2001-01-25	8120000	8520000	8100000	8520000	471
2001-01-26	8320000	8550000	8200000	8400000	582
2001-01-29	8180000	8450000	8100000	8110000	137
2001-01-30	8400000	8500000	8230000	8450000	186
2001-01-31	8650000	8650000	8380000	8580000	140
2001-02-01	8400000	8600000	8150000	8490000	145
2001-02-02	8520000	8630000	8400000	8540000	162
2001-02-05	8000000	8160000	7720000	7730000	151
2001-02-06	7750000	8200000	7700000	8150000	130
2001-02-07	8050000	8300000	8000000	8170000	193
2001-02-08	8140000	8150000	7740000	7800000	259
2001-02-09	7690000	8100000	7600000	7990000	149
2001-02-13	7940000	7950000	7450000	7490000	164
2001-02-14	7250000	7250000	7010000	7040000	174
2001-02-15	7000000	7150000	6910000	7020000	192
2001-02-16	7290000	7300000	6990000	7010000	195
2001-02-19	6990000	7010000	6610000	6900000	141
2001-02-20	6980000	7030000	6830000	6950000	123
2001-02-21	6650000	6780000	6500000	6600000	109
2001-02-22	6200000	6300000	5990000	6210000	171
2001-02-23	6110000	7030000	6050000	7000000	162
2001-02-26	6950000	6990000	6600000	6640000	62
2001-02-27	6720000	6820000	6700000	6700000	59
2001-02-28	6600000	6700000	6360000	6440000	80
2001-03-01	6300000	6600000	6240000	6400000	60
2001-03-02	6500000	6650000	6300000	6350000	77
2001-03-05	6250000	6250000	5650000	6110000	391
2001-03-06	6400000	6600000	6220000	6570000	202
2001-03-07	7000000	7010000	6490000	6550000	164
2001-03-08	5550000	5850000	5550000	5750000	1294
2001-03-09	5550000	5940000	5500000	5700000	436
2001-03-12	5490000	5660000	5480000	5660000	112
2001-03-13	5100000	5150000	4850000	5020000	433
2001-03-14	5220000	5350000	4920000	4930000	212
2001-03-15	4700000	5000000	4670000	5000000	399
2001-03-16	5100000	5400000	5100000	5320000	347
2001-03-19	5120000	5590000	5120000	5550000	232
2001-03-21	5350000	5540000	5350000	5540000	140
2001-03-22	5490000	5490000	5150000	5200000	144
2001-03-23	5440000	5690000	5410000	5690000	366
2001-03-26	5910000	6050000	5830000	5990000	283
2001-03-27	6090000	6090000	5650000	5790000	133
2001-03-28	6050000	6060000	5800000	5890000	146
2001-03-29	5600000	5700000	5590000	5610000	83
2001-03-30	5600000	5700000	5250000	5320000	176
2001-04-02	5320000	5320000	5150000	5170000	165
2001-04-03	5070000	5200000	5030000	5120000	177
2001-04-04	5020000	5020000	4900000	4950000	378
2001-04-05	5400000	5410000	5150000	5410000	220
2001-04-06	5910000	5920000	5650000	5720000	464
2001-04-09	5500000	5500000	5170000	5210000	122
2001-04-10	5250000	5350000	5170000	5210000	92
2001-04-11	5320000	5370000	5220000	5330000	186
2001-04-12	5490000	5550000	5380000	5460000	165
2001-04-13	5550000	5580000	5300000	5350000	169
2001-04-16	5300000	5300000	5100000	5170000	85
2001-04-17	5070000	5160000	5050000	5100000	78
2001-04-18	5200000	5340000	5200000	5290000	134
2001-04-19	5600000	5670000	5530000	5620000	437
2001-04-20	5690000	5760000	5540000	5610000	182
2001-04-23	5010000	5150000	4990000	5000000	550
2001-04-24	4850000	4900000	4650000	4790000	662
2001-04-25	4790000	4960000	4790000	4900000	159
2001-04-26	4850000	4910000	4680000	4750000	264
2001-04-27	4770000	4880000	4760000	4850000	147
2001-05-01	5000000	5070000	4980000	5000000	140
2001-05-02	5100000	5100000	4960000	5080000	159
2001-05-07	5130000	5280000	5060000	5250000	220
2001-05-08	5350000	5470000	5280000	5350000	348
2001-05-09	5400000	5450000	5290000	5300000	185
2001-05-10	5200000	5230000	5100000	5170000	64
2001-05-11	5100000	5110000	5060000	5070000	72
2001-05-14	5050000	5050000	4960000	4960000	79
2001-05-15	4910000	4990000	4900000	4960000	195
2001-05-16	4980000	4980000	4900000	4900000	82
2001-05-17	5000000	5070000	4860000	4900000	142
2001-05-18	4900000	4950000	4860000	4880000	99
2001-05-21	4870000	4920000	4860000	4870000	40
2001-05-22	5000000	5000000	4920000	4970000	157
2001-05-23	5000000	5000000	4870000	4880000	76
2001-05-24	4850000	4860000	4750000	4760000	111
2001-05-25	4770000	4810000	4650000	4810000	125
2001-05-28	4710000	4710000	4600000	4610000	125
2001-05-29	4580000	4580000	4400000	4570000	181
2001-05-30	4430000	4500000	4400000	4440000	101
2001-05-31	4290000	4320000	3970000	4040000	593
2001-06-01	4050000	4090000	3820000	3860000	376
2001-06-04	3860000	3860000	3600000	3790000	227
2001-06-05	3790000	4250000	3710000	4240000	247
2001-06-06	4300000	4330000	4030000	4030000	271
2001-06-07	3920000	4090000	3850000	4090000	234
2001-06-08	4170000	4180000	4070000	4120000	110
2001-06-11	4070000	4070000	3940000	3980000	89
2001-06-12	3930000	3930000	3840000	3850000	111
2001-06-13	3900000	3910000	3860000	3900000	48
2001-06-14	3870000	3910000	3840000	3900000	71
2001-06-15	3880000	3880000	3770000	3830000	104
2001-06-18	3800000	3830000	3700000	3700000	69
2001-06-19	4110000	4160000	3890000	3920000	324
2001-06-20	3950000	3950000	3820000	3830000	272
2001-06-21	4030000	4330000	4010000	4330000	512
2001-06-22	4530000	4800000	4500000	4560000	769
2001-06-25	4600000	4630000	4320000	4450000	153
2001-06-26	4400000	4480000	4310000	4390000	122
2001-06-27	4490000	4490000	4330000	4350000	137
2001-06-28	4340000	4340000	4010000	4100000	180
2001-06-29	4150000	4360000	4140000	4280000	171
2001-07-02	4300000	4300000	4160000	4240000	96
2001-07-03	4230000	4320000	4180000	4210000	66
2001-07-04	4060000	4120000	3950000	3970000	152
2001-07-05	3950000	3980000	3850000	3900000	111
2001-07-06	3800000	3880000	3790000	3880000	95
2001-07-09	3530000	3640000	3510000	3550000	212
2001-07-10	3560000	3620000	3460000	3490000	156
2001-07-11	3400000	3420000	3320000	3330000	104
2001-07-12	3630000	3830000	3550000	3830000	202
2001-07-13	3900000	3970000	3790000	3870000	259
2001-07-16	3830000	3830000	3580000	3680000	85
2001-07-17	3500000	3530000	3400000	3480000	212
2001-07-18	3490000	3540000	3350000	3390000	102
2001-07-19	3330000	3470000	3320000	3470000	115
2001-07-23	3470000	3470000	3280000	3370000	103
2001-07-24	3360000	3460000	3320000	3450000	88
2001-07-25	3310000	3550000	3310000	3450000	95
2001-07-26	3410000	3520000	3370000	3450000	54
2001-07-27	3470000	3510000	3430000	3450000	42
2001-07-30	3470000	3470000	3300000	3380000	111
2001-07-31	3350000	3550000	3350000	3510000	116
2001-08-01	3560000	3610000	3450000	3470000	68
2001-08-02	3540000	3600000	3510000	3580000	107
2001-08-03	3620000	3900000	3550000	3790000	192
2001-08-06	3780000	3800000	3590000	3640000	62
2001-08-07	3630000	3630000	3450000	3450000	144
2001-08-08	3480000	3510000	3400000	3410000	80
2001-08-09	3350000	3440000	3340000	3360000	79
2001-08-10	3370000	3430000	3350000	3400000	45
2001-08-13	3400000	3400000	3300000	3300000	52
2001-08-14	3340000	3440000	3330000	3380000	51
2001-08-15	3310000	3340000	3310000	3320000	35
2001-08-16	3300000	3310000	3220000	3250000	84
2001-08-17	3260000	3270000	3170000	3190000	76
2001-08-20	3100000	3100000	3000000	3010000	108
2001-08-21	3010000	3070000	2990000	3000000	105
2001-08-22	2950000	2950000	2850000	2850000	182
2001-08-23	2870000	2890000	2650000	2650000	196
2001-08-24	2480000	2550000	2450000	2480000	308
2001-08-27	2570000	2650000	2560000	2580000	100
2001-08-28	2530000	2530000	2380000	2460000	183
2001-08-29	2360000	2410000	2330000	2360000	80
2001-08-30	2300000	2440000	2270000	2420000	108
2001-08-31	2240000	2300000	2220000	2220000	254
2001-09-03	2020000	2020000	2020000	2020000	50
2001-09-04	1820000	2000000	1820000	2000000	503
2001-09-05	1880000	2200000	1880000	2050000	677
2001-09-06	2070000	2250000	2030000	2250000	505
2001-09-07	2250000	2450000	2220000	2450000	626
2001-09-10	2450000	2650000	2250000	2630000	524
2001-09-11	2710000	2800000	2700000	2750000	370
2001-09-12	2550000	2550000	2550000	2550000	56
2001-09-13	2450000	2450000	2450000	2450000	61
2001-09-14	2350000	2510000	2350000	2450000	480
2001-09-17	2330000	2330000	2250000	2250000	146
2001-09-18	2270000	2450000	2270000	2450000	354
2001-09-19	2500000	2610000	2430000	2430000	270
2001-09-20	2390000	2390000	2300000	2320000	199
2001-09-21	2180000	2220000	2160000	2180000	216
2001-09-25	2300000	2340000	2220000	2220000	111
2001-09-26	2230000	2260000	2220000	2230000	37
2001-09-27	2210000	2220000	2100000	2120000	227
2001-09-28	2200000	2230000	2180000	2200000	158
2001-10-01	2220000	2400000	2190000	2310000	219
2001-10-02	2330000	2510000	2330000	2480000	242
2001-10-03	2560000	2590000	2480000	2500000	240
2001-10-04	2600000	2610000	2510000	2530000	109
2001-10-05	2550000	2570000	2510000	2530000	171
2001-10-09	2510000	2510000	2410000	2420000	61
2001-10-10	2400000	2500000	2340000	2430000	64
2001-10-11	2520000	2630000	2520000	2630000	229
2001-10-12	2830000	2830000	2830000	2830000	119
2001-10-15	2820000	3030000	2760000	3030000	426
2001-10-16	3100000	3510000	3090000	3330000	730
2001-10-17	3510000	3830000	3510000	3830000	554
2001-10-18	3840000	4220000	3550000	3700000	745
2001-10-19	3950000	4100000	3710000	4050000	422
2001-10-22	4200000	4280000	3900000	4000000	625
2001-10-23	4050000	4060000	3500000	3630000	804
2001-10-24	3580000	3710000	3300000	3610000	464
2001-10-25	3710000	3850000	3620000	3800000	358
2001-10-26	3870000	3910000	3750000	3770000	236
2001-10-29	3720000	3770000	3550000	3700000	195
2001-10-30	3550000	3610000	3470000	3510000	292
2001-10-31	3420000	3530000	3410000	3410000	154
2001-11-01	3510000	3610000	3480000	3560000	155
2001-11-02	3700000	3750000	3530000	3550000	127
2001-11-05	3530000	3550000	3450000	3490000	61
2001-11-06	3620000	3750000	3550000	3750000	230
2001-11-07	3850000	3860000	3680000	3680000	267
2001-11-08	3630000	3650000	3530000	3580000	83
2001-11-09	3530000	3560000	3480000	3490000	116
2001-11-12	3500000	3570000	3490000	3530000	44
2001-11-13	3520000	3520000	3450000	3460000	103
2001-11-14	3560000	3590000	3510000	3580000	153
2001-11-15	3680000	3870000	3650000	3840000	477
2001-11-16	3890000	4200000	3760000	4100000	843
2001-11-19	4200000	4300000	4170000	4180000	332
2001-11-20	4200000	4320000	4170000	4170000	486
2001-11-21	4040000	4120000	3950000	4040000	227
2001-11-22	4000000	4050000	3960000	4000000	101
2001-11-26	4110000	4260000	4110000	4220000	191
2001-11-27	4320000	4720000	4300000	4610000	958
2001-11-28	4590000	5020000	4480000	4830000	857
2001-11-29	4680000	4770000	4400000	4500000	539
2001-11-30	4650000	4710000	4520000	4520000	178
2001-12-03	4540000	4550000	4250000	4350000	149
2001-12-04	4380000	4450000	4280000	4400000	125
2001-12-05	4550000	4650000	4460000	4570000	223
2001-12-06	4800000	4970000	4760000	4780000	547
2001-12-07	4980000	5140000	4920000	4960000	447
2001-12-10	4880000	4910000	4660000	4680000	136
2001-12-11	4650000	4670000	4510000	4650000	161
2001-12-12	4690000	5000000	4690000	4990000	376
2001-12-13	4970000	5060000	4880000	4960000	265
2001-12-14	4760000	4870000	4720000	4850000	240
2001-12-17	4900000	4910000	4720000	4850000	149
2001-12-18	4850000	4850000	4350000	4350000	411
2001-12-19	4300000	4500000	4010000	4090000	465
2001-12-20	4010000	4040000	3650000	3800000	393
2001-12-21	3700000	3870000	3680000	3700000	209
2001-12-25	3750000	3860000	3660000	3850000	96
2001-12-26	3950000	4150000	3900000	3920000	230
2001-12-27	4150000	4150000	4030000	4080000	149
2001-12-28	4150000	4180000	4080000	4080000	113
2002-01-04	4210000	4330000	4170000	4300000	180
2002-01-07	4350000	4420000	4340000	4400000	172
2002-01-08	4410000	4870000	4350000	4800000	483
2002-01-09	4900000	5000000	4820000	4970000	415
2002-01-10	4940000	4940000	4550000	4610000	227
2002-01-11	4500000	4740000	4450000	4720000	191
2002-01-15	4590000	4850000	4590000	4680000	392
2002-01-16	4700000	4920000	4700000	4920000	442
2002-01-17	4840000	4890000	4770000	4810000	230
2002-01-18	4810000	4980000	4770000	4940000	405
2002-01-21	4850000	4930000	4700000	4700000	290
2002-01-22	4450000	4540000	4200000	4200000	588
2002-01-23	3900000	4050000	3880000	3900000	569
2002-01-24	4000000	4080000	3830000	3850000	251
2002-01-25	3960000	4070000	3860000	3930000	193
2002-01-28	3960000	4020000	3850000	3920000	135
2002-01-29	3870000	3920000	3850000	3910000	160
2002-01-30	3880000	4000000	3850000	3920000	121
2002-01-31	4000000	4400000	4000000	4400000	525
2002-02-01	4650000	4680000	4500000	4660000	705
2002-02-04	4630000	4630000	4450000	4460000	178
2002-02-05	4210000	4300000	4160000	4190000	257
2002-02-06	4140000	4490000	4140000	4470000	230
2002-02-07	4420000	4600000	4350000	4600000	460
2002-02-08	4450000	4560000	4310000	4510000	364
2002-02-12	4460000	4550000	4320000	4360000	349
2002-02-13	4400000	4610000	4400000	4600000	208
2002-02-14	4550000	4840000	4550000	4830000	443
2002-02-15	4750000	4780000	4570000	4700000	261
2002-02-18	4550000	4660000	4530000	4620000	110
2002-02-19	4650000	4690000	4540000	4640000	63
2002-02-20	4560000	4630000	4500000	4600000	59
2002-02-21	4600000	4870000	4600000	4860000	390
2002-02-22	4870000	5150000	4870000	5100000	982
2002-02-25	5180000	5350000	5070000	5300000	545
2002-02-26	5300000	5300000	4800000	5010000	309
2002-02-27	5040000	5430000	5040000	5260000	403
2002-02-28	5360000	5530000	5200000	5270000	374
2002-03-01	5440000	5520000	5350000	5490000	454
2002-03-04	5690000	5780000	5250000	5280000	503
2002-03-05	5380000	5410000	5260000	5380000	208
2002-03-06	5280000	5290000	5110000	5120000	250
2002-03-07	5190000	5350000	5120000	5350000	264
2002-03-08	5520000	5800000	5520000	5760000	580
2002-03-11	5910000	6030000	5650000	5950000	571
2002-03-12	5940000	5990000	5700000	5770000	156
2002-03-13	5570000	5700000	5420000	5500000	279
2002-03-14	5300000	5400000	5150000	5190000	481
2002-03-15	5320000	5370000	5210000	5270000	196
2002-03-18	5340000	5380000	5280000	5290000	136
2002-03-19	5310000	5480000	5310000	5410000	164
2002-03-20	5510000	5560000	5430000	5450000	236
2002-03-22	5410000	5410000	5180000	5190000	242
2002-03-25	5190000	5280000	5100000	5270000	182
2002-03-26	2830000	2830000	2700000	2820000	459
2002-03-27	2840000	2990000	2760000	2970000	650
2002-03-28	3010000	3170000	3010000	3170000	746
2002-03-29	3350000	3550000	3180000	3200000	1149
2002-04-01	3220000	3450000	3220000	3270000	391
2002-04-02	3350000	3370000	3280000	3290000	265
2002-04-03	3200000	3260000	3180000	3230000	459
2002-04-04	3210000	3330000	3210000	3290000	317
2002-04-05	3280000	3450000	3280000	3440000	610
2002-04-08	3440000	3470000	3310000	3320000	393
2002-04-09	3340000	3370000	3180000	3270000	354
2002-04-10	3180000	3270000	3170000	3240000	187
2002-04-11	3340000	3420000	3320000	3380000	386
2002-04-12	3230000	3290000	3220000	3260000	230
2002-04-15	3220000	3260000	3180000	3190000	175
2002-04-16	3190000	3200000	3110000	3170000	197
2002-04-17	3210000	3240000	3040000	3120000	386
2002-04-18	3070000	3120000	3060000	3120000	173
2002-04-19	3080000	3100000	3030000	3080000	187
2002-04-22	3010000	3030000	2920000	2970000	393
2002-04-23	2910000	2920000	2870000	2880000	347
2002-04-24	2850000	3080000	2760000	3080000	639
2002-04-25	3030000	3120000	2860000	2880000	549
2002-04-26	2860000	2980000	2820000	2930000	514
2002-04-30	2970000	3030000	2900000	2940000	493
2002-05-01	2970000	2970000	2870000	2870000	216
2002-05-02	2890000	2920000	2840000	2870000	240
2002-05-07	2800000	2930000	2790000	2850000	281
2002-05-08	2840000	2920000	2840000	2870000	95
2002-05-09	2970000	3020000	2900000	2920000	251
2002-05-10	2880000	2880000	2840000	2860000	132
2002-05-13	2750000	2780000	2710000	2730000	334
2002-05-14	2680000	2680000	2530000	2530000	657
2002-05-15	2510000	2550000	2360000	2500000	825
2002-05-16	2560000	2690000	2560000	2610000	556
2002-05-17	2670000	2740000	2670000	2670000	475
2002-05-20	2700000	2790000	2690000	2740000	518
2002-05-21	2740000	2740000	2640000	2650000	365
2002-05-22	2600000	2670000	2580000	2620000	242
2002-05-23	2660000	2670000	2610000	2630000	244
2002-05-24	2630000	2630000	2560000	2570000	321
2002-05-27	2580000	2710000	2560000	2680000	267
2002-05-28	2690000	2770000	2690000	2760000	398
2002-05-29	2750000	2960000	2730000	2910000	1275
2002-05-30	2890000	2910000	2810000	2860000	314
2002-05-31	2810000	2830000	2750000	2790000	300
2002-06-03	2760000	2850000	2760000	2850000	214
2002-06-04	2830000	2930000	2810000	2830000	300
2002-06-05	2830000	2870000	2800000	2830000	101
2002-06-06	2830000	2850000	2750000	2780000	126
2002-06-07	2660000	2690000	2600000	2620000	571
2002-06-10	2620000	2660000	2580000	2590000	315
2002-06-11	2580000	2590000	2520000	2540000	443
2002-06-12	2500000	2520000	2490000	2510000	446
2002-06-13	2500000	2600000	2490000	2600000	444
2002-06-14	2600000	2620000	2500000	2510000	569
2002-06-17	2490000	2490000	2400000	2420000	274
2002-06-18	2480000	2490000	2400000	2420000	319
2002-06-19	2410000	2430000	2220000	2280000	604
2002-06-20	2200000	2380000	2120000	2380000	662
2002-06-21	2340000	2370000	2260000	2330000	232
2002-06-24	2230000	2440000	2230000	2410000	293
2002-06-25	2430000	2480000	2390000	2410000	189
2002-06-26	2390000	2410000	2320000	2320000	162
2002-06-27	2380000	2460000	2350000	2400000	434
2002-06-28	2480000	2580000	2480000	2580000	548
2002-07-01	2600000	2650000	2560000	2630000	300
2002-07-02	2560000	2570000	2530000	2560000	190
2002-07-03	2540000	2650000	2530000	2630000	226
2002-07-04	2650000	2660000	2580000	2610000	182
2002-07-05	2620000	2630000	2570000	2590000	122
2002-07-08	2680000	2700000	2580000	2580000	317
2002-07-09	2590000	2650000	2580000	2640000	136
2002-07-10	2590000	2600000	2540000	2540000	192
2002-07-11	2550000	2550000	2500000	2520000	119
2002-07-12	2570000	2570000	2520000	2570000	152
2002-07-15	2580000	2580000	2530000	2540000	104
2002-07-16	2550000	2590000	2550000	2560000	165
2002-07-17	2580000	2580000	2410000	2450000	272
2002-07-18	2480000	2520000	2440000	2440000	109
2002-07-19	2420000	2450000	2390000	2410000	253
2002-07-22	2380000	2600000	2370000	2500000	450
2002-07-23	2540000	2660000	2530000	2650000	467
2002-07-24	2650000	2660000	2580000	2630000	246
2002-07-25	2660000	2670000	2620000	2640000	199
2002-07-26	2630000	2630000	2550000	2630000	214
2002-07-29	2630000	2650000	2610000	2610000	249
2002-07-30	2630000	2640000	2610000	2610000	88
2002-07-31	2610000	2680000	2590000	2680000	188
2002-08-01	2680000	2730000	2680000	2710000	386
2002-08-02	2700000	2700000	2660000	2700000	185
2002-08-05	2700000	2730000	2670000	2710000	123
2002-08-06	2700000	2700000	2560000	2640000	503
2002-08-07	2700000	2840000	2700000	2840000	896
2002-08-08	2860000	2890000	2750000	2760000	308
2002-08-09	2770000	2800000	2740000	2770000	229
2002-08-12	2790000	2800000	2750000	2780000	134
2002-08-13	2810000	2820000	2780000	2790000	119
2002-08-14	2790000	2790000	2700000	2780000	151
2002-08-15	2810000	2870000	2810000	2850000	256
2002-08-16	2890000	2940000	2870000	2890000	340
2002-08-19	2930000	2930000	2880000	2880000	154
2002-08-20	2920000	2920000	2870000	2870000	199
2002-08-21	2900000	2930000	2890000	2930000	208
2002-08-22	2960000	3100000	2950000	3030000	876
2002-08-23	3080000	3260000	3030000	3200000	1015
2002-08-26	3200000	3260000	3190000	3200000	541
2002-08-27	3200000	3200000	3100000	3120000	233
2002-08-28	3050000	3060000	2910000	2920000	499
2002-08-29	2900000	2930000	2860000	2880000	394
2002-08-30	2980000	3030000	2980000	3000000	282
2002-09-02	3000000	3000000	2910000	2930000	134
2002-09-03	2930000	2990000	2850000	2870000	273
2002-09-04	2800000	2800000	2730000	2750000	358
2002-09-05	2800000	2830000	2780000	2810000	203
2002-09-06	2760000	2840000	2730000	2840000	196
2002-09-09	2910000	2970000	2910000	2920000	143
2002-09-10	2960000	2970000	2900000	2900000	111
2002-09-11	3040000	3050000	2940000	3020000	485
2002-09-12	2990000	3000000	2950000	2960000	207
2002-09-13	2900000	2930000	2880000	2930000	107
2002-09-17	2920000	3010000	2920000	2970000	151
2002-09-18	2950000	2950000	2930000	2950000	153
2002-09-19	2930000	3000000	2850000	2850000	363
2002-09-20	2800000	2860000	2730000	2860000	325
2002-09-24	2840000	2890000	2810000	2850000	354
2002-09-25	1520000	1540000	1440000	1460000	564
2002-09-26	1500000	1520000	1470000	1490000	231
2002-09-27	1500000	1520000	1490000	1500000	271
2002-09-30	1500000	1510000	1470000	1510000	292
2002-10-01	1500000	1500000	1410000	1450000	450
2002-10-02	1480000	1480000	1420000	1430000	139
2002-10-03	1400000	1440000	1350000	1370000	344
2002-10-04	1330000	1340000	1320000	1330000	466
2002-10-07	1250000	1410000	1230000	1410000	551
2002-10-08	1310000	1360000	1300000	1330000	241
2002-10-09	1330000	1350000	1280000	1290000	230
2002-10-10	1300000	1350000	1270000	1350000	331
2002-10-11	1430000	1430000	1380000	1390000	460
2002-10-15	1450000	1480000	1420000	1470000	410
2002-10-16	1450000	1470000	1420000	1460000	342
2002-10-17	1450000	1460000	1430000	1450000	96
2002-10-18	1450000	1500000	1450000	1500000	604
2002-10-21	1510000	1520000	1410000	1420000	433
2002-10-22	1440000	1460000	1410000	1430000	133
2002-10-23	1430000	1500000	1420000	1500000	222
2002-10-24	1490000	1560000	1470000	1540000	1339
2002-10-25	1520000	1520000	1500000	1520000	332
2002-10-28	1520000	1670000	1500000	1630000	1154
2002-10-29	1630000	1630000	1550000	1590000	745
2002-10-30	1590000	1630000	1590000	1590000	757
2002-10-31	1590000	1590000	1550000	1580000	297
2002-11-01	1600000	1610000	1580000	1600000	377
2002-11-05	1610000	1620000	1550000	1580000	371
2002-11-06	1570000	1580000	1500000	1520000	881
2002-11-07	1520000	1530000	1460000	1460000	380
2002-11-08	1450000	1510000	1410000	1490000	280
2002-11-11	1460000	1470000	1390000	1390000	575
2002-11-12	1390000	1480000	1390000	1480000	436
2002-11-13	1480000	1490000	1420000	1420000	97
2002-11-14	1440000	1480000	1440000	1460000	210
2002-11-15	1490000	1490000	1430000	1450000	187
2002-11-18	1460000	1470000	1430000	1440000	167
2002-11-19	1430000	1500000	1400000	1430000	628
2002-11-20	1450000	1460000	1350000	1350000	851
2002-11-21	1390000	1410000	1360000	1360000	857
2002-11-22	1400000	1460000	1390000	1450000	1110
2002-11-25	1440000	1550000	1430000	1540000	1176
2002-11-26	1520000	1520000	1460000	1470000	737
2002-11-27	1470000	1500000	1450000	1490000	335
2002-11-28	1520000	1530000	1480000	1510000	440
2002-11-29	1510000	1510000	1480000	1500000	294
2002-12-02	1500000	1510000	1470000	1510000	441
2002-12-03	1530000	1580000	1530000	1550000	965
2002-12-04	1510000	1520000	1490000	1500000	421
2002-12-05	1500000	1510000	1480000	1480000	455
2002-12-06	1490000	1490000	1430000	1430000	541
2002-12-09	1450000	1510000	1450000	1510000	558
2002-12-10	1490000	1500000	1470000	1490000	307
2002-12-11	1500000	1510000	1480000	1490000	417
2002-12-12	1460000	1480000	1450000	1450000	240
2002-12-13	1460000	1460000	1410000	1430000	674
2002-12-16	1430000	1490000	1400000	1490000	883
2002-12-17	1480000	1490000	1460000	1480000	544
2002-12-18	1480000	1480000	1450000	1470000	317
2002-12-19	1460000	1480000	1420000	1480000	479
2002-12-20	1470000	1470000	1450000	1460000	305
2002-12-24	1460000	1460000	1430000	1440000	304
2002-12-25	1440000	1450000	1410000	1420000	943
2002-12-26	1410000	1480000	1410000	1480000	420
2002-12-27	1470000	1480000	1450000	1470000	178
2002-12-30	1460000	1480000	1460000	1480000	78
2003-01-06	1470000	1520000	1460000	1510000	319
2003-01-07	1500000	1510000	1460000	1470000	180
2003-01-08	1480000	1500000	1470000	1480000	226
2003-01-09	1470000	1490000	1470000	1480000	171
2003-01-10	1490000	1490000	1460000	1460000	153
2003-01-14	1480000	1510000	1480000	1510000	377
2003-01-15	1510000	1510000	1480000	1490000	168
2003-01-16	1500000	1540000	1490000	1530000	627
2003-01-17	1530000	1650000	1530000	1640000	1994
2003-01-20	1680000	1750000	1680000	1680000	2140
2003-01-21	1690000	1760000	1670000	1760000	1874
2003-01-22	1750000	1760000	1700000	1730000	1265
2003-01-23	1690000	1730000	1650000	1690000	1210
2003-01-24	1690000	1790000	1690000	1760000	1378
2003-01-27	1720000	1780000	1720000	1760000	447
2003-01-28	1730000	1740000	1700000	1720000	622
2003-01-29	1730000	1740000	1710000	1710000	431
2003-01-30	1690000	1690000	1660000	1690000	395
2003-01-31	1670000	1690000	1650000	1670000	552
2003-02-03	1700000	1830000	1680000	1800000	1253
2003-02-04	1860000	1940000	1860000	1900000	1794
2003-02-05	1880000	1930000	1860000	1900000	1204
2003-02-06	1920000	1980000	1920000	1970000	1472
2003-02-07	1960000	2050000	1960000	2000000	1976
2003-02-10	2010000	2010000	1900000	1910000	706
2003-02-12	1910000	1920000	1860000	1900000	924
2003-02-13	1900000	1960000	1890000	1930000	689
2003-02-14	1930000	1950000	1900000	1900000	683
2003-02-17	1940000	1990000	1940000	1970000	719
2003-02-18	1970000	1980000	1940000	1950000	314
2003-02-19	1990000	2040000	1980000	2030000	1106
2003-02-20	2070000	2130000	2060000	2080000	1350
2003-02-21	2100000	2110000	2070000	2080000	718
2003-02-24	2090000	2090000	2000000	2020000	455
2003-02-25	2020000	2100000	2010000	2080000	703
2003-02-26	2070000	2080000	2060000	2080000	293
2003-02-27	2070000	2070000	2020000	2030000	708
2003-02-28	2040000	2050000	2030000	2040000	435
2003-03-03	2050000	2060000	1840000	1840000	1881
2003-03-04	1760000	1920000	1730000	1890000	2095
2003-03-05	1850000	1910000	1840000	1860000	604
2003-03-06	1860000	1860000	1790000	1800000	602
2003-03-07	1770000	1770000	1620000	1630000	1285
2003-03-10	1600000	1600000	1460000	1470000	2439
2003-03-11	1510000	1640000	1450000	1540000	2487
2003-03-12	1740000	1740000	1740000	1740000	575
2003-03-13	1740000	1850000	1720000	1780000	2721
2003-03-14	1800000	1850000	1800000	1830000	1660
2003-03-17	1800000	1810000	1760000	1780000	878
2003-03-18	1820000	1830000	1790000	1810000	990
2003-03-19	1610000	1610000	1610000	1610000	4504
2003-03-20	1610000	1690000	1540000	1680000	6631
2003-03-24	1700000	1710000	1640000	1670000	4029
2003-03-25	1640000	1660000	1600000	1650000	3004
2003-03-26	925000	925000	900000	925000	4619
2003-03-27	940000	998000	930000	966000	6271
2003-03-28	964000	984000	960000	973000	3073
2003-03-31	970000	982000	958000	975000	2747
2003-04-01	970000	1060000	970000	1020000	5269
2003-04-02	1010000	1070000	1010000	1050000	3901
2003-04-03	1080000	1210000	1080000	1130000	6348
2003-04-04	1120000	1130000	1070000	1080000	1723
2003-04-07	1120000	1150000	1110000	1130000	2825
2003-04-08	1120000	1160000	1100000	1140000	2593
2003-04-09	1130000	1170000	1120000	1170000	1825
2003-04-10	1160000	1170000	1140000	1160000	1365
2003-04-11	1160000	1210000	1150000	1200000	3131
2003-04-14	1220000	1230000	1150000	1200000	2970
2003-04-15	1200000	1210000	1180000	1210000	1554
2003-04-16	1170000	1180000	1150000	1160000	1929
2003-04-17	1120000	1160000	1100000	1140000	1826
2003-04-18	1160000	1180000	1140000	1180000	1200
2003-04-21	1180000	1180000	1160000	1170000	514
2003-04-22	1160000	1170000	1150000	1160000	940
2003-04-23	1170000	1220000	1140000	1220000	3524
2003-04-24	1220000	1360000	1210000	1340000	4998
2003-04-25	1340000	1540000	1310000	1410000	6496
2003-04-28	1370000	1370000	1250000	1270000	2348
2003-04-30	1270000	1360000	1230000	1340000	3289
2003-05-01	1350000	1420000	1330000	1360000	2195
2003-05-02	1380000	1400000	1360000	1380000	1514
2003-05-06	1390000	1400000	1350000	1360000	2319
2003-05-07	1340000	1440000	1320000	1420000	3876
2003-05-08	1420000	1430000	1380000	1400000	1616
2003-05-09	1360000	1380000	1340000	1360000	1706
2003-05-12	1320000	1330000	1250000	1270000	1995
2003-05-13	1260000	1330000	1250000	1300000	1740
2003-05-14	1320000	1330000	1290000	1320000	1990
2003-05-15	1310000	1310000	1270000	1280000	1153
2003-05-16	1290000	1300000	1270000	1290000	1287
2003-05-19	1280000	1290000	1250000	1260000	877
2003-05-20	1240000	1320000	1160000	1300000	7675
2003-05-21	1350000	1480000	1350000	1450000	11120
2003-05-22	1520000	1560000	1410000	1430000	7011
2003-05-23	1450000	1490000	1370000	1370000	2652
2003-05-26	1410000	1440000	1380000	1430000	3102
2003-05-27	1440000	1440000	1380000	1400000	1538
2003-05-28	1460000	1490000	1430000	1480000	3048
2003-05-29	1490000	1490000	1440000	1470000	1992
2003-05-30	1450000	1470000	1440000	1460000	1363
2003-06-02	1480000	1480000	1440000	1450000	1525
2003-06-03	1450000	1450000	1400000	1420000	1925
2003-06-04	1410000	1460000	1390000	1420000	2272
2003-06-05	1440000	1450000	1430000	1450000	1684
2003-06-06	1460000	1550000	1440000	1550000	4090
2003-06-09	1530000	1550000	1510000	1540000	2686
2003-06-10	1500000	1530000	1500000	1520000	1209
2003-06-11	1500000	1520000	1480000	1500000	1511
2003-06-12	1490000	1500000	1450000	1470000	1644
2003-06-13	1450000	1510000	1450000	1500000	2382
2003-06-16	1490000	1500000	1470000	1470000	742
2003-06-17	1550000	1570000	1520000	1570000	3669
2003-06-18	1580000	1620000	1580000	1620000	3650
2003-06-19	1660000	1770000	1660000	1770000	5066
2003-06-20	1710000	1740000	1700000	1730000	2702
2003-06-23	1750000	1770000	1710000	1740000	2776
2003-06-24	1690000	1720000	1680000	1700000	2040
2003-06-25	1700000	1760000	1700000	1760000	2271
2003-06-26	1760000	1790000	1740000	1760000	2363
2003-06-27	1820000	1850000	1800000	1830000	4294
2003-06-30	1840000	2000000	1840000	1950000	3891
2003-07-01	1960000	1970000	1880000	1900000	4588
2003-07-02	1940000	1970000	1860000	1870000	3250
2003-07-03	1900000	1900000	1750000	1820000	5673
2003-07-04	1810000	1870000	1780000	1840000	2239
2003-07-07	1870000	1950000	1870000	1950000	3527
2003-07-08	2070000	2150000	2020000	2150000	6636
2003-07-09	2150000	2350000	2100000	2230000	7483
2003-07-10	2090000	2140000	2040000	2050000	6235
2003-07-11	1970000	1980000	1850000	1870000	6315
2003-07-14	1950000	2030000	1940000	2030000	3087
2003-07-15	2080000	2120000	1970000	1990000	3488
2003-07-16	2010000	2030000	1930000	1970000	2066
2003-07-17	1890000	1940000	1850000	1910000	2645
2003-07-18	2000000	2100000	1990000	2050000	7871
2003-07-22	2090000	2250000	2080000	2240000	10487
2003-07-23	2200000	2210000	2130000	2160000	6228
2003-07-24	2160000	2160000	2080000	2110000	3589
2003-07-25	2110000	2180000	2080000	2140000	2885
2003-07-28	2180000	2210000	2160000	2180000	2758
2003-07-29	2210000	2230000	2170000	2190000	1897
2003-07-30	2190000	2190000	2120000	2140000	1571
2003-07-31	2100000	2120000	2030000	2040000	4271
2003-08-01	2100000	2130000	2040000	2060000	2996
2003-08-04	2040000	2040000	2000000	2020000	1406
2003-08-05	2000000	2010000	1930000	1970000	2403
2003-08-06	1900000	1950000	1890000	1930000	3002
2003-08-07	1920000	1960000	1890000	1890000	1975
2003-08-08	1910000	1930000	1880000	1900000	1328
2003-08-11	1900000	1910000	1800000	1820000	3337
2003-08-12	1840000	1900000	1800000	1890000	3627
2003-08-13	1950000	2060000	1940000	2030000	4674
2003-08-14	2000000	2030000	1970000	2020000	1480
2003-08-15	2040000	2050000	2000000	2010000	1410
2003-08-18	2030000	2060000	2010000	2040000	1486
2003-08-19	2070000	2100000	2060000	2080000	2000
2003-08-20	2120000	2140000	2100000	2140000	2672
2003-08-21	2080000	2100000	2070000	2090000	1370
2003-08-22	2080000	2080000	2010000	2010000	1965
2003-08-25	1970000	2000000	1950000	1960000	2026
2003-08-26	1940000	1950000	1900000	1930000	2564
2003-08-27	2110000	2120000	2010000	2030000	5201
2003-08-28	2070000	2070000	2000000	2010000	1397
2003-08-29	2030000	2120000	2020000	2110000	2691
2003-09-01	2110000	2120000	2070000	2090000	1861
2003-09-02	2110000	2200000	2100000	2200000	4541
2003-09-03	2250000	2270000	2150000	2160000	5108
2003-09-04	2180000	2190000	2160000	2170000	2175
2003-09-05	2190000	2200000	2100000	2160000	2456
2003-09-08	2120000	2190000	2120000	2180000	1021
2003-09-09	2200000	2210000	2160000	2170000	2123
2003-09-10	2180000	2190000	2170000	2180000	1758
2003-09-11	2170000	2180000	2150000	2170000	1469
2003-09-12	2170000	2260000	2170000	2260000	4229
2003-09-16	2280000	2450000	2280000	2430000	6698
2003-09-17	2480000	2630000	2480000	2540000	6626
2003-09-18	2620000	2740000	2620000	2740000	5055
2003-09-19	2850000	2900000	2590000	2680000	7552
2003-09-22	2600000	2670000	2560000	2640000	5077
2003-09-24	2720000	2820000	2660000	2800000	6885
2003-09-25	1560000	1600000	1480000	1480000	15754
2003-09-26	1420000	1460000	1340000	1390000	11010
2003-09-29	1410000	1470000	1380000	1470000	7148
2003-09-30	1520000	1630000	1490000	1580000	10184
2003-10-01	1600000	1600000	1550000	1590000	7053
2003-10-02	1630000	1640000	1540000	1550000	6844
2003-10-03	1560000	1560000	1490000	1510000	6040
2003-10-06	1530000	1570000	1520000	1550000	4482
2003-10-07	1550000	1620000	1520000	1620000	8176
2003-10-08	1610000	1610000	1560000	1580000	5753
2003-10-09	1600000	1610000	1550000	1570000	4317
2003-10-10	1610000	1670000	1590000	1640000	10773
2003-10-14	1660000	1680000	1620000	1640000	6281
2003-10-15	1640000	1680000	1600000	1680000	6861
2003-10-16	1670000	1880000	1670000	1880000	16461
2003-10-17	1880000	1920000	1780000	1820000	12589
2003-10-20	1800000	1900000	1780000	1880000	7878
2003-10-21	1920000	1930000	1680000	1740000	13882
2003-10-22	1900000	1910000	1700000	1720000	18156
2003-10-23	1640000	1660000	1520000	1540000	14142
2003-10-24	1560000	1650000	1520000	1600000	11630
2003-10-27	1620000	1750000	1610000	1680000	13961
2003-10-28	1650000	1700000	1560000	1570000	12857
2003-10-29	1650000	1650000	1590000	1620000	8088
2003-10-30	1610000	1630000	1580000	1630000	4872
2003-10-31	1650000	1690000	1630000	1630000	15680
2003-11-04	1670000	1680000	1600000	1640000	5481
2003-11-05	1610000	1620000	1560000	1590000	4479
2003-11-06	1560000	1560000	1500000	1500000	5366
2003-11-07	1470000	1500000	1380000	1490000	7268
2003-11-10	1470000	1470000	1420000	1440000	3878
2003-11-11	1320000	1330000	1240000	1310000	11496
2003-11-12	1350000	1380000	1290000	1330000	8156
2003-11-13	1410000	1520000	1390000	1480000	18112
2003-11-14	1440000	1460000	1370000	1410000	6511
2003-11-17	1330000	1360000	1310000	1330000	6520
2003-11-18	1350000	1370000	1280000	1350000	10055
2003-11-19	1270000	1320000	1260000	1290000	5327
2003-11-20	1330000	1360000	1270000	1310000	6837
2003-11-21	1290000	1360000	1280000	1310000	10184
2003-11-25	1350000	1380000	1320000	1360000	15851
2003-11-26	1400000	1550000	1400000	1500000	34403
2003-11-27	1560000	1580000	1490000	1540000	32867
2003-11-28	1490000	1530000	1440000	1440000	43582
2003-12-01	1400000	1490000	1360000	1450000	20961
2003-12-02	1500000	1520000	1470000	1480000	10220
2003-12-03	1480000	1520000	1470000	1520000	6667
2003-12-04	1490000	1510000	1460000	1460000	6157
2003-12-05	1470000	1490000	1460000	1480000	2713
2003-12-08	1450000	1460000	1380000	1390000	6163
2003-12-09	1410000	1440000	1360000	1370000	5574
2003-12-10	1350000	1360000	1310000	1330000	4511
2003-12-11	1310000	1350000	1310000	1340000	4275
2003-12-12	1320000	1320000	1250000	1270000	12823
2003-12-15	1330000	1380000	1290000	1380000	6257
2003-12-16	1310000	1360000	1300000	1320000	5306
2003-12-17	1350000	1360000	1320000	1330000	3825
2003-12-18	1350000	1360000	1310000	1320000	2354
2003-12-19	1330000	1340000	1310000	1310000	2843
2003-12-22	1300000	1330000	1290000	1320000	3021
2003-12-24	1340000	1360000	1330000	1350000	4517
2003-12-25	1360000	1410000	1360000	1380000	5605
2003-12-26	1400000	1420000	1370000	1410000	4702
2003-12-29	1420000	1440000	1410000	1420000	2680
2003-12-30	1430000	1450000	1430000	1440000	2438
2004-01-05	1470000	1480000	1450000	1450000	1902
2004-01-06	1470000	1490000	1440000	1440000	2582
2004-01-07	1440000	1470000	1420000	1460000	2674
2004-01-08	1480000	1530000	1460000	1470000	7722
2004-01-09	1490000	1500000	1450000	1480000	4771
2004-01-13	1500000	1520000	1460000	1460000	3807
2004-01-14	1470000	1480000	1460000	1470000	2517
2004-01-15	1460000	1480000	1430000	1430000	3156
2004-01-16	1450000	1470000	1430000	1470000	3069
2004-01-19	1490000	1520000	1480000	1520000	6014
2004-01-20	1530000	1560000	1510000	1550000	8205
2004-01-21	1550000	1560000	1520000	1520000	4723
2004-01-22	1550000	1550000	1480000	1480000	4531
2004-01-23	1480000	1510000	1470000	1500000	2891
2004-01-26	1500000	1530000	1500000	1530000	3075
2004-01-27	1530000	1540000	1500000	1500000	4023
2004-01-28	1500000	1530000	1490000	1530000	4026
2004-01-29	1540000	1550000	1520000	1530000	11374
2004-01-30	1540000	1580000	1520000	1560000	28359
2004-02-02	1560000	1610000	1560000	1590000	15228
2004-02-03	1600000	1670000	1590000	1650000	12472
2004-02-04	1670000	1700000	1600000	1600000	11057
2004-02-05	1570000	1620000	1570000	1600000	5755
2004-02-06	1630000	1660000	1620000	1640000	6853
2004-02-09	1690000	1750000	1670000	1750000	12078
2004-02-10	1770000	1780000	1710000	1730000	7296
2004-02-12	1750000	1780000	1740000	1750000	4213
2004-02-13	1760000	1760000	1730000	1750000	3056
2004-02-16	1760000	1820000	1750000	1810000	7195
2004-02-17	1810000	1830000	1780000	1820000	4305
2004-02-18	1910000	1970000	1860000	1860000	21525
2004-02-19	1890000	1950000	1880000	1920000	7986
2004-02-20	1920000	1920000	1880000	1920000	5261
2004-02-23	1920000	2040000	1910000	2040000	15555
2004-02-24	2070000	2160000	2010000	2010000	17099
2004-02-25	1910000	2040000	1870000	1900000	26947
2004-02-26	1890000	1940000	1870000	1920000	14030
2004-02-27	1960000	2020000	1960000	2010000	9336
2004-03-01	2090000	2170000	2070000	2150000	11602
2004-03-02	2210000	2240000	2130000	2180000	10783
2004-03-03	2170000	2200000	2160000	2180000	5643
2004-03-04	2180000	2190000	2150000	2150000	4837
2004-03-05	2130000	2140000	2080000	2110000	6542
2004-03-08	2090000	2160000	2070000	2140000	6084
2004-03-09	2130000	2190000	2120000	2160000	7014
2004-03-10	2170000	2180000	2120000	2120000	5207
2004-03-11	2080000	2100000	2040000	2060000	8903
2004-03-12	1980000	2020000	1980000	2000000	13210
2004-03-15	2040000	2070000	2030000	2040000	6260
2004-03-16	2030000	2060000	2010000	2020000	6926
2004-03-17	2050000	2120000	2050000	2120000	6031
2004-03-18	2190000	2190000	2100000	2120000	8320
2004-03-19	2100000	2150000	2100000	2150000	5098
2004-03-22	2140000	2190000	2120000	2140000	7203
2004-03-23	2110000	2150000	2100000	2150000	4942
2004-03-24	2160000	2160000	2060000	2090000	7491
2004-03-25	2100000	2120000	2040000	2100000	10817
2004-03-26	1080000	1210000	1070000	1170000	50906
2004-03-29	1210000	1310000	1210000	1280000	45659
2004-03-30	1340000	1350000	1270000	1280000	26396
2004-03-31	1270000	1280000	1230000	1270000	18017
2004-04-01	1310000	1320000	1260000	1300000	22030
2004-04-02	1310000	1380000	1300000	1350000	27649
2004-04-05	1390000	1430000	1370000	1390000	24669
2004-04-06	1400000	1410000	1320000	1330000	23664
2004-04-07	1280000	1310000	1260000	1270000	23305
2004-04-08	1330000	1340000	1290000	1310000	12269
2004-04-09	1280000	1290000	1230000	1250000	12530
2004-04-12	1270000	1310000	1260000	1290000	9376
2004-04-13	1310000	1320000	1270000	1280000	7697
2004-04-14	1260000	1270000	1250000	1250000	6348
2004-04-15	1260000	1270000	1200000	1230000	11635
2004-04-16	1230000	1240000	1190000	1210000	8917
2004-04-19	1220000	1240000	1160000	1200000	9298
2004-04-20	1210000	1220000	1170000	1210000	13143
2004-04-21	1200000	1240000	1190000	1220000	15729
2004-04-22	1260000	1280000	1200000	1200000	13885
2004-04-23	1210000	1220000	1170000	1190000	8879
2004-04-26	1170000	1180000	1150000	1160000	7021
2004-04-27	1150000	1200000	1140000	1190000	13998
2004-04-28	1190000	1300000	1190000	1300000	24856
2004-04-30	1260000	1290000	1240000	1250000	15927
2004-05-06	1280000	1280000	1250000	1250000	5853
2004-05-07	1230000	1240000	1210000	1220000	6122
2004-05-10	1170000	1190000	1080000	1100000	10664
2004-05-11	1050000	1130000	1040000	1080000	14106
2004-05-12	1150000	1160000	1110000	1130000	13095
2004-05-13	1140000	1140000	1100000	1110000	6979
2004-05-14	1090000	1110000	1050000	1050000	8833
2004-05-17	1050000	1050000	923000	978000	11296
2004-05-18	969000	1020000	950000	1020000	9797
2004-05-19	1060000	1070000	1040000	1060000	8543
2004-05-20	1050000	1080000	1020000	1020000	8584
2004-05-21	1040000	1070000	1030000	1060000	6538
2004-05-24	1060000	1090000	1050000	1060000	10693
2004-05-25	1070000	1100000	1050000	1070000	14609
2004-05-26	1120000	1130000	1070000	1080000	13896
2004-05-27	1080000	1100000	1070000	1090000	7917
2004-05-28	1100000	1160000	1080000	1160000	24649
2004-05-31	1140000	1160000	1120000	1150000	20464
2004-06-01	1130000	1160000	1130000	1150000	9143
2004-06-02	1150000	1150000	1130000	1140000	6541
2004-06-03	1150000	1150000	1100000	1110000	7482
2004-06-04	1090000	1100000	1050000	1090000	11183
2004-06-07	1090000	1110000	1070000	1100000	12407
2004-06-08	1110000	1120000	1070000	1090000	6134
2004-06-09	1090000	1090000	1050000	1060000	7232
2004-06-10	1040000	1050000	1020000	1030000	11603
2004-06-11	1020000	1060000	1020000	1040000	14199
2004-06-14	1050000	1060000	1030000	1030000	6721
2004-06-15	1030000	1040000	1020000	1030000	5051
2004-06-16	1040000	1040000	1020000	1020000	5186
2004-06-17	1030000	1040000	1020000	1040000	10387
2004-06-18	1030000	1040000	1000000	1010000	9330
2004-06-21	1020000	1030000	1000000	1000000	5519
2004-06-22	1000000	1010000	958000	976000	13543
2004-06-23	960000	964000	928000	930000	17809
2004-06-24	950000	1010000	933000	994000	22056
2004-06-25	984000	1000000	970000	992000	7427
2004-06-28	1010000	1040000	1000000	1040000	8891
2004-06-29	1050000	1060000	1030000	1050000	6216
2004-06-30	1050000	1070000	1040000	1060000	6514
2004-07-01	1070000	1080000	1060000	1060000	8156
2004-07-02	1020000	1040000	1010000	1010000	7020
2004-07-05	1000000	1010000	994000	994000	5140
2004-07-06	980000	997000	955000	960000	8116
2004-07-07	930000	980000	926000	980000	11700
2004-07-08	960000	961000	940000	943000	10114
2004-07-09	943000	961000	935000	957000	12518
2004-07-12	967000	967000	953000	958000	4542
2004-07-13	950000	954000	931000	937000	7625
2004-07-14	945000	948000	925000	932000	7120
2004-07-15	930000	930000	860000	890000	20855
2004-07-16	880000	918000	851000	908000	12534
2004-07-20	880000	886000	863000	868000	9449
2004-07-21	882000	926000	881000	920000	13731
2004-07-22	907000	971000	890000	953000	33544
2004-07-23	950000	952000	925000	930000	8519
2004-07-26	910000	922000	900000	900000	8984
2004-07-27	883000	905000	862000	865000	10144
2004-07-28	900000	905000	881000	899000	7745
2004-07-29	896000	896000	861000	866000	8248
2004-07-30	872000	884000	870000	884000	4800
2004-08-02	870000	874000	855000	860000	6990
2004-08-03	867000	878000	851000	860000	6952
2004-08-04	831000	839000	804000	833000	14903
2004-08-05	836000	871000	814000	850000	10287
2004-08-06	821000	847000	821000	844000	6510
2004-08-09	834000	868000	822000	863000	6490
2004-08-10	889000	951000	888000	948000	23052
2004-08-11	971000	990000	952000	959000	16431
2004-08-12	956000	972000	943000	960000	6227
2004-08-13	950000	975000	942000	956000	9611
2004-08-16	955000	964000	938000	958000	5627
2004-08-17	969000	982000	952000	952000	8822
2004-08-18	952000	967000	946000	960000	4059
2004-08-19	977000	1020000	970000	1010000	30041
2004-08-20	1020000	1060000	1010000	1050000	19357
2004-08-23	1080000	1090000	1050000	1070000	13830
2004-08-24	1080000	1100000	1070000	1100000	17055
2004-08-25	1110000	1120000	1060000	1070000	19999
2004-08-26	1090000	1100000	1030000	1040000	10405
2004-08-27	1030000	1060000	1020000	1050000	7621
2004-08-30	1070000	1080000	1050000	1080000	7136
2004-08-31	1070000	1080000	1050000	1050000	4434
2004-09-01	1050000	1080000	1050000	1060000	6023
2004-09-02	1080000	1080000	1060000	1080000	4132
2004-09-03	1080000	1090000	1060000	1060000	4795
2004-09-06	1070000	1080000	1060000	1070000	3841
2004-09-07	1080000	1100000	1070000	1100000	8527
2004-09-08	1090000	1100000	1070000	1080000	6836
2004-09-09	1070000	1080000	1060000	1060000	4775
2004-09-10	1060000	1070000	1050000	1070000	8157
2004-09-13	1070000	1090000	1070000	1080000	4375
2004-09-14	1090000	1090000	1070000	1070000	4415
2004-09-15	1070000	1080000	1050000	1050000	5219
2004-09-16	1040000	1050000	1020000	1040000	5765
2004-09-17	1040000	1040000	1020000	1040000	4521
2004-09-21	1030000	1070000	1030000	1060000	7607
2004-09-22	1060000	1070000	1030000	1030000	6923
2004-09-24	1000000	1010000	992000	1000000	16012
2004-09-27	523000	525000	481000	499000	35946
2004-09-28	489000	490000	473000	480000	17859
2004-09-29	485000	486000	462000	464000	14133
2004-09-30	474000	490000	468000	490000	16995
2004-10-01	506000	510000	497000	503000	14148
2004-10-04	521000	523000	509000	512000	12066
2004-10-05	510000	510000	503000	505000	6433
2004-10-06	500000	517000	500000	517000	11020
2004-10-07	522000	529000	515000	518000	14362
2004-10-08	514000	520000	512000	517000	4685
2004-10-12	517000	520000	507000	508000	6650
2004-10-13	511000	516000	508000	509000	7124
2004-10-14	510000	515000	503000	509000	9309
2004-10-15	504000	509000	501000	508000	7570
2004-10-18	510000	511000	507000	508000	6488
2004-10-19	513000	521000	507000	518000	10996
2004-10-20	513000	514000	504000	504000	9815
2004-10-21	499000	503000	480000	480000	17697
2004-10-22	475000	481000	463000	481000	17040
2004-10-25	466000	476000	465000	472000	8107
2004-10-26	473000	481000	472000	480000	7708
2004-10-27	488000	490000	482000	485000	7711
2004-10-28	494000	494000	485000	485000	7869
2004-10-29	490000	490000	474000	479000	7631
2004-11-01	470000	477000	470000	475000	4399
2004-11-02	473000	482000	473000	478000	5376
2004-11-04	483000	486000	466000	469000	9010
2004-11-05	473000	476000	471000	474000	7319
2004-11-08	472000	473000	463000	465000	8073
2004-11-09	463000	466000	454000	456000	11653
2004-11-10	451000	459000	450000	459000	7753
2004-11-11	467000	475000	464000	464000	9286
2004-11-12	467000	483000	467000	482000	14120
2004-11-15	500000	514000	497000	512000	26672
2004-11-16	510000	512000	495000	500000	11418
2004-11-17	490000	499000	489000	496000	7208
2004-11-18	499000	507000	498000	499000	9253
2004-11-19	505000	506000	496000	497000	6358
2004-11-22	502000	502000	488000	497000	6885
2004-11-24	495000	497000	489000	491000	6896
2004-11-25	486000	490000	481000	483000	7264
2004-11-26	483000	484000	478000	480000	7143
2004-11-29	478000	489000	476000	485000	7545
2004-11-30	478000	482000	474000	474000	9193
2004-12-01	469000	471000	460000	460000	14202
2004-12-02	470000	474000	468000	472000	8927
2004-12-03	476000	483000	476000	480000	9936
2004-12-06	476000	479000	472000	478000	5773
2004-12-07	474000	476000	468000	469000	4962
2004-12-08	462000	469000	461000	467000	6915
2004-12-09	470000	470000	461000	465000	5837
2004-12-10	466000	468000	462000	462000	11975
2004-12-13	461000	464000	458000	459000	4760
2004-12-14	463000	464000	457000	464000	7035
2004-12-15	460000	466000	460000	462000	6265
2004-12-16	461000	464000	457000	460000	7652
2004-12-17	460000	471000	460000	469000	9916
2004-12-20	465000	470000	464000	465000	5213
2004-12-21	466000	471000	465000	468000	7425
2004-12-22	470000	472000	463000	466000	7121
2004-12-24	467000	471000	466000	467000	7586
2004-12-27	470000	470000	465000	465000	5269
2004-12-28	466000	482000	465000	482000	13347
2004-12-29	487000	501000	485000	490000	30385
2004-12-30	496000	497000	490000	492000	6569
2005-01-04	484000	490000	482000	490000	6113
2005-01-05	483000	487000	482000	482000	7004
2005-01-06	477000	482000	476000	477000	8327
2005-01-07	478000	480000	475000	476000	6979
2005-01-11	479000	486000	478000	483000	9038
2005-01-12	482000	483000	478000	480000	5644
2005-01-13	478000	480000	472000	475000	6081
2005-01-14	478000	484000	475000	483000	12168
2005-01-17	490000	505000	489000	503000	34388
2005-01-18	502000	503000	495000	500000	16466
2005-01-19	504000	511000	496000	498000	31215
2005-01-20	496000	506000	488000	503000	19481
2005-01-21	505000	509000	502000	508000	25423
2005-01-24	510000	530000	509000	529000	52447
2005-01-25	527000	530000	523000	528000	22312
2005-01-26	530000	532000	523000	523000	13584
2005-01-27	528000	528000	523000	527000	6396
2005-01-28	524000	526000	516000	523000	8511
2005-01-31	522000	527000	514000	525000	10386
2005-02-01	528000	531000	525000	527000	11777
2005-02-02	530000	548000	530000	541000	41923
2005-02-03	545000	547000	538000	545000	17388
2005-02-04	547000	552000	541000	549000	23791
2005-02-07	554000	567000	551000	563000	34963
2005-02-08	561000	564000	556000	561000	13438
2005-02-09	561000	561000	548000	555000	11287
2005-02-10	555000	559000	550000	555000	10200
2005-02-14	557000	558000	547000	547000	12409
2005-02-15	546000	547000	531000	532000	17204
2005-02-16	531000	534000	522000	531000	18412
2005-02-17	539000	546000	528000	530000	30078
2005-02-18	524000	525000	517000	519000	14293
2005-02-21	519000	530000	509000	526000	12362
2005-02-22	527000	532000	524000	525000	9680
2005-02-23	523000	523000	514000	515000	9626
2005-02-24	516000	520000	515000	518000	6654
2005-02-25	520000	521000	515000	520000	10879
2005-02-28	521000	527000	520000	525000	7921
2005-03-01	525000	530000	523000	523000	8799
2005-03-02	523000	525000	521000	522000	7315
2005-03-03	521000	524000	519000	524000	6069
2005-03-04	527000	534000	525000	527000	14491
2005-03-07	532000	533000	527000	532000	9396
2005-03-08	530000	531000	525000	527000	6864
2005-03-09	526000	526000	521000	524000	7202
2005-03-10	521000	522000	517000	518000	13594
2005-03-11	516000	520000	514000	519000	19274
2005-03-14	519000	520000	506000	508000	13757
2005-03-15	508000	509000	496000	500000	17961
2005-03-16	508000	515000	500000	503000	28503
2005-03-17	503000	506000	500000	503000	13676
2005-03-18	502000	514000	502000	510000	13766
2005-03-22	513000	520000	512000	513000	15825
2005-03-23	514000	518000	511000	515000	16775
2005-03-24	516000	517000	512000	514000	16681
2005-03-25	522000	527000	518000	525000	41686
2005-03-28	265000	266000	252000	258000	41960
2005-03-29	257000	258000	249000	252000	30834
2005-03-30	251000	255000	247000	250000	22700
2005-03-31	252000	254000	249000	251000	19959
2005-04-01	247000	249000	246000	248000	13924
2005-04-04	249000	249000	241000	244000	18321
2005-04-05	244000	246000	241000	243000	24653
2005-04-06	244000	250000	243000	248000	32012
2005-04-07	245000	248000	245000	248000	16126
2005-04-08	248000	254000	247000	253000	40219
2005-04-11	250000	253000	249000	250000	16923
2005-04-12	249000	250000	243000	244000	20054
2005-04-13	244000	245000	242000	245000	11948
2005-04-14	242000	244000	240000	243000	11460
2005-04-15	240000	241000	238000	240000	14574
2005-04-18	235000	236000	230000	231000	24115
2005-04-19	235000	237000	231000	233000	13547
2005-04-20	240000	241000	235000	236000	21725
2005-04-21	235000	241000	231000	237000	31770
2005-04-22	243000	245000	240000	244000	38123
2005-04-25	240000	244000	240000	243000	13796
2005-04-26	243000	243000	240000	241000	10302
2005-04-27	238000	239000	237000	239000	14576
2005-04-28	235000	237000	232000	235000	22914
2005-05-02	231000	233000	230000	232000	20175
2005-05-06	231000	232000	229000	231000	21833
2005-05-09	230000	230000	228000	229000	12491
2005-05-10	228000	228000	226000	228000	14040
2005-05-11	226000	227000	222000	223000	21703
2005-05-12	223000	224000	220000	220000	16993
2005-05-13	219000	225000	219000	221000	29025
2005-05-16	222000	223000	216000	217000	12136
2005-05-17	219000	220000	209000	211000	21051
2005-05-18	212000	213000	208000	211000	16988
2005-05-19	218000	223000	214000	221000	29364
2005-05-20	223000	229000	220000	221000	31114
2005-05-23	222000	223000	219000	221000	12309
2005-05-24	222000	222000	216000	218000	20071
2005-05-25	216000	217000	212000	214000	17905
2005-05-26	214000	214000	210000	212000	16860
2005-05-27	213000	218000	211000	217000	33629
2005-05-30	217000	224000	216000	222000	37482
2005-05-31	222000	222000	217000	219000	33080
2005-06-01	222000	226000	220000	221000	45112
2005-06-02	223000	228000	222000	226000	53536
2005-06-03	227000	230000	226000	227000	32140
2005-06-06	225000	229000	224000	229000	24209
2005-06-07	228000	231000	225000	228000	28671
2005-06-08	229000	229000	223000	224000	16386
2005-06-09	222000	224000	215000	217000	22947
2005-06-10	216000	221000	216000	217000	32041
2005-06-13	216000	219000	215000	216000	11341
2005-06-14	216000	217000	211000	213000	22673
2005-06-15	212000	214000	211000	212000	14414
2005-06-16	213000	215000	212000	212000	13329
2005-06-17	214000	217000	213000	215000	14329
2005-06-20	217000	223000	216000	222000	33355
2005-06-21	222000	227000	220000	225000	39624
2005-06-22	226000	229000	223000	226000	34656
2005-06-23	225000	226000	222000	223000	9311
2005-06-24	220000	223000	220000	223000	12273
2005-06-27	220000	222000	219000	222000	12676
2005-06-28	220000	228000	219000	227000	27869
2005-06-29	230000	235000	229000	234000	59501
2005-06-30	237000	238000	232000	233000	22310
2005-07-01	229000	236000	229000	234000	23561
2005-07-04	237000	239000	233000	234000	29777
2005-07-05	233000	234000	229000	231000	14087
2005-07-06	233000	235000	231000	233000	17876
2005-07-07	232000	233000	230000	232000	11228
2005-07-08	232000	235000	231000	231000	19016
2005-07-11	233000	235000	232000	233000	13158
2005-07-12	233000	234000	232000	232000	12748
2005-07-13	233000	234000	232000	234000	14073
2005-07-14	233000	238000	233000	235000	17703
2005-07-15	237000	237000	232000	235000	16475
2005-07-19	235000	235000	231000	233000	12976
2005-07-20	233000	233000	229000	230000	23133
2005-07-21	235000	239000	234000	236000	31626
2005-07-22	233000	234000	232000	232000	11073
2005-07-25	233000	233000	231000	232000	12591
2005-07-26	232000	234000	231000	231000	10578
2005-07-27	230000	235000	230000	233000	16942
2005-07-28	236000	238000	234000	234000	15912
2005-07-29	237000	244000	236000	242000	41586
2005-08-01	241000	249000	241000	249000	35710
2005-08-02	248000	248000	245000	246000	17492
2005-08-03	250000	250000	245000	246000	24674
2005-08-04	244000	246000	242000	244000	24296
2005-08-05	242000	243000	239000	242000	16112
2005-08-08	237000	241000	236000	241000	26159
2005-08-09	240000	244000	240000	242000	12314
2005-08-10	245000	248000	244000	244000	22501
2005-08-11	248000	251000	246000	246000	23806
2005-08-12	250000	257000	248000	256000	53931
2005-08-15	256000	258000	253000	254000	33342
2005-08-16	255000	256000	250000	253000	13827
2005-08-17	253000	267000	252000	261000	61842
2005-08-18	265000	269000	262000	263000	63304
2005-08-19	264000	264000	260000	261000	21867
2005-08-22	264000	267000	264000	265000	18584
2005-08-23	268000	270000	264000	264000	32401
2005-08-24	265000	267000	264000	265000	13189
2005-08-25	265000	265000	260000	261000	17769
2005-08-26	262000	264000	261000	264000	14945
2005-08-29	261000	262000	260000	260000	9653
2005-08-30	259000	263000	257000	263000	17052
2005-08-31	259000	262000	259000	260000	11416
2005-09-01	261000	263000	258000	258000	9355
2005-09-02	260000	260000	256000	258000	9444
2005-09-05	260000	262000	259000	262000	16207
2005-09-06	263000	266000	262000	262000	24734
2005-09-07	264000	265000	258000	260000	15975
2005-09-08	260000	262000	259000	261000	12222
2005-09-09	259000	261000	258000	261000	41666
2005-09-12	263000	264000	261000	261000	11844
2005-09-13	262000	262000	257000	258000	19205
2005-09-14	256000	258000	255000	256000	16949
2005-09-15	255000	256000	254000	256000	13994
2005-09-16	254000	255000	250000	254000	30459
2005-09-20	251000	252000	250000	251000	30063
2005-09-21	250000	251000	247000	247000	29402
2005-09-22	245000	246000	241000	244000	32277
2005-09-26	245000	250000	245000	247000	34413
2005-09-27	124000	125000	120000	121000	36485
2005-09-28	123000	124000	121000	123000	32518
2005-09-29	125000	132000	123000	129000	104439
2005-09-30	132000	137000	130000	133000	113338
2005-10-03	134000	136000	130000	131000	45483
2005-10-04	131000	134000	131000	132000	41985
2005-10-05	135000	136000	130000	132000	52715
2005-10-06	131000	132000	126000	127000	44700
2005-10-07	128000	129000	126000	129000	21468
2005-10-11	127000	128000	124000	127000	46720
2005-10-12	127000	128000	124000	125000	33382
2005-10-13	125000	125000	123000	125000	28433
2005-10-14	124000	124000	122000	122000	27663
2005-10-17	122000	123000	119000	120000	40859
2005-10-18	120000	122000	120000	121000	30549
2005-10-19	122000	122000	119000	121000	36318
2005-10-20	122000	123000	120000	121000	43821
2005-10-21	120000	123000	120000	123000	35020
2005-10-24	123000	123000	118000	118000	55799
2005-10-25	119000	121000	118000	118000	48841
2005-10-26	118000	119000	117000	118000	40118
2005-10-27	118000	121000	118000	120000	48034
2005-10-28	121000	126000	118000	126000	142474
2005-10-31	124000	126000	123000	123000	37488
2005-11-01	125000	126000	123000	124000	18877
2005-11-02	123000	124000	120000	122000	46057
2005-11-04	123000	123000	119000	121000	48372
2005-11-07	120000	121000	118000	120000	79036
2005-11-08	120000	121000	119000	120000	29232
2005-11-09	119000	121000	118000	119000	44322
2005-11-10	121000	132000	120000	128000	232185
2005-11-11	130000	135000	128000	130000	151398
2005-11-14	131000	136000	129000	134000	100287
2005-11-15	131000	135000	131000	132000	61995
2005-11-16	133000	139000	131000	139000	135293
2005-11-17	138000	149000	137000	146000	214232
2005-11-18	155000	157000	149000	151000	140696
2005-11-21	151000	152000	145000	151000	93345
2005-11-22	148000	148000	144000	147000	99038
2005-11-24	145000	148000	141000	143000	83431
2005-11-25	141000	143000	137000	143000	71277
2005-11-28	143000	146000	141000	143000	71786
2005-11-29	144000	152000	143000	147000	100498
2005-11-30	148000	149000	145000	145000	42890
2005-12-01	144000	145000	141000	145000	55534
2005-12-02	147000	148000	144000	147000	46083
2005-12-05	146000	147000	143000	143000	44571
2005-12-06	143000	145000	141000	141000	37186
2005-12-07	141000	142000	139000	140000	40483
2005-12-08	141000	144000	137000	139000	94348
2005-12-09	139000	143000	138000	142000	107774
2005-12-12	144000	146000	141000	142000	88140
2005-12-13	142000	142000	139000	141000	46766
2005-12-14	143000	145000	140000	142000	101605
2005-12-15	143000	146000	142000	145000	114453
2005-12-16	145000	146000	139000	139000	93203
2005-12-19	141000	147000	139000	146000	146070
2005-12-20	150000	158000	148000	155000	313597
2005-12-21	157000	159000	151000	152000	182529
2005-12-22	155000	156000	152000	155000	75399
2005-12-26	157000	164000	156000	160000	177883
2005-12-27	162000	168000	160000	164000	169667
2005-12-28	165000	165000	161000	163000	61615
2005-12-29	165000	183000	165000	181000	312500
2005-12-30	185000	190000	178000	179000	225587
2006-01-04	185000	185000	176000	176000	87438
2006-01-05	177000	180000	173000	174000	80028
2006-01-06	173000	178000	168000	172000	126164
2006-01-10	174000	175000	169000	169000	79524
2006-01-11	170000	180000	168000	179000	120911
2006-01-12	178000	180000	172000	175000	90690
2006-01-13	174000	178000	172000	177000	100371
2006-01-16	177000	182000	173000	179000	129606
2006-01-17	173000	176000	162000	164000	190132
2006-01-18	149000	153000	137000	148000	279715
2006-01-19	155000	168000	154000	168000	243805
2006-01-20	173000	173000	155000	159000	201144
2006-01-23	150000	156000	149000	151000	120439
2006-01-24	163000	165000	159000	161000	187125
2006-01-25	161000	163000	155000	155000	119484
2006-01-26	158000	159000	152000	154000	67000
2006-01-27	157000	160000	155000	158000	67601
2006-01-30	160000	161000	156000	157000	58105
2006-01-31	158000	159000	156000	159000	78330
2006-02-01	157000	158000	152000	153000	90009
2006-02-02	155000	156000	152000	152000	54695
2006-02-03	152000	155000	150000	154000	78737
2006-02-06	155000	155000	151000	152000	64544
2006-02-07	153000	159000	152000	159000	93257
2006-02-08	156000	159000	152000	153000	118165
2006-02-09	155000	155000	149000	151000	110556
2006-02-10	149000	150000	142000	145000	144196
2006-02-13	141000	141000	135000	136000	112136
2006-02-14	133000	141000	126000	141000	187799
2006-02-15	137000	139000	131000	132000	131918
2006-02-16	133000	135000	130000	135000	73011
2006-02-17	139000	145000	134000	134000	180964
2006-02-20	132000	132000	126000	128000	102456
2006-02-21	132000	141000	131000	138000	135069
2006-02-22	138000	140000	131000	133000	110682
2006-02-23	137000	141000	135000	139000	129130
2006-02-24	136000	139000	136000	139000	83153
2006-02-27	141000	146000	139000	139000	146764
2006-02-28	139000	140000	136000	138000	93143
2006-03-01	136000	137000	133000	134000	81526
2006-03-02	138000	139000	136000	137000	65703
2006-03-03	137000	138000	135000	136000	60345
2006-03-06	140000	147000	140000	146000	155303
2006-03-07	144000	145000	141000	145000	76301
2006-03-08	144000	145000	139000	142000	90130
2006-03-09	143000	148000	142000	147000	67986
2006-03-10	147000	149000	145000	146000	61518
2006-03-13	149000	150000	148000	150000	39149
2006-03-14	150000	153000	149000	150000	58723
2006-03-15	151000	153000	151000	152000	37658
2006-03-16	152000	153000	148000	148000	24214
2006-03-17	148000	149000	145000	147000	51138
2006-03-20	147000	148000	146000	146000	63889
2006-03-22	144000	145000	138000	138000	92516
2006-03-23	141000	142000	137000	139000	77928
2006-03-24	139000	141000	137000	139000	44374
2006-03-27	139000	139000	136000	136000	55874
2006-03-28	66900	67000	65300	66100	179695
2006-03-29	68100	71100	68000	71100	262168
2006-03-30	72900	75000	70300	70700	261070
2006-03-31	72000	72100	71200	71800	128670
2006-04-03	72500	72900	72000	72400	118517
2006-04-04	72600	72800	70200	71000	124690
2006-04-05	71500	71600	70000	70200	95056
2006-04-06	70200	70900	69500	69900	97468
2006-04-07	69800	70100	68800	69900	108693
2006-04-10	69400	69800	69100	69500	60535
2006-04-11	69200	69300	68100	68500	93906
2006-04-12	68000	68400	67100	68000	126725
2006-04-13	68100	68400	67100	67600	89188
2006-04-14	68000	68000	67000	67000	66655
2006-04-17	66500	66900	65300	66000	154687
2006-04-18	65500	67100	65400	66500	96750
2006-04-19	68500	69100	67000	67000	136424
2006-04-20	69000	69000	66100	66700	125041
2006-04-21	68000	68200	65200	66100	161676
2006-04-24	65300	65700	64000	64200	167951
2006-04-25	65200	68000	65000	67500	135681
2006-04-26	67500	67600	66000	66900	89796
2006-04-27	67400	67900	66800	67500	75622
2006-04-28	67000	67300	65600	66400	89420
2006-05-01	66100	66800	65300	65600	64814
2006-05-02	65100	65800	65000	65700	79514
2006-05-08	66400	67000	65800	66700	86504
2006-05-09	66600	67400	66000	66500	104198
2006-05-10	67400	67800	65600	66900	126538
2006-05-11	66500	67800	66100	66400	108513
2006-05-12	65700	65900	65100	65300	110920
2006-05-15	65000	65400	64300	64900	86099
2006-05-16	65900	66400	64000	64000	119130
2006-05-17	64300	64800	60200	62600	165222
2006-05-18	60100	60800	59000	60500	217500
2006-05-19	60300	62500	59400	62000	234745
2006-05-22	61800	62100	60100	60200	159569
2006-05-23	60000	60600	59000	59500	128748
2006-05-24	59800	60100	57700	59300	132842
2006-05-25	59000	61700	58500	60800	214559
2006-05-26	62000	62200	60900	61700	157596
2006-05-29	61500	61900	60300	61600	103525
2006-05-30	61600	62000	61000	61600	79172
2006-05-31	60100	60500	58500	59100	136551
2006-06-01	59500	60000	57200	57700	141093
2006-06-02	57800	59300	54400	58700	169507
2006-06-05	57900	58400	57000	57800	121662
2006-06-06	57000	57000	55500	56300	122899
2006-06-07	55800	56000	54300	55000	126332
2006-06-08	53500	53600	51200	52000	186725
2006-06-09	52400	54200	50500	53300	185632
2006-06-12	53000	56100	52500	55200	135159
2006-06-13	54200	57400	53700	55000	195216
2006-06-14	55100	58400	54500	57700	165363
2006-06-15	58700	59000	57800	57800	122345
2006-06-16	59300	59400	57400	57400	96463
2006-06-19	57100	60300	57000	59700	147346
2006-06-20	59400	60400	58200	58900	108585
2006-06-21	57900	58800	55900	56700	118503
2006-06-22	57900	58900	57200	58000	117941
2006-06-23	57400	59300	57400	58800	126643
2006-06-26	58300	61300	58300	60900	141681
2006-06-27	60500	61200	59900	60200	103329
2006-06-28	59200	60300	57900	58500	171823
2006-06-29	59200	60100	58600	59400	302957
2006-06-30	60100	60800	59200	60600	122614
2006-07-03	60800	62100	60100	61600	119254
2006-07-04	62200	62300	61600	62000	81931
2006-07-05	61600	62200	61400	62100	70975
2006-07-06	61900	62200	60900	61700	94461
2006-07-07	61900	61900	59400	60600	130660
2006-07-10	59600	61800	59500	61600	115912
2006-07-11	61900	62000	60300	61000	80269
2006-07-12	61000	61600	60400	60500	69593
2006-07-13	60100	60900	59200	59600	96456
2006-07-14	58600	58800	57100	57300	133586
2006-07-18	56800	56800	54900	55700	113530
2006-07-19	54700	54800	52100	52800	125941
2006-07-20	53800	56300	53800	56200	129696
2006-07-21	55700	55700	53300	55000	126791
2006-07-24	53000	53000	50300	51100	336404
2006-07-25	52600	53400	50000	50000	286032
2006-07-26	49850	49900	45700	46200	466415
2006-07-27	45400	48250	43250	46200	519231
2006-07-28	46600	47400	46000	46700	152894
2006-07-31	47900	49150	47400	48500	178900
2006-08-01	48500	49100	47700	48550	169672
2006-08-02	48350	48900	48000	48500	162686
2006-08-03	48900	49050	47200	47500	144055
2006-08-04	47500	47950	47000	47550	87795
2006-08-07	47400	47450	45400	45800	130125
2006-08-08	45600	45800	44100	44950	155098
2006-08-09	44850	46850	43800	46500	137498
2006-08-10	46350	46400	45500	46100	95148
2006-08-11	45500	45800	45250	45500	73572
2006-08-14	45100	46250	45100	46200	93123
2006-08-15	46300	47850	45700	47100	241960
2006-08-16	47900	49200	47600	48450	302436
2006-08-17	50000	51700	48800	49500	329447
2006-08-18	51300	51700	50300	51000	195027
2006-08-21	51000	51000	49200	49500	182241
2006-08-22	48850	49900	48650	49350	133006
2006-08-23	49450	50100	49100	49500	138418
2006-08-24	49200	49400	48650	49000	109439
2006-08-25	48700	48750	46300	46600	181885
2006-08-28	45950	46000	44850	45100	179331
2006-08-29	45500	46400	44800	45250	168690
2006-08-30	45250	45250	44000	44000	156030
2006-08-31	44250	45350	44000	44050	162918
2006-09-01	44050	44300	43300	43750	180478
2006-09-04	44950	46050	44000	45800	278996
2006-09-05	46100	47300	45900	47000	216513
2006-09-06	46550	46650	44850	45150	235397
2006-09-07	44350	44600	44050	44250	182602
2006-09-08	44000	44150	43250	43600	264168
2006-09-11	43650	43900	42600	42650	202110
2006-09-12	42850	43500	42150	42250	239033
2006-09-13	42650	42800	41000	41150	147635
2006-09-14	41200	42050	40200	41150	241535
2006-09-15	41550	42400	41150	41700	174055
2006-09-19	42450	42700	41200	41450	138262
2006-09-20	40300	40350	39600	40200	210851
2006-09-21	40500	40750	39200	39750	191865
2006-09-22	40550	41700	40350	41350	288228
2006-09-25	42300	42550	40600	41250	269498
2006-09-26	42050	43150	41400	41850	177889
2006-09-27	42250	42800	41850	42650	120432
2006-09-28	42700	43900	42350	43750	171869
2006-09-29	44100	44650	43650	44450	206816
2006-10-02	44100	44300	43200	43400	180725
2006-10-03	42200	42900	41800	42700	196764
2006-10-04	43000	43600	41600	42000	229239
2006-10-05	43500	43550	42300	42400	122426
2006-10-06	42000	42200	41100	41750	115137
2006-10-10	40950	41400	40450	40600	128643
2006-10-11	40650	40650	39150	39300	296626
2006-10-12	39700	41700	39500	41200	317911
2006-10-13	41450	43600	41450	43350	369306
2006-10-16	43750	47050	43700	46950	455565
2006-10-17	46550	47650	45500	46900	335802
2006-10-18	46850	49100	45650	48650	379219
2006-10-19	48600	49100	46900	47450	323015
2006-10-20	46700	47150	45150	45500	261883
2006-10-23	43900	46500	43550	45500	332762
2006-10-24	48700	49500	48650	49500	314130
2006-10-25	49350	49350	46900	47200	576388
2006-10-26	47900	48550	47050	47200	234192
2006-10-27	47600	47900	46250	47300	203050
2006-10-30	46900	47500	45850	45900	257831
2006-10-31	46150	46700	45500	45500	120724
2006-11-01	45300	46250	44600	45800	178195
2006-11-02	45700	46450	45150	45900	161809
2006-11-06	45600	45700	43650	43700	167915
2006-11-07	44000	44400	43650	43650	129635
2006-11-08	43350	43550	40600	41050	340545
2006-11-09	42650	43350	41600	42350	203546
2006-11-10	42000	42100	41400	41500	199343
2006-11-13	41050	43150	40900	42550	171123
2006-11-14	42950	43450	42550	43100	143701
2006-11-15	43450	43550	42050	42350	136231
2006-11-16	42100	42950	41450	41700	109195
2006-11-17	41350	41700	40000	40200	185760
2006-11-20	40500	40800	39950	40050	230022
2006-11-21	41000	41800	40550	41200	217080
2006-11-22	41500	43550	41000	43100	241664
2006-11-24	43100	43200	41500	42900	169037
2006-11-27	42500	44300	42350	43650	164596
2006-11-28	43000	44900	42600	44400	169460
2006-11-29	44750	45200	44450	44700	125788
2006-11-30	44750	46000	44600	45600	224212
2006-12-01	45450	46250	45200	45550	119974
2006-12-04	45600	47300	45400	46850	179672
2006-12-05	47100	47800	46800	46850	173217
2006-12-06	47000	47600	45850	47050	180449
2006-12-07	47300	47800	46900	47350	135242
2006-12-08	47000	47500	46350	46800	145456
2006-12-11	46800	46950	45300	45550	134320
2006-12-12	45700	46350	45450	45900	103792
2006-12-13	45300	45800	45050	45800	69154
2006-12-14	46500	47150	45850	47050	133460
2006-12-15	47550	48700	47300	48550	315768
2006-12-18	49350	52200	49100	51400	540467
2006-12-19	50500	50800	48800	49000	272841
2006-12-20	49050	51100	49000	50600	213932
2006-12-21	50600	51800	49950	50100	174374
2006-12-22	49600	49850	48550	49100	100874
2006-12-25	48500	48750	48100	48400	76630
2006-12-26	48000	49400	48000	49000	81478
2006-12-27	49250	49750	47750	48200	95567
2006-12-28	48200	48600	47100	47900	114822
2006-12-29	47500	47750	47000	47400	70210
2007-01-04	48400	50000	48300	49300	120498
2007-01-05	49350	49850	48900	49300	127598
2007-01-09	49350	49650	48400	48750	120420
2007-01-10	48750	49000	47550	47850	127605
2007-01-11	47850	47900	46600	47000	115501
2007-01-12	47150	47250	46300	46600	101959
2007-01-15	46500	46850	45800	46150	148555
2007-01-16	45900	47650	45900	47200	171695
2007-01-17	47150	47150	46300	46650	128476
2007-01-18	47000	47800	46750	47450	371722
2007-01-19	47700	49400	47400	48900	328005
2007-01-22	49300	49800	48150	48500	277626
2007-01-23	48250	48350	47350	47550	188625
2007-01-24	48100	48300	47500	47750	168292
2007-01-25	49600	49750	48200	48250	295413
2007-01-26	47850	48100	47150	47450	196020
2007-01-29	47500	47500	46250	46600	269415
2007-01-30	46600	46950	45500	45950	236898
2007-01-31	46050	46150	44600	45200	201136
2007-02-01	45050	45150	44100	44300	263970
2007-02-02	44650	44900	43000	43300	324301
2007-02-05	42550	43150	42200	42700	261645
2007-02-06	43000	43750	42800	42850	223488
2007-02-07	43250	44200	43000	43200	213799
2007-02-08	43600	43800	43200	43700	207589
2007-02-09	43750	43950	42700	43600	230390
2007-02-13	43750	44000	43350	43750	130303
2007-02-14	43800	43950	43400	43450	92476
2007-02-15	43450	44600	42350	44350	278434
2007-02-16	44050	45600	43800	45050	329283
2007-02-19	45800	45900	45050	45250	168317
2007-02-20	45300	45650	45050	45300	125117
2007-02-21	45600	45650	45100	45300	164175
2007-02-22	45450	45550	44550	45200	155607
2007-02-23	45250	46650	44950	46500	202034
2007-02-26	47000	47200	46150	46600	169556
2007-02-27	46650	46750	45700	46350	124248
2007-02-28	43950	45350	43550	44950	196032
2007-03-01	45050	45200	42600	43800	184959
2007-03-02	43000	44200	42900	43600	162652
2007-03-05	42800	42900	40650	41000	272792
2007-03-06	41000	41800	40700	41750	322735
2007-03-07	42300	42450	41400	41650	130131
2007-03-08	41250	41700	40900	41700	141425
2007-03-09	41600	41800	41150	41500	233921
2007-03-12	41700	41900	40500	40650	257670
2007-03-13	40700	41050	40500	40500	206537
2007-03-14	40100	40300	39500	39650	269804
2007-03-15	39800	40100	39550	39600	144953
2007-03-16	40000	40100	39350	39550	243368
2007-03-19	39400	39900	39200	39450	169434
2007-03-20	40100	41000	40100	40650	206306
2007-03-22	41400	41650	41200	41300	124844
2007-03-23	41350	41450	40700	41100	127037
2007-03-26	41500	41800	41250	41550	96286
2007-03-27	41000	41300	40300	40650	103979
2007-03-28	40700	41550	40600	40800	132941
2007-03-29	40400	41050	39900	40450	146366
2007-03-30	40750	40900	40300	40650	95542
2007-04-02	41450	42150	41000	41050	250074
2007-04-03	41450	41750	40700	41550	181868
2007-04-04	41800	41800	41150	41250	109960
2007-04-05	41350	41350	40500	40650	124573
2007-04-06	40650	40650	40050	40150	135272
2007-04-09	40150	40250	39850	40000	126294
2007-04-10	40100	40400	39950	40000	109885
2007-04-11	39950	40050	39150	39500	147637
2007-04-12	39500	39600	39200	39400	158163
2007-04-13	39650	39950	39400	39450	142042
2007-04-16	39500	39900	39450	39700	121861
2007-04-17	39750	39800	39000	39200	215183
2007-04-18	39450	39450	38350	38800	268079
2007-04-19	38400	38550	37600	38200	238102
2007-04-20	38050	38100	37300	37500	157504
2007-04-23	36700	36850	35750	36050	323631
2007-04-24	35950	36900	35900	36750	248525
2007-04-25	39550	40750	39050	40750	630929
2007-04-26	40750	43500	40650	42600	716356
2007-04-27	42650	43450	41500	41650	281804
2007-05-01	42250	42600	42100	42400	170410
2007-05-02	42800	42950	42150	42750	204197
2007-05-07	45150	45700	44350	44800	414603
2007-05-08	44800	46350	44300	45500	413723
2007-05-09	45400	45650	44300	45100	209592
2007-05-10	44550	45000	43450	43600	185998
2007-05-11	43400	43400	42700	43000	146021
2007-05-14	43350	43850	42600	42900	133518
2007-05-15	42450	43400	41900	42050	179851
2007-05-16	42000	42050	41250	41750	121250
2007-05-17	41700	41900	40400	40850	223035
2007-05-18	40500	40650	39550	40250	177762
2007-05-21	41450	41500	40250	40900	160248
2007-05-22	41400	42500	41250	42050	238581
2007-05-23	42100	42300	41000	41600	152356
2007-05-24	41600	41600	40600	41200	145154
2007-05-25	40800	42700	40800	42300	250608
2007-05-28	42000	43200	41850	42600	193426
2007-05-29	42200	42850	42100	42450	105197
2007-05-30	42550	43000	41750	42000	132391
2007-05-31	42350	42400	41600	41700	101263
2007-06-01	42100	42600	41500	42500	145031
2007-06-04	42900	42950	42100	42300	102199
2007-06-05	42350	43250	42200	42900	161688
2007-06-06	42700	42750	42200	42400	75026
2007-06-07	41700	42600	41600	42300	99376
2007-06-08	42050	42100	41300	41600	152793
2007-06-11	41950	42100	41100	41250	102341
2007-06-12	41000	41150	40350	40850	128817
2007-06-13	40100	40550	39850	40450	89572
2007-06-14	41150	42000	41150	41750	168549
2007-06-15	43750	43800	42500	43350	273820
2007-06-18	43700	44050	43200	43850	167191
2007-06-19	43850	43900	42600	43250	158795
2007-06-20	42850	43100	42100	42300	130336
2007-06-21	42000	43000	42000	42850	87541
2007-06-22	42850	43600	42550	43100	143771
2007-06-25	42600	43250	42500	42800	102753
2007-06-26	42450	42600	41550	41850	177283
2007-06-27	41900	41900	41500	41600	68706
2007-06-28	42000	42350	41650	42050	70917
2007-06-29	42100	42100	41300	41850	71292
2007-07-02	42200	42200	41700	41850	70571
2007-07-03	42050	42350	41750	41900	106021
2007-07-04	42300	42350	41750	42200	74308
2007-07-05	42100	42500	41750	41950	85337
2007-07-06	42000	42100	41600	42000	88424
2007-07-09	42100	42450	41850	42400	108055
2007-07-10	42000	42000	40850	41250	212545
2007-07-11	40850	41450	40650	40800	112461
2007-07-12	40850	41250	40500	40650	124598
2007-07-13	41250	41450	40850	41150	105955
2007-07-17	41100	41250	40400	40900	125640
2007-07-18	40350	40650	39850	40050	164852
2007-07-19	40250	40450	40100	40300	92320
2007-07-20	40450	40450	39800	39850	101443
2007-07-23	40000	40000	39350	39600	127580
2007-07-24	38800	40800	38350	40750	251985
2007-07-25	39350	39500	38650	39400	224261
2007-07-26	38500	38500	38000	38000	193136
2007-07-27	36800	37650	36650	37150	173342
2007-07-30	36900	37400	36600	37400	101392
2007-07-31	37500	37600	36950	37150	91954
2007-08-01	37000	37050	36150	36550	151437
2007-08-02	36600	37150	36350	36950	176844
2007-08-03	38150	38200	37150	37300	120702
2007-08-06	37350	38200	37050	38050	140531
2007-08-07	38050	38050	37100	37150	135485
2007-08-08	36900	37650	36550	37500	178847
2007-08-09	38100	40350	37950	38500	371910
2007-08-10	38950	40550	38850	39950	299509
2007-08-13	39550	39650	38000	39650	182546
2007-08-14	39600	39600	38400	38800	120159
2007-08-15	38850	38850	37950	38200	166825
2007-08-16	37000	37700	36400	37450	209738
2007-08-17	36650	37000	35200	35650	242055
2007-08-20	36250	36550	35900	36000	111785
2007-08-21	36000	36800	35750	36600	128438
2007-08-22	37400	40000	37300	38700	298377
2007-08-23	40000	40450	39200	39800	304711
2007-08-24	39800	40200	38550	38650	193419
2007-08-27	39450	40450	39450	40000	176613
2007-08-28	39850	42050	39850	41550	285612
2007-08-29	40750	43950	40400	43450	430420
2007-08-30	43200	44000	42850	43300	183181
2007-08-31	42950	43400	42150	42900	150659
2007-09-03	44000	44000	42850	43100	105551
2007-09-04	42700	43650	42650	42950	66098
2007-09-05	44000	44750	43200	43600	284022
2007-09-06	43000	44200	42400	44050	114099
2007-09-07	44000	44000	42950	43400	164809
2007-09-10	42900	43300	42600	43000	120548
2007-09-11	43050	44050	42400	43500	153490
2007-09-12	44000	44150	41650	42000	190517
2007-09-13	41800	41850	40550	40900	153829
2007-09-14	40650	41000	40400	40750	147006
2007-09-18	40350	40350	39200	39300	138606
2007-09-19	40200	40600	40050	40300	98659
2007-09-20	40400	40700	39800	40600	131533
2007-09-21	40200	40450	39800	39900	113704
2007-09-25	40700	40750	39900	40500	144523
2007-09-26	40150	41400	39850	41050	202814
2007-09-27	41250	42500	41250	42400	122968
2007-09-28	43200	44250	42850	43500	211704
2007-10-01	43500	43950	42950	43250	90092
2007-10-02	44050	44850	43400	43900	190766
2007-10-03	44550	46650	44200	46350	295868
2007-10-04	46350	49050	45650	49000	425786
2007-10-05	49800	51800	49100	51000	454867
2007-10-09	49500	50700	48800	49750	279389
2007-10-10	50400	51200	49900	50500	231941
2007-10-11	50600	53900	50100	53500	255949
2007-10-12	52000	52700	50500	51200	239281
2007-10-15	51700	52100	50600	51200	110107
2007-10-16	50700	51000	49050	49100	190048
2007-10-17	50800	52000	50300	51700	315112
2007-10-18	51700	54000	51700	53800	209390
2007-10-19	54000	57200	53000	56200	343480
2007-10-22	54200	58100	54000	57600	359889
2007-10-23	58600	59000	54500	54900	352165
2007-10-24	56500	57200	55300	55800	243381
2007-10-25	50800	51400	50800	50800	300234
2007-10-26	50300	50300	49200	50300	651149
2007-10-29	51300	53700	51200	52500	516219
2007-10-30	53000	53200	51400	52100	308238
2007-10-31	52300	52400	50200	51100	161612
2007-11-01	51200	52300	50200	51700	193570
2007-11-02	50200	54500	50200	53500	404093
2007-11-05	54000	54200	50100	50500	280647
2007-11-06	50600	52200	49050	49150	300256
2007-11-07	49500	50500	47000	47950	449471
2007-11-08	46750	48800	46100	48100	329296
2007-11-09	48150	48500	46200	46550	226398
2007-11-12	45350	46050	44650	45700	302750
2007-11-13	45650	46200	44750	45000	266798
2007-11-14	46950	48700	46900	48000	199146
2007-11-15	48000	48200	47000	47200	164879
2007-11-16	46300	47200	46100	47000	106787
2007-11-19	48150	49100	46750	46800	162377
2007-11-20	46000	47700	45150	47400	162639
2007-11-21	48000	48800	46900	47250	121327
2007-11-22	47300	51200	46750	50500	375439
2007-11-26	50300	52600	49950	51600	240622
2007-11-27	51100	53800	50900	53500	248397
2007-11-28	53500	54800	53000	54400	227819
2007-11-29	55400	55500	53300	54800	208741
2007-11-30	53600	54400	53100	53300	159044
2007-12-03	54300	54800	53300	54300	177171
2007-12-04	55800	57900	54300	54300	432573
2007-12-05	54800	56300	53200	55000	266078
2007-12-06	55500	55700	53300	53600	169838
2007-12-07	54100	54600	53100	53200	120047
2007-12-10	53000	53800	51600	52000	126997
2007-12-11	52100	52600	50500	51900	152014
2007-12-12	50900	53700	50900	53200	185965
2007-12-13	52500	52800	50300	50600	143931
2007-12-14	50100	51600	49800	50300	174557
2007-12-17	49400	49950	47100	47300	202741
2007-12-18	46500	49550	46450	48550	194387
2007-12-19	49750	49950	48600	48850	115199
2007-12-20	49000	49350	47750	48100	88719
2007-12-21	48700	50700	47750	50300	231599
2007-12-25	51900	52700	51000	51200	113893
2007-12-26	51600	51600	50000	51000	94431
2007-12-27	51000	51500	49900	50200	93241
2007-12-28	49000	50600	49000	50000	55601
2008-01-04	48650	48700	47500	48100	100963
2008-01-07	47700	49100	46800	47300	128428
2008-01-08	47050	48850	47050	48850	110811
2008-01-09	47250	49400	47050	49100	144123
2008-01-10	49150	49700	47850	48000	103454
2008-01-11	48400	48700	46200	46550	203113
2008-01-15	46950	48200	45350	45950	145748
2008-01-16	44350	44400	41950	41950	394973
2008-01-17	42750	43450	41000	42350	215251
2008-01-18	40700	44250	40650	43850	185487
2008-01-21	43000	44850	42550	42850	118008
2008-01-22	41000	41300	38850	38850	204951
2008-01-23	40350	40600	39700	40050	180200
2008-01-24	40450	41500	39450	39550	135762
2008-01-25	40150	41000	39450	41000	232706
2008-01-28	40600	40850	37650	38250	267077
2008-01-29	38800	39300	38300	38600	237477
2008-01-30	36700	38200	36200	36800	302336
2008-01-31	38000	40800	37550	40800	438912
2008-02-01	42600	43800	40850	42000	456223
2008-02-04	46000	46000	46000	46000	20272
2008-02-05	50000	50000	47400	48300	708301
2008-02-06	46900	47700	45650	45650	336823
2008-02-07	45450	46350	44800	45200	232976
2008-02-08	45000	47950	44900	46600	178174
2008-02-12	47800	48100	46800	47550	187470
2008-02-13	47700	47900	46900	47500	178199
2008-02-14	48300	49800	48000	49150	160685
2008-02-15	49550	51400	49300	51400	234491
2008-02-18	51400	52100	50300	51300	180379
2008-02-19	50800	51200	49550	49850	155547
2008-02-20	49200	49900	46950	47450	235697
2008-02-21	48500	49500	47850	48550	153497
2008-02-22	49750	49750	47850	48150	149031
2008-02-25	48300	49700	48000	49650	128455
2008-02-26	50600	50600	48750	48900	113487
2008-02-27	49950	50000	49400	49600	105259
2008-02-28	48400	49600	48300	48700	92424
2008-02-29	47750	48100	45500	47050	187128
2008-03-03	45200	46250	44950	44950	122299
2008-03-04	45000	45550	43200	43900	192043
2008-03-05	44700	46050	44350	45050	160022
2008-03-06	47450	49050	46750	47500	193780
2008-03-07	46200	47000	45600	46100	157384
2008-03-10	45700	46150	43500	43600	153186
2008-03-11	43000	44800	42350	44300	145571
2008-03-12	46700	47400	46200	46750	154352
2008-03-13	46650	47250	45500	47100	131201
2008-03-14	47150	47150	45100	46250	186966
2008-03-17	45500	47000	43850	46200	133137
2008-03-18	46600	47000	44600	45850	146796
2008-03-19	47450	47850	46350	47850	145105
2008-03-21	47850	49350	47550	49250	127138
2008-03-24	49900	51000	49450	50500	165514
2008-03-25	51000	52400	50600	51900	178573
2008-03-26	52800	54000	52700	53800	187101
2008-03-27	53300	53500	52400	52900	90449
2008-03-28	53000	54600	52400	53600	122651
2008-03-31	52600	54500	51300	51800	167128
2008-04-01	51800	52400	49900	50900	158196
2008-04-02	51400	52300	50300	52300	242687
2008-04-03	52300	53700	51900	53300	113588
2008-04-04	53500	53500	51300	51600	107387
2008-04-07	52900	55400	52600	54300	346456
2008-04-08	54700	55000	51300	51400	196144
2008-04-09	50900	51200	48550	50200	270068
2008-04-10	51500	53200	50400	51600	254440
2008-04-11	53100	53200	51800	53000	132829
2008-04-14	52500	52900	51700	52000	72317
2008-04-15	52200	53500	52200	52900	81334
2008-04-16	53300	53300	51800	52200	80124
2008-04-17	53200	53900	52600	53000	89380
2008-04-18	53900	53900	52000	53000	106387
2008-04-21	54000	54000	52200	53200	126641
2008-04-22	53100	53200	52600	53000	65177
2008-04-23	52100	54200	51700	52700	114928
2008-04-24	52700	53100	51500	51500	91056
2008-04-25	51700	52200	51200	52100	73256
2008-04-28	49700	50400	47100	47150	414051
2008-04-30	46800	47350	44750	46050	361447
2008-05-01	46250	47650	45750	46550	206049
2008-05-02	47500	47500	46750	47250	142574
2008-05-07	45650	46800	45100	45700	389433
2008-05-08	44900	44900	43000	43250	416340
2008-05-09	44050	44850	43650	43900	210318
2008-05-12	44300	44750	43800	44450	136845
2008-05-13	44850	45850	44500	45450	159438
2008-05-14	46250	46350	45450	45850	154668
2008-05-15	46200	46300	45000	45000	158969
2008-05-16	46000	46000	45100	45200	126175
2008-05-19	45600	45700	44800	45100	126051
2008-05-20	45150	45300	42100	43300	252845
2008-05-21	42950	43850	42500	43400	159989
2008-05-22	43450	43800	42500	43800	165281
2008-05-23	44400	44950	43800	43850	189978
2008-05-26	45850	46550	45550	45850	280665
2008-05-27	46150	46200	45150	45150	123768
2008-05-28	45300	45700	44650	44800	132500
2008-05-29	45200	45300	44200	44200	155951
2008-05-30	44500	45500	44300	45350	113787
2008-06-02	45550	46500	45450	46000	267163
2008-06-03	45900	46000	44050	44550	243592
2008-06-04	44700	44850	44050	44700	218771
2008-06-05	44950	45600	44650	44850	181809
2008-06-06	45100	45300	44050	44200	159266
2008-06-09	43800	43800	42800	43750	131072
2008-06-10	43850	43900	42200	42200	197838
2008-06-11	42200	42700	41600	41950	220789
2008-06-12	41750	42250	41000	42200	149557
2008-06-13	42200	42250	41100	41550	226378
2008-06-16	41700	42300	41300	42100	155957
2008-06-17	42000	43050	41900	42950	168921
2008-06-18	43150	44750	43100	44450	209336
2008-06-19	44350	44400	42500	42900	194757
2008-06-20	43100	43700	42050	42550	139911
2008-06-23	42150	42300	41550	41950	133844
2008-06-24	41700	41750	41050	41250	122990
2008-06-25	41250	41300	40350	40550	180486
2008-06-26	40650	42600	40650	42100	182042
2008-06-27	41500	41700	40900	41300	157169
2008-06-30	41100	41250	40500	40900	135399
2008-07-01	40750	41100	40400	41000	134145
2008-07-02	41050	41600	40400	40800	173628
2008-07-03	40500	41800	40100	41600	217018
2008-07-04	41750	42750	41200	41650	197941
2008-07-07	41850	43050	41500	42800	119325
2008-07-08	43600	43650	42200	42450	170522
2008-07-09	42700	43050	41550	41750	140568
2008-07-10	41650	43300	41600	42550	126560
2008-07-11	42400	42400	40450	41000	131417
2008-07-14	40600	40650	39500	39750	171936
2008-07-15	39050	39250	38250	38700	170635
2008-07-16	38000	38200	36550	37050	258120
2008-07-17	37900	38950	37800	38550	179245
2008-07-18	39100	39200	37650	38250	172812
2008-07-22	37650	37900	37050	37600	136874
2008-07-23	38000	39700	37800	39350	146348
2008-07-24	40150	41500	39600	41250	191707
2008-07-25	40050	40800	39800	40000	93024
2008-07-28	41200	42750	41000	41700	233437
2008-07-29	40500	41250	40000	40700	104627
2008-07-30	41200	41600	41150	41450	80124
2008-07-31	41750	41850	40750	41150	77774
2008-08-01	40350	40450	38750	39050	143246
2008-08-04	38600	39200	37550	38050	128320
2008-08-05	38000	38050	37050	37300	136634
2008-08-06	37500	38100	37500	37650	134520
2008-08-07	37700	38050	37100	37850	153776
2008-08-08	37200	37850	37200	37750	106235
2008-08-11	38100	39300	38050	38850	107548
2008-08-12	38900	39350	38500	38900	93638
2008-08-13	38200	38450	37200	37350	139289
2008-08-14	36650	37250	36200	36600	164297
2008-08-15	36850	37400	36800	37400	79812
2008-08-18	37300	37900	37100	37400	65262
2008-08-19	36700	38600	36700	38500	131676
2008-08-20	37900	40350	37750	40000	195695
2008-08-21	41200	41400	40550	41150	207137
2008-08-22	41150	42250	40300	42200	170789
2008-08-25	42300	42400	40700	40950	96811
2008-08-26	40500	42000	40100	41750	62726
2008-08-27	41700	42800	41600	42150	107813
2008-08-28	42550	42600	41500	41950	60977
2008-08-29	40950	42300	40800	42050	89162
2008-09-01	41450	41850	40850	41000	69291
2008-09-02	41000	41550	39550	40500	79656
2008-09-03	40100	41300	39750	40350	83997
2008-09-04	40400	41200	39950	40700	89819
2008-09-05	39700	40850	39250	40800	160502
2008-09-08	41500	41850	41050	41500	73746
2008-09-09	41400	41400	39900	40050	67160
2008-09-10	39250	41700	39250	41400	92293
2008-09-11	40750	41300	39850	40300	122390
2008-09-12	40350	40800	39900	40150	77077
2008-09-16	36550	36550	36150	36150	381895
2008-09-17	36500	36600	32800	35850	416353
2008-09-18	34500	36600	34350	35900	201857
2008-09-19	37000	37750	35850	36050	182011
2008-09-22	36450	37000	35050	35600	149655
2008-09-24	34800	35200	34050	34550	187207
2008-09-25	34800	35250	34400	34700	127398
2008-09-26	35500	35600	34650	35000	133663
2008-09-29	35000	36150	34800	34900	97205
2008-09-30	32500	34450	32250	33700	104702
2008-10-01	33700	34900	33000	33000	129048
2008-10-02	33200	33850	32600	33200	125962
2008-10-03	32500	33100	31800	31850	129921
2008-10-06	29550	30450	28980	29940	242117
2008-10-07	28140	31900	28120	30500	249394
2008-10-08	29700	29850	28780	28970	322169
2008-10-09	28070	29430	27880	28070	223575
2008-10-10	25670	27270	25600	26320	167214
2008-10-14	28500	29280	27720	27960	236488
2008-10-15	27360	30000	27070	29770	308487
2008-10-16	27970	27970	26780	26850	234567
2008-10-17	29550	29850	28690	29850	302454
2008-10-20	30950	31750	29530	30350	189164
2008-10-21	32350	32400	30100	30650	144616
2008-10-22	29850	30850	28330	28500	191954
2008-10-23	27000	27730	26440	27400	217964
2008-10-24	26900	27000	26180	26530	185889
2008-10-27	28030	29470	26920	28260	379567
2008-10-28	28260	30000	28190	28760	344561
2008-10-29	30950	31800	29900	31600	228019
2008-10-30	30800	31150	29910	30700	317209
2008-10-31	30700	32600	30150	31850	190841
2008-11-04	32650	33850	32100	33450	161461
2008-11-05	35050	35800	34250	35800	233082
2008-11-06	34800	35300	34100	34750	222489
2008-11-07	34000	35000	32500	34000	203435
2008-11-10	34800	35100	34600	34950	153371
2008-11-11	34500	35700	34150	34850	162487
2008-11-12	34100	34550	33500	34050	148250
2008-11-13	33000	33250	32000	32700	194190
2008-11-14	33900	34000	32350	33000	105099
2008-11-17	32200	34100	31850	33150	137673
2008-11-18	33200	33300	31000	31350	165565
2008-11-19	32150	32550	31550	32550	128001
2008-11-20	31500	31700	30750	30750	137598
2008-11-21	30000	30400	28720	30100	229069
2008-11-25	30350	31150	29540	30150	174423
2008-11-26	28850	30250	28850	29320	52865
2008-11-27	29520	30150	29520	29900	71076
2008-11-28	29910	31350	29610	31000	111608
2008-12-01	30650	30700	29950	30400	76330
2008-12-02	29000	29280	28600	28640	102419
2008-12-03	29480	30700	29370	30550	206798
2008-12-04	30950	31550	30100	30700	139425
2008-12-05	31500	31500	30450	30500	86210
2008-12-08	31300	33650	31200	33000	145317
2008-12-09	33700	33900	33150	33300	129234
2008-12-10	33300	33800	32450	33700	134893
2008-12-11	34100	35000	33800	35000	157461
2008-12-12	35050	35600	33150	33600	178172
2008-12-15	34000	34350	33750	33850	78101
2008-12-16	34250	35250	34150	34850	107422
2008-12-17	35450	35750	33850	34400	101096
2008-12-18	34800	35150	34450	34750	84533
2008-12-19	34700	35650	34600	35150	172347
2008-12-22	35600	37100	35250	36900	149538
2008-12-24	37000	37100	35600	36150	67207
2008-12-25	35850	36700	35850	36150	30176
2008-12-26	36450	36500	36150	36500	25001
2008-12-29	36850	37000	36600	36950	41210
2008-12-30	37000	37000	36250	36500	40737
2009-01-05	37150	37450	36550	37000	48382
2009-01-06	37300	40300	37200	38400	222607
2009-01-07	38400	39900	35100	36050	225114
2009-01-08	36200	36300	33500	33950	203189
2009-01-09	34750	34850	33900	34100	132310
2009-01-13	34050	34050	33050	33450	70888
2009-01-14	33050	33250	32200	32300	105613
2009-01-15	31500	32700	31050	31950	125157
2009-01-16	32150	32850	32100	32200	86489
2009-01-19	32950	33100	31450	31700	87701
2009-01-20	31700	31800	30500	30850	127819
2009-01-21	30000	30900	29950	30200	93142
2009-01-22	30550	31200	30150	30900	85875
2009-01-23	30950	32200	30900	30950	123499
2009-01-26	31350	31650	30650	31200	63029
2009-01-27	30900	31400	30200	31250	164271
2009-01-28	32050	32150	29370	29950	392676
2009-01-29	30250	30350	28920	29080	262025
2009-01-30	28260	29110	28000	28990	205971
2009-02-02	28500	28710	27770	28170	157671
2009-02-03	28470	29830	28300	28760	116263
2009-02-04	29000	29400	28920	29340	105599
2009-02-05	29200	29320	28140	28340	105705
2009-02-06	28640	28770	27790	27900	99569
2009-02-09	28300	28380	27500	27800	91960
2009-02-10	27700	28110	27000	27440	125904
2009-02-12	27010	27290	26550	26630	111438
2009-02-13	26930	28590	26820	28190	142625
2009-02-16	28190	29020	28150	28590	94458
2009-02-17	28400	28480	27520	27560	81026
2009-02-18	27590	28260	27300	28210	79276
2009-02-19	28220	28670	28050	28140	78357
2009-02-20	27840	28020	25900	25970	177838
2009-02-23	25730	25800	24140	25100	197891
2009-02-24	24500	26100	24030	25320	177015
2009-02-25	25800	26390	25400	26260	133112
2009-02-26	26280	26890	26170	26460	92326
2009-02-27	27000	28380	26890	28270	211505
2009-03-02	27070	27730	26620	27530	110323
2009-03-03	26330	26520	25000	25060	279974
2009-03-04	24500	25450	24290	25130	154472
2009-03-05	25500	25970	25300	25350	107302
2009-03-06	25000	25080	24350	24830	95003
2009-03-09	24530	24720	22700	22900	221706
2009-03-10	22900	23750	22790	23570	155399
2009-03-11	24240	24570	23990	24100	112850
2009-03-12	24100	24180	22430	22590	196062
2009-03-13	22680	23660	22680	23530	148660
2009-03-16	23660	24680	23610	24170	83282
2009-03-17	24680	25430	24250	25340	148610
2009-03-18	26540	26600	24810	24830	178691
2009-03-19	25000	25720	24890	25520	94855
2009-03-23	26500	27930	25900	27250	218426
2009-03-24	28150	28250	27550	27760	123710
2009-03-25	27000	28060	26990	27950	132307
2009-03-26	27990	28050	27340	27770	101985
2009-03-27	27800	28470	27730	27790	138814
2009-03-30	27710	28400	27000	27000	108706
2009-03-31	26400	27220	25670	25870	96626
2009-04-01	25990	26980	25920	26820	96830
2009-04-02	27420	27580	27010	27380	77298
2009-04-03	27520	27750	27070	27260	91533
2009-04-06	27100	27260	26510	26720	110678
2009-04-07	27180	28220	26850	27800	224009
2009-04-08	28100	28320	27230	27330	178192
2009-04-09	27400	28110	27400	27840	94466
2009-04-10	28140	28240	27560	27770	138909
2009-04-13	27800	27940	26840	27030	103786
2009-04-14	27000	27830	26870	27510	137745
2009-04-15	27500	27830	27190	27250	71275
2009-04-16	27260	27380	26850	26990	150774
2009-04-17	27010	27240	26750	26900	125086
2009-04-20	26920	27000	26360	26460	82767
2009-04-21	25710	25930	25270	25800	105620
2009-04-22	26100	26470	25580	26190	112731
2009-04-23	26050	27760	25800	27540	192509
2009-04-24	27240	27330	26210	26340	140358
2009-04-27	26940	27130	26360	26560	90338
2009-04-28	26790	26920	26350	26350	115956
2009-04-30	26350	26800	24610	24610	329831
2009-05-01	24910	24910	24270	24510	189462
2009-05-07	25200	25260	24260	24480	259047
2009-05-08	24480	25340	24300	25280	203183
2009-05-11	25800	25860	25310	25530	166660
2009-05-12	25310	25480	25010	25020	98830
2009-05-13	25030	25510	24720	25470	108773
2009-05-14	25100	25280	25000	25130	81782
2009-05-15	25290	25450	25190	25190	86627
2009-05-18	25490	25570	25030	25200	99809
2009-05-19	25200	25300	24820	24950	114542
2009-05-20	25250	25250	24920	25100	90364
2009-05-21	25100	25180	24890	24990	97390
2009-05-22	24750	25000	24750	25000	70201
2009-05-25	24840	25110	24750	24880	127332
2009-05-26	24910	24990	24800	24870	108175
2009-05-27	25200	25770	25070	25630	295472
2009-05-28	25660	26070	25370	25600	258630
2009-05-29	25860	25960	25190	25360	134763
2009-06-01	24760	25030	24740	24880	205529
2009-06-02	24930	25080	24810	24860	146018
2009-06-03	24870	25980	24860	25940	241588
2009-06-04	25920	26050	25120	25870	189168
2009-06-05	26220	27630	26030	27280	340037
2009-06-08	27500	27840	26960	27240	172224
2009-06-09	27300	27400	26480	26670	129820
2009-06-10	26750	27500	26510	27500	161988
2009-06-11	27600	28320	27600	28210	288151
2009-06-12	30000	31000	29350	30550	598224
2009-06-15	31750	32150	30600	31950	376205
2009-06-16	31150	31900	30600	31050	270244
2009-06-17	30650	30900	29690	29730	216454
2009-06-18	29580	30800	28870	30000	268328
2009-06-19	30000	31300	29820	31050	248030
2009-06-22	31150	31550	30350	30500	119009
2009-06-23	29400	29650	28820	28830	244717
2009-06-24	29400	29450	28030	28250	288539
2009-06-25	28570	29470	28360	29390	209153
2009-06-26	29580	29930	29340	29780	168295
2009-06-29	29750	29860	28710	28790	120849
2009-06-30	29650	30800	29400	30750	254335
2009-07-01	30500	30950	29850	30150	147961
2009-07-02	29910	30100	29300	29350	149190
2009-07-03	28950	29660	28880	29600	95310
2009-07-06	29900	30650	29410	30550	170994
2009-07-07	31100	31550	30700	30800	174767
2009-07-08	30550	30650	29440	29740	171107
2009-07-09	29500	29590	28830	28880	147192
2009-07-10	29200	29480	28500	28520	103506
2009-07-13	28490	29190	27790	27870	154158
2009-07-14	28100	28440	28020	28140	83802
2009-07-15	28440	28750	28170	28260	88212
2009-07-16	28990	29120	28300	28500	81385
2009-07-17	28350	29470	28310	29460	128319
2009-07-21	30500	30700	29530	29830	118126
2009-07-22	30050	30500	29800	30500	117205
2009-07-23	30550	31000	30100	30200	101562
2009-07-24	30650	31400	30350	31350	160922
2009-07-27	31850	32250	31200	31300	147564
2009-07-28	31600	31900	30600	30950	101024
2009-07-29	31050	33200	31050	33100	318412
2009-07-30	32300	32550	31550	32200	165224
2009-07-31	32250	32400	30600	31050	276273
2009-08-03	31100	31850	31000	31600	118826
2009-08-04	32150	32150	31550	31600	78305
2009-08-05	31650	31900	31200	31250	46768
2009-08-06	31000	31150	30600	30900	86842
2009-08-07	31200	31600	30750	31000	67691
2009-08-10	31550	31600	30850	30900	63049
2009-08-11	30900	31000	30650	30700	61615
2009-08-12	30850	30850	30400	30700	71354
2009-08-13	30950	31100	30500	30950	85434
2009-08-14	30800	31100	30550	30550	62390
2009-08-17	30600	30750	30000	30000	68016
2009-08-18	29630	30150	29520	30050	81422
2009-08-19	30400	30400	29700	29780	77704
2009-08-20	29820	29990	29580	29850	80295
2009-08-21	29990	30000	29200	29290	107451
2009-08-24	30050	30200	29630	29770	84648
2009-08-25	29780	29940	29540	29730	86421
2009-08-26	29970	29970	29580	29920	81561
2009-08-27	29660	30450	29660	30150	145002
2009-08-28	30600	31400	30400	31400	218665
2009-08-31	31000	31950	30550	31650	177044
2009-09-01	31700	32350	31350	32150	175857
2009-09-02	31750	31850	31300	31750	124583
2009-09-03	31800	32000	31500	31600	86112
2009-09-04	31500	31550	30150	30250	128351
2009-09-07	31000	31000	30100	30300	73069
2009-09-08	30100	30200	29230	29500	160782
2009-09-09	29600	29760	29340	29470	85887
2009-09-10	29400	30900	29400	30850	123243
2009-09-11	30900	30900	29950	30600	140438
2009-09-14	30650	30800	30100	30650	60639
2009-09-15	30400	31300	30400	30850	130429
2009-09-16	30850	31150	29740	29840	134531
2009-09-17	29880	30050	29700	29920	83883
2009-09-18	29500	29800	29300	29440	112368
2009-09-24	29820	30250	29700	29930	117311
2009-09-25	29860	29870	29180	29430	91254
2009-09-28	29100	29940	29010	29370	173529
2009-09-29	29580	29790	29320	29470	95761
2009-09-30	29750	30500	29600	30500	143315
2009-10-01	30450	30450	29520	29640	91487
2009-10-02	29080	29170	27870	28160	140924
2009-10-05	28170	28650	27820	27890	74732
2009-10-06	27970	28070	27180	27560	123080
2009-10-07	27570	27930	27420	27820	68570
2009-10-08	27780	28030	27220	27760	139254
2009-10-09	28960	30400	28900	29870	277558
2009-10-13	30000	30200	29560	29670	133326
2009-10-14	29490	29950	29170	29270	84706
2009-10-15	29130	29470	28780	28950	110098
2009-10-16	28990	29230	28240	28700	116444
2009-10-19	28710	29070	28410	29030	72379
2009-10-20	29320	29410	28330	28740	198772
2009-10-21	29040	29680	28810	29540	147750
2009-10-22	29450	29660	28990	29530	108137
2009-10-23	29230	29890	28930	29030	123808
2009-10-26	28890	29010	28550	28850	92352
2009-10-27	28840	29790	28730	29430	167768
2009-10-28	29130	29700	28020	28020	185913
2009-10-29	27800	27980	27500	27980	236067
2009-10-30	28200	28340	27700	28030	102257
2009-11-02	27500	27980	27350	27960	86399
2009-11-04	27850	27850	27450	27760	71796
2009-11-05	27670	27800	27200	27430	101139
2009-11-06	27500	27500	27060	27290	83772
2009-11-09	27150	27390	26870	27000	91989
2009-11-10	26900	27070	26480	26610	119512
2009-11-11	26420	26500	26110	26300	134854
2009-11-12	27370	27830	26950	27500	280070
2009-11-13	27200	27470	26960	27160	120285
2009-11-16	27000	27260	26300	26440	78915
2009-11-17	26300	26540	26050	26240	66822
2009-11-18	26000	26060	25320	25770	121916
2009-11-19	25920	25950	25110	25360	113049
2009-11-20	25300	25930	25300	25910	87239
2009-11-24	26020	26140	25520	25730	88176
2009-11-25	25510	25800	25400	25670	74256
2009-11-26	25500	25950	25470	25780	60268
2009-11-27	25670	25830	25310	25320	75150
2009-11-30	25310	25910	25110	25830	89412
2009-12-01	25230	25840	25230	25800	119658
2009-12-02	25810	25810	25470	25710	149340
2009-12-03	26000	26810	25990	26610	155049
2009-12-04	26610	26800	26350	26720	117660
2009-12-07	27300	27300	26620	26880	124747
2009-12-08	27480	28500	27370	28280	300165
2009-12-09	28270	29930	28250	29670	331760
2009-12-10	29070	29190	28500	28800	147748
2009-12-11	29100	29350	28880	29180	140837
2009-12-14	29020	29120	28670	28950	79271
2009-12-15	28650	29050	28650	28940	43206
2009-12-16	28700	29290	28700	28990	93739
2009-12-17	28880	28990	28090	28120	92945
2009-12-18	28520	28610	27630	27910	149772
2009-12-21	27650	28570	27650	28070	122791
2009-12-22	27770	27880	27450	27610	112602
2009-12-24	27650	28240	27440	28060	116459
2009-12-25	28000	28680	27790	27990	81280
2009-12-28	28020	28370	28010	28010	81117
2009-12-29	28080	28210	27850	27950	50807
2009-12-30	27980	28140	27840	27840	45979
2010-01-04	27690	28320	27670	28190	51514
2010-01-05	29000	29000	28650	28860	111257
2010-01-06	29080	29140	28430	29010	97505
2010-01-07	29240	29640	29100	29480	113788
2010-01-08	29780	29780	28650	28850	120008
2010-01-12	29100	29200	28720	29090	107365
2010-01-13	28800	29200	28780	28830	81629
2010-01-14	28770	29060	28500	29010	73787
2010-01-15	29280	29570	29020	29500	85925
2010-01-18	29400	29760	29090	29160	85221
2010-01-19	29340	29340	28990	29010	52544
2010-01-20	29500	30500	29240	30250	224612
2010-01-21	30450	31400	29840	31350	231315
2010-01-22	31050	31800	30850	31550	199049
2010-01-25	31550	32250	31150	32000	183278
2010-01-26	32150	32600	31300	31350	201581
2010-01-27	32050	32850	31200	31350	224316
2010-01-28	32500	33950	31750	33800	394027
2010-01-29	33850	34750	33650	34250	244418
2010-02-01	34050	34700	33700	34600	158412
2010-02-02	34300	34350	33400	33550	234341
2010-02-03	33450	34300	33200	33750	200915
2010-02-04	34000	34050	31850	32250	261236
2010-02-05	31950	32250	31650	31850	134858
2010-02-08	32400	32900	32200	32250	158580
2010-02-09	32000	32350	31900	32200	103558
2010-02-10	32900	32900	32000	32050	82062
2010-02-12	32250	32300	31500	31550	93095
2010-02-15	31650	31750	31250	31350	54067
2010-02-16	31200	31600	31050	31250	38249
2010-02-17	31800	31850	31150	31250	64903
2010-02-18	31600	32400	31450	31750	138398
2010-02-19	31750	31800	31200	31250	61121
2010-02-22	31400	32000	31350	31950	56265
2010-02-23	32650	32950	32150	32900	111780
2010-02-24	32600	32850	32300	32450	81025
2010-02-25	32750	33300	32550	32950	145309
2010-02-26	33300	33700	32950	33250	200896
2010-03-01	33350	33500	32750	32750	55753
2010-03-02	32450	32650	32000	32500	76232
2010-03-03	32700	32950	32250	32350	81858
2010-03-04	32800	33350	32400	33100	141174
2010-03-05	33500	34300	33350	34150	212772
2010-03-08	34600	35850	34450	35750	263291
2010-03-09	35900	36150	35250	35350	143069
2010-03-10	34850	35150	34300	34750	129759
2010-03-11	35050	35100	34100	34600	95221
2010-03-12	34650	35650	34500	35550	118623
2010-03-15	35200	35400	34600	34700	79593
2010-03-16	34550	34700	34300	34450	66805
2010-03-17	34650	35500	34550	35400	98911
2010-03-18	35400	35400	34550	34600	80278
2010-03-19	34900	35500	34700	35300	121527
2010-03-23	34850	35200	34700	34800	55017
2010-03-24	34950	35950	34850	35800	133494
2010-03-25	35750	35750	34850	34850	92541
2010-03-26	34900	34950	34450	34700	94563
2010-03-29	34150	34450	33700	34150	106301
2010-03-30	34400	34800	34100	34550	127625
2010-03-31	34500	34600	33750	34050	144881
2010-04-01	34550	35550	34350	35050	179713
2010-04-02	35250	35850	35100	35400	111193
2010-04-05	35700	35750	35200	35550	62406
2010-04-06	35700	36700	35500	36450	203905
2010-04-07	36500	37050	36250	36350	104392
2010-04-08	36150	36650	36100	36350	78847
2010-04-09	36200	36800	35800	36050	90422
2010-04-12	36150	37000	36050	36600	116174
2010-04-13	36250	36450	35400	35650	134674
2010-04-14	35700	35900	35200	35350	100553
2010-04-15	35950	35950	35300	35500	86899
2010-04-16	35250	35650	34950	35100	114920
2010-04-19	34400	35500	34350	35150	122107
2010-04-20	35150	35550	34750	35000	101856
2010-04-21	35350	35350	34650	34900	89433
2010-04-22	34550	34800	34100	34650	68360
2010-04-23	34650	36150	34400	36050	202572
2010-04-26	36400	37200	36400	36950	176963
2010-04-27	37150	38500	36900	38400	292540
2010-04-28	37000	37300	34450	36000	323406
2010-04-30	36600	36800	35800	36000	162746
2010-05-06	35200	35550	34800	34900	147548
2010-05-07	34000	34650	33650	34200	151797
2010-05-10	33800	34800	33700	34550	120531
2010-05-11	35350	35550	34400	34400	119160
2010-05-12	34800	35000	34200	34400	67771
2010-05-13	35000	35150	34500	35000	66250
2010-05-14	34300	35000	34250	34600	63285
2010-05-17	34000	34450	33650	33850	76455
2010-05-18	33950	34400	33600	33750	65460
2010-05-19	33200	33500	32700	33350	83989
2010-05-20	33250	33350	32850	33050	86297
2010-05-21	31950	32300	31700	31950	106784
2010-05-24	32200	32800	32100	32250	83701
2010-05-25	31950	31950	30750	30850	124874
2010-05-26	31000	31400	30600	30800	90189
2010-05-27	30450	31800	30150	31650	101934
2010-05-28	32200	32400	31800	31950	79141
2010-05-31	31300	32050	30950	32000	132717
2010-06-01	33000	33650	32700	32800	226058
2010-06-02	32250	32550	31800	32200	151695
2010-06-03	32550	33400	32300	33100	141380
2010-06-04	33300	33650	33150	33500	67131
2010-06-07	32800	33000	31900	32250	118582
2010-06-08	32250	33600	32050	33450	108242
2010-06-09	33100	34150	32900	33650	166956
2010-06-10	33650	33750	32900	33700	106063
2010-06-11	34500	34600	33950	34100	102666
2010-06-14	34500	34750	34400	34600	40248
2010-06-15	34800	34900	33900	34050	84592
2010-06-16	34500	34600	34150	34350	67504
2010-06-17	34500	34500	34050	34350	56917
2010-06-18	34400	35100	34100	35000	96623
2010-06-21	35500	36350	35450	36350	141591
2010-06-22	35900	36250	35850	36050	56786
2010-06-23	36150	36450	35800	35950	113430
2010-06-24	35550	36400	35550	36100	72939
2010-06-25	35650	36050	35450	36000	142748
2010-06-28	36050	36100	35550	35650	78437
2010-06-29	36200	36400	36000	36150	97480
2010-06-30	35450	35850	35350	35650	82362
2010-07-01	35100	35100	34200	34500	149198
2010-07-02	35150	36100	34650	35300	120699
2010-07-05	35650	36200	35450	36000	61627
2010-07-06	35700	36750	35650	36750	115321
2010-07-07	36500	36550	35900	36450	111800
2010-07-08	36750	36950	36150	36350	111062
2010-07-09	35950	36200	35600	35700	146387
2010-07-12	35050	35450	34800	34850	77865
2010-07-13	35000	35350	34800	35100	107757
2010-07-14	35250	35550	34900	35000	89304
2010-07-15	34550	34750	33350	33500	184300
2010-07-16	32900	33550	32550	32850	132222
2010-07-20	32500	33900	32400	33550	146199
2010-07-21	33500	34150	33150	33500	119412
2010-07-22	33000	33500	32700	33150	79781
2010-07-23	33600	34100	33450	33650	99962
2010-07-26	34150	34800	34100	34750	101058
2010-07-27	35550	36000	34850	35150	170652
2010-07-28	35200	35300	34000	34100	180603
2010-07-29	34150	34500	33450	33500	100677
2010-07-30	33950	34100	33250	33250	77744
2010-08-02	33050	33250	32500	32550	120458
2010-08-03	32800	32950	32000	32300	129394
2010-08-04	32000	32150	31300	31400	147279
2010-08-05	31650	31750	31200	31450	91634
2010-08-06	31000	31200	30500	31150	140624
2010-08-09	31100	31950	30850	31950	101565
2010-08-10	31800	32100	31500	31900	98138
2010-08-11	31450	31550	30900	31100	93796
2010-08-12	30400	30750	30050	30250	129398
2010-08-13	30500	30850	30400	30600	118186
2010-08-16	29980	30450	29760	30400	75046
2010-08-17	30100	30700	30000	30600	49433
2010-08-18	30750	31000	30200	30400	81329
2010-08-19	30250	30650	30050	30600	73115
2010-08-20	30300	30650	30200	30400	45702
2010-08-23	30200	30250	29810	29940	114061
2010-08-24	29690	30150	29390	30000	130850
2010-08-25	29890	30300	29660	30000	100908
2010-08-26	30300	31850	30000	31700	229378
2010-08-27	31400	31550	30900	31250	124428
2010-08-30	31450	31750	31150	31300	65884
2010-08-31	31350	31550	30150	30250	101236
2010-09-01	29570	30650	29130	30500	172612
2010-09-02	30700	30900	30300	30650	50029
2010-09-03	30850	31200	30800	31050	66605
2010-09-06	31150	31250	30750	30950	52824
2010-09-07	30800	30950	30500	30700	49624
2010-09-08	30650	30700	29800	30000	111280
2010-09-09	30000	30250	29930	30100	64183
2010-09-10	30050	30250	29840	29960	126749
2010-09-13	30200	30600	30050	30500	50644
2010-09-14	30600	30800	30400	30450	35859
2010-09-15	30600	31100	30500	30850	88323
2010-09-16	30950	31000	30000	30100	67756
2010-09-17	29850	30100	29020	29090	241020
2010-09-21	29900	29970	29380	29610	142076
2010-09-22	29260	29500	28920	29130	112659
2010-09-24	29140	29940	29020	29490	220670
2010-09-27	29500	29590	28990	28990	135604
2010-09-28	29180	29260	29010	29050	148809
2010-09-29	29050	29580	29040	29270	126988
2010-09-30	29200	29290	28840	28840	122130
2010-10-01	28840	28950	28350	28520	106574
2010-10-04	28770	28980	27500	27700	167735
2010-10-05	27820	27960	27360	27570	150315
2010-10-06	28070	28530	27920	28520	157706
2010-10-07	30050	30050	29160	29580	260664
2010-10-08	29700	30250	29610	29920	210223
2010-10-12	30300	30350	29300	29350	156293
2010-10-13	29400	29650	28680	28770	132034
2010-10-14	29700	30650	29520	30600	247378
2010-10-15	30500	30500	29710	30300	143886
2010-10-18	30250	30300	29810	30150	96161
2010-10-19	30050	30050	29190	29490	147070
2010-10-20	29550	29550	28540	28610	186685
2010-10-21	28630	29250	28020	28120	196506
2010-10-22	28400	28400	27770	28040	198322
2010-10-25	28400	29560	28310	28810	241146
2010-10-26	28220	28600	27850	28120	159293
2010-10-27	28290	28690	28000	28550	160565
2010-10-28	28970	28970	28210	28260	214769
2010-10-29	28100	28460	27930	28160	105417
2010-11-01	27880	28090	27610	27670	95903
2010-11-02	27510	27690	27230	27380	97621
2010-11-04	27730	28260	27640	27980	97092
2010-11-05	28350	28540	28160	28240	74302
2010-11-08	28540	28850	28350	28760	77401
2010-11-09	28820	28890	28640	28710	103535
2010-11-10	28880	29140	28640	28800	91601
2010-11-11	28990	29250	28740	29210	120313
2010-11-12	29220	29470	28950	29120	120517
2010-11-15	29070	29250	28820	28880	88735
2010-11-16	28890	29700	28890	29280	162605
2010-11-17	28950	29250	28860	28920	97577
2010-11-18	29150	29390	28910	29310	98868
2010-11-19	29750	29880	29530	29640	105334
2010-11-22	29800	30050	29610	29730	92645
2010-11-24	29240	29700	29000	29550	61954
2010-11-25	29920	30700	29790	30450	136936
2010-11-26	30350	30600	30050	30050	58264
2010-11-29	30200	30550	29990	30050	56178
2010-11-30	30050	30300	29890	30000	63235
2010-12-01	29830	30750	29710	30650	97491
2010-12-02	30850	31500	30450	31300	134385
2010-12-03	31900	32300	31600	31800	107175
2010-12-06	32000	32200	31650	32200	87912
2010-12-07	31700	31900	31200	31300	85425
2010-12-08	31350	31650	31100	31250	115485
2010-12-09	31200	31350	30950	31100	55525
2010-12-10	31000	31200	30700	31000	116271
2010-12-13	31000	31400	30850	31350	38220
2010-12-14	31400	32000	31300	31850	76179
2010-12-15	32000	32300	31850	32050	125479
2010-12-16	31900	32700	31900	32500	107058
2010-12-17	32550	32950	32350	32450	103734
2010-12-20	32250	32600	32050	32150	44386
2010-12-21	32150	32500	32150	32350	44899
2010-12-22	32250	32450	31950	32100	63865
2010-12-24	31800	32050	31750	31900	66397
2010-12-27	32100	32200	31800	31850	36607
2010-12-28	31700	31850	31650	31800	25469
2010-12-29	31650	32000	31650	31950	22166
2010-12-30	31950	31950	31500	31500	30614
2011-01-04	31650	31850	31500	31700	59831
2011-01-05	31050	31400	30150	30200	160383
2011-01-06	30200	30600	30050	30250	114564
2011-01-07	30250	30550	30100	30200	77564
2011-01-11	30100	30550	29560	30450	106584
2011-01-12	30600	30900	30500	30750	69012
2011-01-13	31050	31500	30750	31450	100493
2011-01-14	31350	31700	31150	31250	84267
2011-01-17	31500	31550	31150	31250	50070
2011-01-18	31000	31450	31000	31250	59692
2011-01-19	31200	31650	31150	31600	58818
2011-01-20	31200	31750	30950	31150	118013
2011-01-21	31150	31350	30650	30950	104277
2011-01-24	31300	31450	31050	31300	64200
2011-01-25	31500	32200	31200	31950	108765
2011-01-26	32000	32700	31950	32450	196743
2011-01-27	32450	32500	31850	32200	128867
2011-01-28	32050	32200	31150	31350	124855
2011-01-31	31350	31400	30700	30950	108463
2011-02-01	31150	31300	30650	30900	104723
2011-02-02	31350	31800	31150	31250	84770
2011-02-03	31200	31300	30900	31100	67679
2011-02-04	31050	31150	30600	30700	93010
2011-02-07	31250	31250	30450	30650	98623
2011-02-08	31000	31000	30650	30850	72203
2011-02-09	30900	30950	30600	30750	64241
2011-02-10	30750	30900	30450	30850	61805
2011-02-14	30800	30850	30600	30700	74294
2011-02-15	30600	30650	30450	30550	79127
2011-02-16	30550	30900	30550	30850	83535
2011-02-17	30950	31800	30900	31450	218768
2011-02-18	31400	31450	30950	31150	80795
2011-02-21	30950	31100	30850	31000	114990
2011-02-22	30800	30950	30650	30650	74663
2011-02-23	30550	31250	30500	30950	111523
2011-02-24	30550	30750	30450	30600	102721
2011-02-25	30500	30750	30500	30700	58603
2011-02-28	30500	30800	30200	30750	116137
2011-03-01	30800	31250	30600	31150	155542
2011-03-02	30950	34600	30900	32300	784149
2011-03-03	31600	32250	31550	31800	221997
2011-03-04	31750	32100	31550	31650	124311
2011-03-07	31800	32450	31250	31950	267890
2011-03-08	33000	33650	32600	32950	296131
2011-03-09	33400	33850	33100	33200	213619
2011-03-10	32600	32800	31950	32250	136724
2011-03-11	31800	31950	31500	31600	108634
2011-03-14	29950	30900	29200	29890	287440
2011-03-15	29000	29240	25250	27400	199189
2011-03-16	29400	30300	28620	29880	282798
2011-03-17	28880	29800	28740	29450	141708
2011-03-18	29510	30100	29150	29420	72732
2011-03-22	29920	29920	29360	29690	127535
2011-03-23	29800	30350	29520	30250	110003
2011-03-24	30700	30950	30100	30600	86299
2011-03-25	30800	30950	30450	30850	70560
2011-03-28	30800	31050	30300	30450	61925
2011-03-29	29450	29880	28560	28990	278421
2011-03-30	29300	29530	29140	29390	122537
2011-03-31	29530	29760	29300	29760	73513
2011-04-01	29720	29890	29230	29270	47661
2011-04-04	29240	29590	28640	28680	128910
2011-04-05	28590	28680	28130	28330	102287
2011-04-06	28100	28170	27490	27860	164334
2011-04-07	27910	28300	27870	28070	99113
2011-04-08	27670	28600	27600	28480	73311
2011-04-11	28700	29200	28510	28820	92094
2011-04-12	28320	28380	27900	28060	121419
2011-04-13	28070	28330	28070	28250	76263
2011-04-14	27750	27970	27720	27770	103053
2011-04-15	27750	28190	27430	28040	177776
2011-04-18	28320	28400	28000	28000	80620
2011-04-19	27540	27750	27320	27330	121473
2011-04-20	27590	27780	27370	27670	84425
2011-04-21	27880	28500	27720	28280	94367
2011-04-22	28400	28910	28210	28470	85830
2011-04-25	29970	29980	29260	29430	158350
2011-04-26	29390	29700	29110	29110	105741
2011-04-27	29200	29470	29020	29090	83262
2011-04-28	29020	29610	28950	29590	109186
2011-05-02	29630	29950	29500	29930	62042
2011-05-06	29520	29520	28800	29070	146871
2011-05-09	29090	29320	28850	29250	91014
2011-05-10	28940	29180	28820	28910	49553
2011-05-11	28920	29000	28510	28560	99901
2011-05-12	28500	28930	28450	28670	72387
2011-05-13	28720	28870	28310	28570	79792
2011-05-16	28160	28160	27570	27620	92149
2011-05-17	28000	28430	27610	27700	158728
2011-05-18	27710	28230	27710	28080	73326
2011-05-19	28050	28230	27740	27910	87509
2011-05-20	27990	27990	27650	27850	64702
2011-05-23	27510	27700	26940	27060	154662
2011-05-24	26500	26750	26310	26580	204762
2011-05-25	26580	26720	26580	26700	125980
2011-05-26	27080	27500	27000	27420	141260
2011-05-27	27300	27370	26780	26840	120713
2011-05-30	26850	26880	26650	26810	81829
2011-05-31	26810	26900	26720	26890	112632
2011-06-01	26900	27640	26870	27570	236371
2011-06-02	26770	27500	26680	27240	163204
2011-06-03	27130	27420	27000	27300	128577
2011-06-06	27520	27940	27450	27880	189204
2011-06-07	27590	27650	27140	27500	102229
2011-06-08	27490	27770	27150	27260	69754
2011-06-09	27200	27400	27000	27220	63429
2011-06-10	27480	27720	27010	27100	69280
2011-06-13	26820	27210	26810	27050	51139
2011-06-14	26800	27110	26770	26960	68346
2011-06-15	26980	27170	26630	26710	123116
2011-06-16	26590	26760	26390	26420	77339
2011-06-17	26370	26550	26220	26320	97361
2011-06-20	26130	26270	26040	26160	62789
2011-06-21	26250	26670	26150	26650	69107
2011-06-22	26560	26990	26560	26850	84416
2011-06-23	26780	26790	26420	26530	64535
2011-06-24	26420	26940	26420	26820	52938
2011-06-27	26790	26800	26510	26550	45806
2011-06-28	26890	26970	26720	26760	59101
2011-06-29	26990	27740	26920	27460	117040
2011-06-30	27700	27780	27490	27600	84204
2011-07-01	27750	27930	27410	27570	95970
2011-07-04	27700	27780	27470	27540	69314
2011-07-05	27400	27540	27020	27180	77449
2011-07-06	27210	27600	27090	27180	97088
2011-07-07	27190	27460	27070	27160	72082
2011-07-08	27420	27990	27210	27970	114901
2011-07-11	27860	27970	27500	27530	74859
2011-07-12	27190	27360	26830	26970	77906
2011-07-13	26920	27360	26900	27280	50778
2011-07-14	27100	27400	27010	27280	45318
2011-07-15	27560	28060	27450	27770	156667
2011-07-19	27890	28270	27810	28130	110011
2011-07-20	28220	28380	28160	28350	118583
2011-07-21	28490	28870	28410	28830	143394
2011-07-22	28990	29330	28910	29060	134077
2011-07-25	28060	28200	27850	28000	226212
2011-07-26	27850	27900	27460	27580	120790
2011-07-27	27620	27760	27260	27380	79930
2011-07-28	27120	27340	26980	27290	89938
2011-07-29	27040	27320	27020	27320	109568
2011-08-01	27130	27410	26880	26930	145116
2011-08-02	26840	26860	26460	26690	90416
2011-08-03	26430	26470	26100	26160	85093
2011-08-04	26170	26280	25990	26000	99017
2011-08-05	25130	25380	24550	25310	186351
2011-08-08	25090	25400	24920	25050	105756
2011-08-09	24450	24450	23850	24160	211616
2011-08-10	24660	24700	24250	24400	177921
2011-08-11	23850	24040	23710	23900	223759
2011-08-12	24300	24780	24210	24610	238270
2011-08-15	24930	24970	24800	24850	133308
2011-08-16	25020	25120	24570	24610	97507
2011-08-17	24560	24640	24040	24340	147954
2011-08-18	24000	24000	23700	23800	193706
2011-08-19	23200	23230	22800	22880	211849
2011-08-22	23000	23200	22840	22880	202694
2011-08-23	22880	22890	22060	22170	231465
2011-08-24	22600	22910	22380	22420	163540
2011-08-25	22780	23160	22730	23030	187315
2011-08-26	23000	23920	22940	23840	219917
2011-08-29	24040	24200	23740	23890	140526
2011-08-30	24600	24820	24580	24600	170383
2011-08-31	24730	24920	24500	24670	130991
2011-09-01	24880	24880	24330	24380	145662
2011-09-02	23920	24320	23760	24040	123133
2011-09-05	23830	24140	23760	23880	78148
2011-09-06	23740	23750	23400	23440	90863
2011-09-07	23750	24040	23660	23880	73773
2011-09-08	24150	24590	24050	24130	91300
2011-09-09	23850	24830	23850	24500	143887
2011-09-12	24100	24310	23850	24110	73354
2011-09-13	24320	24690	24000	24640	134512
2011-09-14	24650	24680	23960	24030	126734
2011-09-15	24380	24490	24130	24320	81615
2011-09-16	24340	24400	23700	24310	119355
2011-09-20	24000	24160	23810	23890	72119
2011-09-21	23880	23930	23170	23210	106496
2011-09-22	23070	23160	22610	22750	124767
2011-09-26	22720	22720	21910	22270	164649
2011-09-27	22700	23780	22660	23780	166501
2011-09-28	24020	24120	23650	23750	82842
2011-09-29	23910	24300	23860	24220	170617
2011-09-30	24230	24290	23790	24250	81554
2011-10-03	23750	24300	23670	24190	76510
2011-10-04	23850	24930	23700	24830	143259
2011-10-05	24840	25170	24720	24840	134321
2011-10-06	25030	25860	25030	25380	159175
2011-10-07	25390	26610	25390	25820	192026
2011-10-11	25850	27120	25800	26910	203798
2011-10-12	26900	26900	26150	26580	149615
2011-10-13	26580	26620	26070	26150	141540
2011-10-14	25810	26160	24900	25300	243069
2011-10-17	25310	25840	25110	25150	160341
2011-10-18	25410	25480	24990	25110	149356
2011-10-19	25000	25100	24340	24680	157382
2011-10-20	24440	24730	24230	24460	110962
2011-10-21	24470	25030	24300	24800	109087
2011-10-24	25000	25280	24910	25200	60080
2011-10-25	25370	25410	24900	25220	93169
2011-10-26	24900	25150	24580	24910	64310
2011-10-27	24810	24920	24520	24920	55441
2011-10-28	25200	25730	25100	25380	79494
2011-10-31	25410	25780	25410	25420	119624
2011-11-01	25250	25260	24860	24950	109430
2011-11-02	24700	24710	24100	24380	105021
2011-11-04	24740	24970	24620	24880	71307
2011-11-07	24440	24660	24030	24210	119175
2011-11-08	24060	24320	23540	23650	143825
2011-11-09	24000	24760	24000	24720	125688
2011-11-10	24220	24550	23950	24500	110073
2011-11-11	24200	24240	23810	23920	75155
2011-11-14	24050	24300	23630	23750	78418
2011-11-15	23880	24050	23760	23810	44623
2011-11-16	23760	23820	23390	23420	53402
2011-11-17	23230	23330	23120	23300	40472
2011-11-18	23190	23640	23080	23400	51705
2011-11-21	23450	23690	23080	23150	47725
2011-11-22	23180	23600	23100	23240	52658
2011-11-24	23020	23120	22700	22740	49081
2011-11-25	22510	22840	22500	22560	42604
2011-11-28	22730	22840	22320	22410	59871
2011-11-29	22530	23110	22370	23050	72718
2011-11-30	23200	24500	23160	24090	188814
2011-12-01	24300	25030	24210	25000	193239
2011-12-02	25010	25450	24680	25290	128847
2011-12-05	25280	25340	24760	25200	77350
2011-12-06	25110	25450	24840	24900	79963
2011-12-07	25090	25660	24720	25660	131498
2011-12-08	25220	25410	25040	25220	99221
2011-12-09	24970	25380	24950	25030	94130
2011-12-12	25280	25390	25070	25070	85358
2011-12-13	24720	25210	24510	25120	72581
2011-12-14	24950	25110	24600	24640	61493
2011-12-15	24420	25290	24420	24500	104088
2011-12-16	24360	24570	24210	24310	58495
2011-12-19	24000	24260	23610	23900	71225
2011-12-20	23800	24130	23740	23950	47567
2011-12-21	24050	24350	24030	24230	42834
2011-12-22	24540	25030	24300	24730	81901
2011-12-26	24470	24690	24330	24520	35435
2011-12-27	24730	24800	24510	24610	30258
2011-12-28	24480	24820	24360	24690	37335
2011-12-29	24700	25070	24510	24980	47123
2011-12-30	24800	24980	24630	24790	32544
2012-01-04	25250	25460	25010	25170	70233
2012-01-05	24900	24900	24510	24560	67628
2012-01-06	24700	24720	23980	24360	95349
2012-01-10	24710	25020	24280	24700	135753
2012-01-11	24880	24970	24280	24420	127661
2012-01-12	24190	24240	23600	23770	175731
2012-01-13	23800	23920	23450	23840	118413
2012-01-16	23590	23590	23120	23230	94767
2012-01-17	23250	23470	23150	23320	89170
2012-01-18	23360	24100	23300	23710	145715
2012-01-19	24190	24580	23920	24520	134972
2012-01-20	24600	24650	24400	24550	93676
2012-01-23	24370	24580	24090	24570	79312
2012-01-24	24700	24780	24170	24440	61349
2012-01-25	24700	25200	24680	25090	120511
2012-01-26	25000	25000	24310	24390	99142
2012-01-27	24000	24250	23260	23330	257293
2012-01-30	23320	23380	22960	23030	156249
2012-01-31	23280	23430	23200	23280	105504
2012-02-01	23310	23770	23260	23760	147232
2012-02-02	24000	24380	23960	24180	126969
2012-02-03	24390	24390	23870	24220	95835
2012-02-06	24330	24540	24260	24450	83415
2012-02-07	24390	24650	24280	24640	105412
2012-02-08	24600	24800	24350	24500	159116
2012-02-09	24390	25500	24300	25310	199890
2012-02-10	25610	26160	25460	25930	173579
2012-02-13	25430	25930	25200	25670	102822
2012-02-14	25500	26050	25500	26030	62379
2012-02-15	24750	25160	23920	25070	257950
2012-02-16	25200	25300	24900	25050	90525
2012-02-17	25070	25250	24830	24900	93464
2012-02-20	25150	25170	24780	24930	96446
2012-02-21	25040	25410	24930	25230	79222
2012-02-22	25260	25550	25080	25430	60470
2012-02-23	25560	25690	25330	25680	79323
2012-02-24	25550	25640	25050	25570	100322
2012-02-27	25460	25540	25210	25320	85959
2012-02-28	25240	25890	25200	25820	81488
2012-02-29	25900	26290	25650	25700	95601
2012-03-01	25890	25900	25430	25570	75329
2012-03-02	26160	27080	26080	26940	256200
2012-03-05	26930	27030	26520	26910	140058
2012-03-06	26920	27200	26610	26920	136806
2012-03-07	26580	26660	25930	26400	122767
2012-03-08	26680	26950	26400	26680	87023
2012-03-09	26950	26960	26680	26760	108716
2012-03-12	26770	27220	26730	27120	111460
2012-03-13	26830	27550	26820	27490	144598
2012-03-14	27820	27980	27680	27750	109174
2012-03-15	27700	27750	27220	27610	93893
2012-03-16	27620	27760	27180	27290	88042
2012-03-19	27400	27670	27370	27470	63779
2012-03-21	27210	27450	27130	27240	83413
2012-03-22	27070	27330	27020	27130	105620
2012-03-23	26870	27160	26860	26950	119741
2012-03-26	26820	27030	26360	26550	111097
2012-03-27	26670	27070	26580	26980	125484
2012-03-28	27070	27070	26590	26640	68651
2012-03-29	26960	27290	26860	27220	104367
2012-03-30	27080	27080	26460	26780	121536
2012-04-02	26780	26800	26610	26640	86371
2012-04-03	26420	26430	25370	25890	189458
2012-04-04	25820	26030	25530	25590	104934
2012-04-05	25500	25860	25480	25690	74948
2012-04-06	25490	25680	25260	25460	75698
2012-04-09	25200	25450	25100	25240	51378
2012-04-10	25240	25510	25050	25360	75285
2012-04-11	25310	25470	25060	25090	83187
2012-04-12	25070	25390	25010	25250	55224
2012-04-13	25200	25410	25130	25180	76183
2012-04-16	25000	25270	24840	25210	76731
2012-04-17	25280	25860	25260	25540	81932
2012-04-18	25610	25770	24810	25130	171124
2012-04-19	25200	25300	25040	25100	124340
2012-04-20	25020	25590	25020	25080	124203
2012-04-23	25090	25560	25070	25320	99442
2012-04-24	25000	26160	24780	25760	185337
cut1
1997-11-04	245	245	245	245	2605056
1997-11-05	236	265	214	250	5177344
1997-11-06	253	268	252	252	1802240
1997-11-07	257	265	248	258	868352
1997-11-10	251	252	243	245	327680
1997-11-11	246	246	241	241	180224
1997-11-12	232	232	208	208	278528
1997-11-13	201	232	188	232	442368
1997-11-14	226	226	220	220	32768
1997-11-17	220	232	220	232	98304
1997-11-18	232	232	226	232	81920
1997-11-19	239	246	232	232	196608
1997-11-20	239	251	239	246	106496
1997-11-21	246	257	246	257	139264
1997-11-25	253	253	237	237	122880
1997-11-26	234	237	234	235	65536
1997-11-27	232	234	226	226	98304
1997-11-28	225	225	208	225	147456
1997-12-01	234	253	234	250	172032
1997-12-02	250	262	250	262	466944
1997-12-03	262	269	257	263	491520
1997-12-04	268	295	268	291	884736
1997-12-05	295	340	295	340	1302528
1997-12-08	364	389	362	367	868352
1997-12-09	353	361	348	353	212992
1997-12-10	353	355	331	355	122880
1997-12-11	359	384	355	355	163840
1997-12-15	347	348	336	342	65536
1997-12-16	337	342	337	342	24576
1997-12-17	351	352	342	348	32768
1997-12-18	344	346	344	346	24576
1997-12-19	344	344	330	330	24576
1997-12-22	329	329	324	324	16384
1997-12-24	317	324	317	322	106496
1997-12-25	330	359	330	342	73728
1997-12-26	342	348	342	348	16384
1997-12-29	342	342	333	342	24576
1997-12-30	355	367	355	365	49152
1998-01-05	367	367	365	365	24576
1998-01-06	365	365	355	362	40960
1998-01-07	362	362	355	355	65536
1998-01-08	355	355	353	353	32768
1998-01-12	351	351	342	342	16384
1998-01-13	342	342	342	342	16384
1998-01-14	344	344	331	331	32768
1998-01-16	342	353	341	353	57344
1998-01-19	358	373	355	368	122880
1998-01-20	367	367	367	367	32768
1998-01-21	355	358	355	358	16384
1998-01-22	355	361	355	356	81920
1998-01-23	356	359	355	355	32768
1998-01-26	355	361	355	358	32768
1998-01-27	361	361	355	355	65536
1998-01-28	353	353	353	353	8192
1998-01-29	341	341	330	331	32768
1998-01-30	334	336	330	336	24576
1998-02-02	336	336	336	336	8192
1998-02-03	337	342	336	336	57344
1998-02-04	334	334	330	330	16384
1998-02-05	324	324	324	324	8192
1998-02-06	330	330	318	318	90112
1998-02-09	319	319	319	319	8192
1998-02-10	319	324	319	324	16384
1998-02-12	330	330	330	330	98304
1998-02-13	329	330	329	330	24576
1998-02-16	330	330	329	330	147456
1998-02-17	330	330	319	319	32768
1998-02-18	330	340	330	340	114688
1998-02-19	342	355	342	347	106496
1998-02-20	352	383	352	383	204800
1998-02-23	395	403	376	384	204800
1998-02-24	384	384	370	370	81920
1998-02-25	367	380	367	379	65536
1998-02-26	385	386	385	385	40960
1998-02-27	391	446	391	446	335872
1998-03-02	446	488	444	488	221184
1998-03-03	480	480	464	477	147456
1998-03-04	475	475	463	473	49152
1998-03-05	474	474	451	451	73728
1998-03-10	467	483	467	483	172032
1998-03-11	495	501	489	501	180224
1998-03-12	507	507	489	489	57344
1998-03-13	495	532	495	532	90112
1998-03-16	523	532	523	532	32768
1998-03-17	538	589	538	589	73728
1998-03-18	593	593	578	579	163840
1998-03-19	574	574	562	566	65536
1998-03-20	572	574	568	574	24576
1998-03-23	574	574	574	574	8192
1998-03-24	573	573	538	550	32768
1998-03-25	550	550	544	544	57344
1998-03-26	562	568	556	556	49152
1998-03-27	556	556	550	550	49152
1998-03-30	538	539	538	538	57344
1998-03-31	550	550	549	550	32768
1998-04-01	550	550	550	550	16384
1998-04-02	557	560	557	560	16384
1998-04-03	574	617	573	610	147456
1998-04-06	647	647	641	647	65536
1998-04-07	647	658	646	646	65536
1998-04-08	645	645	645	645	8192
1998-04-09	644	644	643	644	24576
1998-04-10	668	709	668	702	114688
1998-04-13	715	732	709	728	155648
1998-04-14	728	750	728	745	122880
1998-04-15	745	748	739	739	73728
1998-04-16	744	744	744	744	8192
1998-04-17	737	737	728	733	90112
1998-04-20	733	733	728	730	57344
1998-04-21	728	728	707	707	40960
1998-04-23	622	629	622	629	65536
1998-04-24	629	629	544	569	131072
1998-04-27	575	630	575	630	90112
1998-04-28	611	611	562	562	81920
1998-04-30	608	610	599	610	40960
1998-05-01	612	671	612	671	40960
1998-05-06	634	634	611	611	49152
1998-05-08	588	588	586	586	24576
1998-05-11	588	588	586	586	40960
1998-05-12	586	586	586	586	8192
1998-05-14	588	588	586	588	24576
1998-05-18	586	599	586	599	16384
1998-05-19	599	622	599	622	57344
1998-05-20	622	622	622	622	8192
1998-05-22	623	623	599	599	16384
1998-05-26	586	586	586	586	8192
1998-05-27	586	586	586	586	8192
1998-05-28	574	574	574	574	16384
1998-05-29	568	568	562	562	16384
1998-06-01	561	562	561	562	16384
1998-06-04	538	538	538	538	24576
1998-06-05	525	525	477	477	49152
1998-06-08	463	463	416	416	106496
1998-06-09	424	424	424	424	16384
1998-06-10	495	513	489	513	90112
1998-06-11	538	538	533	533	32768
1998-06-12	534	534	519	519	49152
1998-06-15	519	519	514	514	57344
1998-06-16	513	513	513	513	65536
1998-06-17	519	525	519	525	24576
1998-06-18	539	574	539	574	57344
1998-06-19	580	580	574	574	49152
1998-06-23	593	599	593	593	49152
1998-06-24	606	606	606	606	16384
1998-06-25	606	617	599	611	73728
1998-06-26	605	605	605	605	24576
1998-06-29	611	611	599	599	16384
1998-06-30	599	599	593	593	16384
1998-07-01	599	611	599	611	49152
1998-07-02	617	635	617	635	57344
1998-07-03	635	641	635	641	32768
1998-07-06	647	654	647	654	24576
1998-07-07	673	733	673	705	106496
1998-07-08	734	770	733	770	245760
1998-07-09	770	770	745	757	114688
1998-07-10	757	763	756	757	196608
1998-07-13	757	757	757	757	8192
1998-07-15	743	743	743	743	8192
1998-07-17	730	730	730	730	8192
1998-07-21	722	722	722	722	16384
1998-07-22	722	722	702	702	24576
1998-07-23	684	702	678	702	65536
1998-07-27	660	660	660	660	24576
1998-08-03	685	685	685	685	8192
1998-08-04	685	685	685	685	8192
1998-08-06	710	710	672	672	57344
1998-08-07	647	647	647	647	8192
1998-08-10	647	647	647	647	16384
1998-08-11	635	635	635	635	8192
1998-08-12	623	623	623	623	8192
1998-08-13	623	623	623	623	8192
1998-08-17	611	611	586	586	32768
1998-08-18	574	574	574	574	16384
1998-08-19	566	577	566	575	73728
1998-08-20	574	575	573	575	49152
1998-08-21	580	586	580	585	57344
1998-08-24	585	585	585	585	8192
1998-08-27	550	550	550	550	8192
1998-08-28	490	502	489	502	98304
1998-08-31	496	500	490	490	147456
1998-09-01	485	490	477	490	98304
1998-09-02	533	534	533	534	16384
1998-09-03	545	595	544	595	40960
1998-09-04	597	623	597	612	40960
1998-09-07	612	613	612	613	16384
1998-09-08	613	613	613	613	8192
1998-09-09	623	635	623	625	32768
1998-09-10	623	623	623	623	8192
1998-09-11	613	613	586	586	24576
1998-09-16	512	527	512	527	32768
1998-09-17	539	539	539	539	8192
1998-09-18	544	562	544	562	24576
1998-09-21	545	545	545	545	8192
1998-09-22	545	545	544	544	16384
1998-09-24	555	605	555	605	24576
1998-09-30	557	557	557	557	16384
1998-10-01	550	550	544	544	16384
1998-10-02	525	525	525	525	16384
1998-10-05	525	525	525	525	8192
1998-10-06	519	519	519	519	8192
1998-10-07	519	550	519	550	40960
1998-10-08	574	586	574	574	81920
1998-10-09	574	574	574	574	16384
1998-10-12	574	574	557	557	90112
1998-10-13	545	545	519	525	114688
1998-10-14	525	525	525	525	8192
1998-10-15	525	527	525	525	73728
1998-10-16	525	525	525	525	32768
1998-10-19	532	532	532	532	8192
1998-10-21	525	525	525	525	8192
1998-10-22	527	527	527	527	8192
1998-10-23	532	538	532	538	32768
1998-10-26	538	538	538	538	16384
1998-10-27	550	562	550	562	24576
1998-10-28	562	562	562	562	8192
1998-10-30	568	568	568	568	16384
1998-11-02	568	568	568	568	8192
1998-11-04	586	593	586	593	24576
1998-11-05	611	617	611	611	57344
1998-11-06	617	647	617	647	73728
1998-11-09	647	647	647	647	16384
1998-11-10	647	649	647	647	24576
1998-11-11	696	709	695	709	57344
1998-11-12	715	738	715	733	114688
1998-11-13	734	837	734	837	409600
1998-11-16	849	916	849	879	196608
1998-11-17	904	904	818	831	98304
1998-11-18	831	831	782	794	73728
1998-11-19	818	855	818	855	49152
1998-11-20	854	854	854	854	16384
1998-11-24	859	879	859	878	65536
1998-11-25	876	876	861	861	49152
1998-11-26	862	867	861	867	49152
1998-11-27	873	873	873	873	8192
1998-11-30	875	977	875	965	229376
1998-12-01	915	915	892	892	114688
1998-12-02	916	928	892	892	32768
1998-12-03	904	904	904	904	16384
1998-12-04	867	867	849	849	32768
1998-12-07	849	855	849	855	16384
1998-12-09	843	855	843	855	40960
1998-12-10	879	879	867	878	32768
1998-12-11	879	879	879	879	8192
1998-12-15	867	867	867	867	8192
1998-12-16	873	873	873	873	32768
1998-12-17	873	892	873	876	81920
1998-12-18	873	898	867	892	73728
1998-12-21	916	953	916	953	73728
1998-12-22	992	1038	992	1026	147456
1998-12-24	1056	1056	1026	1026	24576
1998-12-25	1014	1014	1014	1014	24576
1998-12-28	1008	1008	1008	1008	24576
1998-12-29	1001	1025	1001	1025	57344
1998-12-30	1037	1037	1037	1037	24576
1999-01-04	1026	1037	1026	1032	24576
1999-01-05	1032	1050	1032	1038	49152
1999-01-06	1050	1160	1050	1160	147456
1999-01-07	1282	1282	1282	1282	73728
1999-01-08	1404	1502	1404	1465	172032
1999-01-11	1709	1709	1709	1709	73728
1999-01-12	1954	1954	1685	1685	131072
1999-01-13	1684	1685	1587	1648	172032
1999-01-14	1648	1648	1587	1612	49152
1999-01-18	1587	1833	1587	1833	122880
1999-01-19	1933	2076	1933	2076	131072
1999-01-20	2076	2076	2015	2015	196608
1999-01-21	1771	1771	1771	1771	49152
1999-01-22	1526	1526	1526	1526	139264
1999-01-25	1491	1685	1491	1648	139264
1999-01-26	1747	1771	1746	1771	40960
1999-01-27	1771	1893	1648	1709	49152
1999-01-28	1587	1587	1587	1587	8192
1999-01-29	1685	1893	1685	1807	212992
1999-02-01	1807	1893	1758	1819	172032
1999-02-02	1807	1807	1771	1807	73728
1999-02-03	1807	1807	1758	1758	40960
1999-02-04	1795	1795	1734	1783	73728
1999-02-05	1709	1758	1709	1746	73728
1999-02-08	1734	1783	1734	1783	40960
1999-02-09	1783	2161	1783	2002	172032
1999-02-10	2015	2137	1893	1966	81920
1999-02-12	2259	2320	2088	2100	57344
1999-02-15	2100	2100	2100	2100	8192
1999-02-16	2222	2222	2076	2076	24576
1999-02-17	2076	2198	2076	2137	57344
1999-02-18	2137	2137	2137	2137	24576
1999-02-19	2137	2442	2137	2161	57344
1999-02-22	2259	2442	2222	2320	90112
1999-02-23	2320	2442	2320	2320	114688
1999-02-24	2503	2503	2430	2442	24576
1999-02-25	2503	2564	2405	2442	57344
1999-02-26	2442	2686	2393	2686	65536
1999-03-01	2686	3296	2686	3052	139264
1999-03-02	3174	3296	2991	3174	131072
1999-03-03	3174	3357	2991	3174	81920
1999-03-04	3418	3418	3052	3199	98304
1999-03-05	2991	3418	2930	3418	65536
1999-03-08	3724	3724	3113	3357	98304
1999-03-09	3663	3663	3052	3077	114688
1999-03-10	3174	3296	2930	3174	122880
1999-03-11	2991	3235	2869	3052	65536
1999-03-12	2991	3052	2747	2784	98304
1999-03-15	2784	3040	2784	2808	57344
1999-03-16	3113	3418	2869	3052	106496
1999-03-17	3480	3480	3113	3235	65536
1999-03-18	3296	3418	3052	3205	90112
1999-03-19	3205	3272	3205	3272	16384
1999-03-23	3418	3785	3235	3333	262144
1999-03-24	3418	3663	3333	3418	311296
1999-03-25	3846	4273	3370	3907	311296
1999-03-26	4151	5494	3907	5005	274432
1999-03-29	5005	6470	5005	5298	290816
1999-03-30	5542	6348	5542	6055	315392
1999-03-31	6104	6226	5860	5860	126976
1999-04-01	6470	6543	5860	6324	294912
1999-04-02	6592	6959	6348	6348	303104
1999-04-05	6714	7325	6641	7154	294912
1999-04-06	7935	8790	7569	7764	524288
1999-04-07	7813	8057	7276	7813	339968
1999-04-08	8301	10254	8301	10254	851968
1999-04-09	10743	14649	10254	14649	946176
1999-04-12	14405	14649	11231	12330	241664
1999-04-13	12330	12330	8912	9888	135168
1999-04-14	9278	12330	9278	12208	299008
1999-04-15	11719	12208	10499	11475	184320
1999-04-16	11475	11963	10010	10499	106496
1999-04-19	10499	10743	8790	8790	106496
1999-04-20	7813	8790	6592	8106	249856
1999-04-21	9034	9278	8179	8619	270336
1999-04-22	9034	10010	8423	10010	151552
1999-04-23	10010	11353	10010	10523	184320
1999-04-26	10987	10987	9400	9766	81920
1999-04-27	10743	11231	10132	10132	81920
1999-04-28	10181	10303	10132	10279	36864
1999-04-30	10276	10276	10108	10108	32768
1999-05-06	9620	9620	9620	9620	28672
1999-05-07	9131	9131	9131	9131	81920
1999-05-10	8643	8643	8643	8643	20480
1999-05-11	8301	8301	8155	8155	73728
1999-05-12	8643	8643	8643	8643	40960
1999-05-13	9131	9131	9131	9131	28672
1999-05-14	9131	9278	9131	9278	40960
1999-05-17	8936	8936	8790	8790	20480
1999-05-18	8787	8790	8301	8301	28672
1999-05-19	8304	8304	8084	8179	36864
1999-05-20	8082	8375	8082	8375	147456
1999-05-21	8326	8326	7886	8033	122880
1999-05-24	8030	8033	7544	7544	36864
1999-05-25	7056	7081	7056	7081	135168
1999-05-26	6592	6592	6592	6592	45056
1999-05-27	6690	6812	6543	6543	180224
1999-05-28	6055	6106	6055	6055	208896
1999-05-31	5567	5567	5567	5567	57344
1999-06-01	5225	6055	5223	6055	159744
1999-06-02	6543	6543	6543	6543	61440
1999-06-03	7032	7032	7032	7032	32768
1999-06-04	7520	7520	7520	7520	53248
1999-06-07	8008	8008	8008	8008	20480
1999-06-08	8497	8497	8497	8497	28672
1999-06-09	8985	8985	8545	8692	217088
1999-06-10	9034	9180	9034	9180	77824
1999-06-11	9522	9571	9180	9180	65536
1999-06-14	9180	9180	8692	8692	12288
1999-06-15	8204	8204	8204	8204	4096
1999-06-16	7715	8057	7715	7715	94208
1999-06-17	8204	8204	8204	8204	28672
1999-06-18	8692	8692	8692	8692	8192
1999-06-21	9180	9180	9180	9180	40960
1999-06-22	9668	9668	9473	9595	77824
1999-06-23	9595	9595	9107	9107	57344
1999-06-24	9109	9253	8985	9107	86016
1999-06-25	9595	9595	9473	9595	36864
1999-06-28	9791	10084	9766	10084	57344
1999-06-29	10572	10572	10572	10572	28672
1999-06-30	11060	11060	11060	11060	28672
1999-07-01	11548	11548	11548	11548	20480
1999-07-02	12037	12037	12037	12037	12288
1999-07-05	12525	12525	12525	12525	77824
1999-07-06	13013	13013	13013	13013	28672
1999-07-07	13428	13428	12525	12525	81920
1999-07-08	12623	13013	12623	12916	114688
1999-07-09	12427	12427	12427	12427	16384
1999-07-12	11939	11939	11939	11939	24576
1999-07-13	11451	11597	11451	11597	81920
1999-07-14	11600	12085	11600	12085	49152
1999-07-15	12574	12574	12574	12574	4096
1999-07-19	13550	13550	13550	13550	4096
1999-07-21	14039	14039	14039	14039	73728
1999-07-22	14527	14527	14527	14527	77824
1999-07-23	15015	15015	15015	15015	57344
1999-07-26	14527	14527	14527	14527	36864
1999-07-27	14039	14039	14039	14039	12288
1999-07-28	13550	13550	13550	13550	32768
1999-07-29	13306	14039	13306	14039	131072
1999-07-30	14039	14039	13990	14039	36864
1999-08-02	14041	14041	14039	14039	20480
1999-08-03	14046	14527	14046	14527	61440
1999-08-04	14524	14524	14039	14039	24576
1999-08-05	13550	13550	13550	13550	12288
1999-08-06	13062	13062	13062	13062	45056
1999-08-09	12574	12574	12574	12574	65536
1999-08-10	12085	12085	12085	12085	28672
1999-08-11	11597	11597	11597	11597	16384
1999-08-12	11548	12085	11548	12085	57344
1999-08-13	12208	12452	12205	12208	90112
1999-08-16	12696	12696	12574	12574	73728
1999-08-17	12574	12574	12574	12574	24576
1999-08-18	12696	13062	12696	13062	45056
1999-08-19	13550	13550	13550	13550	32768
1999-08-20	14039	14039	14039	14039	40960
1999-08-23	14332	14332	13550	13550	20480
1999-08-24	13550	14039	13550	14039	53248
1999-08-25	14041	14527	13990	14527	36864
1999-08-26	14869	14893	14771	14771	49152
1999-08-27	14722	14722	14649	14649	16384
1999-08-30	14646	14700	14646	14700	20480
1999-08-31	14698	15188	14698	15188	32768
1999-09-01	15677	15677	15677	15677	40960
1999-09-02	15699	16165	15699	16165	45056
1999-09-03	16602	16653	16602	16602	32768
1999-09-06	16944	16993	16944	16968	32768
1999-09-07	17115	17457	17090	17334	77824
1999-09-08	17774	17823	17774	17774	45056
1999-09-09	17825	18262	17825	18262	32768
1999-09-10	18265	18555	18265	18555	53248
1999-09-13	18555	19043	18555	19043	57344
1999-09-14	19532	19532	19043	19043	28672
1999-09-16	18555	18555	18555	18555	12288
1999-09-17	18067	18067	18067	18067	12288
1999-09-20	17579	17579	17579	17579	53248
1999-09-21	17093	17093	17090	17090	32768
1999-09-22	16602	16602	16602	16602	4096
1999-09-24	16114	16114	16114	16114	32768
1999-09-27	17090	17090	16114	16114	94208
1999-09-28	16211	17090	16211	17090	71680
1999-09-29	18067	18067	18067	18067	24576
1999-09-30	19043	19043	19043	19043	79872
1999-10-01	20020	20020	20020	20020	24576
1999-10-04	20997	20997	20997	20997	14336
1999-10-05	21973	21973	21973	21973	24576
1999-10-06	22950	22950	22950	22950	34816
1999-10-07	23926	23926	23926	23926	16384
1999-10-08	24903	24903	24903	24903	124928
1999-10-12	25586	25586	23926	23926	122880
1999-10-13	22950	22950	22950	22950	8192
1999-10-14	21973	21973	21973	21973	16384
1999-10-15	20997	20997	20997	20997	16384
1999-10-18	20020	20020	20020	20020	6144
1999-10-19	19043	19043	19043	19043	22528
1999-10-20	20020	20020	20020	20020	83968
1999-10-21	20997	20997	20997	20997	10240
1999-10-22	21973	21973	21973	21973	4096
1999-10-26	23926	23926	23384	23829	243712
1999-10-27	24903	24903	24903	24903	67584
1999-10-28	25879	25879	25586	25879	141312
1999-10-29	26856	26856	26856	26856	81920
1999-11-01	27833	27833	27833	27833	30720
1999-11-02	28809	28809	28809	28809	47104
1999-11-04	29786	29786	29786	29786	55296
1999-11-05	30762	30762	30762	30762	32768
1999-11-08	31739	31739	31739	31739	53248
1999-11-09	32715	32715	32715	32715	32768
1999-11-10	33692	33692	33204	33692	151552
1999-11-11	34668	34668	34668	34668	63488
1999-11-12	35645	35645	35645	35645	16384
1999-11-15	36622	36622	36622	36622	53248
1999-11-16	37110	37110	35645	35889	229376
1999-11-17	35987	35987	34913	34913	59392
1999-11-18	33936	33936	33936	33936	34816
1999-11-19	32959	34913	32959	34913	139264
1999-11-22	35889	35889	34664	34913	120832
1999-11-24	34717	34815	33936	33936	94208
1999-11-25	34913	34913	34668	34668	45056
1999-11-26	34375	34522	33692	34522	47104
1999-11-29	35162	35499	35157	35450	28672
1999-11-30	36426	36426	34668	35157	40960
1999-12-01	34473	34473	34180	34180	43008
1999-12-02	34375	34375	33204	33204	16384
1999-12-03	33008	34180	32959	32959	63488
1999-12-06	33936	33936	33936	33936	4096
1999-12-07	34913	34913	34913	34913	14336
1999-12-08	35889	35889	35889	35889	12288
1999-12-09	36866	36866	36866	36866	30720
1999-12-10	37842	37842	37842	37842	14336
1999-12-13	38819	38819	38819	38819	8192
1999-12-14	39795	39795	39795	39795	36864
1999-12-15	40772	40772	40772	40772	67584
1999-12-16	40772	41749	39844	41700	157696
1999-12-17	42383	42481	40723	40723	79872
1999-12-20	40625	40625	39747	39747	102400
1999-12-21	38770	38770	38770	38770	40960
1999-12-22	39747	39747	39747	39747	18432
1999-12-24	40723	40723	40723	40723	6144
1999-12-27	41700	41700	41700	41700	30720
1999-12-28	42676	42676	42676	42676	4096
1999-12-30	44629	44629	44629	44629	8192
2000-01-04	45606	45606	45606	45606	8192
2000-01-05	45601	45606	44629	44629	94208
2000-01-06	44629	44629	43653	43653	16384
2000-01-07	42676	42676	42676	42676	16384
2000-01-11	43653	43653	43653	43653	6144
2000-01-12	44629	44629	44629	44629	55296
2000-01-13	44825	45606	44532	45606	63488
2000-01-14	46583	46583	46583	46583	55296
2000-01-17	47559	47559	47559	47559	65536
2000-01-18	48536	48536	48536	48536	24576
2000-01-19	49512	49512	48389	48389	110592
2000-01-20	46583	46583	46583	46583	26624
2000-01-21	45704	45752	45606	45606	61440
2000-01-24	45601	46583	45601	46583	53248
2000-01-25	46485	47559	45997	47364	43008
2000-01-27	49317	49317	49317	49317	10240
2000-01-28	50293	50293	50293	50293	6144
2000-01-31	51270	51270	51270	51270	4096
2000-02-01	52247	52247	52247	52247	4096
2000-02-02	53223	53223	53223	53223	6144
2000-02-03	54200	54200	54200	54200	28672
2000-02-04	55176	55176	55176	55176	86016
2000-02-07	56153	56153	56153	56153	14336
2000-02-08	57129	57129	57129	57129	18432
2000-02-09	58106	58106	58106	58106	2048
2000-02-10	59083	59083	59083	59083	30720
2000-02-14	62500	62500	62500	62500	12288
2000-02-15	65918	65918	65918	65918	16384
2000-02-16	69336	69336	69336	69336	18432
2000-02-17	72754	72754	72754	72754	22528
2000-02-18	76172	76172	76172	76172	32768
2000-02-21	79590	79590	79590	79590	47104
2000-02-22	81983	81983	76172	76172	157696
2000-02-23	74805	79590	74708	78589	122880
2000-02-24	77564	77906	75171	76661	22528
2000-02-25	73243	73243	73243	73243	10240
2000-02-28	73243	73243	69825	69825	22528
2000-02-29	69190	73243	68848	73243	73728
2000-03-01	75635	76172	75000	76172	36864
2000-03-02	73780	75196	72754	75196	14336
2000-03-03	73218	73243	71778	71778	10240
2000-03-06	68360	68360	68360	68360	10240
2000-03-07	64942	64942	64942	64942	14336
2000-03-08	61524	65650	61524	65650	45056
2000-03-09	67701	68433	67383	68360	14336
2000-03-10	65625	69043	64942	64942	30720
2000-03-13	61524	61524	61524	61524	16384
2000-03-14	58106	58106	58106	58106	6144
2000-03-15	54688	61524	54688	61524	75776
2000-03-16	62208	64258	58350	58594	53248
2000-03-17	60645	61329	56153	56153	79872
2000-03-21	55127	59571	55127	59571	47104
2000-03-22	62989	62989	62989	62989	28672
2000-03-23	66382	66407	64942	65625	18432
2000-03-24	65625	65625	62208	62208	14336
2000-03-27	58790	58790	58790	58790	30720
2000-03-28	51954	54004	51954	54004	112640
2000-03-29	60840	60840	60840	60840	13312
2000-03-30	67676	67676	64063	67676	141312
2000-03-31	62891	62891	60840	60840	30720
2000-04-03	54004	54004	54004	54004	22528
2000-04-04	47168	47168	47168	47168	14336
2000-04-05	42432	52051	42432	48829	175104
2000-04-06	48145	48633	42579	44825	41984
2000-04-07	41895	48829	41504	48829	70656
2000-04-10	53711	55665	53711	55372	68608
2000-04-11	51270	51954	51270	51758	20480
2000-04-12	47364	49122	46875	48780	43008
2000-04-13	48780	48780	43897	43897	50176
2000-04-14	39014	39014	39014	39014	20480
2000-04-17	34131	34131	34131	34131	21504
2000-04-18	29249	29249	29249	29249	19456
2000-04-19	31202	31202	30372	31202	188416
2000-04-20	36084	36084	36084	36084	79872
2000-04-21	40967	40967	40967	40967	2048
2000-04-24	45850	45850	45850	45850	10240
2000-04-25	50733	50733	50733	50733	37888
2000-04-26	57569	57569	57569	57569	56320
2000-04-27	58252	61524	55567	58106	249856
2000-04-28	56055	56055	51270	52735	61440
2000-05-01	49317	50391	48340	49805	76800
2000-05-02	53711	53711	51661	52442	46080
2000-05-08	52442	52442	49317	49805	14336
2000-05-09	46973	46973	45118	45118	27648
2000-05-10	41993	43458	41504	42969	43008
2000-05-11	38086	39063	38086	38086	48128
2000-05-12	42969	42969	41993	42969	25600
2000-05-15	42920	42920	40040	40040	13312
2000-05-16	40528	41504	38086	38184	52224
2000-05-17	42090	42090	37647	37696	56320
2000-05-18	37696	39063	33204	38575	49152
2000-05-19	33692	36133	33692	36133	81920
2000-05-22	31250	31250	31250	31250	24576
2000-05-23	29786	31153	26368	26368	186368
2000-05-24	24415	26563	24415	24903	221184
2000-05-25	26856	26856	26709	26856	142336
2000-05-26	27833	28809	27344	28809	94208
2000-05-29	30762	30762	30762	30762	9216
2000-05-30	35645	35645	35645	35645	67584
2000-05-31	36622	39063	31250	34571	134144
2000-06-01	32227	37110	32227	36719	40960
2000-06-02	38086	38086	36231	37891	50176
2000-06-05	42286	42774	41797	42774	55296
2000-06-06	42286	45899	41016	45020	82944
2000-06-07	44043	44043	41211	42481	36864
2000-06-08	43116	43116	40040	40821	24576
2000-06-09	40528	40918	40528	40528	19456
2000-06-12	40528	42383	40528	41993	25600
2000-06-13	42481	42969	41993	42627	45056
2000-06-14	43067	44922	43067	44922	39936
2000-06-15	42432	43262	41993	42823	41984
2000-06-16	41944	41993	41504	41993	24576
2000-06-19	42481	42969	42481	42969	12288
2000-06-20	44825	45899	44434	45899	24576
2000-06-21	46387	48829	46192	48438	54272
2000-06-22	48389	48389	43946	43946	48128
2000-06-23	40040	41993	40040	41993	29696
2000-06-26	40040	40528	38965	38965	24576
2000-06-27	37012	38965	37012	38086	19456
2000-06-28	38038	39063	38038	39063	11264
2000-06-29	40528	41016	39161	40528	26624
2000-06-30	40577	41993	40235	41065	23552
2000-07-03	41504	41504	40040	40040	12288
2000-07-04	40040	40430	38086	39063	22528
2000-07-05	39014	39014	37110	37110	10240
2000-07-06	34180	34668	32764	32764	37888
2000-07-07	33838	37110	33838	37110	21504
2000-07-10	39063	39063	38086	38086	17408
2000-07-11	34668	36329	34278	36329	25600
2000-07-12	37647	38086	32715	33301	84992
2000-07-13	35108	37110	34668	37110	47104
2000-07-14	36622	37110	36133	36133	14336
2000-07-17	35157	35547	34180	34180	58368
2000-07-18	33204	33204	31250	32715	53248
2000-07-19	32715	32715	30762	32520	48128
2000-07-21	33008	33692	31739	31836	27648
2000-07-24	31055	31055	29786	30274	41984
2000-07-25	30274	30665	28907	30274	69632
2000-07-26	30274	32227	30274	30372	38912
2000-07-27	28321	28321	27149	27540	47104
2000-07-28	25586	25879	25586	25586	126976
2000-07-31	23633	23633	23633	23633	50176
2000-08-01	25586	25586	25586	25586	54272
2000-08-02	27540	27540	27149	27393	132096
2000-08-03	28174	28174	26075	27051	62464
2000-08-04	27002	27002	25684	25879	50176
2000-08-07	25342	27344	25342	27295	53248
2000-08-08	28321	29249	28321	29249	37888
2000-08-09	31202	31202	31202	31202	34816
2000-08-10	36084	36084	34180	36084	128000
2000-08-11	34180	34668	32227	33692	73728
2000-08-14	34180	35645	34180	35010	9216
2000-08-15	35059	36622	34180	34961	73728
2000-08-16	34961	35157	34668	34961	33792
2000-08-17	33692	33692	32422	33643	8192
2000-08-18	31690	32178	30274	31153	31744
2000-08-21	31153	32325	31153	32129	25600
2000-08-22	32032	32032	30909	31739	13312
2000-08-23	31739	31885	31690	31885	15360
2000-08-24	32032	33204	32032	32862	76800
2000-08-25	33838	34668	33692	34473	100352
2000-08-28	34961	36133	34180	34180	63488
2000-08-29	29786	30274	29297	29297	81920
2000-08-30	28809	31641	28809	30762	53248
2000-08-31	30762	32032	30762	31250	13312
2000-09-01	32715	33106	31250	31739	39936
2000-09-04	31250	31250	28809	29493	28672
2000-09-05	30469	31641	28418	28614	73728
2000-09-06	28614	29786	28028	28711	43008
2000-09-07	27735	28711	27735	27930	44032
2000-09-08	27540	29297	27540	28223	38912
2000-09-11	27637	28028	26856	27833	47104
2000-09-12	27637	27637	26905	27344	26624
2000-09-13	27442	28418	27442	27637	25600
2000-09-14	28223	28223	27344	27393	14336
2000-09-18	25440	25879	25440	25440	57344
2000-09-19	23487	23975	23487	23975	119808
2000-09-20	24415	25879	24415	25782	53248
2000-09-21	25928	26856	25196	25879	70656
2000-09-22	24903	24903	24024	24415	45056
2000-09-25	24463	25391	24463	24952	53248
2000-09-26	25391	28809	25391	28516	46592
2000-09-27	28516	28516	26758	27344	32768
2000-09-28	25489	27149	25489	26465	25600
2000-09-29	27344	27735	25782	26172	46080
2000-10-02	25000	26368	25000	26368	21504
2000-10-03	26368	26368	24219	24805	55296
2000-10-04	23243	25293	22852	25293	52224
2000-10-05	25293	25586	24512	25586	31232
2000-10-06	24415	25391	24415	24805	29184
2000-10-10	24512	24610	23633	24317	45056
2000-10-11	23633	24415	23438	24024	42496
2000-10-12	23340	23438	21875	23145	50688
2000-10-13	21192	24415	21192	24122	89088
2000-10-16	23731	24317	23047	23926	73728
2000-10-17	23633	23731	21875	22461	46592
2000-10-18	18555	19922	18555	19336	158208
2000-10-19	20313	21290	20118	21290	73216
2000-10-20	24024	24415	22852	23633	48640
2000-10-23	23829	23829	22071	22071	22016
2000-10-24	21387	22461	21387	21875	23040
2000-10-25	21485	21875	20313	20313	22528
2000-10-26	19532	20313	19336	20118	27136
2000-10-27	20704	21290	20508	20508	16896
2000-10-30	20508	20508	19727	19922	14336
2000-10-31	19629	19629	18165	18360	71168
2000-11-01	20313	20313	20313	20313	27648
2000-11-02	21094	22266	20704	21875	39424
2000-11-06	22461	24805	22461	24805	72704
2000-11-07	24415	24415	23047	24415	26624
2000-11-08	24512	25977	24415	24903	50688
2000-11-09	24317	24903	23243	23438	32768
2000-11-10	22461	23047	22071	22071	32768
2000-11-13	19336	20411	19141	20118	63488
2000-11-14	20508	21387	20508	21290	30720
2000-11-15	22852	23438	22559	22559	26624
2000-11-16	21875	21875	20704	21290	10240
2000-11-17	21290	21290	20411	20411	33792
2000-11-20	20411	20606	18750	19141	71680
2000-11-21	17579	18360	17403	17422	185344
2000-11-22	16250	16270	15469	15469	153600
2000-11-24	13516	13516	13516	13516	113152
2000-11-27	13692	14258	12696	13184	450560
2000-11-28	12696	15137	11758	15137	237568
2000-11-29	13965	14063	13379	13672	156160
2000-11-30	13477	13965	12696	13282	119296
2000-12-01	12774	14043	12774	13965	155648
2000-12-04	14454	15469	14336	15352	147968
2000-12-05	15547	15723	14981	15391	96256
2000-12-06	17344	17344	17344	17344	118784
2000-12-07	15977	16797	15625	15625	150528
2000-12-08	15235	16524	14747	16524	119296
2000-12-11	17051	17481	16836	17266	113664
2000-12-12	17188	17188	15723	16309	84992
2000-12-13	16290	16583	15254	16563	91648
2000-12-14	15606	16114	15528	15528	47616
2000-12-15	14942	15040	14180	14239	65024
2000-12-18	13848	13848	13047	13770	115200
2000-12-19	13184	13184	12500	12637	105472
2000-12-20	11153	12305	11153	11543	116736
2000-12-21	10743	10938	10000	10157	140288
2000-12-22	10958	11641	10958	11524	59392
2000-12-25	12696	13047	12500	12657	41472
2000-12-26	12696	12696	12129	12403	38400
2000-12-27	12696	13379	12579	13243	92672
2000-12-28	12715	13379	12696	13086	40960
2000-12-29	13282	13360	13125	13125	25088
2001-01-04	13907	13907	12930	13204	66048
2001-01-05	13204	13204	12422	12696	48640
2001-01-09	11719	11739	10743	10743	96768
2001-01-10	11055	11524	10899	11231	63488
2001-01-11	10723	10723	9297	10235	90112
2001-01-12	10176	10430	9532	9942	197120
2001-01-15	9747	10333	9668	9961	297984
2001-01-16	10352	11915	10313	11915	250880
2001-01-17	11583	12110	10938	11875	205312
2001-01-18	12852	13672	12696	13379	299520
2001-01-19	14180	15333	14083	15333	237056
2001-01-22	14551	15137	13379	13555	237568
2001-01-23	13262	13692	12852	13458	75776
2001-01-24	14043	14942	13848	14688	181760
2001-01-25	15860	16641	15821	16641	241152
2001-01-26	16250	16700	16016	16407	297984
2001-01-29	15977	16504	15821	15840	70144
2001-01-30	16407	16602	16075	16504	95232
2001-01-31	16895	16895	16368	16758	71680
2001-02-01	16407	16797	15918	16583	74240
2001-02-02	16641	16856	16407	16680	82944
2001-02-05	15625	15938	15079	15098	77312
2001-02-06	15137	16016	15040	15918	66560
2001-02-07	15723	16211	15625	15958	98816
2001-02-08	15899	15918	15118	15235	132608
2001-02-09	15020	15821	14844	15606	76288
2001-02-13	15508	15528	14551	14629	83968
2001-02-14	14161	14161	13692	13750	89088
2001-02-15	13672	13965	13497	13711	98304
2001-02-16	14239	14258	13653	13692	99840
2001-02-19	13653	13692	12911	13477	72192
2001-02-20	13633	13731	13340	13575	62976
2001-02-21	12989	13243	12696	12891	55808
2001-02-22	12110	12305	11700	12129	87552
2001-02-23	11934	13731	11817	13672	82944
2001-02-26	13575	13653	12891	12969	31744
2001-02-27	13125	13321	13086	13086	30208
2001-02-28	12891	13086	12422	12579	40960
2001-03-01	12305	12891	12188	12500	30720
2001-03-02	12696	12989	12305	12403	39424
2001-03-05	12208	12208	11036	11934	200192
2001-03-06	12500	12891	12149	12833	103424
2001-03-07	13672	13692	12676	12793	83968
2001-03-08	10840	11426	10840	11231	662528
2001-03-09	10840	11602	10743	11133	223232
2001-03-12	10723	11055	10704	11055	57344
2001-03-13	9961	10059	9473	9805	221696
2001-03-14	10196	10450	9610	9629	108544
2001-03-15	9180	9766	9122	9766	204288
2001-03-16	9961	10547	9961	10391	177664
2001-03-19	10000	10918	10000	10840	118784
2001-03-21	10450	10821	10450	10821	71680
2001-03-22	10723	10723	10059	10157	73728
2001-03-23	10625	11114	10567	11114	187392
2001-03-26	11543	11817	11387	11700	144896
2001-03-27	11895	11895	11036	11309	68096
2001-03-28	11817	11836	11329	11504	74752
2001-03-29	10938	11133	10918	10958	42496
2001-03-30	10938	11133	10254	10391	90112
2001-04-02	10391	10391	10059	10098	84480
2001-04-03	9903	10157	9825	10000	90624
2001-04-04	9805	9805	9571	9668	193536
2001-04-05	10547	10567	10059	10567	112640
2001-04-06	11543	11563	11036	11172	237568
2001-04-09	10743	10743	10098	10176	62464
2001-04-10	10254	10450	10098	10176	47104
2001-04-11	10391	10489	10196	10411	95232
2001-04-12	10723	10840	10508	10665	84480
2001-04-13	10840	10899	10352	10450	86528
2001-04-16	10352	10352	9961	10098	43520
2001-04-17	9903	10079	9864	9961	39936
2001-04-18	10157	10430	10157	10333	68608
2001-04-19	10938	11075	10801	10977	223744
2001-04-20	11114	11250	10821	10958	93184
2001-04-23	9786	10059	9747	9766	281600
2001-04-24	9473	9571	9083	9356	338944
2001-04-25	9356	9688	9356	9571	81408
2001-04-26	9473	9590	9141	9278	135168
2001-04-27	9317	9532	9297	9473	75264
2001-05-01	9766	9903	9727	9766	71680
2001-05-02	9961	9961	9688	9922	81408
2001-05-07	10020	10313	9883	10254	112640
2001-05-08	10450	10684	10313	10450	178176
2001-05-09	10547	10645	10333	10352	94720
2001-05-10	10157	10215	9961	10098	32768
2001-05-11	9961	9981	9883	9903	36864
2001-05-14	9864	9864	9688	9688	40448
2001-05-15	9590	9747	9571	9688	99840
2001-05-16	9727	9727	9571	9571	41984
2001-05-17	9766	9903	9493	9571	72704
2001-05-18	9571	9668	9493	9532	50688
2001-05-21	9512	9610	9493	9512	20480
2001-05-22	9766	9766	9610	9708	80384
2001-05-23	9766	9766	9512	9532	38912
2001-05-24	9473	9493	9278	9297	56832
2001-05-25	9317	9395	9083	9395	64000
2001-05-28	9200	9200	8985	9004	64000
2001-05-29	8946	8946	8594	8926	92672
2001-05-30	8653	8790	8594	8672	51712
2001-05-31	8379	8438	7754	7891	303616
2001-06-01	7911	7989	7461	7540	192512
2001-06-04	7540	7540	7032	7403	116224
2001-06-05	7403	8301	7247	8282	126464
2001-06-06	8399	8458	7872	7872	138752
2001-06-07	7657	7989	7520	7989	119808
2001-06-08	8145	8165	7950	8047	56320
2001-06-11	7950	7950	7696	7774	45568
2001-06-12	7676	7676	7500	7520	56832
2001-06-13	7618	7637	7540	7618	24576
2001-06-14	7559	7637	7500	7618	36352
2001-06-15	7579	7579	7364	7481	53248
2001-06-18	7422	7481	7227	7227	35328
2001-06-19	8028	8125	7598	7657	165888
2001-06-20	7715	7715	7461	7481	139264
2001-06-21	7872	8458	7833	8458	262144
2001-06-22	8848	9375	8790	8907	393728
2001-06-25	8985	9043	8438	8692	78336
2001-06-26	8594	8750	8418	8575	62464
2001-06-27	8770	8770	8458	8497	70144
2001-06-28	8477	8477	7833	8008	92160
2001-06-29	8106	8516	8086	8360	87552
2001-07-02	8399	8399	8125	8282	49152
2001-07-03	8262	8438	8165	8223	33792
2001-07-04	7930	8047	7715	7754	77824
2001-07-05	7715	7774	7520	7618	56832
2001-07-06	7422	7579	7403	7579	48640
2001-07-09	6895	7110	6856	6934	108544
2001-07-10	6954	7071	6758	6817	79872
2001-07-11	6641	6680	6485	6504	53248
2001-07-12	7090	7481	6934	7481	103424
2001-07-13	7618	7754	7403	7559	132608
2001-07-16	7481	7481	6993	7188	43520
2001-07-17	6836	6895	6641	6797	108544
2001-07-18	6817	6915	6543	6622	52224
2001-07-19	6504	6778	6485	6778	58880
2001-07-23	6778	6778	6407	6583	52736
2001-07-24	6563	6758	6485	6739	45056
2001-07-25	6465	6934	6465	6739	48640
2001-07-26	6661	6875	6583	6739	27648
2001-07-27	6778	6856	6700	6739	21504
2001-07-30	6778	6778	6446	6602	56832
2001-07-31	6543	6934	6543	6856	59392
2001-08-01	6954	7051	6739	6778	34816
2001-08-02	6915	7032	6856	6993	54784
2001-08-03	7071	7618	6934	7403	98304
2001-08-06	7383	7422	7012	7110	31744
2001-08-07	7090	7090	6739	6739	73728
2001-08-08	6797	6856	6641	6661	40960
2001-08-09	6543	6719	6524	6563	40448
2001-08-10	6583	6700	6543	6641	23040
2001-08-13	6641	6641	6446	6446	26624
2001-08-14	6524	6719	6504	6602	26112
2001-08-15	6465	6524	6465	6485	17920
2001-08-16	6446	6465	6290	6348	43008
2001-08-17	6368	6387	6192	6231	38912
2001-08-20	6055	6055	5860	5879	55296
2001-08-21	5879	5997	5840	5860	53760
2001-08-22	5762	5762	5567	5567	93184
2001-08-23	5606	5645	5176	5176	100352
2001-08-24	4844	4981	4786	4844	157696
2001-08-27	5020	5176	5000	5040	51200
2001-08-28	4942	4942	4649	4805	93696
2001-08-29	4610	4708	4551	4610	40960
2001-08-30	4493	4766	4434	4727	55296
2001-08-31	4375	4493	4336	4336	130048
2001-09-03	3946	3946	3946	3946	25600
2001-09-04	3555	3907	3555	3907	257536
2001-09-05	3672	4297	3672	4004	346624
2001-09-06	4043	4395	3965	4395	258560
2001-09-07	4395	4786	4336	4786	320512
2001-09-10	4786	5176	4395	5137	268288
2001-09-11	5293	5469	5274	5372	189440
2001-09-12	4981	4981	4981	4981	28672
2001-09-13	4786	4786	4786	4786	31232
2001-09-14	4590	4903	4590	4786	245760
2001-09-17	4551	4551	4395	4395	74752
2001-09-18	4434	4786	4434	4786	181248
2001-09-19	4883	5098	4747	4747	138240
2001-09-20	4668	4668	4493	4532	101888
2001-09-21	4258	4336	4219	4258	110592
2001-09-25	4493	4571	4336	4336	56832
2001-09-26	4356	4415	4336	4356	18944
2001-09-27	4317	4336	4102	4141	116224
2001-09-28	4297	4356	4258	4297	80896
2001-10-01	4336	4688	4278	4512	112128
2001-10-02	4551	4903	4551	4844	123904
2001-10-03	5000	5059	4844	4883	122880
2001-10-04	5079	5098	4903	4942	55808
2001-10-05	4981	5020	4903	4942	87552
2001-10-09	4903	4903	4708	4727	31232
2001-10-10	4688	4883	4571	4747	32768
2001-10-11	4922	5137	4922	5137	117248
2001-10-12	5528	5528	5528	5528	60928
2001-10-15	5508	5918	5391	5918	218112
2001-10-16	6055	6856	6036	6504	373760
2001-10-17	6856	7481	6856	7481	283648
2001-10-18	7500	8243	6934	7227	381440
2001-10-19	7715	8008	7247	7911	216064
2001-10-22	8204	8360	7618	7813	320000
2001-10-23	7911	7930	6836	7090	411648
2001-10-24	6993	7247	6446	7051	237568
2001-10-25	7247	7520	7071	7422	183296
2001-10-26	7559	7637	7325	7364	120832
2001-10-29	7266	7364	6934	7227	99840
2001-10-30	6934	7051	6778	6856	149504
2001-10-31	6680	6895	6661	6661	78848
2001-11-01	6856	7051	6797	6954	79360
2001-11-02	7227	7325	6895	6934	65024
2001-11-05	6895	6934	6739	6817	31232
2001-11-06	7071	7325	6934	7325	117760
2001-11-07	7520	7540	7188	7188	136704
2001-11-08	7090	7129	6895	6993	42496
2001-11-09	6895	6954	6797	6817	59392
2001-11-12	6836	6973	6817	6895	22528
2001-11-13	6875	6875	6739	6758	52736
2001-11-14	6954	7012	6856	6993	78336
2001-11-15	7188	7559	7129	7500	244224
2001-11-16	7598	8204	7344	8008	431616
2001-11-19	8204	8399	8145	8165	169984
2001-11-20	8204	8438	8145	8145	248832
2001-11-21	7891	8047	7715	7891	116224
2001-11-22	7813	7911	7735	7813	51712
2001-11-26	8028	8321	8028	8243	97792
2001-11-27	8438	9219	8399	9004	490496
2001-11-28	8965	9805	8750	9434	438784
2001-11-29	9141	9317	8594	8790	275968
2001-11-30	9083	9200	8829	8829	91136
2001-12-03	8868	8887	8301	8497	76288
2001-12-04	8555	8692	8360	8594	64000
2001-12-05	8887	9083	8711	8926	114176
2001-12-06	9375	9708	9297	9336	280064
2001-12-07	9727	10040	9610	9688	228864
2001-12-10	9532	9590	9102	9141	69632
2001-12-11	9083	9122	8809	9083	82432
2001-12-12	9161	9766	9161	9747	192512
2001-12-13	9708	9883	9532	9688	135680
2001-12-14	9297	9512	9219	9473	122880
2001-12-17	9571	9590	9219	9473	76288
2001-12-18	9473	9473	8497	8497	210432
2001-12-19	8399	8790	7833	7989	238080
2001-12-20	7833	7891	7129	7422	201216
2001-12-21	7227	7559	7188	7227	107008
2001-12-25	7325	7540	7149	7520	49152
2001-12-26	7715	8106	7618	7657	117760
2001-12-27	8106	8106	7872	7969	76288
2001-12-28	8106	8165	7969	7969	57856
2002-01-04	8223	8458	8145	8399	92160
2002-01-07	8497	8633	8477	8594	88064
2002-01-08	8614	9512	8497	9375	247296
2002-01-09	9571	9766	9415	9708	212480
2002-01-10	9649	9649	8887	9004	116224
2002-01-11	8790	9258	8692	9219	97792
2002-01-15	8965	9473	8965	9141	200704
2002-01-16	9180	9610	9180	9610	226304
2002-01-17	9454	9551	9317	9395	117760
2002-01-18	9395	9727	9317	9649	207360
2002-01-21	9473	9629	9180	9180	148480
2002-01-22	8692	8868	8204	8204	301056
2002-01-23	7618	7911	7579	7618	291328
2002-01-24	7813	7969	7481	7520	128512
2002-01-25	7735	7950	7540	7676	98816
2002-01-28	7735	7852	7520	7657	69120
2002-01-29	7559	7657	7520	7637	81920
2002-01-30	7579	7813	7520	7657	61952
2002-01-31	7813	8594	7813	8594	268800
2002-02-01	9083	9141	8790	9102	360960
2002-02-04	9043	9043	8692	8711	91136
2002-02-05	8223	8399	8125	8184	131584
2002-02-06	8086	8770	8086	8731	117760
2002-02-07	8633	8985	8497	8985	235520
2002-02-08	8692	8907	8418	8809	186368
2002-02-12	8711	8887	8438	8516	178688
2002-02-13	8594	9004	8594	8985	106496
2002-02-14	8887	9454	8887	9434	226816
2002-02-15	9278	9336	8926	9180	133632
2002-02-18	8887	9102	8848	9024	56320
2002-02-19	9083	9161	8868	9063	32256
2002-02-20	8907	9043	8790	8985	30208
2002-02-21	8985	9512	8985	9493	199680
2002-02-22	9512	10059	9512	9961	502784
2002-02-25	10118	10450	9903	10352	279040
2002-02-26	10352	10352	9375	9786	158208
2002-02-27	9844	10606	9844	10274	206336
2002-02-28	10469	10801	10157	10293	191488
2002-03-01	10625	10782	10450	10723	232448
2002-03-04	11114	11290	10254	10313	257536
2002-03-05	10508	10567	10274	10508	106496
2002-03-06	10313	10333	9981	10000	128000
2002-03-07	10137	10450	10000	10450	135168
2002-03-08	10782	11329	10782	11250	296960
2002-03-11	11543	11778	11036	11622	292352
2002-03-12	11602	11700	11133	11270	79872
2002-03-13	10879	11133	10586	10743	142848
2002-03-14	10352	10547	10059	10137	246272
2002-03-15	10391	10489	10176	10293	100352
2002-03-18	10430	10508	10313	10333	69632
2002-03-19	10372	10704	10372	10567	83968
2002-03-20	10762	10860	10606	10645	120832
2002-03-22	10567	10567	10118	10137	123904
2002-03-25	10137	10313	9961	10293	93184
2002-03-26	11055	11055	10547	11016	117504
2002-03-27	11094	11680	10782	11602	166400
2002-03-28	11758	12383	11758	12383	190976
2002-03-29	13086	13868	12422	12500	294144
2002-04-01	12579	13477	12579	12774	100096
2002-04-02	13086	13165	12813	12852	67840
2002-04-03	12500	12735	12422	12618	117504
2002-04-04	12540	13008	12540	12852	81152
2002-04-05	12813	13477	12813	13438	156160
2002-04-08	13438	13555	12930	12969	100608
2002-04-09	13047	13165	12422	12774	90624
2002-04-10	12422	12774	12383	12657	47872
2002-04-11	13047	13360	12969	13204	98816
2002-04-12	12618	12852	12579	12735	58880
2002-04-15	12579	12735	12422	12461	44800
2002-04-16	12461	12500	12149	12383	50432
2002-04-17	12540	12657	11875	12188	98816
2002-04-18	11993	12188	11954	12188	44288
2002-04-19	12032	12110	11836	12032	47872
2002-04-22	11758	11836	11407	11602	100608
2002-04-23	11368	11407	11211	11250	88832
2002-04-24	11133	12032	10782	12032	163584
2002-04-25	11836	12188	11172	11250	140544
2002-04-26	11172	11641	11016	11446	131584
2002-04-30	11602	11836	11329	11485	126208
2002-05-01	11602	11602	11211	11211	55296
2002-05-02	11290	11407	11094	11211	61440
2002-05-07	10938	11446	10899	11133	71936
2002-05-08	11094	11407	11094	11211	24320
2002-05-09	11602	11797	11329	11407	64256
2002-05-10	11250	11250	11094	11172	33792
2002-05-13	10743	10860	10586	10665	85504
2002-05-14	10469	10469	9883	9883	168192
2002-05-15	9805	9961	9219	9766	211200
2002-05-16	10000	10508	10000	10196	142336
2002-05-17	10430	10704	10430	10430	121600
2002-05-20	10547	10899	10508	10704	132608
2002-05-21	10704	10704	10313	10352	93440
2002-05-22	10157	10430	10079	10235	61952
2002-05-23	10391	10430	10196	10274	62464
2002-05-24	10274	10274	10000	10040	82176
2002-05-27	10079	10586	10000	10469	68352
2002-05-28	10508	10821	10508	10782	101888
2002-05-29	10743	11563	10665	11368	326400
2002-05-30	11290	11368	10977	11172	80384
2002-05-31	10977	11055	10743	10899	76800
2002-06-03	10782	11133	10782	11133	54784
2002-06-04	11055	11446	10977	11055	76800
2002-06-05	11055	11211	10938	11055	25856
2002-06-06	11055	11133	10743	10860	32256
2002-06-07	10391	10508	10157	10235	146176
2002-06-10	10235	10391	10079	10118	80640
2002-06-11	10079	10118	9844	9922	113408
2002-06-12	9766	9844	9727	9805	114176
2002-06-13	9766	10157	9727	10157	113664
2002-06-14	10157	10235	9766	9805	145664
2002-06-17	9727	9727	9375	9454	70144
2002-06-18	9688	9727	9375	9454	81664
2002-06-19	9415	9493	8672	8907	154624
2002-06-20	8594	9297	8282	9297	169472
2002-06-21	9141	9258	8829	9102	59392
2002-06-24	8711	9532	8711	9415	75008
2002-06-25	9493	9688	9336	9415	48384
2002-06-26	9336	9415	9063	9063	41472
2002-06-27	9297	9610	9180	9375	111104
2002-06-28	9688	10079	9688	10079	140288
2002-07-01	10157	10352	10000	10274	76800
2002-07-02	10000	10040	9883	10000	48640
2002-07-03	9922	10352	9883	10274	57856
2002-07-04	10352	10391	10079	10196	46592
2002-07-05	10235	10274	10040	10118	31232
2002-07-08	10469	10547	10079	10079	81152
2002-07-09	10118	10352	10079	10313	34816
2002-07-10	10118	10157	9922	9922	49152
2002-07-11	9961	9961	9766	9844	30464
2002-07-12	10040	10040	9844	10040	38912
2002-07-15	10079	10079	9883	9922	26624
2002-07-16	9961	10118	9961	10000	42240
2002-07-17	10079	10079	9415	9571	69632
2002-07-18	9688	9844	9532	9532	27904
2002-07-19	9454	9571	9336	9415	64768
2002-07-22	9297	10157	9258	9766	115200
2002-07-23	9922	10391	9883	10352	119552
2002-07-24	10352	10391	10079	10274	62976
2002-07-25	10391	10430	10235	10313	50944
2002-07-26	10274	10274	9961	10274	54784
2002-07-29	10274	10352	10196	10196	63744
2002-07-30	10274	10313	10196	10196	22528
2002-07-31	10196	10469	10118	10469	48128
2002-08-01	10469	10665	10469	10586	98816
2002-08-02	10547	10547	10391	10547	47360
2002-08-05	10547	10665	10430	10586	31488
2002-08-06	10547	10547	10000	10313	128768
2002-08-07	10547	11094	10547	11094	229376
2002-08-08	11172	11290	10743	10782	78848
2002-08-09	10821	10938	10704	10821	58624
2002-08-12	10899	10938	10743	10860	34304
2002-08-13	10977	11016	10860	10899	30464
2002-08-14	10899	10899	10547	10860	38656
2002-08-15	10977	11211	10977	11133	65536
2002-08-16	11290	11485	11211	11290	87040
2002-08-19	11446	11446	11250	11250	39424
2002-08-20	11407	11407	11211	11211	50944
2002-08-21	11329	11446	11290	11446	53248
2002-08-22	11563	12110	11524	11836	224256
2002-08-23	12032	12735	11836	12500	259840
2002-08-26	12500	12735	12461	12500	138496
2002-08-27	12500	12500	12110	12188	59648
2002-08-28	11915	11954	11368	11407	127744
2002-08-29	11329	11446	11172	11250	100864
2002-08-30	11641	11836	11641	11719	72192
2002-09-02	11719	11719	11368	11446	34304
2002-09-03	11446	11680	11133	11211	69888
2002-09-04	10938	10938	10665	10743	91648
2002-09-05	10938	11055	10860	10977	51968
2002-09-06	10782	11094	10665	11094	50176
2002-09-09	11368	11602	11368	11407	36608
2002-09-10	11563	11602	11329	11329	28416
2002-09-11	11875	11915	11485	11797	124160
2002-09-12	11680	11719	11524	11563	52992
2002-09-13	11329	11446	11250	11446	27392
2002-09-17	11407	11758	11407	11602	38656
2002-09-18	11524	11524	11446	11524	39168
2002-09-19	11446	11719	11133	11133	92928
2002-09-20	10938	11172	10665	11172	83200
2002-09-24	11094	11290	10977	11133	90624
2002-09-25	11875	12032	11250	11407	72192
2002-09-26	11719	11875	11485	11641	29568
2002-09-27	11719	11875	11641	11719	34688
2002-09-30	11719	11797	11485	11797	37376
2002-10-01	11719	11719	11016	11329	57600
2002-10-02	11563	11563	11094	11172	17792
2002-10-03	10938	11250	10547	10704	44032
2002-10-04	10391	10469	10313	10391	59648
2002-10-07	9766	11016	9610	11016	70528
2002-10-08	10235	10625	10157	10391	30848
2002-10-09	10391	10547	10000	10079	29440
2002-10-10	10157	10547	9922	10547	42368
2002-10-11	11172	11172	10782	10860	58880
2002-10-15	11329	11563	11094	11485	52480
2002-10-16	11329	11485	11094	11407	43776
2002-10-17	11329	11407	11172	11329	12288
2002-10-18	11329	11719	11329	11719	77312
2002-10-21	11797	11875	11016	11094	55424
2002-10-22	11250	11407	11016	11172	17024
2002-10-23	11172	11719	11094	11719	28416
2002-10-24	11641	12188	11485	12032	171392
2002-10-25	11875	11875	11719	11875	42496
2002-10-28	11875	13047	11719	12735	147712
2002-10-29	12735	12735	12110	12422	95360
2002-10-30	12422	12735	12422	12422	96896
2002-10-31	12422	12422	12110	12344	38016
2002-11-01	12500	12579	12344	12500	48256
2002-11-05	12579	12657	12110	12344	47488
2002-11-06	12266	12344	11719	11875	112768
2002-11-07	11875	11954	11407	11407	48640
2002-11-08	11329	11797	11016	11641	35840
2002-11-11	11407	11485	10860	10860	73600
2002-11-12	10860	11563	10860	11563	55808
2002-11-13	11563	11641	11094	11094	12416
2002-11-14	11250	11563	11250	11407	26880
2002-11-15	11641	11641	11172	11329	23936
2002-11-18	11407	11485	11172	11250	21376
2002-11-19	11172	11719	10938	11172	80384
2002-11-20	11329	11407	10547	10547	108928
2002-11-21	10860	11016	10625	10625	109696
2002-11-22	10938	11407	10860	11329	142080
2002-11-25	11250	12110	11172	12032	150528
2002-11-26	11875	11875	11407	11485	94336
2002-11-27	11485	11719	11329	11641	42880
2002-11-28	11875	11954	11563	11797	56320
2002-11-29	11797	11797	11563	11719	37632
2002-12-02	11719	11797	11485	11797	56448
2002-12-03	11954	12344	11954	12110	123520
2002-12-04	11797	11875	11641	11719	53888
2002-12-05	11719	11797	11563	11563	58240
2002-12-06	11641	11641	11172	11172	69248
2002-12-09	11329	11797	11329	11797	71424
2002-12-10	11641	11719	11485	11641	39296
2002-12-11	11719	11797	11563	11641	53376
2002-12-12	11407	11563	11329	11329	30720
2002-12-13	11407	11407	11016	11172	86272
2002-12-16	11172	11641	10938	11641	113024
2002-12-17	11563	11641	11407	11563	69632
2002-12-18	11563	11563	11329	11485	40576
2002-12-19	11407	11563	11094	11563	61312
2002-12-20	11485	11485	11329	11407	39040
2002-12-24	11407	11407	11172	11250	38912
2002-12-25	11250	11329	11016	11094	120704
2002-12-26	11016	11563	11016	11563	53760
2002-12-27	11485	11563	11329	11485	22784
2002-12-30	11407	11563	11407	11563	9984
2003-01-06	11485	11875	11407	11797	40832
2003-01-07	11719	11797	11407	11485	23040
2003-01-08	11563	11719	11485	11563	28928
2003-01-09	11485	11641	11485	11563	21888
2003-01-10	11641	11641	11407	11407	19584
2003-01-14	11563	11797	11563	11797	48256
2003-01-15	11797	11797	11563	11641	21504
2003-01-16	11719	12032	11641	11954	80256
2003-01-17	11954	12891	11954	12813	255232
2003-01-20	13125	13672	13125	13125	273920
2003-01-21	13204	13750	13047	13750	239872
2003-01-22	13672	13750	13282	13516	161920
2003-01-23	13204	13516	12891	13204	154880
2003-01-24	13204	13985	13204	13750	176384
2003-01-27	13438	13907	13438	13750	57216
2003-01-28	13516	13594	13282	13438	79616
2003-01-29	13516	13594	13360	13360	55168
2003-01-30	13204	13204	12969	13204	50560
2003-01-31	13047	13204	12891	13047	70656
2003-02-03	13282	14297	13125	14063	160384
2003-02-04	14532	15157	14532	14844	229632
2003-02-05	14688	15079	14532	14844	154112
2003-02-06	15000	15469	15000	15391	188416
2003-02-07	15313	16016	15313	15625	252928
2003-02-10	15704	15704	14844	14922	90368
2003-02-12	14922	15000	14532	14844	118272
2003-02-13	14844	15313	14766	15079	88192
2003-02-14	15079	15235	14844	14844	87424
2003-02-17	15157	15547	15157	15391	92032
2003-02-18	15391	15469	15157	15235	40192
2003-02-19	15547	15938	15469	15860	141568
2003-02-20	16172	16641	16094	16250	172800
2003-02-21	16407	16485	16172	16250	91904
2003-02-24	16329	16329	15625	15782	58240
2003-02-25	15782	16407	15704	16250	89984
2003-02-26	16172	16250	16094	16250	37504
2003-02-27	16172	16172	15782	15860	90624
2003-02-28	15938	16016	15860	15938	55680
2003-03-03	16016	16094	14375	14375	240768
2003-03-04	13750	15000	13516	14766	268160
2003-03-05	14454	14922	14375	14532	77312
2003-03-06	14532	14532	13985	14063	77056
2003-03-07	13829	13829	12657	12735	164480
2003-03-10	12500	12500	11407	11485	312192
2003-03-11	11797	12813	11329	12032	318336
2003-03-12	13594	13594	13594	13594	73600
2003-03-13	13594	14454	13438	13907	348288
2003-03-14	14063	14454	14063	14297	212480
2003-03-17	14063	14141	13750	13907	112384
2003-03-18	14219	14297	13985	14141	126720
2003-03-19	12579	12579	12579	12579	576512
2003-03-20	12579	13204	12032	13125	848768
2003-03-24	13282	13360	12813	13047	515712
2003-03-25	12813	12969	12500	12891	384512
2003-03-26	14454	14454	14063	14454	295616
2003-03-27	14688	15594	14532	15094	401344
2003-03-28	15063	15375	15000	15204	196672
2003-03-31	15157	15344	14969	15235	175808
2003-04-01	15157	16563	15157	15938	337216
2003-04-02	15782	16719	15782	16407	249664
2003-04-03	16875	18907	16875	17657	406272
2003-04-04	17500	17657	16719	16875	110272
2003-04-07	17500	17969	17344	17657	180800
2003-04-08	17500	18125	17188	17813	165952
2003-04-09	17657	18282	17500	18282	116800
2003-04-10	18125	18282	17813	18125	87360
2003-04-11	18125	18907	17969	18750	200384
2003-04-14	19063	19219	17969	18750	190080
2003-04-15	18750	18907	18438	18907	99456
2003-04-16	18282	18438	17969	18125	123456
2003-04-17	17500	18125	17188	17813	116864
2003-04-18	18125	18438	17813	18438	76800
2003-04-21	18438	18438	18125	18282	32896
2003-04-22	18125	18282	17969	18125	60160
2003-04-23	18282	19063	17813	19063	225536
2003-04-24	19063	21250	18907	20938	319872
2003-04-25	20938	24063	20469	22032	415744
2003-04-28	21407	21407	19532	19844	150272
2003-04-30	19844	21250	19219	20938	210496
2003-05-01	21094	22188	20782	21250	140480
2003-05-02	21563	21875	21250	21563	96896
2003-05-06	21719	21875	21094	21250	148416
2003-05-07	20938	22500	20625	22188	248064
2003-05-08	22188	22344	21563	21875	103424
2003-05-09	21250	21563	20938	21250	109184
2003-05-12	20625	20782	19532	19844	127680
2003-05-13	19688	20782	19532	20313	111360
2003-05-14	20625	20782	20157	20625	127360
2003-05-15	20469	20469	19844	20000	73792
2003-05-16	20157	20313	19844	20157	82368
2003-05-19	20000	20157	19532	19688	56128
2003-05-20	19375	20625	18125	20313	491200
2003-05-21	21094	23125	21094	22657	711680
2003-05-22	23750	24375	22032	22344	448704
2003-05-23	22657	23282	21407	21407	169728
2003-05-26	22032	22500	21563	22344	198528
2003-05-27	22500	22500	21563	21875	98432
2003-05-28	22813	23282	22344	23125	195072
2003-05-29	23282	23282	22500	22969	127488
2003-05-30	22657	22969	22500	22813	87232
2003-06-02	23125	23125	22500	22657	97600
2003-06-03	22657	22657	21875	22188	123200
2003-06-04	22032	22813	21719	22188	145408
2003-06-05	22500	22657	22344	22657	107776
2003-06-06	22813	24219	22500	24219	261760
2003-06-09	23907	24219	23594	24063	171904
2003-06-10	23438	23907	23438	23750	77376
2003-06-11	23438	23750	23125	23438	96704
2003-06-12	23282	23438	22657	22969	105216
2003-06-13	22657	23594	22657	23438	152448
2003-06-16	23282	23438	22969	22969	47488
2003-06-17	24219	24532	23750	24532	234816
2003-06-18	24688	25313	24688	25313	233600
2003-06-19	25938	27657	25938	27657	324224
2003-06-20	26719	27188	26563	27032	172928
2003-06-23	27344	27657	26719	27188	177664
2003-06-24	26407	26875	26250	26563	130560
2003-06-25	26563	27500	26563	27500	145344
2003-06-26	27500	27969	27188	27500	151232
2003-06-27	28438	28907	28125	28594	274816
2003-06-30	28750	31250	28750	30469	249024
2003-07-01	30625	30782	29375	29688	293632
2003-07-02	30313	30782	29063	29219	208000
2003-07-03	29688	29688	27344	28438	363072
2003-07-04	28282	29219	27813	28750	143296
2003-07-07	29219	30469	29219	30469	225728
2003-07-08	32344	33594	31563	33594	424704
2003-07-09	33594	36719	32813	34844	478912
2003-07-10	32657	33438	31875	32032	399040
2003-07-11	30782	30938	28907	29219	404160
2003-07-14	30469	31719	30313	31719	197568
2003-07-15	32500	33125	30782	31094	223232
2003-07-16	31407	31719	30157	30782	132224
2003-07-17	29532	30313	28907	29844	169280
2003-07-18	31250	32813	31094	32032	503744
2003-07-22	32657	35157	32500	35000	671168
2003-07-23	34375	34532	33282	33750	398592
2003-07-24	33750	33750	32500	32969	229696
2003-07-25	32969	34063	32500	33438	184640
2003-07-28	34063	34532	33750	34063	176512
2003-07-29	34532	34844	33907	34219	121408
2003-07-30	34219	34219	33125	33438	100544
2003-07-31	32813	33125	31719	31875	273344
2003-08-01	32813	33282	31875	32188	191744
2003-08-04	31875	31875	31250	31563	89984
2003-08-05	31250	31407	30157	30782	153792
2003-08-06	29688	30469	29532	30157	192128
2003-08-07	30000	30625	29532	29532	126400
2003-08-08	29844	30157	29375	29688	84992
2003-08-11	29688	29844	28125	28438	213568
2003-08-12	28750	29688	28125	29532	232128
2003-08-13	30469	32188	30313	31719	299136
2003-08-14	31250	31719	30782	31563	94720
2003-08-15	31875	32032	31250	31407	90240
2003-08-18	31719	32188	31407	31875	95104
2003-08-19	32344	32813	32188	32500	128000
2003-08-20	33125	33438	32813	33438	171008
2003-08-21	32500	32813	32344	32657	87680
2003-08-22	32500	32500	31407	31407	125760
2003-08-25	30782	31250	30469	30625	129664
2003-08-26	30313	30469	29688	30157	164096
2003-08-27	32969	33125	31407	31719	332864
2003-08-28	32344	32344	31250	31407	89408
2003-08-29	31719	33125	31563	32969	172224
2003-09-01	32969	33125	32344	32657	119104
2003-09-02	32969	34375	32813	34375	290624
2003-09-03	35157	35469	33594	33750	326912
2003-09-04	34063	34219	33750	33907	139200
2003-09-05	34219	34375	32813	33750	157184
2003-09-08	33125	34219	33125	34063	65344
2003-09-09	34375	34532	33750	33907	135872
2003-09-10	34063	34219	33907	34063	112512
2003-09-11	33907	34063	33594	33907	94016
2003-09-12	33907	35313	33907	35313	270656
2003-09-16	35625	38282	35625	37969	428672
2003-09-17	38750	41094	38750	39688	424064
2003-09-18	40938	42813	40938	42813	323520
2003-09-19	44532	45313	40469	41875	483328
2003-09-22	40625	41719	40000	41250	324928
2003-09-24	42500	44063	41563	43750	440640
2003-09-25	48750	50000	46250	46250	504128
2003-09-26	44375	45625	41875	43438	352320
2003-09-29	44063	45938	43125	45938	228736
2003-09-30	47500	50938	46563	49375	325888
2003-10-01	50000	50000	48438	49688	225696
2003-10-02	50938	51250	48125	48438	219008
2003-10-03	48750	48750	46563	47188	193280
2003-10-06	47813	49063	47500	48438	143424
2003-10-07	48438	50625	47500	50625	261632
2003-10-08	50313	50313	48750	49375	184096
2003-10-09	50000	50313	48438	49063	138144
2003-10-10	50313	52188	49688	51250	344736
2003-10-14	51875	52500	50625	51250	200992
2003-10-15	51250	52500	50000	52500	219552
2003-10-16	52188	58750	52188	58750	526752
2003-10-17	58750	60000	55625	56875	402848
2003-10-20	56250	59375	55625	58750	252096
2003-10-21	60000	60313	52500	54375	444224
2003-10-22	59375	59688	53125	53750	580992
2003-10-23	51250	51875	47500	48125	452544
2003-10-24	48750	51563	47500	50000	372160
2003-10-27	50625	54688	50313	52500	446752
2003-10-28	51563	53125	48750	49063	411424
2003-10-29	51563	51563	49688	50625	258816
2003-10-30	50313	50938	49375	50938	155904
2003-10-31	51563	52813	50938	50938	501760
2003-11-04	52188	52500	50000	51250	175392
2003-11-05	50313	50625	48750	49688	143328
2003-11-06	48750	48750	46875	46875	171712
2003-11-07	45938	46875	43125	46563	232576
2003-11-10	45938	45938	44375	45000	124096
2003-11-11	41250	41563	38750	40938	367872
2003-11-12	42188	43125	40313	41563	260992
2003-11-13	44063	47500	43438	46250	579584
2003-11-14	45000	45625	42813	44063	208352
2003-11-17	41563	42500	40938	41563	208640
2003-11-18	42188	42813	40000	42188	321760
2003-11-19	39688	41250	39375	40313	170464
2003-11-20	41563	42500	39688	40938	218784
2003-11-21	40313	42500	40000	40938	325888
2003-11-25	42188	43125	41250	42500	507232
2003-11-26	43750	48438	43750	46875	1100896
2003-11-27	48750	49375	46563	48125	1051744
2003-11-28	46563	47813	45000	45000	1394624
2003-12-01	43750	46563	42500	45313	670752
2003-12-02	46875	47500	45938	46250	327040
2003-12-03	46250	47500	45938	47500	213344
2003-12-04	46563	47188	45625	45625	197024
2003-12-05	45938	46563	45625	46250	86816
2003-12-08	45313	45625	43125	43438	197216
2003-12-09	44063	45000	42500	42813	178368
2003-12-10	42188	42500	40938	41563	144352
2003-12-11	40938	42188	40938	41875	136800
2003-12-12	41250	41250	39063	39688	410336
2003-12-15	41563	43125	40313	43125	200224
2003-12-16	40938	42500	40625	41250	169792
2003-12-17	42188	42500	41250	41563	122400
2003-12-18	42188	42500	40938	41250	75328
2003-12-19	41563	41875	40938	40938	90976
2003-12-22	40625	41563	40313	41250	96672
2003-12-24	41875	42500	41563	42188	144544
2003-12-25	42500	44063	42500	43125	179360
2003-12-26	43750	44375	42813	44063	150464
2003-12-29	44375	45000	44063	44375	85760
2003-12-30	44688	45313	44688	45000	78016
2004-01-05	45938	46250	45313	45313	60864
2004-01-06	45938	46563	45000	45000	82624
2004-01-07	45000	45938	44375	45625	85568
2004-01-08	46250	47813	45625	45938	247104
2004-01-09	46563	46875	45313	46250	152672
2004-01-13	46875	47500	45625	45625	121824
2004-01-14	45938	46250	45625	45938	80544
2004-01-15	45625	46250	44688	44688	100992
2004-01-16	45313	45938	44688	45938	98208
2004-01-19	46563	47500	46250	47500	192448
2004-01-20	47813	48750	47188	48438	262560
2004-01-21	48438	48750	47500	47500	151136
2004-01-22	48438	48438	46250	46250	144992
2004-01-23	46250	47188	45938	46875	92512
2004-01-26	46875	47813	46875	47813	98400
2004-01-27	47813	48125	46875	46875	128736
2004-01-28	46875	47813	46563	47813	128832
2004-01-29	48125	48438	47500	47813	363968
2004-01-30	48125	49375	47500	48750	907488
2004-02-02	48750	50313	48750	49688	487296
2004-02-03	50000	52188	49688	51563	399104
2004-02-04	52188	53125	50000	50000	353824
2004-02-05	49063	50625	49063	50000	184160
2004-02-06	50938	51875	50625	51250	219296
2004-02-09	52813	54688	52188	54688	386496
2004-02-10	55313	55625	53438	54063	233472
2004-02-12	54688	55625	54375	54688	134816
2004-02-13	55000	55000	54063	54688	97792
2004-02-16	55000	56875	54688	56563	230240
2004-02-17	56563	57188	55625	56875	137760
2004-02-18	59688	61563	58125	58125	688800
2004-02-19	59063	60938	58750	60000	255552
2004-02-20	60000	60000	58750	60000	168352
2004-02-23	60000	63750	59688	63750	497760
2004-02-24	64688	67500	62813	62813	547168
2004-02-25	59688	63750	58438	59375	862304
2004-02-26	59063	60625	58438	60000	448960
2004-02-27	61250	63125	61250	62813	298752
2004-03-01	65313	67813	64688	67188	371264
2004-03-02	69063	70000	66563	68125	345056
2004-03-03	67813	68750	67500	68125	180576
2004-03-04	68125	68438	67188	67188	154784
2004-03-05	66563	66875	65000	65938	209344
2004-03-08	65313	67500	64688	66875	194688
2004-03-09	66563	68438	66250	67500	224448
2004-03-10	67813	68125	66250	66250	166624
2004-03-11	65000	65625	63750	64375	284896
2004-03-12	61875	63125	61875	62500	422720
2004-03-15	63750	64688	63438	63750	200320
2004-03-16	63438	64375	62813	63125	221632
2004-03-17	64063	66250	64063	66250	192992
2004-03-18	68438	68438	65625	66250	266240
2004-03-19	65625	67188	65625	67188	163136
2004-03-22	66875	68438	66250	66875	230496
2004-03-23	65938	67188	65625	67188	158144
2004-03-24	67500	67500	64375	65313	239712
2004-03-25	65625	66250	63750	65625	346144
2004-03-26	67500	75625	66875	73125	814496
2004-03-29	75625	81875	75625	80000	730544
2004-03-30	83750	84375	79375	80000	422336
2004-03-31	79375	80000	76875	79375	288272
2004-04-01	81875	82500	78750	81250	352480
2004-04-02	81875	86250	81250	84375	442384
2004-04-05	86875	89375	85625	86875	394704
2004-04-06	87500	88125	82500	83125	378624
2004-04-07	80000	81875	78750	79375	372880
2004-04-08	83125	83750	80625	81875	196304
2004-04-09	80000	80625	76875	78125	200480
2004-04-12	79375	81875	78750	80625	150016
2004-04-13	81875	82500	79375	80000	123152
2004-04-14	78750	79375	78125	78125	101568
2004-04-15	78750	79375	75000	76875	186160
2004-04-16	76875	77500	74375	75625	142672
2004-04-19	76250	77500	72500	75000	148768
2004-04-20	75625	76250	73125	75625	210288
2004-04-21	75000	77500	74375	76250	251664
2004-04-22	78750	80000	75000	75000	222160
2004-04-23	75625	76250	73125	74375	142064
2004-04-26	73125	73750	71875	72500	112336
2004-04-27	71875	75000	71250	74375	223968
2004-04-28	74375	81250	74375	81250	397696
2004-04-30	78750	80625	77500	78125	254832
2004-05-06	80000	80000	78125	78125	93648
2004-05-07	76875	77500	75625	76250	97952
2004-05-10	73125	74375	67500	68750	170624
2004-05-11	65625	70625	65000	67500	225696
2004-05-12	71875	72500	69375	70625	209520
2004-05-13	71250	71250	68750	69375	111664
2004-05-14	68125	69375	65625	65625	141328
2004-05-17	65625	65625	57688	61125	180736
2004-05-18	60563	63750	59375	63750	156752
2004-05-19	66250	66875	65000	66250	136688
2004-05-20	65625	67500	63750	63750	137344
2004-05-21	65000	66875	64375	66250	104608
2004-05-24	66250	68125	65625	66250	171088
2004-05-25	66875	68750	65625	66875	233744
2004-05-26	70000	70625	66875	67500	222336
2004-05-27	67500	68750	66875	68125	126672
2004-05-28	68750	72500	67500	72500	394384
2004-05-31	71250	72500	70000	71875	327424
2004-06-01	70625	72500	70625	71875	146288
2004-06-02	71875	71875	70625	71250	104656
2004-06-03	71875	71875	68750	69375	119712
2004-06-04	68125	68750	65625	68125	178928
2004-06-07	68125	69375	66875	68750	198512
2004-06-08	69375	70000	66875	68125	98144
2004-06-09	68125	68125	65625	66250	115712
2004-06-10	65000	65625	63750	64375	185648
2004-06-11	63750	66250	63750	65000	227184
2004-06-14	65625	66250	64375	64375	107536
2004-06-15	64375	65000	63750	64375	80816
2004-06-16	65000	65000	63750	63750	82976
2004-06-17	64375	65000	63750	65000	166192
2004-06-18	64375	65000	62500	63125	149280
2004-06-21	63750	64375	62500	62500	88304
2004-06-22	62500	63125	59875	61000	216688
2004-06-23	60000	60250	58000	58125	284944
2004-06-24	59375	63125	58313	62125	352896
2004-06-25	61500	62500	60625	62000	118832
2004-06-28	63125	65000	62500	65000	142256
2004-06-29	65625	66250	64375	65625	99456
2004-06-30	65625	66875	65000	66250	104224
2004-07-01	66875	67500	66250	66250	130496
2004-07-02	63750	65000	63125	63125	112320
2004-07-05	62500	63125	62125	62125	82240
2004-07-06	61250	62313	59688	60000	129856
2004-07-07	58125	61250	57875	61250	187200
2004-07-08	60000	60063	58750	58938	161824
2004-07-09	58938	60063	58438	59813	200288
2004-07-12	60438	60438	59563	59875	72672
2004-07-13	59375	59625	58188	58563	122000
2004-07-14	59063	59250	57813	58250	113920
2004-07-15	58125	58125	53750	55625	333680
2004-07-16	55000	57375	53188	56750	200544
2004-07-20	55000	55375	53938	54250	151184
2004-07-21	55125	57875	55063	57500	219696
2004-07-22	56688	60688	55625	59563	536704
2004-07-23	59375	59500	57813	58125	136304
2004-07-26	56875	57625	56250	56250	143744
2004-07-27	55188	56563	53875	54063	162304
2004-07-28	56250	56563	55063	56188	123920
2004-07-29	56000	56000	53813	54125	131968
2004-07-30	54500	55250	54375	55250	76800
2004-08-02	54375	54625	53438	53750	111840
2004-08-03	54188	54875	53188	53750	111232
2004-08-04	51938	52438	50250	52063	238448
2004-08-05	52250	54438	50875	53125	164592
2004-08-06	51313	52938	51313	52750	104160
2004-08-09	52125	54250	51375	53938	103840
2004-08-10	55563	59438	55500	59250	368832
2004-08-11	60688	61875	59500	59938	262896
2004-08-12	59750	60750	58938	60000	99632
2004-08-13	59375	60938	58875	59750	153776
2004-08-16	59688	60250	58625	59875	90032
2004-08-17	60563	61375	59500	59500	141152
2004-08-18	59500	60438	59125	60000	64944
2004-08-19	61063	63750	60625	63125	480656
2004-08-20	63750	66250	63125	65625	309712
2004-08-23	67500	68125	65625	66875	221280
2004-08-24	67500	68750	66875	68750	272880
2004-08-25	69375	70000	66250	66875	319984
2004-08-26	68125	68750	64375	65000	166480
2004-08-27	64375	66250	63750	65625	121936
2004-08-30	66875	67500	65625	67500	114176
2004-08-31	66875	67500	65625	65625	70944
2004-09-01	65625	67500	65625	66250	96368
2004-09-02	67500	67500	66250	67500	66112
2004-09-03	67500	68125	66250	66250	76720
2004-09-06	66875	67500	66250	66875	61456
2004-09-07	67500	68750	66875	68750	136432
2004-09-08	68125	68750	66875	67500	109376
2004-09-09	66875	67500	66250	66250	76400
2004-09-10	66250	66875	65625	66875	130512
2004-09-13	66875	68125	66875	67500	70000
2004-09-14	68125	68125	66875	66875	70640
2004-09-15	66875	67500	65625	65625	83504
2004-09-16	65000	65625	63750	65000	92240
2004-09-17	65000	65000	63750	65000	72336
2004-09-21	64375	66875	64375	66250	121712
2004-09-22	66250	66875	64375	64375	110768
2004-09-24	62500	63125	62000	62500	256192
2004-09-27	65375	65625	60125	62375	287568
2004-09-28	61125	61250	59125	60000	142872
2004-09-29	60625	60750	57750	58000	113064
2004-09-30	59250	61250	58500	61250	135960
2004-10-01	63250	63750	62125	62875	113184
2004-10-04	65125	65375	63625	64000	96528
2004-10-05	63750	63750	62875	63125	51464
2004-10-06	62500	64625	62500	64625	88160
2004-10-07	65250	66125	64375	64750	114896
2004-10-08	64250	65000	64000	64625	37480
2004-10-12	64625	65000	63375	63500	53200
2004-10-13	63875	64500	63500	63625	56992
2004-10-14	63750	64375	62875	63625	74472
2004-10-15	63000	63625	62625	63500	60560
2004-10-18	63750	63875	63375	63500	51904
2004-10-19	64125	65125	63375	64750	87968
2004-10-20	64125	64250	63000	63000	78520
2004-10-21	62375	62875	60000	60000	141576
2004-10-22	59375	60125	57875	60125	136320
2004-10-25	58250	59500	58125	59000	64856
2004-10-26	59125	60125	59000	60000	61664
2004-10-27	61000	61250	60250	60625	61688
2004-10-28	61750	61750	60625	60625	62952
2004-10-29	61250	61250	59250	59875	61048
2004-11-01	58750	59625	58750	59375	35192
2004-11-02	59125	60250	59125	59750	43008
2004-11-04	60375	60750	58250	58625	72080
2004-11-05	59125	59500	58875	59250	58552
2004-11-08	59000	59125	57875	58125	64584
2004-11-09	57875	58250	56750	57000	93224
2004-11-10	56375	57375	56250	57375	62024
2004-11-11	58375	59375	58000	58000	74288
2004-11-12	58375	60375	58375	60250	112960
2004-11-15	62500	64250	62125	64000	213376
2004-11-16	63750	64000	61875	62500	91344
2004-11-17	61250	62375	61125	62000	57664
2004-11-18	62375	63375	62250	62375	74024
2004-11-19	63125	63250	62000	62125	50864
2004-11-22	62750	62750	61000	62125	55080
2004-11-24	61875	62125	61125	61375	55168
2004-11-25	60750	61250	60125	60375	58112
2004-11-26	60375	60500	59750	60000	57144
2004-11-29	59750	61125	59500	60625	60360
2004-11-30	59750	60250	59250	59250	73544
2004-12-01	58625	58875	57500	57500	113616
2004-12-02	58750	59250	58500	59000	71416
2004-12-03	59500	60375	59500	60000	79488
2004-12-06	59500	59875	59000	59750	46184
2004-12-07	59250	59500	58500	58625	39696
2004-12-08	57750	58625	57625	58375	55320
2004-12-09	58750	58750	57625	58125	46696
2004-12-10	58250	58500	57750	57750	95800
2004-12-13	57625	58000	57250	57375	38080
2004-12-14	57875	58000	57125	58000	56280
2004-12-15	57500	58250	57500	57750	50120
2004-12-16	57625	58000	57125	57500	61216
2004-12-17	57500	58875	57500	58625	79328
2004-12-20	58125	58750	58000	58125	41704
2004-12-21	58250	58875	58125	58500	59400
2004-12-22	58750	59000	57875	58250	56968
2004-12-24	58375	58875	58250	58375	60688
2004-12-27	58750	58750	58125	58125	42152
2004-12-28	58250	60250	58125	60250	106776
2004-12-29	60875	62625	60625	61250	243080
2004-12-30	62000	62125	61250	61500	52552
2005-01-04	60500	61250	60250	61250	48904
2005-01-05	60375	60875	60250	60250	56032
2005-01-06	59625	60250	59500	59625	66616
2005-01-07	59750	60000	59375	59500	55832
2005-01-11	59875	60750	59750	60375	72304
2005-01-12	60250	60375	59750	60000	45152
2005-01-13	59750	60000	59000	59375	48648
2005-01-14	59750	60500	59375	60375	97344
2005-01-17	61250	63125	61125	62875	275104
2005-01-18	62750	62875	61875	62500	131728
2005-01-19	63000	63875	62000	62250	249720
2005-01-20	62000	63250	61000	62875	155848
2005-01-21	63125	63625	62750	63500	203384
2005-01-24	63750	66250	63625	66125	419576
2005-01-25	65875	66250	65375	66000	178496
2005-01-26	66250	66500	65375	65375	108672
2005-01-27	66000	66000	65375	65875	51168
2005-01-28	65500	65750	64500	65375	68088
2005-01-31	65250	65875	64250	65625	83088
2005-02-01	66000	66375	65625	65875	94216
2005-02-02	66250	68500	66250	67625	335384
2005-02-03	68125	68375	67250	68125	139104
2005-02-04	68375	69000	67625	68625	190328
2005-02-07	69250	70875	68875	70375	279704
2005-02-08	70125	70500	69500	70125	107504
2005-02-09	70125	70125	68500	69375	90296
2005-02-10	69375	69875	68750	69375	81600
2005-02-14	69625	69750	68375	68375	99272
2005-02-15	68250	68375	66375	66500	137632
2005-02-16	66375	66750	65250	66375	147296
2005-02-17	67375	68250	66000	66250	240624
2005-02-18	65500	65625	64625	64875	114344
2005-02-21	64875	66250	63625	65750	98896
2005-02-22	65875	66500	65500	65625	77440
2005-02-23	65375	65375	64250	64375	77008
2005-02-24	64500	65000	64375	64750	53232
2005-02-25	65000	65125	64375	65000	87032
2005-02-28	65125	65875	65000	65625	63368
2005-03-01	65625	66250	65375	65375	70392
2005-03-02	65375	65625	65125	65250	58520
2005-03-03	65125	65500	64875	65500	48552
2005-03-04	65875	66750	65625	65875	115928
2005-03-07	66500	66625	65875	66500	75168
2005-03-08	66250	66375	65625	65875	54912
2005-03-09	65750	65750	65125	65500	57616
2005-03-10	65125	65250	64625	64750	108752
2005-03-11	64500	65000	64250	64875	154192
2005-03-14	64875	65000	63250	63500	110056
2005-03-15	63500	63625	62000	62500	143688
2005-03-16	63500	64375	62500	62875	228024
2005-03-17	62875	63250	62500	62875	109408
2005-03-18	62750	64250	62750	63750	110128
2005-03-22	64125	65000	64000	64125	126600
2005-03-23	64250	64750	63875	64375	134200
2005-03-24	64500	64625	64000	64250	133448
2005-03-25	65250	65875	64750	65625	333488
2005-03-28	66250	66500	63000	64500	167840
2005-03-29	64250	64500	62250	63000	123336
2005-03-30	62750	63750	61750	62500	90800
2005-03-31	63000	63500	62250	62750	79836
2005-04-01	61750	62250	61500	62000	55696
2005-04-04	62250	62250	60250	61000	73284
2005-04-05	61000	61500	60250	60750	98612
2005-04-06	61000	62500	60750	62000	128048
2005-04-07	61250	62000	61250	62000	64504
2005-04-08	62000	63500	61750	63250	160876
2005-04-11	62500	63250	62250	62500	67692
2005-04-12	62250	62500	60750	61000	80216
2005-04-13	61000	61250	60500	61250	47792
2005-04-14	60500	61000	60000	60750	45840
2005-04-15	60000	60250	59500	60000	58296
2005-04-18	58750	59000	57500	57750	96460
2005-04-19	58750	59250	57750	58250	54188
2005-04-20	60000	60250	58750	59000	86900
2005-04-21	58750	60250	57750	59250	127080
2005-04-22	60750	61250	60000	61000	152492
2005-04-25	60000	61000	60000	60750	55184
2005-04-26	60750	60750	60000	60250	41208
2005-04-27	59500	59750	59250	59750	58304
2005-04-28	58750	59250	58000	58750	91656
2005-05-02	57750	58250	57500	58000	80700
2005-05-06	57750	58000	57250	57750	87332
2005-05-09	57500	57500	57000	57250	49964
2005-05-10	57000	57000	56500	57000	56160
2005-05-11	56500	56750	55500	55750	86812
2005-05-12	55750	56000	55000	55000	67972
2005-05-13	54750	56250	54750	55250	116100
2005-05-16	55500	55750	54000	54250	48544
2005-05-17	54750	55000	52250	52750	84204
2005-05-18	53000	53250	52000	52750	67952
2005-05-19	54500	55750	53500	55250	117456
2005-05-20	55750	57250	55000	55250	124456
2005-05-23	55500	55750	54750	55250	49236
2005-05-24	55500	55500	54000	54500	80284
2005-05-25	54000	54250	53000	53500	71620
2005-05-26	53500	53500	52500	53000	67440
2005-05-27	53250	54500	52750	54250	134516
2005-05-30	54250	56000	54000	55500	149928
2005-05-31	55500	55500	54250	54750	132320
2005-06-01	55500	56500	55000	55250	180448
2005-06-02	55750	57000	55500	56500	214144
2005-06-03	56750	57500	56500	56750	128560
2005-06-06	56250	57250	56000	57250	96836
2005-06-07	57000	57750	56250	57000	114684
2005-06-08	57250	57250	55750	56000	65544
2005-06-09	55500	56000	53750	54250	91788
2005-06-10	54000	55250	54000	54250	128164
2005-06-13	54000	54750	53750	54000	45364
2005-06-14	54000	54250	52750	53250	90692
2005-06-15	53000	53500	52750	53000	57656
2005-06-16	53250	53750	53000	53000	53316
2005-06-17	53500	54250	53250	53750	57316
2005-06-20	54250	55750	54000	55500	133420
2005-06-21	55500	56750	55000	56250	158496
2005-06-22	56500	57250	55750	56500	138624
2005-06-23	56250	56500	55500	55750	37244
2005-06-24	55000	55750	55000	55750	49092
2005-06-27	55000	55500	54750	55500	50704
2005-06-28	55000	57000	54750	56750	111476
2005-06-29	57500	58750	57250	58500	238004
2005-06-30	59250	59500	58000	58250	89240
2005-07-01	57250	59000	57250	58500	94244
2005-07-04	59250	59750	58250	58500	119108
2005-07-05	58250	58500	57250	57750	56348
2005-07-06	58250	58750	57750	58250	71504
2005-07-07	58000	58250	57500	58000	44912
2005-07-08	58000	58750	57750	57750	76064
2005-07-11	58250	58750	58000	58250	52632
2005-07-12	58250	58500	58000	58000	50992
2005-07-13	58250	58500	58000	58500	56292
2005-07-14	58250	59500	58250	58750	70812
2005-07-15	59250	59250	58000	58750	65900
2005-07-19	58750	58750	57750	58250	51904
2005-07-20	58250	58250	57250	57500	92532
2005-07-21	58750	59750	58500	59000	126504
2005-07-22	58250	58500	58000	58000	44292
2005-07-25	58250	58250	57750	58000	50364
2005-07-26	58000	58500	57750	57750	42312
2005-07-27	57500	58750	57500	58250	67768
2005-07-28	59000	59500	58500	58500	63648
2005-07-29	59250	61000	59000	60500	166344
2005-08-01	60250	62250	60250	62250	142840
2005-08-02	62000	62000	61250	61500	69968
2005-08-03	62500	62500	61250	61500	98696
2005-08-04	61000	61500	60500	61000	97184
2005-08-05	60500	60750	59750	60500	64448
2005-08-08	59250	60250	59000	60250	104636
2005-08-09	60000	61000	60000	60500	49256
2005-08-10	61250	62000	61000	61000	90004
2005-08-11	62000	62750	61500	61500	95224
2005-08-12	62500	64250	62000	64000	215724
2005-08-15	64000	64500	63250	63500	133368
2005-08-16	63750	64000	62500	63250	55308
2005-08-17	63250	66750	63000	65250	247368
2005-08-18	66250	67250	65500	65750	253216
2005-08-19	66000	66000	65000	65250	87468
2005-08-22	66000	66750	66000	66250	74336
2005-08-23	67000	67500	66000	66000	129604
2005-08-24	66250	66750	66000	66250	52756
2005-08-25	66250	66250	65000	65250	71076
2005-08-26	65500	66000	65250	66000	59780
2005-08-29	65250	65500	65000	65000	38612
2005-08-30	64750	65750	64250	65750	68208
2005-08-31	64750	65500	64750	65000	45664
2005-09-01	65250	65750	64500	64500	37420
2005-09-02	65000	65000	64000	64500	37776
2005-09-05	65000	65500	64750	65500	64828
2005-09-06	65750	66500	65500	65500	98936
2005-09-07	66000	66250	64500	65000	63900
2005-09-08	65000	65500	64750	65250	48888
2005-09-09	64750	65250	64500	65250	166664
2005-09-12	65750	66000	65250	65250	47376
2005-09-13	65500	65500	64250	64500	76820
2005-09-14	64000	64500	63750	64000	67796
2005-09-15	63750	64000	63500	64000	55976
2005-09-16	63500	63750	62500	63500	121836
2005-09-20	62750	63000	62500	62750	120252
2005-09-21	62500	62750	61750	61750	117608
2005-09-22	61250	61500	60250	61000	129108
2005-09-26	61250	62500	61250	61750	137652
2005-09-27	62000	62500	60000	60500	72970
2005-09-28	61500	62000	60500	61500	65036
2005-09-29	62500	66000	61500	64500	208878
2005-09-30	66000	68500	65000	66500	226676
2005-10-03	67000	68000	65000	65500	90966
2005-10-04	65500	67000	65500	66000	83970
2005-10-05	67500	68000	65000	66000	105430
2005-10-06	65500	66000	63000	63500	89400
2005-10-07	64000	64500	63000	64500	42936
2005-10-11	63500	64000	62000	63500	93440
2005-10-12	63500	64000	62000	62500	66764
2005-10-13	62500	62500	61500	62500	56866
2005-10-14	62000	62000	61000	61000	55326
2005-10-17	61000	61500	59500	60000	81718
2005-10-18	60000	61000	60000	60500	61098
2005-10-19	61000	61000	59500	60500	72636
2005-10-20	61000	61500	60000	60500	87642
2005-10-21	60000	61500	60000	61500	70040
2005-10-24	61500	61500	59000	59000	111598
2005-10-25	59500	60500	59000	59000	97682
2005-10-26	59000	59500	58500	59000	80236
2005-10-27	59000	60500	59000	60000	96068
2005-10-28	60500	63000	59000	63000	284948
2005-10-31	62000	63000	61500	61500	74976
2005-11-01	62500	63000	61500	62000	37754
2005-11-02	61500	62000	60000	61000	92114
2005-11-04	61500	61500	59500	60500	96744
2005-11-07	60000	60500	59000	60000	158072
2005-11-08	60000	60500	59500	60000	58464
2005-11-09	59500	60500	59000	59500	88644
2005-11-10	60500	66000	60000	64000	464370
2005-11-11	65000	67500	64000	65000	302796
2005-11-14	65500	68000	64500	67000	200574
2005-11-15	65500	67500	65500	66000	123990
2005-11-16	66500	69500	65500	69500	270586
2005-11-17	69000	74500	68500	73000	428464
2005-11-18	77500	78500	74500	75500	281392
2005-11-21	75500	76000	72500	75500	186690
2005-11-22	74000	74000	72000	73500	198076
2005-11-24	72500	74000	70500	71500	166862
2005-11-25	70500	71500	68500	71500	142554
2005-11-28	71500	73000	70500	71500	143572
2005-11-29	72000	76000	71500	73500	200996
2005-11-30	74000	74500	72500	72500	85780
2005-12-01	72000	72500	70500	72500	111068
2005-12-02	73500	74000	72000	73500	92166
2005-12-05	73000	73500	71500	71500	89142
2005-12-06	71500	72500	70500	70500	74372
2005-12-07	70500	71000	69500	70000	80966
2005-12-08	70500	72000	68500	69500	188696
2005-12-09	69500	71500	69000	71000	215548
2005-12-12	72000	73000	70500	71000	176280
2005-12-13	71000	71000	69500	70500	93532
2005-12-14	71500	72500	70000	71000	203210
2005-12-15	71500	73000	71000	72500	228906
2005-12-16	72500	73000	69500	69500	186406
2005-12-19	70500	73500	69500	73000	292140
2005-12-20	75000	79000	74000	77500	627194
2005-12-21	78500	79500	75500	76000	365058
2005-12-22	77500	78000	76000	77500	150798
2005-12-26	78500	82000	78000	80000	355766
2005-12-27	81000	84000	80000	82000	339334
2005-12-28	82500	82500	80500	81500	123230
2005-12-29	82500	91500	82500	90500	625000
2005-12-30	92500	95000	89000	89500	451174
2006-01-04	92500	92500	88000	88000	174876
2006-01-05	88500	90000	86500	87000	160056
2006-01-06	86500	89000	84000	86000	252328
2006-01-10	87000	87500	84500	84500	159048
2006-01-11	85000	90000	84000	89500	241822
2006-01-12	89000	90000	86000	87500	181380
2006-01-13	87000	89000	86000	88500	200742
2006-01-16	88500	91000	86500	89500	259212
2006-01-17	86500	88000	81000	82000	380264
2006-01-18	74500	76500	68500	74000	559430
2006-01-19	77500	84000	77000	84000	487610
2006-01-20	86500	86500	77500	79500	402288
2006-01-23	75000	78000	74500	75500	240878
2006-01-24	81500	82500	79500	80500	374250
2006-01-25	80500	81500	77500	77500	238968
2006-01-26	79000	79500	76000	77000	134000
2006-01-27	78500	80000	77500	79000	135202
2006-01-30	80000	80500	78000	78500	116210
2006-01-31	79000	79500	78000	79500	156660
2006-02-01	78500	79000	76000	76500	180018
2006-02-02	77500	78000	76000	76000	109390
2006-02-03	76000	77500	75000	77000	157474
2006-02-06	77500	77500	75500	76000	129088
2006-02-07	76500	79500	76000	79500	186514
2006-02-08	78000	79500	76000	76500	236330
2006-02-09	77500	77500	74500	75500	221112
2006-02-10	74500	75000	71000	72500	288392
2006-02-13	70500	70500	67500	68000	224272
2006-02-14	66500	70500	63000	70500	375598
2006-02-15	68500	69500	65500	66000	263836
2006-02-16	66500	67500	65000	67500	146022
2006-02-17	69500	72500	67000	67000	361928
2006-02-20	66000	66000	63000	64000	204912
2006-02-21	66000	70500	65500	69000	270138
2006-02-22	69000	70000	65500	66500	221364
2006-02-23	68500	70500	67500	69500	258260
2006-02-24	68000	69500	68000	69500	166306
2006-02-27	70500	73000	69500	69500	293528
2006-02-28	69500	70000	68000	69000	186286
2006-03-01	68000	68500	66500	67000	163052
2006-03-02	69000	69500	68000	68500	131406
2006-03-03	68500	69000	67500	68000	120690
2006-03-06	70000	73500	70000	73000	310606
2006-03-07	72000	72500	70500	72500	152602
2006-03-08	72000	72500	69500	71000	180260
2006-03-09	71500	74000	71000	73500	135972
2006-03-10	73500	74500	72500	73000	123036
2006-03-13	74500	75000	74000	75000	78298
2006-03-14	75000	76500	74500	75000	117446
2006-03-15	75500	76500	75500	76000	75316
2006-03-16	76000	76500	74000	74000	48428
2006-03-17	74000	74500	72500	73500	102276
2006-03-20	73500	74000	73000	73000	127778
2006-03-22	72000	72500	69000	69000	185032
2006-03-23	70500	71000	68500	69500	155856
2006-03-24	69500	70500	68500	69500	88748
2006-03-27	69500	69500	68000	68000	111748
2006-03-28	66900	67000	65300	66100	179695
2006-03-29	68100	71100	68000	71100	262168
2006-03-30	72900	75000	70300	70700	261070
2006-03-31	72000	72100	71200	71800	128670
2006-04-03	72500	72900	72000	72400	118517
2006-04-04	72600	72800	70200	71000	124690
2006-04-05	71500	71600	70000	70200	95056
2006-04-06	70200	70900	69500	69900	97468
2006-04-07	69800	70100	68800	69900	108693
2006-04-10	69400	69800	69100	69500	60535
2006-04-11	69200	69300	68100	68500	93906
2006-04-12	68000	68400	67100	68000	126725
2006-04-13	68100	68400	67100	67600	89188
2006-04-14	68000	68000	67000	67000	66655
2006-04-17	66500	66900	65300	66000	154687
2006-04-18	65500	67100	65400	66500	96750
2006-04-19	68500	69100	67000	67000	136424
2006-04-20	69000	69000	66100	66700	125041
2006-04-21	68000	68200	65200	66100	161676
2006-04-24	65300	65700	64000	64200	167951
2006-04-25	65200	68000	65000	67500	135681
2006-04-26	67500	67600	66000	66900	89796
2006-04-27	67400	67900	66800	67500	75622
2006-04-28	67000	67300	65600	66400	89420
2006-05-01	66100	66800	65300	65600	64814
2006-05-02	65100	65800	65000	65700	79514
2006-05-08	66400	67000	65800	66700	86504
2006-05-09	66600	67400	66000	66500	104198
2006-05-10	67400	67800	65600	66900	126538
2006-05-11	66500	67800	66100	66400	108513
2006-05-12	65700	65900	65100	65300	110920
2006-05-15	65000	65400	64300	64900	86099
2006-05-16	65900	66400	64000	64000	119130
2006-05-17	64300	64800	60200	62600	165222
2006-05-18	60100	60800	59000	60500	217500
2006-05-19	60300	62500	59400	62000	234745
2006-05-22	61800	62100	60100	60200	159569
2006-05-23	60000	60600	59000	59500	128748
2006-05-24	59800	60100	57700	59300	132842
2006-05-25	59000	61700	58500	60800	214559
2006-05-26	62000	62200	60900	61700	157596
2006-05-29	61500	61900	60300	61600	103525
2006-05-30	61600	62000	61000	61600	79172
2006-05-31	60100	60500	58500	59100	136551
2006-06-01	59500	60000	57200	57700	141093
2006-06-02	57800	59300	54400	58700	169507
2006-06-05	57900	58400	57000	57800	121662
2006-06-06	57000	57000	55500	56300	122899
2006-06-07	55800	56000	54300	55000	126332
2006-06-08	53500	53600	51200	52000	186725
2006-06-09	52400	54200	50500	53300	185632
2006-06-12	53000	56100	52500	55200	135159
2006-06-13	54200	57400	53700	55000	195216
2006-06-14	55100	58400	54500	57700	165363
2006-06-15	58700	59000	57800	57800	122345
2006-06-16	59300	59400	57400	57400	96463
2006-06-19	57100	60300	57000	59700	147346
2006-06-20	59400	60400	58200	58900	108585
2006-06-21	57900	58800	55900	56700	118503
2006-06-22	57900	58900	57200	58000	117941
2006-06-23	57400	59300	57400	58800	126643
2006-06-26	58300	61300	58300	60900	141681
2006-06-27	60500	61200	59900	60200	103329
2006-06-28	59200	60300	57900	58500	171823
2006-06-29	59200	60100	58600	59400	302957
2006-06-30	60100	60800	59200	60600	122614
2006-07-03	60800	62100	60100	61600	119254
2006-07-04	62200	62300	61600	62000	81931
2006-07-05	61600	62200	61400	62100	70975
2006-07-06	61900	62200	60900	61700	94461
2006-07-07	61900	61900	59400	60600	130660
2006-07-10	59600	61800	59500	61600	115912
2006-07-11	61900	62000	60300	61000	80269
2006-07-12	61000	61600	60400	60500	69593
2006-07-13	60100	60900	59200	59600	96456
2006-07-14	58600	58800	57100	57300	133586
2006-07-18	56800	56800	54900	55700	113530
2006-07-19	54700	54800	52100	52800	125941
2006-07-20	53800	56300	53800	56200	129696
2006-07-21	55700	55700	53300	55000	126791
2006-07-24	53000	53000	50300	51100	336404
2006-07-25	52600	53400	50000	50000	286032
2006-07-26	49850	49900	45700	46200	466415
2006-07-27	45400	48250	43250	46200	519231
2006-07-28	46600	47400	46000	46700	152894
2006-07-31	47900	49150	47400	48500	178900
2006-08-01	48500	49100	47700	48550	169672
2006-08-02	48350	48900	48000	48500	162686
2006-08-03	48900	49050	47200	47500	144055
2006-08-04	47500	47950	47000	47550	87795
2006-08-07	47400	47450	45400	45800	130125
2006-08-08	45600	45800	44100	44950	155098
2006-08-09	44850	46850	43800	46500	137498
2006-08-10	46350	46400	45500	46100	95148
2006-08-11	45500	45800	45250	45500	73572
2006-08-14	45100	46250	45100	46200	93123
2006-08-15	46300	47850	45700	47100	241960
2006-08-16	47900	49200	47600	48450	302436
2006-08-17	50000	51700	48800	49500	329447
2006-08-18	51300	51700	50300	51000	195027
2006-08-21	51000	51000	49200	49500	182241
2006-08-22	48850	49900	48650	49350	133006
2006-08-23	49450	50100	49100	49500	138418
2006-08-24	49200	49400	48650	49000	109439
2006-08-25	48700	48750	46300	46600	181885
2006-08-28	45950	46000	44850	45100	179331
2006-08-29	45500	46400	44800	45250	168690
2006-08-30	45250	45250	44000	44000	156030
2006-08-31	44250	45350	44000	44050	162918
2006-09-01	44050	44300	43300	43750	180478
2006-09-04	44950	46050	44000	45800	278996
2006-09-05	46100	47300	45900	47000	216513
2006-09-06	46550	46650	44850	45150	235397
2006-09-07	44350	44600	44050	44250	182602
2006-09-08	44000	44150	43250	43600	264168
2006-09-11	43650	43900	42600	42650	202110
2006-09-12	42850	43500	42150	42250	239033
2006-09-13	42650	42800	41000	41150	147635
2006-09-14	41200	42050	40200	41150	241535
2006-09-15	41550	42400	41150	41700	174055
2006-09-19	42450	42700	41200	41450	138262
2006-09-20	40300	40350	39600	40200	210851
2006-09-21	40500	40750	39200	39750	191865
2006-09-22	40550	41700	40350	41350	288228
2006-09-25	42300	42550	40600	41250	269498
2006-09-26	42050	43150	41400	41850	177889
2006-09-27	42250	42800	41850	42650	120432
2006-09-28	42700	43900	42350	43750	171869
2006-09-29	44100	44650	43650	44450	206816
2006-10-02	44100	44300	43200	43400	180725
2006-10-03	42200	42900	41800	42700	196764
2006-10-04	43000	43600	41600	42000	229239
2006-10-05	43500	43550	42300	42400	122426
2006-10-06	42000	42200	41100	41750	115137
2006-10-10	40950	41400	40450	40600	128643
2006-10-11	40650	40650	39150	39300	296626
2006-10-12	39700	41700	39500	41200	317911
2006-10-13	41450	43600	41450	43350	369306
2006-10-16	43750	47050	43700	46950	455565
2006-10-17	46550	47650	45500	46900	335802
2006-10-18	46850	49100	45650	48650	379219
2006-10-19	48600	49100	46900	47450	323015
2006-10-20	46700	47150	45150	45500	261883
2006-10-23	43900	46500	43550	45500	332762
2006-10-24	48700	49500	48650	49500	314130
2006-10-25	49350	49350	46900	47200	576388
2006-10-26	47900	48550	47050	47200	234192
2006-10-27	47600	47900	46250	47300	203050
2006-10-30	46900	47500	45850	45900	257831
2006-10-31	46150	46700	45500	45500	120724
2006-11-01	45300	46250	44600	45800	178195
2006-11-02	45700	46450	45150	45900	161809
2006-11-06	45600	45700	43650	43700	167915
2006-11-07	44000	44400	43650	43650	129635
2006-11-08	43350	43550	40600	41050	340545
2006-11-09	42650	43350	41600	42350	203546
2006-11-10	42000	42100	41400	41500	199343
2006-11-13	41050	43150	40900	42550	171123
2006-11-14	42950	43450	42550	43100	143701
2006-11-15	43450	43550	42050	42350	136231
2006-11-16	42100	42950	41450	41700	109195
2006-11-17	41350	41700	40000	40200	185760
2006-11-20	40500	40800	39950	40050	230022
2006-11-21	41000	41800	40550	41200	217080
2006-11-22	41500	43550	41000	43100	241664
2006-11-24	43100	43200	41500	42900	169037
2006-11-27	42500	44300	42350	43650	164596
2006-11-28	43000	44900	42600	44400	169460
2006-11-29	44750	45200	44450	44700	125788
2006-11-30	44750	46000	44600	45600	224212
2006-12-01	45450	46250	45200	45550	119974
2006-12-04	45600	47300	45400	46850	179672
2006-12-05	47100	47800	46800	46850	173217
2006-12-06	47000	47600	45850	47050	180449
2006-12-07	47300	47800	46900	47350	135242
2006-12-08	47000	47500	46350	46800	145456
2006-12-11	46800	46950	45300	45550	134320
2006-12-12	45700	46350	45450	45900	103792
2006-12-13	45300	45800	45050	45800	69154
2006-12-14	46500	47150	45850	47050	133460
2006-12-15	47550	48700	47300	48550	315768
2006-12-18	49350	52200	49100	51400	540467
2006-12-19	50500	50800	48800	49000	272841
2006-12-20	49050	51100	49000	50600	213932
2006-12-21	50600	51800	49950	50100	174374
2006-12-22	49600	49850	48550	49100	100874
2006-12-25	48500	48750	48100	48400	76630
2006-12-26	48000	49400	48000	49000	81478
2006-12-27	49250	49750	47750	48200	95567
2006-12-28	48200	48600	47100	47900	114822
2006-12-29	47500	47750	47000	47400	70210
2007-01-04	48400	50000	48300	49300	120498
2007-01-05	49350	49850	48900	49300	127598
2007-01-09	49350	49650	48400	48750	120420
2007-01-10	48750	49000	47550	47850	127605
2007-01-11	47850	47900	46600	47000	115501
2007-01-12	47150	47250	46300	46600	101959
2007-01-15	46500	46850	45800	46150	148555
2007-01-16	45900	47650	45900	47200	171695
2007-01-17	47150	47150	46300	46650	128476
2007-01-18	47000	47800	46750	47450	371722
2007-01-19	47700	49400	47400	48900	328005
2007-01-22	49300	49800	48150	48500	277626
2007-01-23	48250	48350	47350	47550	188625
2007-01-24	48100	48300	47500	47750	168292
2007-01-25	49600	49750	48200	48250	295413
2007-01-26	47850	48100	47150	47450	196020
2007-01-29	47500	47500	46250	46600	269415
2007-01-30	46600	46950	45500	45950	236898
2007-01-31	46050	46150	44600	45200	201136
2007-02-01	45050	45150	44100	44300	263970
2007-02-02	44650	44900	43000	43300	324301
2007-02-05	42550	43150	42200	42700	261645
2007-02-06	43000	43750	42800	42850	223488
2007-02-07	43250	44200	43000	43200	213799
2007-02-08	43600	43800	43200	43700	207589
2007-02-09	43750	43950	42700	43600	230390
2007-02-13	43750	44000	43350	43750	130303
2007-02-14	43800	43950	43400	43450	92476
2007-02-15	43450	44600	42350	44350	278434
2007-02-16	44050	45600	43800	45050	329283
2007-02-19	45800	45900	45050	45250	168317
2007-02-20	45300	45650	45050	45300	125117
2007-02-21	45600	45650	45100	45300	164175
2007-02-22	45450	45550	44550	45200	155607
2007-02-23	45250	46650	44950	46500	202034
2007-02-26	47000	47200	46150	46600	169556
2007-02-27	46650	46750	45700	46350	124248
2007-02-28	43950	45350	43550	44950	196032
2007-03-01	45050	45200	42600	43800	184959
2007-03-02	43000	44200	42900	43600	162652
2007-03-05	42800	42900	40650	41000	272792
2007-03-06	41000	41800	40700	41750	322735
2007-03-07	42300	42450	41400	41650	130131
2007-03-08	41250	41700	40900	41700	141425
2007-03-09	41600	41800	41150	41500	233921
2007-03-12	41700	41900	40500	40650	257670
2007-03-13	40700	41050	40500	40500	206537
2007-03-14	40100	40300	39500	39650	269804
2007-03-15	39800	40100	39550	39600	144953
2007-03-16	40000	40100	39350	39550	243368
2007-03-19	39400	39900	39200	39450	169434
2007-03-20	40100	41000	40100	40650	206306
2007-03-22	41400	41650	41200	41300	124844
2007-03-23	41350	41450	40700	41100	127037
2007-03-26	41500	41800	41250	41550	96286
2007-03-27	41000	41300	40300	40650	103979
2007-03-28	40700	41550	40600	40800	132941
2007-03-29	40400	41050	39900	40450	146366
2007-03-30	40750	40900	40300	40650	95542
2007-04-02	41450	42150	41000	41050	250074
2007-04-03	41450	41750	40700	41550	181868
2007-04-04	41800	41800	41150	41250	109960
2007-04-05	41350	41350	40500	40650	124573
2007-04-06	40650	40650	40050	40150	135272
2007-04-09	40150	40250	39850	40000	126294
2007-04-10	40100	40400	39950	40000	109885
2007-04-11	39950	40050	39150	39500	147637
2007-04-12	39500	39600	39200	39400	158163
2007-04-13	39650	39950	39400	39450	142042
2007-04-16	39500	39900	39450	39700	121861
2007-04-17	39750	39800	39000	39200	215183
2007-04-18	39450	39450	38350	38800	268079
2007-04-19	38400	38550	37600	38200	238102
2007-04-20	38050	38100	37300	37500	157504
2007-04-23	36700	36850	35750	36050	323631
2007-04-24	35950	36900	35900	36750	248525
2007-04-25	39550	40750	39050	40750	630929
2007-04-26	40750	43500	40650	42600	716356
2007-04-27	42650	43450	41500	41650	281804
2007-05-01	42250	42600	42100	42400	170410
2007-05-02	42800	42950	42150	42750	204197
2007-05-07	45150	45700	44350	44800	414603
2007-05-08	44800	46350	44300	45500	413723
2007-05-09	45400	45650	44300	45100	209592
2007-05-10	44550	45000	43450	43600	185998
2007-05-11	43400	43400	42700	43000	146021
2007-05-14	43350	43850	42600	42900	133518
2007-05-15	42450	43400	41900	42050	179851
2007-05-16	42000	42050	41250	41750	121250
2007-05-17	41700	41900	40400	40850	223035
2007-05-18	40500	40650	39550	40250	177762
2007-05-21	41450	41500	40250	40900	160248
2007-05-22	41400	42500	41250	42050	238581
2007-05-23	42100	42300	41000	41600	152356
2007-05-24	41600	41600	40600	41200	145154
2007-05-25	40800	42700	40800	42300	250608
2007-05-28	42000	43200	41850	42600	193426
2007-05-29	42200	42850	42100	42450	105197
2007-05-30	42550	43000	41750	42000	132391
2007-05-31	42350	42400	41600	41700	101263
2007-06-01	42100	42600	41500	42500	145031
2007-06-04	42900	42950	42100	42300	102199
2007-06-05	42350	43250	42200	42900	161688
2007-06-06	42700	42750	42200	42400	75026
2007-06-07	41700	42600	41600	42300	99376
2007-06-08	42050	42100	41300	41600	152793
2007-06-11	41950	42100	41100	41250	102341
2007-06-12	41000	41150	40350	40850	128817
2007-06-13	40100	40550	39850	40450	89572
2007-06-14	41150	42000	41150	41750	168549
2007-06-15	43750	43800	42500	43350	273820
2007-06-18	43700	44050	43200	43850	167191
2007-06-19	43850	43900	42600	43250	158795
2007-06-20	42850	43100	42100	42300	130336
2007-06-21	42000	43000	42000	42850	87541
2007-06-22	42850	43600	42550	43100	143771
2007-06-25	42600	43250	42500	42800	102753
2007-06-26	42450	42600	41550	41850	177283
2007-06-27	41900	41900	41500	41600	68706
2007-06-28	42000	42350	41650	42050	70917
2007-06-29	42100	42100	41300	41850	71292
2007-07-02	42200	42200	41700	41850	70571
2007-07-03	42050	42350	41750	41900	106021
2007-07-04	42300	42350	41750	42200	74308
2007-07-05	42100	42500	41750	41950	85337
2007-07-06	42000	42100	41600	42000	88424
2007-07-09	42100	42450	41850	42400	108055
2007-07-10	42000	42000	40850	41250	212545
2007-07-11	40850	41450	40650	40800	112461
2007-07-12	40850	41250	40500	40650	124598
2007-07-13	41250	41450	40850	41150	105955
2007-07-17	41100	41250	40400	40900	125640
2007-07-18	40350	40650	39850	40050	164852
2007-07-19	40250	40450	40100	40300	92320
2007-07-20	40450	40450	39800	39850	101443
2007-07-23	40000	40000	39350	39600	127580
2007-07-24	38800	40800	38350	40750	251985
2007-07-25	39350	39500	38650	39400	224261
2007-07-26	38500	38500	38000	38000	193136
2007-07-27	36800	37650	36650	37150	173342
2007-07-30	36900	37400	36600	37400	101392
2007-07-31	37500	37600	36950	37150	91954
2007-08-01	37000	37050	36150	36550	151437
2007-08-02	36600	37150	36350	36950	176844
2007-08-03	38150	38200	37150	37300	120702
2007-08-06	37350	38200	37050	38050	140531
2007-08-07	38050	38050	37100	37150	135485
2007-08-08	36900	37650	36550	37500	178847
2007-08-09	38100	40350	37950	38500	371910
2007-08-10	38950	40550	38850	39950	299509
2007-08-13	39550	39650	38000	39650	182546
2007-08-14	39600	39600	38400	38800	120159
2007-08-15	38850	38850	37950	38200	166825
2007-08-16	37000	37700	36400	37450	209738
2007-08-17	36650	37000	35200	35650	242055
2007-08-20	36250	36550	35900	36000	111785
2007-08-21	36000	36800	35750	36600	128438
2007-08-22	37400	40000	37300	38700	298377
2007-08-23	40000	40450	39200	39800	304711
2007-08-24	39800	40200	38550	38650	193419
2007-08-27	39450	40450	39450	40000	176613
2007-08-28	39850	42050	39850	41550	285612
2007-08-29	40750	43950	40400	43450	430420
2007-08-30	43200	44000	42850	43300	183181
2007-08-31	42950	43400	42150	42900	150659
2007-09-03	44000	44000	42850	43100	105551
2007-09-04	42700	43650	42650	42950	66098
2007-09-05	44000	44750	43200	43600	284022
2007-09-06	43000	44200	42400	44050	114099
2007-09-07	44000	44000	42950	43400	164809
2007-09-10	42900	43300	42600	43000	120548
2007-09-11	43050	44050	42400	43500	153490
2007-09-12	44000	44150	41650	42000	190517
2007-09-13	41800	41850	40550	40900	153829
2007-09-14	40650	41000	40400	40750	147006
2007-09-18	40350	40350	39200	39300	138606
2007-09-19	40200	40600	40050	40300	98659
2007-09-20	40400	40700	39800	40600	131533
2007-09-21	40200	40450	39800	39900	113704
2007-09-25	40700	40750	39900	40500	144523
2007-09-26	40150	41400	39850	41050	202814
2007-09-27	41250	42500	41250	42400	122968
2007-09-28	43200	44250	42850	43500	211704
2007-10-01	43500	43950	42950	43250	90092
2007-10-02	44050	44850	43400	43900	190766
2007-10-03	44550	46650	44200	46350	295868
2007-10-04	46350	49050	45650	49000	425786
2007-10-05	49800	51800	49100	51000	454867
2007-10-09	49500	50700	48800	49750	279389
2007-10-10	50400	51200	49900	50500	231941
2007-10-11	50600	53900	50100	53500	255949
2007-10-12	52000	52700	50500	51200	239281
2007-10-15	51700	52100	50600	51200	110107
2007-10-16	50700	51000	49050	49100	190048
2007-10-17	50800	52000	50300	51700	315112
2007-10-18	51700	54000	51700	53800	209390
2007-10-19	54000	57200	53000	56200	343480
2007-10-22	54200	58100	54000	57600	359889
2007-10-23	58600	59000	54500	54900	352165
2007-10-24	56500	57200	55300	55800	243381
2007-10-25	50800	51400	50800	50800	300234
2007-10-26	50300	50300	49200	50300	651149
2007-10-29	51300	53700	51200	52500	516219
2007-10-30	53000	53200	51400	52100	308238
2007-10-31	52300	52400	50200	51100	161612
2007-11-01	51200	52300	50200	51700	193570
2007-11-02	50200	54500	50200	53500	404093
2007-11-05	54000	54200	50100	50500	280647
2007-11-06	50600	52200	49050	49150	300256
2007-11-07	49500	50500	47000	47950	449471
2007-11-08	46750	48800	46100	48100	329296
2007-11-09	48150	48500	46200	46550	226398
2007-11-12	45350	46050	44650	45700	302750
2007-11-13	45650	46200	44750	45000	266798
2007-11-14	46950	48700	46900	48000	199146
2007-11-15	48000	48200	47000	47200	164879
2007-11-16	46300	47200	46100	47000	106787
2007-11-19	48150	49100	46750	46800	162377
2007-11-20	46000	47700	45150	47400	162639
2007-11-21	48000	48800	46900	47250	121327
2007-11-22	47300	51200	46750	50500	375439
2007-11-26	50300	52600	49950	51600	240622
2007-11-27	51100	53800	50900	53500	248397
2007-11-28	53500	54800	53000	54400	227819
2007-11-29	55400	55500	53300	54800	208741
2007-11-30	53600	54400	53100	53300	159044
2007-12-03	54300	54800	53300	54300	177171
2007-12-04	55800	57900	54300	54300	432573
2007-12-05	54800	56300	53200	55000	266078
2007-12-06	55500	55700	53300	53600	169838
2007-12-07	54100	54600	53100	53200	120047
2007-12-10	53000	53800	51600	52000	126997
2007-12-11	52100	52600	50500	51900	152014
2007-12-12	50900	53700	50900	53200	185965
2007-12-13	52500	52800	50300	50600	143931
2007-12-14	50100	51600	49800	50300	174557
2007-12-17	49400	49950	47100	47300	202741
2007-12-18	46500	49550	46450	48550	194387
2007-12-19	49750	49950	48600	48850	115199
2007-12-20	49000	49350	47750	48100	88719
2007-12-21	48700	50700	47750	50300	231599
2007-12-25	51900	52700	51000	51200	113893
2007-12-26	51600	51600	50000	51000	94431
2007-12-27	51000	51500	49900	50200	93241
2007-12-28	49000	50600	49000	50000	55601
2008-01-04	48650	48700	47500	48100	100963
2008-01-07	47700	49100	46800	47300	128428
2008-01-08	47050	48850	47050	48850	110811
2008-01-09	47250	49400	47050	49100	144123
2008-01-10	49150	49700	47850	48000	103454
2008-01-11	48400	48700	46200	46550	203113
2008-01-15	46950	48200	45350	45950	145748
2008-01-16	44350	44400	41950	41950	394973
2008-01-17	42750	43450	41000	42350	215251
2008-01-18	40700	44250	40650	43850	185487
2008-01-21	43000	44850	42550	42850	118008
2008-01-22	41000	41300	38850	38850	204951
2008-01-23	40350	40600	39700	40050	180200
2008-01-24	40450	41500	39450	39550	135762
2008-01-25	40150	41000	39450	41000	232706
2008-01-28	40600	40850	37650	38250	267077
2008-01-29	38800	39300	38300	38600	237477
2008-01-30	36700	38200	36200	36800	302336
2008-01-31	38000	40800	37550	40800	438912
2008-02-01	42600	43800	40850	42000	456223
2008-02-04	46000	46000	46000	46000	20272
2008-02-05	50000	50000	47400	48300	708301
2008-02-06	46900	47700	45650	45650	336823
2008-02-07	45450	46350	44800	45200	232976
2008-02-08	45000	47950	44900	46600	178174
2008-02-12	47800	48100	46800	47550	187470
2008-02-13	47700	47900	46900	47500	178199
2008-02-14	48300	49800	48000	49150	160685
2008-02-15	49550	51400	49300	51400	234491
2008-02-18	51400	52100	50300	51300	180379
2008-02-19	50800	51200	49550	49850	155547
2008-02-20	49200	49900	46950	47450	235697
2008-02-21	48500	49500	47850	48550	153497
2008-02-22	49750	49750	47850	48150	149031
2008-02-25	48300	49700	48000	49650	128455
2008-02-26	50600	50600	48750	48900	113487
2008-02-27	49950	50000	49400	49600	105259
2008-02-28	48400	49600	48300	48700	92424
2008-02-29	47750	48100	45500	47050	187128
2008-03-03	45200	46250	44950	44950	122299
2008-03-04	45000	45550	43200	43900	192043
2008-03-05	44700	46050	44350	45050	160022
2008-03-06	47450	49050	46750	47500	193780
2008-03-07	46200	47000	45600	46100	157384
2008-03-10	45700	46150	43500	43600	153186
2008-03-11	43000	44800	42350	44300	145571
2008-03-12	46700	47400	46200	46750	154352
2008-03-13	46650	47250	45500	47100	131201
2008-03-14	47150	47150	45100	46250	186966
2008-03-17	45500	47000	43850	46200	133137
2008-03-18	46600	47000	44600	45850	146796
2008-03-19	47450	47850	46350	47850	145105
2008-03-21	47850	49350	47550	49250	127138
2008-03-24	49900	51000	49450	50500	165514
2008-03-25	51000	52400	50600	51900	178573
2008-03-26	52800	54000	52700	53800	187101
2008-03-27	53300	53500	52400	52900	90449
2008-03-28	53000	54600	52400	53600	122651
2008-03-31	52600	54500	51300	51800	167128
2008-04-01	51800	52400	49900	50900	158196
2008-04-02	51400	52300	50300	52300	242687
2008-04-03	52300	53700	51900	53300	113588
2008-04-04	53500	53500	51300	51600	107387
2008-04-07	52900	55400	52600	54300	346456
2008-04-08	54700	55000	51300	51400	196144
2008-04-09	50900	51200	48550	50200	270068
2008-04-10	51500	53200	50400	51600	254440
2008-04-11	53100	53200	51800	53000	132829
2008-04-14	52500	52900	51700	52000	72317
2008-04-15	52200	53500	52200	52900	81334
2008-04-16	53300	53300	51800	52200	80124
2008-04-17	53200	53900	52600	53000	89380
2008-04-18	53900	53900	52000	53000	106387
2008-04-21	54000	54000	52200	53200	126641
2008-04-22	53100	53200	52600	53000	65177
2008-04-23	52100	54200	51700	52700	114928
2008-04-24	52700	53100	51500	51500	91056
2008-04-25	51700	52200	51200	52100	73256
2008-04-28	49700	50400	47100	47150	414051
2008-04-30	46800	47350	44750	46050	361447
2008-05-01	46250	47650	45750	46550	206049
2008-05-02	47500	47500	46750	47250	142574
2008-05-07	45650	46800	45100	45700	389433
2008-05-08	44900	44900	43000	43250	416340
2008-05-09	44050	44850	43650	43900	210318
2008-05-12	44300	44750	43800	44450	136845
2008-05-13	44850	45850	44500	45450	159438
2008-05-14	46250	46350	45450	45850	154668
2008-05-15	46200	46300	45000	45000	158969
2008-05-16	46000	46000	45100	45200	126175
2008-05-19	45600	45700	44800	45100	126051
2008-05-20	45150	45300	42100	43300	252845
2008-05-21	42950	43850	42500	43400	159989
2008-05-22	43450	43800	42500	43800	165281
2008-05-23	44400	44950	43800	43850	189978
2008-05-26	45850	46550	45550	45850	280665
2008-05-27	46150	46200	45150	45150	123768
2008-05-28	45300	45700	44650	44800	132500
2008-05-29	45200	45300	44200	44200	155951
2008-05-30	44500	45500	44300	45350	113787
2008-06-02	45550	46500	45450	46000	267163
2008-06-03	45900	46000	44050	44550	243592
2008-06-04	44700	44850	44050	44700	218771
2008-06-05	44950	45600	44650	44850	181809
2008-06-06	45100	45300	44050	44200	159266
2008-06-09	43800	43800	42800	43750	131072
2008-06-10	43850	43900	42200	42200	197838
2008-06-11	42200	42700	41600	41950	220789
2008-06-12	41750	42250	41000	42200	149557
2008-06-13	42200	42250	41100	41550	226378
2008-06-16	41700	42300	41300	42100	155957
2008-06-17	42000	43050	41900	42950	168921
2008-06-18	43150	44750	43100	44450	209336
2008-06-19	44350	44400	42500	42900	194757
2008-06-20	43100	43700	42050	42550	139911
2008-06-23	42150	42300	41550	41950	133844
2008-06-24	41700	41750	41050	41250	122990
2008-06-25	41250	41300	40350	40550	180486
2008-06-26	40650	42600	40650	42100	182042
2008-06-27	41500	41700	40900	41300	157169
2008-06-30	41100	41250	40500	40900	135399
2008-07-01	40750	41100	40400	41000	134145
2008-07-02	41050	41600	40400	40800	173628
2008-07-03	40500	41800	40100	41600	217018
2008-07-04	41750	42750	41200	41650	197941
2008-07-07	41850	43050	41500	42800	119325
2008-07-08	43600	43650	42200	42450	170522
2008-07-09	42700	43050	41550	41750	140568
2008-07-10	41650	43300	41600	42550	126560
2008-07-11	42400	42400	40450	41000	131417
2008-07-14	40600	40650	39500	39750	171936
2008-07-15	39050	39250	38250	38700	170635
2008-07-16	38000	38200	36550	37050	258120
2008-07-17	37900	38950	37800	38550	179245
2008-07-18	39100	39200	37650	38250	172812
2008-07-22	37650	37900	37050	37600	136874
2008-07-23	38000	39700	37800	39350	146348
2008-07-24	40150	41500	39600	41250	191707
2008-07-25	40050	40800	39800	40000	93024
2008-07-28	41200	42750	41000	41700	233437
2008-07-29	40500	41250	40000	40700	104627
2008-07-30	41200	41600	41150	41450	80124
2008-07-31	41750	41850	40750	41150	77774
2008-08-01	40350	40450	38750	39050	143246
2008-08-04	38600	39200	37550	38050	128320
2008-08-05	38000	38050	37050	37300	136634
2008-08-06	37500	38100	37500	37650	134520
2008-08-07	37700	38050	37100	37850	153776
2008-08-08	37200	37850	37200	37750	106235
2008-08-11	38100	39300	38050	38850	107548
2008-08-12	38900	39350	38500	38900	93638
2008-08-13	38200	38450	37200	37350	139289
2008-08-14	36650	37250	36200	36600	164297
2008-08-15	36850	37400	36800	37400	79812
2008-08-18	37300	37900	37100	37400	65262
2008-08-19	36700	38600	36700	38500	131676
2008-08-20	37900	40350	37750	40000	195695
2008-08-21	41200	41400	40550	41150	207137
2008-08-22	41150	42250	40300	42200	170789
2008-08-25	42300	42400	40700	40950	96811
2008-08-26	40500	42000	40100	41750	62726
2008-08-27	41700	42800	41600	42150	107813
2008-08-28	42550	42600	41500	41950	60977
2008-08-29	40950	42300	40800	42050	89162
2008-09-01	41450	41850	40850	41000	69291
2008-09-02	41000	41550	39550	40500	79656
2008-09-03	40100	41300	39750	40350	83997
2008-09-04	40400	41200	39950	40700	89819
2008-09-05	39700	40850	39250	40800	160502
2008-09-08	41500	41850	41050	41500	73746
2008-09-09	41400	41400	39900	40050	67160
2008-09-10	39250	41700	39250	41400	92293
2008-09-11	40750	41300	39850	40300	122390
2008-09-12	40350	40800	39900	40150	77077
2008-09-16	36550	36550	36150	36150	381895
2008-09-17	36500	36600	32800	35850	416353
2008-09-18	34500	36600	34350	35900	201857
2008-09-19	37000	37750	35850	36050	182011
2008-09-22	36450	37000	35050	35600	149655
2008-09-24	34800	35200	34050	34550	187207
2008-09-25	34800	35250	34400	34700	127398
2008-09-26	35500	35600	34650	35000	133663
2008-09-29	35000	36150	34800	34900	97205
2008-09-30	32500	34450	32250	33700	104702
2008-10-01	33700	34900	33000	33000	129048
2008-10-02	33200	33850	32600	33200	125962
2008-10-03	32500	33100	31800	31850	129921
2008-10-06	29550	30450	28980	29940	242117
2008-10-07	28140	31900	28120	30500	249394
2008-10-08	29700	29850	28780	28970	322169
2008-10-09	28070	29430	27880	28070	223575
2008-10-10	25670	27270	25600	26320	167214
2008-10-14	28500	29280	27720	27960	236488
2008-10-15	27360	30000	27070	29770	308487
2008-10-16	27970	27970	26780	26850	234567
2008-10-17	29550	29850	28690	29850	302454
2008-10-20	30950	31750	29530	30350	189164
2008-10-21	32350	32400	30100	30650	144616
2008-10-22	29850	30850	28330	28500	191954
2008-10-23	27000	27730	26440	27400	217964
2008-10-24	26900	27000	26180	26530	185889
2008-10-27	28030	29470	26920	28260	379567
2008-10-28	28260	30000	28190	28760	344561
2008-10-29	30950	31800	29900	31600	228019
2008-10-30	30800	31150	29910	30700	317209
2008-10-31	30700	32600	30150	31850	190841
2008-11-04	32650	33850	32100	33450	161461
2008-11-05	35050	35800	34250	35800	233082
2008-11-06	34800	35300	34100	34750	222489
2008-11-07	34000	35000	32500	34000	203435
2008-11-10	34800	35100	34600	34950	153371
2008-11-11	34500	35700	34150	34850	162487
2008-11-12	34100	34550	33500	34050	148250
2008-11-13	33000	33250	32000	32700	194190
2008-11-14	33900	34000	32350	33000	105099
2008-11-17	32200	34100	31850	33150	137673
2008-11-18	33200	33300	31000	31350	165565
2008-11-19	32150	32550	31550	32550	128001
2008-11-20	31500	31700	30750	30750	137598
2008-11-21	30000	30400	28720	30100	229069
2008-11-25	30350	31150	29540	30150	174423
2008-11-26	28850	30250	28850	29320	52865
2008-11-27	29520	30150	29520	29900	71076
2008-11-28	29910	31350	29610	31000	111608
2008-12-01	30650	30700	29950	30400	76330
2008-12-02	29000	29280	28600	28640	102419
2008-12-03	29480	30700	29370	30550	206798
2008-12-04	30950	31550	30100	30700	139425
2008-12-05	31500	31500	30450	30500	86210
2008-12-08	31300	33650	31200	33000	145317
2008-12-09	33700	33900	33150	33300	129234
2008-12-10	33300	33800	32450	33700	134893
2008-12-11	34100	35000	33800	35000	157461
2008-12-12	35050	35600	33150	33600	178172
2008-12-15	34000	34350	33750	33850	78101
2008-12-16	34250	35250	34150	34850	107422
2008-12-17	35450	35750	33850	34400	101096
2008-12-18	34800	35150	34450	34750	84533
2008-12-19	34700	35650	34600	35150	172347
2008-12-22	35600	37100	35250	36900	149538
2008-12-24	37000	37100	35600	36150	67207
2008-12-25	35850	36700	35850	36150	30176
2008-12-26	36450	36500	36150	36500	25001
2008-12-29	36850	37000	36600	36950	41210
2008-12-30	37000	37000	36250	36500	40737
2009-01-05	37150	37450	36550	37000	48382
2009-01-06	37300	40300	37200	38400	222607
2009-01-07	38400	39900	35100	36050	225114
2009-01-08	36200	36300	33500	33950	203189
2009-01-09	34750	34850	33900	34100	132310
2009-01-13	34050	34050	33050	33450	70888
2009-01-14	33050	33250	32200	32300	105613
2009-01-15	31500	32700	31050	31950	125157
2009-01-16	32150	32850	32100	32200	86489
2009-01-19	32950	33100	31450	31700	87701
2009-01-20	31700	31800	30500	30850	127819
2009-01-21	30000	30900	29950	30200	93142
2009-01-22	30550	31200	30150	30900	85875
2009-01-23	30950	32200	30900	30950	123499
2009-01-26	31350	31650	30650	31200	63029
2009-01-27	30900	31400	30200	31250	164271
2009-01-28	32050	32150	29370	29950	392676
2009-01-29	30250	30350	28920	29080	262025
2009-01-30	28260	29110	28000	28990	205971
2009-02-02	28500	28710	27770	28170	157671
2009-02-03	28470	29830	28300	28760	116263
2009-02-04	29000	29400	28920	29340	105599
2009-02-05	29200	29320	28140	28340	105705
2009-02-06	28640	28770	27790	27900	99569
2009-02-09	28300	28380	27500	27800	91960
2009-02-10	27700	28110	27000	27440	125904
2009-02-12	27010	27290	26550	26630	111438
2009-02-13	26930	28590	26820	28190	142625
2009-02-16	28190	29020	28150	28590	94458
2009-02-17	28400	28480	27520	27560	81026
2009-02-18	27590	28260	27300	28210	79276
2009-02-19	28220	28670	28050	28140	78357
2009-02-20	27840	28020	25900	25970	177838
2009-02-23	25730	25800	24140	25100	197891
2009-02-24	24500	26100	24030	25320	177015
2009-02-25	25800	26390	25400	26260	133112
2009-02-26	26280	26890	26170	26460	92326
2009-02-27	27000	28380	26890	28270	211505
2009-03-02	27070	27730	26620	27530	110323
2009-03-03	26330	26520	25000	25060	279974
2009-03-04	24500	25450	24290	25130	154472
2009-03-05	25500	25970	25300	25350	107302
2009-03-06	25000	25080	24350	24830	95003
2009-03-09	24530	24720	22700	22900	221706
2009-03-10	22900	23750	22790	23570	155399
2009-03-11	24240	24570	23990	24100	112850
2009-03-12	24100	24180	22430	22590	196062
2009-03-13	22680	23660	22680	23530	148660
2009-03-16	23660	24680	23610	24170	83282
2009-03-17	24680	25430	24250	25340	148610
2009-03-18	26540	26600	24810	24830	178691
2009-03-19	25000	25720	24890	25520	94855
2009-03-23	26500	27930	25900	27250	218426
2009-03-24	28150	28250	27550	27760	123710
2009-03-25	27000	28060	26990	27950	132307
2009-03-26	27990	28050	27340	27770	101985
2009-03-27	27800	28470	27730	27790	138814
2009-03-30	27710	28400	27000	27000	108706
2009-03-31	26400	27220	25670	25870	96626
2009-04-01	25990	26980	25920	26820	96830
2009-04-02	27420	27580	27010	27380	77298
2009-04-03	27520	27750	27070	27260	91533
2009-04-06	27100	27260	26510	26720	110678
2009-04-07	27180	28220	26850	27800	224009
2009-04-08	28100	28320	27230	27330	178192
2009-04-09	27400	28110	27400	27840	94466
2009-04-10	28140	28240	27560	27770	138909
2009-04-13	27800	27940	26840	27030	103786
2009-04-14	27000	27830	26870	27510	137745
2009-04-15	27500	27830	27190	27250	71275
2009-04-16	27260	27380	26850	26990	150774
2009-04-17	27010	27240	26750	26900	125086
2009-04-20	26920	27000	26360	26460	82767
2009-04-21	25710	25930	25270	25800	105620
2009-04-22	26100	26470	25580	26190	112731
2009-04-23	26050	27760	25800	27540	192509
2009-04-24	27240	27330	26210	26340	140358
2009-04-27	26940	27130	26360	26560	90338
2009-04-28	26790	26920	26350	26350	115956
2009-04-30	26350	26800	24610	24610	329831
2009-05-01	24910	24910	24270	24510	189462
2009-05-07	25200	25260	24260	24480	259047
2009-05-08	24480	25340	24300	25280	203183
2009-05-11	25800	25860	25310	25530	166660
2009-05-12	25310	25480	25010	25020	98830
2009-05-13	25030	25510	24720	25470	108773
2009-05-14	25100	25280	25000	25130	81782
2009-05-15	25290	25450	25190	25190	86627
2009-05-18	25490	25570	25030	25200	99809
2009-05-19	25200	25300	24820	24950	114542
2009-05-20	25250	25250	24920	25100	90364
2009-05-21	25100	25180	24890	24990	97390
2009-05-22	24750	25000	24750	25000	70201
2009-05-25	24840	25110	24750	24880	127332
2009-05-26	24910	24990	24800	24870	108175
2009-05-27	25200	25770	25070	25630	295472
2009-05-28	25660	26070	25370	25600	258630
2009-05-29	25860	25960	25190	25360	134763
2009-06-01	24760	25030	24740	24880	205529
2009-06-02	24930	25080	24810	24860	146018
2009-06-03	24870	25980	24860	25940	241588
2009-06-04	25920	26050	25120	25870	189168
2009-06-05	26220	27630	26030	27280	340037
2009-06-08	27500	27840	26960	27240	172224
2009-06-09	27300	27400	26480	26670	129820
2009-06-10	26750	27500	26510	27500	161988
2009-06-11	27600	28320	27600	28210	288151
2009-06-12	30000	31000	29350	30550	598224
2009-06-15	31750	32150	30600	31950	376205
2009-06-16	31150	31900	30600	31050	270244
2009-06-17	30650	30900	29690	29730	216454
2009-06-18	29580	30800	28870	30000	268328
2009-06-19	30000	31300	29820	31050	248030
2009-06-22	31150	31550	30350	30500	119009
2009-06-23	29400	29650	28820	28830	244717
2009-06-24	29400	29450	28030	28250	288539
2009-06-25	28570	29470	28360	29390	209153
2009-06-26	29580	29930	29340	29780	168295
2009-06-29	29750	29860	28710	28790	120849
2009-06-30	29650	30800	29400	30750	254335
2009-07-01	30500	30950	29850	30150	147961
2009-07-02	29910	30100	29300	29350	149190
2009-07-03	28950	29660	28880	29600	95310
2009-07-06	29900	30650	29410	30550	170994
2009-07-07	31100	31550	30700	30800	174767
2009-07-08	30550	30650	29440	29740	171107
2009-07-09	29500	29590	28830	28880	147192
2009-07-10	29200	29480	28500	28520	103506
2009-07-13	28490	29190	27790	27870	154158
2009-07-14	28100	28440	28020	28140	83802
2009-07-15	28440	28750	28170	28260	88212
2009-07-16	28990	29120	28300	28500	81385
2009-07-17	28350	29470	28310	29460	128319
2009-07-21	30500	30700	29530	29830	118126
2009-07-22	30050	30500	29800	30500	117205
2009-07-23	30550	31000	30100	30200	101562
2009-07-24	30650	31400	30350	31350	160922
2009-07-27	31850	32250	31200	31300	147564
2009-07-28	31600	31900	30600	30950	101024
2009-07-29	31050	33200	31050	33100	318412
2009-07-30	32300	32550	31550	32200	165224
2009-07-31	32250	32400	30600	31050	276273
2009-08-03	31100	31850	31000	31600	118826
2009-08-04	32150	32150	31550	31600	78305
2009-08-05	31650	31900	31200	31250	46768
2009-08-06	31000	31150	30600	30900	86842
2009-08-07	31200	31600	30750	31000	67691
2009-08-10	31550	31600	30850	30900	63049
2009-08-11	30900	31000	30650	30700	61615
2009-08-12	30850	30850	30400	30700	71354
2009-08-13	30950	31100	30500	30950	85434
2009-08-14	30800	31100	30550	30550	62390
2009-08-17	30600	30750	30000	30000	68016
2009-08-18	29630	30150	29520	30050	81422
2009-08-19	30400	30400	29700	29780	77704
2009-08-20	29820	29990	29580	29850	80295
2009-08-21	29990	30000	29200	29290	107451
2009-08-24	30050	30200	29630	29770	84648
2009-08-25	29780	29940	29540	29730	86421
2009-08-26	29970	29970	29580	29920	81561
2009-08-27	29660	30450	29660	30150	145002
2009-08-28	30600	31400	30400	31400	218665
2009-08-31	31000	31950	30550	31650	177044
2009-09-01	31700	32350	31350	32150	175857
2009-09-02	31750	31850	31300	31750	124583
2009-09-03	31800	32000	31500	31600	86112
2009-09-04	31500	31550	30150	30250	128351
2009-09-07	31000	31000	30100	30300	73069
2009-09-08	30100	30200	29230	29500	160782
2009-09-09	29600	29760	29340	29470	85887
2009-09-10	29400	30900	29400	30850	123243
2009-09-11	30900	30900	29950	30600	140438
2009-09-14	30650	30800	30100	30650	60639
2009-09-15	30400	31300	30400	30850	130429
2009-09-16	30850	31150	29740	29840	134531
2009-09-17	29880	30050	29700	29920	83883
2009-09-18	29500	29800	29300	29440	112368
2009-09-24	29820	30250	29700	29930	117311
2009-09-25	29860	29870	29180	29430	91254
2009-09-28	29100	29940	29010	29370	173529
2009-09-29	29580	29790	29320	29470	95761
2009-09-30	29750	30500	29600	30500	143315
2009-10-01	30450	30450	29520	29640	91487
2009-10-02	29080	29170	27870	28160	140924
2009-10-05	28170	28650	27820	27890	74732
2009-10-06	27970	28070	27180	27560	123080
2009-10-07	27570	27930	27420	27820	68570
2009-10-08	27780	28030	27220	27760	139254
2009-10-09	28960	30400	28900	29870	277558
2009-10-13	30000	30200	29560	29670	133326
2009-10-14	29490	29950	29170	29270	84706
2009-10-15	29130	29470	28780	28950	110098
2009-10-16	28990	29230	28240	28700	116444
2009-10-19	28710	29070	28410	29030	72379
2009-10-20	29320	29410	28330	28740	198772
2009-10-21	29040	29680	28810	29540	147750
2009-10-22	29450	29660	28990	29530	108137
2009-10-23	29230	29890	28930	29030	123808
2009-10-26	28890	29010	28550	28850	92352
2009-10-27	28840	29790	28730	29430	167768
2009-10-28	29130	29700	28020	28020	185913
2009-10-29	27800	27980	27500	27980	236067
2009-10-30	28200	28340	27700	28030	102257
2009-11-02	27500	27980	27350	27960	86399
2009-11-04	27850	27850	27450	27760	71796
2009-11-05	27670	27800	27200	27430	101139
2009-11-06	27500	27500	27060	27290	83772
2009-11-09	27150	27390	26870	27000	91989
2009-11-10	26900	27070	26480	26610	119512
2009-11-11	26420	26500	26110	26300	134854
2009-11-12	27370	27830	26950	27500	280070
2009-11-13	27200	27470	26960	27160	120285
2009-11-16	27000	27260	26300	26440	78915
2009-11-17	26300	26540	26050	26240	66822
2009-11-18	26000	26060	25320	25770	121916
2009-11-19	25920	25950	25110	25360	113049
2009-11-20	25300	25930	25300	25910	87239
2009-11-24	26020	26140	25520	25730	88176
2009-11-25	25510	25800	25400	25670	74256
2009-11-26	25500	25950	25470	25780	60268
2009-11-27	25670	25830	25310	25320	75150
2009-11-30	25310	25910	25110	25830	89412
2009-12-01	25230	25840	25230	25800	119658
2009-12-02	25810	25810	25470	25710	149340
2009-12-03	26000	26810	25990	26610	155049
2009-12-04	26610	26800	26350	26720	117660
2009-12-07	27300	27300	26620	26880	124747
2009-12-08	27480	28500	27370	28280	300165
2009-12-09	28270	29930	28250	29670	331760
2009-12-10	29070	29190	28500	28800	147748
2009-12-11	29100	29350	28880	29180	140837
2009-12-14	29020	29120	28670	28950	79271
2009-12-15	28650	29050	28650	28940	43206
2009-12-16	28700	29290	28700	28990	93739
2009-12-17	28880	28990	28090	28120	92945
2009-12-18	28520	28610	27630	27910	149772
2009-12-21	27650	28570	27650	28070	122791
2009-12-22	27770	27880	27450	27610	112602
2009-12-24	27650	28240	27440	28060	116459
2009-12-25	28000	28680	27790	27990	81280
2009-12-28	28020	28370	28010	28010	81117
2009-12-29	28080	28210	27850	27950	50807
2009-12-30	27980	28140	27840	27840	45979
2010-01-04	27690	28320	27670	28190	51514
2010-01-05	29000	29000	28650	28860	111257
2010-01-06	29080	29140	28430	29010	97505
2010-01-07	29240	29640	29100	29480	113788
2010-01-08	29780	29780	28650	28850	120008
2010-01-12	29100	29200	28720	29090	107365
2010-01-13	28800	29200	28780	28830	81629
2010-01-14	28770	29060	28500	29010	73787
2010-01-15	29280	29570	29020	29500	85925
2010-01-18	29400	29760	29090	29160	85221
2010-01-19	29340	29340	28990	29010	52544
2010-01-20	29500	30500	29240	30250	224612
2010-01-21	30450	31400	29840	31350	231315
2010-01-22	31050	31800	30850	31550	199049
2010-01-25	31550	32250	31150	32000	183278
2010-01-26	32150	32600	31300	31350	201581
2010-01-27	32050	32850	31200	31350	224316
2010-01-28	32500	33950	31750	33800	394027
2010-01-29	33850	34750	33650	34250	244418
2010-02-01	34050	34700	33700	34600	158412
2010-02-02	34300	34350	33400	33550	234341
2010-02-03	33450	34300	33200	33750	200915
2010-02-04	34000	34050	31850	32250	261236
2010-02-05	31950	32250	31650	31850	134858
2010-02-08	32400	32900	32200	32250	158580
2010-02-09	32000	32350	31900	32200	103558
2010-02-10	32900	32900	32000	32050	82062
2010-02-12	32250	32300	31500	31550	93095
2010-02-15	31650	31750	31250	31350	54067
2010-02-16	31200	31600	31050	31250	38249
2010-02-17	31800	31850	31150	31250	64903
2010-02-18	31600	32400	31450	31750	138398
2010-02-19	31750	31800	31200	31250	61121
2010-02-22	31400	32000	31350	31950	56265
2010-02-23	32650	32950	32150	32900	111780
2010-02-24	32600	32850	32300	32450	81025
2010-02-25	32750	33300	32550	32950	145309
2010-02-26	33300	33700	32950	33250	200896
2010-03-01	33350	33500	32750	32750	55753
2010-03-02	32450	32650	32000	32500	76232
2010-03-03	32700	32950	32250	32350	81858
2010-03-04	32800	33350	32400	33100	141174
2010-03-05	33500	34300	33350	34150	212772
2010-03-08	34600	35850	34450	35750	263291
2010-03-09	35900	36150	35250	35350	143069
2010-03-10	34850	35150	34300	34750	129759
2010-03-11	35050	35100	34100	34600	95221
2010-03-12	34650	35650	34500	35550	118623
2010-03-15	35200	35400	34600	34700	79593
2010-03-16	34550	34700	34300	34450	66805
2010-03-17	34650	35500	34550	35400	98911
2010-03-18	35400	35400	34550	34600	80278
2010-03-19	34900	35500	34700	35300	121527
2010-03-23	34850	35200	34700	34800	55017
2010-03-24	34950	35950	34850	35800	133494
2010-03-25	35750	35750	34850	34850	92541
2010-03-26	34900	34950	34450	34700	94563
2010-03-29	34150	34450	33700	34150	106301
2010-03-30	34400	34800	34100	34550	127625
2010-03-31	34500	34600	33750	34050	144881
2010-04-01	34550	35550	34350	35050	179713
2010-04-02	35250	35850	35100	35400	111193
2010-04-05	35700	35750	35200	35550	62406
2010-04-06	35700	36700	35500	36450	203905
2010-04-07	36500	37050	36250	36350	104392
2010-04-08	36150	36650	36100	36350	78847
2010-04-09	36200	36800	35800	36050	90422
2010-04-12	36150	37000	36050	36600	116174
2010-04-13	36250	36450	35400	35650	134674
2010-04-14	35700	35900	35200	35350	100553
2010-04-15	35950	35950	35300	35500	86899
2010-04-16	35250	35650	34950	35100	114920
2010-04-19	34400	35500	34350	35150	122107
2010-04-20	35150	35550	34750	35000	101856
2010-04-21	35350	35350	34650	34900	89433
2010-04-22	34550	34800	34100	34650	68360
2010-04-23	34650	36150	34400	36050	202572
2010-04-26	36400	37200	36400	36950	176963
2010-04-27	37150	38500	36900	38400	292540
2010-04-28	37000	37300	34450	36000	323406
2010-04-30	36600	36800	35800	36000	162746
2010-05-06	35200	35550	34800	34900	147548
2010-05-07	34000	34650	33650	34200	151797
2010-05-10	33800	34800	33700	34550	120531
2010-05-11	35350	35550	34400	34400	119160
2010-05-12	34800	35000	34200	34400	67771
2010-05-13	35000	35150	34500	35000	66250
2010-05-14	34300	35000	34250	34600	63285
2010-05-17	34000	34450	33650	33850	76455
2010-05-18	33950	34400	33600	33750	65460
2010-05-19	33200	33500	32700	33350	83989
2010-05-20	33250	33350	32850	33050	86297
2010-05-21	31950	32300	31700	31950	106784
2010-05-24	32200	32800	32100	32250	83701
2010-05-25	31950	31950	30750	30850	124874
2010-05-26	31000	31400	30600	30800	90189
2010-05-27	30450	31800	30150	31650	101934
2010-05-28	32200	32400	31800	31950	79141
2010-05-31	31300	32050	30950	32000	132717
2010-06-01	33000	33650	32700	32800	226058
2010-06-02	32250	32550	31800	32200	151695
2010-06-03	32550	33400	32300	33100	141380
2010-06-04	33300	33650	33150	33500	67131
2010-06-07	32800	33000	31900	32250	118582
2010-06-08	32250	33600	32050	33450	108242
2010-06-09	33100	34150	32900	33650	166956
2010-06-10	33650	33750	32900	33700	106063
2010-06-11	34500	34600	33950	34100	102666
2010-06-14	34500	34750	34400	34600	40248
2010-06-15	34800	34900	33900	34050	84592
2010-06-16	34500	34600	34150	34350	67504
2010-06-17	34500	34500	34050	34350	56917
2010-06-18	34400	35100	34100	35000	96623
2010-06-21	35500	36350	35450	36350	141591
2010-06-22	35900	36250	35850	36050	56786
2010-06-23	36150	36450	35800	35950	113430
2010-06-24	35550	36400	35550	36100	72939
2010-06-25	35650	36050	35450	36000	142748
2010-06-28	36050	36100	35550	35650	78437
2010-06-29	36200	36400	36000	36150	97480
2010-06-30	35450	35850	35350	35650	82362
2010-07-01	35100	35100	34200	34500	149198
2010-07-02	35150	36100	34650	35300	120699
2010-07-05	35650	36200	35450	36000	61627
2010-07-06	35700	36750	35650	36750	115321
2010-07-07	36500	36550	35900	36450	111800
2010-07-08	36750	36950	36150	36350	111062
2010-07-09	35950	36200	35600	35700	146387
2010-07-12	35050	35450	34800	34850	77865
2010-07-13	35000	35350	34800	35100	107757
2010-07-14	35250	35550	34900	35000	89304
2010-07-15	34550	34750	33350	33500	184300
2010-07-16	32900	33550	32550	32850	132222
2010-07-20	32500	33900	32400	33550	146199
2010-07-21	33500	34150	33150	33500	119412
2010-07-22	33000	33500	32700	33150	79781
2010-07-23	33600	34100	33450	33650	99962
2010-07-26	34150	34800	34100	34750	101058
2010-07-27	35550	36000	34850	35150	170652
2010-07-28	35200	35300	34000	34100	180603
2010-07-29	34150	34500	33450	33500	100677
2010-07-30	33950	34100	33250	33250	77744
2010-08-02	33050	33250	32500	32550	120458
2010-08-03	32800	32950	32000	32300	129394
2010-08-04	32000	32150	31300	31400	147279
2010-08-05	31650	31750	31200	31450	91634
2010-08-06	31000	31200	30500	31150	140624
2010-08-09	31100	31950	30850	31950	101565
2010-08-10	31800	32100	31500	31900	98138
2010-08-11	31450	31550	30900	31100	93796
2010-08-12	30400	30750	30050	30250	129398
2010-08-13	30500	30850	30400	30600	118186
2010-08-16	29980	30450	29760	30400	75046
2010-08-17	30100	30700	30000	30600	49433
2010-08-18	30750	31000	30200	30400	81329
2010-08-19	30250	30650	30050	30600	73115
2010-08-20	30300	30650	30200	30400	45702
2010-08-23	30200	30250	29810	29940	114061
2010-08-24	29690	30150	29390	30000	130850
2010-08-25	29890	30300	29660	30000	100908
2010-08-26	30300	31850	30000	31700	229378
2010-08-27	31400	31550	30900	31250	124428
2010-08-30	31450	31750	31150	31300	65884
2010-08-31	31350	31550	30150	30250	101236
2010-09-01	29570	30650	29130	30500	172612
2010-09-02	30700	30900	30300	30650	50029
2010-09-03	30850	31200	30800	31050	66605
2010-09-06	31150	31250	30750	30950	52824
2010-09-07	30800	30950	30500	30700	49624
2010-09-08	30650	30700	29800	30000	111280
2010-09-09	30000	30250	29930	30100	64183
2010-09-10	30050	30250	29840	29960	126749
2010-09-13	30200	30600	30050	30500	50644
2010-09-14	30600	30800	30400	30450	35859
2010-09-15	30600	31100	30500	30850	88323
2010-09-16	30950	31000	30000	30100	67756
2010-09-17	29850	30100	29020	29090	241020
2010-09-21	29900	29970	29380	29610	142076
2010-09-22	29260	29500	28920	29130	112659
2010-09-24	29140	29940	29020	29490	220670
2010-09-27	29500	29590	28990	28990	135604
2010-09-28	29180	29260	29010	29050	148809
2010-09-29	29050	29580	29040	29270	126988
2010-09-30	29200	29290	28840	28840	122130
2010-10-01	28840	28950	28350	28520	106574
2010-10-04	28770	28980	27500	27700	167735
2010-10-05	27820	27960	27360	27570	150315
2010-10-06	28070	28530	27920	28520	157706
2010-10-07	30050	30050	29160	29580	260664
2010-10-08	29700	30250	29610	29920	210223
2010-10-12	30300	30350	29300	29350	156293
2010-10-13	29400	29650	28680	28770	132034
2010-10-14	29700	30650	29520	30600	247378
2010-10-15	30500	30500	29710	30300	143886
2010-10-18	30250	30300	29810	30150	96161
2010-10-19	30050	30050	29190	29490	147070
2010-10-20	29550	29550	28540	28610	186685
2010-10-21	28630	29250	28020	28120	196506
2010-10-22	28400	28400	27770	28040	198322
2010-10-25	28400	29560	28310	28810	241146
2010-10-26	28220	28600	27850	28120	159293
2010-10-27	28290	28690	28000	28550	160565
2010-10-28	28970	28970	28210	28260	214769
2010-10-29	28100	28460	27930	28160	105417
2010-11-01	27880	28090	27610	27670	95903
2010-11-02	27510	27690	27230	27380	97621
2010-11-04	27730	28260	27640	27980	97092
2010-11-05	28350	28540	28160	28240	74302
2010-11-08	28540	28850	28350	28760	77401
2010-11-09	28820	28890	28640	28710	103535
2010-11-10	28880	29140	28640	28800	91601
2010-11-11	28990	29250	28740	29210	120313
2010-11-12	29220	29470	28950	29120	120517
2010-11-15	29070	29250	28820	28880	88735
2010-11-16	28890	29700	28890	29280	162605
2010-11-17	28950	29250	28860	28920	97577
2010-11-18	29150	29390	28910	29310	98868
2010-11-19	29750	29880	29530	29640	105334
2010-11-22	29800	30050	29610	29730	92645
2010-11-24	29240	29700	29000	29550	61954
2010-11-25	29920	30700	29790	30450	136936
2010-11-26	30350	30600	30050	30050	58264
2010-11-29	30200	30550	29990	30050	56178
2010-11-30	30050	30300	29890	30000	63235
2010-12-01	29830	30750	29710	30650	97491
2010-12-02	30850	31500	30450	31300	134385
2010-12-03	31900	32300	31600	31800	107175
2010-12-06	32000	32200	31650	32200	87912
2010-12-07	31700	31900	31200	31300	85425
2010-12-08	31350	31650	31100	31250	115485
2010-12-09	31200	31350	30950	31100	55525
2010-12-10	31000	31200	30700	31000	116271
2010-12-13	31000	31400	30850	31350	38220
2010-12-14	31400	32000	31300	31850	76179
2010-12-15	32000	32300	31850	32050	125479
2010-12-16	31900	32700	31900	32500	107058
2010-12-17	32550	32950	32350	32450	103734
2010-12-20	32250	32600	32050	32150	44386
2010-12-21	32150	32500	32150	32350	44899
2010-12-22	32250	32450	31950	32100	63865
2010-12-24	31800	32050	31750	31900	66397
2010-12-27	32100	32200	31800	31850	36607
2010-12-28	31700	31850	31650	31800	25469
2010-12-29	31650	32000	31650	31950	22166
2010-12-30	31950	31950	31500	31500	30614
2011-01-04	31650	31850	31500	31700	59831
2011-01-05	31050	31400	30150	30200	160383
2011-01-06	30200	30600	30050	30250	114564
2011-01-07	30250	30550	30100	30200	77564
2011-01-11	30100	30550	29560	30450	106584
2011-01-12	30600	30900	30500	30750	69012
2011-01-13	31050	31500	30750	31450	100493
2011-01-14	31350	31700	31150	31250	84267
2011-01-17	31500	31550	31150	31250	50070
2011-01-18	31000	31450	31000	31250	59692
2011-01-19	31200	31650	31150	31600	58818
2011-01-20	31200	31750	30950	31150	118013
2011-01-21	31150	31350	30650	30950	104277
2011-01-24	31300	31450	31050	31300	64200
2011-01-25	31500	32200	31200	31950	108765
2011-01-26	32000	32700	31950	32450	196743
2011-01-27	32450	32500	31850	32200	128867
2011-01-28	32050	32200	31150	31350	124855
2011-01-31	31350	31400	30700	30950	108463
2011-02-01	31150	31300	30650	30900	104723
2011-02-02	31350	31800	31150	31250	84770
2011-02-03	31200	31300	30900	31100	67679
2011-02-04	31050	31150	30600	30700	93010
2011-02-07	31250	31250	30450	30650	98623
2011-02-08	31000	31000	30650	30850	72203
2011-02-09	30900	30950	30600	30750	64241
2011-02-10	30750	30900	30450	30850	61805
2011-02-14	30800	30850	30600	30700	74294
2011-02-15	30600	30650	30450	30550	79127
2011-02-16	30550	30900	30550	30850	83535
2011-02-17	30950	31800	30900	31450	218768
2011-02-18	31400	31450	30950	31150	80795
2011-02-21	30950	31100	30850	31000	114990
2011-02-22	30800	30950	30650	30650	74663
2011-02-23	30550	31250	30500	30950	111523
2011-02-24	30550	30750	30450	30600	102721
2011-02-25	30500	30750	30500	30700	58603
2011-02-28	30500	30800	30200	30750	116137
2011-03-01	30800	31250	30600	31150	155542
2011-03-02	30950	34600	30900	32300	784149
2011-03-03	31600	32250	31550	31800	221997
2011-03-04	31750	32100	31550	31650	124311
2011-03-07	31800	32450	31250	31950	267890
2011-03-08	33000	33650	32600	32950	296131
2011-03-09	33400	33850	33100	33200	213619
2011-03-10	32600	32800	31950	32250	136724
2011-03-11	31800	31950	31500	31600	108634
2011-03-14	29950	30900	29200	29890	287440
2011-03-15	29000	29240	25250	27400	199189
2011-03-16	29400	30300	28620	29880	282798
2011-03-17	28880	29800	28740	29450	141708
2011-03-18	29510	30100	29150	29420	72732
2011-03-22	29920	29920	29360	29690	127535
2011-03-23	29800	30350	29520	30250	110003
2011-03-24	30700	30950	30100	30600	86299
2011-03-25	30800	30950	30450	30850	70560
2011-03-28	30800	31050	30300	30450	61925
2011-03-29	29450	29880	28560	28990	278421
2011-03-30	29300	29530	29140	29390	122537
2011-03-31	29530	29760	29300	29760	73513
2011-04-01	29720	29890	29230	29270	47661
2011-04-04	29240	29590	28640	28680	128910
2011-04-05	28590	28680	28130	28330	102287
2011-04-06	28100	28170	27490	27860	164334
2011-04-07	27910	28300	27870	28070	99113
2011-04-08	27670	28600	27600	28480	73311
2011-04-11	28700	29200	28510	28820	92094
2011-04-12	28320	28380	27900	28060	121419
2011-04-13	28070	28330	28070	28250	76263
2011-04-14	27750	27970	27720	27770	103053
2011-04-15	27750	28190	27430	28040	177776
2011-04-18	28320	28400	28000	28000	80620
2011-04-19	27540	27750	27320	27330	121473
2011-04-20	27590	27780	27370	27670	84425
2011-04-21	27880	28500	27720	28280	94367
2011-04-22	28400	28910	28210	28470	85830
2011-04-25	29970	29980	29260	29430	158350
2011-04-26	29390	29700	29110	29110	105741
2011-04-27	29200	29470	29020	29090	83262
2011-04-28	29020	29610	28950	29590	109186
2011-05-02	29630	29950	29500	29930	62042
2011-05-06	29520	29520	28800	29070	146871
2011-05-09	29090	29320	28850	29250	91014
2011-05-10	28940	29180	28820	28910	49553
2011-05-11	28920	29000	28510	28560	99901
2011-05-12	28500	28930	28450	28670	72387
2011-05-13	28720	28870	28310	28570	79792
2011-05-16	28160	28160	27570	27620	92149
2011-05-17	28000	28430	27610	27700	158728
2011-05-18	27710	28230	27710	28080	73326
2011-05-19	28050	28230	27740	27910	87509
2011-05-20	27990	27990	27650	27850	64702
2011-05-23	27510	27700	26940	27060	154662
2011-05-24	26500	26750	26310	26580	204762
2011-05-25	26580	26720	26580	26700	125980
2011-05-26	27080	27500	27000	27420	141260
2011-05-27	27300	27370	26780	26840	120713
2011-05-30	26850	26880	26650	26810	81829
2011-05-31	26810	26900	26720	26890	112632
2011-06-01	26900	27640	26870	27570	236371
2011-06-02	26770	27500	26680	27240	163204
2011-06-03	27130	27420	27000	27300	128577
2011-06-06	27520	27940	27450	27880	189204
2011-06-07	27590	27650	27140	27500	102229
2011-06-08	27490	27770	27150	27260	69754
2011-06-09	27200	27400	27000	27220	63429
2011-06-10	27480	27720	27010	27100	69280
2011-06-13	26820	27210	26810	27050	51139
2011-06-14	26800	27110	26770	26960	68346
2011-06-15	26980	27170	26630	26710	123116
2011-06-16	26590	26760	26390	26420	77339
2011-06-17	26370	26550	26220	26320	97361
2011-06-20	26130	26270	26040	26160	62789
2011-06-21	26250	26670	26150	26650	69107
2011-06-22	26560	26990	26560	26850	84416
2011-06-23	26780	26790	26420	26530	64535
2011-06-24	26420	26940	26420	26820	52938
2011-06-27	26790	26800	26510	26550	45806
2011-06-28	26890	26970	26720	26760	59101
2011-06-29	26990	27740	26920	27460	117040
2011-06-30	27700	27780	27490	27600	84204
2011-07-01	27750	27930	27410	27570	95970
2011-07-04	27700	27780	27470	27540	69314
2011-07-05	27400	27540	27020	27180	77449
2011-07-06	27210	27600	27090	27180	97088
2011-07-07	27190	27460	27070	27160	72082
2011-07-08	27420	27990	27210	27970	114901
2011-07-11	27860	27970	27500	27530	74859
2011-07-12	27190	27360	26830	26970	77906
2011-07-13	26920	27360	26900	27280	50778
2011-07-14	27100	27400	27010	27280	45318
2011-07-15	27560	28060	27450	27770	156667
2011-07-19	27890	28270	27810	28130	110011
2011-07-20	28220	28380	28160	28350	118583
2011-07-21	28490	28870	28410	28830	143394
2011-07-22	28990	29330	28910	29060	134077
2011-07-25	28060	28200	27850	28000	226212
2011-07-26	27850	27900	27460	27580	120790
2011-07-27	27620	27760	27260	27380	79930
2011-07-28	27120	27340	26980	27290	89938
2011-07-29	27040	27320	27020	27320	109568
2011-08-01	27130	27410	26880	26930	145116
2011-08-02	26840	26860	26460	26690	90416
2011-08-03	26430	26470	26100	26160	85093
2011-08-04	26170	26280	25990	26000	99017
2011-08-05	25130	25380	24550	25310	186351
2011-08-08	25090	25400	24920	25050	105756
2011-08-09	24450	24450	23850	24160	211616
2011-08-10	24660	24700	24250	24400	177921
2011-08-11	23850	24040	23710	23900	223759
2011-08-12	24300	24780	24210	24610	238270
2011-08-15	24930	24970	24800	24850	133308
2011-08-16	25020	25120	24570	24610	97507
2011-08-17	24560	24640	24040	24340	147954
2011-08-18	24000	24000	23700	23800	193706
2011-08-19	23200	23230	22800	22880	211849
2011-08-22	23000	23200	22840	22880	202694
2011-08-23	22880	22890	22060	22170	231465
2011-08-24	22600	22910	22380	22420	163540
2011-08-25	22780	23160	22730	23030	187315
2011-08-26	23000	23920	22940	23840	219917
2011-08-29	24040	24200	23740	23890	140526
2011-08-30	24600	24820	24580	24600	170383
2011-08-31	24730	24920	24500	24670	130991
2011-09-01	24880	24880	24330	24380	145662
2011-09-02	23920	24320	23760	24040	123133
2011-09-05	23830	24140	23760	23880	78148
2011-09-06	23740	23750	23400	23440	90863
2011-09-07	23750	24040	23660	23880	73773
2011-09-08	24150	24590	24050	24130	91300
2011-09-09	23850	24830	23850	24500	143887
2011-09-12	24100	24310	23850	24110	73354
2011-09-13	24320	24690	24000	24640	134512
2011-09-14	24650	24680	23960	24030	126734
2011-09-15	24380	24490	24130	24320	81615
2011-09-16	24340	24400	23700	24310	119355
2011-09-20	24000	24160	23810	23890	72119
2011-09-21	23880	23930	23170	23210	106496
2011-09-22	23070	23160	22610	22750	124767
2011-09-26	22720	22720	21910	22270	164649
2011-09-27	22700	23780	22660	23780	166501
2011-09-28	24020	24120	23650	23750	82842
2011-09-29	23910	24300	23860	24220	170617
2011-09-30	24230	24290	23790	24250	81554
2011-10-03	23750	24300	23670	24190	76510
2011-10-04	23850	24930	23700	24830	143259
2011-10-05	24840	25170	24720	24840	134321
2011-10-06	25030	25860	25030	25380	159175
2011-10-07	25390	26610	25390	25820	192026
2011-10-11	25850	27120	25800	26910	203798
2011-10-12	26900	26900	26150	26580	149615
2011-10-13	26580	26620	26070	26150	141540
2011-10-14	25810	26160	24900	25300	243069
2011-10-17	25310	25840	25110	25150	160341
2011-10-18	25410	25480	24990	25110	149356
2011-10-19	25000	25100	24340	24680	157382
2011-10-20	24440	24730	24230	24460	110962
2011-10-21	24470	25030	24300	24800	109087
2011-10-24	25000	25280	24910	25200	60080
2011-10-25	25370	25410	24900	25220	93169
2011-10-26	24900	25150	24580	24910	64310
2011-10-27	24810	24920	24520	24920	55441
2011-10-28	25200	25730	25100	25380	79494
2011-10-31	25410	25780	25410	25420	119624
2011-11-01	25250	25260	24860	24950	109430
2011-11-02	24700	24710	24100	24380	105021
2011-11-04	24740	24970	24620	24880	71307
2011-11-07	24440	24660	24030	24210	119175
2011-11-08	24060	24320	23540	23650	143825
2011-11-09	24000	24760	24000	24720	125688
2011-11-10	24220	24550	23950	24500	110073
2011-11-11	24200	24240	23810	23920	75155
2011-11-14	24050	24300	23630	23750	78418
2011-11-15	23880	24050	23760	23810	44623
2011-11-16	23760	23820	23390	23420	53402
2011-11-17	23230	23330	23120	23300	40472
2011-11-18	23190	23640	23080	23400	51705
2011-11-21	23450	23690	23080	23150	47725
2011-11-22	23180	23600	23100	23240	52658
2011-11-24	23020	23120	22700	22740	49081
2011-11-25	22510	22840	22500	22560	42604
2011-11-28	22730	22840	22320	22410	59871
2011-11-29	22530	23110	22370	23050	72718
2011-11-30	23200	24500	23160	24090	188814
2011-12-01	24300	25030	24210	25000	193239
2011-12-02	25010	25450	24680	25290	128847
2011-12-05	25280	25340	24760	25200	77350
2011-12-06	25110	25450	24840	24900	79963
2011-12-07	25090	25660	24720	25660	131498
2011-12-08	25220	25410	25040	25220	99221
2011-12-09	24970	25380	24950	25030	94130
2011-12-12	25280	25390	25070	25070	85358
2011-12-13	24720	25210	24510	25120	72581
2011-12-14	24950	25110	24600	24640	61493
2011-12-15	24420	25290	24420	24500	104088
2011-12-16	24360	24570	24210	24310	58495
2011-12-19	24000	24260	23610	23900	71225
2011-12-20	23800	24130	23740	23950	47567
2011-12-21	24050	24350	24030	24230	42834
2011-12-22	24540	25030	24300	24730	81901
2011-12-26	24470	24690	24330	24520	35435
2011-12-27	24730	24800	24510	24610	30258
2011-12-28	24480	24820	24360	24690	37335
2011-12-29	24700	25070	24510	24980	47123
2011-12-30	24800	24980	24630	24790	32544
2012-01-04	25250	25460	25010	25170	70233
2012-01-05	24900	24900	24510	24560	67628
2012-01-06	24700	24720	23980	24360	95349
2012-01-10	24710	25020	24280	24700	135753
2012-01-11	24880	24970	24280	24420	127661
2012-01-12	24190	24240	23600	23770	175731
2012-01-13	23800	23920	23450	23840	118413
2012-01-16	23590	23590	23120	23230	94767
2012-01-17	23250	23470	23150	23320	89170
2012-01-18	23360	24100	23300	23710	145715
2012-01-19	24190	24580	23920	24520	134972
2012-01-20	24600	24650	24400	24550	93676
2012-01-23	24370	24580	24090	24570	79312
2012-01-24	24700	24780	24170	24440	61349
2012-01-25	24700	25200	24680	25090	120511
2012-01-26	25000	25000	24310	24390	99142
2012-01-27	24000	24250	23260	23330	257293
2012-01-30	23320	23380	22960	23030	156249
2012-01-31	23280	23430	23200	23280	105504
2012-02-01	23310	23770	23260	23760	147232
2012-02-02	24000	24380	23960	24180	126969
2012-02-03	24390	24390	23870	24220	95835
2012-02-06	24330	24540	24260	24450	83415
2012-02-07	24390	24650	24280	24640	105412
2012-02-08	24600	24800	24350	24500	159116
2012-02-09	24390	25500	24300	25310	199890
2012-02-10	25610	26160	25460	25930	173579
2012-02-13	25430	25930	25200	25670	102822
2012-02-14	25500	26050	25500	26030	62379
2012-02-15	24750	25160	23920	25070	257950
2012-02-16	25200	25300	24900	25050	90525
2012-02-17	25070	25250	24830	24900	93464
2012-02-20	25150	25170	24780	24930	96446
2012-02-21	25040	25410	24930	25230	79222
2012-02-22	25260	25550	25080	25430	60470
2012-02-23	25560	25690	25330	25680	79323
2012-02-24	25550	25640	25050	25570	100322
2012-02-27	25460	25540	25210	25320	85959
2012-02-28	25240	25890	25200	25820	81488
2012-02-29	25900	26290	25650	25700	95601
2012-03-01	25890	25900	25430	25570	75329
2012-03-02	26160	27080	26080	26940	256200
2012-03-05	26930	27030	26520	26910	140058
2012-03-06	26920	27200	26610	26920	136806
2012-03-07	26580	26660	25930	26400	122767
2012-03-08	26680	26950	26400	26680	87023
2012-03-09	26950	26960	26680	26760	108716
2012-03-12	26770	27220	26730	27120	111460
2012-03-13	26830	27550	26820	27490	144598
2012-03-14	27820	27980	27680	27750	109174
2012-03-15	27700	27750	27220	27610	93893
2012-03-16	27620	27760	27180	27290	88042
2012-03-19	27400	27670	27370	27470	63779
2012-03-21	27210	27450	27130	27240	83413
2012-03-22	27070	27330	27020	27130	105620
2012-03-23	26870	27160	26860	26950	119741
2012-03-26	26820	27030	26360	26550	111097
2012-03-27	26670	27070	26580	26980	125484
2012-03-28	27070	27070	26590	26640	68651
2012-03-29	26960	27290	26860	27220	104367
2012-03-30	27080	27080	26460	26780	121536
2012-04-02	26780	26800	26610	26640	86371
2012-04-03	26420	26430	25370	25890	189458
2012-04-04	25820	26030	25530	25590	104934
2012-04-05	25500	25860	25480	25690	74948
2012-04-06	25490	25680	25260	25460	75698
2012-04-09	25200	25450	25100	25240	51378
2012-04-10	25240	25510	25050	25360	75285
2012-04-11	25310	25470	25060	25090	83187
2012-04-12	25070	25390	25010	25250	55224
2012-04-13	25200	25410	25130	25180	76183
2012-04-16	25000	25270	24840	25210	76731
2012-04-17	25280	25860	25260	25540	81932
2012-04-18	25610	25770	24810	25130	171124
2012-04-19	25200	25300	25040	25100	124340
2012-04-20	25020	25590	25020	25080	124203
2012-04-23	25090	25560	25070	25320	99442
2012-04-24	25000	26160	24780	25760	185337
cut2
2009-01-05	871.01	883.68	870.99	875.91
2009-01-06	883.15	883.15	872.38	876.20
2009-01-07	881.43	896.21	880.26	888.25
2009-01-08	879.64	879.69	860.16	860.89
2009-01-09	863.82	866.15	849.32	855.02
2009-01-13	845.07	845.07	814.12	814.12
2009-01-14	818.21	828.30	813.07	819.39
2009-01-15	807.68	807.68	792.72	795.99
2009-01-16	805.24	820.55	802.75	817.89
2009-01-19	825.18	828.88	815.51	817.73
2009-01-20	812.91	813.46	795.82	805.03
2009-01-21	793.70	795.82	784.50	787.15
2009-01-22	795.05	796.39	781.39	795.91
2009-01-23	789.39	789.39	773.28	773.55
2009-01-26	772.59	777.91	767.82	768.28
2009-01-27	776.84	809.94	776.84	805.49
2009-01-28	802.82	812.32	791.86	804.33
2009-01-29	812.42	822.14	807.50	818.47
2009-01-30	808.58	808.58	787.35	794.03
2009-02-02	785.66	785.66	770.55	777.85
2009-02-03	776.48	796.20	772.02	773.79
2009-02-04	779.56	795.53	776.15	792.78
2009-02-05	790.33	799.56	783.75	786.41
2009-02-06	794.62	803.47	787.54	790.84
2009-02-09	798.85	803.60	778.90	778.90
2009-02-10	787.50	792.24	775.25	778.10
2009-02-12	769.02	770.68	756.98	760.29
2009-02-13	766.19	773.38	759.20	764.59
2009-02-16	763.12	772.45	762.43	770.10
2009-02-17	765.10	765.15	754.89	756.53
2009-02-18	747.84	751.63	744.37	749.26
2009-02-19	753.78	757.28	749.53	751.59
2009-02-20	751.90	752.31	736.21	739.53
2009-02-23	730.60	737.72	721.79	735.28
2009-02-24	725.46	730.28	717.85	730.28
2009-02-25	739.47	746.72	732.99	745.62
2009-02-26	745.11	755.08	739.24	742.53
2009-02-27	744.14	756.76	741.18	756.71
2009-03-02	745.74	745.78	729.86	734.59
2009-03-03	724.04	731.73	714.96	726.80
2009-03-04	720.01	734.54	715.22	732.04
2009-03-05	736.83	751.81	736.52	741.55
2009-03-06	731.93	731.93	720.76	721.39
2009-03-09	721.81	724.26	705.42	710.53
2009-03-10	705.49	708.74	700.83	703.50
2009-03-11	713.91	729.12	713.78	722.28
2009-03-12	717.87	718.03	698.46	700.93
2009-03-13	707.62	726.20	707.35	724.30
2009-03-16	731.61	747.08	731.61	741.69
2009-03-17	747.64	763.29	743.62	760.64
2009-03-18	766.23	772.59	758.36	764.67
2009-03-19	771.25	774.27	762.46	764.77
2009-03-23	766.92	792.13	765.49	791.56
2009-03-24	803.28	815.40	801.84	812.72
2009-03-25	815.77	821.02	809.23	818.49
2009-03-26	813.85	827.38	809.22	826.81
2009-03-27	833.96	845.64	824.53	824.53
2009-03-30	825.06	829.08	789.54	789.54
2009-03-31	785.70	797.26	769.98	773.66
2009-04-01	780.51	795.75	778.21	793.82
2009-04-02	803.57	829.80	803.30	826.69
2009-04-03	835.01	843.46	826.17	831.36
2009-04-06	840.75	847.36	827.22	830.97
2009-04-07	830.93	835.92	827.26	832.60
2009-04-08	824.60	825.64	810.34	815.26
2009-04-09	822.20	842.52	821.70	841.81
2009-04-10	850.64	854.88	836.30	845.97
2009-04-13	844.28	856.08	840.45	848.97
2009-04-14	852.17	852.92	833.73	843.42
2009-04-15	838.53	839.73	825.79	835.25
2009-04-16	843.49	855.19	829.47	832.04
2009-04-17	840.24	848.06	839.40	845.57
2009-04-20	847.00	849.22	838.26	848.30
2009-04-21	838.02	838.02	818.32	830.72
2009-04-22	836.80	838.97	825.26	829.96
2009-04-23	833.28	841.75	823.31	839.50
2009-04-24	839.36	845.06	829.96	830.05
2009-04-27	838.97	843.86	826.70	833.10
2009-04-28	830.07	839.34	811.99	811.99
2009-04-30	822.90	841.67	822.90	837.79
2009-05-01	840.13	849.29	837.68	846.85
2009-05-07	858.93	885.93	858.93	885.93
2009-05-08	883.36	895.35	880.89	895.35
2009-05-11	899.57	906.47	889.91	900.45
2009-05-12	892.54	894.72	885.43	885.43
2009-05-13	886.77	892.04	880.46	888.75
2009-05-14	877.83	878.58	860.44	862.66
2009-05-15	868.78	882.86	867.98	881.65
2009-05-18	872.98	872.98	856.76	859.71
2009-05-19	871.59	883.81	871.46	879.76
2009-05-20	885.62	889.36	881.21	886.30
2009-05-21	880.09	881.44	872.53	881.44
2009-05-22	871.49	879.84	867.65	875.88
2009-05-25	878.48	888.82	878.48	883.00
2009-05-26	887.10	888.20	877.55	883.77
2009-05-27	893.48	898.14	891.46	892.85
2009-05-28	885.76	898.37	885.71	895.59
2009-05-29	897.99	898.34	890.74	897.91
2009-06-01	898.35	914.26	895.56	912.52
2009-06-02	922.92	923.89	913.56	913.56
2009-06-03	914.73	919.71	913.37	914.50
2009-06-04	910.26	918.11	908.36	910.99
2009-06-05	918.59	920.75	913.60	916.56
2009-06-08	923.25	930.23	922.55	926.89
2009-06-09	924.02	927.55	916.90	918.24
2009-06-10	923.33	937.94	922.63	937.01
2009-06-11	937.87	942.62	936.41	940.65
2009-06-12	946.30	954.08	943.66	950.54
2009-06-15	952.88	952.99	944.43	946.82
2009-06-16	936.11	937.24	914.14	914.76
2009-06-17	911.10	924.87	911.10	923.03
2009-06-18	919.54	919.54	906.29	911.21
2009-06-19	917.09	924.84	913.11	918.97
2009-06-22	921.02	928.88	917.35	922.48
2009-06-23	911.60	912.22	897.25	901.69
2009-06-24	905.38	906.78	898.87	902.46
2009-06-25	907.33	926.39	905.18	919.77
2009-06-26	926.65	928.53	920.52	926.80
2009-06-29	925.89	930.91	911.93	915.32
2009-06-30	925.34	932.96	925.11	929.76
2009-07-01	924.18	940.06	922.76	928.30
2009-07-02	933.77	934.55	923.19	924.02
2009-07-03	913.88	920.75	908.77	920.62
2009-07-06	915.32	919.40	910.47	912.42
2009-07-07	914.54	917.25	906.59	909.13
2009-07-08	899.79	899.88	886.64	888.54
2009-07-09	880.63	882.53	873.40	873.91
2009-07-10	877.53	879.94	870.24	872.50
2009-07-13	867.73	878.41	852.11	852.42
2009-07-14	863.04	871.01	862.44	868.57
2009-07-15	872.89	873.79	864.07	866.37
2009-07-16	877.39	886.37	871.12	872.25
2009-07-17	879.15	880.76	874.52	878.29
2009-07-21	889.12	901.55	889.12	901.55
2009-07-22	901.94	909.42	898.48	906.58
2009-07-23	906.14	916.05	905.06	908.69
2009-07-24	918.84	921.65	914.34	920.48
2009-07-27	927.44	936.22	926.39	928.26
2009-07-28	931.12	931.12	925.73	930.13
2009-07-29	925.38	934.81	925.38	930.36
2009-07-30	933.66	938.09	927.67	936.94
2009-07-31	945.55	950.38	943.11	950.26
2009-08-03	953.53	958.45	951.89	957.56
2009-08-04	966.94	970.56	957.05	959.02
2009-08-05	961.61	962.79	949.34	949.58
2009-08-06	950.90	959.88	950.90	957.51
2009-08-07	955.11	956.76	942.72	956.76
2009-08-10	967.02	973.38	965.78	969.24
2009-08-11	968.46	973.83	968.04	973.51
2009-08-12	965.12	966.78	959.87	959.87
2009-08-13	967.19	970.44	965.06	968.41
2009-08-14	969.79	975.88	968.62	973.57
2009-08-17	968.15	968.15	949.23	949.59
2009-08-18	941.78	951.01	941.78	949.66
2009-08-19	950.25	953.77	942.31	943.25
2009-08-20	947.81	961.16	944.29	958.59
2009-08-21	958.16	958.16	938.81	947.34
2009-08-24	960.40	972.31	960.40	970.27
2009-08-25	965.58	969.39	963.05	965.11
2009-08-26	970.27	977.54	968.53	975.59
2009-08-27	970.00	970.73	955.68	964.23
2009-08-28	968.46	970.47	963.76	969.31
2009-08-31	974.87	987.27	959.83	965.73
2009-09-01	962.29	971.91	960.66	968.77
2009-09-02	956.51	956.51	944.18	949.81
2009-09-03	944.74	948.03	940.10	942.77
2009-09-04	944.05	944.16	933.33	935.74
2009-09-07	943.27	945.89	942.21	944.60
2009-09-08	946.41	946.41	940.08	946.40
2009-09-09	943.54	945.39	935.18	939.84
2009-09-10	946.13	959.17	946.13	958.49
2009-09-11	959.94	959.94	947.31	950.41
2009-09-14	946.68	946.68	930.20	934.05
2009-09-15	936.52	940.51	930.97	932.52
2009-09-16	937.75	943.72	930.35	931.43
2009-09-17	939.95	943.45	932.10	939.52
2009-09-18	934.38	939.44	925.50	939.44
2009-09-24	941.79	954.22	940.59	950.20
2009-09-25	936.16	936.18	919.81	922.67
2009-09-28	911.41	912.66	898.08	902.84
2009-09-29	908.02	909.54	901.14	904.00
2009-09-30	905.19	909.84	902.73	909.84
2009-10-01	905.39	906.21	894.08	896.12
2009-10-02	884.19	884.20	871.52	874.67
2009-10-05	875.10	876.65	864.50	867.28
2009-10-06	872.64	873.81	863.78	871.63
2009-10-07	876.59	887.00	875.84	885.69
2009-10-08	886.49	890.98	882.02	887.59
2009-10-09	891.80	897.92	886.08	897.83
2009-10-13	903.29	905.76	900.49	901.40
2009-10-14	901.12	901.12	890.77	894.34
2009-10-15	904.23	909.41	901.54	904.11
2009-10-16	906.23	906.23	896.20	900.95
2009-10-19	895.70	906.30	893.57	905.80
2009-10-20	913.08	914.96	909.75	913.45
2009-10-21	909.21	914.64	908.56	913.70
2009-10-22	907.38	908.60	897.17	908.60
2009-10-23	912.58	912.86	901.40	902.03
2009-10-26	903.50	914.62	902.22	910.72
2009-10-27	904.42	904.95	892.66	895.48
2009-10-28	893.70	896.82	887.00	888.80
2009-10-29	877.94	885.83	874.19	882.26
2009-10-30	891.38	895.61	888.57	894.67
2009-11-02	884.08	884.08	874.66	880.54
2009-11-04	877.69	881.37	875.40	881.27
2009-11-05	879.42	882.07	872.01	874.96
2009-11-06	880.50	880.50	872.29	874.01
2009-11-09	871.86	874.88	866.12	870.67
2009-11-10	877.33	882.21	871.67	872.44
2009-11-11	873.48	879.29	870.94	872.29
2009-11-12	875.79	877.43	865.64	867.70
2009-11-13	866.72	868.76	862.06	866.80
2009-11-16	865.21	865.37	857.67	860.42
2009-11-17	863.16	864.37	854.89	857.00
2009-11-18	853.42	856.17	844.91	850.06
2009-11-19	849.46	849.46	832.09	837.71
2009-11-20	831.03	838.71	829.98	838.71
2009-11-24	839.34	839.40	827.95	829.22
2009-11-25	828.45	834.15	826.36	833.29
2009-11-26	826.97	835.10	824.95	829.56
2009-11-27	819.41	820.40	809.24	811.01
2009-11-30	819.37	839.94	818.86	839.94
2009-12-01	833.52	857.76	829.56	857.76
2009-12-02	854.66	863.14	850.93	858.74
2009-12-03	867.65	888.04	867.57	888.04
2009-12-04	890.62	890.62	883.09	889.58
2009-12-07	898.86	903.65	895.47	898.93
2009-12-08	893.14	898.21	893.12	896.70
2009-12-09	889.00	889.00	883.65	884.94
2009-12-10	882.34	889.02	870.47	873.90
2009-12-11	880.01	888.57	874.62	888.57
2009-12-14	890.48	890.48	877.60	885.08
2009-12-15	882.27	887.52	881.38	884.63
2009-12-16	892.94	903.89	892.94	898.29
2009-12-17	900.87	903.39	896.28	896.28
2009-12-18	892.15	894.80	886.72	893.59
2009-12-21	896.68	897.42	891.48	891.48
2009-12-22	897.58	904.60	895.29	903.06
2009-12-24	907.83	914.73	907.53	913.72
2009-12-25	914.30	914.30	908.95	909.39
2009-12-28	910.91	919.56	910.21	914.78
2009-12-29	913.58	918.58	911.43	915.87
2009-12-30	920.54	920.54	907.59	907.59
2010-01-04	912.48	919.17	912.48	915.75
2010-01-05	926.07	930.51	918.14	919.57
2010-01-06	925.06	932.41	921.12	931.13
2010-01-07	933.84	937.02	929.77	931.85
2010-01-08	937.53	941.81	933.37	941.29
2010-01-12	940.61	955.40	940.61	954.13
2010-01-13	947.66	955.91	943.88	944.02
2010-01-14	948.63	959.01	948.63	959.01
2010-01-15	962.24	966.40	959.04	966.40
2010-01-18	957.72	960.81	953.98	957.55
2010-01-19	957.69	957.69	947.29	949.76
2010-01-20	956.41	956.47	942.99	944.72
2010-01-21	940.92	958.88	937.43	956.03
2010-01-22	942.92	942.95	932.41	940.94
2010-01-25	930.64	938.73	928.87	934.59
2010-01-26	934.01	937.87	916.40	916.40
2010-01-27	917.80	919.66	907.67	907.67
2010-01-28	911.13	920.64	909.59	914.32
2010-01-29	908.73	910.54	901.12	901.12
2010-02-01	900.38	900.38	891.93	898.61
2010-02-02	905.78	913.68	903.73	912.82
2010-02-03	918.20	921.90	915.02	915.68
2010-02-04	918.02	918.55	903.34	911.09
2010-02-05	896.29	896.92	888.73	891.78
2010-02-08	887.90	891.97	882.21	883.01
2010-02-09	877.89	883.58	876.77	881.57
2010-02-10	887.98	890.15	883.50	883.50
2010-02-12	891.80	892.87	887.47	892.16
2010-02-15	892.72	893.85	882.98	883.47
2010-02-16	885.23	887.77	884.03	885.17
2010-02-17	895.87	904.90	895.21	904.63
2010-02-18	907.14	907.36	902.35	904.73
2010-02-19	904.85	906.23	889.08	889.08
2010-02-22	902.76	914.76	902.76	909.75
2010-02-23	906.62	908.42	900.91	907.37
2010-02-24	895.70	897.62	890.86	895.69
2010-02-25	899.59	900.84	889.26	891.41
2010-02-26	890.90	897.13	890.23	894.10
2010-03-01	895.23	902.50	893.81	898.93
2010-03-02	900.50	903.41	896.46	902.71
2010-03-03	900.62	907.11	900.00	905.65
2010-03-04	905.29	906.18	896.68	897.64
2010-03-05	905.11	913.18	905.11	910.81
2010-03-08	923.56	927.31	921.74	927.31
2010-03-09	925.69	927.79	923.60	924.38
2010-03-10	924.10	926.27	921.94	922.44
2010-03-11	928.00	931.16	925.08	930.38
2010-03-12	935.95	937.50	930.94	936.38
2010-03-15	940.81	942.42	935.20	938.91
2010-03-16	936.68	942.33	936.68	938.10
2010-03-17	943.75	949.00	940.84	947.43
2010-03-18	947.83	948.78	940.15	940.79
2010-03-19	944.15	949.01	943.57	948.93
2010-03-23	948.10	951.05	947.13	947.37
2010-03-24	952.53	956.02	948.42	951.97
2010-03-25	955.76	957.26	950.98	952.13
2010-03-26	957.39	966.72	957.30	966.72
2010-03-29	958.19	966.14	957.86	966.13
2010-03-30	971.28	980.33	970.15	979.58
2010-03-31	981.76	984.06	977.39	978.81
2010-04-01	983.99	986.51	978.28	985.26
2010-04-02	991.06	991.06	985.37	989.39
2010-04-05	996.05	996.97	992.41	995.68
2010-04-06	997.11	997.37	986.05	990.79
2010-04-07	991.82	998.97	991.56	995.54
2010-04-08	987.60	992.32	985.80	985.99
2010-04-09	985.62	989.42	983.73	989.42
2010-04-12	996.34	1001.53	994.78	994.78
2010-04-13	993.26	993.56	982.59	988.44
2010-04-14	991.49	995.26	987.40	991.10
2010-04-15	999.54	1001.77	996.97	998.90
2010-04-16	996.55	996.55	987.35	988.84
2010-04-19	974.43	975.87	968.79	970.84
2010-04-20	975.92	980.33	971.41	972.11
2010-04-21	982.98	987.61	982.00	987.07
2010-04-22	981.02	981.02	970.69	978.17
2010-04-23	976.33	979.53	973.74	978.20
2010-04-26	987.35	996.82	987.06	996.71
2010-04-27	992.82	997.63	990.93	997.63
2010-04-28	980.93	981.14	973.57	977.64
2010-04-30	988.94	991.22	984.72	987.04
2010-05-06	970.44	970.46	955.84	956.72
2010-05-07	937.51	937.60	921.36	931.74
2010-05-10	930.54	945.20	927.98	944.64
2010-05-11	954.04	954.34	930.57	932.10
2010-05-12	936.56	940.23	929.39	932.83
2010-05-13	943.48	949.21	941.93	947.90
2010-05-14	936.79	942.89	930.97	936.45
2010-05-17	926.55	928.92	915.23	920.43
2010-05-18	924.23	924.64	910.99	913.91
2010-05-19	902.69	910.64	898.70	910.64
2010-05-20	904.59	908.36	896.01	898.15
2010-05-21	880.90	881.44	872.48	879.69
2010-05-24	878.42	881.73	873.28	880.01
2010-05-25	872.12	872.76	857.00	859.82
2010-05-26	864.18	867.58	854.79	859.00
2010-05-27	851.70	870.26	850.88	869.89
2010-05-28	883.51	884.98	874.41	878.52
2010-05-31	876.60	886.43	876.28	880.46
2010-06-01	879.43	881.55	874.89	880.04
2010-06-02	871.60	882.65	867.20	870.05
2010-06-03	881.66	891.26	881.63	890.64
2010-06-04	891.35	895.09	887.42	890.16
2010-06-07	874.36	874.46	858.90	859.21
2010-06-08	854.33	862.87	853.21	858.33
2010-06-09	854.26	855.11	846.47	850.37
2010-06-10	853.33	857.26	848.87	856.79
2010-06-11	869.60	872.49	865.36	866.44
2010-06-14	875.29	878.56	875.29	878.56
2010-06-15	876.52	881.98	874.91	878.83
2010-06-16	890.53	894.83	889.64	892.38
2010-06-17	889.04	891.62	885.61	887.48
2010-06-18	887.83	887.83	882.34	884.64
2010-06-21	891.86	904.03	891.86	902.49
2010-06-22	895.27	900.39	893.78	894.56
2010-06-23	883.91	884.13	880.33	880.84
2010-06-24	878.91	886.40	878.54	879.77
2010-06-25	870.11	871.63	864.40	867.30
2010-06-28	867.27	867.44	859.30	860.80
2010-06-29	862.55	865.54	850.29	852.19
2010-06-30	839.03	842.19	835.91	841.42
2010-07-01	834.72	834.89	825.06	828.39
2010-07-02	831.30	833.72	825.90	830.98
2010-07-05	833.71	839.30	832.85	836.89
2010-07-06	829.81	847.24	825.26	847.24
2010-07-07	845.69	846.35	837.13	841.51
2010-07-08	856.09	862.73	855.76	861.02
2010-07-09	864.14	864.30	856.99	861.21
2010-07-12	859.21	866.13	857.15	857.70
2010-07-13	861.23	863.78	852.10	854.39
2010-07-14	867.60	874.25	866.18	870.73
2010-07-15	863.75	863.76	856.60	856.60
2010-07-16	850.61	853.21	838.78	840.58
2010-07-20	829.84	837.69	828.73	832.26
2010-07-21	838.64	838.79	827.31	829.35
2010-07-22	824.06	827.43	822.02	825.48
2010-07-23	837.26	844.37	836.08	841.29
2010-07-26	848.79	852.05	845.80	845.88
2010-07-27	846.20	850.12	844.37	846.12
2010-07-28	855.88	866.26	855.88	865.51
2010-07-29	859.39	864.04	859.05	861.27
2010-07-30	858.86	858.86	846.71	849.50
2010-08-02	851.45	858.96	848.82	850.69
2010-08-03	861.82	864.25	855.57	859.18
2010-08-04	854.43	854.71	844.38	845.93
2010-08-05	856.44	859.98	851.31	857.09
2010-08-06	851.83	861.40	851.23	861.17
2010-08-09	853.33	857.62	853.09	857.62
2010-08-10	860.68	862.27	851.22	854.68
2010-08-11	845.54	846.34	833.75	834.45
2010-08-12	820.69	827.78	816.78	827.78
2010-08-13	826.95	832.86	822.92	831.24
2010-08-16	823.05	828.70	820.34	828.63
2010-08-17	820.89	828.17	820.47	826.78
2010-08-18	834.25	838.20	826.99	835.23
2010-08-19	835.41	843.98	835.41	843.98
2010-08-20	833.89	838.20	828.60	829.59
2010-08-23	826.67	828.48	822.45	824.79
2010-08-24	817.23	821.40	814.38	817.73
2010-08-25	809.54	814.31	804.38	807.31
2010-08-26	811.08	811.79	805.18	811.79
2010-08-27	804.44	821.45	804.34	819.62
2010-08-30	830.38	840.68	827.11	829.21
2010-08-31	818.30	818.61	803.78	804.67
2010-09-01	804.25	811.40	800.69	811.40
2010-09-02	822.26	822.98	812.03	819.42
2010-09-03	822.32	825.71	818.95	823.70
2010-09-06	829.48	838.71	827.78	838.71
2010-09-07	833.77	841.17	832.82	834.92
2010-09-08	824.81	825.14	817.21	820.99
2010-09-09	827.95	828.89	823.77	826.84
2010-09-10	830.42	839.75	830.04	833.72
2010-09-13	840.20	843.28	837.14	837.65
2010-09-14	838.07	839.58	833.02	834.87
2010-09-15	832.27	854.11	828.89	848.64
2010-09-16	856.80	856.85	842.59	844.71
2010-09-17	850.03	853.82	848.30	852.09
2010-09-21	858.32	858.77	848.97	849.94
2010-09-22	847.60	851.51	845.38	846.52
2010-09-24	837.79	849.03	835.36	838.41
2010-09-27	844.88	849.30	841.87	849.30
2010-09-28	843.04	845.71	840.69	842.65
2010-09-29	844.06	849.58	841.69	846.97
2010-09-30	846.96	847.92	828.89	829.51
2010-10-01	833.46	833.59	823.90	829.97
2010-10-04	830.35	834.42	821.20	822.74
2010-10-05	820.23	834.30	817.07	832.64
2010-10-06	838.72	844.67	835.37	844.50
2010-10-07	843.11	850.13	842.79	846.06
2010-10-08	844.46	847.23	839.44	839.44
2010-10-12	844.69	844.85	823.32	824.60
2010-10-13	829.74	832.86	821.83	822.65
2010-10-14	830.43	840.11	829.13	836.95
2010-10-15	834.36	834.52	824.31	826.38
2010-10-18	827.21	835.13	826.75	830.52
2010-10-19	830.65	837.74	830.27	833.73
2010-10-20	824.04	824.83	815.67	823.69
2010-10-21	824.11	826.33	815.77	820.40
2010-10-22	821.57	827.69	820.39	824.88
2010-10-25	824.31	826.44	820.14	821.23
2010-10-26	819.30	824.06	817.81	817.94
2010-10-27	821.80	822.79	812.72	817.76
2010-10-28	816.44	819.30	813.35	814.33
2010-10-29	812.98	815.09	803.54	810.91
2010-11-01	806.55	811.18	801.56	803.30
2010-11-02	802.36	803.81	799.64	803.12
2010-11-04	812.97	820.76	812.48	816.33
2010-11-05	827.46	839.56	827.16	834.98
2010-11-08	840.67	842.14	837.08	841.74
2010-11-09	838.18	843.48	836.79	839.95
2010-11-10	843.89	854.15	843.53	852.98
2010-11-11	856.14	858.52	853.43	856.37
2010-11-12	851.99	856.89	846.98	846.98
2010-11-15	851.62	852.26	847.42	851.56
2010-11-16	856.21	857.44	845.51	847.77
2010-11-17	840.22	850.88	840.17	850.30
2010-11-18	851.81	868.81	850.65	868.81
2010-11-19	876.80	877.06	869.13	869.52
2010-11-22	877.04	878.64	873.51	875.48
2010-11-24	862.23	869.96	858.81	866.57
2010-11-25	873.34	874.27	867.56	869.81
2010-11-26	870.10	874.25	866.76	866.81
2010-11-29	868.69	876.43	868.69	874.59
2010-11-30	872.21	874.76	860.94	860.94
2010-12-01	860.70	866.07	858.87	866.07
2010-12-02	878.61	880.72	876.53	877.21
2010-12-03	883.08	884.63	877.37	879.22
2010-12-06	879.49	881.79	878.58	881.41
2010-12-07	880.29	880.47	875.77	879.10
2010-12-08	881.88	887.39	881.48	887.39
2010-12-09	890.08	891.90	887.60	891.60
2010-12-10	896.37	896.37	886.77	888.22
2010-12-13	887.54	897.40	886.38	897.40
2010-12-14	896.40	901.89	896.27	901.89
2010-12-15	904.78	904.99	899.85	902.42
2010-12-16	901.41	907.80	900.54	903.84
2010-12-17	903.78	906.24	901.94	903.14
2010-12-20	903.79	904.95	895.28	898.55
2010-12-21	900.91	908.13	900.68	906.21
2010-12-22	905.43	909.67	903.57	905.78
2010-12-24	901.99	903.96	900.15	901.66
2010-12-27	901.47	906.50	901.47	904.68
2010-12-28	904.31	905.42	902.67	902.83
2010-12-29	902.55	908.01	901.49	908.01
2010-12-30	905.71	906.42	897.74	898.80
2011-01-04	909.44	913.28	906.86	911.80
2011-01-05	912.24	914.29	909.23	911.69
2011-01-06	919.81	924.67	919.41	924.51
2011-01-07	924.69	928.27	923.72	926.42
2011-01-11	923.71	928.70	923.03	926.94
2011-01-12	932.51	935.64	928.78	929.64
2011-01-13	938.00	939.70	934.44	937.74
2011-01-14	933.93	937.66	930.31	930.31
2011-01-17	936.61	936.61	927.09	928.73
2011-01-18	926.59	934.92	925.37	931.58
2011-01-19	935.72	938.03	932.92	936.87
2011-01-20	932.15	933.47	926.32	927.19
2011-01-21	929.01	929.08	909.14	910.85
2011-01-24	914.79	917.54	909.78	917.18
2011-01-25	920.96	931.49	919.04	929.28
2011-01-26	926.27	927.15	922.64	922.64
2011-01-27	925.60	930.50	923.36	929.66
2011-01-28	928.21	929.48	916.95	919.69
2011-01-31	908.30	912.36	904.25	910.08
2011-02-01	914.34	916.41	910.82	913.52
2011-02-02	921.90	931.62	921.47	929.64
2011-02-03	928.42	929.28	925.35	927.57
2011-02-04	935.12	940.80	934.92	935.36
2011-02-07	943.14	943.62	938.57	940.43
2011-02-08	944.76	945.45	942.26	944.00
2011-02-09	948.56	951.28	942.29	944.02
2011-02-10	940.91	948.28	940.49	946.63
2011-02-14	954.24	959.30	953.92	959.19
2011-02-15	960.69	964.31	959.05	962.57
2011-02-16	963.31	971.70	963.27	967.30
2011-02-17	974.33	976.28	971.42	974.14
2011-02-18	973.65	975.69	970.96	973.60
2011-02-21	972.56	974.64	969.27	974.63
2011-02-22	966.87	968.04	955.48	956.70
2011-02-23	947.41	957.25	946.88	946.88
2011-02-24	942.11	942.97	932.35	934.22
2011-02-25	935.21	942.06	934.10	941.93
2011-02-28	941.51	952.56	935.63	951.27
2011-03-01	956.34	964.18	956.19	963.70
2011-03-02	952.05	953.47	942.87	942.87
2011-03-03	945.18	949.43	944.84	948.69
2011-03-04	960.92	962.66	952.90	955.59
2011-03-07	950.22	950.37	938.91	941.63
2011-03-08	941.30	945.50	938.60	939.16
2011-03-09	946.58	951.39	942.75	944.29
2011-03-10	941.31	941.31	928.40	930.84
2011-03-11	920.79	925.41	915.12	915.51
2011-03-14	896.68	896.79	843.86	846.96
2011-03-15	831.52	831.60	725.90	766.73
2011-03-16	779.89	818.94	779.89	817.63
2011-03-17	800.80	821.47	782.19	810.80
2011-03-18	821.27	833.86	820.74	830.39
2011-03-22	848.12	869.95	847.17	868.13
2011-03-23	868.62	868.91	854.91	861.10
2011-03-24	860.78	863.80	853.43	853.95
2011-03-25	865.10	865.16	851.05	857.38
2011-03-28	858.77	858.93	850.72	857.85
2011-03-29	846.22	854.17	839.04	850.21
2011-03-30	852.04	866.09	849.73	866.09
2011-03-31	871.02	871.08	861.76	869.38
2011-04-01	870.96	874.35	862.62	862.62
2011-04-04	867.33	868.72	859.52	859.75
2011-04-05	859.91	860.17	841.76	847.16
2011-04-06	850.00	850.00	837.14	839.61
2011-04-07	845.00	850.02	840.26	841.10
2011-04-08	837.64	855.87	833.06	853.13
2011-04-11	850.75	856.01	849.04	852.34
2011-04-12	842.47	843.52	835.11	838.51
2011-04-13	835.40	845.90	835.27	844.59
2011-04-14	841.02	848.90	838.58	846.72
2011-04-15	845.52	846.77	840.80	841.29
2011-04-18	841.98	843.32	836.34	836.34
2011-04-19	828.11	831.23	824.61	827.56
2011-04-20	833.72	839.13	832.13	837.17
2011-04-21	842.35	844.28	837.53	841.72
2011-04-22	836.92	845.13	834.87	842.18
2011-04-25	842.88	847.84	839.52	840.68
2011-04-26	837.94	838.15	831.80	833.64
2011-04-27	838.42	844.36	836.85	839.87
2011-04-28	844.45	851.85	842.12	851.85
2011-05-02	861.60	866.65	860.49	865.55
2011-05-06	855.16	857.83	851.17	856.50
2011-05-09	859.83	860.43	851.27	853.21
2011-05-10	854.47	859.40	851.09	856.46
2011-05-11	863.01	864.19	856.38	857.62
2011-05-12	852.08	857.64	849.34	849.34
2011-05-13	851.36	851.62	833.19	839.94
2011-05-16	833.98	834.90	829.28	829.55
2011-05-17	827.09	830.42	823.99	828.85
2011-05-18	831.40	839.98	830.30	837.96
2011-05-19	841.70	843.23	830.36	831.89
2011-05-20	830.40	834.05	827.77	827.77
2011-05-23	824.05	824.08	816.43	817.68
2011-05-24	812.88	820.27	812.52	819.16
2011-05-25	821.98	823.53	816.98	817.74
2011-05-26	822.72	827.90	822.27	827.08
2011-05-27	824.70	830.22	821.62	824.90
2011-05-30	823.05	826.56	818.96	823.68
2011-05-31	823.02	838.48	822.70	838.48
2011-06-01	839.41	839.59	834.50	839.41
2011-06-02	826.94	827.22	823.24	825.76
2011-06-03	825.11	828.28	816.34	816.57
2011-06-06	814.57	817.03	805.84	807.99
2011-06-07	807.07	814.60	805.23	813.76
2011-06-08	812.93	815.31	810.12	814.45
2011-06-09	811.00	813.34	807.14	812.95
2011-06-10	818.64	823.79	815.50	817.38
2011-06-13	808.95	813.06	806.99	812.26
2011-06-14	812.13	823.83	811.49	822.86
2011-06-15	826.74	827.35	820.90	824.65
2011-06-16	817.14	819.76	812.41	812.41
2011-06-17	814.71	814.83	801.78	805.34
2011-06-20	807.88	812.20	805.94	806.83
2011-06-21	811.37	815.73	808.38	815.73
2011-06-22	819.51	831.30	819.51	828.99
2011-06-23	823.73	830.85	822.78	825.51
2011-06-24	827.25	834.37	826.90	833.20
2011-06-27	831.21	831.32	824.64	825.64
2011-06-28	832.51	835.51	828.79	830.34
2011-06-29	839.82	844.11	837.99	844.11
2011-06-30	849.14	850.32	844.31	849.22
2011-07-01	854.83	856.79	852.20	853.86
2011-07-04	863.50	866.53	862.00	864.11
2011-07-05	864.12	867.46	863.78	865.18
2011-07-06	865.65	873.51	863.53	873.51
2011-07-07	869.98	873.05	867.87	870.48
2011-07-08	878.39	879.48	873.96	874.34
2011-07-11	869.49	873.01	868.52	870.16
2011-07-12	859.14	861.72	855.33	857.19
2011-07-13	853.48	862.76	853.37	860.53
2011-07-14	858.01	862.06	853.98	856.88
2011-07-15	856.06	860.76	855.85	859.36
2011-07-19	854.81	857.45	852.89	853.75
2011-07-20	862.39	864.91	858.89	860.66
2011-07-21	862.07	862.31	858.13	860.11
2011-07-22	866.95	870.71	864.83	868.81
2011-07-25	866.12	866.71	861.12	861.91
2011-07-26	863.67	869.84	861.95	866.20
2011-07-27	861.69	861.69	854.82	859.11
2011-07-28	850.14	851.88	844.92	848.37
2011-07-29	846.65	848.46	841.37	841.37
2011-08-01	847.12	858.53	847.02	851.70
2011-08-02	845.77	846.66	841.98	843.96
2011-08-03	831.48	831.76	825.18	826.75
2011-08-04	829.07	835.53	824.10	826.36
2011-08-05	809.46	809.72	794.16	800.96
2011-08-08	789.59	791.94	779.87	782.86
2011-08-09	766.94	770.39	744.46	770.39
2011-08-10	783.91	785.29	775.27	776.73
2011-08-11	762.70	770.88	759.78	770.88
2011-08-12	778.78	779.67	765.38	768.19
2011-08-15	778.07	779.72	772.68	777.12
2011-08-16	781.36	783.95	776.36	779.06
2011-08-17	775.38	778.02	771.12	776.65
2011-08-18	775.57	775.66	765.91	767.31
2011-08-19	753.27	756.08	749.45	751.69
2011-08-22	748.51	752.86	742.13	742.84
2011-08-23	747.97	751.85	742.76	750.39
2011-08-24	756.95	759.40	740.51	742.24
2011-08-25	749.93	759.34	749.21	751.82
2011-08-26	750.82	756.35	749.99	756.07
2011-08-29	757.21	765.64	751.53	758.83
2011-08-30	768.00	773.31	766.37	767.30
2011-08-31	767.14	771.18	764.48	770.60
2011-09-01	776.04	780.99	774.52	778.28
2011-09-02	772.42	774.26	766.38	769.78
2011-09-05	760.77	762.15	753.46	755.82
2011-09-06	750.31	750.34	740.39	741.20
2011-09-07	750.87	755.29	748.43	753.63
2011-09-08	761.76	763.08	754.12	757.41
2011-09-09	754.60	760.52	754.60	755.70
2011-09-12	742.61	743.09	738.80	741.26
2011-09-13	745.40	751.75	741.90	749.82
2011-09-14	749.81	753.56	740.58	741.69
2011-09-15	750.53	754.92	749.46	751.76
2011-09-16	760.41	768.42	759.56	768.13
2011-09-20	760.62	760.74	753.37	755.04
2011-09-21	755.50	758.93	754.39	757.13
2011-09-22	749.56	749.77	740.85	744.54
2011-09-26	744.06	744.28	727.33	728.85
2011-09-27	738.64	748.55	737.55	748.55
2011-09-28	746.96	755.60	746.59	754.07
2011-09-29	747.78	762.30	747.26	762.30
2011-09-30	764.06	765.89	755.56	761.17
2011-10-03	748.80	749.67	735.94	747.11
2011-10-04	737.20	737.61	729.80	736.18
2011-10-05	739.36	739.81	724.77	726.25
2011-10-06	732.60	740.28	732.25	736.86
2011-10-07	742.98	748.64	740.95	741.55
2011-10-11	752.38	757.93	751.83	755.00
2011-10-12	750.71	754.92	747.25	753.44
2011-10-13	761.08	762.02	758.00	758.83
2011-10-14	752.81	753.87	748.71	748.81
2011-10-17	759.36	763.85	757.93	761.88
2011-10-18	752.62	754.94	749.64	751.24
2011-10-19	756.34	757.88	749.62	751.49
2011-10-20	749.73	749.88	742.72	746.02
2011-10-21	744.77	746.59	741.87	744.21
2011-10-24	749.25	755.55	749.14	755.44
2011-10-25	757.60	757.85	747.04	747.70
2011-10-26	741.78	751.21	736.50	746.48
2011-10-27	748.11	763.04	744.28	762.79
2011-10-28	774.11	777.39	769.94	771.43
2011-10-31	768.55	779.08	764.06	764.06
2011-11-01	757.77	761.04	753.39	754.50
2011-11-02	745.07	745.21	737.21	738.58
2011-11-04	747.69	752.85	745.38	752.02
2011-11-07	749.96	750.45	747.06	750.45
2011-11-08	747.57	749.93	736.89	738.03
2011-11-09	744.06	749.87	740.90	749.40
2011-11-10	735.64	735.64	727.72	730.30
2011-11-11	731.54	733.19	726.24	729.13
2011-11-14	738.04	739.91	733.93	735.85
2011-11-15	731.82	735.04	729.97	730.91
2011-11-16	730.57	731.61	723.58	724.11
2011-11-17	720.63	728.63	717.78	727.71
2011-11-18	719.57	721.47	717.61	719.98
2011-11-21	716.87	719.19	716.28	717.08
2011-11-22	710.31	719.50	709.86	717.79
2011-11-24	708.05	712.00	704.77	706.08
2011-11-25	704.08	711.36	703.88	706.60
2011-11-28	714.09	719.61	713.98	715.70
2011-11-29	721.94	729.83	719.84	729.68
2011-11-30	724.87	728.60	721.10	728.46
2011-12-01	740.70	746.44	738.76	740.01
2011-12-02	741.83	744.46	739.52	744.14
2011-12-05	748.99	749.39	745.32	748.61
2011-12-06	745.55	747.77	738.01	738.01
2011-12-07	741.36	750.61	739.97	749.63
2011-12-08	745.62	748.39	742.32	745.11
2011-12-09	735.99	741.82	735.99	738.12
2011-12-12	747.73	749.29	745.65	746.69
2011-12-13	737.47	743.74	736.05	740.71
2011-12-14	738.45	739.46	734.90	736.98
2011-12-15	732.17	733.06	724.97	725.02
2011-12-16	726.99	727.85	722.75	723.56
2011-12-19	720.38	720.65	712.27	716.38
2011-12-20	718.13	720.59	717.79	718.49
2011-12-21	727.00	727.60	723.88	725.68
2011-12-22	724.19	725.33	721.77	723.12
2011-12-26	730.21	730.21	725.82	726.44
2011-12-27	723.75	725.38	723.02	724.25
2011-12-28	723.34	725.06	721.04	721.45
2011-12-29	719.24	722.98	716.82	722.12
2011-12-30	724.39	728.62	723.08	728.61
2012-01-04	737.52	744.30	736.60	742.99
2012-01-05	740.70	740.70	736.09	736.28
2012-01-06	735.25	735.82	725.97	729.60
2012-01-10	732.95	736.26	731.37	731.93
2012-01-11	733.45	735.89	731.36	733.47
2012-01-12	732.13	732.21	725.28	727.15
2012-01-13	731.82	735.85	731.82	734.60
2012-01-16	728.54	728.54	722.85	725.24
2012-01-17	727.97	731.61	727.14	731.53
2012-01-18	730.23	740.51	728.66	734.98
2012-01-19	738.68	743.78	738.68	740.68
2012-01-20	750.48	757.10	750.01	755.47
2012-01-23	755.41	758.41	753.83	756.79
2012-01-24	760.25	760.61	755.76	757.40
2012-01-25	761.79	769.36	759.97	767.40
2012-01-26	768.14	768.61	763.28	764.61
2012-01-27	763.95	766.38	759.97	761.13
2012-01-30	758.56	760.96	756.32	757.01
2012-01-31	755.69	759.31	753.84	755.27
2012-02-01	755.10	760.87	754.84	757.96
2012-02-02	762.52	766.13	762.13	762.45
2012-02-03	761.63	764.85	759.58	760.69
2012-02-06	770.01	772.36	768.54	769.85
2012-02-07	768.64	773.17	768.64	772.77
2012-02-08	776.52	782.34	775.75	782.34
2012-02-09	781.34	785.62	777.67	784.49
2012-02-10	785.88	786.05	778.78	779.07
2012-02-13	777.45	784.08	777.42	781.68
2012-02-14	780.25	787.70	778.65	786.80
2012-02-15	791.76	806.63	791.03	802.96
2012-02-16	799.87	806.38	798.40	800.25
2012-02-17	810.69	815.45	809.64	810.45
2012-02-20	821.45	823.48	817.34	819.03
2012-02-21	817.82	821.86	814.61	816.29
2012-02-22	816.99	826.11	815.43	825.40
2012-02-23	825.19	831.37	820.93	829.35
2012-02-24	829.80	835.10	828.67	834.29
2012-02-27	841.45	843.74	834.80	835.25
2012-02-28	828.69	839.20	825.81	838.48
2012-02-29	843.82	847.83	834.21	835.96
2012-03-01	840.23	845.91	827.56	831.54
2012-03-02	839.33	841.00	833.34	837.82
2012-03-05	836.44	840.39	830.90	832.86
2012-03-06	833.14	835.58	824.79	827.35
2012-03-07	815.87	823.42	815.85	822.71
2012-03-08	830.20	836.32	829.82	836.16
2012-03-09	847.26	853.35	842.07	848.71
2012-03-12	854.37	854.37	845.28	845.28
2012-03-13	846.54	854.24	845.33	845.33
2012-03-14	858.64	861.46	857.11	857.11
2012-03-15	862.62	866.42	858.71	863.61
2012-03-16	862.48	866.73	861.81	866.73
2012-03-19	868.38	870.35	866.99	868.35
2012-03-21	864.47	865.42	858.42	858.78
2012-03-22	856.39	863.82	856.21	862.07
2012-03-23	852.77	854.97	852.31	852.53
2012-03-26	854.41	856.03	851.82	851.82
2012-03-27	862.36	872.42	862.30	872.42
2012-03-28	863.02	865.32	859.42	864.43
2012-03-29	860.92	862.15	855.38	857.74
2012-03-30	857.20	857.26	851.63	854.35
2012-04-02	862.07	863.23	855.49	856.05
2012-04-03	854.17	854.33	849.24	851.02
2012-04-04	851.77	853.39	833.97	835.36
2012-04-05	827.07	835.05	823.77	832.57
2012-04-06	826.67	829.92	823.61	825.71
2012-04-09	815.36	819.42	813.69	813.69
2012-04-10	817.28	822.05	811.67	813.43
2012-04-11	802.16	808.50	800.57	805.84
2012-04-12	807.90	811.44	804.30	809.88
2012-04-13	816.40	819.14	813.28	815.48
2012-04-16	806.99	809.16	803.62	803.83
2012-04-17	805.14	807.92	802.72	803.09
2012-04-18	813.55	820.42	813.51	819.27
2012-04-19	814.37	817.08	812.47	814.13
2012-04-20	814.12	814.22	810.08	811.94
2012-04-23	814.41	818.88	808.18	809.54
2012-04-24	802.85	806.78	801.20	803.94
cut3
2009-01-05	8991.21	9127.38	8987.36	9043.12
2009-01-06	9130.01	9171.03	9029.94	9080.84
2009-01-07	9133.80	9325.35	9106.05	9239.24
2009-01-08	9143.21	9148.83	8876.42	8876.42
2009-01-09	8932.71	8956.85	8773.20	8836.80
2009-01-13	8732.63	8732.93	8405.50	8413.91
2009-01-14	8425.75	8516.07	8359.16	8438.45
2009-01-15	8309.38	8309.38	7997.73	8023.31
2009-01-16	8125.20	8283.91	8067.47	8230.15
2009-01-19	8318.26	8351.68	8221.84	8256.85
2009-01-20	8187.14	8190.42	7962.46	8065.79
2009-01-21	7949.96	8009.22	7829.30	7901.64
2009-01-22	7988.30	8051.74	7809.89	8051.74
2009-01-23	7965.41	7965.41	7745.25	7745.25
2009-01-26	7714.26	7807.16	7671.04	7682.14
2009-01-27	7782.90	8115.15	7782.07	8061.07
2009-01-28	8052.25	8171.63	7936.59	8106.29
2009-01-29	8201.16	8305.38	8138.99	8251.24
2009-01-30	8142.88	8142.88	7922.39	7994.05
2009-02-02	7908.51	7955.75	7795.27	7873.98
2009-02-03	7862.95	8084.41	7800.80	7825.51
2009-02-04	7897.24	8084.97	7863.65	8038.94
2009-02-05	7985.53	8093.96	7901.04	7949.65
2009-02-06	8054.27	8169.04	8033.24	8076.62
2009-02-09	8178.07	8257.71	7969.03	7969.03
2009-02-10	8066.94	8124.79	7917.27	7945.94
2009-02-12	7842.53	7862.52	7685.68	7705.36
2009-02-13	7789.35	7887.74	7730.27	7779.40
2009-02-16	7732.68	7804.24	7694.73	7750.17
2009-02-17	7690.13	7710.43	7615.94	7645.51
2009-02-18	7539.96	7565.79	7479.18	7534.44
2009-02-19	7604.22	7642.69	7537.56	7557.65
2009-02-20	7544.07	7554.70	7382.33	7416.38
2009-02-23	7314.30	7417.18	7209.43	7376.16
2009-02-24	7266.68	7270.90	7155.16	7268.56
2009-02-25	7368.44	7471.03	7330.44	7461.22
2009-02-26	7470.60	7599.81	7433.06	7457.93
2009-02-27	7463.42	7589.77	7414.40	7568.42
2009-03-02	7454.28	7454.28	7234.96	7280.15
2009-03-03	7177.79	7288.14	7088.47	7229.72
2009-03-04	7146.71	7320.65	7104.63	7290.96
2009-03-05	7336.02	7532.87	7336.02	7433.49
2009-03-06	7328.29	7328.29	7167.07	7173.10
2009-03-09	7191.13	7241.02	7028.49	7086.03
2009-03-10	7059.77	7100.77	7021.28	7054.98
2009-03-11	7165.39	7393.81	7161.85	7376.12
2009-03-12	7320.45	7345.02	7198.25	7198.25
2009-03-13	7301.12	7571.45	7300.87	7569.28
2009-03-16	7630.20	7754.75	7630.20	7704.15
2009-03-17	7767.34	7967.03	7723.94	7949.13
2009-03-18	8006.86	8054.35	7895.28	7972.17
2009-03-19	8017.93	8034.09	7902.49	7945.96
2009-03-23	7943.14	8229.13	7922.55	8215.53
2009-03-24	8334.68	8504.41	8297.27	8488.30
2009-03-25	8499.69	8553.01	8392.56	8479.99
2009-03-26	8430.22	8640.28	8383.99	8636.33
2009-03-27	8711.72	8843.18	8626.97	8626.97
2009-03-30	8621.85	8651.06	8236.08	8236.08
2009-03-31	8199.43	8383.74	8088.45	8109.53
2009-04-01	8173.36	8351.91	8084.62	8351.91
2009-04-02	8453.73	8741.67	8449.87	8719.78
2009-04-03	8814.10	8884.63	8697.17	8749.84
2009-04-06	8856.84	8992.06	8812.36	8857.93
2009-04-07	8838.66	8884.45	8778.92	8832.85
2009-04-08	8746.73	8765.64	8556.75	8595.01
2009-04-09	8665.16	8920.86	8664.26	8916.06
2009-04-10	9041.23	9068.80	8856.69	8964.11
2009-04-13	8930.35	9024.45	8888.10	8924.43
2009-04-14	8955.90	8961.73	8749.92	8842.68
2009-04-15	8777.68	8800.52	8681.03	8742.96
2009-04-16	8848.43	9030.00	8720.62	8755.26
2009-04-17	8854.33	8953.34	8834.63	8907.58
2009-04-20	8899.59	8933.80	8813.72	8924.75
2009-04-21	8802.09	8802.09	8612.76	8711.33
2009-04-22	8777.53	8802.90	8683.27	8727.30
2009-04-23	8776.94	8860.55	8647.80	8847.01
2009-04-24	8832.10	8852.83	8694.92	8707.99
2009-04-27	8783.34	8840.53	8648.51	8726.34
2009-04-28	8678.28	8808.64	8493.77	8493.77
2009-04-30	8615.45	8844.77	8615.45	8828.26
2009-05-01	8848.84	9017.32	8827.13	8977.37
2009-05-07	9102.35	9385.70	9100.11	9385.70
2009-05-08	9351.40	9464.43	9349.57	9432.83
2009-05-11	9460.72	9503.91	9342.75	9451.98
2009-05-12	9358.25	9389.61	9298.61	9298.61
2009-05-13	9305.79	9379.47	9278.89	9340.49
2009-05-14	9212.30	9223.77	9052.41	9093.73
2009-05-15	9150.21	9272.08	9140.90	9265.02
2009-05-18	9167.05	9167.82	8997.74	9038.69
2009-05-19	9172.56	9326.75	9166.97	9290.29
2009-05-20	9372.72	9399.40	9311.61	9344.64
2009-05-21	9280.35	9286.35	9189.92	9264.15
2009-05-22	9156.14	9267.82	9126.69	9225.81
2009-05-25	9245.95	9402.76	9245.95	9347.00
2009-05-26	9364.22	9379.78	9231.93	9310.81
2009-05-27	9426.62	9491.13	9411.54	9438.77
2009-05-28	9353.33	9492.66	9353.33	9451.39
2009-05-29	9478.21	9522.50	9426.52	9522.50
2009-06-01	9517.49	9691.73	9491.26	9677.75
2009-06-02	9774.55	9793.47	9704.31	9704.31
2009-06-03	9723.82	9774.55	9718.70	9741.67
2009-06-04	9690.93	9743.46	9654.45	9668.96
2009-06-05	9752.51	9774.98	9719.14	9768.01
2009-06-08	9829.28	9914.07	9816.17	9865.63
2009-06-09	9824.17	9855.89	9755.81	9786.82
2009-06-10	9833.64	9991.49	9813.77	9991.49
2009-06-11	9992.98	10022.23	9958.98	9981.33
2009-06-12	10088.21	10170.82	10050.96	10135.82
2009-06-15	10126.55	10126.55	10029.59	10039.67
2009-06-16	9914.40	9941.50	9752.88	9752.88
2009-06-17	9705.73	9845.52	9705.73	9840.85
2009-06-18	9778.35	9778.35	9649.83	9703.72
2009-06-19	9757.85	9822.83	9703.87	9786.26
2009-06-22	9788.07	9887.13	9763.37	9826.27
2009-06-23	9695.76	9712.09	9511.45	9549.61
2009-06-24	9596.78	9624.41	9534.24	9590.32
2009-06-25	9628.18	9871.14	9619.02	9796.08
2009-06-26	9852.92	9893.00	9793.95	9877.39
2009-06-29	9866.28	9943.83	9748.73	9783.47
2009-06-30	9896.56	10000.30	9894.95	9958.44
2009-07-01	9889.34	10086.18	9874.00	9939.93
2009-07-02	9993.77	9993.77	9869.54	9876.15
2009-07-03	9751.69	9816.07	9691.16	9816.07
2009-07-06	9738.49	9771.76	9650.75	9680.87
2009-07-07	9713.62	9734.43	9619.68	9647.79
2009-07-08	9548.81	9557.39	9407.98	9420.75
2009-07-09	9342.33	9384.24	9291.06	9291.06
2009-07-10	9338.76	9370.15	9265.24	9287.28
2009-07-13	9242.13	9343.86	9050.33	9050.33
2009-07-14	9173.14	9284.38	9166.63	9261.81
2009-07-15	9307.45	9333.64	9250.73	9269.25
2009-07-16	9393.98	9489.67	9317.11	9344.16
2009-07-17	9413.86	9420.61	9360.59	9395.32
2009-07-21	9512.52	9652.02	9508.92	9652.02
2009-07-22	9642.79	9750.63	9608.77	9723.16
2009-07-23	9712.10	9861.29	9703.84	9792.94
2009-07-24	9909.40	9950.04	9863.87	9944.55
2009-07-27	10020.87	10179.59	10015.11	10088.66
2009-07-28	10116.89	10116.89	10049.63	10087.26
2009-07-29	10035.91	10161.95	10035.91	10113.24
2009-07-30	10155.71	10169.68	10067.51	10165.21
2009-07-31	10275.14	10359.07	10259.57	10356.83
2009-08-03	10355.23	10380.57	10322.00	10352.47
2009-08-04	10438.40	10479.19	10358.48	10375.01
2009-08-05	10405.05	10406.10	10252.53	10252.53
2009-08-06	10266.01	10426.14	10266.01	10388.09
2009-08-07	10365.71	10412.09	10249.04	10412.09
2009-08-10	10530.62	10585.37	10487.70	10524.26
2009-08-11	10514.74	10587.36	10512.21	10585.46
2009-08-12	10486.36	10526.35	10434.09	10435.00
2009-08-13	10508.09	10556.28	10489.12	10517.19
2009-08-14	10551.33	10630.38	10539.43	10597.33
2009-08-17	10521.02	10521.02	10268.61	10268.61
2009-08-18	10208.12	10325.06	10181.10	10284.96
2009-08-19	10269.97	10327.17	10195.11	10204.00
2009-08-20	10254.53	10407.30	10204.26	10383.41
2009-08-21	10375.22	10375.22	10142.22	10238.20
2009-08-24	10389.57	10591.00	10388.78	10581.05
2009-08-25	10515.57	10566.31	10469.69	10497.36
2009-08-26	10554.44	10668.74	10526.51	10639.71
2009-08-27	10570.78	10575.79	10400.83	10473.97
2009-08-28	10546.88	10560.44	10471.63	10534.14
2009-08-31	10608.16	10767.00	10423.90	10492.53
2009-09-01	10453.37	10577.19	10438.51	10530.06
2009-09-02	10378.08	10378.08	10215.60	10280.46
2009-09-03	10240.02	10290.83	10185.38	10214.64
2009-09-04	10256.49	10256.49	10143.26	10187.11
2009-09-07	10288.56	10326.45	10274.38	10320.94
2009-09-08	10353.51	10393.23	10300.50	10393.23
2009-09-09	10343.80	10390.25	10249.11	10312.14
2009-09-10	10385.35	10521.14	10382.99	10513.67
2009-09-11	10519.33	10522.23	10407.97	10444.33
2009-09-14	10388.32	10388.32	10170.74	10202.06
2009-09-15	10255.73	10292.47	10183.30	10217.62
2009-09-16	10288.72	10394.56	10247.24	10270.77
2009-09-17	10373.25	10443.80	10340.14	10443.80
2009-09-18	10379.21	10399.42	10292.50	10370.54
2009-09-24	10405.53	10566.98	10386.88	10544.22
2009-09-25	10395.45	10395.45	10228.60	10265.98
2009-09-28	10136.24	10151.60	9971.05	10009.52
2009-09-29	10089.11	10125.88	10059.87	10100.20
2009-09-30	10105.17	10138.04	10063.69	10133.23
2009-10-01	10072.64	10077.22	9965.06	9978.64
2009-10-02	9848.07	9852.17	9714.18	9731.87
2009-10-05	9733.07	9751.56	9669.31	9674.49
2009-10-06	9744.42	9744.42	9628.67	9691.80
2009-10-07	9750.24	9821.74	9735.68	9799.60
2009-10-08	9806.03	9863.34	9746.15	9832.47
2009-10-09	9893.63	10016.39	9857.39	10016.39
2009-10-13	10066.85	10116.72	10039.62	10076.56
2009-10-14	10096.01	10096.01	10016.24	10060.21
2009-10-15	10194.91	10272.62	10184.81	10238.65
2009-10-16	10276.40	10290.31	10216.14	10257.56
2009-10-19	10179.46	10264.44	10125.33	10236.51
2009-10-20	10329.30	10357.59	10307.59	10336.84
2009-10-21	10292.39	10350.45	10292.39	10333.39
2009-10-22	10251.09	10274.61	10159.66	10267.17
2009-10-23	10335.34	10364.17	10271.71	10282.99
2009-10-26	10271.84	10397.69	10250.84	10362.62
2009-10-27	10283.46	10290.92	10201.93	10212.46
2009-10-28	10182.14	10209.20	10060.39	10075.05
2009-10-29	9941.57	9945.41	9850.12	9891.10
2009-10-30	10006.84	10049.59	9984.33	10034.74
2009-11-02	9903.77	9904.61	9736.14	9802.95
2009-11-04	9785.28	9845.00	9767.91	9844.31
2009-11-05	9803.50	9825.82	9691.69	9717.44
2009-11-06	9812.73	9849.72	9767.00	9789.35
2009-11-09	9777.64	9845.90	9731.59	9808.99
2009-11-10	9904.68	9979.46	9857.10	9870.73
2009-11-11	9890.18	9949.71	9849.99	9871.68
2009-11-12	9921.45	9943.57	9788.49	9804.49
2009-11-13	9793.19	9805.20	9719.38	9770.31
2009-11-16	9784.14	9802.53	9725.81	9791.18
2009-11-17	9834.58	9847.64	9715.28	9729.93
2009-11-18	9723.23	9788.81	9631.83	9676.80
2009-11-19	9691.86	9691.86	9496.07	9549.47
2009-11-20	9460.21	9513.60	9423.66	9497.68
2009-11-24	9511.00	9511.00	9397.79	9401.58
2009-11-25	9381.90	9453.70	9366.33	9441.64
2009-11-26	9354.80	9457.24	9324.99	9383.24
2009-11-27	9257.20	9257.20	9076.41	9081.52
2009-11-30	9189.17	9353.74	9162.18	9345.55
2009-12-01	9281.82	9572.20	9233.20	9572.20
2009-12-02	9551.79	9643.32	9514.15	9608.94
2009-12-03	9707.02	9977.67	9707.02	9977.67
2009-12-04	10019.24	10022.59	9902.83	10022.59
2009-12-07	10131.90	10204.58	10106.31	10167.60
2009-12-08	10079.81	10149.32	10079.81	10140.47
2009-12-09	10048.94	10048.94	9985.97	10004.72
2009-12-10	9964.05	10035.92	9834.22	9862.82
2009-12-11	9958.00	10107.87	9916.21	10107.87
2009-12-14	10126.61	10126.61	10009.60	10105.68
2009-12-15	10053.45	10111.76	10033.90	10083.48
2009-12-16	10179.33	10222.22	10117.12	10177.41
2009-12-17	10199.92	10260.12	10163.80	10163.80
2009-12-18	10110.64	10157.25	10027.85	10142.05
2009-12-21	10196.71	10215.49	10183.47	10183.47
2009-12-22	10256.00	10378.03	10235.22	10378.03
2009-12-24	10413.37	10558.41	10413.37	10536.92
2009-12-25	10546.97	10546.97	10476.65	10494.71
2009-12-28	10521.81	10652.99	10513.55	10634.23
2009-12-29	10611.85	10683.12	10597.41	10638.06
2009-12-30	10707.51	10707.51	10546.44	10546.44
2010-01-04	10609.34	10694.49	10608.14	10654.79
2010-01-05	10719.44	10791.04	10655.57	10681.83
2010-01-06	10709.55	10768.61	10661.17	10731.45
2010-01-07	10742.75	10774.00	10636.67	10681.66
2010-01-08	10743.30	10816.45	10677.56	10798.32
2010-01-12	10770.35	10905.39	10763.68	10879.14
2010-01-13	10795.48	10866.62	10729.86	10735.03
2010-01-14	10778.07	10909.94	10774.25	10907.68
2010-01-15	10917.41	10982.10	10878.83	10982.10
2010-01-18	10887.61	10895.10	10781.03	10855.08
2010-01-19	10866.83	10866.83	10749.47	10764.90
2010-01-20	10834.91	10860.93	10724.57	10737.52
2010-01-21	10704.79	10886.64	10649.84	10868.41
2010-01-22	10740.21	10768.07	10528.33	10590.55
2010-01-25	10478.31	10557.64	10414.58	10512.69
2010-01-26	10506.15	10566.49	10324.98	10325.28
2010-01-27	10344.07	10373.82	10252.08	10252.08
2010-01-28	10309.73	10462.70	10296.98	10414.29
2010-01-29	10308.05	10324.37	10198.04	10198.04
2010-02-01	10212.36	10224.83	10129.91	10205.02
2010-02-02	10310.98	10396.48	10287.74	10371.09
2010-02-03	10428.12	10436.52	10356.03	10404.33
2010-02-04	10434.52	10438.41	10279.57	10355.98
2010-02-05	10162.34	10166.30	10036.33	10057.09
2010-02-08	10007.47	10063.53	9942.05	9951.82
2010-02-09	9876.61	9956.79	9867.39	9932.90
2010-02-10	10024.26	10049.87	9963.99	9963.99
2010-02-12	10085.35	10099.46	10014.50	10092.19
2010-02-15	10097.82	10119.47	10012.53	10013.30
2010-02-16	10044.53	10062.27	10019.43	10034.25
2010-02-17	10161.72	10306.83	10150.24	10306.83
2010-02-18	10324.98	10340.38	10285.05	10335.69
2010-02-19	10334.40	10354.42	10123.58	10123.58
2010-02-22	10302.07	10449.75	10298.15	10400.47
2010-02-23	10327.64	10375.55	10280.33	10352.10
2010-02-24	10205.15	10243.20	10129.65	10198.83
2010-02-25	10256.10	10267.43	10087.21	10101.96
2010-02-26	10107.08	10171.22	10085.13	10126.03
2010-03-01	10128.73	10215.15	10116.86	10172.06
2010-03-02	10199.19	10238.96	10150.30	10221.84
2010-03-03	10193.95	10274.09	10186.68	10253.14
2010-03-04	10255.69	10263.48	10134.37	10145.72
2010-03-05	10254.61	10376.41	10254.61	10368.96
2010-03-08	10538.12	10586.97	10514.28	10585.92
2010-03-09	10567.32	10593.27	10542.81	10567.65
2010-03-10	10555.85	10588.69	10547.34	10563.92
2010-03-11	10627.99	10664.95	10588.86	10664.95
2010-03-12	10777.49	10777.49	10699.30	10751.26
2010-03-15	10802.11	10808.84	10708.97	10751.98
2010-03-16	10718.63	10776.52	10716.20	10721.71
2010-03-17	10789.24	10864.30	10761.89	10846.98
2010-03-18	10841.71	10845.73	10731.66	10744.03
2010-03-19	10785.88	10826.76	10775.57	10824.72
2010-03-23	10793.05	10817.78	10766.65	10774.15
2010-03-24	10829.36	10880.62	10769.18	10815.03
2010-03-25	10857.81	10872.45	10808.71	10828.85
2010-03-26	10895.29	11001.59	10889.33	10996.37
2010-03-29	10909.98	10990.75	10901.20	10986.47
2010-03-30	11022.20	11108.82	11003.35	11097.14
2010-03-31	11138.72	11147.62	11085.55	11089.94
2010-04-01	11178.92	11272.73	11118.18	11244.40
2010-04-02	11274.18	11313.98	11235.98	11286.09
2010-04-05	11380.63	11408.17	11306.62	11339.30
2010-04-06	11350.70	11358.45	11217.25	11282.32
2010-04-07	11296.31	11350.57	11260.42	11292.83
2010-04-08	11179.67	11232.36	11159.45	11168.20
2010-04-09	11174.62	11204.87	11148.80	11204.34
2010-04-12	11300.79	11351.55	11248.57	11251.90
2010-04-13	11202.92	11212.65	11088.48	11161.23
2010-04-14	11213.46	11270.20	11161.01	11204.90
2010-04-15	11310.07	11316.66	11255.80	11273.79
2010-04-16	11230.53	11230.53	11084.72	11102.18
2010-04-19	10929.64	10953.90	10878.99	10908.77
2010-04-20	10960.11	10999.57	10890.82	10900.68
2010-04-21	11031.79	11093.30	11023.88	11090.05
2010-04-22	10988.01	10989.62	10866.23	10949.09
2010-04-23	10930.90	10967.12	10865.92	10914.46
2010-04-26	11039.59	11165.79	11035.66	11165.79
2010-04-27	11109.26	11213.45	11108.72	11212.66
2010-04-28	11005.51	11008.88	10882.40	10924.79
2010-04-30	11062.12	11092.52	11035.27	11057.40
2010-05-06	10847.90	10847.90	10682.61	10695.69
2010-05-07	10472.70	10472.77	10257.32	10364.59
2010-05-10	10399.05	10540.52	10362.34	10530.70
2010-05-11	10643.24	10643.28	10392.47	10411.10
2010-05-12	10465.03	10501.72	10345.19	10394.03
2010-05-13	10542.77	10638.23	10535.94	10620.55
2010-05-14	10453.40	10551.69	10390.77	10462.51
2010-05-17	10329.37	10331.87	10158.30	10235.76
2010-05-18	10313.14	10336.53	10197.74	10242.64
2010-05-19	10087.33	10186.84	10041.93	10186.84
2010-05-20	10121.85	10161.70	9999.59	10030.31
2010-05-21	9823.84	9829.36	9696.63	9784.54
2010-05-24	9754.99	9793.46	9693.07	9758.40
2010-05-25	9632.14	9642.60	9432.09	9459.89
2010-05-26	9567.35	9622.40	9477.45	9522.66
2010-05-27	9419.30	9642.16	9395.29	9639.72
2010-05-28	9806.06	9824.85	9699.14	9762.98
2010-05-31	9735.47	9831.73	9728.42	9768.70
2010-06-01	9747.26	9747.26	9658.44	9711.83
2010-06-02	9604.45	9763.41	9560.16	9603.24
2010-06-03	9758.82	9920.18	9758.82	9914.19
2010-06-04	9909.24	9962.42	9868.09	9901.19
2010-06-07	9705.36	9714.42	9502.62	9520.80
2010-06-08	9448.79	9587.65	9434.81	9537.94
2010-06-09	9475.57	9487.91	9378.23	9439.13
2010-06-10	9476.78	9544.04	9437.47	9542.65
2010-06-11	9701.32	9764.73	9673.60	9705.25
2010-06-14	9824.93	9882.05	9819.95	9879.85
2010-06-15	9845.65	9911.73	9826.44	9887.89
2010-06-16	10043.10	10109.86	10033.24	10067.15
2010-06-17	10009.25	10053.45	9989.21	9999.40
2010-06-18	10027.98	10029.11	9965.16	9995.02
2010-06-21	10109.27	10251.90	10109.27	10238.01
2010-06-22	10131.91	10183.89	10105.48	10112.89
2010-06-23	9973.00	9975.01	9912.80	9923.70
2010-06-24	9902.62	10009.37	9893.75	9928.34
2010-06-25	9778.70	9794.98	9697.87	9737.48
2010-06-28	9758.91	9762.23	9679.17	9693.94
2010-06-29	9718.40	9760.02	9548.98	9570.67
2010-06-30	9396.45	9420.14	9347.07	9382.64
2010-07-01	9296.86	9297.45	9147.68	9191.60
2010-07-02	9234.56	9259.14	9160.47	9203.71
2010-07-05	9230.00	9282.90	9213.79	9266.78
2010-07-06	9158.28	9351.11	9091.70	9338.04
2010-07-07	9322.56	9329.64	9223.80	9279.65
2010-07-08	9462.82	9545.99	9456.19	9535.74
2010-07-09	9601.62	9610.59	9516.97	9585.32
2010-07-12	9540.47	9632.89	9533.56	9548.11
2010-07-13	9592.87	9629.98	9502.24	9537.23
2010-07-14	9707.50	9807.36	9693.33	9795.24
2010-07-15	9701.14	9726.94	9667.00	9685.53
2010-07-16	9600.88	9636.84	9392.21	9408.36
2010-07-20	9267.24	9378.91	9247.76	9300.46
2010-07-21	9404.07	9404.07	9230.84	9278.83
2010-07-22	9216.28	9254.19	9176.12	9220.88
2010-07-23	9380.12	9474.11	9367.30	9430.96
2010-07-26	9520.73	9561.82	9493.61	9503.66
2010-07-27	9512.67	9541.59	9487.65	9496.85
2010-07-28	9614.47	9760.31	9614.47	9753.27
2010-07-29	9653.51	9732.76	9648.97	9696.02
2010-07-30	9658.61	9658.61	9505.64	9537.30
2010-08-02	9574.64	9676.41	9548.86	9570.31
2010-08-03	9716.57	9750.88	9632.86	9694.01
2010-08-04	9606.82	9613.13	9474.67	9489.34
2010-08-05	9640.34	9688.87	9578.31	9653.92
2010-08-06	9560.65	9658.04	9545.41	9642.12
2010-08-09	9535.51	9572.83	9523.63	9572.49
2010-08-10	9629.97	9667.04	9505.34	9551.05
2010-08-11	9423.03	9445.04	9283.60	9292.85
2010-08-12	9125.49	9212.59	9065.94	9212.59
2010-08-13	9214.81	9278.46	9163.25	9253.46
2010-08-16	9145.22	9210.97	9095.94	9196.67
2010-08-17	9089.85	9188.48	9084.24	9161.68
2010-08-18	9261.50	9280.59	9153.42	9240.54
2010-08-19	9242.84	9362.68	9242.84	9362.68
2010-08-20	9239.36	9287.50	9169.17	9179.38
2010-08-23	9147.24	9171.42	9090.96	9116.69
2010-08-24	9024.60	9069.22	8983.52	8995.14
2010-08-25	8904.44	8949.41	8807.41	8845.39
2010-08-26	8908.02	8910.62	8834.09	8906.48
2010-08-27	8811.47	9021.75	8810.46	8991.06
2010-08-30	9141.45	9280.70	9117.87	9149.26
2010-08-31	8998.80	9008.36	8819.26	8824.06
2010-09-01	8833.32	8932.15	8796.45	8927.02
2010-09-02	9069.62	9083.37	8958.81	9062.84
2010-09-03	9097.59	9141.09	9050.83	9114.13
2010-09-06	9199.99	9303.00	9173.38	9301.32
2010-09-07	9232.02	9311.02	9206.81	9226.00
2010-09-08	9098.86	9105.89	8997.63	9024.60
2010-09-09	9120.54	9136.26	9069.93	9098.39
2010-09-10	9132.25	9291.50	9132.25	9239.17
2010-09-13	9325.64	9390.90	9317.30	9321.82
2010-09-14	9319.50	9330.05	9258.18	9299.31
2010-09-15	9243.48	9578.53	9199.08	9516.56
2010-09-16	9613.89	9620.90	9479.63	9509.50
2010-09-17	9581.65	9643.28	9554.36	9626.09
2010-09-21	9700.88	9704.25	9579.83	9602.11
2010-09-22	9563.20	9625.41	9546.82	9566.32
2010-09-24	9435.04	9601.77	9415.96	9471.67
2010-09-27	9557.37	9613.47	9539.12	9603.14
2010-09-28	9547.03	9553.91	9471.35	9495.76
2010-09-29	9530.05	9600.93	9502.34	9559.38
2010-09-30	9554.83	9566.84	9369.35	9369.35
2010-10-01	9440.52	9470.36	9358.62	9404.23
2010-10-04	9410.92	9508.36	9358.82	9381.06
2010-10-05	9337.43	9538.35	9332.19	9518.76
2010-10-06	9588.47	9693.74	9567.00	9691.43
2010-10-07	9661.09	9716.92	9649.17	9684.81
2010-10-08	9675.59	9675.59	9588.88	9588.88
2010-10-12	9648.59	9650.20	9387.73	9388.64
2010-10-13	9473.78	9510.19	9393.52	9403.51
2010-10-14	9515.17	9614.44	9481.51	9583.51
2010-10-15	9551.21	9553.57	9478.64	9500.25
2010-10-18	9517.68	9564.85	9471.48	9498.49
2010-10-19	9490.09	9575.14	9489.68	9539.45
2010-10-20	9399.65	9402.05	9316.97	9381.60
2010-10-21	9443.95	9479.25	9326.78	9376.48
2010-10-22	9404.09	9457.62	9390.14	9426.71
2010-10-25	9424.44	9460.70	9392.68	9401.16
2010-10-26	9386.55	9455.04	9362.13	9377.38
2010-10-27	9436.01	9453.79	9329.86	9387.03
2010-10-28	9368.61	9413.68	9350.44	9366.03
2010-10-29	9327.25	9337.92	9179.15	9202.45
2010-11-01	9166.85	9258.32	9134.47	9154.72
2010-11-02	9140.28	9172.03	9123.62	9159.98
2010-11-04	9282.79	9377.76	9276.71	9358.78
2010-11-05	9500.60	9680.68	9497.45	9625.99
2010-11-08	9699.79	9737.01	9677.87	9732.92
2010-11-09	9667.44	9726.60	9659.86	9694.49
2010-11-10	9748.61	9842.90	9747.03	9830.52
2010-11-11	9878.11	9885.37	9834.82	9861.46
2010-11-12	9810.30	9851.42	9724.81	9724.81
2010-11-15	9782.11	9830.59	9744.99	9827.51
2010-11-16	9892.82	9908.30	9774.12	9797.10
2010-11-17	9693.21	9817.37	9693.21	9811.66
2010-11-18	9820.60	10013.63	9798.56	10013.63
2010-11-19	10124.84	10130.23	10019.08	10022.39
2010-11-22	10133.48	10157.97	10091.99	10115.19
2010-11-24	9942.54	10064.31	9904.92	10030.11
2010-11-25	10117.85	10123.27	10038.79	10079.76
2010-11-26	10078.09	10134.64	10039.56	10039.56
2010-11-29	10075.66	10149.94	10044.01	10125.99
2010-11-30	10070.31	10101.42	9937.04	9937.04
2010-12-01	9939.80	9988.05	9918.55	9988.05
2010-12-02	10151.15	10187.59	10143.08	10168.52
2010-12-03	10231.78	10254.00	10154.99	10178.32
2010-12-06	10175.23	10179.82	10143.58	10167.23
2010-12-07	10152.92	10167.10	10094.41	10141.10
2010-12-08	10200.16	10258.90	10184.32	10232.33
2010-12-09	10277.87	10298.25	10240.11	10285.88
2010-12-10	10373.70	10373.70	10194.27	10211.95
2010-12-13	10212.52	10293.89	10199.73	10293.89
2010-12-14	10272.70	10319.24	10267.22	10316.77
2010-12-15	10338.14	10340.69	10282.85	10309.78
2010-12-16	10297.55	10347.39	10282.54	10311.29
2010-12-17	10307.17	10327.42	10280.80	10303.83
2010-12-20	10296.55	10301.15	10182.74	10216.41
2010-12-21	10277.14	10372.51	10266.29	10370.53
2010-12-22	10348.48	10394.22	10327.60	10346.48
2010-12-24	10275.97	10290.89	10264.06	10279.19
2010-12-27	10292.84	10376.91	10292.84	10355.99
2010-12-28	10330.01	10340.84	10292.63	10292.63
2010-12-29	10282.84	10344.54	10276.61	10344.54
2010-12-30	10303.13	10315.47	10209.93	10228.92
2011-01-04	10352.19	10409.17	10321.28	10398.10
2011-01-05	10387.95	10413.45	10358.01	10380.77
2011-01-06	10477.52	10530.11	10477.52	10529.76
2011-01-07	10506.72	10550.71	10503.02	10541.04
2011-01-11	10484.62	10538.39	10476.29	10510.68
2011-01-12	10562.28	10576.51	10502.96	10512.80
2011-01-13	10593.46	10620.57	10565.28	10589.76
2011-01-14	10502.77	10579.97	10497.26	10499.04
2011-01-17	10562.09	10562.39	10482.69	10502.86
2011-01-18	10461.04	10548.21	10456.71	10518.98
2011-01-19	10567.88	10580.68	10534.02	10557.10
2011-01-20	10492.32	10505.63	10421.83	10437.31
2011-01-21	10459.04	10461.07	10257.99	10274.52
2011-01-24	10318.60	10347.94	10278.81	10345.11
2011-01-25	10374.64	10480.24	10355.80	10464.42
2011-01-26	10410.29	10440.35	10392.46	10401.90
2011-01-27	10466.34	10496.40	10421.75	10478.66
2011-01-28	10460.29	10480.08	10332.65	10360.34
2011-01-31	10219.98	10265.96	10182.57	10237.92
2011-02-01	10281.55	10299.38	10245.75	10274.50
2011-02-02	10368.48	10479.75	10366.96	10457.36
2011-02-03	10424.68	10450.46	10409.84	10431.36
2011-02-04	10526.39	10580.36	10524.13	10543.52
2011-02-07	10641.30	10644.21	10572.72	10592.04
2011-02-08	10645.03	10648.80	10623.70	10635.98
2011-02-09	10686.72	10701.92	10599.50	10617.83
2011-02-10	10574.80	10631.94	10570.62	10605.65
2011-02-14	10689.65	10725.54	10674.44	10725.54
2011-02-15	10738.81	10760.99	10712.49	10746.67
2011-02-16	10758.86	10842.31	10752.61	10808.29
2011-02-17	10882.91	10891.60	10820.48	10836.64
2011-02-18	10829.15	10862.43	10810.12	10842.80
2011-02-21	10820.78	10859.39	10789.41	10857.53
2011-02-22	10745.52	10761.22	10639.78	10664.70
2011-02-23	10588.82	10666.72	10569.93	10579.10
2011-02-24	10533.20	10542.70	10428.38	10452.71
2011-02-25	10475.78	10532.21	10454.46	10526.76
2011-02-28	10504.54	10628.76	10448.83	10624.09
2011-03-01	10676.24	10754.03	10672.91	10754.03
2011-03-02	10609.28	10629.50	10492.38	10492.38
2011-03-03	10529.77	10593.34	10522.40	10586.02
2011-03-04	10730.91	10768.43	10664.37	10693.66
2011-03-07	10626.97	10626.97	10472.51	10505.02
2011-03-08	10513.08	10565.10	10509.23	10525.19
2011-03-09	10607.23	10662.07	10563.82	10589.50
2011-03-10	10544.13	10549.17	10410.10	10434.38
2011-03-11	10298.64	10378.55	10254.43	10254.43
2011-03-14	10044.17	10049.92	9578.65	9620.49
2011-03-15	9441.66	9441.66	8227.63	8605.15
2011-03-16	8767.20	9168.51	8763.95	9093.72
2011-03-17	8913.35	9093.61	8639.56	8962.67
2011-03-18	9083.95	9275.60	9073.25	9206.75
2011-03-22	9403.05	9625.03	9395.85	9608.32
2011-03-23	9590.38	9592.40	9387.95	9449.47
2011-03-24	9479.77	9509.85	9418.71	9435.01
2011-03-25	9565.49	9569.71	9469.37	9536.13
2011-03-28	9541.76	9541.76	9405.20	9478.53
2011-03-29	9348.60	9502.23	9317.38	9459.08
2011-03-30	9489.51	9708.79	9479.03	9708.79
2011-03-31	9765.28	9765.81	9658.64	9755.10
2011-04-01	9757.28	9822.06	9698.60	9708.39
2011-04-04	9773.91	9808.60	9718.89	9718.89
2011-04-05	9732.26	9732.26	9566.05	9615.55
2011-04-06	9668.11	9671.46	9562.16	9584.37
2011-04-07	9652.65	9687.18	9578.47	9590.93
2011-04-08	9586.17	9804.28	9536.68	9768.08
2011-04-11	9731.32	9775.60	9700.75	9719.70
2011-04-12	9603.50	9603.64	9513.27	9555.26
2011-04-13	9516.50	9655.83	9516.50	9641.18
2011-04-14	9579.79	9676.97	9553.75	9653.92
2011-04-15	9646.22	9651.96	9577.96	9591.52
2011-04-18	9593.41	9608.92	9539.03	9556.65
2011-04-19	9448.65	9478.87	9405.19	9441.03
2011-04-20	9536.79	9631.00	9519.36	9606.82
2011-04-21	9686.27	9725.12	9643.92	9685.77
2011-04-22	9623.00	9732.63	9611.38	9682.21
2011-04-25	9705.90	9748.40	9657.24	9671.96
2011-04-26	9627.43	9627.43	9536.51	9558.69
2011-04-27	9631.53	9728.03	9630.37	9691.84
2011-04-28	9749.36	9849.74	9712.58	9849.74
2011-05-02	9964.39	10017.47	9936.27	10004.20
2011-05-06	9851.15	9876.10	9782.66	9859.20
2011-05-09	9881.67	9897.88	9776.74	9794.38
2011-05-10	9804.27	9853.13	9755.93	9818.76
2011-05-11	9912.48	9929.04	9842.67	9864.26
2011-05-12	9770.71	9836.71	9716.65	9716.65
2011-05-13	9751.45	9751.45	9552.93	9648.77
2011-05-16	9571.13	9598.33	9552.86	9558.30
2011-05-17	9513.50	9593.13	9502.39	9567.02
2011-05-18	9602.88	9693.60	9590.40	9662.08
2011-05-19	9711.32	9731.12	9600.81	9620.82
2011-05-20	9612.87	9670.92	9600.48	9607.08
2011-05-23	9547.86	9547.86	9437.87	9460.63
2011-05-24	9406.04	9490.35	9406.04	9477.17
2011-05-25	9492.43	9503.77	9415.33	9422.88
2011-05-26	9499.45	9567.46	9489.12	9562.05
2011-05-27	9514.16	9588.70	9493.44	9521.94
2011-05-30	9489.78	9540.02	9448.00	9504.97
2011-05-31	9500.60	9697.35	9497.40	9693.73
2011-06-01	9708.05	9720.73	9659.31	9719.61
2011-06-02	9560.63	9575.01	9517.28	9555.04
2011-06-03	9550.59	9603.03	9490.68	9492.21
2011-06-06	9467.37	9490.55	9359.78	9380.35
2011-06-07	9371.21	9454.66	9358.31	9442.95
2011-06-08	9428.62	9458.43	9394.74	9449.46
2011-06-09	9411.71	9472.41	9383.12	9467.15
2011-06-10	9536.84	9613.75	9493.05	9514.44
2011-06-13	9405.16	9466.51	9391.26	9448.21
2011-06-14	9446.84	9565.65	9438.41	9547.79
2011-06-15	9589.54	9599.90	9520.93	9574.32
2011-06-16	9478.63	9497.59	9411.28	9411.28
2011-06-17	9443.32	9447.44	9318.62	9351.40
2011-06-20	9383.24	9421.38	9346.76	9354.32
2011-06-21	9420.43	9459.66	9378.36	9459.66
2011-06-22	9525.25	9658.62	9525.25	9629.43
2011-06-23	9557.72	9652.41	9553.30	9596.74
2011-06-24	9626.43	9695.09	9609.24	9678.71
2011-06-27	9633.92	9638.49	9570.58	9578.31
2011-06-28	9668.01	9703.41	9632.54	9648.98
2011-06-29	9767.91	9797.26	9734.66	9797.26
2011-06-30	9838.09	9849.69	9785.89	9816.09
2011-07-01	9878.69	9900.57	9849.64	9868.07
2011-07-04	9980.16	10005.75	9940.47	9965.09
2011-07-05	9959.39	9995.17	9948.38	9972.46
2011-07-06	9987.25	10082.48	9967.87	10082.48
2011-07-07	10047.15	10102.19	10029.33	10071.14
2011-07-08	10205.71	10207.91	10126.47	10137.73
2011-07-11	10069.01	10109.30	10057.77	10069.53
2011-07-12	9943.32	9971.14	9907.81	9925.92
2011-07-13	9891.11	9979.67	9887.33	9963.14
2011-07-14	9929.18	9998.49	9884.00	9936.12
2011-07-15	9919.59	9985.32	9919.10	9974.47
2011-07-19	9921.50	9945.72	9889.72	9889.72
2011-07-20	10008.00	10042.75	9986.33	10005.90
2011-07-21	10019.94	10026.18	9974.00	10010.39
2011-07-22	10096.89	10149.18	10076.53	10132.11
2011-07-25	10080.82	10091.02	10039.10	10050.01
2011-07-26	10078.48	10130.25	10043.99	10097.72
2011-07-27	10048.34	10053.50	10009.37	10047.19
2011-07-28	9936.97	9945.98	9853.85	9901.35
2011-07-29	9866.82	9914.22	9824.34	9833.03
2011-08-01	9907.04	10040.13	9906.17	9965.01
2011-08-02	9872.17	9873.46	9822.30	9844.59
2011-08-03	9705.02	9705.02	9610.38	9637.14
2011-08-04	9685.50	9768.40	9615.35	9659.18
2011-08-05	9469.16	9469.16	9264.09	9299.88
2011-08-08	9169.67	9215.09	9057.29	9097.56
2011-08-09	8911.80	8949.97	8656.79	8944.48
2011-08-10	9110.81	9144.33	9021.36	9038.74
2011-08-11	8876.54	8983.33	8832.42	8981.94
2011-08-12	9064.17	9070.27	8926.89	8963.72
2011-08-15	9082.53	9117.11	9032.84	9086.41
2011-08-16	9135.06	9150.31	9072.53	9107.43
2011-08-17	9047.89	9081.80	9003.70	9057.26
2011-08-18	9043.18	9043.18	8931.25	8943.76
2011-08-19	8771.70	8796.42	8707.47	8719.24
2011-08-22	8686.91	8756.10	8619.21	8628.13
2011-08-23	8703.31	8747.41	8630.49	8733.01
2011-08-24	8812.16	8825.27	8620.89	8639.61
2011-08-25	8748.01	8849.94	8741.74	8772.36
2011-08-26	8746.77	8805.10	8742.46	8797.78
2011-08-29	8802.25	8926.27	8751.29	8851.35
2011-08-30	8958.50	8992.86	8935.85	8953.90
2011-08-31	8938.31	8967.80	8905.52	8955.20
2011-09-01	9017.01	9098.15	8998.01	9060.80
2011-09-02	8980.56	9014.27	8914.65	8950.74
2011-09-05	8828.46	8842.56	8757.18	8784.46
2011-09-06	8687.11	8709.92	8588.34	8590.57
2011-09-07	8723.05	8773.26	8702.78	8763.41
2011-09-08	8863.24	8876.49	8757.58	8793.12
2011-09-09	8739.49	8803.75	8726.25	8737.66
2011-09-12	8578.71	8584.05	8520.97	8535.67
2011-09-13	8588.74	8635.88	8549.06	8616.55
2011-09-14	8623.55	8671.24	8499.34	8518.57
2011-09-15	8645.38	8695.95	8634.02	8668.86
2011-09-16	8785.28	8864.16	8774.16	8864.16
2011-09-20	8763.61	8771.72	8704.23	8721.24
2011-09-21	8717.00	8771.62	8707.16	8741.16
2011-09-22	8643.02	8643.02	8545.46	8560.26
2011-09-26	8547.62	8556.66	8359.70	8374.13
2011-09-27	8496.33	8609.95	8479.05	8609.95
2011-09-28	8607.92	8665.54	8595.85	8615.65
2011-09-29	8527.64	8706.14	8501.42	8701.23
2011-09-30	8713.91	8756.00	8653.62	8700.29
2011-10-03	8567.98	8577.52	8455.67	8545.48
2011-10-04	8426.39	8470.76	8359.24	8456.12
2011-10-05	8491.43	8501.03	8343.01	8382.98
2011-10-06	8463.83	8545.79	8463.83	8522.02
2011-10-07	8593.22	8663.57	8589.36	8605.62
2011-10-11	8733.54	8806.44	8725.62	8773.68
2011-10-12	8719.67	8763.02	8689.38	8738.90
2011-10-13	8832.81	8854.16	8808.73	8823.25
2011-10-14	8782.91	8787.89	8733.71	8747.96
2011-10-17	8881.44	8911.70	8856.58	8879.60
2011-10-18	8758.74	8773.99	8727.39	8741.91
2011-10-19	8808.25	8831.55	8745.36	8772.54
2011-10-20	8733.60	8733.60	8672.21	8682.15
2011-10-21	8679.04	8700.49	8652.26	8678.89
2011-10-24	8764.92	8844.28	8762.02	8843.98
2011-10-25	8866.43	8870.37	8755.38	8762.31
2011-10-26	8673.85	8798.80	8642.56	8748.47
2011-10-27	8779.40	8926.54	8731.77	8926.54
2011-10-28	9059.04	9086.43	9003.09	9050.47
2011-10-31	9011.18	9152.39	8988.39	8988.39
2011-11-01	8880.75	8946.00	8831.69	8835.52
2011-11-02	8719.92	8719.92	8640.42	8640.42
2011-11-04	8761.27	8814.71	8729.08	8801.40
2011-11-07	8771.82	8779.55	8741.25	8767.09
2011-11-08	8744.36	8772.04	8644.00	8655.51
2011-11-09	8724.60	8762.98	8678.78	8755.44
2011-11-10	8597.28	8600.21	8500.67	8500.80
2011-11-11	8545.92	8566.49	8485.77	8514.47
2011-11-14	8631.28	8655.31	8595.43	8603.70
2011-11-15	8546.99	8584.30	8527.63	8541.93
2011-11-16	8546.75	8567.64	8459.30	8463.16
2011-11-17	8421.88	8491.82	8400.22	8479.63
2011-11-18	8374.16	8398.80	8359.60	8374.91
2011-11-21	8349.05	8375.60	8343.93	8348.27
2011-11-22	8263.86	8350.35	8261.01	8314.74
2011-11-24	8189.42	8212.95	8157.39	8165.18
2011-11-25	8138.19	8199.67	8135.79	8160.01
2011-11-28	8269.91	8322.01	8259.71	8287.49
2011-11-29	8371.01	8477.82	8351.38	8477.82
2011-11-30	8407.63	8435.34	8361.69	8434.61
2011-12-01	8581.20	8653.88	8577.09	8597.38
2011-12-02	8603.23	8643.75	8591.93	8643.75
2011-12-05	8697.78	8704.48	8668.04	8695.98
2011-12-06	8644.97	8671.54	8571.09	8575.16
2011-12-07	8629.19	8729.81	8614.21	8722.17
2011-12-08	8664.35	8688.81	8625.19	8664.58
2011-12-09	8521.33	8576.68	8503.03	8536.46
2011-12-12	8652.98	8682.47	8633.35	8653.82
2011-12-13	8535.52	8594.32	8518.80	8552.81
2011-12-14	8513.77	8540.57	8486.37	8519.13
2011-12-15	8449.05	8455.32	8374.25	8377.37
2011-12-16	8416.21	8433.18	8390.98	8401.72
2011-12-19	8363.39	8364.93	8272.26	8296.12
2011-12-20	8318.03	8354.90	8317.73	8336.48
2011-12-21	8442.74	8471.11	8440.89	8459.98
2011-12-22	8429.54	8436.59	8383.96	8395.16
2011-12-26	8504.46	8516.53	8477.37	8479.34
2011-12-27	8443.00	8458.26	8428.53	8440.56
2011-12-28	8426.09	8456.66	8414.90	8423.62
2011-12-29	8367.71	8404.89	8330.87	8398.89
2011-12-30	8434.90	8455.35	8415.56	8455.35
2012-01-04	8549.54	8581.45	8547.70	8560.11
2012-01-05	8515.66	8519.16	8481.83	8488.71
2012-01-06	8488.98	8488.98	8349.33	8390.35
2012-01-10	8422.99	8450.59	8405.18	8422.26
2012-01-11	8440.96	8463.72	8426.03	8447.88
2012-01-12	8423.10	8426.83	8360.33	8385.59
2012-01-13	8471.10	8509.76	8458.68	8500.02
2012-01-16	8409.79	8409.79	8352.23	8378.36
2012-01-17	8420.12	8475.66	8413.22	8466.40
2012-01-18	8458.29	8595.78	8446.09	8550.58
2012-01-19	8596.68	8668.94	8596.68	8639.68
2012-01-20	8751.18	8791.39	8725.32	8766.36
2012-01-23	8753.91	8795.27	8744.54	8765.90
2012-01-24	8815.36	8825.09	8768.51	8785.33
2012-01-25	8842.01	8911.62	8816.09	8883.69
2012-01-26	8890.49	8894.60	8834.93	8849.47
2012-01-27	8851.02	8886.02	8810.89	8841.22
2012-01-30	8803.79	8832.48	8774.23	8793.05
2012-01-31	8781.44	8836.68	8776.65	8802.51
2012-02-01	8789.06	8830.28	8780.10	8809.79
2012-02-02	8865.28	8893.22	8849.25	8876.82
2012-02-03	8849.17	8877.57	8825.98	8831.93
2012-02-06	8939.99	8949.32	8916.94	8929.20
2012-02-07	8904.16	8928.44	8887.19	8917.52
2012-02-08	8971.88	9015.59	8956.78	9015.59
2012-02-09	8996.13	9018.49	8942.87	9002.24
2012-02-10	9010.53	9016.97	8947.17	8947.17
2012-02-13	8954.03	9023.50	8948.54	8999.18
2012-02-14	8978.72	9072.08	8972.74	9052.07
2012-02-15	9108.84	9314.37	9107.68	9260.34
2012-02-16	9232.38	9308.93	9214.23	9238.10
2012-02-17	9371.12	9435.03	9369.25	9384.17
2012-02-20	9534.03	9549.31	9462.67	9485.09
2012-02-21	9458.68	9517.04	9440.41	9463.02
2012-02-22	9459.74	9564.05	9442.58	9554.00
2012-02-23	9549.77	9609.84	9513.65	9595.57
2012-02-24	9594.85	9647.38	9576.21	9647.38
2012-02-27	9726.20	9736.11	9628.25	9633.93
2012-02-28	9567.12	9722.52	9528.77	9722.52
2012-02-29	9771.62	9866.41	9706.22	9723.24
2012-03-01	9771.34	9865.75	9666.02	9707.37
2012-03-02	9797.05	9803.75	9729.24	9777.03
2012-03-05	9756.98	9792.14	9673.75	9698.59
2012-03-06	9704.61	9732.89	9602.84	9637.63
2012-03-07	9509.10	9603.19	9509.10	9576.06
2012-03-08	9674.94	9768.96	9659.15	9768.96
2012-03-09	9911.07	10007.62	9853.18	9929.74
2012-03-12	10015.92	10021.51	9889.86	9889.86
2012-03-13	9921.25	10011.88	9888.30	9899.08
2012-03-14	10064.12	10115.79	10050.52	10050.52
2012-03-15	10115.40	10158.74	10077.55	10123.28
2012-03-16	10110.57	10148.47	10090.09	10129.83
2012-03-19	10150.85	10172.64	10134.48	10141.99
2012-03-21	10100.87	10132.02	10073.90	10086.49
2012-03-22	10055.16	10136.85	10052.50	10127.08
2012-03-23	10014.87	10031.96	9999.37	10011.47
2012-03-26	10040.10	10056.20	10016.05	10018.24
2012-03-27	10152.90	10255.15	10152.25	10255.15
2012-03-28	10153.51	10197.80	10124.28	10182.57
2012-03-29	10134.78	10147.04	10084.77	10114.79
2012-03-30	10089.10	10111.09	10033.19	10083.56
2012-04-02	10161.72	10190.35	10109.87	10109.87
2012-04-03	10082.21	10084.07	10040.99	10050.39
2012-04-04	10045.88	10059.19	9819.99	9819.99
2012-04-05	9737.50	9806.40	9692.70	9767.61
2012-04-06	9710.25	9739.59	9659.16	9688.45
2012-04-09	9565.31	9618.10	9535.33	9546.26
2012-04-10	9583.36	9648.12	9509.72	9538.02
2012-04-11	9398.60	9487.84	9388.14	9458.74
2012-04-12	9486.19	9541.04	9449.49	9524.79
2012-04-13	9613.19	9690.29	9603.97	9637.99
2012-04-16	9522.70	9532.91	9464.49	9470.64
2012-04-17	9476.15	9519.80	9455.32	9464.71
2012-04-18	9594.30	9682.88	9594.30	9667.26
2012-04-19	9578.68	9617.93	9565.83	9588.38
2012-04-20	9574.99	9580.87	9532.68	9561.36
2012-04-23	9599.20	9643.27	9520.98	9542.17
2012-04-24	9466.08	9510.70	9423.30	9468.04
cut4
2009-01-05	48.49	48.66	48.12	48.19
2009-01-06	48.25	48.87	48.11	48.78
2009-01-07	48.78	48.84	47.06	47.24
2009-01-08	47.15	47.18	46.52	46.60
2009-01-09	46.67	46.86	46.52	46.82
2009-01-13	46.62	46.70	46.23	46.40
2009-01-14	46.45	46.57	45.82	45.82
2009-01-15	45.67	45.67	45.34	45.48
2009-01-16	45.53	45.65	45.46	45.52
2009-01-19	45.65	45.69	45.38	45.52
2009-01-20	45.48	45.53	44.62	44.87
2009-01-21	44.77	44.77	44.55	44.62
2009-01-22	44.61	44.87	44.40	44.87
2009-01-23	44.83	44.93	44.61	44.80
2009-01-26	44.79	44.88	44.60	44.66
2009-01-27	44.75	44.90	44.69	44.88
2009-01-28	44.94	44.94	44.07	44.43
2009-01-29	44.47	44.55	43.71	43.88
2009-01-30	43.87	44.04	43.53	43.95
2009-02-02	43.79	43.83	43.35	43.54
2009-02-03	43.57	43.76	43.40	43.66
2009-02-04	43.54	43.55	43.28	43.48
2009-02-05	43.43	43.55	43.01	43.08
2009-02-06	43.07	43.10	42.52	42.64
2009-02-09	42.66	42.70	42.15	42.21
2009-02-10	42.17	42.19	41.77	42.02
2009-02-12	41.81	41.81	41.26	41.28
2009-02-13	41.35	42.29	41.35	42.27
2009-02-16	42.37	42.57	42.13	42.40
2009-02-17	42.24	42.24	41.69	41.71
2009-02-18	41.60	42.02	41.57	42.01
2009-02-19	41.96	42.12	41.72	41.98
2009-02-20	41.84	41.84	40.49	40.62
2009-02-23	40.39	40.39	39.69	39.84
2009-02-24	39.53	39.72	39.07	39.46
2009-02-25	39.70	40.39	39.70	40.38
2009-02-26	40.42	40.83	40.42	40.82
2009-02-27	40.71	41.79	40.55	41.72
2009-03-02	41.61	41.61	41.20	41.31
2009-03-03	41.12	41.12	39.78	39.91
2009-03-04	39.75	39.81	39.61	39.78
2009-03-05	39.86	40.14	39.77	39.80
2009-03-06	39.62	39.63	39.47	39.51
2009-03-09	39.47	39.54	38.31	38.31
2009-03-10	38.27	38.54	38.27	38.45
2009-03-11	38.67	38.86	38.46	38.49
2009-03-12	38.44	38.44	37.59	37.66
2009-03-13	37.93	38.34	37.93	38.22
2009-03-16	38.34	38.93	38.32	38.75
2009-03-17	38.65	39.04	38.44	39.04
2009-03-18	39.32	39.88	38.63	38.79
2009-03-19	38.88	39.43	38.87	39.43
2009-03-23	39.47	40.47	39.47	40.40
2009-03-24	40.60	40.84	40.50	40.84
2009-03-25	40.82	41.12	40.73	41.12
2009-03-26	40.74	40.91	40.53	40.88
2009-03-27	40.98	41.45	40.98	41.33
2009-03-30	41.38	41.56	41.10	41.14
2009-03-31	41.03	41.15	40.60	41.04
2009-04-01	41.24	41.39	40.59	40.72
2009-04-02	40.79	41.11	40.71	41.08
2009-04-03	41.32	41.33	40.93	41.01
2009-04-06	41.05	41.28	40.89	40.98
2009-04-07	40.96	41.31	40.88	41.01
2009-04-08	40.94	41.29	40.87	41.29
2009-04-09	41.21	41.75	41.20	41.75
2009-04-10	41.88	41.91	41.63	41.72
2009-04-13	41.75	41.77	41.36	41.38
2009-04-14	41.29	41.32	41.05	41.19
2009-04-15	41.13	41.22	40.99	41.19
2009-04-16	41.24	41.42	41.23	41.35
2009-04-17	41.41	41.54	41.34	41.42
2009-04-20	41.44	41.47	41.12	41.24
2009-04-21	41.00	41.00	40.47	40.64
2009-04-22	40.69	41.01	40.69	40.97
2009-04-23	40.92	41.60	40.69	41.49
2009-04-24	41.52	41.59	41.41	41.53
2009-04-27	41.63	41.88	41.63	41.83
2009-04-28	41.81	41.91	41.43	41.47
2009-04-30	41.61	41.74	40.86	40.86
2009-05-01	40.91	40.92	40.38	40.59
2009-05-07	40.73	40.97	40.60	40.67
2009-05-08	40.74	41.51	40.74	41.51
2009-05-11	41.73	41.88	41.58	41.67
2009-05-12	41.64	41.83	41.61	41.68
2009-05-13	41.73	42.01	41.73	42.01
2009-05-14	41.89	42.03	41.80	41.95
2009-05-15	42.05	42.17	42.04	42.17
2009-05-18	42.10	42.11	41.90	41.97
2009-05-19	42.09	42.21	41.98	42.06
2009-05-20	42.12	42.36	42.11	42.36
2009-05-21	42.33	42.50	42.31	42.47
2009-05-22	42.26	42.47	42.18	42.47
2009-05-25	42.57	42.97	42.57	42.97
2009-05-26	42.98	43.00	42.85	42.94
2009-05-27	43.07	43.53	43.07	43.44
2009-05-28	43.35	43.62	43.35	43.62
2009-05-29	43.75	43.90	43.64	43.79
2009-06-01	43.87	43.94	43.67	43.69
2009-06-02	43.84	43.91	43.61	43.78
2009-06-03	43.80	44.11	43.80	44.05
2009-06-04	43.97	44.12	43.89	44.09
2009-06-05	44.25	44.85	44.24	44.77
2009-06-08	44.83	45.00	44.80	44.93
2009-06-09	44.98	45.07	44.66	44.71
2009-06-10	44.77	45.17	44.77	45.14
2009-06-11	45.19	45.75	45.16	45.75
2009-06-12	45.92	47.13	45.90	47.13
2009-06-15	47.36	48.02	47.35	47.82
2009-06-16	47.72	47.77	47.28	47.29
2009-06-17	47.22	47.29	46.88	47.22
2009-06-18	47.29	47.39	47.13	47.38
2009-06-19	47.43	47.90	47.43	47.90
2009-06-22	47.92	47.93	47.61	47.61
2009-06-23	47.37	47.37	46.98	46.98
2009-06-24	47.02	47.23	46.73	46.99
2009-06-25	47.05	47.42	46.89	47.41
2009-06-26	47.55	47.68	47.44	47.68
2009-06-29	47.84	48.10	47.47	47.47
2009-06-30	47.73	48.58	47.73	48.54
2009-07-01	48.60	48.87	48.60	48.86
2009-07-02	48.94	49.02	48.52	48.71
2009-07-03	48.54	48.85	48.46	48.84
2009-07-06	48.94	49.48	48.93	49.48
2009-07-07	49.61	49.81	49.23	49.30
2009-07-08	49.20	49.20	48.60	48.92
2009-07-09	48.71	48.71	48.05	48.09
2009-07-10	48.09	48.19	47.66	47.78
2009-07-13	47.56	47.64	46.14	46.22
2009-07-14	46.40	46.80	46.40	46.80
2009-07-15	46.75	46.94	46.67	46.94
2009-07-16	47.13	47.35	47.03	47.03
2009-07-17	47.17	48.13	47.06	48.11
2009-07-21	48.31	48.62	48.31	48.62
2009-07-22	48.55	48.85	48.53	48.71
2009-07-23	48.75	48.99	48.74	48.88
2009-07-24	49.04	49.28	48.73	49.28
2009-07-27	49.48	49.66	49.44	49.51
2009-07-28	49.57	49.63	49.20	49.60
2009-07-29	49.50	49.89	49.48	49.89
2009-07-30	49.98	49.98	49.46	49.79
2009-07-31	49.90	50.09	49.35	49.44
2009-08-03	49.66	49.82	49.60	49.78
2009-08-04	50.00	50.10	49.79	49.87
2009-08-05	49.89	49.91	49.64	49.68
2009-08-06	49.64	49.81	49.59	49.80
2009-08-07	49.82	49.82	49.52	49.70
2009-08-10	50.01	50.12	49.99	50.08
2009-08-11	50.08	50.34	49.96	50.34
2009-08-12	50.27	50.40	49.97	50.39
2009-08-13	50.48	50.65	50.28	50.37
2009-08-14	50.34	50.47	50.30	50.47
2009-08-17	50.43	50.43	49.93	50.01
2009-08-18	49.85	49.92	49.69	49.87
2009-08-19	49.95	50.11	49.47	49.53
2009-08-20	49.60	49.62	49.48	49.62
2009-08-21	49.66	49.72	49.31	49.39
2009-08-24	49.53	49.55	49.40	49.55
2009-08-25	49.65	49.80	49.63	49.73
2009-08-26	49.72	49.75	49.67	49.75
2009-08-27	49.74	49.79	49.48	49.79
2009-08-28	49.82	50.15	49.80	50.00
2009-08-31	49.99	50.49	49.83	50.49
2009-09-01	50.40	50.92	50.35	50.92
2009-09-02	50.78	50.82	50.65	50.82
2009-09-03	50.77	50.84	50.61	50.69
2009-09-04	50.72	50.77	49.80	49.80
2009-09-07	49.90	50.08	49.68	49.83
2009-09-08	49.77	49.77	49.42	49.56
2009-09-09	49.56	49.63	49.49	49.63
2009-09-10	49.61	50.26	49.61	50.26
2009-09-11	50.33	50.35	50.13	50.32
2009-09-14	50.30	50.30	50.15	50.29
2009-09-15	50.26	50.38	50.10	50.38
2009-09-16	50.29	50.34	50.05	50.24
2009-09-17	50.33	50.33	49.98	50.11
2009-09-18	50.04	50.05	49.65	49.93
2009-09-24	49.99	50.16	49.89	49.96
2009-09-25	49.78	49.80	49.52	49.74
2009-09-28	49.57	49.57	49.24	49.42
2009-09-29	49.47	49.55	49.35	49.46
2009-09-30	49.49	49.72	49.45	49.72
2009-10-01	49.72	49.84	49.46	49.50
2009-10-02	49.06	49.06	48.50	48.61
2009-10-05	48.60	48.60	48.15	48.34
2009-10-06	48.42	48.47	48.08	48.28
2009-10-07	48.33	48.56	48.12	48.56
2009-10-08	48.48	48.48	48.25	48.35
2009-10-09	48.34	49.30	48.34	49.28
2009-10-13	49.36	49.41	49.25	49.29
2009-10-14	49.27	49.27	48.90	48.95
2009-10-15	49.09	49.13	48.86	48.90
2009-10-16	48.87	48.87	48.68	48.79
2009-10-19	48.71	48.97	48.65	48.90
2009-10-20	48.96	49.05	48.78	48.93
2009-10-21	48.96	49.29	48.89	49.25
2009-10-22	49.20	49.32	49.13	49.30
2009-10-23	49.35	49.40	48.98	49.01
2009-10-26	49.03	49.11	49.02	49.07
2009-10-27	49.00	49.08	48.88	48.91
2009-10-28	48.98	48.98	48.17	48.27
2009-10-29	48.05	48.11	47.79	48.10
2009-10-30	48.25	48.59	48.25	48.59
2009-11-02	48.39	48.40	48.06	48.38
2009-11-04	48.43	48.52	48.23	48.33
2009-11-05	48.30	48.30	47.81	47.94
2009-11-06	48.02	48.05	47.60	47.81
2009-11-09	47.71	47.71	47.21	47.31
2009-11-10	47.44	47.66	47.34	47.34
2009-11-11	47.39	47.49	47.11	47.32
2009-11-12	47.35	47.40	47.08	47.25
2009-11-13	47.14	47.20	46.96	47.14
2009-11-16	46.94	47.02	46.05	46.12
2009-11-17	46.12	46.15	45.20	45.24
2009-11-18	45.27	45.34	44.63	44.79
2009-11-19	44.80	44.92	44.60	44.80
2009-11-20	44.67	45.35	44.57	45.35
2009-11-24	45.37	45.46	45.08	45.10
2009-11-25	45.18	45.23	45.01	45.16
2009-11-26	45.10	45.43	45.06	45.33
2009-11-27	45.14	45.14	44.79	45.06
2009-11-30	45.16	45.73	45.16	45.73
2009-12-01	45.70	45.85	45.60	45.85
2009-12-02	45.96	46.08	45.73	45.78
2009-12-03	45.84	46.18	45.83	46.14
2009-12-04	46.20	46.43	46.00	46.39
2009-12-07	46.59	46.76	46.56	46.70
2009-12-08	46.64	47.36	46.64	47.33
2009-12-09	47.21	47.60	47.19	47.50
2009-12-10	47.52	47.67	47.33	47.49
2009-12-11	47.51	47.55	47.20	47.52
2009-12-14	47.48	47.52	47.07	47.08
2009-12-15	47.15	47.37	47.10	47.37
2009-12-16	47.33	47.38	47.23	47.32
2009-12-17	47.39	47.41	47.04	47.17
2009-12-18	47.18	47.37	46.99	47.35
2009-12-21	47.27	47.50	47.26	47.44
2009-12-22	47.55	47.76	47.49	47.69
2009-12-24	47.69	48.22	47.60	48.18
2009-12-25	48.19	48.32	48.01	48.11
2009-12-28	48.01	48.20	47.97	48.14
2009-12-29	48.23	48.51	48.22	48.51
2009-12-30	48.49	48.52	48.26	48.36
2010-01-04	48.54	48.60	48.33	48.47
2010-01-05	48.53	48.76	48.50	48.54
2010-01-06	48.57	48.67	48.49	48.67
2010-01-07	48.64	48.90	48.54	48.59
2010-01-08	48.62	48.82	48.41	48.58
2010-01-12	48.63	48.94	48.63	48.88
2010-01-13	48.83	49.01	48.83	48.90
2010-01-14	48.96	48.97	48.79	48.82
2010-01-15	48.87	49.40	48.83	49.38
2010-01-18	49.29	49.37	49.07	49.37
2010-01-19	49.28	49.31	49.17	49.18
2010-01-20	49.29	49.61	49.20	49.61
2010-01-21	49.50	50.18	49.49	50.18
2010-01-22	49.99	49.99	49.79	49.93
2010-01-25	49.84	50.92	49.84	50.89
2010-01-26	50.87	50.92	50.12	50.14
2010-01-27	50.18	50.38	50.01	50.19
2010-01-28	50.34	51.14	50.34	51.14
2010-01-29	51.29	51.60	51.29	51.52
2010-02-01	51.32	51.35	51.11	51.17
2010-02-02	51.38	51.59	51.08	51.54
2010-02-03	51.65	51.65	51.35	51.45
2010-02-04	51.39	51.45	50.82	50.92
2010-02-05	50.56	50.56	50.12	50.22
2010-02-08	50.15	50.15	49.40	49.42
2010-02-09	49.34	49.66	49.26	49.56
2010-02-10	49.69	49.85	49.69	49.77
2010-02-12	49.91	50.00	49.79	50.00
2010-02-15	49.91	49.96	49.46	49.50
2010-02-16	49.61	50.15	49.59	50.08
2010-02-17	50.15	50.29	50.10	50.19
2010-02-18	50.22	50.27	50.14	50.21
2010-02-19	50.15	50.18	49.93	49.98
2010-02-22	50.01	50.13	49.97	50.08
2010-02-23	50.09	50.26	50.06	50.26
2010-02-24	50.20	50.27	50.13	50.27
2010-02-25	50.38	50.46	50.30	50.36
2010-02-26	50.34	50.50	50.33	50.47
2010-03-01	50.50	51.16	50.48	51.06
2010-03-02	50.85	51.14	50.67	50.81
2010-03-03	50.81	50.86	50.70	50.79
2010-03-04	50.86	50.94	50.84	50.84
2010-03-05	50.95	51.66	50.69	51.49
2010-03-08	51.59	52.01	51.51	51.63
2010-03-09	51.86	52.31	51.84	52.30
2010-03-10	52.03	52.28	51.87	51.92
2010-03-11	51.84	51.88	51.71	51.80
2010-03-12	52.08	52.32	52.01	52.26
2010-03-15	52.29	52.31	52.15	52.26
2010-03-16	52.23	52.38	52.23	52.34
2010-03-17	52.46	52.62	52.44	52.62
2010-03-18	52.66	52.71	52.58	52.64
2010-03-19	52.67	52.88	52.65	52.88
2010-03-23	53.00	53.35	52.99	53.35
2010-03-24	53.46	53.49	53.35	53.48
2010-03-25	53.48	53.59	53.08	53.58
2010-03-26	53.41	53.43	52.96	53.01
2010-03-29	52.60	52.92	52.58	52.88
2010-03-30	52.99	53.47	52.94	53.47
2010-03-31	53.31	53.56	53.13	53.20
2010-04-01	53.25	53.61	53.19	53.52
2010-04-02	53.64	53.82	53.61	53.71
2010-04-05	53.79	53.97	53.79	53.95
2010-04-06	53.99	54.56	53.68	54.48
2010-04-07	54.60	54.64	54.24	54.38
2010-04-08	54.32	54.72	54.31	54.72
2010-04-09	54.74	54.90	54.73	54.90
2010-04-12	55.03	55.09	54.97	55.07
2010-04-13	55.08	55.09	54.44	54.65
2010-04-14	54.58	54.58	54.23	54.30
2010-04-15	54.31	54.55	54.03	54.55
2010-04-16	54.58	54.72	54.51	54.66
2010-04-19	54.12	54.31	53.79	54.28
2010-04-20	54.39	54.48	54.10	54.11
2010-04-21	54.43	54.61	54.27	54.42
2010-04-22	54.49	54.86	54.40	54.85
2010-04-23	54.95	55.71	54.94	55.71
2010-04-26	55.92	56.47	55.92	56.47
2010-04-27	56.52	57.08	56.37	57.06
2010-04-28	56.43	56.43	56.02	56.40
2010-04-30	56.60	56.62	56.04	56.15
2010-05-06	55.62	55.70	55.18	55.29
2010-05-07	54.48	54.48	53.47	53.99
2010-05-10	53.83	54.31	53.82	53.97
2010-05-11	54.28	55.03	54.09	54.09
2010-05-12	54.15	54.34	53.66	53.82
2010-05-13	54.06	54.23	53.97	54.23
2010-05-14	54.03	54.03	53.88	54.02
2010-05-17	53.61	53.61	52.92	53.05
2010-05-18	53.05	53.10	52.41	52.61
2010-05-19	52.14	52.14	51.66	52.05
2010-05-20	51.86	52.01	51.42	51.70
2010-05-21	50.85	50.85	49.80	50.04
2010-05-24	50.14	50.29	49.86	50.11
2010-05-25	50.04	50.04	49.12	49.23
2010-05-26	49.29	49.36	49.01	49.33
2010-05-27	48.93	49.90	48.86	49.90
2010-05-28	50.40	50.62	50.08	50.26
2010-05-31	50.24	50.75	50.23	50.68
2010-06-01	50.93	51.54	50.93	51.48
2010-06-02	51.02	51.06	50.57	50.60
2010-06-03	50.84	50.95	50.79	50.95
2010-06-04	51.20	51.61	51.20	51.42
2010-06-07	51.00	51.00	50.69	50.73
2010-06-08	50.38	51.06	50.32	51.06
2010-06-09	51.07	51.13	50.79	50.91
2010-06-10	50.94	51.21	50.87	51.17
2010-06-11	51.61	51.77	51.33	51.65
2010-06-14	51.69	51.85	51.69	51.85
2010-06-15	51.87	51.88	51.66	51.66
2010-06-16	51.82	51.87	51.72	51.83
2010-06-17	51.76	51.88	51.65	51.79
2010-06-18	51.87	51.87	51.52	51.74
2010-06-21	51.94	52.49	51.93	52.21
2010-06-22	52.15	52.54	52.14	52.46
2010-06-23	52.27	52.27	52.17	52.22
2010-06-24	52.20	52.23	52.13	52.19
2010-06-25	52.08	52.09	51.77	51.97
2010-06-28	51.95	51.96	51.43	51.46
2010-06-29	51.48	51.52	50.95	51.00
2010-06-30	50.63	51.01	50.43	51.01
2010-07-01	50.88	50.90	50.49	50.66
2010-07-02	50.71	50.95	50.66	50.95
2010-07-05	50.98	51.36	50.97	51.16
2010-07-06	51.02	51.67	50.91	51.67
2010-07-07	51.71	51.71	51.42	51.50
2010-07-08	51.70	51.82	51.66	51.75
2010-07-09	51.83	51.96	51.76	51.96
2010-07-12	51.72	51.92	51.44	51.50
2010-07-13	51.54	51.57	51.34	51.42
2010-07-14	51.49	51.60	51.40	51.47
2010-07-15	51.42	51.44	50.72	50.72
2010-07-16	50.71	50.80	50.29	50.40
2010-07-20	50.32	50.32	50.20	50.30
2010-07-21	50.30	50.34	50.07	50.14
2010-07-22	50.01	50.02	49.85	49.99
2010-07-23	50.15	50.30	49.95	50.02
2010-07-26	50.08	50.26	50.08	50.17
2010-07-27	50.42	51.11	50.40	50.93
2010-07-28	50.96	51.08	50.95	51.07
2010-07-29	51.02	51.15	50.80	51.14
2010-07-30	51.13	51.14	50.85	50.96
2010-08-02	50.88	50.98	50.11	50.13
2010-08-03	50.20	50.27	50.02	50.13
2010-08-04	50.06	50.10	49.88	49.96
2010-08-05	50.03	50.17	49.36	49.45
2010-08-06	49.25	49.25	49.01	49.14
2010-08-09	49.03	49.07	48.95	49.07
2010-08-10	49.13	49.14	48.94	49.02
2010-08-11	48.98	48.99	48.74	48.80
2010-08-12	48.23	48.38	47.89	48.11
2010-08-13	48.10	48.44	48.07	48.44
2010-08-16	48.32	48.45	48.24	48.45
2010-08-17	48.13	48.41	48.12	48.38
2010-08-18	48.49	48.51	48.18	48.25
2010-08-19	48.28	48.44	48.24	48.44
2010-08-20	48.35	48.43	48.28	48.29
2010-08-23	48.29	48.32	47.92	48.03
2010-08-24	47.95	48.04	47.78	47.99
2010-08-25	47.83	47.90	47.61	47.71
2010-08-26	47.78	48.35	47.76	48.22
2010-08-27	48.12	48.30	48.09	48.26
2010-08-30	48.41	48.69	48.41	48.50
2010-08-31	48.41	48.41	48.20	48.32
2010-09-01	48.19	48.19	48.04	48.06
2010-09-02	48.14	48.15	47.77	47.98
2010-09-03	47.88	47.96	47.81	47.88
2010-09-06	47.98	48.21	47.89	48.18
2010-09-07	48.13	48.15	48.01	48.10
2010-09-08	47.95	47.95	47.85	47.90
2010-09-09	48.02	48.19	47.72	47.91
2010-09-10	47.92	48.24	47.91	48.19
2010-09-13	48.24	48.42	48.24	48.35
2010-09-14	48.41	48.54	48.38	48.42
2010-09-15	48.35	48.46	48.28	48.37
2010-09-16	48.42	48.46	48.17	48.22
2010-09-17	48.20	48.27	48.16	48.25
2010-09-21	48.39	48.43	48.34	48.40
2010-09-22	48.28	48.45	48.17	48.19
2010-09-24	48.06	48.34	48.05	48.28
2010-09-27	48.35	48.36	47.94	48.21
2010-09-28	48.13	48.20	48.06	48.18
2010-09-29	48.22	48.39	48.20	48.35
2010-09-30	48.23	48.27	47.87	47.87
2010-10-01	47.93	47.93	47.81	47.92
2010-10-04	47.91	47.91	47.62	47.62
2010-10-05	47.56	47.59	47.41	47.49
2010-10-06	47.55	47.67	47.51	47.67
2010-10-07	47.55	47.80	47.55	47.78
2010-10-08	47.76	47.86	47.75	47.81
2010-10-12	48.08	48.11	47.77	47.82
2010-10-13	47.79	47.84	47.30	47.32
2010-10-14	47.49	47.88	47.49	47.70
2010-10-15	47.80	47.99	47.53	47.58
2010-10-18	47.56	47.60	47.51	47.60
2010-10-19	47.51	47.52	47.39	47.46
2010-10-20	47.36	47.36	46.90	46.95
2010-10-21	46.95	46.95	46.66	46.72
2010-10-22	46.65	46.67	46.54	46.64
2010-10-25	46.67	47.09	46.62	46.66
2010-10-26	46.66	46.87	46.64	46.83
2010-10-27	46.89	46.95	46.83	46.91
2010-10-28	46.91	46.92	46.78	46.82
2010-10-29	46.85	46.92	46.73	46.92
2010-11-01	46.87	46.87	46.68	46.75
2010-11-02	46.72	46.74	46.50	46.52
2010-11-04	46.60	46.63	46.35	46.43
2010-11-05	46.46	46.67	46.46	46.67
2010-11-08	46.75	47.02	46.75	47.01
2010-11-09	47.12	47.34	47.12	47.34
2010-11-10	47.27	47.46	47.27	47.37
2010-11-11	47.43	47.51	47.41	47.51
2010-11-12	47.60	47.60	47.40	47.58
2010-11-15	47.61	47.68	47.52	47.68
2010-11-16	47.66	47.66	47.49	47.55
2010-11-17	47.39	47.61	47.37	47.61
2010-11-18	47.67	47.96	47.67	47.96
2010-11-19	48.12	48.27	48.12	48.19
2010-11-22	48.36	48.63	48.35	48.60
2010-11-24	48.36	48.90	48.29	48.89
2010-11-25	49.06	49.31	48.99	49.04
2010-11-26	49.23	49.28	48.85	48.92
2010-11-29	49.15	49.24	48.99	49.06
2010-11-30	49.10	49.16	48.95	49.08
2010-12-01	49.06	49.41	49.05	49.41
2010-12-02	49.74	50.04	49.74	49.91
2010-12-03	50.28	50.55	50.28	50.55
2010-12-06	50.64	50.85	50.64	50.85
2010-12-07	50.95	50.96	50.76	50.87
2010-12-08	50.92	51.10	50.92	51.03
2010-12-09	51.06	51.15	50.89	51.15
2010-12-10	51.13	51.13	51.02	51.06
2010-12-13	51.18	51.49	51.16	51.49
2010-12-14	51.50	51.59	51.29	51.46
2010-12-15	51.59	51.70	51.59	51.67
2010-12-16	51.72	52.10	51.72	51.77
2010-12-17	52.03	52.37	52.03	52.37
2010-12-20	52.47	52.57	52.30	52.36
2010-12-21	52.36	52.42	52.21	52.23
2010-12-22	52.30	52.35	52.00	52.07
2010-12-24	52.09	52.40	52.08	52.40
2010-12-27	52.41	52.51	52.04	52.09
2010-12-28	52.07	52.14	52.00	52.13
2010-12-29	52.13	52.29	52.09	52.29
2010-12-30	52.13	52.30	51.96	52.30
2011-01-04	52.46	52.63	52.46	52.63
2011-01-05	52.57	52.59	52.20	52.20
2011-01-06	52.51	52.68	52.31	52.49
2011-01-07	52.53	52.54	52.40	52.54
2011-01-11	52.57	52.89	52.56	52.89
2011-01-12	53.00	53.12	52.94	53.04
2011-01-13	53.19	53.33	53.13	53.32
2011-01-14	53.31	53.66	53.28	53.33
2011-01-17	53.67	53.70	53.47	53.59
2011-01-18	53.64	53.97	53.62	53.78
2011-01-19	53.89	54.39	53.88	54.39
2011-01-20	54.29	54.30	53.96	54.05
2011-01-21	54.00	54.00	53.07	53.24
2011-01-24	53.26	53.63	53.22	53.63
2011-01-25	53.83	54.01	53.83	54.00
2011-01-26	53.99	54.48	53.92	54.39
2011-01-27	54.44	54.46	54.35	54.46
2011-01-28	54.44	54.44	54.10	54.26
2011-01-31	53.96	54.10	53.85	54.03
2011-02-01	54.14	54.41	54.14	54.41
2011-02-02	54.61	54.91	54.61	54.91
2011-02-03	54.92	54.94	54.23	54.32
2011-02-04	54.31	54.37	54.09	54.18
2011-02-07	54.36	54.61	54.28	54.60
2011-02-08	54.72	54.80	54.62	54.80
2011-02-09	54.83	54.87	54.66	54.68
2011-02-10	54.67	54.92	54.61	54.92
2011-02-14	55.05	55.32	55.05	55.31
2011-02-15	55.12	55.14	54.96	55.14
2011-02-16	55.02	55.08	54.90	55.07
2011-02-17	55.16	55.23	55.09	55.14
2011-02-18	55.17	55.27	55.12	55.27
2011-02-21	55.35	55.54	55.32	55.51
2011-02-22	55.45	55.46	54.96	55.12
2011-02-23	54.75	55.04	54.68	54.85
2011-02-24	54.61	54.61	53.97	54.12
2011-02-25	54.02	54.23	53.96	54.23
2011-02-28	54.47	54.96	54.40	54.96
2011-03-01	55.11	55.38	55.11	55.38
2011-03-02	55.16	55.89	55.09	55.57
2011-03-03	55.74	55.88	55.73	55.83
2011-03-04	56.11	56.21	55.94	56.14
2011-03-07	56.13	56.16	55.82	55.95
2011-03-08	56.09	56.15	55.99	56.04
2011-03-09	56.36	56.42	55.85	55.95
2011-03-10	55.96	55.98	55.31	55.47
2011-03-11	55.02	55.33	54.98	55.15
2011-03-14	54.02	54.02	49.08	49.68
2011-03-15	49.20	49.20	44.28	45.46
2011-03-16	45.50	47.65	45.50	47.42
2011-03-17	46.85	48.07	46.11	47.89
2011-03-18	48.28	49.69	48.28	49.69
2011-03-22	50.24	51.32	50.24	51.32
2011-03-23	51.45	51.83	51.30	51.60
2011-03-24	51.72	51.93	51.33	51.78
2011-03-25	52.03	52.10	51.65	51.76
2011-03-28	51.77	51.77	51.24	51.29
2011-03-29	50.71	50.85	50.45	50.82
2011-03-30	50.91	51.15	50.87	51.13
2011-03-31	51.34	51.40	51.16	51.29
2011-04-01	51.33	51.41	51.11	51.11
2011-04-04	51.22	51.28	51.07	51.10
2011-04-05	51.12	51.12	50.53	50.59
2011-04-06	50.50	50.50	50.00	50.32
2011-04-07	50.34	50.43	50.13	50.38
2011-04-08	50.23	50.56	50.19	50.56
2011-04-11	50.57	50.78	50.56	50.78
2011-04-12	50.61	50.65	49.76	49.84
2011-04-13	49.64	50.17	49.63	50.17
2011-04-14	50.03	50.34	50.03	50.34
2011-04-15	50.37	50.51	50.28	50.47
2011-04-18	50.56	50.63	50.51	50.54
2011-04-19	50.22	50.27	50.08	50.24
2011-04-20	50.39	50.57	50.37	50.54
2011-04-21	50.67	50.93	50.56	50.87
2011-04-22	50.81	51.06	50.76	51.04
2011-04-25	51.13	51.56	51.13	51.50
2011-04-26	51.53	51.57	51.34	51.39
2011-04-27	51.44	51.51	51.28	51.40
2011-04-28	51.47	51.89	51.45	51.89
2011-05-02	51.93	52.32	51.93	52.32
2011-05-06	52.13	52.24	52.05	52.24
2011-05-09	52.33	52.41	52.10	52.18
2011-05-10	52.23	52.31	52.17	52.30
2011-05-11	52.40	52.46	52.32	52.38
2011-05-12	52.15	52.30	52.13	52.21
2011-05-13	52.33	52.34	51.97	52.15
2011-05-16	51.93	51.93	51.12	51.13
2011-05-17	50.95	50.95	50.73	50.84
2011-05-18	50.91	51.23	50.81	51.20
2011-05-19	51.40	51.71	51.38	51.45
2011-05-20	51.46	51.82	51.45	51.82
2011-05-23	51.69	51.70	51.23	51.28
2011-05-24	51.09	51.32	51.07	51.23
2011-05-25	51.33	51.40	51.08	51.11
2011-05-26	51.20	51.38	51.15	51.38
2011-05-27	51.31	51.38	51.22	51.36
2011-05-30	51.40	51.56	51.38	51.54
2011-05-31	51.54	51.89	51.52	51.89
2011-06-01	51.87	51.97	51.83	51.94
2011-06-02	51.65	51.68	51.56	51.66
2011-06-03	51.60	51.86	51.48	51.60
2011-06-06	51.56	51.64	51.39	51.50
2011-06-07	51.46	51.89	51.46	51.89
2011-06-08	51.83	51.86	51.50	51.70
2011-06-09	51.60	51.66	51.48	51.58
2011-06-10	51.63	51.74	51.55	51.74
2011-06-13	51.57	51.73	51.55	51.61
2011-06-14	51.69	51.75	51.54	51.59
2011-06-15	51.67	51.67	51.44	51.54
2011-06-16	51.43	51.54	51.31	51.37
2011-06-17	51.50	51.57	51.41	51.52
2011-06-20	51.58	51.60	51.28	51.28
2011-06-21	51.35	51.61	51.35	51.61
2011-06-22	51.67	51.84	51.67	51.75
2011-06-23	51.67	51.70	51.48	51.54
2011-06-24	51.40	51.42	51.26	51.41
2011-06-27	51.48	51.57	51.47	51.48
2011-06-28	51.57	51.65	51.30	51.39
2011-06-29	51.52	51.97	51.49	51.97
2011-06-30	52.00	52.11	51.89	52.11
2011-07-01	52.11	52.11	51.72	51.85
2011-07-04	52.16	52.18	51.97	52.06
2011-07-05	52.10	52.10	52.01	52.07
2011-07-06	52.12	52.37	52.09	52.37
2011-07-07	52.27	52.41	52.24	52.38
2011-07-08	52.49	52.66	52.49	52.50
2011-07-11	52.48	52.61	52.48	52.53
2011-07-12	52.38	52.38	52.14	52.24
2011-07-13	52.10	52.39	52.10	52.35
2011-07-14	52.44	52.53	52.38	52.53
2011-07-15	52.49	52.50	52.39	52.43
2011-07-19	52.41	52.53	52.38	52.53
2011-07-20	52.84	52.91	52.59	52.66
2011-07-21	52.86	53.12	52.68	53.08
2011-07-22	53.15	53.39	53.14	53.35
2011-07-25	53.41	53.42	53.27	53.32
2011-07-26	53.35	53.36	52.57	52.66
2011-07-27	52.57	52.57	52.40	52.46
2011-07-28	52.30	52.36	52.11	52.12
2011-07-29	52.07	52.13	51.84	51.93
2011-08-01	51.94	52.36	51.94	52.27
2011-08-02	52.18	52.18	51.99	52.10
2011-08-03	51.78	51.79	51.61	51.79
2011-08-04	51.84	51.96	51.77	51.77
2011-08-05	51.21	51.30	50.58	51.30
2011-08-08	51.00	51.01	49.75	50.33
2011-08-09	49.65	49.92	48.58	49.92
2011-08-10	50.28	50.63	49.45	49.47
2011-08-11	49.03	49.48	48.84	49.48
2011-08-12	49.70	49.93	49.58	49.73
2011-08-15	49.88	50.15	49.88	50.13
2011-08-16	50.26	50.35	50.15	50.19
2011-08-17	50.21	50.43	50.21	50.42
2011-08-18	50.45	50.59	50.39	50.41
2011-08-19	50.03	50.18	49.18	49.24
2011-08-22	49.22	49.34	48.53	48.58
2011-08-23	48.59	48.62	48.23	48.38
2011-08-24	48.49	48.52	47.98	48.00
2011-08-25	48.21	48.49	48.20	48.47
2011-08-26	48.51	49.00	48.49	48.80
2011-08-29	49.14	49.43	49.00	49.26
2011-08-30	49.48	49.89	49.48	49.53
2011-08-31	49.86	50.44	49.85	50.44
2011-09-01	50.49	50.64	50.34	50.42
2011-09-02	50.33	50.33	50.08	50.13
2011-09-05	49.85	50.14	49.83	50.14
2011-09-06	49.95	49.97	49.16	49.16
2011-09-07	49.33	49.45	49.27	49.42
2011-09-08	49.73	49.96	49.70	49.91
2011-09-09	49.56	49.99	49.55	49.59
2011-09-12	49.69	49.69	49.45	49.45
2011-09-13	49.39	49.39	49.12	49.16
2011-09-14	49.40	49.47	48.96	48.96
2011-09-15	48.99	49.06	48.87	48.87
2011-09-16	49.10	49.36	49.03	49.33
2011-09-20	49.30	49.34	49.12	49.22
2011-09-21	49.04	49.29	49.04	49.24
2011-09-22	48.77	48.87	48.40	48.47
2011-09-26	48.29	48.29	47.00	47.06
2011-09-27	47.66	48.02	47.58	47.93
2011-09-28	48.20	48.26	48.02	48.08
2011-09-29	48.06	48.59	48.03	48.59
2011-09-30	48.75	48.76	48.57	48.69
2011-10-03	48.59	48.59	48.23	48.32
2011-10-04	48.04	48.34	47.83	48.33
2011-10-05	48.46	48.52	48.12	48.21
2011-10-06	48.27	48.71	48.26	48.45
2011-10-07	48.71	48.99	48.60	48.67
2011-10-11	49.02	49.26	48.92	48.97
2011-10-12	49.23	49.26	49.16	49.20
2011-10-13	49.18	49.40	49.16	49.38
2011-10-14	49.19	49.24	49.07	49.11
2011-10-17	48.86	48.89	48.69	48.69
2011-10-18	48.50	48.55	48.35	48.51
2011-10-19	48.48	48.54	48.34	48.44
2011-10-20	48.38	48.39	48.12	48.17
2011-10-21	48.07	48.10	47.90	48.05
2011-10-24	48.14	48.22	48.10	48.22
2011-10-25	48.48	48.55	48.43	48.43
2011-10-26	48.35	48.37	48.24	48.28
2011-10-27	48.20	48.43	48.20	48.39
2011-10-28	48.58	48.62	48.31	48.33
2011-10-31	48.53	48.74	48.50	48.65
2011-11-01	48.61	48.62	48.38	48.51
2011-11-02	48.35	48.35	48.00	48.07
2011-11-04	48.25	48.42	48.24	48.37
2011-11-07	48.48	48.65	48.34	48.41
2011-11-08	48.34	48.43	48.07	48.13
2011-11-09	47.91	47.98	47.76	47.78
2011-11-10	47.86	47.86	47.67	47.67
2011-11-11	47.58	47.62	47.46	47.49
2011-11-14	47.45	47.69	47.45	47.69
2011-11-15	47.56	47.56	47.27	47.27
2011-11-16	47.29	47.31	46.96	47.01
2011-11-17	46.72	46.81	46.67	46.81
2011-11-18	46.64	46.64	46.48	46.55
2011-11-21	46.56	46.58	46.26	46.26
2011-11-22	46.07	46.30	45.92	46.30
2011-11-24	46.02	46.02	45.88	45.96
2011-11-25	45.72	45.85	45.64	45.64
2011-11-28	45.57	45.81	45.57	45.69
2011-11-29	45.93	46.15	45.93	46.13
2011-11-30	46.42	46.80	46.42	46.65
2011-12-01	47.33	47.55	47.33	47.49
2011-12-02	47.61	47.74	47.56	47.74
2011-12-05	47.95	48.22	47.95	48.22
2011-12-06	48.18	48.21	47.77	47.88
2011-12-07	47.74	47.92	47.69	47.92
2011-12-08	48.11	48.21	47.83	47.93
2011-12-09	47.91	47.95	47.85	47.88
2011-12-12	48.12	48.31	48.12	48.21
2011-12-13	48.11	48.28	48.03	48.28
2011-12-14	48.21	48.23	47.81	47.91
2011-12-15	47.73	47.89	47.69	47.69
2011-12-16	47.63	47.71	47.52	47.61
2011-12-19	47.54	47.56	47.29	47.48
2011-12-20	47.38	47.70	47.38	47.70
2011-12-21	47.77	47.90	47.77	47.88
2011-12-22	47.91	48.05	47.80	47.83
2011-12-26	48.11	48.14	47.89	47.93
2011-12-27	47.85	47.92	47.72	47.72
2011-12-28	47.64	47.90	47.63	47.81
2011-12-29	47.77	47.87	47.72	47.86
2011-12-30	48.02	48.12	47.92	48.12
2012-01-04	48.21	48.33	48.21	48.33
2012-01-05	48.54	48.58	48.36	48.39
2012-01-06	48.28	48.29	47.97	48.08
2012-01-10	48.16	48.20	48.02	48.11
2012-01-11	48.23	48.29	48.15	48.25
2012-01-12	48.12	48.12	47.73	47.82
2012-01-13	47.65	47.79	47.57	47.74
2012-01-16	47.51	47.51	47.34	47.37
2012-01-17	47.34	47.34	47.16	47.29
2012-01-18	47.36	47.69	47.30	47.69
2012-01-19	47.77	47.80	47.66	47.67
2012-01-20	48.03	48.16	47.98	48.14
2012-01-23	48.14	48.25	48.13	48.20
2012-01-24	48.37	48.40	48.21	48.30
2012-01-25	48.34	48.61	48.30	48.46
2012-01-26	48.63	48.65	48.53	48.64
2012-01-27	48.32	48.32	47.79	47.81
2012-01-30	47.87	47.91	47.65	47.74
2012-01-31	47.70	47.89	47.68	47.87
2012-02-01	47.93	48.04	47.86	48.02
2012-02-02	48.24	48.57	48.24	48.52
2012-02-03	48.50	48.62	48.50	48.54
2012-02-06	48.88	48.92	48.75	48.88
2012-02-07	48.83	48.91	48.68	48.91
2012-02-08	49.02	49.19	48.99	49.19
2012-02-09	49.17	49.56	49.17	49.52
2012-02-10	49.63	50.01	49.63	50.01
2012-02-13	49.98	50.14	49.92	50.14
2012-02-14	49.98	50.04	49.80	49.94
2012-02-15	50.16	50.28	50.05	50.28
2012-02-16	49.82	50.00	49.76	49.97
2012-02-17	49.90	49.99	49.64	49.74
2012-02-20	49.77	49.77	49.32	49.43
2012-02-21	49.55	49.88	49.55	49.78
2012-02-22	49.90	50.41	49.90	50.41
2012-02-23	50.50	50.75	50.49	50.75
2012-02-24	50.78	51.03	50.78	51.03
2012-02-27	51.10	51.21	51.05	51.17
2012-02-28	50.96	51.03	50.79	51.02
2012-02-29	51.25	51.37	51.20	51.26
2012-03-01	51.25	51.28	50.84	50.86
2012-03-02	50.83	51.17	50.80	50.92
2012-03-05	51.45	51.70	51.44	51.70
2012-03-06	51.63	51.72	51.54	51.61
2012-03-07	51.16	51.22	51.03	51.20
2012-03-08	51.28	51.58	51.28	51.58
2012-03-09	51.78	51.93	51.74	51.93
2012-03-12	52.04	52.40	52.03	52.14
2012-03-13	52.49	52.74	52.48	52.70
2012-03-14	52.97	53.26	52.91	53.26
2012-03-15	53.37	53.46	53.28	53.34
2012-03-16	53.34	53.53	53.30	53.40
2012-03-19	53.37	53.48	53.37	53.45
2012-03-21	53.51	53.74	53.51	53.74
2012-03-22	53.59	53.68	53.57	53.68
2012-03-23	53.52	53.64	53.45	53.59
2012-03-26	53.63	53.70	53.23	53.25
2012-03-27	53.27	53.31	53.17	53.31
2012-03-28	52.88	53.05	52.84	53.05
2012-03-29	53.08	53.38	52.92	53.11
2012-03-30	53.43	53.68	53.39	53.68
2012-04-02	53.59	54.00	53.59	53.92
2012-04-03	53.90	53.91	53.69	53.77
2012-04-04	53.45	53.60	53.12	53.22
2012-04-05	53.02	53.19	52.88	53.17
2012-04-06	53.14	53.20	53.07	53.19
2012-04-09	53.03	53.07	52.99	53.01
2012-04-10	52.93	53.01	52.75	52.75
2012-04-11	52.57	52.63	52.42	52.63
2012-04-12	52.64	52.74	52.63	52.74
2012-04-13	52.91	53.08	52.90	53.08
2012-04-16	53.08	53.27	53.01	53.16
2012-04-17	53.15	53.30	53.15	53.28
2012-04-18	53.49	53.70	53.46	53.70
2012-04-19	53.49	53.55	53.40	53.48
2012-04-20	53.42	53.45	53.35	53.44
2012-04-23	53.37	53.41	53.18	53.25
2012-04-24	53.24	53.33	53.20	53.29
