package Net::IPAM::Util;

use 5.10.0;
use strict;
use warnings;
use utf8;

use Carp ();

use Exporter 'import';
our @EXPORT_OK = qw(incr_n);

=head1 NAME

Net::IPAM::Util - A selection of general utility subroutines for Net::IPAM

=head1 SYNOPSIS

  use Net::IPAM::Util qw(incr_n);

  $n = incr_n("\x0a\x00\x00\x01");                                  # 10.0.0.2
  $n = incr_n( pack( 'n8', 0x2001, 0xdb8, 0, 0, 0, 0, 0, 1, ) );    # 2001:db8::2

=cut

=head1 FUNCTIONS

=head2 incr_n

Increment a packed IPv4 or IPv6 address in network byte order. Returns undef on overflow.

This increment function is needed in L<Net::IPAM::IP> and L<Net::IPAM::Block> for transparent handling
of IPv4 and IPv6 addresses and blocks.

This is the only math operation needed for iterating over blocks, splitting blocks to CIDRs
or aggregate IPs and blocks to CIDRs.

No need for L<Math::BigInt>, this pure perl algorithm works for all uint_n in network byte order,
where n is a multiple of 32: uint_32, uint_64, uint_96, uint_128, ...

=cut

sub incr_n {
  my $n = shift;
  Carp::croak("missing argument") unless defined $n;

	my $l = length($n);

  # work in chunks of 32 bits = 4 bytes
  my $p = $l / 4 - 1;

  # only uint_32, uint_64, uint_96, uint_128, ... allowed
  Carp::croak("wrong bitlen") if $l % 4 != 0 or $p < 0;

  # start at least significant chunk position
  my $chunk = vec( $n, $p, 32 );

  # carry?
  while ( $chunk == 0xffff_ffff ) {

    # OVERFLOW, this chunk is already the most significant chunk!
    return if $p == 0;

    # set this chunk to zero: 0xffff_ffff + 1 = 0x0000_0000 + carry
    vec( $n, $p, 32 ) = 0;

    # carry on to next more significant chunk position
    $chunk = vec( $n, --$p, 32 );
  }

  # incr this chunk
  vec( $n, $p, 32 ) = ++$chunk;
  return $n;
}

=head1 AUTHOR

Karl Gaissmaier, C<< <karl.gaissmaier(at)uni-ulm.de> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-net-ipam-util at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Net-IPAM-Util>.
I will be notified, and then you'll automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Net::IPAM::Util

You can also look for information at:

=over 4

=item * on github

TODO

=back

=head1 SEE ALSO

L<Net::IPAM::IP>
L<Net::IPAM::Block>
L<Net::IPAM::Tree>

=head1 LICENSE AND COPYRIGHT

This software is copyright (c) 2020 by Karl Gaissmaier.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

1;    # End of Net::IPAM::Util
