package Net::SSLLabs::Cert;

use strict;
use warnings;
use URI();

sub new {
    my ( $class, $json ) = @_;
    my $self = $json;
    bless $self, $class;
    return $self;
}

sub alt_names {
    my ($self) = @_;
    return @{ $self->{altNames} };
}

sub issuer_subject {
    my ($self) = @_;
    return $self->{issuerSubject};
}

sub issues {
    my ($self) = @_;
    return $self->{issues};
}

sub ocsp_revocation_status {
    my ($self) = @_;
    return $self->{ocspRevocationStatus};
}

sub ocsp_uris {
    my ($self) = @_;
    return map { URI->new($_) } @{ $self->{ocspURIs} };
}

sub revocation_info {
    my ($self) = @_;
    return $self->{revocationInfo};
}

sub sgc {
    my ($self) = @_;
    return $self->{sgc} ? 1 : 0;
}

sub validation_type {
    my ($self) = @_;
    return $self->{validationType};
}

sub sct {
    my ($self) = @_;
    return $self->{sct} ? 1 : 0;
}

sub sig_alg {
    my ($self) = @_;
    return $self->{sigAlg};
}

sub common_names {
    my ($self) = @_;
    return map { URI->new($_) } @{ $self->{commonNames} };
}

sub crl_uris {
    my ($self) = @_;
    return map { URI->new($_) } @{ $self->{crlURIs} };
}

sub issuer_label {
    my ($self) = @_;
    return $self->{issuerLabel};
}

sub subject {
    my ($self) = @_;
    return $self->{subject};
}

sub not_before {
    my ($self) = @_;
    return $self->{notBefore};
}

sub revocation_status {
    my ($self) = @_;
    return $self->{revocationStatus};
}

sub not_after {
    my ($self) = @_;
    return $self->{notAfter};
}

sub crl_revocation_status {
    my ($self) = @_;
    return $self->{crlRevocationStatus};
}

1;
__END__

=head1 NAME

Net::SSLLabs::Cert - Cert object

=head1 METHODS

=head2 new

a new C<Net::SSLLabs::Cert> object, accepts a JSON object as it's parameter.

=head2 subject

certificate subject

=head2 common_names

list of common names extracted from the subject

=head2 alt_names

alternative names

=head2 not_before

UNIX timestamp before which the certificate is not valid

=head2 not_after

UNIX timestamp after which the certificate is not valid

=head2 issuer_subject

issuer subject

=head2 sig_alg

certificate signature algorithm

=head2 issuer_label

issuer name

=head2 revocation_info

a number that represents revocation information present in the certificate:

=over 2

=item bit 0 (1) - CRL information available

=item bit 1 (2) - OCSP information available

=back

=head2 crl_uris

list of CRL L<URI|URI>s extracted from the certificate.

=head2 ocsp_uris

list of OCSP L<URI|URI>s extracted from the certificate

=head2 revocation_status

a number that describes the revocation status of the certificate:

=over 2

=item 0 - not checked

=item 1 - certificate revoked

=item 2 - certificate not revoked

=item 3 - revocation check error

=item 4 - no revocation information

=item 5 - internal error

=back

=head2 crl_revocation_status

same as revocationStatus, but only for the CRL information (if any).

=head2 ocsp_revocation_status

same as revocationStatus, but only for the OCSP information (if any).

=head2 sgc

Server Gated Cryptography support; integer:

=over 2

=item bit 1 (1) - Netscape SGC

=item bit 2 (2) - Microsoft SGC

=back

=head2 validation_type

E for Extended Validation certificates; may be null if unable to determine

=head2 issues

list of certificate issues, one bit per issue:

=over 2

=item bit 0 (1) - no chain of trust

=item bit 1 (2) - not before

=item bit 2 (4) - not after

=item bit 3 (8) - hostname mismatch

=item bit 4 (16) - revoked

=item bit 5 (32) - bad common name

=item bit 6 (64) - self-signed

=item bit 7 (128) - blacklisted

=item bit 8 (256) - insecure signature

=back

=head2 sct

true if the certificate contains an embedded SCT; false otherwise.

