#!/usr/bin/perl

# $Id: session_cleanup.pl,v 1.1 2003/02/21 03:21:50 lachoy Exp $

# session_cleanup.pl

use strict;
use OpenInteract::Startup;
use Storable qw( thaw );
use MIME::Base64;

my $TEMP_TRACK_FILE = 'tmp_session_id';

{
    $| = 1;
    my $usage = "Usage: $0 --website_dir=/path/to/my_site days-old";

    my ( $OPT_analyze, $OPT_debug, $OPT_help, $OPT_decode );
    my $R = OpenInteract::Startup->setup_static_environment_options(
                    $usage,
                    { 'analyze'  => \$OPT_analyze,
                      'debug'    => \$OPT_debug,
                      'decode=s' => \$OPT_decode,
                      'help'     => \$OPT_help },
                    { temp_lib => 'lazy' } );
    if ( $OPT_help ) { 
        print usage(); exit();
    }

    if ( $OPT_analyze ) {
        print "Analysis mode on; no changes will be made\n";
    }

    $OPT_decode ||= 'base64';

    my ( $days_old ) = @ARGV;
    $days_old ||= 60;
    my ( $removed, $kept, $count );

    my $initial_time = time;

    my $db = $R->db;

    # Grab all the session_ids and print them to a file, one per line,
    # so we only have to keep one handle open

    my ( $id );
    my $sql = qq/ SELECT id FROM sessions /;
    my $sth = $db->prepare( $sql );
    $sth->execute;

    open( IDLIST, "+> $TEMP_TRACK_FILE" );
    $sth->bind_col( 1, \$id );
    while ( $sth->fetch ) {
        print IDLIST "$id\n";
        $count++;
    }
    $sth->finish;
    print "Scanning [$count] sessions...\n";

    seek( IDLIST, 0, 0 );

    my $below_thresh = time - ( $days_old * 86400 );
    print "Removing those before ", scalar( localtime( $below_thresh ) ), "\n";

    my $current = 0;
ID:
    while ( <IDLIST> ) {
        chomp;
        my $sth_d = $db->prepare( "SELECT a_session FROM sessions WHERE id = ?" );
        $sth_d->execute( $_ );
        my ( $raw_data ) = $sth_d->fetchrow_array;

        # Pick one of these...

        my $session = decode( $OPT_decode, $raw_data );

        my $timestamp = $session->{timestamp} || 0;
        if ( scalar keys %{ $session } == 0 ) {
            $OPT_analyze || $db->do( "DELETE FROM sessions WHERE id = '$_'" );
            $removed++;
        }
        elsif ( $timestamp < $below_thresh ) {
            $OPT_analyze || $db->do( "DELETE FROM sessions WHERE id = '$_'" );
            $removed++;
        }
        else {
            $kept++;
        }
        print " $current" if ( $current> 0 && $current % 500 == 0 );
        if ( $current > 0 && $current % 2500 == 0 ) {
            print "\n";
            $OPT_debug && print "[Item: $current] ",
                                "[", scalar( localtime( $timestamp ) ), "]",
                                "[Keys: ", join( ', ', keys %{ $session } ), "]\n";
        }
        $current++;
    }
    close( IDLIST );
    print "\n",
          "Results:\n",
          "   Sessions: $count\n",
          "   Removed:  $removed\n",
          "   Kept:     $kept\n",
          "   Time:     ", ( time - $initial_time ), " seconds\n";
    unlink( $TEMP_TRACK_FILE )
}

sub decode {
    my ( $type, $data ) = @_;
    if ( $type eq 'base64' ) {
        return thaw( decode_base64( $_[0] ) );
    }
    elsif ( $type eq 'storable' ) {
        return thaw( $_[0] );
    }
    else {
        die "Cannot decode type [$type] unknown\n";
    }
}

sub usage {
    return <<USAGE;
$0 [ days-previous ] options

Cleanup old/stale sessions from your database. Depends on the
'timestamp' field being set.

Options:

 --help
     Print this message

 --analyze
     Show a report on the number of sessions and those
     that would be removed, but don't actually remove

 --decode=type
     Type of session encoding. Types are:
        base64   - Storable + Base64
        storable - Storable

 --debug
     Display extra debugging along the way

USAGE
}
