package OpenInteract::Handler::User;

# $Id: User.pm,v 1.7 2001/10/31 13:58:24 lachoy Exp $

use strict;
use Data::Dumper  qw( Dumper );
use OpenInteract::Handler::GenericDispatcher qw( DEFAULT_SECURITY_KEY );
use SPOPS::Secure qw( :level :scope );

@OpenInteract::Handler::User::ISA     = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::User::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::User::author            = 'chris@cwinters.com';
$OpenInteract::Handler::User::default_method    = 'search_form';
@OpenInteract::Handler::User::forbidden_methods = ();
%OpenInteract::Handler::User::security          = ();

use constant MAIN_SCRIPT => '/User';

sub search_form {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{title} = 'Search for User Records';
    return $R->template->handler( {}, $p,
                                  { name => 'base_user::user_search_form' } );
}


sub search {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { main_script => MAIN_SCRIPT,
                   status_msg  => $p->{status_msg},
                   error_msg   => $p->{error_msg} };
    my $fetch_params = $class->_search_params;
    $R->DEBUG && $R->scrib( 2, "Searching for users with parameters\n",
                               Dumper( $fetch_params ) );
    $params->{user_iterator} = eval { $R->user->fetch_iterator( $fetch_params) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw({ code => 404 });
        $params->{error_msg} .= "<br>Error running search: $@";
    }
    $R->{page}{title} = 'User Listing';
    return $R->template->handler( {}, $params,
                                  { name => 'base_user::user_search_results' } );
}


sub _search_params {
    my ( $class ) = @_;
    my $R   = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my @search_fields = qw( first_name last_name email login_name );
    my %s = map { $_ => $apr->param( $_ ) } @search_fields;
    my $user_class = $R->user;
    return $class->_search_params_ldap( \%s ) if ( $user_class->isa( 'SPOPS::LDAP' ) );
    return $class->_search_params_dbi( \%s )  if ( $user_class->isa( 'SPOPS::DBI' ) );
    return ();
}


sub _search_params_dbi {
    my ( $class, $search ) = @_;
    my @where = ();
    my @value = ();
    foreach my $field ( keys %{ $search } ) {
        next unless ( $search->{ $field } );
        push @where, " $field LIKE ? ";
        push @value, "%$search->{ $field }%";
    }
    return { order => 'login_name',
             where => join( ' AND ', @where ),
             value => \@value };
}


sub _search_params_ldap {
    my ( $class, $search ) = @_;
    my $R      = OpenInteract::Request->instance;
    my $field_map = $R->user->CONFIG->{field_map} || {};
    my @filter_chunk = ();
    foreach my $field ( keys %{ $search } ) {
        next unless ( $search->{ $field } );
        my $ldap_prop = $field_map->{ $field };
        push @filter_chunk, "($ldap_prop=*$search->{ $field }*)";
    }
    return {} unless ( scalar @filter_chunk );
    return { filter => '(&' . join( '', @filter_chunk ) . ')' };
}


# Display a single user record, either in static or editable format.

sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    # Add this to the hashref below if you want users to be able to choose themes
    # theme_list => eval { $R->theme->fetch_group( { order => 'title' } ) },

    my $params = { main_script => MAIN_SCRIPT,
                   error_msg   => $p->{error_msg},
                   status_msg  => $p->{status_msg} };
    $params->{user} = $p->{user};
    unless ( $params->{user} ) {
        my $user_id = $R->apache->param( 'user_id' ) || $R->apache->param( 'uid' );
        if ( $user_id ) {
            $params->{user}  = eval { $R->user->fetch( $user_id ) };
            if ( $@ ) {
                OpenInteract::Error->set( SPOPS::Error->get );
                $R->throw({ code => 404 });
                $R->scrib( 0, "Cannot retrieve user ($user_id) ($@) $SPOPS::Error::system_msg" );
                return $class->search_form({ error_msg => "Could not retrieve requested record ($@)" });
            }
        }
    }

    # If the user has only READ access to module and there is no UID
    # or user object, there's no reason to continue

    unless ( $params->{user} or $R->{auth}{is_admin} ) {
        my $msg = 'Sorry, you do not have access to create a new user object. Returning to listing.';
        $R->scrib( 0, "No permission to create new user." );
        return $class->search_form({ error_msg => $msg });
    }

    my $obj_level = ( $params->{user} )
                      ? $params->{user}{tmp_security_level} : SEC_LEVEL_WRITE;

    $params->{user} ||= $R->user->new({ theme_id => $R->{theme}{theme_id} });
    $R->DEBUG && $R->scrib( 1, "Security  $params->{user}{user_id}: ($obj_level)" );
    $params->{group_list} = eval { $params->{user}->group };

    my $tmpl_name = ( $obj_level == SEC_LEVEL_WRITE )
                      ? 'user_form' : 'user_detail';
    $R->{page}{title} = 'View User Detail';
    return $R->template->handler( {}, $params,
                                  { name => "base_user::$tmpl_name"  } );
}


sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{return_url} = '/User/';
    my $apr = $R->apache;

    my $user_id = $apr->param( 'user_id' );
    $R->DEBUG && $R->scrib( 1, "Trying to modify ID <<$user_id>>" );

    my $is_new = 0;
    my $user = $p->{user} ||
               eval { $R->user->fetch( $user_id ) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 404 } );
    }

    my $obj_level = ( $user ) ? $user->{tmp_security_level} : SEC_LEVEL_WRITE;
    if ( $obj_level < SEC_LEVEL_WRITE ) {
        my $user_msg = 'Sorry, you do not have access to modify this user object. Returning to listing.';
        return $class->search_form({ error_msg => $user_msg });
    }

    # Note if this user is new so we can do some security fun after
    # the information is saved

    unless ( $user ) {
        $user ||= $R->user->new;
        $is_new++;
    }

    my ( @status );

    # Uncomment the code if you are allowing users to change their own themes
    # Track this so we can change the theme if the user changed it.

    # my $pre_theme_id = $user->{theme_id} || $R->{theme}{theme_id};

    my @field_list = qw( first_name last_name email title login_name notes );
    foreach my $field ( @field_list ) {
        $user->{ $field } = $apr->param( $field );
    }
    my $password = $apr->param( 'password' );
    if ( $password ) {
        my $password_confirm = $apr->param( 'password_conf' );
        unless ( $password eq $password_confirm ) {
            return $class->show({ user      => $user,
                                  error_msg => 'Passwords do not match. Please re-enter.' });
        }
        $R->DEBUG && $R->scrib( 1, "User asked to change password. Changing." );
        $password = $R->user->crypt_it( $password ) if ( $R->CONFIG->{login}{crypt_password} );
        $user->{password} = $password;
        push @status, "Password changed";
    }

    eval { $user->save };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $OpenInteract::Error::user_msg = 'Error saving user object.';
        $R->throw({ code => 407 });
        $R->scrib( 0, "Cannot save user object: $SPOPS::Error::system_msg" );
        return $class->show({ user      => $user,
                              error_msg => "Error saving user record ($@)" });
    }

    # If this is a new user, allow the user to edit his/her own record;
    # other security settings (WORLD, GROUP for site group) should be
    # done in the normal way (via conf/spops.perl->{creation_security};
    # also make the user a member of group 'public'

    if ( $is_new ) {
        my $site_admin_id = $R->CONFIG->{default_objects}{site_admin_group};
        eval {
            $user->set_security({ scope => SEC_SCOPE_USER,
                                  level => SEC_LEVEL_WRITE,
                                  scope_id => $user->id });
            $user->make_public();
        };
        if ( $@ ) {
            $R->scrib( 0, "Error modifying group membership: $@" );
            my $error_msg = "Record created but group membership inconsistent.";
            return $class->show({ user      => $user,
                                  error_msg => $error_msg });
        }
        push @status, "User record made part of 'public' group";
    }


    # Uncomment the code if you are allowing users to change their own themes
    # If the user changed his/her theme, show it immediately

# if ( $user->{user_id} == $R->{auth}{user}{user_id} and
#      $user->{theme_id} != $pre_theme_id ) {
#   $R->{theme} = eval { $R->theme->fetch( $user->{theme_id} ) };
# }


    my $action = ( $is_new ) ? 'created' : 'updated';
    unshift @status, "User record successfully $action";
    return $class->show({ user       => $user,
                          status_msg => join( "<br>\n", @status ) });
}


sub remove {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    if ( my $user_id = $R->apache->param( 'user_id' ) ) {
        my $user = eval { $R->user->fetch( $user_id ) };
        if ( $@ or ! $user ) {
            return $class->search_form({ error_msg => 'Cannot remove user -- object not created properly.' });
        }
        if ( $user->{tmp_security_level} < SEC_LEVEL_WRITE ) {
            my $user_msg = 'Sorry, you do not have access to remove this user object. Returning to listing.';
            return $class->search_form({ error_msg => $user_msg });
        }

        eval { $user->remove };
        if ( $@ ) {
            OpenInteract::Error->set( SPOPS::Error->get );
            $R->throw({ code => 405 });
            $R->scrib( 0, "Failed to remove user object: ($@) $SPOPS::Error::system_msg" );
            my $error_msg = "Error removing user object. ($@)";
            return $class->search_form({ error_msg => $error_msg });
        }
    }
    return $class->search_form({ status_msg => 'User record successfully removed' });
}

1;
