package OpenInteract::Handler::Page;

# $Id: Page.pm,v 1.13 2001/11/27 12:08:36 lachoy Exp $

use strict;
use Class::Date ();
use OpenInteract::CommonHandler qw( OK ERROR );
use OpenInteract::Handler::GenericDispatcher qw( DEFAULT_SECURITY_KEY );
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::Page::ISA     = qw( OpenInteract::CommonHandler  SPOPS::Secure );
$OpenInteract::Handler::Page::VERSION = sprintf("%d.%02d", q$Revision: 1.13 $ =~ /(\d+)\.(\d+)/);
$OpenInteract::Handler::Page::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Page::default_method    = 'show';
@OpenInteract::Handler::Page::forbidden_methods = ();
%OpenInteract::Handler::Page::security          = (
   DEFAULT_SECURITY_KEY() => SEC_LEVEL_WRITE,
   show   => SEC_LEVEL_NONE, notify  => SEC_LEVEL_READ,
);

# 52 weeks -- default expiration for page

use constant DEFAULT_EXPIRE  => '365D';

# Use this to check whether the file retrieved is displayable in the
# browser; others (pdf, ps, mov, etc.) get sent to the user directly

my %DISPLAY_TYPES = map { $_ => 1 } ( 'text/html', 'text/plain', 'text/xml' );

# Use this to separate your single document into multiple pages

my $PAGE_SEPARATOR = '<!--PAGE-->';


sub MY_PACKAGE                 { return 'base_page' }
sub MY_HANDLER_PATH            { return '/Page' }
sub MY_OBJECT_TYPE             { return 'page' }
sub MY_OBJECT_CLASS            { return OpenInteract::Request->instance->page }
sub MY_SEARCH_FIELDS           { return () }
sub MY_EDIT_RETURN_URL         { return '/' }
sub MY_EDIT_DISPLAY_TASK       { return 'show' }
sub MY_EDIT_FIELDS             { return qw( location title author keywords
                                            boxes main_template notes content
                                            storage content_location ) }
sub MY_EDIT_FIELDS_TOGGLED     { return qw( is_active template_parse ) }
sub MY_EDIT_FIELDS_DATE        { return qw( active_on expires_on ) }
sub MY_ALLOW_SEARCH_FORM       { return 1 }
sub MY_ALLOW_SEARCH            { return 1 }
sub MY_ALLOW_CREATE            { return 1 }
sub MY_OBJECT_CREATE_SECURITY  { return SEC_LEVEL_WRITE }
sub MY_ALLOW_SHOW              { return 1 }
sub MY_ALLOW_EDIT              { return 1 }
sub MY_ALLOW_REMOVE            { return 1 }
sub MY_ALLOW_NOTIFY            { return 1 }
sub MY_ALLOW_WIZARD            { return undef }


# Overrides entry in OpenInteract::Handler::GenericDispatcher

sub _get_task {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    return 'show'  if ( $R->{path}{full}->[0] ne 'Page' );
    return lc shift @{ $R->{path}{current} } ||
           $OpenInteract::Handler::Page::default_method;
}



# Retrieve all directories, expanding the one we were asked to (if at
# all). Note that these are just the objects in the database, although
# there should be a corresponding content entry for every one of these
# in the filesystem or database.

sub directory_list {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $selected_dir = $R->apache->param( 'selected_dir' );
    my $params = { selected_dir => $selected_dir,
                   error_msg    => $p->{error_msg},
                   status_msg   => $p->{status_msg} };

    $params->{dir_list} = eval { $R->page->list_directories };
    if ( $@ ) {
        OpenIntereact::Error->set( SPOPS::Error->get );
        $R->throw({ code => 403 });
        $params->{error_msg} .= "\nCannot retrieve directories: $@";
        $params->{dir_list} = [];
    }

    # Store the pages found using the directory as a key pointing to a
    # listref of files it contains

    if ( $selected_dir ) {
        $params->{children_files} = $R->page->fetch_iterator({
                                               where => 'directory = ?',
                                               value => [ $selected_dir ] });
    }

    $R->{page}{title} = 'Document Listing';
    return $R->template->handler( {}, $params,
                                  { name => 'base_page::page_directory_list' } );
}


# Yes, this is out of the normal order. It's just that show() is so
# big and includes so much stuff...

sub _edit_customize {
    my ( $class, $page, $old_data ) = @_;
    my $R = OpenInteract::Request->instance;

    my %opts = ();

    # If the user changed the location, then we need to set the ID so
    # the UPDATE works properly.

    if ( $page->{location} ne $old_data->{location} ) {
        $opts{use_id} = $old_data->{location};
    }

    # Default the expires_on field

    unless ( $page->{expires_on} ) {
        my $expire_date = Class::Date->now + DEFAULT_EXPIRE;
        $page->{expires_on} = $expire_date->strftime( '%Y-%m-%d' );;
    }

    # See if the upload should be there

    my $has_upload = $class->_read_field_toggled( $R->apache, 'use_upload' );
    if ( $has_upload eq 'yes' ) {
        $R->DEBUG && $R->scrib( 1, "User is requesting content from uploaded file..." );
        my $upload  = $R->apache->upload( 'content_upload' );
        unless ( $upload ) {
            my $error_msg = 'You checked off that you wanted to upload a ' .
                            'file but did not upload one. Why are you teasing me?';
            my %error_opts = ( method => 'show', error_msg => $error_msg, edit => 1 );
            return ( ERROR, \%error_opts );
        }
        $R->DEBUG && $R->scrib( 1, "Upload seems to be retrieved ok. Here is some info:\n",
                                   "Filename: (", $upload->filename, ") Size: (",
                                   $upload->size, ") Type: (", $upload->type, ")" );
        $page->{size}      = $upload->size;
        $page->{mime_type} = $upload->type;
        if ( $class->_is_displayable( $page->{mime_type} ) ) {
            my $fh = $upload->fh;
            local $/ = undef;
            my $content = <$fh>;
            $page->{content} = \$content;
        }
        else {
            $page->{content} = $upload->fh;
        }
    }

    $page->{mime_type} ||= 'text/html';

    # Non-displayable docs always get saved to the filesystem (for
    # now); also, remove 'content' from the list of fields to be
    # processed by the FullText indexer

    unless ( $class->_is_displayable( $page->{mime_type} ) ) {
        $page->{is_file} = 'yes';
        if ( $page->CONFIG->{fulltext_field} ) {
            $opts{fulltext_field} = [ grep ! /^content$/, @{ $page->{fulltext_field} } ];
        }
    }

    return ( OK, \%opts );
}



# Why do we set the content-type when returning errors? See note on
# error content-type forcing in POD...

sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { error_msg   => $p->{error_msg},
                   status_msg  => $p->{status_msg} };

    # Get the actual location requested -- this can be passed in
    # directly, specified in a parameter or taken from the URL.

    my $location = $class->_clean_location( $class->_find_location( $p ) );

    # Try and find a page object (just the metadata) matching with our
    # location. Note that _find_page_object() will also treat
    # $location as a directory request, where appropriate

    my $page = $p->{page} || eval { $class->_find_page_object( $location ) };
    if ( $@ ) {
        $R->{page}{content_type} = 'text/html';
        $R->DEBUG && $R->scrib( 1, "Could not retrieve page. Error ($@)" );
        if ( $@ =~ /^security/ ) {
            my $admin_email = $R->CONFIG->{mail}{admin_email};
            $R->throw({ code => 303 });
        }
        elsif ( $@ =~ /^access/ ) {
            return "<h2>Cannot Access</h2><p>Failure accessing page.</p>";
        }
    }

    my $do_edit = $p->{edit} || $R->apache->param( 'edit' );
    unless ( $page or $do_edit ) {
        $R->{page}{content_type} = 'text/html';
        $R->throw({ code => 314, system_msg => $location });
    }

    # Now we have a page; just check to see if we were instructed to
    # display the editable form for this page, and if so ensure this
    # user can do so.

    if ( $do_edit ) {
        $page ||= $R->page->new;

        # If this is an editable doc, fetch the content, otherwise,
        # mark it as non-editable to the form

        $page->{storage} ||= 'file';
        if ( $class->_is_displayable( $page->{mime_type} ) ) {
            $page->fetch_content;
        }
        else {
            $params->{non_editable} = 1;
        }
        $R->DEBUG && $R->scrib( 1, "This page should be in an editable form" );
        $params->{page} = $page;
        $R->{page}{title}   = 'Edit a Document';
        return $R->template->handler( {}, $params, { name => 'base_page::page_form' } );
    }

    # If we specified that we're going to send a separate file to the
    # user (usually not HTML, text, etc.) then set the information and
    # quit processing

    unless ( $class->_is_displayable( $page->{mime_type} ) ) {
        $R->{page}{content_type}   = $page->{mime_type};
        $R->{page}{send_file}      = join( '', $R->CONFIG->get_dir( 'html' ), $page->{location} );
        $R->{page}{send_file_size} = $page->{size};
        $R->DEBUG && $R->scrib( 1, "File being retrieved is not directly displayable.",
                                   "Set 'send_file' to ($page->{location})" );
        return undef;
    }

    # We have a page and we can display it, so deal with the fact that
    # it may be an alias and then grab the relevant content

    while ( $page->{storage} eq 'alias' ) {
        $page = eval { $class->_find_page_object( $page->{content_location} ) };
        if ( $@ or ! $page ) {
            $R->scrib( 0, "Location for alias ($page->{content_location})",
                          "wasn't found. ($@)" );
            return "Request was for an alias, but aliased page unavailable.";
        }
    }

    $page->fetch_content;

    # Otherwise, ensure the page is active, set metadata and send it
    # off to be displayed

    $R->DEBUG && $R->scrib( 1, "Display ($page->{location}) as normal HTML" );
    unless ( $class->_is_active( $page ) ) {
        $R->DEBUG && $R->scrib( 1, "Page is not currently active; return error" );
        $R->{page}{title} = 'Page not active';
        return '<h2 align="center">Not Active</h2><p>Sorry, this page is not active.</p>';
    }

    $R->{page}{title} = $page->{title};

    # Allows the page to define the main template it will use; if
    # the page doesn't define one then the main UI module will use
    # the default for the current theme

    $R->{page}{_template_name_} = $page->{main_template};

    $class->_add_object_boxes( $page, $p );

    # You can split your page into multiple viewable pages -- see
    # _split_pages() for more info

    my $display_content = $class->_split_pages( $page );

    return $display_content if ( $page->{template_parse} eq 'no' );
    return $R->template->handler( {}, $params, { text => \$display_content } );
}


# True means page is displayable in browser, false means it's not. We
# treat an empty mime_type as an HTML page. (Might change)

sub _is_displayable {
    my ( $class, $mime_type) = @_;
    return 1 unless ( $mime_type );
    return 1 if ( $DISPLAY_TYPES{ $mime_type } );
    return undef;
}


# Find the location from whatever is available -- passed parameters,
# GET/POST parameters, or the original path

sub _find_location {
    my ( $class, $p ) = @_;
    return $p->{page}{location} if ( $p->{page} );
    my $R = OpenInteract::Request->instance;
    return $p->{location} ||
           $R->apache->param( 'location' ) ||
           $R->{path}{original};
}


# Security -- remove all '.' from the beginning of the location
# requested so people don't try to go up the directory tree. Also
# remove any two-dot sequence.
#
# In the future we might flag these as bad requests (die from here)
# and simply bail with a stern scoling.

sub _clean_location {
    my ( $class, $location ) = @_;
    return undef unless ( $location );
    $location =~ s/^\.+//;
    $location =~ s/\.\./_/;
    return $location;
}


# Find object with $location in the database. We also try to do the
# work so that you can request a directory index ('home',
# 'index.html', etc.);

sub _find_page_object {
    my ( $class, $location ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Trying to retrieve object with location ($location)" );

    # Chop off any query strings and put in a case-consistent format

    $location    = $class->_remove_query_string( $location );
    $location    = lc $location;

    # Just like 'DirectoryIndex' in Apache...

    my $index_names = $R->CONFIG->{action}{page}{directory_index} || [];
    $R->DEBUG && $R->scrib( 1, "Using the following for index names: ",
                               join( ', ', @{ $index_names } ) );

    my @locations = ( $location );
    if ( $location =~ m|/$| ) {
        $R->DEBUG && $R->scrib( 1, "Explicit directory request; add indexes" );
        push @locations, map { "$location$_" } @{ $index_names };
    }
    elsif ( $location !~ /\.\w+$/ ) {
        $R->DEBUG && $R->scrib( 1, "Location has no extension; add indexes " ),
        push @locations, map { "$location/$_" } @{ $index_names };
    }
    else {
        my ( $sans_extension );
        ( $sans_extension = $location ) =~ s/\.\w+$//;
        $R->DEBUG && $R->scrib( 1, "Also check location without the extension ",
                                  "using $sans_extension" );
        push @locations, $sans_extension;
    }

    my ( $page );
    my ( $error_type );
    foreach my $location ( @locations ) {
        $page = eval { $R->page->fetch( $location ) };
        if ( $@ ) {
            $R->scrib( 0, "Encountered error trying to retrieve ($location);",
                          "continuing with other locations." );
            $error_type = SPOPS::Error->get->{type};
        }
        if ( $page ) {
            $R->DEBUG && $R->scrib( 1, "Found matching location: ($page->{location})" );
            return $page;
        }
    }

    # Returning undef means there were no errors, we just didn't find the page

    return undef unless ( $error_type );
    die $error_type;
}


# Lop off the query string from $text

sub _remove_query_string {
    my ( $class, $text ) = @_;
    $text =~  s|^(.*)\?.*$|$1|;
    return $text;
}


# A page can have one or more tags that declare it wants itself split
# into multiple pieces for display. This routine does the
# splitting. This is still under development...

sub _split_pages {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;

    # Split the page into separate pages -- first check and see if the
    # document IS paged, then do the splitting and other contortions

    if ( $page->{content} =~ /$PAGE_SEPARATOR/ ) {
        my @text_pages      = split /$PAGE_SEPARATOR/, $page->{content};
        my $page_num        = $R->apache->param( 'pagenum' ) || 1;
        my $this_page       =  $text_pages[ $page_num - 1 ];
        my $total_pages     = scalar @text_pages;
        my $current_pagenum = $page_num;
        $this_page .= <<PCOUNT;
     <p align="right"><font size="-1">
     [%- PROCESS page_count( total_pages     = $total_pages,
                             url             = '$page->{location}',
                             current_pagenum = $current_pagenum ) -%]
     </font></p>
PCOUNT
       return $this_page;
    }
    return $page->{content};
}


sub _add_object_boxes {
    my ( $class, $page, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $box_string = $page->{boxes};

    # Add boxes as necessary -- names beginning with a '-' should be
    # tagged for removal

    if ( $box_string ) {
        $box_string =~ s/^\s+//;
        my @boxes = split /\s+/, $box_string;
        foreach my $box_name ( @boxes ) {
            next if ( $box_name =~ /^\s*$/ );
            $R->DEBUG && $R->scrib( 1, "Adding box name ($box_name) from page definition" );
            my $box_params = { name => $box_name };
            if ( $box_name =~ s/^\-// ) {
                $box_params->{name}   = $box_name;
                $box_params->{remove} = 'yes';
            }
            push @{ $R->{boxes} }, $box_params
        }
    }

    # If this person has WRITE access to the module, give them a box
    # so they can edit/remove this document

    if ( $p->{level} >= SEC_LEVEL_WRITE ) {
        push @{ $R->{boxes} }, { name   => 'edit_document_box',
                                 params => { page => $page } };
    }
    return undef;
}


sub _is_active {
    my ( $class, $page ) = @_;

    return undef if ( $page->{is_active} eq 'no' );
    return 1 unless ( $page->{active_on} );

    my $active  = Class::Date::date([ split /\D/, $page->{active_on} ]);
    my $now     = Class::Date->now;
    my $expires = Class::Date::date([ split /\D/, $page->{expires_on} ]);

    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Active on: $active; Expires on: $expires" );

    return ( $active <= $now and $now <= $expires );

}


1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Page - Display HTML pages and other documents from the database and/or filesystem

=head1 SYNOPSIS

=head1 DESCRIPTION

Displays a 'static' page from information in the database. The URL to
the page looks like a normal page rather than a database call or other
GET request, although it B<can> look like a GET request if you want it
to.

=head2 Error Content-Type Forcing

We have to force the content-type when returning an error in C<show()>
because the user might have requested a file that actually exists in
the filesystem and which Apache has already mapped a content-type. You
will know when this happens because you will be prompted to d/l the
file or a plugin (like Acrobat Reader) will try to display it, but the
*actual* content will be plain old HTML...

=head1 METHODS

We use L<OpenInteract::CommonHandler|OpenInteract::CommonHandler> but
override the C<show()> method for our special needs.

B<directory_list>: implemented in this class

B<search_form>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<search>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<show>: implemented in this class

B<edit>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<remove>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<notify>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
