package OpenInteract::Handler::SystemDoc;

# $Id: SystemDoc.pm,v 1.1.1.1 2001/02/02 06:20:42 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper  qw( Dumper );
use Pod::Parser;

@OpenInteract::Handler::SystemDoc::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::SystemDoc::VERSION = sprintf("%d.%02d", q$Revision: 1.1.1.1 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::SystemDoc::author            = 'chris@cwinters.com';
$OpenInteract::Handler::SystemDoc::default_method    = 'listing';
@OpenInteract::Handler::SystemDoc::forbidden_methods = qw( _colonify _uncolonify );
%OpenInteract::Handler::SystemDoc::security          = ( 
 listing => SEC_LEVEL_READ,  show   => SEC_LEVEL_READ, 
);

use constant MAIN_SCRIPT => '/SystemDoc';

sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { error_msg => $p->{error_msg}, main_script => MAIN_SCRIPT  };
  $R->{page}->{title} = 'OpenInteract System Documentation / Menu';
  return $R->template->handler( {}, 
                                $params, 
                                { db => 'system_doc_menu', 
                                  package => 'system_doc' } );
}



sub package_list {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { error_msg => $p->{error_msg}, main_script => MAIN_SCRIPT };

  # Grab a list of all the packages and find the 'titles' document in
  # each or read the files directly

  my $db_file = join( '/', $R->CONFIG->{dir}->{base}, $OpenInteract::Package::PKG_DB_FILE );
  $params->{package_list} = eval { OpenInteract::Package->fetch_group({ 
                                       directory => $R->CONFIG->{dir}->{base} 
                                   }) };

PACKAGE:
  foreach my $pkg ( @{ $params->{package_list} } ) {
    $R->scrib( 1, "Finding documents for package $pkg->{name}-$pkg->{version}" );
    my $doc_dir = join( '/', $pkg->{website_dir}, $pkg->{package_dir}, 'doc' );
    unless ( -d $doc_dir ) {
      $doc_dir = join( '/', $pkg->{base_dir}, $pkg->{package_dir}, 'doc' );
      next PACKAGE unless ( -d $doc_dir );
    }
    $R->scrib( 1, "Found doc dir: $doc_dir" );
    if ( -f "$doc_dir/titles" ) {
      $R->scrib( 1, "Opening titles file in dir" );
      eval { open( TITLES, "$doc_dir/titles" ) || die $! };
      if ( $@ ) {
        warn " (SystemDoc/package_list): Error opening titles file; reading filenames directly. (Error: $@)\n";
      }
      else {
        while ( <TITLES> ) {
          chomp;
          next if ( /^\s*\#/ );
          next if ( /^\s*$/ );
          my ( $filename, $title ) = /^([\w\.]+)\s+(.*)$/;
          $R->scrib( 1, "Found $filename / $title from $_" );
          push @{ $pkg->{tmp_doc_list} }, { filename => $filename, title => $title };
        }
        close( TITLES );
        next PACKAGE;
      }
    }
    eval { opendir( DOC, $doc_dir ) || die $! };
    if ( $@ ) {
      warn " (SystemDoc/package_list): Could not open $doc_dir: $@\n";
      next PACKAGE;
    }
    my @doc_files = grep { -f "$doc_dir/$_" } grep ! /^(tmp|\.|\_)/, grep ! /~$/, readdir( DOC );
    closedir( DOC );
    $pkg->{tmp_doc_list} = [ map { { filename => $_, title => "$_ (no title)" } } @doc_files ];
  }
  $R->{page}->{title} = 'OpenInteract Package Documentation';
  return $R->template->handler( {}, 
                                $params, 
                                { db => 'package_doc',
                                  package => 'system_doc' } );
}



sub module_list {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { error_msg => $p->{error_msg}, main_script => MAIN_SCRIPT  };
  my $view = $R->apache->param( 'view' ) || 'local';

  # Now sort INC and chop up the files into packages -- depending on
  # what option we were given, restrict the modules to either
  # SPOPS/OpenInteract or Everything
  
  my %this_inc = %INC;
  my @top = ();
  my $website_name = $R->CONFIG->{website_name};
  $R->scrib( 1, "Website name being filtered: ($website_name)" );

  if ( $view eq 'local' ) {
    @top = ( [ 'OpenInteract', [ 'OpenInteract' ] ], [ 'SPOPS', [ 'SPOPS' ] ], [ $website_name, [] ] );
    foreach my $local_module ( sort grep /^(OpenInteract|SPOPS|$website_name)/, keys %this_inc ) {
      my ( $module_type ) = $local_module =~ /^(\w+)\//;
      push @{ $top[0]->[1] }, _colonify( $local_module )  if ( $module_type eq 'OpenInteract' );
      push @{ $top[1]->[1] }, _colonify( $local_module )  if ( $module_type eq 'SPOPS' );
      push @{ $top[2]->[1] }, _colonify( $local_module )  if ( $module_type eq $website_name );
      $R->scrib( 1, "Found package $local_module ($module_type)" );
    }
  }
  elsif ( $view eq 'general' ) {
    
    # Get rid of the SPOPS/OI/ThisApp packages

    foreach ( grep /^OpenInteract/, keys %this_inc ) { delete $this_inc{ $_ } }
    foreach ( grep /^SPOPS/, keys %this_inc ) { delete $this_inc{ $_ } }
    foreach ( grep /^$website_name/, keys %this_inc ) { delete $this_inc{ $_ } }
    
    my $count = -1;
    my $curr_parent = undef;
    foreach my $full_pkg ( sort keys %this_inc ) {
      next unless ( $full_pkg =~ /\.pm$/ );
      my ( $first ) = split /\//, $full_pkg;
      if ( $first ne $curr_parent ) {
        $count++;
        $R->scrib( 1, "First item != parent: ($first) / ($curr_parent)" );
        $curr_parent   = $first;
        $curr_parent   =~ s/\.pm$//;
        $top[ $count ] = [ $curr_parent, [] ]; #  _colonify( $curr_parent ) 
      }
      $R->scrib( 1, "Found package $full_pkg" );
      push @{ $top[ $count ]->[1] }, _colonify( $full_pkg );
    }
  }
  $params->{module_list}  = \@top;
  $R->scrib( 1, "Number of module parents found: ", scalar @top );
  
  $R->{page}->{title} = 'OpenInteract System Documentation / Module Documentation';
  return $R->template->handler( {}, 
                                $params, 
                                { db => 'system_module_listing',
                                  package => 'system_doc' } );
}



sub _colonify {
  my ( $text ) = @_;
  $text =~ s|\.pm$||;
  $text =~ s|/|::|g;
  return $text;
}


sub _uncolonify {
  my ( $text ) = @_;
  $text .= '.pm';
  $text =~ s|::|/|g;
  return $text;
}


sub show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $package = $R->apache->param( 'package' );

  my ( $pod_file, $html_file, $text_file );

  # If this is a package, display the doc

  if ( $package ) {
    my $doc = $R->apache->param( 'doc' );
    my $pkg = OpenInteract::Package->fetch_by_name({ 
                   name => $package, 
                   directory => $R->CONFIG->{dir}->{base}
              });
    if ( $doc =~ /\.(html|txt|pod)$/ ) {
      my $full_filename = $pkg->find_file( "doc/$doc" );
      $R->scrib( 1, "Found ($full_filename) using $doc and $package" );
      $pod_file  = $full_filename  if ( $doc =~ /\.pod$/ );
      $html_file = $full_filename  if ( $doc =~ /\.html$/ );
      $text_file = $full_filename  if ( $doc =~ /\.txt$/ );
    }
  }
  else {
    my $module = $p->{module} || $R->apache->param( 'module' );
    $pod_file = `perldoc -l $module`;
    chomp $pod_file;
    $R->scrib( 1, "Found ($pod_file) from module ($module)" );
    unless ( $pod_file ) {
      $pod_file = $INC{ _uncolonify( $module ) };   
      $R->scrib( 1, "Found ($pod_file) from module ($module)" );
    }
  }
  if ( -f $pod_file ) {
    $R->scrib( 1, "Trying to view pod in $pod_file" );
    chdir( '/tmp' );
    my $pod_text = `pod2html $pod_file`;
    $pod_text =~ s/^.*<BODY>//;
    $pod_text =~ s|</BODY>.*$||;
    return '<p>Sorry, no docs for this module.' unless $pod_text;
    $R->{page}->{title} = 'OpenInteract System Documentation Display';
    return $pod_text;
  } 
  if ( -f $html_file ) {
    eval { open( HTML, $html_file ) || die $! };
    return "<p>Error opening HTML file: $@"  if ( $@ );
    local $/ = undef;
    my $html_text = <HTML>;
    close( HTML );
    $html_text =~ s/^.*<BODY>//;
    $html_text =~ s|</BODY>.*$||;
    return '<p>Sorry, no docs for this module.' unless $html_text;
    $R->{page}->{title} = 'OpenInteract System Documentation Display';
    return $html_text;
  }
  if ( -f $text_file ) {
    eval { open( TEXT, $text_file ) || die $! };
    return "<p>Error opening TEXT file: $@"  if ( $@ );
    local $/ = undef;
    my $text_text = <TEXT>;
    close( TEXT );
    return '<p>Sorry, no docs for this module.' unless $text_text;
    $R->{page}->{title} = 'OpenInteract System Documentation Display';
    return qq(<pre><font size="-1">$text_text</font></pre>);
  }
  return '<p>Sorry, I cannot find documentation by that name';
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::SystemDoc - Display system documentation in HTML format

=head1 SYNOPSIS

=head1 DESCRIPTION

Display documentation for the OpenInteract system, SPOPS modules, and any
other perl modules used.

=head1 METHODS

B<listing()>

List the OpenInteract system documentation and all the modules used by
the system -- we display both the C<OpenInteract> modules and the
C<SPOPS> modules first.

B<package_list( \%params )>

B<module_list( \%params )>

B<show( \%params )>

Display a particular document or module, filtering through C<pod2html>.

Parameters:

 filename
   Full filename of document to extract POD from.

 module
   Perl module to extract POD from; we match up the module to a file
   using %INC

=head1 NOTES

=head1 TO DO

B<Get more meta information>

System documentation needs more meta information so we can better
display title and other information on the listing page.

B<Change POD parser>

The C<pod2html> program is not entirely suitable for what we want to
do. It would be nice to have code sections marked in a certain way and
to handle internal references properly. (That is, something like:
LE<lt>OpenInteract::PackageE<gt> would be handled with a link to the
page within this documentation system.)

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
