package OpenInteract::Handler::SiteTemplate;

# $Id: SiteTemplate.pm,v 1.9 2001/02/01 05:25:03 cwinters Exp $

use strict;
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::SiteTemplate::ISA     = qw( OpenInteract::Handler::GenericDispatcher  SPOPS::Secure );
$OpenInteract::Handler::SiteTemplate::VERSION = sprintf("%d.%02d", q$Revision: 1.9 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::SiteTemplate::author            = 'chris@cwinters.com';
$OpenInteract::Handler::SiteTemplate::default_method    = 'listing';
%OpenInteract::Handler::SiteTemplate::security          = ( 
 listing => SEC_LEVEL_READ,  show   => SEC_LEVEL_READ, 
 edit    => SEC_LEVEL_WRITE, remove => SEC_LEVEL_WRITE,
);

my $DEBUG  = 0;
my $MAIN_SCRIPT = '/SiteTemplate';

sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  $params->{template_list} = eval { $R->site_template->fetch_group({ 
                                                  order => 'package, name' 
                                    }) };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 403 } );
    $params->{template_list} = [];
  }
  
  $R->{page}->{title} = 'Listing of Template Items';
  return $R->template->handler( {}, $params, { db => 'template_list', 
                                               package => 'base_template' } );
}

sub show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => $MAIN_SCRIPT, 
                 error_msg => $p->{error_msg}, 
                 package_list => $R->CONFIG->{package_list} };

  my $template = $p->{template};
  my $template_id = $R->apache->param( 'template_id' );
  $template  ||= eval { $R->site_template->fetch( $template_id ) };
  if ( $@ ) { 
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 404 } );   
  }
 
  # If the user has less than WRITE access to module and there is no
  # template_id or template object, there's no reason to continue

  if ( ! $template and $p->{level} < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to create a new template object. Returning to listing.';
    return $class->listing( { error_msg => $user_msg } );
  }

  # Object security defaults to WRITE: if there is no object,
  # then you're creating it and security then depends on the application

  my $obj_level = ( $template ) ? $template->{tmp_security_level} : SEC_LEVEL_WRITE;

  $template  ||= $R->site_template->new;
  $params->{tmpl} = $template;

  my $update_items = eval { $template->fetch_updates( 5 ) };
  foreach my $update_info ( @{ $update_items } ) {
    my $user = eval { $R->user->fetch( $update_info->[0] ) };
    my $username = ( $user ) ? $user->{login_name} : 'admin';
    push @{ $params->{update_list} }, { login_name => $username, date => $update_info->[1] };
  }

  my $tmpl_name = 'template_detail';
  if ( $p->{level} == SEC_LEVEL_WRITE and $obj_level == SEC_LEVEL_WRITE ) {
    $tmpl_name = 'template_form';
  }
  $R->scrib( 1, "Displaying $tmpl_name with template ID $template->{template_id}" );
  $R->{page}->{title} = 'Detail of Template Item';
  return $R->template->handler( {}, $params, { db => $tmpl_name } ); 
}

sub edit {
  my ( $class, $p ) = @_;
  my $R   = OpenInteract::Request->instance;
  my $apr = $R->apache;
  $R->{page}->{return_url} = '/SiteTemplate/';
  my $template_id = $apr->param( 'template_id' );
  $R->scrib( 1, "Trying to modify ID <<$template_id>>" );
  my $template = eval { $R->site_template->fetch( $template_id ); };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 404 } );
    return $R->listing( { error_msg => 'Cannot retrieve template to modify' } );
  }
  my $obj_level = ( $template ) ? $template->{tmp_security_level} : SEC_LEVEL_WRITE;
  if ( $obj_level < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to modify this template object. Returning to listing.';
    return $class->show( { template => $template, error_msg => $user_msg } );   
  }

  $template ||= $R->site_template->new;
  my $old_name = $template->{name};
  foreach my $field ( @{ $R->site_template->field_list } ) {
    next if ( $field eq 'template_id' );
    $template->{ $field } = $apr->param( $field );
  }
  if ( my $alt = $apr->param( 'package_alt' ) ) {
    $template->{package} = $alt;   
    $R->scrib( 1, "Set package to alternate typed in by user." );
  }
  
  eval { $template->save };
  if ( $@ ) {
    warn "Error saving template ($SPOPS::Error::system_msg); trying again.";
    $template->{name} = ( $old_name ) ? $old_name : $template->generate_random_code( 8 );
    eval { $template->save };
    if ( $@ ) {
      $p->{error_msg} = "Error saving template with new name. Please try again.";
    }
    else {
      $p->{error_msg} = "Template name you entered conflicted with another. New name: $template->{name}.";
    }
  }
  $template->{tmp_security_level} = SEC_LEVEL_WRITE;
  $p->{template} = $template;
  return $class->listing( $p ); 
}

sub remove {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  if ( my $template_id = $R->apache->param( 'template_id' ) ) {
    my $template = eval { $R->site_template->fetch( $template_id ) };
    if ( $@ or ! $template ) {
      return $class->listing( { error_msg => "Cannot remove template: error retrieving template for removal." } );
    }
    
    if ( $template->{tmp_security_level} < SEC_LEVEL_WRITE ) {
      my $user_msg = 'Sorry, you do not have access to remove this template object. Returning to listing.';
      return $class->listing( { error_msg => $user_msg } );   
    }
    eval { $template->remove };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 405 } );
      $p->{error_msg} = 'Cannot remove template object! See error log for details.';
    }
  }
  return $class->listing( $p );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::SiteTemplate - List, display and edit templates

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
