package OpenInteract::Handler::News;

# $Id: News.pm,v 1.12 2001/02/01 05:26:26 cwinters Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper  qw( Dumper );

@OpenInteract::Handler::News::ISA     = qw( OpenInteract::Handler::GenericDispatcher  SPOPS::Secure );
$OpenInteract::Handler::News::VERSION = sprintf("%d.%02d", q$Revision: 1.12 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::News::author            = 'chris@cwinters.com';
$OpenInteract::Handler::News::default_method    = 'listing';
@OpenInteract::Handler::News::forbidden_methods = qw( _process_listing );
%OpenInteract::Handler::News::security          = ( 
 latest  => SEC_LEVEL_READ,
 listing => SEC_LEVEL_READ,  show   => SEC_LEVEL_READ, search => SEC_LEVEL_READ,
 edit    => SEC_LEVEL_WRITE, remove => SEC_LEVEL_WRITE,
 notify  => SEC_LEVEL_READ 
);

my $DEFAULT_EXPIRE = 60 * 60 * 24 * 7 * 12;     # 12 weeks -- default expiration for news item
my $MAIN_SCRIPT    = '/News';

sub latest {
  my ( $class, $p ) = @_; 
  my $R = OpenInteract::Request->instance;
  $p->{num_items} ||= 1;
 
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  my $where = "active_on <= ? AND active = ? ";
  my @values = ( $R->news->now, 'yes' );
  my $news_list = $class->_process_listing({ 
                                  where => $where,
                                  value => \@values });
  @{ $params->{news_list} } = splice( @{ $news_list }, 0, $p->{num_items} );
  return $R->template->handler( {}, $params, { db => 'news_latest',
                                               package => 'news' } );
}


# List current news items

sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  my $where = " active_on <= ? AND active = ? ";
  my @values = ( $R->news->now, 'yes' );
  my $option = $R->apache->param( 'option' );
  if ( $option eq 'older' ) {
    $where .= ' AND ( expires_on < ? ) ';
  }
  else { 
    $where .= ' AND ( expires_on IS NULL OR expires_on > ? ) ';
  }
  push @values, $R->news->now;
  $params->{news_list} = $class->_process_listing({
                                         where => $where, 
                                         value => \@values });

 $R->{page}->{title} = 'News Items';
 return $R->template->handler( {}, $params, { db => 'news_list',
                                              package => 'news' } );
}



sub _process_listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  $R->scrib( 1, "Processing news listing with ($p->{where})" );
  my $date_format = { posted_on => "DATE_FORMAT( posted_on, '%a %b %d %h:%i %p' )",
                      expires_on => "DATE_FORMAT( expires_on, '%a %b %d %h:%i %p' )" };
  my $news_list = eval { $R->news->fetch_group({
                                      where => $p->{where},
                                      value => $p->{value}, 
                                      field_alter => $date_format,
                                      order => 'posted_on DESC' }) };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 403 } );
    $news_list = [];
  }

  # Only grab the first 'chunk' of the news item, as split up by the
  # separator; also create information about the user who posted the
  # story
  
  my %posters = ();
  foreach my $news ( @{ $news_list } ) {
    ( $news->{tmp_content} ) = split '<!--BREAK-->', $news->{news_item};

    # If there is template content in the news item, process it
    
    if ( $news->{tmp_content} =~ m|\[\%| ) {
      $R->scrib( 1, "Processing template directive content for news item ($news->{title})" );
      $news->{tmp_content} = $R->template->handler( {}, {}, 
                                                    { text => $news->{tmp_content} } );
    }
    unless ( $posters{ $news->{posted_by} } ) {
      my $user = eval { $news->posted_by_user() };
      my $poster_info = {};
      if ( $@ ) {
        $poster_info->{login_name} = 'admin';
        $poster_info->{user_id}    = undef;
      }
      else { 
        $poster_info->{login_name} = $user->{login_name};
        $poster_info->{user_id}    = $user->{user_id};
      }
      $posters{ $user->{user_id} } = $poster_info;
    }
    $news->{tmp_poster_info} = $posters{ $news->{posted_by} };
  }
  return $news_list;
}


# Search current news items listing results

sub search {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $search = $R->apache->param( 'search' );
  
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  
  my $where = " active_on <= NOW() AND active = ? ";
  if ( $search ) {
    $search =~ s/ /\', \'/goi;
    my $in_clause = "IN ( '$search' )";
    $where .= " AND ( title $in_clause OR news_item $in_clause ) ";
  }
  my @values = ( 'yes' );
  my $option = $R->apache->param( 'option' );
  if ( $option eq 'older' ) {
    $where .= ' AND ( expires_on < ? ) ';
  }
  else { 
    $where .= ' AND ( expires_on IS NULL OR expires_on > ? ) ';
  }
  push @values, $R->news->now;
  $params->{news_list} = $class->_process_listing({
                                       where => $where, 
                                       value => \@values });
  
 $R->{page}->{title} = 'News Search';
 $params->{search} = $search;
 return $R->template->handler( {}, $params, { db => 'news_list', 
                                              package => 'news' } );
}


# Display detail of a news item -- displays it in editable form for those 
# who have permission to edit the object.

sub show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  my $do_edit = ( $R->apache->param( 'edit' ) and 
                  $p->{level} >= SEC_LEVEL_WRITE );

  $params->{news} = $p->{news} || 
                    eval { $class->_create_object( { _class => $R->news, _id_field => 'news_id' } ) };
  return $class->listing( { error_msg => $@ } ) if ( $@ );
  
  my $date_format = {};
  unless ( $do_edit ) {
    $date_format = { posted_on => "DATE_FORMAT( posted_on, '%a %b %d %h:%i %p' )" };
  }

  # If the user has less than WRITE access to module and there is no
  # news_id or news object, there's no reason to continue

  unless ( $params->{news} or $do_edit ) {
    my $error_msg = 'Sorry, the news object you requested does not exist. Returning to listing.';
    return $class->listing( { error_msg => $error_msg } );
  }
  
  # Object security defaults to WRITE: if there is no object,
  # then you're creating it and security then depends on the application

  my $obj_level = ( $params->{news} ) 
                   ? $params->{news}->{tmp_security_level} 
                   : SEC_LEVEL_WRITE;
 
  $params->{news}  ||= $R->news->new;
  my $tmpl_name = 'news_detail';
  if ( $do_edit and $obj_level == SEC_LEVEL_WRITE ) {
    $tmpl_name = 'news_form';
  }
  $R->{page}->{title} = 'Detail of News Item';
  return $R->template->handler( {}, $params, { db => $tmpl_name, 
                                               package => 'news' } ); 
}



# Make changes to a news item -- either create it or update it

sub edit {
  my ( $class, $p ) = @_; 
  my $R = OpenInteract::Request->instance;
  my $apr = $R->apache;
  $R->{page}->{return_url} = '/News/';

  my $news = eval { $class->_create_object( { _class => $R->news, 
                                              _id_field => 'news_id' } ) };
  return $class->listing( { error_msg => $@ } ) if ( $@ );
  
  # If we didn't retrieve an object, assume it's new and
  # that our default security for this object is WRITE; if the
  # user isn't supposed to be creating news objects at all,
  # he/she should have a READ permission only for the module

  my $obj_level = ( $news ) ? $news->{tmp_security_level} : SEC_LEVEL_WRITE;
  if ( $obj_level < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to modify this news object. Returning to listing.';
    return $class->show( { news => $news, error_msg => $user_msg } );   
  }

  # Again, create a new object if one wasn't created before.
  $news ||= $R->news->new;
  my %no_read = ( posted_by => 1, expires_on => 1, active_on => 1, news_id => 1 );
  foreach my $field ( @{ $R->news->field_list } ) {
    next if ( $no_read{ $field } );
    $R->scrib( 1, "Find value for field <<$field>>" );
    $news->{ $field } = $apr->param( $field );
  }

  # Set our date fields

  $news->{expires_on} = $class->date_read( 'expires_on' );
  $news->{active_on}  = $class->date_read( 'active_on' ) || $news->now;
  
  # substitute <p> for hard returns where needed

  $news->{news_item} =~ s/(\r\n\r\n)(?!<p>)/$1<p>/g;

  # Set the expiration date if it wasn't set

  unless ( $news->{expires_on} ) {
    my $expire_time = $R->{time} + $DEFAULT_EXPIRE;
    my @time_info = localtime( $expire_time );
    $news->{expires_on} = join '-', $time_info[5] + 1900, $time_info[4] + 1, $time_info[3];
  }
  
  # Set other defaults
  
  $news->{posted_by} ||= $R->{auth}->{user}->{user_id};
  $news->{posted_on} ||= $news->now;
  eval { $news->save };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 407 } );
    $p->{error_msg} = 'Cannot save news object! See error log for details.';
  }
  $p->{news} = $news;
  return $class->show( $p ); 
}



sub remove {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $news = eval { $class->_create_object( { _class => $R->news, _id_field => 'news_id' } ) };
  return $class->listing( { error_msg => $@ } ) if ( $@ );
  return $class->listing( { error_msg => 'Cannot location object to remove.' } ) unless ( $news );
  if ( $news->{tmp_security_level} < SEC_LEVEL_WRITE ) {
    my $error_msg = 'Sorry, you do not have access to remove this news object. Returning to listing.';
    return $class->listing( { error_msg => $error_msg } );   
  }
  eval { $news->remove };
  if ( $@ ) {
    my $ei = OpenInteract::Error->set( SPOPS::Error->get );
    $R->scrib( 1, "Cannot remove News: $@\n", Dumper( $ei ) );
    $R->throw( { code => 405 } );
    $p->{error_msg} = "Cannot remove news object! See error log.";
  }
  return $class->listing( $p );
}

sub notify {
  my ( $class ) = @_;
  my $R = OpenInteract::Request->instance;
  my @news_id  = $R->apache->param( 'news_id' );
  my $email    = $R->apache->param( 'email' );
  if ( ! $email or ! scalar @news_id ) {
    return '<h2 align="center">Error</h2>' .
           '<p>Error: Cannot notify anyone about an object when no ID/email is given.</p>';
  }
  my @news_list = ();
  foreach my $nid ( @news_id ) {
    my $news = $R->news->fetch( $nid );
    push @news_list, $news   if ( $news );
  }
  if ( $R->news->notify( { email => $email, subject => 'News notification', 
                           object => \@news_list, type => 'news' } ) ) {
    return '<h2 align="center">Success!</h2>' .
           '<p>Notification sent properly!</p>';
  }
  return '<h2 align="center">Error</h2>' .
         '<p>Error sending email. Please check error logs!</p>';   
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::News - List, display and edit simple news items

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 NOTES

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
