#!/usr/bin/perl
use strict;
use warnings;
use bytes;

use lib 't';
use Test::More tests => 9;
use File::Spec::Functions qw( catfile );
use KinoSearch::Store::RAMInvIndex;
use KinoSearch::Index::CompoundFileReader;
use KinoSearchTestInvIndex qw( get_uscon_docs );

my $DEBUG = 0;
is( $DEBUG, 0, "debug should be disabled" );

# load index pre-generated by java Lucene into memory
my $lucene_index = KinoSearch::Store::RAMInvIndex->new(
    path => catfile( 't', 'lucene_index' ), );
my $lu_comp_file_reader = KinoSearch::Index::CompoundFileReader->new(
    invindex => $lucene_index,
    seg_name => "_1g",
);

{
    no warnings 'redefine';
    no warnings 'once';
    *KinoSearch::InvIndexer::_new_seg_name = sub {'_1g'};
    $KinoSearch::Index::TermInfosWriter::INDEX_INTERVAL = 128;
}
my $whitespace_tokenizer = KinoSearch::Analysis::Tokenizer->new(
    token_re => qr/\S+/,
);
my $invindex = KinoSearch::Store::RAMInvIndex->new( create => 1 );
my $invindexer = KinoSearch::InvIndexer->new(
    invindex => $invindex,
    analyzer => $whitespace_tokenizer,
);
$invindexer->spec_field( name => 'bodytext' );

my $source_docs = get_uscon_docs();
my @sorted_filenames = sort keys %$source_docs;
for my $filename (@sorted_filenames) {
    my $fields = $source_docs->{$filename};
    my $doc = $invindexer->new_doc;
    $doc->set_value( bodytext => $fields->{bodytext} );
    $invindexer->add_doc($doc);
}
$invindexer->finish;

my $kino_comp_file_reader = KinoSearch::Index::CompoundFileReader->new(
    invindex => $invindex,
    seg_name => "_1g",
);

# note: skip 'segments', 'deletable' files
my @files
    = qw( _1g.f0 _1g.fdt _1g.fdx _1g.fnm _1g.frq _1g.prx _1g.tii _1g.tis );

for (@files) {
    my $kino_data = $kino_comp_file_reader->slurp_file($_);
    my $lu_data   = $lu_comp_file_reader->slurp_file($_);

    my $message = "'$_' should match lucene-generated file";
    ok( $lu_data eq $kino_data, $message );
    if ( $lu_data ne $kino_data ) {
        dump_files( $lu_data, $kino_data );
    }
}

sub dump_files {
    return unless $DEBUG;
    eval <<'HEXDUMPING';
    use Data::Hexdumper;
    my ($lu_data, $kino_data) = @_;
    while (length $lu_data or length $kino_data) {
        if (length $lu_data) {
            my $hex = hexdump( data => substr($lu_data, 0, 16, '') );
            $hex =~ s/./L/;
            print $hex;
        }
        if (length $kino_data) {
            my $hex = hexdump( data => substr($kino_data, 0, 16, '') );
            $hex =~ s/./K/;
            print $hex;
        }
    }
    print "\n";
HEXDUMPING
}
