package Net::Async::Graphite;

=head1 NAME

Net::Async::Graphite - Request data from graphite.

=head1 SYNOPSIS

  use Net::Async::Graphite;

  my $graphite = Net::Async::Graphite->new(
      endpoint => 'https://graphite.example.com/',
  );

  # Find (requires graphite-api)
  my $metrics = $graphite->metrics_asperl('foo.*.datum')->get;

  # Download
  my $data = $graphite->render_asperl('foo.*.datum')->get;

  # Draw
  my $png = $graphite->render(png => 'foo.*.datum')->get;
  # or my $png = $graphite->png('foo.*.datum')->get;

  # Draw with gnuplot (requires gnuplot)
  my $ascii = $graphite->plot('foo.*.datum')->get;
  print $ascii;

=head1 DESCRIPTION

An interface to Graphite's data-request APIs graphite-web and
graphite-api (which include's graphite-web's /render API).

Preliminary support exists in this version for most of the
functionality of C</render>, C</metrics/expand>, C</metrics/find> and
C</metrics/index.json> which is described in detail in
L<Net::Async::Graphite::API>.

There is also support to pass the raw data obtained from C</render>
into gnuplot, which includes the ability to format it for terminal
output. This support will probably be made optional at some point or
maybe split into its own package.

=head1 BUGS

No attempt is made to configure a timeout for the HTTP request.

See individual components for their bugs that I know about.

=cut

use v5.10;
use strictures 2;
use Moo;
use Carp;

use namespace::clean;

=head1 OBJECT

Net::Async::Graphite is an object built from individual roles using
L<Moo>. It uses L<IO::Async> internally to provide an asynchronous
interface to the remote API and so all methods return a L<Future>
object.

If you don't know about L<Future> or don't want to, you must only
remember to call C<get> on the return value of any
L<Net::Async::Graphite> method:

    my $data_when_it_comes = $graphite->render('me.tr.ic');
    ...; # Later
    my $data_finally = $data_when_it_comes->get();

    # or

    my $data_now = $graphite->render('me.tr.ic')->get();

L<Net::Async::Graphite>'s API consists of all of the methods,
attributes and accessors defined in its roles (and theirs (and theirs
(etc.))) which do not begin with an C<_>. Being perl you are welcome
to use these private methods, and even dig around inside the object's
guts if you wish, but they do not constitute any part of the stable[*]
API. You are thus also welcome to deal with the ensuing breakage.

[*] Hahaha.

L<Net::Async::Graphite>'s functionality comes from these roles:

=over

=item L<Net::Async::Graphite::API>

Interface between perl and Graphite's APIs (C</metrics> and
C</render>). Of most interest are:

=begin comment

usename and password don't both need a description but then it looks silly.

=end comment

=over

=item username

Mixed in by L<Net::Async::Graphite::HTTPClient>.

=item password

Mixed in by L<Net::Async::Graphite::HTTPClient>.

=item endpoint

=item metrics()

=item render()

=item find_target_from_spec()

=back

=item L<Net::Async::Graphite::Draw>

Render data in various ways after it has been downloaded. Note that
Graphite's C</render> API include its own drawing routines which are
not related to or affected by this role. This role uses data obtained
from C</render> API call with C<format=raw>; the relationship ends
there.

Of most interest are:

=over

=item last_value()

=item plot()

=back

=cut

with 'Net::Async::Graphite::API';

with 'Net::Async::Graphite::Draw';

=back

=head1 ATTRIBUTES

I don't know if I want these.

=over

=item default_from

=item default_until

Default values for C<from=> and C<until=> URI parameters.

=cut

has default_from => is => rw => predicate => 1;

has default_until => is => rw => predicate => 1;

1;

=back

=head1 SEE ALSO

L<Net::Async::Graphite::API>

L<Net::Async::Graphite::Draw>

L<Future>

L<Moo>

L<Net::Async::HTTP>

=head1 AUTHOR

Matthew King <matthew.king@cloudbeds.com>

=cut
