#!/usr/bin/perl

use strict;
use warnings;
use Config;
use Math::Random::MTwist qw(:rand :seed :dist :state);
use Test::More;

use constant IS_32_BIT => ~0 == 0xffff_ffff;
use constant NVMANTBITS => Math::Random::MTwist::NVMANTBITS;
use constant ADVANCE_CT =>
  NVMANTBITS <= 32 ? 3 : NVMANTBITS <= 64 ? 2 : 0;

sub is_string {
  my $what = shift;
  ($what & ~$what) ne '0';
}

# Depending on the NV precision, the PRNG state advances by 1, 2 or 4 steps
# per mts_(l)drand() call. To ensure we can test the same numbers regardless of
# the NV precision, we need to advance the PRNG as if we had maximum
# precision.
sub mt_advance {
    irand32() for 1 .. ADVANCE_CT;
}

# If you change the order of the tests the expected results will change!

# We cannot test all of the distributions because most of them contain loops
# whose loop count we cannot predict.

ok(srand(1_000_686_894) == 1_000_686_894, 'srand');

ok(irand32() == 2_390_553_143, 'irand32');

ok(rand32() =~ /^2\.32830643.+e-0?10$/i, 'rand32');

ok(rand()   =~ /^0\.28705174/, 'rand');
mt_advance();

ok(rd_triangular(0, 2, 1)  =~ /^0\.64310118/, 'rd_triangular');
ok(rd_ltriangular(0, 2, 1) =~ /^0\.88599040/, 'rd_ltriangular');
mt_advance();

ok(rd_uniform(-73, 73)  =~  /^12\.1245369/, 'rd_uniform');
ok(rd_luniform(-73, 73) =~ /^-33\.8653015/, 'rd_luniform');
mt_advance();

ok(rd_double() =~ /^-7.51470990.+e\+141$/i, 'rd_double');
ok(randstr(19) eq "\312\220\304`;*m\342\256\3653\331\245\200\201\213\207\2X", 'randstr');

do {
  my $seed = 0b1010000100111100011110010000101;
  my $d1 = do { seed32($seed); rd_double(); };
  my $d2 = do { seed32($seed); rd_double(0); };
  my $d3 = do { seed32($seed); (rd_double())[0]; };
  ok($d1 == $d2, 'rd_double(0)');
  ok($d2 == $d3, '(rd_double())[0]');
};

do {
  my $state = getstate();
  ok(length $state > 624*4, 'getstate');
  ok(irand32() == 2_419_637_362, 'irand32 before setstate');
  my $rs = rd_double(2);
  ok($rs eq "\220ct\245\221\262g\366", 'rd_double(2) before setstate');

  setstate($state);
  ok(irand32() == 2_419_637_362, 'irand32 after setstate');
  ok($rs eq rd_double(2), 'rd_double(2) after setstate');
};

do {
  my $i = irand64();
  if (! defined $i) {
    ok(IS_32_BIT && !Math::Random::MTwist::HAS_UINT64_T, 'irand64 is undef on 32-bit Perl w/o long long');
    # Dummy calls to advance the state pointer like a regular 64-bit call would.
    irand32();
    irand32();
  }
  elsif (is_string($i)) {
    ok(IS_32_BIT && Math::Random::MTwist::HAS_UINT64_T, 'irand64 is string on 32-bit Perl with long long');
    ok($i eq '8280813916528444879', "irand64 string value");
  }
  else {
    ok($i == 8280813916528444879, 'irand64 numeric value');
  }
};

do {
  my $i = rd_iuniform64(-73, 0);
  if (defined $i) {
    ok($i == -68, 'rd_iuniform64 numeric value');
  }
  else {
    ok(IS_32_BIT, 'rd_iuniform64 is undef on 32-bit Perl');
  }
};

done_testing();
