#!/usr/bin/perl

use strict;
use warnings;
use Config;
use Math::Random::MTwist qw(:rand :seed :dist :state);
use Test::More;

use constant IS_32_BIT => ~0 == 0xffff_ffff;

sub is_string {
  my $what = shift;
  ($what & ~$what) ne '0';
}

# If you change the order of the tests the expected results will change!

ok(srand(1_000_686_894) == 1_000_686_894, 'srand');

ok(irand32() == 2_390_553_143, 'irand32');

do {
  my $i = irand64();
  if (! defined $i) {
    ok(IS_32_BIT && !$Config{d_longlong}, 'irand64 is undef on 32-bit Perl w/o long long');
    # Dummy calls to advance the state pointer like a regular 64-bit call would.
    irand32();
    irand32();
  }
  elsif (is_string($i)) {
    ok(IS_32_BIT && $Config{d_longlong}, 'irand64 is string on 32-bit Perl with long long');
    ok($i eq '5527845158', "irand64 string value");
  }
  else {
    ok($i == 5527845158, 'irand64 numeric value');
  }
};

ok(rand()   =~ /^0\.9457734/, 'rand');
ok(rand32() =~ /^0\.3981395/, 'rand32');

ok(rd_erlang(2, 1)  =~ /^1\.2556495/, 'rd_erlang');
ok(rd_lerlang(2, 1) =~ /^0\.3129639/, 'rd_lerlang');

ok(rd_exponential(1)  =~ /^1\.3165971/, 'rd_exponential');
ok(rd_lexponential(1) =~ /^0\.3011559/, 'rd_lexponential');

ok(rd_lognormal(1, 0)  =~ /^0\.8843699/, 'rd_lognormal');
ok(rd_llognormal(1, 0) =~ /^1\.2886502/, 'rd_llognormal');

ok(rd_normal(5, 1)  =~ /^3\.4375795/, 'rd_normal');
ok(rd_lnormal(5, 1) =~ /^4\.8138142/, 'rd_lnormal');

ok(rd_triangular(0, 2, 1)  =~ /^1\.0779715/, 'rd_triangular');
ok(rd_ltriangular(0, 2, 1) =~ /^1\.3103709/, 'rd_ltriangular');

ok(rd_weibull(1.5, 1)  =~ /^0\.7575942/, 'rd_weibull');
ok(rd_lweibull(1.5, 1) =~ /^0\.5284899/, 'rd_lweibull');

ok(rd_double() =~ /^8.6196948.+e-145$/, 'rd_double');

{
  my $seed = 0b1010000100111100011110010000101;
  my $d1 = do { seed32($seed); rd_double(); };
  my $d2 = do { seed32($seed); rd_double(0); };
  my $d3 = do { seed32($seed); (rd_double())[0]; };
  ok($d1 == $d2, 'rd_double(0)');
  ok($d2 == $d3, '(rd_double())[0]');
}

{
  my $state = getstate();
  ok(length $state > 624*4, 'getstate');
  ok(irand32() == 2_419_637_362, 'irand32 before setstate');
  my $rs = rd_double(2);
  ok($rs eq "\220ct\245\221\262g\366", 'rd_double(2) before setstate');

  setstate($state);
  ok(irand32() == 2_419_637_362, 'irand32 after setstate');
  ok($rs eq rd_double(2), 'rd_double(2) after setstate');
}

ok(randstr(19) eq "\0270\352\20\a)GB\205\332\325\201\32\236\13\30\317\5\312", 'randstr');

do {
  my $i = rd_iuniform64(-42, 42);
  if (defined $i) {
    ok($i == 30, 'rd_iuniform64 numeric value');
  }
  else {
    ok(IS_32_BIT, 'rd_iuniform64 is undef on 32-bit Perl');
    # Dummy calls to advance the state pointer like a regular 64-bit call would.
    irand32();
    irand32();
  }
};

ok(rd_iuniform32(-42, 42) == 2,  'rd_iuniform32');

done_testing();
