#test error reporting for different types of bad input

use t::TestMRCConverter;
use Test::More;
plan tests => 2 * blocks;
use Test::XML;
use Test::LongString;

filters {
	log => 'fix_version',
	tbx => 'fix_version',
	input => 'convert'
};
#loop through all blocks, convert the input, and compare with the output
for my $block ( blocks('input') ){
	my ($tbx, $log) = @{ $block->input };
	
	#compare tbx; check for undefs first
	if( defined $tbx and defined $block->tbx ){
		is_xml($tbx, $block->tbx, "'" . $block->name . "' correct TBX output");
	}else{
		is($tbx,$block->tbx, "'" . $block->name . "' correct TBX output");
	}
	
	#check log for correct messages
	is_string($log, $block->log,  "'" . $block->name . "' correct error(s) logged")
		or print $log;
}

__DATA__
=== No IDs in file

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] The file contained no concepts or parties.

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
		</body>
	</text>
</martif>

=== No IDs in file, and bad header

--- input no_tbx
=MRCtermTable
A	sourceDesc	a restaurant menu in English and French

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] TBX header could not be completed because a required A-row is missing or malformed.
[ERROR] The file contained no concepts or parties.

=== subjectField in header

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
A	subjectField	Restaurant Menus
C005	subjectField	Restaurant Menus
C005en1	term	chick peas
C005en1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Termbase-wide subject fields are recorded in the <titleStmt> element of the TBX header.
[MSG] File includes IDs:
	C005
	C005en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
				<note>entire termbase concerns subject: Restaurant Menus</note>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C005">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="en">
					<tig id="C005en1">
						<term>chick peas</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== missing links

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C005	subjectField	Restaurant Menus
C005	transactionType	origination	Responsibility: Jack	Date: 2007-01-31
C005	transactionType	origination	Link: R005
C005	transactionType	origination	Date: 2007-01-31
C005	xGraphic	garbanzo beans
C005	crossReference	see turkey
C005	externalCrossReference	see chicken website
C005en1	term	chick peas
C005en1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Consider adding information: No responsible party linked in line 5.
[ERROR] Consider adding information: No responsible party linked in line 7.
[ERROR] xGraphic without Link in line 8, skipped.
[ERROR] crossReference without Link in line 9, skipped.
[ERROR] externalCrossReference without Link in line 10, skipped.
[MSG] File includes links to:
	R005
[MSG] File includes IDs:
	C005
	C005en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C005">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-31</date>
					<transacNote type="responsibility">Jack</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<transacNote type="responsibility" target="R005">Responsible Party</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-31</date>
				</transacGrp>
				<langSet xml:lang="en">
					<tig id="C005en1">
						<term>chick peas</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== date formats
One good format, the rest bad

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C005	subjectField	Restaurant Menus
C005	transactionType	origination	Responsibility: Jack	Link: R007	Date: 2007-01-31
C005	transactionType	origination	Responsibility: Jill	Link: R008	Date: 0000-01-31
C005	transactionType	origination	Responsibility: Jim	Link: R008	Date: 2007-00-31
C005	transactionType	origination	Responsibility: Jan	Link: R008	Date: 2007-01-00
C005	transactionType	origination	Responsibility: Jake	Link: R007	Date: 2007-01-01
C005	transactionType	origination	Responsibility: Joanna	Link: R007	Date: 2007-15-01
C005	transactionType	origination	Responsibility: John	Link: R007	Date: 2007-1-21
C005en1	term	chick peas
C005en1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Consider correcting: Zeroes in date '0000-01-31', line 6.
[ERROR] Consider correcting: Zeroes in date '2007-00-31', line 7.
[ERROR] Consider correcting: Zeroes in date '2007-01-00', line 8.
[ERROR] Consider double-checking: Month and day are ambiguous in '2007-01-01', line 9.
[ERROR] Consider correcting: Month 15 is nonsense in line 10.
[ERROR] Date '2007-1-21' not in ISO format (yyyy-mm-dd) in line 11, ignored.
[MSG] File includes links to:
	R007
	R008
	R008
	R008
	R007
	R007
	R007
[MSG] File includes IDs:
	C005
	C005en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C005">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-31</date>
					<transacNote type="responsibility" target="R007">Jack</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>0000-01-31</date>
					<transacNote type="responsibility" target="R008">Jill</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-00-31</date>
					<transacNote type="responsibility" target="R008">Jim</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-00</date>
					<transacNote type="responsibility" target="R008">Jan</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-01</date>
					<transacNote type="responsibility" target="R007">Jake</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-15-01</date>
					<transacNote type="responsibility" target="R007">Joanna</transacNote>
				</transacGrp>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<transacNote type="responsibility" target="R007">John</transacNote>
				</transacGrp>
				<langSet xml:lang="en">
					<tig id="C005en1">
						<term>chick peas</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>


=== disallowed keywords
Also checks that keywords are allowed in correct places

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus	Source: La Restaurant
C003	xGraphic	garbanzo beans	Link: http://flickr.com/photos/lilgreen/432468210/
C003	transactionType	origination	Responsibility: Jill	Link: R007	Date: 2007-01-31
C003	crossReference	see turkey	Link: C004
C003	externalCrossReference	see chicken website	Link: www.chickens.com
C003fr	definition	a squaky bird	Source: http://en.wikipedia.org/wiki/Chicken
C003fr1	term	poulet	Source: a cooking textbook
C003fr1	context	pass the poulet, please	Source: I made it up	Note: hmm
C003fr1	partOfSpeech	noun	Link: www.nouns.com

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Data category term does not allow keyword 'Source', ignored in line 10.
[ERROR] You may attach a source or note to an entire term entry (or a language section or concept entry) by placing it on its own line with the appropriate ID, like this: 
	C003fr1	source	a cooking textbook
[ERROR] Data category context does not allow keyword 'Note', ignored in line 11.
[ERROR] You may attach a source or note to an entire term entry (or a language section or concept entry) by placing it on its own line with the appropriate ID, like this: 
	C003fr1	note	hmm
[ERROR] Data category partOfSpeech does not allow keyword 'Link', ignored in line 12.
[MSG] File includes links to:
	http://flickr.com/photos/lilgreen/432468210/
	R007
	C004
	www.chickens.com
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
					<admin type="source">La Restaurant</admin>
				</descripGrp>
				<xref type="xGraphic" target="http://flickr.com/photos/lilgreen/432468210/">garbanzo beans</xref>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-31</date>
					<transacNote type="responsibility" target="R007">Jill</transacNote>
				</transacGrp>
				<ref type="crossReference" target="C004">see turkey</ref>
				<xref type="externalCrossReference" target="www.chickens.com">see chicken website</xref>
				<langSet xml:lang="fr">
					<descripGrp>
						<descrip type="definition">a squaky bird</descrip>
						<admin type="source">http://en.wikipedia.org/wiki/Chicken</admin>
					</descripGrp>
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
						<descripGrp>
							<descrip type="context">pass the poulet, please</descrip>
							<admin type="source">I made it up</admin>
						</descripGrp>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== disallowed language field
Also checks that no errors are raised for allowed language field locations (Responsibility and Source)

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003	xGraphic	garbanzo beans	Link: [en] http://flickr.com/photos/lilgreen/432468210/
C003	transactionType	origination	Responsibility: [en] Jill	Link: R007	Date: 2007-01-31
C003fr	definition	a squaky bird	Source: [en] http://en.wikipedia.org/wiki/Chicken
C003fr1	term	poulet
C003fr1	partOfSpeech	noun
C004	subjectField	Restaurant Menus
C004fr	definition	[en] a big bird
C004fr1	term	Big Bird
C004fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Language tag makes no sense with keyword 'Link' in line 5, ignored.
[MSG] File includes links to:
	http://flickr.com/photos/lilgreen/432468210/
	R007
[MSG] File includes IDs:
	C003
	C003fr1
	C004
	C004fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<xref type="xGraphic" target="http://flickr.com/photos/lilgreen/432468210/">garbanzo beans</xref>
				<transacGrp>
					<transac type="transactionType">origination</transac>
					<date>2007-01-31</date>
					<transacNote type="responsibility" target="R007" xml:lang="en">Jill</transacNote>
				</transacGrp>
				<langSet xml:lang="fr">
					<descripGrp>
						<descrip type="definition">a squaky bird</descrip>
						<admin type="source" xml:lang="en">http://en.wikipedia.org/wiki/Chicken</admin>
					</descripGrp>
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
			<termEntry id="C004">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<descripGrp>
						<descrip type="definition" xml:lang="en">a big bird</descrip>
					</descripGrp>
					<tig id="C004fr1">
						<term>Big Bird</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== unparseable extra field

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun	bad

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Can't parse additional field 'bad' in line 6, ignored.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== unkown termLocation
Tests that all of the suggested categories raise no errors, and the unkown does.

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun
C003fr1	termLocation	menuItem
C003fr1	termLocation	dialogBox
C003fr1	termLocation	groupBox
C003fr1	termLocation	textBox
C003fr1	termLocation	comboBox
C003fr1	termLocation	comboBoxElement
C003fr1	termLocation	checkBox
C003fr1	termLocation	tab
C003fr1	termLocation	pushButton
C003fr1	termLocation	radioButton
C003fr1	termLocation	spinBox
C003fr1	termLocation	progressBar
C003fr1	termLocation	slider
C003fr1	termLocation	informativeMessage
C003fr1	termLocation	interactiveMessage
C003fr1	termLocation	toolTip
C003fr1	termLocation	tableText
C003fr1	termLocation	userDefinedType
C003fr1	termLocation	userdefinedtype
C003fr1	termLocation	unknown

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Correcting 'userdefinedtype' to 'userDefinedType' in line 25.
[ERROR] Unfamiliar termLocation 'unknown' in line 26. If this is a location in a user interface, consult the suggested values in the TBX spec.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
						<termNote type="termLocation">menuItem</termNote>
						<termNote type="termLocation">dialogBox</termNote>
						<termNote type="termLocation">groupBox</termNote>
						<termNote type="termLocation">textBox</termNote>
						<termNote type="termLocation">comboBox</termNote>
						<termNote type="termLocation">comboBoxElement</termNote>
						<termNote type="termLocation">checkBox</termNote>
						<termNote type="termLocation">tab</termNote>
						<termNote type="termLocation">pushButton</termNote>
						<termNote type="termLocation">radioButton</termNote>
						<termNote type="termLocation">spinBox</termNote>
						<termNote type="termLocation">progressBar</termNote>
						<termNote type="termLocation">slider</termNote>
						<termNote type="termLocation">informativeMessage</termNote>
						<termNote type="termLocation">interactiveMessage</termNote>
						<termNote type="termLocation">toolTip</termNote>
						<termNote type="termLocation">tableText</termNote>
						<termNote type="termLocation">userDefinedType</termNote>
						<termNote type="termLocation">userDefinedType</termNote>
						<termNote type="termLocation">unknown</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== bad ID format

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003	note	I can't parley-voo any French.
C003fr1	term	poulet
C003fr1	partOfSpeech	noun
X003fr1	note	bad row id

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Bad ID 'X003fr1' (format not recognized) in line 8, skipped.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<note>I can't parley-voo any French.</note>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== missing language

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C0031	term	poulet
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Bad ID 'C0031' (no language section) in line 6, skipped.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== incomplete row

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun
C003fr1	geographicalUsage	
C003fr1	

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Incomplete row in line 7, skipped.
[ERROR] Incomplete row in line 8, skipped.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== no =MRCtermTable

--- input
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C001	subjectField	Restaurant Menus
C001fr1	term	poulet
C001fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] The input MRC is missing a line beginning with =MRCTermTable. You must include such a line to switch on the TBX converter -- all preceding material is ignored.

=== No respParty type
Also tests the code for different types of parties

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun
R007	fn	Jill Johnson
R007	email	jill@example.com
R007	title	bean expert
R007	type	person
R008	fn	Jan Jameson
R009	fn	Chickens, Inc.
R009	type	organization

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] At least one of your responsible parties has no type (person, organization, etc.) and has been provisionally printed as a respParty. To conform to TBX-Basic, you must list each party as either a person or an organization.
[MSG] File includes IDs:
	C003
	C003fr1
	R007
	R009
	R008

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
		<back>
			<refObjectList type="respPerson">
				<refObject id="R007">
					<item type="fn">Jill Johnson</item>
					<item type="email">jill@example.com</item>
					<item type="title">bean expert</item>
				</refObject>
			</refObjectList>
			<refObjectList type="respOrg">
				<refObject id="R009">
					<item type="fn">Chickens, Inc.</item>
				</refObject>
			</refObjectList>
			<refObjectList type="respParty">
				<refObject id="R008">
					<item type="fn">Jan Jameson</item>
				</refObject>
			</refObjectList>
		</back>
	</text>
</martif>

=== No respParty type (2)

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun
R007	fn	Some Company
R008	fn 	Some Other Company

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] At least one of your responsible parties has no type (person, organization, etc.) and has been provisionally printed as a respParty. To conform to TBX-Basic, you must list each party as either a person or an organization.
[MSG] File includes IDs:
	C003
	C003fr1
	R007
	R008

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
		<back>
			<refObjectList type="respParty">
				<refObject id="R007">
					<item type="fn">Some Company</item>
				</refObject>
				<refObject id="R008">
					<item type="fn">Some Other Company</item>
				</refObject>
			</refObjectList>
		</back>
	</text>
</martif>
	
=== A C R order incorrect

--- input no_tbx
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
R007	type	person
R007	fn	Jill Johnson
C005	subjectField	Restaurant Menus
C005en1	term	chick peas
C005en1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Don't know what to do with line 6, processing stopped. The rows in your file are not in proper A C R order.

=== LangSet level out of order

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a book about animals
C003	subjectField	animal book
C003en1	term	chicken
C003en1	partOfSpeech	noun
C003en	note	my English is pretty good, isn't it?

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] LangSet-level row out of order in line 7, skipped.
[MSG] File includes IDs:
	C003
	C003en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a book about animals</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">animal book</descrip>
				</descripGrp>
				<langSet xml:lang="en">
					<tig id="C003en1">
						<term>chicken</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== Concept level out of order

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a book about animals
C003	subjectField	animal book
C003en1	term	chicken
C003en1	partOfSpeech	noun
C003	note	that's a real story

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Concept-level row out of order in line 7, skipped.
[MSG] File includes IDs:
	C003
	C003en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a book about animals</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">animal book</descrip>
				</descripGrp>
				<langSet xml:lang="en">
					<tig id="C003en1">
						<term>chicken</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== Term level disallowed data category

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a book about animals
C003	subjectField	animal book
C003en1	xGraphic	My pet chicken is named wanda.	Link: wanda.jpg
C003en1	subjectField	Restaurant Menus
C003en1	definition	a squaky bird
C003en1	term	chicken
C003en1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Data category 'xGraphic' not allowed at the Term level in line 5, skipped.
[ERROR] Data category 'subjectField' not allowed at the Term level in line 6, skipped.
[ERROR] Data category 'definition' not allowed at the Term level in line 7, skipped.
[MSG] File includes IDs:
	C003
	C003en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a book about animals</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">animal book</descrip>
				</descripGrp>
				<langSet xml:lang="en">
					<tig id="C003en1">
						<term>chicken</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== LangSet level disallowed data category

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a book about animals
C003en	xGraphic	my pet chicken	Link: wanda.jpg
C003en	subjectField	Restaurant Menus
C003en	context	the chicken pecked my eye
C003en	geographicalUsage	North America
C003en	grammaticalGender	feminine
C003en	partOfSpeech	noun
C003en	termLocation	menuItem
C003en	termType	fullForm
C003en	term	chicken
C003en	administrativeStatus	admittedTerm-admn-sts

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Data category 'xGraphic' not allowed at the LangSet level in line 4, skipped.
[ERROR] Data category 'subjectField' not allowed at the LangSet level in line 5, skipped.
[ERROR] Data category 'context' not allowed at the LangSet level in line 6, skipped.
[ERROR] Data category 'geographicalUsage' not allowed at the LangSet level in line 7, skipped.
[ERROR] Data category 'grammaticalGender' not allowed at the LangSet level in line 8, skipped.
[ERROR] Data category 'partOfSpeech' not allowed at the LangSet level in line 9, skipped.
[ERROR] Data category 'termLocation' not allowed at the LangSet level in line 10, skipped.
[ERROR] Data category 'termType' not allowed at the LangSet level in line 11, skipped.
[ERROR] Data category 'term' not allowed at the LangSet level in line 12, skipped.
[ERROR] Data category 'administrativeStatus' not allowed at the LangSet level in line 13, skipped.
[MSG] File includes IDs:
	C003

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a book about animals</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<langSet xml:lang="en">
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== Concept level disallowed data category

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a book about animals
C003	subjectField	animals
C003	administrativeStatus	admittedTerm-admn-sts
C003	context	the chicken pecked my eye
C003	definition	a squaky bird
C003	geographicalUsage	North America
C003	grammaticalGender	feminine
C003	partOfSpeech	noun
C003	termLocation	menuItem
C003	termType	fullForm
C003	term	chicken

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Data category 'administrativeStatus' not allowed at the Concept level in line 5, skipped.
[ERROR] Data category 'context' not allowed at the Concept level in line 6, skipped.
[ERROR] Data category 'definition' not allowed at the Concept level in line 7, skipped.
[ERROR] Data category 'geographicalUsage' not allowed at the Concept level in line 8, skipped.
[ERROR] Data category 'grammaticalGender' not allowed at the Concept level in line 9, skipped.
[ERROR] Data category 'partOfSpeech' not allowed at the Concept level in line 10, skipped.
[ERROR] Data category 'termLocation' not allowed at the Concept level in line 11, skipped.
[ERROR] Data category 'termType' not allowed at the Concept level in line 12, skipped.
[ERROR] Data category 'term' not allowed at the Concept level in line 13, skipped.
[MSG] File includes IDs:
	C003

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a book about animals</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">animals</descrip>
				</descripGrp>
			</termEntry>
		</body>
	</text>
</martif>

=== invalid header category

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
A	definition	some random definition
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Could not interpret header line 4, skipped.
[MSG] File includes IDs:
	C003
	C003fr1
	
--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== missing sourceDesc in header

--- input
=MRCtermTable
A	workingLanguage	en
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] TBX header could not be completed because a required A-row is missing or malformed.

--- tbx eval
undef

=== missing workingLanguage in header

--- input
=MRCtermTable
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] TBX header could not be completed because a required A-row is missing or malformed.

--- tbx eval
undef

=== bad header
--- input
=MRCtermTable
A	workingLanguage	en
A	partOfSpeech	noun
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Could not interpret header line 3, skipped.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== duplicate workingLanguage entry

--- input
=MRCtermTable
A	workingLanguage	en
A	workingLanguage	fr
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Duplicate workingLanguage ignored in line 3.
[ERROR] Could not interpret header line 3, skipped.
[MSG] File includes IDs:
	C003
	C003fr1
	
--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== invalid value

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	junk
C004	subjectField	Restaurant Menus
C004en	definition	a squaky bird 
C004en1	context	The chicken pecked my eye.
C004en1	term	chicken

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] 'junk' not a valid partOfSpeech in line 6, skipped. See picklist for valid values.
[ERROR] Term C003fr1 lacks a partOfSpeech row. This TBX file may not be machine processed. See line 6.
[ERROR] Term C003fr1 (see line 6) is lacking an element necessary for TBX-Basic.
	To make it valid for human use only, add one of:
		a definition (at the language level)
		an example of use in context (at the term level).
	To make it valid for human or machine processing, add its part of speech (at the term level).
[ERROR] Term C004en1 lacks a partOfSpeech row. This TBX file may not be machine processed. See line 9.
[MSG] File includes IDs:
	C003
	C003fr1
	C004
	C004en1
	
--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
					</tig>
				</langSet>
			</termEntry>
			<termEntry id="C004">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="en">
					<descripGrp>
						<descrip type="definition">a squaky bird</descrip>
					</descripGrp>
					<tig id="C004en1">
						<term>chicken</term>
						<descripGrp>
							<descrip type="context">The chicken pecked my eye.</descrip>
						</descripGrp>
					</tig>
					
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== missing term

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C005	subjectField	Restaurant Menus
C005en1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] There is no term row for 'C005en1', although other data categories describe such a term. See line 4.
[MSG] File includes IDs:
	C005
	C005en1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C005">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="en">
					<tig id="C005en1">
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== missing partOfSpeech

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Term C003fr1 lacks a partOfSpeech row. This TBX file may not be machine processed. See line 4.
[ERROR] Term C003fr1 (see line 4) is lacking an element necessary for TBX-Basic.
	To make it valid for human use only, add one of:
		a definition (at the language level)
		an example of use in context (at the term level).
	To make it valid for human or machine processing, add its part of speech (at the term level).
[MSG] File includes IDs:
	C003
	C003fr1
	
--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>

=== incorrect category capitalization

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectfield	Restaurant Menus
C003fr1	term	poulet
C003fr1	partOfSpeech	NOUN

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Correcting 'subjectfield' to 'subjectField' in line 4.
[ERROR] Correcting 'NOUN' to 'noun' in line 6.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>


=== bad category

--- input
=MRCtermTable
A	workingLanguage	en
A	sourceDesc	a restaurant menu in English and French
C003	subjectField	Restaurant Menus
C003fr1	term	poulet
C003fr1	POS	noun
C003fr1	partOfSpeech	noun

--- log
[MSG] MRC2TBX converter version [version]
[ERROR] Unknown data category 'POS' in line 6, skipped.
[MSG] File includes IDs:
	C003
	C003fr1

--- tbx
<?xml version='1.0' encoding="UTF-8"?>
<!DOCTYPE martif SYSTEM "TBXBasiccoreStructV02.dtd">
<martif type="TBX-Basic-V1" xml:lang="en">
	<martifHeader>
		<fileDesc>
			<titleStmt>
				<title>termbase from MRC file</title>
			</titleStmt>
			<sourceDesc>
				<p>generated by Convert::MRC version [version]</p>
			</sourceDesc>
			<sourceDesc>
				<p>a restaurant menu in English and French</p>
			</sourceDesc>
		</fileDesc>
		<encodingDesc>
			<p type="DCSName">TBXBasicXCSV02.xcs</p>
		</encodingDesc>
	</martifHeader>
	<text>
		<body>
			<termEntry id="C003">
				<descripGrp>
					<descrip type="subjectField">Restaurant Menus</descrip>
				</descripGrp>
				<langSet xml:lang="fr">
					<tig id="C003fr1">
						<term>poulet</term>
						<termNote type="partOfSpeech">noun</termNote>
					</tig>
				</langSet>
			</termEntry>
		</body>
	</text>
</martif>
