=head1 NAME

Sys::Statistics::Linux::PgSwStats - Collect linux paging and swapping statistics.

=head1 SYNOPSIS

   use Sys::Statistics::Linux::PgSwStats;

   my $lxs = new Sys::Statistics::Linux::PgSwStats;
   $lxs->init;
   sleep 1;
   my $stats = $lxs->get;

=head1 DESCRIPTION

This module collects statistics by the virtual F</proc> filesystem (procfs) and is statseloped on default vanilla
kernels. It is tested on x86 hardware with the distributions SuSE (SuSE on s390 and s390x architecture as well),
Red Hat, Debian, Asianux, Slackware and Mandrake on kernel versions 2.4 and 2.6 and should run on all linux
kernels with a default vanilla kernel as well. It is possible that this module doesn't run on all distributions
if the procfs is too much changed.

Further it is necessary to run it as a user with the authorization to read the F</proc> filesystem.

=head1 DELTAS

It's necessary to initialize the statistics by calling C<init()>, because the statistics are deltas between
the call of C<init()> and C<get()>. By calling C<get()> the deltas be generated and the initial values will
be updated automatically. This way making it possible that the call of C<init()> is only necessary
after the call of C<new()>. Further it's recommended to sleep for a while - at least one second - between
the call of C<init()> and/or C<get()> if you want to get useful statistics.

=head1 PAGING AND SWAPPING STATISTICS

Generated by F</proc/stat> or F</proc/vmstat>.

   pgpgin   -  Number of kilobytes the system has paged in from disk per second.
   pgpgout  -  Number of kilobytes the system has paged out to disk per second.
   pswpin   -  Number of kilobytes the system has swapped in from disk per second.
   pswpout  -  Number of kilobytes the system has swapped out to disk per second.

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

   my $lxs = new Sys::Statistics::Linux::PgSwStats;

=head2 init()

Call C<init()> to initialize the statistics.

   $lxs->init;

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

   my $stats = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<proc(5)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::PgSwStats;
our $VERSION = '0.07';

use strict;
use warnings;
use Carp qw(croak);

sub new {
   my $class = shift;
   my %self = (
      files => {
         stat => '/proc/stat',
         vmstat => '/proc/vmstat',
         uptime => '/proc/uptime',
      },
   );
   return bless \%self, $class;
}

sub init {
   my $self = shift;
   $self->{uptime} = $self->_uptime;
   $self->{init} = $self->_load;
}

sub get {
   my $self  = shift;
   my $class = ref $self;

   croak "$class: there are no initial statistics defined"
      unless exists $self->{init};

   $self->{stats} = $self->_load;
   $self->_deltas;
   return $self->{stats};
}

#
# private stuff
#

sub _load {
   my $self  = shift;
   my $class = ref $self;
   my $file  = $self->{files};
   my %stats = ();

   open my $fh, '<', $file->{stat} or croak "$class: unable to open $file->{stat} ($!)";

   while (my $line = <$fh>) {
      if ($line =~ /^page\s+(\d+)\s+(\d+)$/) {
         @stats{qw(pgpgin pgpgout)} = ($1, $2);
      } elsif ($line =~ /^swap\s+(\d+)\s+(\d+)$/) {
         @stats{qw(pswpin pswpout)} = ($1, $2);
      }
   }

   close($fh);

   # if paging and swapping are not found in /proc/stat
   # then let's try a look into /proc/vmstat (since 2.6)

   unless (defined $stats{pswpout}) {
      open my $fh, '<', $file->{vmstat} or croak "$class: unable to open $file->{vmstat} ($!)";
      while (my $line = <$fh>) {
         next unless $line =~ /^(pgpgin|pgpgout|pswpin|pswpout)\s+(\d+)/;
         $stats{$1} = $2;
      }
      close($fh);
   }

   return \%stats;
}

sub _deltas {
   my $self   = shift;
   my $class  = ref $self;
   my $istat  = $self->{init};
   my $lstat  = $self->{stats};
   my $uptime = $self->_uptime;
   my $delta  = sprintf('%.2f', $uptime - $self->{uptime});
   $self->{uptime} = $uptime;

   while (my ($k, $v) = each %{$lstat}) {
      croak "$class: different keys in statistics"
         unless defined $istat->{$k} && defined $lstat->{$k};
      croak "$class: value of '$k' is not a number"
         unless $v =~ /^\d+$/ && $istat->{$k} =~ /^\d+$/;

      $lstat->{$k} =
         $lstat->{$k} == $istat->{$k}
            ? sprintf('%.2f', 0)
            : $delta > 0
               ? sprintf('%.2f', ($lstat->{$k} - $istat->{$k}) / $delta)
               : sprintf('%.2f', $lstat->{$k} - $istat->{$k});


      $istat->{$k}  = $v;
   }
}

sub _uptime {
   my $self  = shift;
   my $class = ref $self;
   my $file  = $self->{files};
   open my $fh, '<', $file->{uptime} or croak "$class: unable to open $file->{uptime} ($!)";
   my ($up, $idle) = split /\s+/, <$fh>;
   close($fh);
   return $up;
}

1;
