=head1 CLASS
Name:	POP::Persistent
Desc:	This is the persistent base class for POP.  It handles all of the
	persistence logic by using a tied hash implementation to intercept
	all attribute fetches and stores.  The basic algorithm is to update
	the changed attribute on a store, and reload the object from
	persistence on fetch if any attribute has been changed by another
	process since the last load.  A shared memory segment with one byte
	per object is used to implement an object version scheme.  There
	are a number of additional optimizations.  See the POP documentation
	for more details.
=cut
require 5.005;
package POP::Persistent;

$VERSION = do{my(@r)=q$Revision: 1.14 $=~/d+/g;sprintf '%d.'.'%02d'x$#r,@r};

use strict;
use vars qw/@ISA $pid_factory %CLASSES %OBJECTS $VERSION/;
use Tie::Hash;
use DBI;
use Carp;
use POP::Environment;
use POSIX qw/floor/;
use constant NOT_FOUND	=> -1;
use constant SHM_KEY => 1999;
use constant SHM_SIZE => 2**17; # 131,072
use constant SEM_STEP => 8192;
use IPC::SysV qw(IPC_CREAT S_IRWXU S_IRWXG);
use IPC::Semaphore;
use Devel::WeakRef;
use POP::Lazy_object;
use POP::Lazy_object_list;
use POP::Lazy_object_hash;
use POP::List;
use POP::Hash;
use POP::Pid_factory;
use POP::POX_parser;

@ISA = qw/Tie::StdHash/;

my $shmid = shmget(SHM_KEY, SHM_SIZE, S_IRWXU|S_IRWXG|IPC_CREAT);
unless ($shmid) { croak "shmget failed: $!" }
my $semset = IPC::Semaphore::->new(SHM_KEY, SHM_SIZE / SEM_STEP, S_IRWXU|S_IRWXG|IPC_CREAT);
unless ($semset) {
  if ($! == 22) {
    croak "SEM_STEP [".&SEM_STEP."] is too small";
  }
  croak "semget failed: $!";
}
my $pid_factory = POP::Pid_factory->new;

# %OBJECTS is our object "cache"; we don't want to interfere with
# normal ref-counting, so we use the convenient Devel::WeakRef::Table :)
tie(%OBJECTS, 'Devel::WeakRef::Table') or croak "object cache tie failed";

# This is used to parse the XML class definition files:
my $parser = POP::POX_parser::->new();

my $dsn;
if ($DBI_DRIVER eq 'Sybase') {
  $dsn = "dbi:Sybase:server=$DB_SERVER;database=$DB_DB";
} else {
  croak "Unknown driver [$DBI_DRIVER]";
}
my $dbh = DBI->connect($dsn, $DB_USER, $DB_PASSWD,
		     { RaiseError => 1,
		       AutoCommit => 0 }) or
  croak "Couldn't connect to [$dsn]: $DBI::errstr";

sub new {
  my $class = shift;
  unless ($CLASSES{$class}) {
    my $class_def_file = &POP::POX_parser::pox_find($class);
    unless ($class_def_file) {
      croak "Couldn't find POX for [$class]. POXLIB=($ENV{POXLIB})";
    }
    $CLASSES{$class} = $parser->parse($class_def_file);
  }
  my $pid;
  if (@_ & 1) { # Odd number of parameters
    $pid = shift;
  }
  my %this = @_;
  my $this = bless \%this, $class;
  if ($pid) { # Pid supplied to constructor
    if ($OBJECTS{$pid}) {
      return $OBJECTS{$pid};
    }
    $this->_POP__Persistent_restore_from_pid($pid);
  } else { # Create a new object; nothing supplied to constructor
    $pid = $this->{'_pop__persistent_pid'} =
      $class->_POP__Persistent_new_pid;
    shmwrite($shmid, $this->{'_pop__persistent_version'} = "\cA", $pid, 1);
    # call our calling classes' initializing routine, if it exists.
    if ($this->can('initialize')) {
      $this->initialize;
    }
    $this->_POP__Persistent_store_all;
  }
  tie(%this, $class, %this);
  $OBJECTS{$pid} = $this;
  return $this;
}

sub DESTROY {
  my $this = shift;
  my $tied = tied %$this;
  untie %$this if $tied;
#  $this->_POP__Persistent_store_all if $this->{'_pop__persistent_changed'};
}

sub TIEHASH {
  my $class = shift;
  my $this = bless {@_}, $class;
  return $this;
}

sub FETCH {
  my($this, $key) = @_;
  my $ver;
  shmread($shmid, $ver, $this->{'_pop__persistent_pid'}, 1) or
    croak "shmread failed on [$this->{'_pop__persistent_pid'}]: $!";
  if ($ver ne $this->{'_pop__persistent_version'}) {
    $this->{'_pop__persistent_version'} = $ver;
    $this->_POP__Persistent_load;
  }
  return $this->{$key};
}

sub STORE {
  # @subkeys is used when it's actuall a collection underneath us
  # informing us that something's changed.  See POP::Hash::STORE and
  # POP::Lazy_object_hash::STORE.
  my($this, $key, $value, @subkeys) = @_;
  $this->{$key} = $value unless @subkeys;
  my $pid = $this->{'_pop__persistent_pid'};
  # Take semaphore
  my $sem_num = floor($pid/SEM_STEP);
  $semset->op($sem_num, 0, 0,
	      $sem_num, 1, 0) or
    croak "semaphore take on [$sem_num] failed: $!";
  $this->{'_pop__persistent_version'} =
    pack("C", unpack("C", $this->{'_pop__persistent_version'})+1);
  unless (shmwrite($shmid, $this->{'_pop__persistent_version'}, $pid, 1)) {
    my $shm_fail = "$!";
    # Give semaphore
    $semset->op($sem_num, -1, 0) or
      croak "shmwrite failed on [$pid]: $shm_fail and semaphore give on".
	    " [$sem_num] failed: $!";
    croak "shmwrite failed on [$pid]: $shm_fail";
  }
  eval {
    $this->_POP__Persistent_store_attr($key, @subkeys);
  };
  if ($@) {
    $dbh->rollback;
    $semset->op($sem_num, -1, 0) or
      croak "STORE on [$pid] {$key} failed: $@".
	    " and semaphore give on [$sem_num] failed: $!";
    croak "STORE on [$pid] {$key} failed: $@";
  } else {
    $dbh->commit;
  }
  # Give semaphore
  $semset->op($sem_num, -1, 0) or
    croak "semaphore give on [$sem_num] failed: $!";
  return $value;
}

sub delete {
  my $this = shift;
  my $pid = $this->pid;
  eval {
    my $class_def = $CLASSES{ref $this};
    my $proc = $class_def->{'abbr'} || lc($class_def->{'name'});
    my $sth = $dbh->prepare("exec ${proc}#DEL $pid");
    $sth->execute;
    if (($sth->fetchrow)[0] == 1) { # Object still referenced
      croak "object still referenced";
    }
  };
  if ($@) {
    $dbh->rollback;
    croak "delete on [$pid] failed: $@";
  } else {
    $dbh->commit;
    untie %$this;
  }
}

sub pid {
  my $this = shift;
  if (my $tied = tied $this) {
    $this = $tied;
  }
  return $this->{'_pop__persistent_pid'};
}

sub all {
  my($this, @attr) = @_;
  if (my $tied = tied $this) {
    $this = $tied;
  }
  my %opts;
  # Pull off leading hash ref of options:
  if (UNIVERSAL::isa($attr[0], 'HASH')) {
    %opts = %{shift @attr};
  }
  my $where_clause;
  my $class = ref $this ? ref $this : $this;
  unless ($CLASSES{$class}) {
    my $class_def_file = &POP::POX_parser::pox_find($class);
    unless ($class_def_file) {
      croak "Couldn't find POX for [$class]. POXLIB=($ENV{POXLIB})";
    }
    $CLASSES{$class} = $parser->parse($class_def_file);
  }
  if ($opts{'where'}) {
    $where_clause = $this->_POP__Persistent_compute_where_clause($opts{'where'});
  }
  my $c = $CLASSES{$class};
  my $lc_name = $c->{'abbr'} || lc($c->{'name'});
  my(@abbr, @type);
  unless (@attr) {
    @attr = ('pid');
  }
  foreach my $attr (@attr) {
    if ($attr eq 'pid') {
      push(@abbr, 'pid');
      push(@type, 'pidtype');
    } elsif (my $a = $c->{'attributes'}{$attr}) {
      if ($a->{'list'} || $a->{'hash'}) {
        croak "Cannot select multi-valued attribute for return";
      }
      push(@abbr, $a->{'dbname'});
      push(@type, $a->{'type'});
    } elsif (my $p = $c->{'participants'}{$attr}) {
      push(@abbr, $p->{'dbname'});
      push(@type, $p->{'type'});
    } else {
      croak "Unknown attribute [$attr]";
    }
  }
  my $select_cols = join ',',@abbr;
  my $ob_name;
  if ($opts{'sort'}) {
    $ob_name = $c->{'attributes'}{$opts{'sort'}}{'dbname'} ||
		$c->{'participants'}{$opts{'sort'}}{'dbname'};
  } else {
    $ob_name = 'pid';
  }
  my $sth = $dbh->prepare("select $select_cols from $lc_name $where_clause order by $ob_name");
  $sth->execute;
  my $result = $sth->fetchall_arrayref;
  $sth->finish;
  my @return;
  $#return = $#{$result};
  for (my $i; $i < @$result; $i++) {
    my $row = $result->[$i];
    if (@$row > 1) {
      for (my $j; $j<@$row; $j++) {
	push(@{$return[$i]},
	  &_POP__Persistent_type_from_db($type[$j], $row->[$j]));
      }
    } else {
      $return[$i] = &_POP__Persistent_type_from_db($type[0], $row->[0]);
    }
  }
  return wantarray ? @return : \@return;
}

sub _POP__Persistent_compute_where_clause {
  my($this, $where) = @_;
  # Where clauses should be supplied like this:
  # [ [ ATTR, OP, VALUE ], CONNECTOR, [ATTR, OP, VALUE] ]
  # where OP is one of {'=', '>', '<', '>=', '<=', '!='}
  # and CONNECTOR is one of {'AND', 'OR'}
  # ( yeah, I know this is incomplete, but it's a start )
  my $sql = 'where ';
  my $class = ref $this ? ref $this : $this;
  my $c = $CLASSES{$class};
  foreach my $expr_or_conn (@$where) {
    if (ref $expr_or_conn) {
      my($attr, $op, $val) = @$expr_or_conn;
      if ($c->{'attributes'}{'list'} ||
	  $c->{'attributes'}{'hash'}) {
	croak "Cannot use multi-valued attribute in where clause";
      }
      if ($c->{'attributes'}{$attr}) {
        $val = &_POP__Persistent_type_to_db(
	  $c->{'attributes'}{$attr}{'type'}, $val);
        $sql .= "$c->{'attributes'}{$attr}{'dbname'} $op $val";
      } elsif ($c->{'participants'}{$attr}) {
        $val = &_POP__Persistent_type_to_db(
	  $c->{'participants'}{$attr}{'type'}, $val);
        $sql .= "$c->{'participants'}{$attr}{'dbname'} $op $val";
      } else { croak "[$attr] is neither an attribute nor a participant" }
    } else {
      $sql .= " $expr_or_conn ";
    }
  }
  return $sql;
}

sub _POP__Persistent_new_pid {
  my $this = shift;
  my $class = ref $this || $this;
  my $new_pid = $pid_factory->next;
  return $new_pid;
}

sub _POP__Persistent_restore_from_pid {
  my($this, $pid) = @_;
  $this->{'_pop__persistent_pid'} = $pid;
  # Take semaphore
  my $sem_num = floor($pid/SEM_STEP);
  $semset->op($sem_num, 0, 0,
	      $sem_num, 1, 0) or
    croak "semaphore take on [$sem_num] failed: $!";
  my $ver;
  unless (shmread($shmid, $ver, $pid, 1)) {
    my $shm_fail = "$!";
    $semset->op($sem_num, -1, 0) or
      croak "shmread failed on [$pid]: $shm_fail and semaphore give on".
	    " [$sem_num] failed: $!";
    croak "shmread failed on [$pid]: $shm_fail";
  }
  $this->{'_pop__persistent_version'} = $ver;
  # Give semaphore
  $semset->op($sem_num, -1, 0) or
    croak "semaphore give on [$sem_num] failed: $!";
  $this->_POP__Persistent_load;
}

sub _POP__Persistent_load {
  my $this = shift;
  my $class_def = $CLASSES{ref $this};
  my $pid = $this->pid;
  my $proc = $class_def->{'abbr'} || lc($class_def->{'name'});
  eval {
    my $sth = $dbh->prepare("exec ${proc}#GET $pid");
    $sth->execute;
    my $result = $sth->fetchall_arrayref();
    $sth->finish;
    unless (@$result > 0) {
      croak "Object [$pid] not found.";
    }
    my $i;
    # NOTE - we do rely on the hash-walking ordering being the same
    # between $class_def here and poxdb.
    foreach (values %{$class_def->{'participants'}}, values %{$class_def->{'scalar_attributes'}}) {
      $this->{$_->{'name'}} =
        &_POP__Persistent_type_from_db($_->{'type'}, $result->[0][$i++]);
    }
    # now the list ones...
    foreach my $att (values %{$class_def->{'list_attributes'}}) {
      next if $this->{'_pop__persistent_mv_attr_vers'}{$att->{'name'}} ==
	      $result->[0][$i++];
      $this->{'_pop__persistent_mv_attr_vers'}{$att->{'name'}} =
	 $result->[0][$i-1];
      my $name = $att->{'abbr'} || lc($att->{'name'});
      $sth = $dbh->prepare("exec ${proc}#GET\@$name $pid");
      $sth->execute();
      my $list_result = $sth->fetchall_arrayref();
      $sth->finish;
      $this->{$att->{'name'}} = $this->_POP__Persistent_list_from_db(
	$att->{'type'}, $att->{'name'}, map {$_->[0]} @$list_result);
    }
    # now the hash ones...
    foreach my $att (values %{$class_def->{'hash_attributes'}}) {
      next if $this->{'_pop__persistent_mv_attr_vers'}{$att->{'name'}} ==
	      $result->[0][$i++];
      $this->{'_pop__persistent_mv_attr_vers'}{$att->{'name'}} =
	 $result->[0][$i-1];
      my $name = $att->{'abbr'} || lc($att->{'name'});
      $sth = $dbh->prepare("exec ${proc}#GET\@$name $pid");
      $sth->execute();
      my $list_result = $sth->fetchall_arrayref();
      $sth->finish;
      $this->{$att->{'name'}} = $this->_POP__Persistent_hash_from_db(
	$att->{'val_type'},
	$att->{'name'},
	{map {&_POP__Persistent_type_from_db($att->{'key_type'}, $_->[0]),
	     $_->[1]} @$list_result});
    }
  };
  if ($@) {
    $dbh->rollback;
    croak "load of [$pid] failed: $@";
  } else {
    $dbh->commit;
  }
}

sub _POP__Persistent_store_attr {
  my($this, $key, @subkeys) = @_;
  my $pid = $this->pid;
  my $class_def = $CLASSES{ref $this};
  my $attr = $class_def->{'attributes'}{$key} ||
		$class_def->{'participants'}{$key};
  my $proc = $class_def->{'dbname'};
  my $name = $attr->{'dbname'};
  if ($attr->{'hash'}) {
    if (@subkeys) {
     for my $subkey (@subkeys) {
      $subkey = &_POP__Persistent_type_to_db($attr->{'key_type'}, $subkey);
      $dbh->do("exec ${proc}#DEL\@$name $pid, $subkey");
      # Yuck. This isn't so good. this has shared knowledge with
      # hash_to_db and Lazy_object_hash
      if ($attr->{'val_type'} =~ /::/) {
	$dbh->do("exec ${proc}#SET\@$name $pid, $subkey, ".
	  $this->{$key}{$subkey}->pid);
      } else {
	$dbh->do("exec ${proc}#SET\@$name $pid, $subkey, ".
	  &_POP__Persistent_type_to_db($attr->{'val_type'},
					$this->{$key}{$subkey}));
      }
     }
    } else {
      $dbh->do("exec ${proc}#DEL\@$name $pid");
      my %values =
      &_POP__Persistent_hash_to_db($attr->{'key_type'},
				   $attr->{'val_type'},
				   $this->{$key});
      while (my($k, $v) = each %values) {
        $dbh->do("exec ${proc}#SET\@$name $pid, $k, $v");
      }
    }
    my $sth = $dbh->prepare("exec ${proc}#VER\@$name $pid");
    $sth->execute();
    $this->{'_pop__persistent_mv_attr_vers'}{$attr->{'name'}} =
	($sth->fetch)[0]->[0];
    $sth->finish();
  } elsif ($attr->{'list'}) {
    if (@subkeys) {
     for my $subkey (@subkeys) {
      $dbh->do("exec ${proc}#DEL\@$name $pid, $subkey");
      # Yuck. This isn't so good. this has shared knowledge with
      # list_to_db and Lazy_object_list
      if ($attr->{'type'} =~ /::/) {
	$dbh->do("exec ${proc}#SET\@$name $pid, ".$this->{$key}[$subkey]->pid.
		 ", $subkey");
      } else {
	$dbh->do("exec ${proc}#SET\@$name $pid, ".
	  &_POP__Persistent_type_to_db($attr->{'type'}, $this->{$key}[$subkey]).
	  ", $subkey");
      }
     }
    } else {
      $dbh->do("exec ${proc}#DEL\@$name $pid");
      my @values =
        &_POP__Persistent_list_to_db($attr->{'type'}, $this->{$key});
      for (my $i = 0; $i < @values; $i++) {
        $dbh->do("exec ${proc}#SET\@$name $pid, $values[$i], $i");
      }
    }
    my $sth = $dbh->prepare("exec ${proc}#VER\@$name $pid");
    $sth->execute();
    $this->{'_pop__persistent_mv_attr_vers'}{$attr->{'name'}} =
	($sth->fetch)[0]->[0];
    $sth->finish();
  } else {
    $dbh->do("exec ${proc}#SET\$$name $pid, ".
	&_POP__Persistent_type_to_db($attr->{'type'}, $this->{$key}));
  }
}

sub _POP__Persistent_store_all {
  my($this, $attr) = @_;
  my $pid = $this->pid;
  my $class_def = $CLASSES{ref $this};
  my $proc = $class_def->{'abbr'} || lc($class_def->{'name'});
  eval {
    $dbh->do("exec ${proc}#SET ".
      join(', ', $pid,
	(map {&_POP__Persistent_type_to_db($_->{'type'}, $this->{$_->{'name'}})}
	    values %{$class_def->{'participants'}},
            values %{$class_def->{'scalar_attributes'}}),
        map {$this->{'_pop__persistent_mv_attr_vers'}{$_}||0}
	    keys %{$class_def->{'list_attributes'}},
	    keys %{$class_def->{'hash_attributes'}}));
    foreach (keys %{$class_def->{'list_attributes'}},
	     keys %{$class_def->{'hash_attributes'}}) {
      $this->_POP__Persistent_store_attr($_);
    }
  };
  if ($@) {
    $dbh->rollback;
    croak "store-all of [$pid] failed: $@";
  } else {
    $dbh->commit;
  }
}

sub _POP__Persistent_list_to_db {
  my($type, $elems) = @_;
  if ($type =~ /::/) {
    if (tied @$elems) {
      return (tied @$elems)->PIDS;   
    } else {
      return map {ref $_ ? $_->pid : $_} @$elems;
    }
  }
  return map {&_POP__Persistent_type_to_db($type, $_)} @$elems;
}

sub _POP__Persistent_list_from_db {
  my($this, $type, $name) = splice(@_,0,3);
  my @temp;
  if ($type =~ /::/) {
    # Embedded object.
    tie(@temp, 'POP::Lazy_object_list', $type, $name, $this, @_);
  } else {
    tie(@temp, 'POP::List', $name, $this,
      map {&_POP__Persistent_type_from_db($type, $_)} @_);
  }
  return \@temp;
}

sub _POP__Persistent_hash_to_db {
  my($key_type, $val_type, $elems) = @_;
  if ($val_type =~ /::/) {
    if (tied %$elems) {
      return (tied %$elems)->PIDS;   
    } else {
      my %ret;
      while (my($k,$v) = each %$elems) {
        $ret{&_POP__Persistent_type_to_db($key_type, $k)} =
	  (ref $v ? $v->pid : 0);
      }
      return wantarray ? %ret : \%ret;
    }
  }
  my %ret;
  while (my($k,$v) = each %$elems) {
    $ret{&_POP__Persistent_type_to_db($key_type, $k)} =
      &_POP__Persistent_type_to_db($val_type, $v);
  }
  return wantarray ? %ret : \%ret;
}

sub _POP__Persistent_hash_from_db {
  my($this, $val_type, $name, $elems) = @_;
  my %temp;
  if ($val_type =~ /::/) {
    # Embedded object.
    tie(%temp, 'POP::Lazy_object_hash', $val_type, $name, $this, $elems);
  } else {
    foreach (keys %$elems) {
      $elems->{$_} = &_POP__Persistent_type_from_db($val_type, $elems->{$_});
    }
    tie(%temp, 'POP::Hash', $name, $this, $elems)
  }
  return wantarray ? %temp : \%temp;
}

sub _POP__Persistent_type_from_db {
  my($type, $val) = @_;
  if ($type =~ /::/) {
    # Embedded object. We just get the pid back from the db, so tie it;
    # on its first access, $temp will be replaced with the actual object
    if ($val) {
      my $temp;
      tie($temp, 'POP::Lazy_object', \$temp, $type, $val);
      return \$temp;
    } else {
      return \do{my $a};
    }
  }
  if ($type =~ /^numeric/ || $type eq 'pidtype' || $type eq 'int') {
    return $val;
  } elsif ($type eq 'datetime') {
    return &_POP__Persistent_date_from_db($val);
  } elsif ($type =~ /^(?:var)?char/) {
    return &_POP__Persistent_char_from_db($val);
  } elsif ($type eq 'text' || $type eq 'bit') {
    return $val;
  } else {
    croak "unknown type [$type]";
  }
  $val;
}

sub _POP__Persistent_type_to_db {
  my($type, $val) = @_;
  if ($type =~ /::/) {
    # Be careful not to restore a lazy-load object if we don't want to:
    if (tied $val) {
      return (tied $val)->pid;
    } elsif (ref $val && UNIVERSAL::isa($val, __PACKAGE__))  {
      return $val->pid;
    } elsif (ref $val eq 'REF' && ref $$val && UNIVERSAL::isa($$val, __PACKAGE__)) {
      return ($$val)->pid;
    } else {
      # Hmm, should be an object, but there's nothing there.  croak? XXX
#      print "VAL NOT AN OBJECT [$type, $val]\n";
      return 0;
    }
  }
  if ($type =~ /^numeric/ || $type eq 'pidtype' || $type eq 'int') {
    return &_POP__Persistent_num_to_db($val);
  } elsif ($type eq 'datetime') {
    return &_POP__Persistent_date_to_db($val);
  } elsif ($type =~ /^(?:var)?char\((\d+)\)$/) {
    return &_POP__Persistent_char_to_db($val, $1);
  } elsif ($type eq 'text') {
    return &_POP__Persistent_text_to_db($val);
  } elsif ($type eq 'bit') {
    return &_POP__Persistent_bit_to_db($val);
  } else {
    croak "unknown type [$type]";
  }
  $val;
}

sub _POP__Persistent_char_from_db {
  my($val) = @_;
  if (defined($val)) {
    $val =~ s/^\s+//;
    $val =~ s/\s+$//;
  }
  $val;
}

sub _POP__Persistent_char_to_db {
  my($val, $width) = @_;
  if (!defined($val) or $val eq '') {
    return "NULL";
  }
  if (length($val) > $width) {
    # XXX Do we want a warning here?
    substr($val, $width) = '';
  }
  $val =~ s/"/""/g;
  qq,"$val",;
}

sub _POP__Persistent_text_to_db {
  my($val) = @_;
  if (!defined($val) or $val eq '') {
    return "NULL";
  }
  $val =~ s/"/""/g;
  qq,"$val",;
}

sub _POP__Persistent_bit_to_db {
  my($val) = @_;
  return $val ? '1' : '0';
}

sub _POP__Persistent_num_to_db {
  my($val) = @_;
  if (!defined($val)) {
    return "NULL";
  } 
  0+$val;
}

sub _POP__Persistent_date_from_db {
  my($val) = @_;
  qq,"$val",;
}

sub _POP__Persistent_date_to_db {
  my($val) = @_;
  $val;
}

$VERSION = $VERSION;
