use v6;

unit package HTML::BoreDOM;

my class Element {
	# HTML tag, e.g. "div"
	has $.tag is rw;

	# HTML attributes, e.g. { color => "red" }
	has $.attrs is rw = {};

	# Children can be either:
	#  other nested elements
	#  or raw HTML strings
	has $.children is rw = [];

	sub escape (Str $html is copy --> Str) {
		$html .= subst("&", "&amp;", :g);
		$html .= subst("<", "&lt;", :g);
		$html .= subst(">", "&gt;", :g);
		return $html;
	}

	# The opening tag e.g. <h1>
	method open {
		my @parts = $.tag.fmt("%s"),
		    |$.attrs.kv.map({ qq{$^a="$^b"} }).sort;

		return @parts.join(" ").fmt: "<%s>";
	}

	# The closing tag e.g. </h1>
	method close {
		$.tag.fmt: "</%s>"
	}

	# The string "operator" is overloaded recursively on every element
	# i.e. the element will be stringified along with all its descendants
	method Str { render(self) }
	multi sub render (Str $html --> Str) { escape $html }
	multi sub render (::?CLASS $element --> Str) {
		~ $element.open
		~ ($element.children.grep(* ~~ ::?CLASS) && "\n")
		~ $element.children.map(&render).join
		~ $element.close
		~ "\n"
	}
}

#
# Very basic helper function for writing HTML in Perl 6
#
# These characters will be escaped in raw HTML (but not in tags/attributes):
#  & --> &amp;
#  < --> &lt;
#  > --> &gt;
#
# For example, you can do:
# h("html",
#  h("body",
#   h("h1", "Hello, World!"),
#   h("p", "Lorem ipsum...")))
#
# Which stringifies to:
# <html>
# <body>
# <h1>Hello, World!</h1>
# <p>Lorem ipsum...</p>
# </body>
# </html>
#
sub h (Str $tag, *@children, *%attrs) is export {
	Element.new:
	    tag => $tag,
	    attrs => %attrs,
	    children => @children;
}
