package Promises6::Deferred;
use Evo::Base -base;
use Promises6::Util ':all';

# [state, val]
has [qw(result builder)];
has subscribers => sub { [] };
has 'is_adaptor';

# public methods -----------
sub promise($self) { $self->builder->promise(deferred => $self) }

sub resolve($self, $val) {
  return if $self->is_adaptor;
  $self->builder->resolver->resolve($self, $val);
  $self;
}

# don't try to resolve
sub reject($self, $reason) {
  return if $self->is_adaptor;
  $self->change_state(REJECTED, $reason);
  $self;
}

sub notify($self, $val) {
  $self->state eq PENDING && $self->broadcast(PROGRESS, $val);
}

# implementation methods -----------
sub state($self) { $self->result or return PENDING; $self->result->[0] }

sub change_state($self, $state, $val) {
  return unless $self->state == PENDING;
  $self->result([$state, $val])->broadcast($state, $val);

  # prevent future errors, don't do delete $self->{subscribers}
  $self->{subscribers} = 0;
  $self;
}

sub broadcast($self, $state, $val) {
  $self->send_msg($_, $state, $val) for $self->subscribers->@*;
}


# you can use loop here
sub send_msg($self, $listener, $state, $val) {
  $listener->get_msg($state, $val);
}

sub subscribe($self, $listener) {
  return push $self->subscribers->@*, $listener if $self->state eq PENDING;

  # already resolved
  $self->send_msg($listener, $self->result->@*);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Promises6::Deferred

=head1 VERSION

version 0.007

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Promises6::Builder;
  use Promises6::Deferred;

  my $bldr = Promises6::Builder->new(deferred_class => 'Promises6::Deferred');
  my $deferred = $bldr->deferred;
  my $promise  = $deferred->promise;

  $promise->then(
    sub($v) { say "resolved $v" },
    sub($r) { say "rejected $r" },
    sub($p) { say "Progress $p" }
  );

  $deferred->notify(10);
  $deferred->notify(20);
  $deferred->resolve('val');

=head1 DESCRIPTION

  This object represents a state for the promise an a links to other promises,
  created by L<Promises6/"then"> method by sending messages to L</"subscribers">
  when the state changes

  Promises/A+ says that than could be called several times on the same promise.
  You can consider that it's a sheperd of the flock of promises that have
  been called with then on the same parent promise.

  So each C<promise> has only one C<deferred>, while each C<deferred> can have
  one or many C<promises>

  Not all method should be used directly.

=head1 ATTRIBUTES

=head2 result, is_adaptor, subscribers

  Internal attributes

=head2 builder

An instance of L<Promises6::Builder>

=head1 METHODS

=head2 promise

  Builds a promise associated with this object

=head2 resolve

  $d->resolve('value')

Starts the resolution procedure L<https://github.com/promises-aplus/promises-spec#the-promise-resolution-procedure>
If value were provided, fulfills all connected promises with that value
invoking L</"change_state"> (which sends messages about that to all
subscribers)

If value is a thenable object or promise, it will follow it.

=head2 reject

  $deferred->reject('reason');

Rejects all connected promises with the reason and send messages to
subsribers. If value is a thenable object or a promise, it WON'T
follow it, because this method changes a state immidiately. So rejecting
promises with other promises is a mistake.

=head2 notify

Notify subscribers about the progress or something else. Can be called many
times before deferred object is resolved

  $deferred->notify(10);
  $deferred->notify(20);

It's not a part of the promises/A+. The behaviour is almast the same as JS Q.
If promise is PENDING, all subscribers will be notified with given value by
calling 3rd argument of L<then>. This invocation should return a value,
which will be passed to the next member of chain.

The difference with JS Q is if C<on_notify> causes an error, the promises
will be rejected but this could be changed in the future.
(Q does nothing with that but I think it's a mistake in implementation)

=head2 state

  Returns current state of deferred object.

=head2 change_state

  Immidiately change a state.
  This is an internal method and shouldn't be used directly

=head2 broadcast

  Send message to all subscribers
  This is an internal method and shouldn't be used directly

=head2 send_msg

  Sends a message to the listener. This method should be subclassed to run
  out from recursions and avoid "deep recursion" problems by sending a message
  in the "next tick".
  This is an internal method and shouldn't be used directly

=head2 subscribe

  Subscribe a listener, by adding it to the subscribers list
  This is an internal method and shouldn't be used directly

=head1 AUTHOR

alexbyk <alexbyk.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
