package Evo::Comp::Role::Exporter;
use Evo;
use Evo::Util;
use Carp 'croak';
use Module::Load;


sub data { shift->{data} }
sub new { bless {data => {}}, __PACKAGE__ }

sub init_slot {
  my ($self, $src, $name, $val) = @_;
  croak "$src already adds $name" if $self->data->{$src}{export}{$name};
  $self->data->{$src}{export}{$name} = $val;
}

sub add_methods {
  my ($self, $role, @methods) = @_;
  no strict 'refs';    ## no critic
  foreach my $m (@methods) {
    my $full = "${role}::$m";
    my $sub = *{$full}{CODE} or croak "Method $full doesn't exists";
    $self->init_slot($role, $m, {type => 'method', value => $sub});
  }
}

sub add_gen {
  my ($self, $role, $name, $gen) = @_;
  $self->init_slot($role, $name, {type => 'gen', value => $gen});
}

sub request_gen {
  my ($self, $role, $name, $comp) = @_;
  my $slot = $self->data->{$role}{export}{$name}
    or croak qq#"$role" doesn't provide method "$name"#;

  # it's important to return same function for the same module
  return $slot->{cache}{$comp} if $slot->{cache}{$comp};
  return $slot->{cache}{$comp} = $slot->{value}->($comp);
}

sub add_attr {
  my ($self, $role, $name, @opts) = @_;
  Evo::Util::check_subname($name) || croak(qq{Attribute "$name" invalid});
  $self->init_slot($role, $name, {type => 'attr', value => \@opts});
}

sub _map {
  my ($self, $type, $role) = @_;
  my $all = $self->data->{$role}{export};
  map { ($_, $all->{$_}->{value}) } grep { $all->{$_}->{type} eq $type } keys %$all;
}

sub methods {
  my ($self, $role, $comp) = @_;
  my %gens = $self->_map('gen', $role);
  my %gms = map { $_ => $gens{$_}->($comp) } keys %gens;
  (%gms, $self->_map('method', $role),);
}

sub attrs { shift->_map('attr', shift); }

sub hooks {
  my ($self, $role, @hooks) = @_;

  return $self->data->{$role}
    && $self->data->{$role}{hooks} ? ($self->data->{$role}{hooks}->@*) : ()
    if !@hooks;
  push @{($self->data->{$role}{hooks} ||= [])}, @hooks;
}

sub proxy {
  my ($self, $dst, $src) = @_;
  load $src;

  my $data_src = $self->data->{$src} || {};
  my $add_src  = $data_src->{export} || {};
  my $data_dst = $self->data->{$dst} ||= {};
  my $add_dst  = $data_dst->{export} ||= {};


  exists $add_dst->{$_} and croak qq#"$_" in "$src" clashes with "$dst"# for keys %$add_src;

  $add_dst->{$_} = $add_src->{$_} for keys %$add_src;

  my @hooks = $self->hooks($src);
  $self->hooks($dst, @hooks) if @hooks;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo::Comp::Role::Exporter

=head1 VERSION

version 0.0173

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
