package Echo::StreamServer::Items;

use 5.008008;
use strict;
use warnings;

use Echo::StreamServer::Client;
our @ISA = ( 'Echo::StreamServer::Client' );

use Echo::StreamServer::MuxRequest;

use Echo::StreamServer::Core;
$Echo::StreamServer::Core::DEBUG=1;

use Data::Dumper;
our $DEBUG=1;

our $VERSION = '0.01';

sub count {
	my ($self, $query) = @_;

	my %params = (
                'appkey' => $self->{'account'}->{'appkey'},
		'q' => $query,
	);

	my $json_hash_ref = send_request($self->{'account'}, 'count', \%params);
	if ($DEBUG) { print STDERR "count returned: " . Dumper($json_hash_ref) . "\n"; }

        # Return count or zero when impossible.
        if (exists($json_hash_ref->{'count'})) {
        	return $json_hash_ref->{'count'};
        }
        else {
                return 0;
        }
}

sub search {
	my ($self, $query, $since) = @_;

	my %params = (
                'appkey' => $self->{'account'}->{'appkey'},
		'q' => $query,
	);
	if (defined $since) {
		$params{'since'} = $since;
	}

	my $json_hash_ref = send_request($self->{'account'}, 'search', \%params);

        # Return results data structure.
        return $json_hash_ref;
}

sub submit {
	my ($self, $content, $mode) = @_;

	my %params = (
		'content' => $content, # ActivityStreams XML
		'mode' => (defined $mode)? $mode: 'update',
	);

	my $json_hash_ref = send_request($self->{'account'}, 'submit', \%params, 'post');

        # Return results data structure.
        return $json_hash_ref;
}

sub mux {
	my ($self, @mux_requests) = @_;
	my $mux_json = Echo::StreamServer::MuxRequest::requests_json(@mux_requests);

	my %params = (
                'appkey' => $self->{'account'}->{'appkey'},
		'requests' => $mux_json,
	);

	my $json_hash_ref = send_request($self->{'account'}, 'mux', \%params, 'post');

        # Return results data structure.
        return $json_hash_ref;
}

1;
__END__

=head1 NAME

Echo::StreamServer::Items - Items API

=head1 SYNOPSIS

  use Echo::StreamServer::Account;
  use Echo::StreamServer::Items;
  use Echo::StreamServer::MuxRequest;

  my $acct = new Echo::StreamServer::Account($appkey, $secret);
  my $client = new Echo::StreamServer::Items($acct);

  # Count EQL Query: Trap StreamServer exceptions.
  eval {
    my $cnt = $client->count("scope:http://www.example.com/*");
  };
  print STDERR "Query Count Failed: $@\n" if ($@);

  # Search EQL Query:
  my $results = $client->search("scope:http://www.example.com/*");
  my @entries = @{$results->{'entries'}};

  # Submit Activity Streams XML document to Echo.
  my $results = $client->submit($activity_streams_xml);

  # Send @mux_requests to Echo.
  my $client = new Echo::StreamServer::Items();
  my $mux_resp = $client->mux(@mux_requests);

=head1 DESCRIPTION

The Items API is Echo::StreamServer::Items and requires an Echo::StreamServer::Account.

Echo Items API is designed for providing interface for working with items in Activity Streams format.
Items API is the core component of the Platform, which allows you to submit items and after that build
the various data sets using the Echo Query Language.

=head2 Client Methods

=over

=item C<count>

Count the number of items within a dataset generated by a certain query, using Echo Query Language.

=item C<search>

Search for items using the Echo Query Language.

=item C<submit>

Submit items C<$content> in Activity Streams XML format.

=item C<mux>

This single API call can C<multiplex> items C<search> and C<count> requests.
C<@mux_requests> is a list of Echo::StreamServer::MuxRequest objects.

=back

=head1 SEE ALSO

Echo::StreamServer::Account
Echo::StreamServer::MuxRequest

=head1 AUTHOR

Andrew Droffner, E<lt>adroffne@advance.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2012 by Andrew Droffner

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
