package Pcore::Util::GeoIP;

use Pcore qw[-autoload];
use Const::Fast qw[];

Const::Fast::const our $GEOIP_STANDARD     => 0;    # PP
Const::Fast::const our $GEOIP_MEMORY_CACHE => 1;    # PP
Const::Fast::const our $GEOIP_CHECK_CACHE  => 2;    # when using memory cache you can force a reload if the file is updated by setting GEOIP_CHECK_CACHE
Const::Fast::const our $GEOIP_INDEX_CACHE  => 4;    # caches the most frequently accessed index portion of the database, resulting in faster lookups than GEOIP_STANDARD, but less memory usage than GEOIP_MEMORY_CACHE - useful for larger databases such as GeoIP Legacy Organization and GeoIP City. Note, for GeoIP Country, Region and Netspeed databases, GEOIP_INDEX_CACHE is equivalent to GEOIP_MEMORY_CACHE

our $GEOIP_PURE_PERL          = 0;                  # force to use pure perl mode
our $GEOIP_COUNTRY_CACHE_MODE = $GEOIP_MEMORY_CACHE;
our $GEOIP_CITY_CACHE_MODE    = $GEOIP_INDEX_CACHE;

our $GEOIP_COUNTRY_PATH    = P->res->get_local('geoip_country.dat');
our $GEOIP_COUNTRY_PATH_V6 = P->res->get_local('geoip_country_v6.dat');
our $GEOIP_CITY_PATH       = P->res->get_local('geoip_city.dat');
our $GEOIP_CITY_PATH_V6    = P->res->get_local('geoip_city_v6.dat');

my $H;

sub country_path {
    my $self = shift;

    return $GEOIP_COUNTRY_PATH;
}

sub city_path {
    my $self = shift;

    return $GEOIP_CITY_PATH;
}

sub _get_h {
    my $self = shift;
    my $type = shift;

    if ( !defined $H->{$type} ) {
        my $db_path;
        my $default_cache_mode;

        my $use_pure_perl = $GEOIP_PURE_PERL;

        if ($use_pure_perl) {
            require Geo::IP::PurePerl;    ## no critic qw(Modules::ProhibitEvilModules)
        }
        else {
            $use_pure_perl = try {
                require Geo::IP;          ## no critic qw(Modules::ProhibitEvilModules)

                return 0;
            }
            catch {
                require Geo::IP::PurePerl;    ## no critic qw(Modules::ProhibitEvilModules)

                return 1;
            };
        }

        if ( $type eq 'country' ) {
            $db_path = $GEOIP_COUNTRY_PATH;

            $default_cache_mode = $GEOIP_COUNTRY_CACHE_MODE;
        }
        elsif ( $type eq 'country_v6' ) {
            $db_path = $GEOIP_COUNTRY_PATH_V6;

            $default_cache_mode = $GEOIP_COUNTRY_CACHE_MODE;
        }
        elsif ( $type eq 'city' ) {
            $db_path = $GEOIP_CITY_PATH;

            $default_cache_mode = $GEOIP_CITY_CACHE_MODE;
        }
        elsif ( $type eq 'city_v6' ) {
            $db_path = $GEOIP_CITY_PATH_V6;

            $default_cache_mode = $GEOIP_CITY_CACHE_MODE;
        }

        # use $GEOIP_MEMORY_CACHE instead of $GEOIP_INDEX_CACHE if $GEOIP_INDEX_CACHE is not supported
        my $flags = $use_pure_perl && $default_cache_mode == $GEOIP_INDEX_CACHE ? $GEOIP_MEMORY_CACHE : $default_cache_mode;

        $flags = $flags | $GEOIP_CHECK_CACHE if !$use_pure_perl;

        my $class = $use_pure_perl ? 'Geo::IP::PurePerl' : 'Geo::IP';

        $H->{$type} = $class->open( $db_path, $flags );
    }

    return $H->{$type};
}

sub reconnect {
    my $self = shift;

    undef $H;

    return;
}

sub update {
    my $self = shift;

    require IO::Uncompress::Gunzip;

    $H = undef;

    my $cv = AE::cv;

    # geoip country
    P->ua->request(
        'http://geolite.maxmind.com/download/geoip/database/GeoLiteCountry/GeoIP.dat.gz',
        chunk_size  => 1,
        on_progress => 1,
        blocking    => $cv,
        on_finish   => sub ($res) {
            IO::Uncompress::Gunzip::gunzip( $res->body, $GEOIP_COUNTRY_PATH->to_string, BinModeOut => 1 ) if $res->status == 200;

            return;
        }
    );

    # geoip country v6
    P->ua->request(
        'http://geolite.maxmind.com/download/geoip/database/GeoIPv6.dat.gz',
        chunk_size  => 1,
        on_progress => 1,
        blocking    => $cv,
        on_finish   => sub ($res) {
            IO::Uncompress::Gunzip::gunzip( $res->body->path, $GEOIP_COUNTRY_PATH_V6->to_string, BinModeOut => 1 ) if $res->status == 200;

            return;
        }
    );

    # geoip city
    P->ua->request(
        'http://geolite.maxmind.com/download/geoip/database/GeoLiteCity.dat.gz',
        chunk_size  => 1,
        on_progress => 1,
        blocking    => $cv,
        on_finish   => sub ($res) {
            IO::Uncompress::Gunzip::gunzip( $res->body->path, $GEOIP_CITY_PATH->to_string, BinModeOut => 1 ) if $res->status == 200;

            return;
        }
    );

    # geoip city v6
    P->ua->request(
        'http://geolite.maxmind.com/download/geoip/database/GeoLiteCityv6-beta/GeoLiteCityv6.dat.gz',
        chunk_size  => 1,
        on_progress => 1,
        blocking    => $cv,
        on_finish   => sub ($res) {
            IO::Uncompress::Gunzip::gunzip( $res->body->path, $GEOIP_CITY_PATH_V6->to_string, BinModeOut => 1 ) if $res->status == 200;

            return;
        }
    );

    $cv->recv;

    # $self->_connect;

    return;
}

sub autoload {
    my $self   = shift;
    my $method = shift;

    return sub {
        my $self = shift;

        return $self->_get_h('country')->$method(@_);
    };
}

# city methods
sub record_by_addr {
    my $self = shift;

    return $self->_get_h('city')->record_by_addr(@_);
}

sub record_by_name {
    my $self = shift;

    return $self->_get_h('city')->record_by_name(@_);
}

1;
__END__
