#!/usr/bin/env perl

package main v0.1.0;

use Pcore;
use Pcore::Util::URI::Host;
use Pcore::Whois;

our $TEST_DATA_PATH = 'test-data.perl';

our $TEST_DATA = -e $TEST_DATA_PATH ? P->cfg->load($TEST_DATA_PATH) : {};

our $NOT_AVAIL_DOMAIN = 'cwxtwtqcuduvo78hr0dbrfxpuhv8gmtm';

our $WHOIS = Pcore::Whois->new;

if ( $ARGV{command} eq 'test' ) {
    cmd_test();
}
elsif ( $ARGV{command} eq 'supported' ) {
    cmd_supported();
}
elsif ( $ARGV{command} eq 'unsupported' ) {
    cmd_unsupported();
}
elsif ( $ARGV{command} eq 'uncovered' ) {
    cmd_uncovered();
}
else {
    say qq[Unknown command "$ARGV{command}"];
}

sub cmd_test {
    my $cv = AE::cv;

    my $error = {};

    my $total_requests = 0;

    my $progress;

    if ( !$ARGV{no_progress} ) {
        for my $tld ( keys $TEST_DATA->%* ) {
            $total_requests++;

            $total_requests += scalar $TEST_DATA->{$tld}->@*;
        }

        $progress = P->progress->get_indicator( total => $total_requests );
    }

    my $done_requests = 0;

    my $request = sub ( $domain, $tld, $is_registered ) {
        $cv->begin;

        $WHOIS->whois(
            $domain,
            sub($res) {
                $progress->update( value => ++$done_requests ) if $progress;

                if ( $res->is_success ) {
                    my $parsing_error;

                    if ( $is_registered && !$res->is_found ) {
                        $parsing_error = 1;

                        $error->{PARSING_ERROR}->{$tld}->{$domain} = 1;
                    }
                    elsif ( !$is_registered && $res->is_found ) {
                        $parsing_error = 1;

                        $error->{PARSING_ERROR}->{$tld}->{$domain} = 1;
                    }

                    if ( $parsing_error && $ARGV{interactive} ) {
                        say 'DOMAIN:' . $res->query;

                        say 'WHOIS SERVER: ' . $res->server->host;

                        say qq[WHOIS SERVER RESPONSE CONTENT:\n];

                        say $res->content;

                        print 'Press ENTER to continue...';
                        <STDIN>;

                        say q[-] x 50;
                    }
                }
                else {
                    if ( $res->server ) {
                        $error->{CONNECTION_ERROR}->{$tld} = $res->server->host;
                    }
                    else {
                        $error->{NOT_SUPPORTED_TLD}->{$tld} = 1;
                    }
                }

                $cv->end;

                return;
            }
        );

        return;
    };

    for my $tld ( keys $TEST_DATA->%* ) {
        for my $domain ( $TEST_DATA->{$tld}->@* ) {
            $request->( $domain, $tld, 1 );
        }

        # unregistered domain
        $request->( $NOT_AVAIL_DOMAIN . q[.] . $tld, $tld, 0 );
    }

    $cv->recv;

    # print CONNECTION ERROR report
    say 'CONNECTION ERROR:';

    for my $tld ( sort keys $error->{CONNECTION_ERROR}->%* ) {
        say qq[$tld = $error->{CONNECTION_ERROR}->{$tld}];
    }

    say q[-] x 50;

    # print PARSING ERROR report
    say 'PARSING ERROR:';

    for my $tld ( sort keys $error->{PARSING_ERROR}->%* ) {
        say $tld;

        for my $domain ( sort keys $error->{PARSING_ERROR}->{$tld}->%* ) {
            say q[ ] x 4 . $domain;
        }
    }

    say q[-] x 50;

    # print UNSUPPORTED TLDs report
    say 'UNSUPPORTED TLDs:';

    for my $tld ( sort keys $error->{NOT_SUPPORTED_TLD}->%* ) {
        say $tld;
    }

    return;
}

sub cmd_supported {
    my $supported = get_supported_tlds();

    for my $tld ( sort keys $supported->%* ) {
        say qq[$tld = $supported->{$tld}];
    }

    return;
}

sub cmd_unsupported {
    my $unsupported = get_unsupported_tlds();

    for my $tld ( sort keys $unsupported->%* ) {
        say $tld;
    }

    return;
}

sub cmd_uncovered {
    for my $tld ( sort keys Pcore::Util::URI::Host->tlds->%* ) {
        if ( !exists $TEST_DATA->{$tld} || !scalar $TEST_DATA->{$tld}->@* ) {
            say $tld;
        }
    }

    return;
}

sub get_supported_tlds {
    my $supported = {};

    my $cv = AE::cv;

    for my $tld ( keys Pcore::Util::URI::Host->tlds->%* ) {
        $cv->begin;

        $WHOIS->get_tld_server(
            $tld,
            sub($server = undef) {
                $cv->end;

                $supported->{$tld} = $server if $server;

                return;
            }
        );
    }

    $cv->recv;

    return $supported;
}

sub get_unsupported_tlds {
    my $unsupported = {};

    my $cv = AE::cv;

    for my $tld ( keys Pcore::Util::URI::Host->tlds->%* ) {
        $cv->begin;

        $WHOIS->get_tld_server(
            $tld,
            sub($server = undef) {
                $cv->end;

                $unsupported->{$tld} = 1 if !$server;

                return;
            }
        );
    }

    $cv->recv;

    return $unsupported;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 43, 109, 123, 132,   │ References::ProhibitDoubleSigils - Double-sigil dereference                                                    │
## │      │ 135, 145, 155, 165,  │                                                                                                                │
## │      │ 173, 187, 212        │                                                                                                                │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 REQUIRED ARGUMENTS

=over

=item <command>

Available commands:

    test           - perform tests
    supported      - print list of supported TLDs
    unsupported    - print list of unsupported TLDs
    uncovered      - print list of TLDs, uncovered by tests

=back

=head1 OPTIONS

=over

=item -q | --no-progress

Do not show progress bar during testing.

=item -i | --interactive

Interactive mode for testing.

=back

=cut
