package Pcore::App::Nginx;

use Pcore qw[-cli -class];

extends qw[Pcore::App::Alien];

has '+name' => ( default => 'nginx' );
has '+ns'   => ( default => 'Pcore::Nginx' );

has nginx_ver => ( is => 'lazy', isa => Str, init_arg => undef );

our $CFG = {
    NGINX => {    #
        NGINX_VER => undef,
    },
};

sub _build_nginx_ver {
    my $self = shift;

    return $self->cfg->{NGINX}->{NGINX_VER} || $DIST->{NGINX}->{NGINX_VER};
}

# APP
around _build_cfg => sub {
    my $orig = shift;
    my $self = shift;

    return P->hash->merge( $self->$orig, $CFG );
};

around _create_local_cfg => sub {
    my $orig = shift;
    my $self = shift;

    my $local_cfg = {};

    return P->hash->merge( $self->$orig, $CFG, $local_cfg );
};

sub _build_alien_dir {
    my $self = shift;

    return $self->app_dir . 'nginx-' . $self->nginx_ver . q[/];
}

sub _build_alien_bin_path {
    my $self = shift;

    return $self->alien_dir . 'sbin/nginx';
}

sub _build_alien_cfg_path {
    my $self = shift;

    return $self->app_dir . 'nginx.conf';
}

around app_build => sub {
    my $orig = shift;
    my $self = shift;

    $self->$orig;

    my $module = {
        nginx_http_auth_request_module => 0,
        nginx_echo_module              => 0,
        nginx_memc_module              => 0,
        nginx_headers_more_module      => 0,
        nginx_substitutions_module     => 0,
        nginx_upload_progress_module   => 0,
        nginx_push_stream_module       => 0,
        nginx_psgi_module              => 0,
        nginx_upload_module            => 0,
    };

    my $module_temp_dir = sub {
        my $module_dir = shift;

        P->file->mkpath( $PROC->{TEMP_DIR} . $module_dir );

        return $PROC->{TEMP_DIR} . $module_dir . q[/];
    };

    if ( -d $self->alien_dir ) {
        $self->_appx_report_warn( q["] . $self->alien_dir . q[" already exists. Remove it manually to rebuild] );
    }
    else {
        my $res = try {

            # pcre
            {
                my $pcre_dir = $module_temp_dir->('pcre');

                # P->sys->system(qq[wget -O - ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre2-$DIST->{NGINX}->{PCRE_VER}.tar.gz | tar -C $pcre_dir --strip-components=1 -xzvf -]);
                P->sys->system(qq[wget -O - ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-$DIST->{NGINX}->{PCRE_VER}.tar.gz | tar -C $pcre_dir --strip-components=1 -xzvf -]);
            }

            # nginx_http_auth_request_module
            if ( $module->{nginx_http_auth_request_module} ) {
                my $nginx_http_auth_request_module_dir = $module_temp_dir->('nginx_http_auth_request_module');
                P->sys->system(qq[wget -O - http://mdounin.ru/hg/ngx_http_auth_request_module/archive/tip.tar.gz | tar -C $nginx_http_auth_request_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx_echo_module
            if ( $module->{nginx_echo_module} ) {
                my $nginx_echo_module_dir = $module_temp_dir->('nginx_echo_module');
                P->sys->system(qq[wget --no-check-certificate -O - https://github.com/agentzh/echo-nginx-module/archive/master.tar.gz | tar -C $nginx_echo_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx_memc_module
            if ( $module->{nginx_memc_module} ) {
                my $nginx_memc_module_dir = $module_temp_dir->(q[nginx_memc_module]);
                P->sys->system(qq[wget --no-check-certificate -O - https://github.com/agentzh/memc-nginx-module/archive/master.tar.gz | tar -C $nginx_memc_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx_headers_more_module
            if ( $module->{nginx_headers_more_module} ) {
                my $nginx_headers_more_module_dir = $module_temp_dir->(q[nginx_headers_more_module]);
                P->sys->system(qq[wget --no-check-certificate -O - https://github.com/agentzh/headers-more-nginx-module/archive/master.tar.gz | tar -C $nginx_headers_more_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx_substitutions_module
            if ( $module->{nginx_substitutions_module} ) {
                my $nginx_substitutions_module_dir = $module_temp_dir->(q[nginx_substitutions_module]);
                P->sys->system(qq[svn checkout http://substitutions4nginx.googlecode.com/svn/trunk/ $nginx_substitutions_module_dir]);
            }

            # nginx_upload_progress_module
            if ( $module->{nginx_upload_progress_module} ) {
                my $nginx_upload_progress_module_dir = $module_temp_dir->(q[nginx_upload_progress_module]);
                P->sys->system(qq[wget --no-check-certificate -O - https://github.com/masterzen/nginx-upload-progress-module/archive/master.tar.gz | tar -C $nginx_upload_progress_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx_push_stream_module
            if ( $module->{nginx_push_stream_module} ) {
                my $nginx_push_stream_module_dir = $module_temp_dir->(q[nginx_push_stream_module]);
                P->sys->system(qq[wget --no-check-certificate -O - https://github.com/wandenberg/nginx-push-stream-module/archive/master.tar.gz | tar -C $nginx_push_stream_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx_psgi_module
            if ( $module->{nginx_psgi_module} ) {
                my $nginx_psgi_module_dir = $module_temp_dir->(q[nginx_psgi_module]);
                P->sys->system(qq[wget --no-check-certificate -O - https://github.com/yko/ngx_mod_psgi/archive/master.tar.gz | tar -C $nginx_psgi_module_dir --strip-components=1 -xzvf -]);
            }

            # nginx itself
            my $nginx_dir = $module_temp_dir->(q[nginx]);
            P->sys->system( q[wget --no-check-certificate -O - http://nginx.org/download/nginx-] . $self->nginx_ver . qq[.tar.gz | tar -C $nginx_dir --strip-components=1 -xzvf -] );

            {
                my $chdir_guard = P->file->chdir($nginx_dir) or die;

                my @params = (
                    q[--with-http_ssl_module],    #
                    q[--with-http_spdy_module],
                    q[--with-http_geoip_module],
                    q[--with-http_realip_module],
                    q[--with-http_addition_module],
                    q[--with-http_sub_module],
                    q[--with-http_gzip_static_module],
                    q[--with-http_perl_module],
                    q[--with-pcre=] . $module_temp_dir->(q[pcre]),
                    q[--with-pcre-jit],
                    map { q[--add-module=] . $module_temp_dir->($_) } grep { $module->{$_} } keys %{$module},
                );
                my $cmd = q[./configure --prefix=] . $self->alien_dir . q[ --with-ld-opt="-Wl,-rpath,/usr/local/lib64 -L /usr/local/lib64" ] . join q[ ], @params;

                P->sys->system($cmd);
                P->sys->system( 'make', q[-j] . P->sys->cpus_num );
                P->sys->system(qw[make install]);
            }

            return 1;
        }
        catch {
            my $e = shift;

            return 0;
        };

        unless ($res) {
            P->file->rmtree( $self->alien_dir );

            $self->_appx_report_fatal(qq[Error building application "$@"]);
        }
    }

    P->file->mkpath( $self->app_dir . 'vhosts' );    # create vhosts directory

    return;
};

around app_deploy => sub {
    my $orig = shift;
    my $self = shift;

    $self->$orig;

    return;
};

around app_test => sub {
    my $orig = shift;
    my $self = shift;

    $self->$orig;

    say 'Config test';

    $self->generate_alien_cfg;

    my $res = P->capture->sys( $self->alien_bin_path, '-t', '-c', $self->alien_cfg_path );

    if ( $res =~ /test\hfailed/sm ) {
        say 'FAILED';

        say $res;
    }
    else {
        say 'OK';
    }

    return;
};

sub generate_alien_cfg {
    my $self = shift;

    my $cfg;

    # TODO PCORE-19

    my $params = {
        app_name           => $self->name,
        app_dir            => $self->app_dir,
        alien_dir          => $self->alien_dir,
        log_dir            => $PROC->{LOG_DIR},
        vhosts_dir         => $self->app_dir . 'vhosts',
        geoip_country_path => P->geoip->country_path,
        geoip_city_path    => P->geoip->city_path,
        ssl                => 0,                           # $self->openssl->is_enabled, # TODO
    };

    $cfg = P->tmpl( type => 'text' )->render( 'nginx/server.nginx', $params );

    $self->store_alien_cfg($cfg);

    return;
}

sub master_proc {
    my $self = shift;

    # плавное завершение
    $SIG->{QUIT} = AE::signal QUIT => sub {
        debug 'SIGQUIT received';

        $self->term_state(1);

        kill 'QUIT', $self->alien_pid;    ## no critic qw(InputOutput::RequireCheckedSyscalls)
    };

    # изменение конфигурации, обновление изменившейся временной зоны (только для FreeBSD и Linux), запуск новых рабочих процессов с новой конфигурацией, плавное завершение старых рабочих процессов
    $SIG->{HUP} = AE::signal HUP => sub {
        debug 'SIGHUP received';

        kill 'HUP', $self->alien_pid;     ## no critic qw(InputOutput::RequireCheckedSyscalls)
    };

    # переоткрытие лог-файлов
    $SIG->{USR1} = AE::signal USR1 => sub {
        debug 'SIGUSR1 received';

        kill 'USR1', $self->alien_pid;    ## no critic qw(InputOutput::RequireCheckedSyscalls)
    };

    # обновление исполняемого файла
    $SIG->{USR2} = AE::signal USR2 => sub {
        debug 'SIGUSR2 received';

        kill 'USR2', $self->alien_pid;    ## no critic qw(InputOutput::RequireCheckedSyscalls)
    };

    # плавное завершение рабочих процессов
    $SIG->{WINCH} = AE::signal WINCH => sub {
        debug 'SIGWINCH received';

        kill 'WINCH', $self->alien_pid;    ## no critic qw(InputOutput::RequireCheckedSyscalls)
    };

    return;
}

sub alien_proc {
    my $self = shift;

    $self->generate_alien_cfg;

    exec $self->alien_bin_path, q[-c], $self->alien_cfg_path or die;
}

1;
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::App::Nginx - pcore nginx application

=head1 SYNOPSIS

    use Pcore::App::Nginx;

    Pcore::App::Nginx->new->run;

=head1 DESCRIPTION

Additional signals supported:

    QUIT  - graceful shutdown (SIGQUIT)
    HUP   - graceful reloading worker processes, reload config (SIGHUP)
    USR1  - re-opening log files (SIGUSR1)
    USR2  - upgrading an executable file (SIGUSR2)
    WINCH - graceful shutdown of worker processes (SIGWINCH)

=cut
