package Pcore::DBD v0.4.0;

use Pcore -dist, -role;
use Pcore::Util::Text qw[escape_scalar];
use DBI;

with qw[Pcore::Core::H::Role::Wrapper];

requires qw[_build__connection];

has _connection    => ( is => 'lazy', isa => ArrayRef, init_arg => undef );
has dbd_type       => ( is => 'lazy', isa => Str,      init_arg => undef );
has _builder_class => ( is => 'lazy', isa => Str,      init_arg => undef );
has _ddl_class     => ( is => 'lazy', isa => Str,      init_arg => undef );

sub _build_dbd_type ($self) {
    ( my $type ) = ref($self) =~ /::([^:]+)\z/sm;

    return $type;
}

sub _build__builder_class ($self) {
    return P->class->load( $self->dbd_type, ns => 'Pcore::DBD::Builder' );
}

sub _build__ddl_class ($self) {
    return P->class->load( $self->dbd_type, ns => 'Pcore::DBD::DDL' );
}

# H
sub h_connect {
    my $self = shift;

    my $h = DBI->connect( $self->_connection->@* );

    return $h;
}

sub h_disconnect {
    my $self = shift;

    $self->h->disconnect;

    return;
}

# QUERY EXECUTION METHODS
sub selectall {
    my $self  = shift;
    my $query = shift;
    my %args  = @_;

    $args{slice} //= {};

    return $self->selectall_arrayref( $query, %args );
}

sub selectall_arrayref {
    my $self  = shift;
    my $query = shift;
    my %args  = (
        slice    => undef,
        max_rows => undef,
        attr     => {},
        bind     => undef,
        nocache  => 0,
        @_,
    );

    # if slice is ArrayRef - return ArrayRef[ArrayRef] with columns, specified in slice by 0-based indexes, eg: [0, 1, -1, -2]
    # if slice is HashRef - return ArrayRef[HashRef] with only columns, specified in slice HashRef, if slice is empty HashRef - returns all columns, eg: {col1 => 1, col2 => 1}
    # if slice is \HashRef - return ArrayRef[HashRef] with only columns, specifies be theirs 0-based indexes, eg: \{0 => 'col1', -5 => 'col2'}

    $args{attr}->{Slice} = $args{slice} if $args{slice};

    $args{attr}->{MaxRows} = $args{max_rows} if $args{max_rows};

    my $res = $self->h->selectall_arrayref( ref $query ? $args{nocache} ? $query->sql : $query->sth : $query, $args{attr}, $args{bind} ? $args{bind}->@* : ref $query ? $query->bind->@* : () );

    if ( $res->@* ) {
        return $res;
    }
    else {
        return;
    }
}

sub selectall_hashref {
    my $self  = shift;
    my $query = shift;
    my %args  = (
        key_cols => 0,       # ex: 1, 'col1', [1, 2, 5], ['col1', 'col5'], index is 0-based
        attr     => {},
        bind     => undef,
        nocache  => 0,
        @_,
    );

    my @key_cols = ref $args{key_cols} ? $args{key_cols}->@* : ( $args{key_cols} );

    for (@key_cols) {
        $_++ if DBI::looks_like_number($_);
    }

    my $res = $self->h->selectall_hashref( ref $query ? $args{nocache} ? $query->sql : $query->sth : $query, \@key_cols, $args{attr}, $args{bind} ? $args{bind}->@* : ref $query ? $query->bind->@* : () );

    if ( keys $res->%* ) {
        return $res;
    }
    else {
        return;
    }
}

sub selectrow {
    my $self = shift;

    return $self->selectrow_hashref(@_);
}

sub selectrow_hashref {
    my $self  = shift;
    my $query = shift;
    my %args  = (
        attr    => {},
        bind    => undef,
        nocache => 0,
        @_,
    );

    return $self->h->selectrow_hashref( ref $query ? $args{nocache} ? $query->sql : $query->sth : $query, $args{attr}, $args{bind} ? $args{bind}->@* : ref $query ? $query->bind->@* : () );
}

sub selectrow_arrayref {
    my $self  = shift;
    my $query = shift;
    my %args  = (
        attr    => {},
        bind    => undef,
        nocache => 0,
        @_,
    );

    return $self->h->selectrow_arrayref( ref $query ? $args{nocache} ? $query->sql : $query->sth : $query, $args{attr}, $args{bind} ? $args{bind}->@* : ref $query ? $query->bind->@* : () );
}

sub selectcol {
    my $self = shift;

    return $self->selectcol_arrayref(@_);
}

sub selectcol_arrayref {
    my $self  = shift;
    my $query = shift;
    my %args  = (
        cols     => undef,    # required columns indexes, index is 0-based, ex: 5, [1, 2, 5]
        max_rows => undef,    # specify max. rows to proceed
        attr     => {},
        bind     => undef,
        nocache  => 0,
        @_,
    );

    $args{attr}->{Columns} = [ map { $_ + 1 } ref $args{cols} ? $args{cols}->@* : ( $args{cols} ) ] if $args{cols};

    $args{attr}->{MaxRows} = $args{max_rows} if $args{max_rows};

    my $res = $self->h->selectcol_arrayref( ref $query ? $args{nocache} ? $query->sql : $query->sth : $query, $args{attr}, $args{bind} ? $args{bind}->@* : ref $query ? $query->bind->@* : () );

    if ( $res->@* ) {
        return $res;
    }
    else {
        return;
    }
}

sub selectval {
    my $self  = shift;
    my $query = shift;
    my %args  = (
        col     => 0,       # ex: 1, 'col1', index is 0-based
        attr    => {},
        bind    => undef,
        nocache => 0,
        @_,
    );

    if ( DBI::looks_like_number( $args{col} ) ) {
        if ( my $res = $self->selectrow_arrayref( $query, %args ) ) {
            return \$res->[ $args{col} ];
        }
    }
    else {
        if ( my $res = $self->selectrow_hashref( $query, %args ) ) {
            return \$res->{ $args{col} };
        }
    }

    return;
}

# TODO sth caching not supported for multiple queries
# add workaround for queries, that contains multiply queries
# see DBD::SQLite do implementation
sub do {    ## no critic qw[Subroutines::ProhibitBuiltinHomonyms]
    my $self  = shift;
    my $query = shift;
    my %args  = (
        attr    => {},
        bind    => undef,
        nocache => 0,
        @_,
    );

    if ( ref($query) && !$args{nocache} ) {
        $query->sth->execute( $args{bind} ? $args{bind}->@* : $query->bind->@* );

        return $query->sth->rows;
    }
    else {
        my $res = $self->h->do( ref $query ? $query->sql : $query, $args{attr}, $args{bind} ? $args{bind}->@* : ref $query ? $query->bind->@* : () );

        return $res eq '0E0' ? 0 : $res;
    }
}

# DBD
sub _default_attr {
    my $self = shift;

    my $attr = {
        Warn        => 1,
        PrintWarn   => 1,
        PrintError  => 0,
        RaiseError  => 1,
        HandleError => sub {
            my $msg = shift;

            escape_scalar $msg;

            die $msg;
        },
        ShowErrorStatement => 1,
        AutoCommit         => 1,
        Callbacks          => {
            connected => sub {
                P->log->debug( 'Connected to: ' . $_[1] );

                return;
            },
            prepare => sub {
                return;
            },
            do => sub {
                P->log->debug( 'Do: ' . $_[1] );

                return;
            },
            ChildCallbacks => {
                execute => sub {
                    P->log->debug( 'Execute: ' . $_[0]->{Statement} );

                    return;
                }
            }
        }
    };

    return $attr;
}

sub last_insert_id {
    my $self = shift;
    my %args = (
        catalog => undef,
        schema  => undef,
        table   => undef,
        field   => undef,
        attr    => undef,
        @_,
    );

    return $self->h->last_insert_id( $args{catalog}, $args{schema}, $args{table}, $args{field}, $args{attr} );
}

sub quote {
    my $self = shift;

    return $self->h->quote(@_);
}

sub quote_id {
    my $self = shift;

    if ( scalar @_ == 1 ) {
        my $res = $self->h->quote_identifier( split /[.]/sm, $_[0] );

        $res =~ s/["`'][*]["`']/*/smg;    # unquote *

        return $res;
    }
    else {
        return $self->h->quote_identifier(@_);
    }
}

# SQL BUILDER
sub query {
    my $self = shift;
    my @args = @_;

    my $class = $self->_builder_class;

    my $query = $class->new( { dbh => $self } );

    $query->_build_query(@args);

    return $query;
}

# DDL
sub ddl {
    my $self = shift;
    my $args = shift;

    $args->{dbh} = $self;

    my $class = $self->_ddl_class;

    return $class->new($args);
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 107                  │ References::ProhibitDoubleSigils - Double-sigil dereference                                                    │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    3 │ 230                  │ Subroutines::ProhibitUnusedPrivateSubroutines - Private subroutine/method '_default_attr' declared but not     │
## │      │                      │ used                                                                                                           │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::DBD

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut
