package Pcore::Chrome::Tab;

use Pcore -class, -res;
use Pcore::Util::Data qw[to_json from_json from_b64];
use Pcore::Util::Scalar qw[weaken];
use Pcore::WebSocket::raw;

use overload    #
  '&{}' => sub ( $self, @ ) {
    return sub {
        return $self->_call(@_);
    };
  },
  fallback => undef;

has chrome => ( required => 1 );
has id     => ( required => 1 );

has listen => ();    # {method => callback} hook

has network_enabled => ( 0, init_arg => undef );
has page_enabled    => ( 0, init_arg => undef );

has _ws  => ();                                              # websocket connection
has _cb  => ();                                              # { msgid => callback }
has _sem => sub { Coro::Semaphore->new(1) }, is => 'lazy';

our $_MSG_ID = 0;

sub new_tab ( $self, @args ) {
    $self->{chrome}->new_tab(@args);

    return;
}

sub close ( $self ) {                                        ## no critic qw[Subroutines::ProhibitBuiltinHomonyms NamingConventions::ProhibitAmbiguousNames]
    return P->http->get("http://$self->{chrome}->{host}:$self->{chrome}->{port}/json/close/$self->{id}");
}

sub activate ( $self ) {
    return P->http->get("http://$self->{chrome}->{host}:$self->{chrome}->{port}/json/activate/$self->{id}");
}

sub listen ( $self, $event, $cb = undef ) {                  ## no critic qw[Subroutines::ProhibitBuiltinHomonyms]
    if ($cb) {
        $self->{listen}->{$event} = $cb;
    }
    else {
        return delete $self->{listen}->{$event};
    }

    return;
}

sub _call ( $self, $method, $args = undef ) {
    my $h = $self->{_ws} // $self->_connect;

    my $id = $_MSG_ID++;

    my $cv = P->cv;

    $self->{_cb}->{$id} = $cv;

    $self->{_ws}->send_text(
        \to_json {
            id     => $id,
            method => $method,
            params => $args,
        }
    );

    return $cv->recv;
}

sub _connect ( $self ) {
    weaken $self;

    my $guard = $self->_sem->guard;

    return $self->{_ws} ||= Pcore::WebSocket::raw->connect(
        "ws://$self->{chrome}->{host}:$self->{chrome}->{port}/devtools/page/$self->{id}",
        connect_timeout  => 1000,
        max_message_size => 0,
        compression      => 0,
        on_disconnect    => sub ( $ws ) {
            return if ${^GLOBAL_PHASE} eq 'DESTRUCT';

            undef $self->{_ws};

            # call pending callbacks
            if ( my $callbacks = delete $self->{_cb} ) {
                for my $cb ( values $callbacks->%* ) {
                    $cb->( res 500 );
                }
            }

            # call pending events callbacks
            for my $cb ( values $self->{listen}->%* ) {
                $cb->( undef, undef );
            }

            return;
        },
        on_text => sub ( $ws, $data_ref ) {
            my $msg = from_json $data_ref;

            if ( exists $msg->{id} ) {
                if ( my $cb = delete $self->{_cb}->{ $msg->{id} } ) {
                    my $res;

                    if ( my $error = $msg->{error} ) {
                        $res = res [ 400, "$error->{message} $error->{data}" ], $msg->{result};
                    }
                    else {
                        $res = res 200, $msg->{result};
                    }

                    $cb->($res);
                }
            }
            elsif ( $msg->{method} ) {
                if ( my $cb = $self->{listen}->{ $msg->{method} } ) {
                    $cb->( $msg->{method}, $msg->{params} );
                }
            }
            else {
                die $msg;
            }

            return;
        },
    );
}

# NETWORK
sub network_enable ( $self ) {
    return res 200 if $self->{network_enabled};

    my $res = $self->_call('Network.enable');

    $self->{network_enabled} = 1 if $res;

    return $res;
}

sub network_disable ( $self ) {
    return res 200 if !$self->{network_enabled};

    my $res = $self->_call('Network.disable');

    $self->{network_enabled} = 0 if $res;

    return $res;
}

# PAGE
sub page_enable ( $self ) {
    return res 200 if $self->{page_enabled};

    my $res = $self->_call('Page.enable');

    $self->{page_enabled} = 1 if $res;

    return $res;
}

sub page_disable ( $self, $cb = undef ) {
    return res 200 if !$self->{page_enabled};

    my $res = $self->_call('Page.disable');

    $self->{page_enabled} = 0 if $res;

    return $res;
}

# COOKIES
sub get_cookies ( $self ) {
    my $res = $self->_call('Network.getCookies');

    my $cookies;

    if ( !$res ) {
        warn $res;
    }
    else {
        $cookies = $self->convert_cookies( $res->{data}->{cookies} );
    }

    return $cookies;
}

sub convert_cookies ( $self, $chrome_cookies ) {
    my $cookies;

    for my $cookie ( $chrome_cookies->@* ) {
        $cookies->{ $cookie->{domain} }->{ $cookie->{path} }->{ $cookie->{name} } = $cookie;

        $cookie->{val} = delete $cookie->{value};

        delete $cookie->{expires} if $cookie->{expires} && $cookie->{expires} < 0;
    }

    return $cookies;
}

# format, png, jpeg
# quality, 0 .. 100, for jpeg only
# clip, { x, y, width, height, scale }
sub get_screenshot ( $self, %args ) {
    if ( delete $args{full} ) {
        my $metrics = $self->_call('Page.getLayoutMetrics');

        my $res = $self->_call(
            'Emulation.setVisibleSize',
            {   width  => $metrics->{data}->{contentSize}->{width},
                height => $metrics->{data}->{contentSize}->{height}
            }
        );
    }

    my $res = $self->_call( 'Page.captureScreenshot', \%args, );

    my $img;

    if ($res) {
        $img = from_b64 $res->{data}->{data};
    }
    else {
        warn $res;
    }

    return $img;
}

# TODO !!! "Page.loadEventFired" is not fired
sub navigate_to ( $self, $url, %args ) {
    my $listener     = $self->{listen}->{'Page.loadEventFired'};
    my $page_enabled = $self->{page_enabled};

    my $res;

    if ( !$page_enabled ) {
        $res = $self->page_enable;

        return $res if !$res;
    }

    my $cv = P->cv;

    $self->listen( 'Page.loadEventFired', $cv );

    $res = $self->_call( 'Page.navigate', { url => $url, %args } );

    goto CLEANUP if !$res;

    $cv->recv;

  CLEANUP:
    $self->page_disable if !$page_enabled;
    $self->listen( 'Page.loadEventFired', $listener );

    return $res;
}

1;
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::Chrome::Tab

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 ATTRIBUTES

=head1 METHODS

=head1 SEE ALSO

=cut
