package Pcore::Chrome v0.1.0;

use Pcore -dist, -const, -class;
use Pcore::Chrome::Tab;
use Pcore::Util::Scalar qw[is_plain_coderef];
use Pcore::Util::Data qw[from_json];
use Pcore::WebSocket;

has host => ( is => 'ro', isa => Str, default => '127.0.0.1' );
has port => ( is => 'ro', isa => Int, default => 9222 );

has _proc => ( is => 'ro', isa => InstanceOf ['Pcore::Util:PM::Proc'], init_arg => undef );
has user_data_dir => ( is => 'ro', init_arg => undef );

const our $DEFAULT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.101 Safari/537.36';

# https://chromedevtools.github.io/devtools-protocol/tot/

sub run ( $self, @args ) {
    my $blocking_cv = defined wantarray ? AE::cv : undef;

    my $cb = is_plain_coderef $args[-1] ? pop @args : undef;

    my %args = @args;

    $args{bin} ||= $MSWIN ? q["c:/Program Files (x86)/Google/Chrome/Application/chrome.exe"] : '/usr/bin/google-chrome';

    $args{host} ||= '127.0.0.1';

    $args{port} ||= ( P->sys->get_free_port( $args{host} ) // die q[Error get free port] );

    my $useragent = !exists $args{useragent} ? $DEFAULT_USERAGENT : $args{useragent};

    $self->{user_data_dir} = P->file->tempdir;

    my $cmd = [
        $args{bin},

        "--remote-debugging-address=$args{host}",
        "--remote-debugging-port=$args{port}",

        '--disable-background-networking',
        '--disable-client-side-phishing-detection',
        '--disable-component-update',
        '--disable-default-apps',
        '--disable-hang-monitor',
        '--disable-prompt-on-repost',
        '--disable-sync',
        '--disable-web-resources',

        # '--window-size=800x600',
        '--start-maximized',
        '--disable-default-apps',
        '--no-first-run',
        '--disable-infobars',
        '--disable-popup-blocking',
        '--disable-default-apps',
        '--disable-web-security',

        "--user-data-dir=$self->{user_data_dir}",
        '--no-sandbox',    # required for run under docker

        # TODO defined $useragent ? qq[--user-agent="$useragent"] : (),

        $args{headless} || !$MSWIN ? ( '--headless', '--disable-gpu' ) : (),
        $args{args} ? $args{args}->@* : (),
    ];

    P->pm->run_proc(
        $cmd,
        win32_create_no_window => 1,
        on_ready               => sub ($proc) {
            $self->{_proc} = $proc;

            $self->{host} = $args{host};
            $self->{port} = $args{port};

            $cb->($self) if $cb;

            $blocking_cv->($self) if $blocking_cv;

            return;
        },
    );

    return $blocking_cv ? $blocking_cv->recv : ();
}

sub get_tabs ( $self, $cb ) {
    P->http->get(
        "http://$self->{host}:$self->{port}/json",
        on_finish => sub ($res) {
            my $tabs = from_json $res->body;

            for my $tab ( $tabs->@* ) {
                $tab = bless { chrome => $self, id => $tab->{id} }, 'Pcore::Chrome::Tab';
            }

            $cb->( $self, $tabs );

            return;
        }
    );

    return;
}

sub new_tab ( $self, @args ) {
    my $cb = is_plain_coderef $args[-1] ? pop @args : undef;

    my $url = $args[0] ? "?$args[0]" : q[];

    P->http->get(
        "http://$self->{host}:$self->{port}/json/new$url",
        on_finish => sub ($res) {
            my $data = from_json $res->body;

            my $tab = bless { chrome => $self, id => $data->{id} }, 'Pcore::Chrome::Tab';

            $cb->( $self, $tab );

            return;
        }
    );

    return;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## +------+----------------------+----------------------------------------------------------------------------------------------------------------+
## | Sev. | Lines                | Policy                                                                                                         |
## |======+======================+================================================================================================================|
## |    2 | 65                   | CodeLayout::ProhibitQuotedWordLists - List of quoted literal words                                             |
## +------+----------------------+----------------------------------------------------------------------------------------------------------------+
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::Chrome

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 ATTRIBUTES

=head1 METHODS

=head1 SEE ALSO

=head1 AUTHOR

zdm <zdm@softvisio.net>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by zdm.

=cut
