#!perl -T

# t/01-test.t #

#   This file is part of Software-License-OrLaterPack.
#
#   Software-License-OrLaterPack is free software: you can redistribute it
#   and/or modify it under the terms of the GNU General Public License as
#   published by the Free Software Foundation, either version 3 of the
#   License, or (at your option) any later version.
#
#   Software-License-OrLaterPack is distributed in the hope that it will be
#   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#   Public License for more details.
#
#   You should have received a copy of the GNU General Public License along
#   with Software-License-OrLaterPack.  If not, see
#   <http://www.gnu.org/licenses/>.

use strict;
use warnings;

use Test::More;

#   Licenses to test.
my %licenses = (
    AGPL_3  => 'GNU Affero General Public License',
    GPL_3   => 'GNU General Public License',
);

plan tests => keys( %licenses ) * 12;

foreach my $lic ( sort( keys( %licenses ) ) ) {

    diag( "Checking $lic..." );

    my $super = "Software::License::${lic}";   # Parent class name.
    my $class = "${super}::or_later";          # Our license class name.

    use_ok( $super ) or next;
    use_ok( $class ) or next;

    my $s = new_ok( $super => [ { holder => 'Jane Doe', year => '2007' } ] );
    my $l = new_ok( $class => [ { holder => 'John Doe', year => '2010' } ] );

    #   Make sure there is ", or any later" clause at the end of license name.
    #
    isnt( $l->name, $s->name,                                        'Name changed'              );
    is(   $l->name, "The $licenses{ $lic } version 3, or any later", 'Name ends with the clause' );

    #   Make sure there is '+' at the end of version.
    #
    isnt( $l->version, $s->version,       'Version changed'             );
    is(   $l->version, $s->version . '+', 'Version ends with plus sign' );

    #   Verify notice. Standard `Software::Licence` notice looks like
    #
    #       This software is Copyright (c) YEAR by HOLDER.
    #
    #       This is free software, licensed under:
    #
    #           license name
    #
    #   Our notice is copyright statement followed by standard GNU notice:
    #
    #       Copyright (C) YEAR HOLDER
    #
    #       This program is free software: you can redistribute it and/or modify it under the terms
    #       of the LICENSE as published by the Free Software Foundation, either version 3 of the
    #       License, or (at your option) any later version.
    #
    #       This program is distributed in the hope that it will be useful, but WITHOUT ANY
    #       WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
    #       PARTICULAR PURPOSE.  See the LICENSE for more details.
    #
    #       You should have received a copy of the LICENSE along with this program.  If not, see
    #       <http://www.gnu.org/licenses/>.
    #
    #   where *LICENSE* is the name of particular license.
    #
    note( $l->notice );
    isnt( $l->notice, $s->notice, 'Notice changed' );
    like(
        $l->notice,
        qr{
            \A
            \QCopyright (C) 2010 John Doe\E \n
            \n
            \QThis program is free software: you can \E .*? \n
            \n
            \QThis program is distributed in the hope \E .*? \n
            \n
            \QYou should have received a copy of the \E .*? \n
            \z
        }sx,
        'Notice starts with copyright statement and have 3 more paragraphs'
    );
    #   We cannot search for plain license name because it may be wrapped across lines. Let us
    #   construct regexp by replacing every space in the license name with `\s+`:
    ( my $regexp = $licenses{ $lic } ) =~ s{ }{\\s+}g;
    my @count = ( $l->notice =~ m{\bthe\s+$regexp\b}g );  # Count phrases.
    is( scalar( @count ), 3, "Phrase \"the $licenses{ $lic }\" appears 3 times in the notice" );

    # Non-default program name.
    like(
        $l->notice( 'assa' ),
        qr{
            \A
            \QCopyright (C) 2010 John Doe\E \n
            \n
            \QAssa is free software: you can \E .*? \n
            \n
            \QAssa is distributed in the hope \E .*? \n
            \n
            \QYou should have received a copy of the \E .*? along \s with \s assa\. .*? \n
            \z
        }sx,
        'Non-default program name'
    );

}; # foreach $lic

# end of file #
