#!perl
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: t/metaresources-template.t
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-MetaResources-Template.
#
#   perl-Dist-Zilla-Plugin-MetaResources-Template is free software: you can redistribute it and/or
#   modify it under the terms of the GNU General Public License as published by the Free Software
#   Foundation, either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-MetaResources-Template is distributed in the hope that it will be
#   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-MetaResources-Template. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#   The test is written using `Moose`-based `Test::Routine`. It is not big deal, because we are
#   testing plugin for `Dist::Zilla`, and `Dist-Zilla` is also `Moose`-based.

use strict;
use warnings;
use lib 't';

use Software::License::GPL_3::or_later;
use Test::Deep qw{ re };
use Test::More;
use Test::Routine::Util;
use Dist::Zilla;

plan tests => 8;

my $role = 'MetaResourcesTemplateTester';
my $license = Software::License::GPL_3::or_later->new( { holder => 'John Doe', year => '2007' } );

#   License resource in `Dist::Zilla::MetaResources` before was a `Str`, starting from v4.300039
#   they convert it to `ArrayRef[Str]`. Let us adopt it to let the test pass on older `Dist::Zilla`
#   versions.
sub license($) {
    my ( $license ) = @_;
    if ( Dist::Zilla->VERSION >= 4.300039 ) {
        $license = [ $license ];
    };
    return $license;
};

run_tests( 'Successful build', $role, {
    resources => {
        #   Check various datatypes of resources:
        'homepage'          => 'http://example.org/{{ $dist->name }}',              # String.
        'bugtracker.web'    => 'https://example.org/{{ $dist->name }}/bugs',        # Hash.
        'license'           => [ '{{ $dist->license->url }}' ],                     # Array.
        'bugtracker.mailto' => 'mailto:bugs+{{ $dist->name }}@example.org',
        'x_plugin'          => '{{ $plugin->plugin_name }}',
    },
    exp_resources => {
        homepage   => 'http://example.org/Dummy',
        bugtracker => {
            web    => 'https://example.org/Dummy/bugs',
            mailto => 'mailto:bugs+Dummy@example.org',
        },
        license    => license(
            $license->url,
        ),
        x_plugin   => 'MetaResources::Template',
    },
} );

{
    #   `MetaResources` has `BUILDARGS` method wich mangles all the options.
    #   Make sure `TextTemplater` options are not mangled.
    local ( $MY::name, $MY::bt_web, $MY::bt_mail );
    $MY::name    = "Foo";
    $MY::bt_web  = "https://example.org/$MY::name/bugs";
    $MY::bt_mail = "mailto:bugs+$MY::name\@example.org";
    run_tests( 'package works', $role, {
        resources => {
            package             => 'MY',
            'bugtracker.web'    => '{{$bt_web}}',
            'bugtracker.mailto' => '{{$bt_mail}}',
        },
        exp_resources => {
            bugtracker => {
                web    => 'https://example.org/Foo/bugs',
                mailto => 'mailto:bugs+Foo@example.org',
            },
        },
    } );
    run_tests( 'prepend works', $role, {
        resources => {
            prepend             => 'package MY;',
            'bugtracker.web'    => '{{$bt_web}}',
            'bugtracker.mailto' => '{{$bt_mail}}',
        },
        exp_resources => {
            bugtracker => {
                web    => 'https://example.org/Foo/bugs',
                mailto => 'mailto:bugs+Foo@example.org',
            },
        },
    } );
    run_tests( 'delimiters work', $role, {
        resources => {
            delimiters          => '(* *)',
            'bugtracker.web'    => '(*$MY::bt_web*)',
            'bugtracker.mailto' => '(*$MY::bt_mail*)',
        },
        exp_resources => {
            bugtracker => {
                web    => 'https://example.org/Foo/bugs',
                mailto => 'mailto:bugs+Foo@example.org',
            },
        },
    } );
}

run_tests( 'Error in string', $role, {
    resources => {
        'prepend'           => 'use strict;',
        'homepage'          => '{{$home}}',
        #^^^^^^^^
    },
    exp_exception => 'Dist::Zilla died in construction: Aborting...',
    exp_messages => [
        re( qr{^Global symbol "\$home" requires explicit package name.* at homepage line 1\b} ),
        'Problem code fragment begins at homepage line 1.',
        #                                ^^^^^^^^
        'homepage:',
        #^^^^^^^^
        '    1: {{$home}}',
        '       ^^^ Code fragment begins in line above ^^^',
        '       ^^^ Code fragment died in line above ^^^',
    ],
} );

run_tests( 'Error in hash', $role, {
    resources => {
        'prepend'           => 'use strict;',
        'bugtracker.web'    => '{{$bt_web}}',
        #^^^^^^^^^^^^^^
    },
    exp_exception => 'Dist::Zilla died in construction: Aborting...',
    exp_messages => [
        re( qr{^Global symbol "\$bt_web" requires explicit package name.* at bugtracker\.web line 1\b} ),
        'Problem code fragment begins at bugtracker.web line 1.',
        #                                ^^^^^^^^^^^^^^
        'bugtracker.web:',
        #^^^^^^^^^^^^^^
        '    1: {{$bt_web}}',
        '       ^^^ Code fragment begins in line above ^^^',
        '       ^^^ Code fragment died in line above ^^^',
    ],
} );

run_tests( 'Error in array', $role, {
    resources => {
        'prepend'           => 'use strict;',
        'license'           => [ '{{$lic}}' ],          # Array.
        #^^^^^^^^
    },
    exp_exception => 'Dist::Zilla died in construction: Aborting...',
    exp_messages => [
        re( qr{^Global symbol "\$lic" requires explicit package name.* at license#1 line 1\b} ),
        'Problem code fragment begins at license#1 line 1.',
        #                                ^^^^^^^^^
        'license#1:',
        #^^^^^^^^
        '    1: {{$lic}}',
        '       ^^^ Code fragment begins in line above ^^^',
        '       ^^^ Code fragment died in line above ^^^',
    ],
} );

run_tests( 'warning about undefined variable', $role, {
    resources => {
        'prepend'           => 'use warnings;',
        'homepage'          => '{{$OUT .= $MY::home}}',
            # ^ `{{$MY::hone}} does not cause a warning. Perl issue a warning when undefined value
            #   is used in an expression.
    },
    exp_resources => {
    },
    exp_messages => [
        re( qr{^Use of uninitialized value \$MY::home in.* at homepage line 1\b} ),
    ],
} );

done_testing;

exit( 0 );

# end of file #
