use Ranker::Ranking;
use Ranker::Error;

#| Rankings class that stores indexable Ranking objects.
unit class Ranker::Rankings is export;
=comment also does Iterable does Iterator;

# Public attributes
has $.strategy is rw; #= Ranking strategy to use on values

# Private attributes
has           @!scores;
has Ranking:D @!rankings;
has Int:D     $!index = 0;
has           %!errors;

#|«
Return the mean (sum of all scores / number of scores).
Same as self.total / self.scores-num.
»
method mean {
    Error.new('Rankings', %!errors).throw unless self!scores-are-valid;
    self.total / self.scores-num;
}

# Return the number of scores.
method scores-num( --> Int ) {
    @!scores.elems
}

#| Return the total sum of scores.
method total {
    @!scores.sum
}

#| Return the total difference.
method total-diff {
    @!scores.map((* - self.mean) ** 2).sum
}

#| Return the variance.
method variance {
    self.total-diff / @!scores.elems
}

#| Return the standard deviation among the scores.
method standard-dev {
    self.variance.sqrt;
}

#| Create a Ranking object and return it.
method create-ranking( $rank, $score, @rankables --> Ranking:D ) {
    @!scores.append: Array.new: $score xx @rankables.elems;

    # instantiate a Ranking object with the given data
    my Ranking $ranking .= new(
        :rankings(self), :index(self.elems), :$rank, :$score, :@rankables
    );

    @!rankings.push: $ranking;

    return $ranking;
}

# PRIVATE methods:

method !scores-are-valid( --> Bool ) {
    self!validate;
    return %!errors.elems == 0;
}

method !validate( --> Nil ) {
    %!errors = Empty;

    # check if scores have Any values
    my $has-Any-values = @!scores.grep(* === Any) ?? True !! False;
    %!errors<scores> = 'contains Any values' if $has-Any-values;
}

# OVERLOADED methods for positional subscripting and iteration

# Return number of Ranking objects in the class.
multi method elems( ::?CLASS:D: ) {
    return @!rankings.elems
}

# Return Ranking object at index. Otherwise, Nil.
multi method AT-POS( ::?CLASS:D: Int:D $index ) {
    return @!rankings[$index] if $index < @!rankings;
    return Nil;
}

=begin comment
# Iterate over the Ranking objects.
method iterator(){ self }
method pull-one( --> Mu ){
    if $!index < @!rankings {
        my $ranking =  @!rankings[$!index];
        $!index++;
        return $ranking;
    }
    else {
        return IterationEnd;
    }
}
=end comment
