package Shipwright::Backend::SVN;

use warnings;
use strict;
use Carp;
use File::Spec::Functions qw/catfile/;
use Shipwright::Util;
use File::Temp qw/tempdir/;
use File::Copy qw/copy/;
use File::Copy::Recursive qw/dircopy/;

our %REQUIRE_OPTIONS = ( import => [qw/source/] );

use base qw/Shipwright::Backend::Base/;

=head1 NAME

Shipwright::Backend::SVN - SVN repository backend

=head1 DESCRIPTION

This module implements a SVN repository backend for Shipwright.

=head1 METHODS

=over

=item initialize

initialize a project.

=cut

sub initialize {
    my $self = shift;
    my $dir  = $self->SUPER::initialize(@_);

    $self->delete;    # clean repository in case it exists
    $self->import(
        source      => $dir,
        _initialize => 1,
        comment     => 'created project',
    );

}

=item import

=cut

sub import {
    my $self = shift;
    return unless @_;
    return $self->SUPER::import( @_, delete => 1 );
}

# a cmd generating factory
sub _cmd {
    my $self = shift;
    my $type = shift;
    my %args = @_;
    $args{path}    ||= '';
    $args{comment} ||= '';

    for ( @{ $REQUIRE_OPTIONS{$type} } ) {
        croak "$type need option $_" unless $args{$_};
    }

    my @cmd;

    if ( $type eq 'checkout' ) {
        @cmd =
          [ $ENV{'SHIPWRIGHT_SVN'}, 'checkout', $self->repository . $args{path}, $args{target} ];
    }
    elsif ( $type eq 'export' ) {
        @cmd =
          [ $ENV{'SHIPWRIGHT_SVN'}, 'export', $self->repository . $args{path}, $args{target} ];
    }
    elsif ( $type eq 'import' ) {
        if ( $args{_initialize} ) {
            @cmd = [
                $ENV{'SHIPWRIGHT_SVN'}, 'import', $args{source},
                $self->repository . ( $args{path} || '' ),
                '-m', $args{comment},
            ];
        }
        elsif ( $args{_extra_tests} ) {
            @cmd = [
                $ENV{'SHIPWRIGHT_SVN'},         'import',
                $args{source}, $self->repository . '/t/extra',
                '-m',          $args{comment},
            ];
        }
        else {
            if ( my $script_dir = $args{build_script} ) {
                @cmd = [
                    $ENV{'SHIPWRIGHT_SVN'},       'import',
                    $script_dir, $self->repository . "/scripts/$args{name}/",
                    '-m',        $args{comment},
                ];
            }
            else {
                if ( $self->has_branch_support ) {
                    @cmd = [
                        $ENV{'SHIPWRIGHT_SVN'},
                        'import',
                        $args{source},
                        $self->repository . "/sources/$args{name}/$args{as}",
                        '-m',
                        $args{comment},
                    ];
                }
                else {
                    @cmd = [
                        $ENV{'SHIPWRIGHT_SVN'},
                        'import',
                        $args{source},
                        $self->repository . "/dists/$args{name}",
                        '-m',
                        $args{comment},
                    ];

                }
            }
        }
    }
    elsif ( $type eq 'list' ) {
        @cmd = [ $ENV{'SHIPWRIGHT_SVN'}, 'list', $self->repository . $args{path} ];
    }
    elsif ( $type eq 'commit' ) {
        @cmd =
          [ $ENV{'SHIPWRIGHT_SVN'}, 'commit', '-m', $args{comment}, $args{path} ];
    }
    elsif ( $type eq 'delete' ) {
        @cmd = [
            $ENV{'SHIPWRIGHT_SVN'}, 'delete', '-m',
            'delete ' . $args{path},
            $self->repository . $args{path},
        ];
    }
    elsif ( $type eq 'move' ) {
        @cmd = [
            $ENV{'SHIPWRIGHT_SVN'},
            'move',
            '-m',
            "move $args{path} to $args{new_path}",
            $self->repository . $args{path},
            $self->repository . $args{new_path}
        ];
    }
    elsif ( $type eq 'info' ) {
        @cmd = [ $ENV{'SHIPWRIGHT_SVN'}, 'info', $self->repository . $args{path} ];
    }
    elsif ( $type eq 'cat' ) {
        @cmd = [ $ENV{'SHIPWRIGHT_SVN'}, 'cat', $self->repository . $args{path} ];
    }
    else {
        croak "invalid command: $type";
    }

    return @cmd;
}

sub _yml {
    my $self = shift;
    my $path = shift;
    my $yml  = shift;

    $path = '/' . $path unless $path =~ m{^/};

    my ( $p_dir, $f );
    if ( $path =~ m{(.*)/(.*)$} ) {
        $p_dir = $1;
        $f     = $2;
    }

    if ($yml) {
        my $dir =
          tempdir( 'shipwright_backend_svn_XXXXXX', CLEANUP => 1, TMPDIR => 1 );
        my $file = catfile( $dir, $f );

        $self->checkout(
            path   => $p_dir,
            target => $dir,
        );

        Shipwright::Util::DumpFile( $file, $yml );
        $self->commit( path => $file, comment => "updated $path" );
    }
    else {
        my ($out) =
          Shipwright::Util->run( [ $ENV{'SHIPWRIGHT_SVN'}, 'cat', $self->repository . $path ] );
        return Shipwright::Util::Load($out);
    }
}

=item info

a wrapper around svn's info command.

=cut

sub info {
    my $self = shift;
    my ( $info, $err ) = $self->SUPER::info(@_);

    if (wantarray) {
        return $info, $err;
    }
    else {
        if ($err) {
            $err =~ s/\s+$//;
            $self->log->warn($err);
            return;
        }
        return $info;
    }
}

=item check_repository

check if the given repository is valid.

=cut

sub check_repository {
    my $self = shift;
    my %args = @_;

    if ( $args{action} eq 'create' ) {

        my ( $info, $err ) = $self->info;

        my $repo = $self->repository;

        # $err like
        # file:///tmp/svn/foo:  (Not a valid URL)
        # usually means foo doesn't exist, which is valid for create
        return 1 if $info || $err && $err =~ m{^\Q$repo\E:}m;

    }
    else {
        return $self->SUPER::check_repository(@_);
    }
    return;
}

sub _update_file {
    my $self   = shift;
    my $path   = shift;
    my $latest = shift;

    if ( $path =~ m{(.*)/(.*)$} ) {
        my $dir =
          tempdir( 'shipwright_backend_svn_XXXXXX', CLEANUP => 1, TMPDIR => 1 );
        my $file = catfile( $dir, $2 );

        $self->checkout(
            path   => $1,
            target => $dir,
        );

        copy( $latest, $file ) or confess "can't copy $latest to $file: $!";
        $self->commit(
            path    => $file,
            comment => "updated $path",
        );
    }
}

sub _update_dir {
    my $self   = shift;
    my $path   = shift;
    my $latest = shift;

    $self->delete( path => $path );
    $self->import( path => $path, source => $latest, _initialize => 1 );
}


=back

=cut

1;

__END__

=head1 AUTHORS

sunnavy  C<< <sunnavy@bestpractical.com> >>

=head1 LICENCE AND COPYRIGHT

Shipwright is Copyright 2007-2009 Best Practical Solutions, LLC.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.
