# ************************************************************************* 
# Copyright (c) 2014, SUSE LLC
# 
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
# 
# 3. Neither the name of SUSE LLC nor the names of its contributors may be
# used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# ************************************************************************* 
#
# parser module
#
package App::Dochazka::CLI::Parser;

use 5.012;
use strict;
use warnings;

use App::CELL qw( $CELL $log );
use App::Dochazka::CLI::Help qw( no_help );
use App::Dochazka::CLI::Set qw( set_employee );
use App::Dochazka::CLI::Show qw( show_employee );
use Web::MREST::CLI::UserAgent qw( send_req );
use Test::Deep::NoTest;

=head1 NAME

App::Dochazka::CLI::Parser - Parser for Dochazka command line client




=head1 VERSION

Version 0.096

=cut

our $VERSION = '0.096';
our $anything = qr/^.+$/i;



=head1 SYNOPSIS

    use Try::Tiny;
    
    my $status;
    my @tokens = split /\s+/, 'MY SAMPLE COMMAND';
    try { 
        App::Dochazka::CLI::Parse::parse_tokens( [], \@tokens ); 
    } catch { 
        $status = $_; 
    };




=head1 CLI COMMANDS

The parsing of CLI commands takes place in the C<parse_tokens> function,
which calls itself recursively until it gets to a rather macabre-sounding

    die send_req . . .

This causes control to return to the while loop in C<bin/dochazka-cli> with the
return value of the C<send_req>, which is a status object.

All tokens should be chosen to be distinguishable by their first
three characters.

=cut

sub parse_tokens {
    my ( $pre, $tokens ) = @_; 
    return $CELL->status_err( "No more tokens" ) unless ref( $tokens );
    my @tokens = @$tokens;
    my $token = shift @tokens;

    if ( @$pre == 0 ) { 

        if ( $token =~  m/^GET/i ) {
        # GET
            parse_tokens( [ 'GET' ], \@tokens ) if @tokens;
            die send_req( 'GET', '' );
        }
        elsif ( $token =~ m/^PUT/i ) {
        # PUT
            parse_tokens( [ 'PUT' ], \@tokens ) if @tokens;
            die send_req( 'PUT', '' );
        } 
        elsif ( $token =~ m/^POS/i ) {
        # POST 
            parse_tokens( [ 'POST' ], \@tokens ) if @tokens;
            die send_req( 'POST', '' );
        } 
        elsif ( $token =~ m/^DEL/i ) {
        # DELETE ''
            parse_tokens( [ 'DELETE' ], \@tokens ) if @tokens;
            die send_req( 'DELETE', '' );
        }
        elsif ( $token =~ m/^SHO/i ) {
        # SHOW
            parse_tokens( [ 'SHOW' ], \@tokens ) if @tokens;
            die no_help( $token, @tokens );
        }
        elsif ( $token =~ m/^SET/i ) {
        # SET
            parse_tokens( [ 'SET' ], \@tokens ) if @tokens;
            die no_help( $token, @tokens );
        }
        elsif ( $token =~ m/^(exi)|(qu)|(\\q)/i ) { 
        # EXIT, QUIT, and the like
            die $CELL->status_ok( 'DOCHAZKA_CLI_EXIT' );
        }   
        die $CELL->status_err( 'DOCHAZKA_CLI_PARSE_ERROR' );
    }
    elsif ( @$pre >= 1 ) { 

        my $t1 = $pre->[0];

        if ( $t1 =~ m/^(GET)|(POST)|(PUT)|(DELETE)$/ ) {
            http_test_and_die( $pre, $token, @tokens );
            die "We were supposed to die in http_test_and_die, but we didn't!";
        } 
        elsif ( $t1 eq 'SHOW' ) {

            #print "The method is SHOW\n";

            if ( $token =~ m/^EMP/i ) {
            # SHOW EMPLOYEE [$KEY]
                die show_employee( @tokens );
            }

        } elsif ( $t1 eq 'SET' ) {

            #print "The method is SET\n";

            if ( $token =~ m/^EMP/i ) {
            # SET EMPLOYEE
                die set_employee( @tokens );
            }

        }

    }
    # we have gone all the way through the state machine without a match
    die $CELL->status_err( 'DOCHAZKA_CLI_PARSE_ERROR' );

}


=head2 http_test_and_die

When the first token is GET, POST, PUT, or DELETE, we enter the "HTTP testing"
branch of the parser, which is implemented by this routine.

As the name indicates, the routine should not return.

=cut

sub http_test_and_die {
    my ( $pre, $token, @tokens ) = @_;
    my $method = $pre->[0];

    #print "http_test_and_die, method is $method, pre is " . @$pre . "\n";

    if ( @$pre == 1 ) {

        #
        # root resource
        #
        if ( $token eq '/' ) {
            die send_req( $method, '/' );
        }

        #
        # activity resource: recurse
        #
        if ( $token =~ m/^act/i ) {
            parse_tokens( [ $method, 'ACTIVITY' ], \@tokens ) if @tokens;
            die send_req( $method, 'activity' );
        }

        #
        # employee resource: recurse
        #
        if ( $token =~ m/^emp/i ) {
            parse_tokens( [ $method, 'EMPLOYEE' ], \@tokens ) if @tokens;
            die send_req( $method, 'employee' );
        }

        #
        # priv resource: recurse
        #
        if ( $token =~ m/^pri/i ) {
            parse_tokens( [ $method, 'PRIV' ], \@tokens ) if @tokens;
            die send_req( $method, 'priv' );
        }

        #
        # schedule resource: recurse
        #
        if ( $token =~ m/^sch/i ) {
            parse_tokens( [ $method, 'SCHEDULE' ], \@tokens ) if @tokens;
            die send_req( $method, 'schedule' );
        }

        #
        # interval resource: recurse
        #
        if ( $token =~ m/^int/i ) {
            parse_tokens( [ $method, 'INTERVAL' ], \@tokens ) if @tokens;
            die send_req( $method, 'interval' );
        }

        #
        # lock resource: recurse
        #
        if ( $token =~ m/^loc/i ) {
            parse_tokens( [ $method, 'LOCK' ], \@tokens ) if @tokens;
            die send_req( $method, 'lock' );
        }

        #
        # top-level resource: handle it here
        #
        # "/bugreport"
        if ( $token =~ m/^bug/i ) {
            die send_req( $method, 'bugreport' );
        }

        # "/cookies"
        if ( $token =~ m/^coo/i ) {
            die $CELL->status_ok( 'COOKIE_JAR', payload => App::Dochazka::CLI::HTTP::cookie_jar() );
        }
    
        # "/dbstatus" 
        if ( $token =~ m/^dbs/i ) {
            die send_req( $method, 'dbstatus' );
        }

        # "/docu/{pod,html,text} \"$RESOURCE'""
        if ( $token =~ m/^doc/i ) { 
            if ( scalar( @tokens ) == 2 ) {
                my $pod_html = shift @tokens;
                my $json = join( ' ', @tokens );
                if ( $json ) {
                    $json = '"' . $json . '"' unless $json =~ m/^".*"$/;
                    if ( $pod_html =~ m/^pod/i ) {
                        die send_req( $method, 'docu/pod', $json );
                    } elsif ( $pod_html =~ m/^htm/i ) {
                        die send_req( $method, 'docu/html', $json );
                    } elsif ( $pod_html =~ m/^tex/i ) {
                        die send_req( $method, 'docu/text', $json );
                    } 
                } else {
                    print "You should specify a resource\n";
                }
            } else {
                die send_req( $method, 'docu' );
            }
        }   

        # "/echo [$JSON]"
        if ( $token =~ m/^ech/i ) { 
            die send_req( $method, 'echo', join(' ', @tokens) );
        }   

        # "/employee..."
        if ( $token =~ m/^emp/i ) {
            parse_tokens( [ $method, 'EMPLOYEE' ], \@tokens ) if @tokens;
            die send_req( $method, 'employee' );
        }

        # "/forbidden"
        if ( $token =~ m/^for/i ) {
            die send_req( $method, "forbidden" );
        }

        # "/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, "help", join( ' ', @tokens ) );
        }
    
        # "/param $TYPE $JSON"
        # "/param/:type/:param"
        if ( $token =~ m/^par/i ) {
            if ( scalar( @tokens ) > 1 ) {
                my $type = $tokens[0];
                my $param = $tokens[1];
                my $json = join( ' ', @tokens[2..$#tokens] );
                die send_req( $method, "param/$type/$param", $json );
            }
        }
    
        # "/noop"
        if ( $token =~ m/^noo/i ) {
            die send_req( $method, "noop" );
        }

        # "/session"
        if ( $token =~ m/^ses/i ) {
            die send_req( $method, "session" );
        }

        # "/test/?:specs"
        if ( $token =~ m/^tes/i ) {
            my $uri_path;
            if ( @tokens ) {
                my $uri_ext .= join( ' ', @tokens ) if @tokens;
                $uri_path = "test/$uri_ext";
            } else {
                $uri_path = "test";
            }
            die send_req( $method, $uri_path );
        }

        # "/version"
        if ( $token =~ m/^ver/i and eq_deeply( $pre, [ $method ] ) ) {
            die send_req( $method, 'version' );
        }   

        # "/whoami"
        if ( $token =~ m/^who/i and eq_deeply( $pre, [ $method ] ) ) {
            die send_req( $method, 'whoami' );
        }   
    }

    #
    # interval resource handlers
    #
    if ( exists $pre->[1] and $pre->[1] eq 'INTERVAL' ) {

        # "/interval/eid/:eid/:tsrange"
        if ( $token =~ m/^eid$/ ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^\d+$/ ) {
                    my $eid = shift @tokens;
                    if ( @tokens ) {
                        my $tsrange = join(' ', @tokens);
                        if ( $tsrange =~ m/\[.+\)/ ) {
                            die send_req( $method, "interval/eid/$eid/$tsrange" );
                        }
                    }
                }
            }
        }

        # "/interval/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, 'interval/help' );
        }

        # "/interval/iid"
        # "/interval/iid/:iid"
        if ( $token =~ m/^iid$/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^[\[{]/ ) {
                    die send_req( $method, "interval/iid", join(' ', @tokens) );
                } elsif ( $tokens[0] =~ m/^\d+/ ) {
                    die send_req( $method, "interval/iid/$tokens[0]", join(' ', @tokens[1..$#tokens]) );
                }
            }
        }

        # "/interval/new"
        if ( $token =~ m/^new/i ) {
            if ( @tokens ) {
                die send_req( $method, 'interval/new', join(' ', @tokens) );
            }
        }

        # "/interval/nick/:nick/:tsrange"
        if ( $token =~ m/^nick$/ ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^[A-Za-z0-9_].+/ ) {
                    my $nick = shift @tokens;
                    if ( @tokens ) {
                        my $tsrange = join(' ', @tokens);
                        if ( $tsrange =~ m/\[.+\)/ ) {
                            die send_req( $method, "interval/nick/$nick/$tsrange" );
                        }
                    }
                }
            }
        }

        # "/interval/self/:tsrange"
        if ( $token =~ m/^self$/ ) {
            if ( @tokens ) {
                 my $tsrange = join(' ', @tokens);
                 if ( $tsrange =~ m/\[.+\)/ ) {
                     die send_req( $method, "interval/self/$tsrange" );
                 }
            }
        }

        # "/interval/summary/?:qualifiers"
        if ( $token =~ m/^sum/ ) {
            if ( @tokens ) {
                die send_req( $method, 'interval/summary/' . join( ' ', @tokens ) );
            }
            die send_req( $method, "interval/summary" );
        }

    }


    #
    # lock resource handlers
    #
    if ( exists $pre->[1] and $pre->[1] eq 'LOCK' ) {

        # "/lock/eid/:eid/:tsrange"
        if ( $token =~ m/^eid$/ ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^\d+$/ ) {
                    my $eid = shift @tokens;
                    if ( @tokens ) {
                        my $tsrange = join(' ', @tokens);
                        if ( $tsrange =~ m/\[.+\)/ ) {
                            die send_req( $method, "lock/eid/$eid/$tsrange" );
                        }
                    }
                }
            }
        }

        # "/lock/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, 'lock/help' );
        }

        # "/lock/lid"
        # "/lock/lid/:lid"
        if ( $token =~ m/^lid$/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^[\[{]/ ) {
                    die send_req( $method, "lock/lid", join(' ', @tokens) );
                } elsif ( $tokens[0] =~ m/^\d+/ ) {
                    die send_req( $method, "lock/lid/$tokens[0]", join(' ', @tokens[1..$#tokens]) );
                }
            }
        }

        # "/lock/new"
        if ( $token =~ m/^new/i ) {
            if ( @tokens ) {
                die send_req( $method, 'lock/new', join(' ', @tokens) );
            }
        }

        # "/lock/nick/:nick/:tsrange"
        if ( $token =~ m/^nick$/ ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^[A-Za-z0-9_].+/ ) {
                    my $nick = shift @tokens;
                    if ( @tokens ) {
                        my $tsrange = join(' ', @tokens);
                        if ( $tsrange =~ m/\[.+\)/ ) {
                            die send_req( $method, "lock/nick/$nick/$tsrange" );
                        }
                    }
                }
            }
        }

        # "/lock/self/:tsrange"
        if ( $token =~ m/^self$/ ) {
            if ( @tokens ) {
                 my $tsrange = join(' ', @tokens);
                 if ( $tsrange =~ m/\[.+\)/ ) {
                     die send_req( $method, "lock/self/$tsrange" );
                 }
            }
        }

    }


    #
    # schedule resource handlers
    #
    if ( exists $pre->[1] and $pre->[1] eq 'SCHEDULE' ) {

        # "/schedule/all"
        # "/schedule/all/disabled"
        if ( $token =~ m/^all/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^dis/i ) {
                    die send_req( $method, "schedule/all/disabled" );
                }
            }
            die send_req( $method, "schedule/all" );
        }

        # "/schedule/eid/:eid/?:ts"
        if ( $token =~ m/^eid/i ) {
            if ( @tokens ) {
                my $urip = "schedule/eid/$tokens[0]";
                if ( defined $tokens[1] ) {
                    $urip .= "/" . join( ' ', @tokens[1..$#tokens] );
                }
                die send_req( $method, $urip );
            }
        }
        
        # "/schedule/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, 'schedule/help' );
        }

        # "/schedule/history..."
        if ( $token =~ m/^his/i ) {
    
            # "/schedule/history"
            if ( not @tokens ) {
                die send_req( $method, "schedule/history" );
            }

            # "/schedule/history/eid/:eid [$JSON]"
            # "/schedule/history/eid/:eid/:tsrange"
            if ( $tokens[0] and $tokens[0] =~ m/^eid/i and $tokens[1] and $tokens[1] =~ m/^\d+$/ ) {
                if ( $tokens[2] and $tokens[2] =~ m/^\[/ ) {
                    die send_req( $method, "schedule/history/eid/$tokens[1]/" . join(' ', @tokens[2..$#tokens]) );
                }
                die send_req( $method, "schedule/history/eid/$tokens[1]", join(' ', @tokens[2..$#tokens]) );
            }
            
            # "/schedule/history/nick/:nick [$JSON]
            # "/schedule/history/nick/:nick/:tsrange
            if ( $tokens[0] and $tokens[0] =~ m/^nic/i and $tokens[1] ) {
                if ( $tokens[2] and $tokens[2] =~ m/^\[/ ) {
                    die send_req( $method, "schedule/history/nick/$tokens[1]/" . join(' ', @tokens[2..$#tokens]) );
                }
                die send_req( $method, "schedule/history/nick/$tokens[1]", join(' ', @tokens[2..$#tokens]));
            }

            # "/schedule/history/self"
            # "/schedule/history/self/:tsrange"
            if ( $tokens[0] and $tokens[0] =~ m/^sel/i ) {
                if ( $tokens[1] and $tokens[1] =~ m/^\[/ ) {
                    die send_req( $method, "schedule/history/self/" . join(' ', @tokens[1..$#tokens]) );
                }
                die send_req( $method, "schedule/history/self" );
            }

            # "/schedule/history/shid/:shid
            if ( $tokens[0] and $tokens[0] =~ m/^shi/i and $tokens[1] ) {
                die send_req( $method, "schedule/history/shid/$tokens[1]" );
            }

        }

        # "/schedule/intervals"
        if ( $token =~ m/^int/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^({)|(\[)/ ) {
                    die send_req( $method, "schedule/intervals", join(' ', @tokens) );
                }
            }
            die send_req( $method, "schedule/intervals" );
        }

        # "/schedule/new"
        if ( $token =~ m/^new/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^({)|(\[)/ ) {
                    die send_req( $method, "schedule/new", join(' ', @tokens) );
                }
            }
        }

        # "/schedule/nick/:nick/?:ts"
        if ( $token =~ m/^nic/i ) {
            if ( @tokens ) {
                my $urip = "schedule/nick/$tokens[0]";
                if ( defined $tokens[1] ) {
                    $urip .= "/" . join( ' ', @tokens[1..$#tokens] );
                }
                die send_req( $method, $urip );
            }
        }

        # "/schedule/self/?:ts"
        if ( $token =~ m/^sel/i ) {
            my $urip = "schedule/self";
            if ( @tokens ) {
                $urip .= "/" . join( ' ', @tokens );
            }
            die send_req( $method, $urip );
        }

        # "/schedule/sid/:sid"
        if ( $token =~ m/^sid/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^\d+/ ) {
                    if ( $method =~ m/^(GET)|(DELETE)$/ ) {
                        die send_req( $method, "schedule/sid/$tokens[0]" );
                    }
                    if ( exists $tokens[1] ) {
                        die send_req( $method, "schedule/sid/$tokens[0]", join(' ', @tokens[1..$#tokens]) );
                    }
                }
            }
        }

    }

    #
    # activity resource handlers
    #
    if ( exists $pre->[1] and $pre->[1] eq 'ACTIVITY' ) {
        
        # "/activity/aid"
        # "/activity/aid/:aid"
        if ( $token =~ m/^aid/i ) {
#            if ( $tokens[0] =~ m/^\d+$/ ) {
            if ( @tokens ) {
                my $aid = $tokens[0];
                if ( $method =~ m/^(GET)|(DELETE)$/ ) {
                    die send_req( $method, "activity/aid/$aid" );
                } elsif ( $method =~ m/PUT$/ ) {
                    die send_req( $method, "activity/aid/$aid", join(' ', @tokens[1..$#tokens]) );
                } elsif ( $method =~ m/POST$/ ) {
                    die send_req( $method, "activity/aid", join(' ', @tokens) );
                }
            } else {
                die send_req( $method, "activity/aid" );
            }
        }

        # "/activity/all"
        # "/activity/all/disabled"
        if ( $token =~ m/^all/i ) {
            if ( @tokens ) {
                die send_req( $method, 'activity/all/disabled' ) if $tokens[0] =~ m/^dis/;
            } else {
                die send_req( $method, 'activity/all' );
            }
        }

        # "/activity/code"
        # "/activity/code/:code"
        if ( $token =~ m/^cod/i ) {
#            if ( $tokens[0] =~ m/^\d+$/ ) {
            if ( @tokens ) {
                my $code = $tokens[0];
                if ( $method =~ m/^(GET)|(DELETE)$/ ) {
                    die send_req( $method, "activity/code/$code" );
                } elsif ( $method =~ m/^PUT$/ ) {
                    die send_req( $method, "activity/code/$code", join(' ', @tokens[1..$#tokens]) );
                } elsif ( $method =~ m/^POST$/ ) {
                    die send_req( $method, "activity/code", join(' ', @tokens) );
                }
            } else {
                die send_req( $method, "activity/code" );
            }
        }

        # "/activity/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, 'activity/help' );
        }

    }

    #
    # employee resource handlers
    #
    if ( exists $pre->[1] and $pre->[1] eq 'EMPLOYEE' ) {

        #print "EMPLOYEE\n";

        # "/employee/count"
        # "/employee/count/:priv"
        if ( $token =~ m/^cou/i ) {
            parse_tokens( [ $method, 'EMPLOYEE', 'COUNT' ], \@tokens ) if @tokens;
            die send_req( $method, 'employee/count' );
        } elsif ( $token =~ $anything and eq_deeply( $pre, [ $method, 'EMPLOYEE', 'COUNT' ] ) ) {
            die send_req( $method, 'employee/count/' . $token );
        }

        # "/employee/current"
        # "/employee/current/priv"
        if ( $token =~ m/^cur/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^pri/i ) {
                    die send_req( $method, 'employee/current/priv' );
                } elsif ( $method ne 'GET' ) {
                    die send_req( $method, 'employee/current', join(' ', @tokens ) );
                }
            } else {
                die send_req( $method, 'employee/current' );
            }
        }

        # "/employee/self"
        # "/employee/self/priv"
        if ( $token =~ m/^sel/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^pri/i ) {
                    die send_req( $method, 'employee/self/priv' );
                } elsif ( $method ne 'GET' ) {
                    die send_req( $method, 'employee/self', join(' ', @tokens ) );
                }
            } else {
                die send_req( $method, 'employee/self' );
            }
        }

        # "/employee/eid [$JSON]"
        # "/employee/eid/:eid [$JSON]"
        if ( $token =~ m/^eid/i ) {
            print "employee eid\n";
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^\d+/ ) {
                    my $eid = $tokens[0];
                    die send_req( $method, "employee/eid/$eid", join(' ', @tokens[1..$#tokens]) );
                } else {
                    die send_req( $method, 'employee/eid', join(' ', @tokens) );
                }
            }   
        }

        # "/employee/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, 'employee/help' );
        }

        # "/employee/nick [$JSON]"
        # "/employee/nick/:nick [$JSON]"
        if ( $token =~ m/^nic/i ) {
            if ( @tokens ) {
                if ( $tokens[0] =~ m/^\{/ ) {
                    die send_req( $method, "employee/nick", join(' ', @tokens) );
                } else {
                    my $nick = $tokens[0];
                    die send_req( $method, "employee/nick/$nick", join(' ', @tokens[1..$#tokens]) );
                }
            }
        }

        # "/employee/search/nick/:key"
        if ( $token =~ m/^sea/i ) {
            my $token = shift @tokens;
            if ( $token =~ m/^nic/i ) {
                my $key = shift @tokens;
                if ( $key ) {
                    die send_req( $method, "employee/search/nick/$key" );
                }
            }
            die send_req( $method, "employee/search" );
        }
    }   

    #
    # priv resource handlers
    #
    if ( exists $pre->[1] and $pre->[1] eq 'PRIV' ) {

        # "/priv"
        if ( $token =~ m/^pri/i and eq_deeply( $pre, [ $method ] ) ) {
            parse_tokens( [ $method, 'PRIV' ], \@tokens ) if @tokens;
            die send_req( $method, "priv" );
        }
    
        # "/priv/self/?:ts"
        if ( $token =~ m/^sel/i ) {
            my $urip = "schedule/self";
            if ( @tokens ) {
                $urip .= "/" . join( ' ', @tokens );
            }
            die send_req( $method, $urip );
        }

        # "/priv/eid/:eid/?:ts"
        if ( $token =~ m/^eid/i ) {
            if ( @tokens ) {
                my $urip = "priv/eid/$tokens[0]";
                if ( defined $tokens[1] ) {
                    $urip .= "/" . join( ' ', @tokens[1..$#tokens] );
                }
                die send_req( $method, $urip );
            }
        }
        
        # "/priv/help"
        if ( $token =~ m/^hel/i ) {
            die send_req( $method, 'priv/help' );
        }
    
        # "/priv/history..."
        if ( $token =~ m/^his/i ) {
    
            # "/priv/history"
            if ( not @tokens ) {
                die send_req( $method, "priv/history" );
            }

            # "/priv/history/eid/:eid [$JSON]"
            # "/priv/history/eid/:eid/:tsrange"
            if ( @tokens and $tokens[0] =~ m/^eid/i and defined( $tokens[1] ) ) {
                my $rest_of_tokens = join(' ', @tokens[2..$#tokens]);
                if ( defined( $rest_of_tokens ) ) {
                    if ( $rest_of_tokens =~ m/^[[(].*,.*[])]$/ ) {
                        die send_req( $method, "priv/history/eid/$tokens[1]/$rest_of_tokens" );
                    }
                    die send_req( $method, "priv/history/eid/$tokens[1]", $rest_of_tokens );
                }
                die send_req( $method, "priv/history/eid/$tokens[1]" );
            }
            

            # "/priv/history/nick/:nick [$JSON]
            # "/priv/history/nick/:nick/:tsrange
            if ( $tokens[0] and $tokens[0] =~ m/^nic/i and defined( $tokens[1] ) ) {
                my $rest_of_tokens = join(' ', @tokens[2..$#tokens]);
                if ( defined( $rest_of_tokens ) ) {
                    if ( $rest_of_tokens =~ m/^[[(].*,.*[])]$/ ) {
                        die send_req( $method, "priv/history/nick/$tokens[1]/$rest_of_tokens" );
                    }
                    die send_req( $method, "priv/history/nick/$tokens[1]", $rest_of_tokens );
                }
                die send_req( $method, "priv/history/nick/$tokens[1]" );
            }

            # "/priv/history/phid/:phid
            if ( $tokens[0] and $tokens[0] =~ m/^phi/i and $tokens[1] ) {
                die send_req( $method, "priv/history/phid/$tokens[1]" );
            }

            # "/priv/history/self"
            # "/priv/history/self/:tsrange"
            if ( $tokens[0] and $tokens[0] =~ m/^sel/i ) {
                if ( $tokens[1] and $tokens[1] =~ m/^\[/ ) {
                    die send_req( $method, "priv/history/self/" . join(' ', @tokens[1..$#tokens]) );
                }
                die send_req( $method, "priv/history/self" );
            }

        }

        # "/priv/nick/:nick/?:ts"
        if ( $token =~ m/^nic/i ) {
            if ( @tokens ) {
                my $urip = "priv/nick/$tokens[0]";
                if ( defined $tokens[1] ) {
                    $urip .= "/" . join( ' ', @tokens[1..$#tokens] );
                }
                die send_req( $method, $urip );
            }
        }

    
    }

    # we have gone all the way through the state machine without a match
    die $CELL->status_err( 'DOCHAZKA_CLI_PARSE_ERROR' );
}

1;
